#' Azure DevOps Project Repositories
#'
#' @description
#' These functions will allow you to scrape project information from Azure DevOps.
#'
#' @details
#' For more information about repository API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/git}.
#'
#' @param domain The name of the Azure DevOps organization
#' @param project Project ID or project name
#' @param repo the name of the repository in \code{project} to look at. Leave as \code{""} to get all repositories
#' within all projects of the domain
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param quiet logical whether want general running information from printing. Any issue with the API call will
#' still show up if set to \code{TRUE}
#'
#' @examples
#' \dontrun{
#' # Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key("<username>", "<password>")
#'
#' # Get repo list
#' vsts_get_repos("domain", "project", auth_key)
#'
#' # Create new repo
#' vsts_create_repo("domain", "project", "repo", auth_key)
#'
#' # Delete existing repo
#' vsts_delete_repo("domain", "project", "repo", auth_key)
#' }
#'
#' @rdname vsts_repo
#' @export
vsts_get_repos <- function(domain, project, auth_key, quiet = FALSE) {
  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/git/repositories?api-version=6.0")

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))
  if (httr::status_code(response) != 200) {
    send_failure_message(response, "get repos list")
    return(NULL)
  }

  content <- httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)$value
  if (!quiet) cat("Available repositories:", paste(content$name, collapse = ", "), "\n")

  content
}

#' @rdname vsts_repo
#' @export
vsts_create_repo <- function(domain, project, repo, auth_key, quiet = FALSE) {
  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/git/repositories?api-version=6.0")
  response <- httr::POST(
    uri,
    httr::add_headers(Authorization = auth_key),
    httr::content_type_json(),
    body = jsonlite::toJSON(list(name = repo), auto_unbox = TRUE)
  )

  if (httr::status_code(response) != 201) {
    send_failure_message(response, "create repository")
    return(NULL)
  }

  if (!quiet) cat(repo, "repository has been created in", project, "\n")

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)
}

#' @rdname vsts_repo
#' @export
vsts_delete_repo <- function(domain, project, repo, auth_key, quiet = FALSE) {
  repos <- vsts_get_repos(domain, project, auth_key, quiet = TRUE)
  repo_id <- repos[repos$name == repo, "id"]

  if (is.null(repo_id) || length(repo_id) == 0) {
    cat("Unable to find", repo, "in", project, "\n")
    return(NULL)
  }

  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/git/repositories", paste0(repo_id, "?api-version=6.0"))
  response <- httr::DELETE(uri, httr::add_headers(Authorization = auth_key))

  if (httr::status_code(response) != 204) {
    send_failure_message(response, "delete repository")
    return(NULL)
  }

  if (!quiet) cat(repo, "repository has been deleted from", project, "\n")
  return(TRUE)
}
