#' Azure DevOps Project Get Work Items
#'
#' @description
#' These functions will allow you to scrape work item information from a particular Azure DevOps project.
#'
#' @details
#' For more information about work item API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/wit/work-items}.
#'
#' @param domain The name of the Azure DevOps organization.
#' @param project Project ID or project name
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param id ID of the work item to retrieve
#' @param query a list of extra parameters that can be sent to the API call:
#' \describe{
#' \item{\code{ids}}{[character] a comma-separated list of up to 200 IDs of the work items to get}
#' \item{\code{fields}}{[character] a comma-separated list of up to 100 fields to get with each work item.
#' If not specified, all fields with values are returned. Calculated fields such as Attached File Count must be specifically
#' queried for using this parameter.}
#' \item{\code{asOf}}{[Date] gets the work items as they existed at this time}
#' \item{\code{ErrorPolicy}}{[character] determines if the call will throw an error when encountering a work item (default behavior)
#' that doesn't exist (\code{throw}) or simply omit it (\code{omit})}
#' }
#'
#' @rdname vsts_get_wk
#' @export
vsts_get_workitems <- function(domain, project, auth_key, id, query = list()) {
  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/wit/workitemsbatch?api-version=6.0")

  response <- httr::POST(
    uri,
    httr::add_headers(Authorization = auth_key),
    httr::content_type_json(),
    query = query
  )

  if (httr::status_code(response) >= 300) {
    send_failure_message(response, "get work items list")
    return(NULL)
  }

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)$value
}

#' @rdname vsts_get_wk
#' @export
vsts_get_workitem <- function(domain, project, auth_key, id) {
  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/wit/workitems", paste0(id, "?api-version=6.0"))
  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))

  if (httr::status_code(response) != 200) {
    send_failure_message(response, paste0("get work item #", id))
    return(NULL)
  }

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)
}

#' Azure DevOps Project Work Items
#'
#' @description
#' These functions will allow you to scrape work item information from a particular Azure DevOps project.
#'
#' @details
#' For more information about work item API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/wit/work-items}.
#'
#' @param domain The name of the Azure DevOps organization.
#' @param project the name of the project in \code{domain} to look at
#' @param item_type the type of work item to be created
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param ... arguments passed to \code{\link{vsts_get_workitem_fields}}
#'
#' @rdname vsts_create_wk
#' @export
vsts_create_workitem <- function(domain, project, item_type, auth_key, ...) {
  item_type_info <- vsts_get_itemtypes(domain, project, auth_key)
  item_types <- item_type_info$name
  item_type <- item_types[match(tolower(item_type), tolower(item_types))]

  if (is.na(item_type)) {
    cat("item_type not available for project. Select from:", paste(item_types, collapse = ", "), "\n")
    return(NULL)
  }

  request_info <- vsts_get_workitem_fields(...)
  request_body <- jsonlite::toJSON(request_info, auto_unbox = TRUE)

  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/wit/workitems", paste0("$", item_type, "?api-version=4.1"))

  response <- httr::POST(
    utils::URLencode(uri),
    httr::add_headers(Authorization = auth_key),
    httr::content_type("application/json-patch+json"),
    body = request_body
  )

  if (httr::status_code(response) != 200) {
    send_failure_message(response, paste("add", item_type, "to", project))
    return(NULL)
  }

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)
}

vsts_get_itemtypes <- function(domain, project, auth_key) {
  uri <- file.path(AZURE_HOME_URL, domain, project, "_apis/wit/workitemtypes?api-version=4.1")
  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))

  if (httr::status_code(response) != 200) {
    send_failure_message(response, "get work item types")
    return(NULL)
  }

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)$value
}

#' Azure DevOps Work Item Fields
#'
#' @description
#' This contains all the fields required of any work item in a visual studio project and helps
#' add/rename the fields of the selected work item.
#'
#' @details
#' For more information about work item fields API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/wit/Fields}.
#'
#' @param System.Title [character] title of the Azure DevOps work item
#' @param System.Description [character] description of the Azure DevOps work item
#' @param System.TeamProject [character] name of the Azure DevOps project
#' @param System.AreaPath [character] path of the Azure DevOps work item
#' @param System.IterationPath [character] name of the Azure DevOps iteration path
#' @param Microsoft.VSTS.Common.Priority [integer] priority of the work item - 1 to 4
#' @param ... other fields that might have been missed out originally
#'
#' @export
vsts_get_workitem_fields <- function(System.Title, System.Description, System.TeamProject,
                                     System.AreaPath, System.IterationPath,
                                     Microsoft.VSTS.Common.Priority, ...) {
  field_list <- as.list(match.call())
  field_list <- field_list[-1]

  data.frame(
    op = "add",
    path = file.path("/fields", names(field_list)),
    from = NA,
    value = unlist(field_list, use.names = FALSE)
  )
}
