% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rank_regression.R
\name{rank_regression.default}
\alias{rank_regression.default}
\title{Rank Regression for Parametric Lifetime Distributions}
\usage{
\method{rank_regression}{default}(
  x,
  y,
  status,
  distribution = c("weibull", "lognormal", "loglogistic", "sev", "normal", "logistic",
    "weibull3", "lognormal3", "loglogistic3", "exponential", "exponential2"),
  conf_level = 0.95,
  direction = c("x_on_y", "y_on_x"),
  control = list(),
  options = list(),
  ...
)
}
\arguments{
\item{x}{A numeric vector which consists of lifetime data. Lifetime data
could be every characteristic influencing the reliability of a product,
e.g. operating time (days/months in service), mileage (km, miles), load
cycles.}

\item{y}{A numeric vector which consists of estimated failure probabilities
regarding the lifetime data in \code{x}.}

\item{status}{A vector of binary data (0 or 1) indicating whether a unit is
a right censored observation (= 0) or a failure (= 1).}

\item{distribution}{Supposed distribution of the random variable.}

\item{conf_level}{Confidence level of the interval.}

\item{direction}{Direction of the dependence in the regression model.}

\item{control}{A list of control parameters (see \link[stats:optim]{optim}).

\code{control} is in use only if a three-parametric distribution was specified.
If this is the case, \code{optim} (always with \code{method = "L-BFGS-B"} and
\code{control$fnscale = -1}) is called to determine the threshold parameter
(see \link{r_squared_profiling}).}

\item{options}{A list of named options. See 'Options'.}

\item{...}{Further arguments passed to or from other methods. Currently not used.}
}
\value{
A list with classes \code{wt_model}, \code{wt_rank_regression} and \code{wt_model_estimation}
which contains:
\itemize{
\item \code{coefficients} : A named vector of estimated coefficients (parameters of the
assumed distribution). \strong{Note}: The parameters are given in the
(log-)location-scale-parameterization.
\item \code{confint} : Confidence intervals for the (log-)location-scale parameters.
For threshold distributions no confidence interval for the threshold parameter
can be computed. If \code{direction = "y_on_x"}, back-transformed confidence
intervals are provided.
\item \code{shape_scale_coefficients} : Only included if \code{distribution} is \code{"weibull"}
or \code{"weibull3"} (parameterization used in \link[stats:Weibull]{Weibull}).
\item \code{shape_scale_confint} : Only included if \code{distribution} is \code{"weibull"}
or \code{"weibull3"}. Approximated confidence intervals for scale \eqn{\eta} and
shape \eqn{\beta} (and threshold \eqn{\gamma} if \code{distribution = "weibull3"}).
\item \code{varcov} : Only provided if \code{options = list(conf_method = "HC")} (default).
Estimated heteroscedasticity-consistent (\strong{HC}) variance-covariance matrix
for the (log-)location-scale parameters.
\item \code{r_squared} : Coefficient of determination.
\item \code{data} : A \code{tibble} with columns \code{x}, \code{status} and \code{prob}.
\item \code{distribution} : Specified distribution.
\item \code{direction} : Specified direction.
}
}
\description{
This function fits a regression model to a linearized parametric lifetime
distribution for complete and (multiple) right-censored data. The parameters
are determined in the frequently used (log-)location-scale parameterization.

For the Weibull, estimates are additionally transformed such that they are in
line with the parameterization provided by the \emph{stats} package
(see \link[stats:Weibull]{Weibull}).
}
\details{
The confidence intervals of the parameters are computed on the basis of a
heteroscedasticity-consistent (\strong{HC}) covariance matrix. Here it should be
said that there is no statistical foundation to determine the standard errors
of the parameters using \emph{Least Squares} in context of \emph{Rank Regression}.
For an accepted statistical method use \link[=ml_estimation]{maximum likelihood}.

If \code{options = list(conf_method = "Mock")}, the argument \code{distribution} must be
one of \code{"weibull"} and \code{"weibull3"}. The approximated confidence intervals
for the Weibull parameters can then only be estimated on the following
confidence levels (see 'References' \emph{(Mock, 1995)}):
\itemize{
\item \code{conf_level = 0.90}
\item \code{conf_level = 0.95}
\item \code{conf_level = 0.99}
}
}
\section{Options}{

Argument \code{options} is a named list of options:\tabular{ll}{
   Name \tab Value \cr
   \code{conf_method} \tab \code{"HC"} (default) or \code{"Mock"} \cr
}
}

\examples{
# Vectors:
obs <- seq(10000, 100000, 10000)
status_1 <- c(0, 1, 1, 0, 0, 0, 1, 0, 1, 0)

cycles <- alloy$cycles
status_2 <- alloy$status

# Example 1 - Fitting a two-parametric weibull distribution:
tbl_john <- estimate_cdf(
  x = obs,
  status = status_1,
  method = "johnson"
)

rr <- rank_regression(
  x = tbl_john$x,
  y = tbl_john$prob,
  status = tbl_john$status,
  distribution = "weibull",
  conf_level = 0.90
)

# Example 2 - Fitting a three-parametric lognormal distribution:
tbl_kaplan <- estimate_cdf(
  x = cycles,
  status = status_2,
  method = "kaplan"
)

rr_2 <- rank_regression(
  x = tbl_kaplan$x,
  y = tbl_kaplan$prob,
  status = tbl_kaplan$status,
  distribution = "lognormal3"
)

}
\references{
\itemize{
\item Mock, R., Methoden zur Datenhandhabung in Zuverlässigkeitsanalysen,
vdf Hochschulverlag AG an der ETH Zürich, 1995
\item Meeker, William Q; Escobar, Luis A., Statistical methods for reliability data,
New York: Wiley series in probability and statistics, 1998
}
}
\seealso{
\link{rank_regression}
}
