% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base_a_zfunction.R, R/base_b_zfold.R,
%   R/base_c_zfitter.R
\name{zfunction}
\alias{zfunction}
\alias{zfold}
\alias{zfitter}
\title{Create a pipe-friendly version of a function}
\usage{
zfunction(fun, x, x_not_found = c("error", "warning", "ok"))

zfold(fun, x, x_not_found = c("error", "warning", "ok"))

zfitter(fun)
}
\arguments{
\item{fun}{The function to adapt (for \code{zfitter()} this should be a fitting
function that takes \code{formula} and \code{data} parameters). The name should not be
quoted, rather, the actual function should be passed (prefixed with package
if needed).}

\item{x}{The name of the argument that should be moved to the front of the
argument list. Can be passed with or without quotes, and is processed using
non-standard evaluation unless surrounded with curlies, as in \code{{value}},
see details below.}

\item{x_not_found}{How to handle the case where the value of \code{x} is not the
name of a parameter in \code{fun}. If \code{error}, abort the function. If \code{ok},
prepend the value to the existing parameter list. This can be useful if
looking to pipe data into a parameter that is hidden by a \code{...}.}
}
\description{
These functions all serve the role of rearranging the arguments of other
functions, in order to create pipe-friendly versions.

\code{zfunction()} rearranges the arguments of any function moving the specified
argument to the front of the list, so that this argument becomes the
recipient of piping. It returns a copy of the input function, that is
identical to the original except for the order of the arguments.

\code{zfold()} creates a pipe-friendly version of a function of the standard
format by creating a fold (or wrapper) around it with the parameters
reordered. Compared to using \code{zfunction()}, which makes a copy of the
original function with rearranged the parameters, this creates a wrapper that
in turn will call the original function with all passed parameters. This is
good for making pipe-friendly versions of \code{S3} generics, whereas rearranging
parameters directly will break the \code{S3} dispatch mechanism.

\code{zfitter()} creates a pipe-friendly version of a fitting function of the
standard format –– that is a function with a \code{formula} parameter followed by
a \code{data} parameter. It also shortens very long data names (longer than 32
characters by default), which otherwise are a nuisance when the data comes
from the pipe, because the pipeline gets converted to a very long function
call.
}
\details{
The \code{x} parameter is processed using non-standard evaluation, which can be
disabled using curly brackets. In other words, the following are all
equivalent, and return a file renaming function with the \code{to} parameter as
the first one:
\itemize{
\item \code{zfunction(file.rename, to)}
\item \code{zfunction(file.rename, "to")}
\item \verb{param_name <- "to"; zfunction(file.rename, \{param_name\})}
}
}
\examples{
# A a grep function with x as first param is often useful
zgrep <- zfunction(grep, x)
carnames <- rownames(mtcars)
grep("ll", carnames, value=TRUE)
zgrep(carnames, "ll", value=TRUE)

# zfunction() is the best approach to wrapping functions such as
# `pls::plsr()` that hide the data parameter behind the `...`.
if (requireNamespace("pls")) {
  zplsr <- zfunction(pls::plsr, data, x_not_found = "ok")
  zplsr(cars, dist ~ speed)
}

# Curly {x} handling: These are all equivalent
param_name <- "to";
f1 <- zfunction(file.rename, to)
f2 <- zfunction(file.rename, "to")
f3 <- zfunction(file.rename, {param_name})

# Using zfold() to create a grep() wrapper with the desired arg order
zgrep <- zfold(grep, x)
carnames <- rownames(mtcars)
grep("ll", carnames, value=TRUE)
zgrep(carnames, "ll", value=TRUE)

# Using zfitter to wrap around a fitting function
# (this is the actual way zlm_robust is defined in this package)
if (requireNamespace("estimatr", quietly = TRUE)) {
  zlm_robust <- zfitter(estimatr::lm_robust)
  zlm_robust(cars, speed~dist)

  # The resulting function works well the native pipe ...
  if ( getRversion() >= "4.1.0" ) {
    cars |> zlm_robust( speed ~ dist )
  }
}

# ... or with dplyr
if ( require("dplyr", warn.conflicts=FALSE) ) {

  # Pipe cars dataset into zlm_robust for fitting
  cars \%>\% zlm_robust( speed ~ dist )

  # Process iris with filter() before piping. Print a summary()
  # of the fitted model using zprint() before assigning the
  # model itself (not the summary) to m.
  m <- iris \%>\%
    dplyr::filter(Species=="setosa") \%>\%
    zlm_robust(Sepal.Length ~ Sepal.Width + Petal.Width) \%>\%
    zprint(summary)
}

}
