% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/karfima.R
\name{KARFIMA.functions}
\alias{KARFIMA.functions}
\alias{KARFIMA.sim}
\alias{KARFIMA.extract}
\alias{KARFIMA.fit}
\title{Functions to simulate, extract components and fit KARFIMA models}
\usage{
KARFIMA.sim(n = 1, burn = 0, xreg = NULL, rho = 0.5, y.lower = 0,
  y.upper = 1, coefs = list(alpha = 0, beta = NULL, phi = NULL, theta =
  NULL, d = 0, nu = 20), y.start = NULL, xreg.start = NULL,
  xregar = TRUE, error.scale = 1, complete = FALSE, inf = 1000,
  linkg = c("logit", "logit"), seed = NULL, rngtype = 2, debug = FALSE)

KARFIMA.extract(yt, xreg = NULL, nnew = 0, xnew = NULL, p, q,
  rho = 0.5, y.lower = 0, y.upper = 1, coefs = list(), lags = list(),
  fixed.values = list(), fixed.lags = list(), y.start = NULL,
  xreg.start = NULL, xregar = TRUE, error.scale = 1, inf = 1000,
  m = 0, linkg = c("logit", "logit"), llk = TRUE, sco = FALSE,
  info = FALSE, extra = FALSE, debug = FALSE)

KARFIMA.fit(yt, xreg = NULL, nnew = 0, xnew = NULL, p = 0, d = TRUE,
  q = 0, m = 0, inf = 1000, rho = 0.5, y.lower = 0, y.upper = 1,
  start = list(), ignore.start = FALSE, lags = list(),
  fixed.values = list(), fixed.lags = list(), lower = list(nu = 0),
  upper = list(nu = Inf), linkg = c("logit", "logit"), sco = FALSE,
  info = FALSE, extra = FALSE, xregar = TRUE, y.start = NULL,
  xreg.start = NULL, error.scale = 1, control = list(), report = TRUE,
  debug = FALSE, ...)
}
\arguments{
\item{n}{a strictly positive integer. The sample size of yt (after burn-in).
Default is 1.}

\item{burn}{a non-negative integer. The length of the "burn-in" period. Default is 0.}

\item{xreg}{optionally, a vector or matrix of external regressors.
For simulation purposes, the length of xreg must be \code{n+burn}.
Default is \code{NULL}. For extraction or fitting purposes, the length
of \code{xreg} must be the same as the length of the observed time series
\eqn{y_t}.}

\item{rho}{a positive number, between 0 and 1, indicating the quantile
to be modeled so that \eqn{\mu_t} is the conditional \eqn{rho}-quantile.}

\item{y.lower}{the lower limit for the density support. Default is 0.}

\item{y.upper}{the upper limit for the density support. Default is 1.}

\item{coefs}{a list with the coefficients of the model. An empty list will result
in an error. The arguments that can be passed through this list are:
\itemize{
\item \code{alpha} optionally, a numeric value corresponding to the intercept.
If the argument is missing, it will be treated as zero. See
\sQuote{The BTSR structure} in \code{\link{btsr.functions}}.

\item \code{beta} optionally, a vector of coefficients corresponding to the
regressors in \code{xreg}. If \code{xreg} is provided but \code{beta} is
missing in the \code{coefs} list, an error message is issued.

\item \code{phi} optionally, for the simulation function this must be a vector
of size \eqn{p}, corresponding to the autoregressive coefficients
(including the ones that are zero), where \eqn{p} is the AR order. For
the extraction and fitting functions, this is a vector with the non-fixed
values in the vector of autoregressive coefficients.

\item \code{theta} optionally, for the simulation function this must be a vector
of size \eqn{q}, corresponding to the moving average coefficients
(including the ones that are zero), where \eqn{q} is the MA order. For
the extraction and fitting functions, this is a vector with the non-fixed
values in the vector of moving average coefficients.

\item \code{d} optionally, a numeric value corresponding to the long memory
parameter. If the argument is missing, it will be treated as zero.

\item \code{nu} the dispersion parameter. If missing, an error message is issued.

}}

\item{y.start}{optionally, an initial value for yt (to be used
in the recursions). Default is \code{NULL}, in which case, the recursion assumes
that \eqn{g_2(y_t) = 0}, for \eqn{t < 1}.}

\item{xreg.start}{optionally, a vector of initial value for xreg
(to be used in the recursions). Default is \code{NULL}, in which case, the recursion assumes
that \eqn{X_t = 0}, for \eqn{t < 1}. If \code{xregar = FALSE} this argument
is ignored.}

\item{xregar}{logical; indicates if xreg is to be included in the
AR part of the model.  See \sQuote{The BTSR structure}. Default is \code{TRUE}.}

\item{error.scale}{the scale for the error term. See \sQuote{The BTSR structure}
in \code{\link{btsr.functions}}. Default is 1.}

\item{complete}{logical; if \code{FALSE} the function returns only the simulated
time series yt, otherwise, additional time series are provided.
Default is \code{FALSE}}

\item{inf}{the truncation point for infinite sums. Default is 1,000.
In practice, the Fortran subroutine uses \eqn{inf = q}, if \eqn{d = 0}.}

\item{linkg}{character or a two character vector indicating which
links must be used in the model.  See \sQuote{The BTSR structure}
in \code{\link{btsr.functions}} for details and \code{\link{link.btsr}}
for valid links. If only one value is provided, the same link is used
for \eqn{mu_t} and for \eqn{y_t} in the AR part of the model.
Default is \code{c("logit", "logit")}. For the linear link, the constant
will be always 1.}

\item{seed}{optionally, an integer which gives the value of the fixed
seed to be used by the random number generator. If missing, a random integer
is chosen uniformly from 1,000 to 10,000.}

\item{rngtype}{optionally, an integer indicating which random number generator
is to be used. Default is 2: the Mersenne Twister algorithm. See \sQuote{Common Arguments}
in \code{\link{btsr.functions}}.}

\item{debug}{logical, if \code{TRUE} the output from FORTRAN is return (for
debugging purposes).  Default is \code{FALSE} for all models.}

\item{yt}{a numeric vector with the observed time series. If missing, an error
message is issued.}

\item{nnew}{optionally, the number of out-of sample predicted values required.
Default is 0.}

\item{xnew}{a vector or matrix, with \code{nnew} observations of the
regressors observed/predicted values corresponding to the period of
out-of-sample forecast. If \code{xreg = NULL}, \code{xnew} is ignored.}

\item{p}{a non-negative integer. The order of AR polynomial.
If missing, the value of \code{p} is calculated from length(coefs$phi)
and length(fixed.values$phi). For fitting, the default is 0.}

\item{q}{a non-negative integer. The order of the MA polynomial.
If missing, the value of \code{q} is calculated from length(coefs$theta)
and length(fixed.values$theta). For fitting, the default is 0.}

\item{lags}{optionally, a list with the lags that the values in \code{coefs} correspond to.
The names of the entries in this list must match the ones in \code{coefs}.
For one dimensional coefficients, the \code{lag} is obviously always 1 and can
be suppressed. An empty list indicates that either the argument \code{fixed.lags}
is provided or all lags must be used.}

\item{fixed.values}{optionally, a list with the values of the coefficients
that are fixed. By default, if a given vector (such as the vector of AR coefficients)
has fixed values and the corresponding entry in this list is empty, the fixed values
are set as zero. The names of the entries in this list must match the ones
in \code{coefs}.}

\item{fixed.lags}{optionally, a list with the lags that the fixed values
in \code{fixed.values} correspond to. The names of the entries in this list must
match the ones in \code{fixed.values}. ##' For one dimensional coefficients, the
\code{lag} is obviously always 1 and can be suppressed. If an empty list is provided
and the model has fixed lags, the argument \code{lags} is used as reference.}

\item{m}{a non-negative integer indicating the starting time for the sum of the
partial log-likelihoods, that is \eqn{\ell = \sum_{t = m+1}^n \ell_t}. Default is
0.}

\item{llk}{logical, if \code{TRUE} the value of the log-likelihood function
is returned. Default is \code{TRUE}.}

\item{sco}{logical, if \code{TRUE} the score vector is returned.
Default is \code{FALSE}.}

\item{info}{logical, if \code{TRUE} the information matrix is returned.
Default is \code{FALSE}. For the fitting function, \code{info} is automatically
set to \code{TRUE} when \code{report = TRUE}.}

\item{extra}{logical, if \code{TRUE} the matrices and vectors used to
calculate the score vector and the information matrix are returned.
Default is \code{FALSE}.}

\item{d}{logical, if \code{TRUE}, the parameter \code{d} is included
in the model either as fixed or non-fixed. If \code{d = FALSE} the value is
fixed as 0. The default is \code{TRUE}.}

\item{start}{a list with the starting values for the non-fixed coefficients
of the model. If an empty list is provided, the function \code{\link{coefs.start}}
is used to obtain starting values for the parameters.}

\item{ignore.start}{logical,  if starting values are not provided, the
function uses the default values and \code{ignore.start} is ignored.
In case starting values are provided and \code{ignore.start = TRUE}, those
starting values are ignored and recalculated. The default is \code{FALSE}.}

\item{lower}{optionally, list with the lower bounds for the
parameters. The names of the entries in these lists must match the ones
in \code{start}. The default is to assume that the parameters have no lower
bound except for \code{nu}, for which de default is 0. Only the bounds for
bounded parameters need to be specified.}

\item{upper}{optionally, list with the upper bounds for the
parameters. The names of the entries in these lists must match the ones
in \code{start}. The default is to assume that the parameters have no upper
bound. Only the bounds for bounded parameters need to be specified.}

\item{control}{a list with configurations to be passed to the
optimization subroutines. Missing arguments will receive default values. See
\cite{\link{fit.control}}.}

\item{report}{logical, if \code{TRUE} the summary from model estimation is
printed and \code{info} is automatically set to \code{TRUE}. Default is \code{TRUE}.}

\item{...}{further arguments passed to the internal functions.}
}
\value{
The function \code{KARFIMA.sim} returns the simulated time series yt  by default.
If \code{complete = TRUE}, a list with the following components
is returned instead:
\itemize{
\item \code{model}: string with the text \code{"KARFIMA"}

\item \code{yt}: the simulated time series

\item \code{mut}: the conditional mean

\item \code{etat}: the linear predictor \eqn{g(\mu_t)}

\item \code{error}: the error term \eqn{r_t}

\item \code{xreg}: the regressors (if included in the model).

\item \code{debug}: the output from FORTRAN (if requested).

}

The function \code{KARFIMA.extract} returns a list with the following components.

\itemize{
\item \code{model}: string with the text \code{"KARFIMA"}

\item \code{coefs}: the coefficients of the model passed through the
\code{coefs} argument

\item \code{yt}: the observed time series

\item \code{gyt}: the transformed time series \eqn{g_2(y_t)}

\item \code{mut}: the conditional mean

\item \code{etat}: the linear predictor \eqn{g_1(\mu_t)}

\item \code{error}: the error term \eqn{r_t}

\item \code{xreg}: the regressors (if included in the model).

\item \code{sll}: the sum of the conditional log-likelihood (if requested)

\item \code{sco}: the score vector  (if requested)

\item \code{info}: the information matrix  (if requested)

\item \code{Drho}, \code{T}, \code{E}, \code{h}: additional matrices and vectors
used to calculate the score vector and the information matrix.  (if requested)

\item \code{yt.new}: the out-of-sample forecast  (if requested)

\item \code{out.Fortran}: FORTRAN output  (if requested)
}

The function \code{btsr.fit} returns a list with the following components.
Each particular model can have additional components in this list.

\itemize{
\item \code{model}: string with the text \code{"KARFIMA"}

\item \code{convergence}: An integer code. 0 indicates successful completion.
The error codes depend on the algorithm used.

\item \code{message}: A character string giving any additional information
returned by the optimizer, or NULL.

\item \code{counts}: an integer giving the number of function evaluations.

\item \code{control}: a list of control parameters.

\item \code{start}: the starting values used by the algorithm.

\item \code{coefficients}: 	The best set of parameters found.

\item \code{n}: the sample size used for estimation.

\item \code{series}: the observed time series

\item \code{gyt}: the transformed time series \eqn{g_2(y_t)}

\item \code{fitted.values}: the conditional mean, which corresponds to
the in-sample forecast, also denoted fitted values

\item \code{etat}: the linear predictor \eqn{g_1(\mu_t)}

\item \code{error.scale}: the scale for the error term.

\item \code{error}: the error term \eqn{r_t}

\item \code{residual}: the observed minus the fitted values. The same as
the \code{error} term if \code{error.scale = 0}.

\item \code{forecast}: the out-of-sample forecast (if requested).

\item \code{xnew}: the observations of the regressors observed/predicted
values corresponding to the period of out-of-sample forecast.
Only inlcudes if \code{xreg} is not \code{NULL} and \code{nnew > 0}.

\item \code{sll}: the sum of the conditional log-likelihood (if requested)

\item \code{info.Matrix}: the information matrix  (if requested)

\item \code{configs}: a list with the configurations adopted to fit the model.
This information is used by the prediction function.

\item \code{out.Fortran}: FORTRAN output  (if requested)

\item \code{call}: a string with the description of the fitted model.

}
}
\description{
These functions can be used to simulate, extract components
and fit any model of the class \code{karfima}. A model with
class \code{karfima} is a special case of a model with class \code{btsr} .
See \sQuote{The BTSR structure} in \code{\link{btsr.functions}} for
more details on the general structure.

The KARMA model, the Kumaraswamy regression and a i.i.d. sample
from a Kumaraswamy distribution can be obtained as special cases.
See \sQuote{Details}.
}
\details{
The KARMA model  and the Kumaraswamy regression can be
obtained as special cases of the KARFIMA model.

\itemize{
\item KARFIMA: is obtained by default.

\item KARMA: is obtained by setting \code{d = 0}.

\item Kumaraswamy regression: is obtained by setting \code{p = 0},
\code{q = 0} and \code{d = FALSE}. The \code{error.scale} is irrelevant.
The second argument in \code{linkg} is irrelevant.

\item an i.i.d. sample from a Kumaraswamy distribution
is obtained by  setting \code{linkg = "linear"}, \code{p = 0}, \code{q = 0},
\code{coefs$d = 0}, \code{d = FALSE}. (\code{error.scale} and
\code{xregar} are irrelevant)
}

The function \code{KARFIMA.sim} generates a random sample from a KARFIMA(p,d,q)
model.

The function \code{KARFIMA.extract} allows the user to extract the
components \eqn{y_t}, \eqn{\mu_t},  \eqn{\eta_t = g(\mu_t)}, \eqn{r_t},
the log-likelihood, and the vectors and matrices used to calculate the
score vector and the information matrix associated to a given set of parameters.

This function can be used by any user to create an objective function
that can be passed to optimization algorithms not available in the BTSR Package.

The function \code{KARFIMA.fit} fits a KARFIMA model to a given univariate time
series. For now, available optimization algorithms are \code{"L-BFGS-B"} and
\code{"Nelder-Mead"}. Both methods accept bounds for the parameters. For
\code{"Nelder-Mead"}, bounds are set via parameter transformation.
}
\examples{
# Generating a Kumaraswamy model were mut does not vary with time
# For linear link, alpha = mu
#
# Warning:
#      |log(1-rho)| >> |log(1 - mu^nu)|
# may cause numerical instability.

y <- KARFIMA.sim(linkg = "linear", n = 1000, seed = 2021,
                 coefs = list(alpha = 0.7, nu = 2))
hist(y)

 #------------------------------------------------------------
 # Generating a Kumaraswamy model were mut does not vary with time
 # For linear link, alpha = mu
 #
 # Warning:
 #      |log(1-rho)| >> |log(1 - mu^nu)|
 # may cause numerical instability.
 #------------------------------------------------------------

 m1 <- KARFIMA.sim(linkg = "linear",n = 100,
                   complete = TRUE, seed = 2021,
                   coefs = list(alpha = 0.7, nu = 2))

 #------------------------------------------------------------
 #  Extracting the conditional time series given yt and
 #  a set of parameters
 #------------------------------------------------------------

 # Assuming that all coefficients are non-fixed
 e1 = KARFIMA.extract(yt = m1$yt, coefs = list(alpha = 0.7, nu = 2),
                      link = "linear", llk = TRUE,
                      sco = TRUE, info = TRUE)

 #----------------------------------------------------
 # comparing the simulated and the extracted values
 #----------------------------------------------------
 cbind(head(m1$mut), head(e1$mut))

 #---------------------------------------------------------
 # the log-likelihood, score vector and information matrix
 #---------------------------------------------------------
 e1$sll
 e1$score
 e1$info.Matrix


# Generating a Kumaraswamy model were mut does not vary with time
# For linear link, alpha = mu
#
# Warning:
#      |log(1-rho)| >> |log(1 - mu^nu)|
# may cause numerical instability.

y <- KARFIMA.sim(linkg = "logit", n = 100, seed = 2021,
               coefs = list(alpha = 0.7, nu = 2))

# fitting the model
f <- KARFIMA.fit(yt = y, report = TRUE,
                 start = list(alpha = 0.5, nu = 1),
                 linkg = "logit", d = FALSE)

}
\seealso{
\code{\link{btsr.sim}}

\code{\link{btsr.extract}}

\code{\link{btsr.fit}}
}
