% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_secondary.R
\name{estimate_secondary}
\alias{estimate_secondary}
\title{Estimate a Secondary Observation from a Primary Observation}
\usage{
estimate_secondary(
  data,
  secondary = secondary_opts(),
  delays = delay_opts(LogNormal(meanlog = Normal(2.5, 0.5), sdlog = Normal(0.47, 0.25),
    max = 30), weight_prior = FALSE),
  truncation = trunc_opts(),
  obs = obs_opts(),
  stan = stan_opts(),
  burn_in = 14,
  CrIs = c(0.2, 0.5, 0.9),
  filter_leading_zeros = FALSE,
  zero_threshold = Inf,
  priors = NULL,
  model = NULL,
  weigh_delay_priors = FALSE,
  verbose = interactive(),
  ...,
  reports
)
}
\arguments{
\item{data}{A \verb{<data.frame>} containing the \code{date} of report and both
\code{primary} and \code{secondary} reports.}

\item{secondary}{A call to \code{\link[=secondary_opts]{secondary_opts()}} or a list containing the
following  binary variables: cumulative, historic, primary_hist_additive,
current, primary_current_additive. These parameters control the structure of
the secondary model, see \code{\link[=secondary_opts]{secondary_opts()}} for details.}

\item{delays}{A call to \code{\link[=delay_opts]{delay_opts()}} defining delay distributions between
primary and secondary observations See the documentation of \code{\link[=delay_opts]{delay_opts()}}
for details. By default a diffuse prior  is assumed with a mean of 14 days
and standard deviation of 7 days (with a standard deviation of 0.5 and 0.25
respectively on the log scale).}

\item{truncation}{A call to \code{\link[=trunc_opts]{trunc_opts()}} defining the truncation of
the observed data. Defaults to \code{\link[=trunc_opts]{trunc_opts()}}, i.e. no truncation.  See the
\code{\link[=estimate_truncation]{estimate_truncation()}} help file for an approach to estimating this from
data where the \code{dist} list element returned by \code{\link[=estimate_truncation]{estimate_truncation()}} is
used as the \code{truncation} argument here, thereby propagating the uncertainty
in the estimate.}

\item{obs}{A list of options as generated by \code{\link[=obs_opts]{obs_opts()}} defining the
observation model. Defaults to \code{\link[=obs_opts]{obs_opts()}}.}

\item{stan}{A list of stan options as generated by \code{\link[=stan_opts]{stan_opts()}}. Defaults
to \code{\link[=stan_opts]{stan_opts()}}. Can be used to override \code{data}, \code{init}, and \code{verbose}
settings if desired.}

\item{burn_in}{Integer, defaults to 14 days. The number of data points to
use for estimation but not to fit to at the beginning of the time series.
This must be less than the number of observations.}

\item{CrIs}{Numeric vector of credible intervals to calculate.}

\item{filter_leading_zeros}{Logical, defaults to TRUE. Should zeros at the
start of the time series be filtered out.}

\item{zero_threshold}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}} Numeric defaults
to Inf. Indicates if detected zero cases are meaningful by using a threshold
number of cases based on the 7-day average. If the average is above this
threshold then the zero is replaced using \code{fill}.}

\item{priors}{A \verb{<data.frame>} of named priors to be used in model fitting
rather than the defaults supplied from other arguments. This is typically
useful if wanting to inform an estimate from the posterior of another model
fit.}

\item{model}{A compiled stan model to override the default model. May be
useful for package developers or those developing extensions.}

\item{weigh_delay_priors}{Logical. If TRUE, all delay distribution priors
will be weighted by the number of observation data points, in doing so
approximately placing an independent prior at each time step and usually
preventing the posteriors from shifting. If FALSE (default), no weight will
be applied, i.e. delay distributions will be treated as a single
parameters.}

\item{verbose}{Logical, should model fitting progress be returned. Defaults
to \code{\link[=interactive]{interactive()}}.}

\item{...}{Additional parameters to pass to \code{\link[=stan_opts]{stan_opts()}}.}

\item{reports}{Deprecated; use \code{data} instead.}
}
\value{
A list containing: \code{predictions} (a \verb{<data.frame>} ordered by date
with the primary, and secondary observations, and a summary of the model
estimated secondary observations), \code{posterior} which contains a summary of
the entire model posterior, \code{data} (a list of data used to fit the
model), and \code{fit} (the \code{stanfit} object).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}
Estimates the relationship between a primary and secondary observation, for
example hospital admissions and deaths or hospital admissions and bed
occupancy. See \code{\link[=secondary_opts]{secondary_opts()}} for model structure options. See parameter
documentation for model defaults and options. See the examples for case
studies using synthetic data and
\href{https://gist.github.com/seabbs/4f09d7609df298db7a86c31612ff9d17}{here}
for an example of forecasting Covid-19 deaths from Covid-19 cases. See
\href{https://gist.github.com/seabbs/4dad3958ca8d83daca8f02b143d152e6}{here} for
a prototype function that may be used to estimate and forecast a secondary
observation from a primary across multiple regions and
\href{https://github.com/epiforecasts/covid.german.forecasts/blob/master/rt-forecast/death-from-cases.R}{here} # nolint
for an application forecasting Covid-19 deaths in Germany and Poland.
}
\examples{
\donttest{
# set number of cores to use
old_opts <- options()
options(mc.cores = ifelse(interactive(), 4, 1))

# load data.table for manipulation
library(data.table)

#### Incidence data example ####

# make some example secondary incidence data
cases <- example_confirmed
cases <- as.data.table(cases)[, primary := confirm]
# Assume that only 40 percent of cases are reported
cases[, scaling := 0.4]
# Parameters of the assumed log normal delay distribution
cases[, meanlog := 1.8][, sdlog := 0.5]

# Simulate secondary cases
cases <- convolve_and_scale(cases, type = "incidence")
#
# fit model to example data specifying a weak prior for fraction reported
# with a secondary case
inc <- estimate_secondary(cases[1:60],
  obs = obs_opts(scale = list(mean = 0.2, sd = 0.2), week_effect = FALSE)
)
plot(inc, primary = TRUE)

# forecast future secondary cases from primary
inc_preds <- forecast_secondary(
  inc, cases[seq(61, .N)][, value := primary]
)
plot(inc_preds, new_obs = cases, from = "2020-05-01")

#### Prevalence data example ####

# make some example prevalence data
cases <- example_confirmed
cases <- as.data.table(cases)[, primary := confirm]
# Assume that only 30 percent of cases are reported
cases[, scaling := 0.3]
# Parameters of the assumed log normal delay distribution
cases[, meanlog := 1.6][, sdlog := 0.8]

# Simulate secondary cases
cases <- convolve_and_scale(cases, type = "prevalence")

# fit model to example prevalence data
prev <- estimate_secondary(cases[1:100],
  secondary = secondary_opts(type = "prevalence"),
  obs = obs_opts(
    week_effect = FALSE,
    scale = list(mean = 0.4, sd = 0.1)
  )
)
plot(prev, primary = TRUE)

# forecast future secondary cases from primary
prev_preds <- forecast_secondary(
 prev, cases[seq(101, .N)][, value := primary]
)
plot(prev_preds, new_obs = cases, from = "2020-06-01")

options(old_opts)
}
}
