#' gsnImportDAVID
#'
#' @description Add DAVID search data to a GSNData object, as generated by the the DAVID web application
#' (\href{https://david.ncifcrf.gov/}{https://david.ncifcrf.gov/}) output using either the
#' "Functional Annotation Chart" or "Functional Annotation Cluster" results output options. The data set
#' can be either in the form of a data.frame or specified as import from an output text file. (See Details below)
#'
#' @param object A GSNData object.
#'
#' @param pathways_data An (optional) data.frame containing the results of DAVID analysis. (Either this or the
#' \code{filename} argument must be set. Such a data.frame can be obtained by using the \code{read_david_data_file()}
#' function to parse a DAVID "Functional Annotation Chart" or "Functional Annotation Cluster" results text file with
#' the default options (output = "flat", redundant = FALSE, sep = "\\t").
#'
#' @param filename An (optional) filename for data sets read from a text file containing DAVID results. This is ignored
#' if the \code{pathways_data} argument is set.
#'
#' @param id_col (optional) A character vector of length 1 indicating the name of the column used as a key for gene
#' sets or modules. This is normally the \code{Term} field of DAVID data, which must be the same as the names of gene
#' sets in the gene set collection specified with the \code{geneSetCollection} argument used when building the gene
#' set network. By default this value is \code{'Term'}. The IDs must correspond to the names of the gene sets provided,
#' or an error will be thrown.
#'
#' @param stat_col (optional) A character vector of length 1 indicating the name of the column used as a statistic
#' to evaluate the quality of pathways results. The function scans through possible \code{stat_col} values
#' ("FDR", "Bonferroni", "Benjamini", "PValue" ), and uses the first one it finds.
#'
#' @param sig_order (optional) Either \code{'loToHi'} (default) or \code{'hiToLo'} depending on the statistic used to
#' evaluate pathways results.
#'
#' @param n_col (optional) Specifies the column containing the number of genes in the gene set. Generally, this is the number
#' of genes in the gene set that are attested in an expression data set (Defaults to 'Count', if that is present, otherwise
#  the function searches for columns: "SIZE", "N1" or "N").
#'
#' @param sep A separator for text file import, defaults to "\\t". Ignored if \code{filename} is not specified.
#'
#' @return This returns a GSNData object containing imported pathways data.
#'
#' @details
#'
#' Note: An error is thrown if all gene set IDs in the genePresenceAbsense are not present in the GSEA \code{NAME} column.
#' However, if there are gene set IDs present in the pathways data that are absent from the \code{$genePresenceAbsence}
#' matrix, then this method emits a warning. It also checks for the standard GSEA data set column names, and if some are
#' missing, it will emit a warning.
#'
#' @export
#'
#'
#' @seealso
#'  \code{\link{gsnAddPathwaysData}}
#'  \code{\link{gsnImportCERNO}}
#'  \code{\link{gsnImportGSNORA}}
#'  \code{\link{gsnImportGenericPathways}}
#'
#' @importFrom utils read.table
#'
gsnImportDAVID <- function( object, pathways_data = NULL, filename = NULL, id_col = NULL, stat_col = NULL, sig_order = NULL, n_col = NULL, sep = "\t" ){
  stopifnot( "GSNData" %in% class( object ) )

  if( is.null( pathways_data ) && is.null( filename ) )
    stop( "The 'pathways_data' and 'filename' arguments cannot both be NULL." )
  if( is.null( pathways_data ) ){
    pathways_data <- read_david_data_file( file = filename, output = "flat", sep = sep )
  }
  if( !is.null(sig_order) && ! sig_order %in% c( "loToHi", "hiToLo" ) )
    stop( "Invalid sig_order: ", as.character( sig_order ) )
  if( ! is.null(stat_col) && ! stat_col %in% colnames( pathways_data ) )
    stop( "stat_col '", stat_col, "' not found in pathways data."  )

  david_fieldnames <- c("Category", "Term", "Count", "%", "PValue", "Genes", "List Total",
                        "Pop Hits", "Pop Total", "Fold Enrichment", "Bonferroni", "Benjamini",  "FDR")

  if( length( missing_fieldnames <- david_fieldnames[! david_fieldnames %in% colnames(pathways_data)] ) > 0 ){
    warning( "Data is missing the following DAVID fields:", paste0( missing_fieldnames,  collapse = ", " ) )
  }
  # Check for id_col
  .id_col = 'Term'
  if( ! .id_col %in% colnames(pathways_data) ) .id_col <- NULL

  # Scan for standard stat_cols
  .stat_col_names <- c("FDR", "Bonferroni", "Benjamini", "PValue" )
  .stat_col <- .stat_col_names[.stat_col_names %in% colnames(pathways_data)]
  .sig_order <- "loToHi"
  if( length( .stat_col ) == 0 ){
    .stat_col <- NULL
    .sig_order <- NULL
  } else .stat_col <- .stat_col[1]

  # Scan for standard n_cols
  .n_col_names <- c("Count", "SIZE", "N1", "N" )
  .n_col <- .n_col_names[.n_col_names %in% colnames(pathways_data)]
  if( length( .n_col ) == 0 ) .n_col <- NULL else .n_col <- .n_col[1]

  pathways <- list( data = pathways_data, type = "david", id_col = .id_col, stat_col = .stat_col, sig_order = .sig_order, n_col = .n_col )
  if( !is.null(id_col) ) pathways$id_col <- id_col
  if( !is.null(stat_col) ) pathways$stat_col <- stat_col
  if( !is.null(sig_order) ) pathways$sig_order <- sig_order
  if( !is.null(n_col) ) pathways$n_col <- n_col

  if( is.null( pathways$id_col ) ) stop("id_col not defined.")
  if( is.null( pathways$stat_col ) ) stop("stat_col not defined.")
  if( is.null( pathways$sig_order ) ) stop("sig_order not defined.")

  if( ! all( colnames( object$genePresenceAbsence ) %in% pathways$data[[pathways$id_col]] ) )
    stop("Error: Pathways data do not match gene set collection. They are missing gene sets from gene set collection.")
  if( ! all( pathways$data[[pathways$id_col]] %in% colnames( object$genePresenceAbsence ) ) )
    warning("Warning: Pathways data do not match gene set collection. They contain gene sets not present in gene set collection.")

  object$pathways <- pathways
  object
}



