% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IncrementalTimeSuppression.R
\name{IncrementalTimeSuppression}
\alias{IncrementalTimeSuppression}
\title{Incremental Time Suppression}
\usage{
IncrementalTimeSuppression(
  data,
  fun,
  timeVar,
  formula,
  suppressedData = NULL,
  subTotals = FALSE,
  finalTotal = FALSE,
  totalPriority = !isFALSE(finalTotal),
  ...
)
}
\arguments{
\item{data}{Input data as a data frame}

\item{fun}{A function: \code{\link{GaussSuppressionFromData}} or one of its wrappers such as
\code{\link{SuppressSmallCounts}} and \code{\link{SuppressDominantCells}}.}

\item{timeVar}{The time period variable}

\item{formula}{formula A formula defining tables within the time periods.
Therefore, the variable \code{timeVar}  should not be included.}

\item{suppressedData}{A data frame or a list of data frames as output from \code{\link{GaussSuppressionFromData}}.}

\item{subTotals}{Whether all cumulative totals over time should be included.}

\item{finalTotal}{When \code{FALSE}, the \code{timeVar} total is named according to time period categories.}

\item{totalPriority}{When \code{FALSE}, the \code{timeVar}  totals are not prioritized
for publication. In other words, these totals are preferred for secondary suppression.}

\item{...}{Further parameters to \code{\link{GaussSuppressionFromData}}}
}
\value{
A data.frame
}
\description{
Suppression function useful for sequential calculations over successive time periods.
}
\note{
This function has been made internal since it is new and future non-backward compatible changes may occur.
}
\examples{
# Generating a dataset spanning four quarters 
d2s <- SSBtoolsData("d2s")
d <- rbind(d2s, d2s, d2s, d2s)
set.seed(10)
d$freq[25:96] <- round(d$freq[25:96] + 9 * rnorm(72))
d$freq[d$freq < 0] <- 0
d$quarter <- rep(c("Q1", "Q2", "Q3", "Q4"), each = 24)

# Redefine the function so that several parameters are preset. 
# Also, a simpler function name.   
STS <- function(data, ...) {
  IncrementalTimeSuppression(data, 
          fun = SuppressSmallCounts, 
          timeVar = "quarter", 
          formula = ~main_income * size, 
          maxN = 15, freqVar = "freq", ...)}
          
          
# Default settings without suppressedData as input
a1 <- STS(d[1:24, ])   
a2 <- STS(d[1:48, ])
a3 <- STS(d[1:72, ])
a4 <- STS(d, finalTotal = TRUE)

# The quarters and named totals
unique(a1$quarter)
unique(a2$quarter)
unique(a3$quarter)
unique(a4$quarter)

# Default settings with suppressedData as input
b2 <- STS(d[1:48, ], suppressedData = a1)
b3 <- STS(d[1:72, ], suppressedData = b2)
b4 <- STS(d, finalTotal = TRUE, suppressedData = b3)

# Without totalPriority, suppression will be the same as before. 
# suppressedData makes no difference.
# However, if, for example, there is a changed version of the suppression 
# algorithm, it may be important to use suppressedData 
identical(a2$suppressed, b2$suppressed)
identical(a3$suppressed, b3$suppressed)
identical(a4$suppressed, b4$suppressed) # totalPriority here, since finalTotal 

# With totalPriority and  all the subtotals 
# Note: subtotals are not prioritized
c2 <- STS(d[1:48, ], subTotals = TRUE, totalPriority = TRUE)
c3 <- STS(d[1:72, ], subTotals = TRUE, totalPriority = TRUE)
c4 <- STS(d, subTotals = TRUE, finalTotal = TRUE)
unique(c2$quarter)
unique(c3$quarter)
unique(c4$quarter)

# With such a method, we can see that is important to take into account 
# previously published results.
# Here this is not done and we see differences.
a2[a2$suppressed | c2$suppressed, ]
c2[a2$suppressed | c2$suppressed, ]
c3[SSBtools::Match( c2[a2$suppressed | c2$suppressed, 1:4], c3[1:4]), ]
c4[SSBtools::Match( c2[a2$suppressed | c2$suppressed, 1:4], c4[1:4]), ]


# Here we take into account previously published results.
d2 <- STS(d[1:48, ], subTotals = TRUE, totalPriority = TRUE, suppressedData = a1)
d3 <- STS(d[1:72, ], subTotals = TRUE, totalPriority = TRUE, suppressedData = d2)
d4 <- STS(d, subTotals = TRUE, finalTotal = TRUE, suppressedData = d3)

SSBtools::SortRows(d2[d2$suppressed, ])
SSBtools::SortRows(d3[d3$suppressed, ])

# With such a method, some annual totals must be suppressed
SSBtools::SortRows(d4[d4$suppressed, ]) 

# If necessary, several suppressed data sets can be taken into account
e4 <- STS(d, finalTotal = TRUE, suppressedData = list(a1, a2))

}
\seealso{
\code{\link{AdditionalSuppression}}
}
\keyword{internal}
