#' Create an AlignmentsTrack for a custom JBrowse 2 view
#'
#' Creates the necessary configuration string for an
#' indexed BAM or CRAM alignment so that it can be used
#' in a JBrowse custom linear genome view.
#'
#' It is important to note that while only the BAM/CRAM file is
#' passed as an argument, \code{tracks_alignment} assumes that a BAM/CRAM
#' index of the same name is located with the file
#'
#' For example:
#'
#' \code{track_alignments("data/alignments.bam")}
#'
#' Assumes that \code{data/alignments.bam.bai} also exists.
#'
#' This is a JBrowse 2 convention, and the default naming output of samtools
#'
#' For more information on creating an index with samtools, visit
#' \url{https://www.htslib.org/}
#'
#' @param track_data the URL to the BAM/CRAM alignments
#' @param assembly the config string generated by \code{assembly}
#'
#' @return a character vector of stringified AlignmentTrack JSON configuration
#'
#' @export
#'
#' @examples
#' assembly <- assembly("https://jbrowse.org/genomes/hg19/fasta/hg19.fa.gz", bgzip = TRUE)
#'
#' track_alignments("alignments.bam", assembly)
#' track_alignments("alignments.cram", assembly)
track_alignments <- function(track_data, assembly) {
  type <- "AlignmentsTrack"
  name <- get_name(track_data)
  assembly_name <- get_assembly_name(assembly)
  track_id <- stringr::str_c(assembly_name, "_", name)
  adapter <- get_alignment_adapter(track_data, assembly)

  # return the stringified JSON of track config
  as.character(
    stringr::str_glue(
      "{{",
      '"type": "{type}", ',
      '"name": "{name}", ',
      '"assemblyNames": ["{assembly_name}"], ',
      '"trackId": "{track_id}", ',
      "{adapter} ",
      "}}"
    )
  )
}

get_alignment_adapter <- function(track_data, assembly) {
  # strip the .gz extension before checking for BAM or CRAM
  track_non_gz <- strip_gz(track_data)

  if (stringr::str_ends(track_non_gz, ".bam")) {
    index <- stringr::str_c(track_data, ".bai")
    as.character(
      stringr::str_glue(
        '"adapter": {{ ',
        '"type": "BamAdapter", ',
        '"bamLocation": {{ ',
        '"uri": "{track_data}" ',
        "}}, ",
        '"index": {{ "location": {{ "uri": "{index}" }} }} ',
        "}}"
      )
    )
  } else if (stringr::str_ends(track_non_gz, ".cram")) {
    index <- stringr::str_c(track_data, ".crai")
    sequence_adapter <- get_assembly_adapter(assembly)
    as.character(
      stringr::str_glue(
        '"adapter": {{ ',
        '"type": "CramAdapter", ',
        '"cramLocation": {{ ',
        '"uri": "{track_data}" ',
        "}}, ",
        '"craiLocation": {{ "uri": "{index}" }}, ',
        '"sequenceAdapter": {sequence_adapter} ',
        "}}"
      )
    )
  } else {
    stop("alignment data must be either BAM or CRAM")
  }
}
