% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emmeans_ED.R
\name{emmeans_ED}
\alias{emmeans_ED}
\title{Make emmeans object for an expected dose}
\usage{
emmeans_ED(
  object,
  specs = ~0,
  left = NULL,
  right = NULL,
  tran = NULL,
  p = 0.5,
  p.name = "probability"
)
}
\arguments{
\item{object}{An object that can be given to \code{\link[emmeans]{emmeans}}. Typically a model fitted by \code{\link{glm}}.}

\item{specs}{As for \code{\link[emmeans]{emmeans}}. Typically as one-sided \code{\link{formula}}. Defaults to \code{~0}.}

\item{left}{A list specifying the left end point of the linear span of continuous variables in which to measure the ED values. Defaults to \code{NULL}.}

\item{right}{A list specifying the right end point of the linear span of continuous variables in which to measure the ED values. Defaults to \code{NULL}.}

\item{tran}{Possible transformation of the scale of the ED values. If given then backtransformation can be done using the technology of the \code{\link[emmeans]{emmeans}}. The default value \code{tran=NULL} corresponds to no transformation.}

\item{p}{Numeric vector given the targeted predictions. Typically probabilities, where the default value \code{p=0.5} corresponds to ED50.}

\item{p.name}{The name of the variable containing \code{p}. If \code{p} contains more than one value, then this will also appear in \code{@misc$by.vars} in the \code{emmGrid} object. Defaults to \code{p.name="probability"}.}
}
\value{
An object of class \code{\link[emmeans]{emmGrid-class}}.
}
\description{
Solves linear equations in continuous explanatory variables in order to find the expected dose. A typical application could be to find LD50, i.e. the lethal dose killing 50 percent of the population, from a probit analysis fitted by \code{\link{glm}}. The associated variance-covariance matrix is found using the Delta method.
}
\details{
Find the 'expected dose' along a gradient in the space of numeric predictor variables. The options 'left' and 'right' specify the endpoints of this gradient. Typically these endpoints should be chosen as 0 and 1 for the numeric predictor of interest. If both endpoints are chosen as NULL then these choices are taken for all numeric predictors.
}
\examples{
# Data from: C.I. Bliss, "The calculation of the dose-mortality curve", 
# Annals of Applied Biology, 134–167, 1935.
\donttest{
# import data from dobson package
library(dobson)
data(beetle)
m0 <- glm(cbind(y,n-y)~x,data=beetle,family=binomial(link="cloglog"))
# ED50 computation
summary(emmeans_ED(m0,tran="log10"),type="response")
# Visualization using the tidyverse
library(tidyverse)
LCL <- Vectorize(function(y,n) binom.test(y,n)$conf.int[1])
UCL <- Vectorize(function(y,n) binom.test(y,n)$conf.int[2])
beetle <- mutate(beetle,LCL=LCL(y,n),UCL=UCL(y,n))
emmeans_ED(m0,p=seq(0.001,0.999,length.out=100),tran="log10") \%>\% 
  summary(type="response") \%>\% as.data.frame() \%>\% 
  mutate(probability=as.numeric(as.character(probability))) \%>\%
  ggplot(aes(x=probability,y=response,ymin=asymp.LCL,ymax=asymp.UCL)) + 
  geom_ribbon(alpha=0.2,fill="blue") + geom_line() +
  xlab("Death probability") +
  ylab(expression(expected~dose~CS[2]~mg/l)) +
  geom_errorbarh(aes(xmin=LCL,xmax=UCL,y=10^x),beetle,inherit.aes=FALSE) +
  geom_point(aes(x=y/n,y=10^x),beetle,inherit.aes=FALSE)
}

}
\author{
Bo Markussen
}
