% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/metab.bayesian.R
\name{metab.bayesian}
\alias{metab.bayesian}
\title{Metabolism model based on a bayesian parameter estimation framework}
\usage{
metab.bayesian(do.obs, do.sat, k.gas, z.mix, irr, wtr, priors, ...)
}
\arguments{
\item{do.obs}{Vector of dissovled oxygen concentration observations, mg L^-1}

\item{do.sat}{Vector of dissolved oxygen saturation values based on water temperature. Calculate using \link{o2.at.sat}}

\item{k.gas}{Vector of kGAS values calculated from any of the gas flux models
(e.g., \link{k.cole}) and converted to kGAS using \link{k600.2.kGAS}}

\item{z.mix}{Vector of mixed-layer depths in meters. To calculate, see \link{ts.meta.depths}}

\item{irr}{Vector of photosynthetically active radiation in \eqn{\mu mol\ m^{-2} s^{-1}}{micro mols / m^2 / s}}

\item{wtr}{Vector of water temperatures in \eqn{^{\circ}C}{degrees C}. Used in scaling respiration with temperature}

\item{priors}{Parameter priors supplied as a named numeric vector (example: c("gppMu"=0, "gppSig2"=1E5, "rMu"=0, "rSig2"=1E5, "kSig2"=NA))}

\item{...}{additional arguments; currently "datetime" is the only recognized argument passed through \code{...}}
}
\value{
A data.frame with columns corresponding to components of metabolism
\describe{
\item{GPP}{numeric estimate of Gross Primary Production, \eqn{mg O_2 L^{-1} d^{-1}}{mg O2 / L / d}}
\item{R}{numeric estimate of Respiration, \eqn{mg O_2 L^{-1} d^{-1}}{mg O2 / L / d}}
\item{NEP}{numeric estimate of Net Ecosystem production, \eqn{mg O_2 L^{-1} d^{-1}}{mg O2 / L / d}}
}
}
\description{
This function runs the bayesian metabolism model on the supplied
gas concentration and other supporting data. This allows for both estimates of
metabolism along with uncertainty around the parameters.
}
\examples{
\dontrun{
library(rLakeAnalyzer)

doobs = load.ts(system.file('extdata',
                           'sparkling.doobs', package="LakeMetabolizer"))
wtr = load.ts(system.file('extdata',
                         'sparkling.wtr', package="LakeMetabolizer"))
wnd = load.ts(system.file('extdata',
                         'sparkling.wnd', package="LakeMetabolizer"))
irr = load.ts(system.file('extdata',
                         'sparkling.par', package="LakeMetabolizer"))

#Subset a day
mod.date = as.POSIXct('2009-07-08', 'GMT')
doobs = doobs[trunc(doobs$datetime, 'day') == mod.date, ]
wtr = wtr[trunc(wtr$datetime, 'day') == mod.date, ]
wnd = wnd[trunc(wnd$datetime, 'day') == mod.date, ]
irr = irr[trunc(irr$datetime, 'day') == mod.date, ]

k600 = k.cole.base(wnd[,2])
k.gas = k600.2.kGAS.base(k600, wtr[,3], 'O2')
do.sat = o2.at.sat(wtr[,1:2], altitude=300)

metab.bayesian(irr=irr[,2], z.mix=rep(1, length(k.gas)),
              do.sat=do.sat[,2], wtr=wtr[,2],
              k.gas=k.gas, do.obs=doobs[,2])
}
}
\author{
Ryan Batt, Luke A. Winslow
}
\references{
Holtgrieve, Gordon W., Daniel E. Schindler, Trevor a. Branch, and Z. Teresa A'mar.
2010. \emph{Simultaneous Quantification of Aquatic Ecosystem Metabolism and Reaeration
Using a Bayesian Statistical Model of Oxygen Dynamics}.
Limnology and Oceanography 55 (3): 1047-1062. doi:10.4319/lo.2010.55.3.1047.
http://www.aslo.org/lo/toc/vol_55/issue_3/1047.html.
}
\seealso{
\link{metab.mle}, \link{metab.bookkeep}, \link{metab.kalman}
}

