% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse_SAR.CWOSL.R
\name{analyse_SAR.CWOSL}
\alias{analyse_SAR.CWOSL}
\title{Analyse SAR CW-OSL measurements}
\usage{
analyse_SAR.CWOSL(
  object,
  signal.integral.min = NA,
  signal.integral.max = NA,
  background.integral.min = NA,
  background.integral.max = NA,
  OSL.component = NULL,
  rejection.criteria = list(),
  dose.points = NULL,
  trim_channels = FALSE,
  mtext.outer = "",
  plot = TRUE,
  plot_onePage = FALSE,
  plot_singlePanels = FALSE,
  onlyLxTxTable = FALSE,
  ...
)
}
\arguments{
\item{object}{\linkS4class{RLum.Analysis} (\strong{required}):
input object containing data for analysis, alternatively a \link{list} of
\linkS4class{RLum.Analysis} objects can be provided. The object should contain \strong{only} curves
considered part of the SAR protocol (see Details).}

\item{signal.integral.min}{\link{integer} (\strong{required}):
lower bound of the signal integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the minimum signal integral for the \code{Tx} curve. Can be set to \code{NA}, in this
case no integrals are taken into account.}

\item{signal.integral.max}{\link{integer} (\strong{required}):
upper bound of the signal integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the maximum signal integral for the \code{Tx} curve. Can be set to \code{NA}, in this
case no integrals are taken into account.}

\item{background.integral.min}{\link{integer} (\strong{required}):
lower bound of the background integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the minimum background integral for the \code{Tx} curve. Can be set to \code{NA}, in this
case no integrals are taken into account.}

\item{background.integral.max}{\link{integer} (\strong{required}):
upper bound of the background integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the maximum background integral for the \code{Tx} curve. Can be set to \code{NA}, in this
case no integrals are taken into account.}

\item{OSL.component}{\link{character} or \link{integer} (\emph{optional}): s single index
or a \link{character} defining the signal component to be evaluated.
It requires that the object was processed by \code{OSLdecomposition::RLum.OSL_decomposition}.
This argument can either be the name of the OSL component assigned by
\code{OSLdecomposition::RLum.OSL_global_fitting} or the index in the descending
order of decay rates. Then \code{"1"} selects the fastest decaying component, \code{"2"}
the second fastest and so on. Can be a \link{list} of \link{integer}s or strings (or mixed)
If object is a \link{list} and this parameter is provided as \link{list} it alternates over
the elements (aliquots) of the object list, e.g., \code{list(1,2)} processes the first
aliquot with component \code{1} and the second aliquot with component \code{2}.
\code{NULL} does not process any component.}

\item{rejection.criteria}{\link{list} (\emph{with default}):
provide a \emph{named} list and set rejection criteria in \strong{percentage}
for further calculation. Can be a \link{list} in a \link{list}, if \code{object} is of type \link{list}.
Note: If an \emph{unnamed} \link{list} is provided the new settings are ignored!

Allowed arguments are \code{recycling.ratio}, \code{recuperation.rate},
\code{palaeodose.error}, \code{testdose.error}, \code{exceed.max.regpoint = TRUE/FALSE},
\code{recuperation_reference = "Natural"} (or any other dose point, e.g., \code{"R1"}).
Example: \code{rejection.criteria = list(recycling.ratio = 10)}.
Per default all numerical values are set to 10, \code{exceed.max.regpoint = TRUE}.
Every criterion can be set to \code{NA}. In this value are calculated, but not considered, i.e.
the RC.Status becomes always \code{'OK'}}

\item{dose.points}{\link{numeric} (\emph{optional}):
a numeric vector containing the dose points values. Using this argument
overwrites dose point values extracted from other data. Can be a \link{list} of
\link{numeric} vectors, if \code{object} is of type \link{list}}

\item{trim_channels}{\link{logical} (\emph{with default}): trim channels per record category
to the lowest number of channels in the category by using \link{trim_RLum.Data}.
Applies only to \code{OSL} and \code{IRSL} curves. For a more granular control use \link{trim_RLum.Data}
before passing the input object.}

\item{mtext.outer}{\link{character} (\emph{optional}):
option to provide an outer margin \code{mtext}. Can be a \link{list} of \link{character}s,
if \code{object} is of type \link{list}}

\item{plot}{\link{logical} (\emph{with default}): enable/disable the plot output.}

\item{plot_onePage}{\link{logical} (\emph{with default}): enable/disable one page
plot output.}

\item{plot_singlePanels}{\link{logical} (\emph{with default}) or \link{numeric} (\emph{optional}):
single plot output (\code{TRUE/FALSE}) to allow for plotting the results in single plot windows.
If a \link{numeric} vector is provided the plots can be selected individually, i.e.
\code{plot_singlePanels = c(1,2,3,4)} will plot the TL and Lx, Tx curves but
not the legend (5) or the
growth curve (6), (7) and (8) belong to rejection criteria plots. Requires
\code{plot = TRUE}.}

\item{onlyLxTxTable}{\link{logical} (\emph{with default}): If \code{TRUE} the dose response
curve fitting and plotting is skipped.
This allows to get hands on the \code{Lx/Tx} table for large datasets
without the need for a curve fitting.}

\item{...}{further arguments that will be passed to the function
\link{plot_GrowthCurve} or \link{calc_OSLLxTxRatio}
(supported: \code{background.count.distribution}, \code{sigmab}, \code{sig0}).
\strong{Please note} that if you consider to use the early light subtraction
method you should provide your own \code{sigmab} value!}
}
\value{
A plot (\emph{optional}) and an \linkS4class{RLum.Results} object is
returned containing the following elements:

\item{data}{\link{data.frame} containing De-values, De-error and further parameters}
\item{LnLxTnTx.values}{\link{data.frame} of all calculated Lx/Tx values including signal,
background counts and the dose points}
\item{rejection.criteria}{\link{data.frame} with values that might by used as rejection criteria.
\code{NA} is produced if no R0 dose point exists.}
\item{Formula}{\link{formula} formula that have been used for the growth curve fitting}

The output should be accessed using the function \link{get_RLum}.
}
\description{
The function performs a SAR CW-OSL analysis on an
\linkS4class{RLum.Analysis} object including growth curve fitting.
}
\details{
The function performs an analysis for a standard SAR protocol measurements
introduced by Murray and Wintle (2000) with CW-OSL curves. For the
calculation of the \code{Lx/Tx} value the function \link{calc_OSLLxTxRatio} is
used. For \strong{changing the way the Lx/Tx error is calculated} use the argument
\code{background.count.distribution} and \code{sigmab}, which will be passed to the function
\link{calc_OSLLxTxRatio}.

\strong{What is part of a SAR sequence?}

The function is rather picky when it comes down to accepted curve input (OSL,IRSL,...) and structure.
A SAR sequence is basically a set of \eqn{L_{x}/T_{x}} curves. Hence, every 2nd curve
is considered a shine-down curve related to the test dose. It also means that the number of
curves for \eqn{L_{x}} has to be equal to the number of \eqn{T_{x}} curves, and that
hot-bleach curves \strong{do not} belong into a SAR sequence; at least not for the analysis.
Other curves allowed and processed are preheat curves, or preheat curves measured as TL, and
irradiation curves. The later one indicates the duration of the irradiation, the
dose and test dose points, e.g., as part of XSYG files.

\strong{Argument \code{object} is of type \code{list}}

If the argument \code{object} is of type \link{list} containing \strong{only}
\linkS4class{RLum.Analysis} objects, the function re-calls itself as often as elements
are in the list. This is useful if an entire measurement wanted to be analysed without
writing separate for-loops. To gain in full control of the parameters (e.g., \code{dose.points}) for
every aliquot (corresponding to one \linkS4class{RLum.Analysis} object in the list), in
this case the arguments can be provided as \link{list}. This \code{list} should
be of similar length as the \code{list} provided with the argument \code{object},
otherwise the function will create an own list of the requested length.
Function output will be just one single \linkS4class{RLum.Results} object.

Please be careful when using this option. It may allow a fast an efficient data analysis, but
the function may also break with an unclear error message, due to wrong input data.

\strong{Working with IRSL data}

The function was originally designed to work just for 'OSL' curves,
following the principles of the SAR protocol. An IRSL measurement protocol
may follow this procedure, e.g., post-IR IRSL protocol (Thomsen et al.,
2008). Therefore this functions has been enhanced to work with IRSL data,
however, the function is only capable of analysing curves that follow the
SAR protocol structure, i.e., to analyse a post-IR IRSL protocol, curve data
have to be pre-selected by the user to fit the standards of the SAR
protocol, i.e., Lx,Tx,Lx,Tx and so on.

Example: Imagine the measurement contains \code{pIRIR50} and \code{pIRIR225} IRSL curves.
Only one curve type can be analysed at the same time: The \code{pIRIR50} curves or
the \code{pIRIR225} curves.

\strong{Supported rejection criteria}

\verb{[recycling.ratio]}: calculated for every repeated regeneration dose point.

\verb{[recuperation.rate]}: recuperation rate calculated by comparing the
\code{Lx/Tx} values of the zero regeneration point with the \code{Ln/Tn} value (the \code{Lx/Tx}
ratio of the natural signal). For methodological background see Aitken and
Smith (1988). As a variant with the argument \code{recuperation_reference} another dose point can be
selected as reference instead of \code{Ln/Tn}.

\verb{[testdose.error]}: set the allowed error for the test dose, which per
default should not exceed 10\%. The test dose error is calculated as \code{Tx_net.error/Tx_net}.
The calculation of the \eqn{T_{n}} error is detailed in \link{calc_OSLLxTxRatio}.

\verb{[palaeodose.error]}: set the allowed error for the De value, which per
default should not exceed 10\%.

\strong{Irradiation times}

The function makes two attempts to extra irradiation data (dose points)
automatically from the input object, if the argument \code{dose.points} was not
set (aka set to \code{NULL}).
\enumerate{
\item It searches in every curve for an info object called \code{IRR_TIME}. If this was set, any value
set here is taken as dose point.
\item If the object contains curves of type \code{irradiation}, the function tries to
use this information to assign these values to the curves. However, the function
does \strong{not} overwrite values preset in \code{IRR_TIME}.
}
}
\note{
This function must not be mixed up with the function
\link{Analyse_SAR.OSLdata}, which works with
\linkS4class{Risoe.BINfileData} objects.

\strong{The function currently does support only 'OSL', 'IRSL' and 'POSL' data!}
}
\section{Function version}{
 0.10.5
}

\examples{

##load data
##ExampleData.BINfileData contains two BINfileData objects
##CWOSL.SAR.Data and TL.SAR.Data
data(ExampleData.BINfileData, envir = environment())

##transform the values from the first position in a RLum.Analysis object
object <- Risoe.BINfileData2RLum.Analysis(CWOSL.SAR.Data, pos=1)

##perform SAR analysis and set rejection criteria
results <- analyse_SAR.CWOSL(
object = object,
signal.integral.min = 1,
signal.integral.max = 2,
background.integral.min = 900,
background.integral.max = 1000,
log = "x",
fit.method = "EXP",
rejection.criteria = list(
  recycling.ratio = 10,
  recuperation.rate = 10,
  testdose.error = 10,
  palaeodose.error = 10,
  recuperation_reference = "Natural",
  exceed.max.regpoint = TRUE)
)

##show De results
get_RLum(results)

##show LnTnLxTx table
get_RLum(results, data.object = "LnLxTnTx.table")

} 

\section{How to cite}{
Kreutzer, S., 2025. analyse_SAR.CWOSL(): Analyse SAR CW-OSL measurements. Function version 0.10.5. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.0.0. https://r-lum.github.io/Luminescence/
}

\references{
Aitken, M.J. and Smith, B.W., 1988. Optical dating: recuperation
after bleaching. Quaternary Science Reviews 7, 387-393.

Duller, G., 2003. Distinguishing quartz and feldspar in single grain
luminescence measurements. Radiation Measurements, 37 (2), 161-165.

Murray, A.S. and Wintle, A.G., 2000. Luminescence dating of quartz using an
improved single-aliquot regenerative-dose protocol. Radiation Measurements
32, 57-73.

Thomsen, K.J., Murray, A.S., Jain, M., Boetter-Jensen, L., 2008. Laboratory
fading rates of various luminescence signals from feldspar-rich sediment
extracts. Radiation Measurements 43, 1474-1486.
doi:10.1016/j.radmeas.2008.06.002
}
\seealso{
\link{calc_OSLLxTxRatio}, \link{plot_GrowthCurve}, \linkS4class{RLum.Analysis},
\linkS4class{RLum.Results}, \link{get_RLum}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
, RLum Developer Team}
\keyword{datagen}
\keyword{plot}
