% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matmul.R
\name{matmult}
\alias{matmult}
\alias{\%*\%,matrix,CsparseMatrix-method}
\alias{\%*\%,float32,CsparseMatrix-method}
\alias{tcrossprod,matrix,RsparseMatrix-method}
\alias{tcrossprod,float32,RsparseMatrix-method}
\alias{crossprod,matrix,CsparseMatrix-method}
\alias{crossprod,float32,CsparseMatrix-method}
\alias{tcrossprod,RsparseMatrix,matrix-method}
\alias{\%*\%,RsparseMatrix,matrix-method}
\alias{\%*\%,RsparseMatrix,float32-method}
\alias{tcrossprod,RsparseMatrix,float32-method}
\alias{\%*\%,RsparseMatrix,numeric-method}
\alias{\%*\%,RsparseMatrix,logical-method}
\alias{\%*\%,RsparseMatrix,integer-method}
\alias{\%*\%,RsparseMatrix,sparseVector-method}
\title{Multithreaded Sparse-Dense Matrix and Vector Multiplications}
\usage{
\S4method{\%*\%}{matrix,CsparseMatrix}(x, y)

\S4method{\%*\%}{float32,CsparseMatrix}(x, y)

\S4method{tcrossprod}{matrix,RsparseMatrix}(x, y)

\S4method{tcrossprod}{float32,RsparseMatrix}(x, y)

\S4method{crossprod}{matrix,CsparseMatrix}(x, y)

\S4method{crossprod}{float32,CsparseMatrix}(x, y)

\S4method{tcrossprod}{RsparseMatrix,matrix}(x, y)

\S4method{\%*\%}{RsparseMatrix,matrix}(x, y)

\S4method{\%*\%}{RsparseMatrix,float32}(x, y)

\S4method{tcrossprod}{RsparseMatrix,float32}(x, y)

\S4method{\%*\%}{RsparseMatrix,numeric}(x, y)

\S4method{\%*\%}{RsparseMatrix,logical}(x, y)

\S4method{\%*\%}{RsparseMatrix,integer}(x, y)

\S4method{\%*\%}{RsparseMatrix,sparseVector}(x, y)
}
\arguments{
\item{x, y}{dense (\code{matrix} / \code{float32})
and sparse (\code{RsparseMatrix} / \code{CsparseMatrix}) matrices or vectors
(\code{sparseVector}, \code{numeric}, \code{integer}, \code{logical}).}
}
\value{
A dense \code{matrix} object in most cases, with some exceptions which might
come in sparse format (see the 'Details' section).
}
\description{
Multithreaded <matrix, matrix> multiplications
(`\%*\%`, `crossprod`, and `tcrossprod`)
and <matrix, vector> multiplications (`\%*\%`),
for <sparse, dense> matrix combinations and <sparse, vector> combinations
(See signatures for supported combinations).

Objects from the `float` package are also supported for some combinations.
}
\details{
Will try to use the maximum available number of threads for the computations
when appropriate. The number of threads can be controlled through the package options
(e.g. `options("MatrixExtra.nthreads" = 1)` - see \link{MatrixExtra-options}) and will
be set to 1 after running \link{restore_old_matrix_behavior}.

Be aware that sparse-dense matrix multiplications might suffer from reduced
numerical precision, especially when using objects of type `float32`
(from the `float` package).

Internally, these functions use BLAS level-1 routines, so their speed might depend on
the BLAS backend being used (e.g. MKL, OpenBLAS) - that means: they might be quite slow
on a default install of R for Windows (see
\href{https://github.com/david-cortes/R-openblas-in-windows}{this link} for
a tutorial about getting OpenBLAS in R for Windows).

Doing computations in float32 precision depends on the package
\href{https://cran.r-project.org/package=float}{float}, and as such comes
with some caveats:\itemize{
\item On Windows, if installing `float` from CRAN, it will use very unoptimized
routines which will likely result in a slowdown compared to using regular
double (numeric) type. Getting it to use an optimized BLAS library is not as
simple as substituting the Rblas DLL - see the
\href{https://github.com/wrathematics/float}{package's README} for details.
\item On macOS, it will use static linking for `float`, thus if changing the BLAS
library used by R, it will not change the float32 functions, and getting good
performance out of it might require compiling it from source with `-march=native`
flag.
}

When multiplying a sparse matrix by a sparse vector, their indices
will be sorted in-place (see \link{sort_sparse_indices}).

In order to match exactly with base R's behaviors, when passing vectors to these
operators, will assume their shape as follows:\itemize{
\item MatMult(Matrix, vector): column vector if the matrix has more than one column
or is empty, row vector if the matrix has only one column.
\item MatMult(vector, Matrix): row vector if the matrix has more than one row,
column vector if the matrix has only one row.
\item MatMul(vector, vector): LHS is a row vector, RHS is a column vector.
\item crossprod(Matrix, vector): column vector if the matrix has more than one row,
row vector if the matrix has only one row.
\item crossprod(vector, Matrix): column vector.
\item crossprod(vector, vector): column vector.
\item tcrossprod(Matrix, vector): row vector if the matrix has only one row,
column vector if the matrix has only one column, and will throw an error otherwise.
\item tcrossprod(vector, Matrix): row vector if the matrix has more than one column,
column vector if the matrix has only one column.
\item tcrossprod(vector, vector): column vector.
}

In general, the output returned by these functions will be a dense matrix from base R,
or a dense matrix from `float` when one of the inputs is also from the `float` package,
with the following exceptions:\itemize{
\item MatMult(RsparseMatrix[n,1], vector) -> `dgRMatrix`.
\item MatMult(RsparseMatrix[n,1], sparseVector) -> `dgCMatrix`.
\item MatMult(float32[n], CsparseMatrix[1,m]) -> `dgCMatrix`.
\item tcrossprod(float32[n], RsparseMatrix[m,1]) -> `dgCMatrix`.
}
}
\examples{
library(Matrix)
library(MatrixExtra)
### To use all available threads (default)
options("MatrixExtra.nthreads" = parallel::detectCores())
### Example will run with only 1 thread (CRAN policy)
options("MatrixExtra.nthreads" = 1)

## Generate random matrices
set.seed(1)
A <- rsparsematrix(5,4,.5)
B <- rsparsematrix(4,3,.5)

## Now multiply in some supported combinations
as.matrix(A) \%*\% as.csc.matrix(B)
as.csr.matrix(A) \%*\% as.matrix(B)
crossprod(as.matrix(B), as.csc.matrix(B))
tcrossprod(as.csr.matrix(A), as.matrix(A))

### Restore the number of threads
options("MatrixExtra.nthreads" = parallel::detectCores())
}
