// Copyright 2017 Kakao Corp. <http://www.kakaocorp.com>
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.


#include <fcntl.h>
#include <iostream>
#include <unistd.h>
#include <string>

#if (defined(_WIN32) || defined(_WIN64))

#include <windows.h>
#include <memoryapi.h>

#else // original code with mmap() and munmap(), Linux

#include <sys/mman.h>

#endif

#include <sys/stat.h>
#include <stdexcept>

#include "n2/mmap.h"

namespace n2 {

Mmap::Mmap(char const* fname) {
    Map(fname);
}
   
Mmap::~Mmap() {
    UnMap();
    if(file_handle_ != -1) {
        close(file_handle_);
        file_handle_ = -1;
    }
}

#if (defined(_WIN32) || defined(_WIN64))

#include <windows.h>
#include <memoryapi.h>

// UnmapViewOfFile (LPCVOID lpBaseAddress);
// MapViewOfFileFromApp (HANDLE hFileMappingObject, ULONG DesiredAccess, ULONG64 FileOffset, SIZE_T NumberOfBytesToMap);
// VirtualAllocFromApp(PVOID BaseAddress, SIZE_T Size, ULONG AllocationType, ULONG  Protection);

void Mmap::Map(char const* fname) {
    UnMap();
    if (fname == nullptr) throw std::runtime_error("[Error] Invalid file name received. (nullptr)");
    file_handle_ = open(fname, O_RDONLY);
    if (file_handle_ == -1) throw std::runtime_error("[Error] Failed to read file: " + std::string(fname));
    file_size_ = QueryFileSize();
    if (file_size_ <= 0) throw std::runtime_error("[Error] Memory mapping failed! (file_size==zero)");
    data_ = static_cast<char*>(VirtualAlloc(nullptr, file_size_, MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE));
    // If the function fails, the return value is NULL.
    if (data_ == NULL) throw std::runtime_error("[Error] Memory mapping failed!");
}

void Mmap::UnMap() {
    if (data_ != nullptr) {
        int ret = UnmapViewOfFile(const_cast<char*>(data_));
        // If the function succeeds, the return value is nonzero.
        // If the function fails, the return value is zero.
        if (ret == 0) throw std::runtime_error("[Error] Memory unmapping failed!");
    }
    data_ = nullptr;
    file_size_ = 0;
    if(file_handle_ != -1) {
        close(file_handle_);
        file_handle_ = -1;
    }    
}


#else // original code with mmap() and munmap(), Linux

// /* Deallocate any mapping for the region starting at ADDR and extending LEN
// bytes.  Returns 0 if successful, -1 for errors (and sets errno).  */
// extern int munmap (void *__addr, size_t __len) __THROW;

// /* Return value of `mmap' in case of an error.  */
// #define MAP_FAILED        ((void *) -1)

// extern void *mmap (void *__addr, size_t __len, int __prot,
//                   int __flags, int __fd, __off_t __offset) __THROW;

// void *mmap(void *addr, size_t len, int prot, int flags,
//       int fildes, off_t off);

#include <sys/mman.h>


void Mmap::Map(char const* fname) {
    UnMap();
    if (fname == nullptr) throw std::runtime_error("[Error] Invalid file name received. (nullptr)");
    file_handle_ = open(fname, O_RDONLY);
    if (file_handle_ == -1) throw std::runtime_error("[Error] Failed to read file: " + std::string(fname));
    file_size_ = QueryFileSize();
    if (file_size_ <= 0) throw std::runtime_error("[Error] Memory mapping failed! (file_size==zero)");
    data_ = static_cast<char*>(mmap(0, file_size_, PROT_READ, MAP_SHARED, file_handle_, 0));
    if (data_ == MAP_FAILED) throw std::runtime_error("[Error] Memory mapping failed!");
}

void Mmap::UnMap() {
    if (data_ != nullptr) {
        int ret = munmap(const_cast<char*>(data_), file_size_);
        if (ret != 0) throw std::runtime_error("[Error] Memory unmapping failed!");
    }
    data_ = nullptr;
    file_size_ = 0;
    if(file_handle_ != -1) {
        close(file_handle_);
        file_handle_ = -1;
    }    
}

#endif 

size_t Mmap::QueryFileSize() const {
    struct stat sbuf;
    if (fstat(file_handle_, &sbuf) == -1) {
        return 0;
    } else {
        return (size_t)sbuf.st_size;
    }
}

}
