#' Intersects objects of the PharmacoSet class, subsetting them to the common
#' drugs and/or cell lines as selected by the user.
#' 
#' Given a list of PharmacoSets, the function will find the common drugs,
#' and/or cell lines, and return PharmacoSets that contain data only pertaining 
#' to the common drugs, and/or cell lines. The mapping between dataset 
#' drug and cell names is done either using annotations found in the 
#' PharmacoSet object's internal curation slot, or according to matching tables 
#' provided by the user
#' 
#' @examples 
#' data(GDSCsmall)
#' data(CCLEsmall)
#' common <- intersectPSet(list('GDSC'=GDSCsmall,
#'   'CCLE'=CCLEsmall), intersectOn = c("drugs", "cell.lines"))
#' common$CGP
#' common$CCLE
#' 
#' 
#' @param pSets [list] a list of PharmacoSet objects, of which the function
#'   should find the intersection
#' @param intersectOn [character] which identifiers to intersect on, 
#'   drugs or cell lines
#' @param drugs a vector of common drugs between pSets.
#' In case user is intersted on getting intersection on certain drugs,
#' they can provide their list of drugs.
#' @param cells a vector of common cell lines between pSets.
#' In case user is intersted on getting intersection on certain cell lines,
#' they can provide their list of cell lines
#' @param strictIntersect [boolean] Should the intersection keep only the drugs 
#'   and cell lines that have been tested on together?
#' @param verbose [boolean] Should the function announce its key steps?
#' @return [list] a list of pSets, contatining only the intersection
#' @export
#' 
intersectPSet <- function (pSets, intersectOn=c("drugs", "cell.lines", "concentrations"), cells, drugs, strictIntersect=FALSE, verbose=TRUE) {
  
  if (verbose) {
    message("Intersecting large PSets may take a long time ...")
  }
  ## TODO: Fix the strict intersection!!!!!!
  if (length(pSets) == 1) {
    return(pSets) 
  }
  if (length(pSets) > 1) {
    
    if ("drugs" %in% intersectOn){
      common.drugs <- intersectList(lapply(pSets, function(x){return(drugNames(x))}))
      if(!missing(drugs)) {
        common.drugs <- intersect(common.drugs, drugs)
      }
      if (length(common.drugs) == 0) {
        stop("No drugs is in common between pSets!")
      }
    }
    if ("cell.lines" %in% intersectOn){
      common.cells <- intersectList(lapply(pSets, function(x){return(cellNames(x))}))
      if(!missing(cells)) {
        common.cells <- intersect(common.cells, cells)
      }
      if (length(common.cells) == 0) {
        stop("No cell lines is in common between pSets!")
      }
    } 
    if (("drugs" %in% intersectOn) & ("cell.lines" %in% intersectOn)) {
      common.exps <- intersectList(lapply(pSets, function (x){
        if ("cellid" %in% colnames(x@sensitivity$info) & "drugid" %in% colnames(x@sensitivity$info)) {
          paste(x@sensitivity$info$cellid, x@sensitivity$info$drugid, sep = "_")
        } else { NULL }
      }))
      expMatch <- data.frame(lapply(pSets, 
      function (x, common.exps){
      if ("cellid" %in% colnames(x@sensitivity$info) & "drugid" %in% colnames(x@sensitivity$info)){
      return(rownames(x@sensitivity$info)[which(paste(x@sensitivity$info$cellid, x@sensitivity$info$drugid, sep = "_") %in% common.exps)])
      }else { NULL }
      }, common.exps=common.exps))
      expMatch <- as.matrix(expMatch)
      rownames(expMatch) <- common.exps
    }
    if (("drugs" %in% intersectOn) & ("cell.lines" %in% intersectOn) & ("concentrations" %in% intersectOn)) {
      pSets <- .calculateSensitivitiesStar(pSets, exps=expMatch, cap=100)
    }
    molecular.types  <- NULL
    for (pSet in pSets) {
      for (eSet in pSet@molecularProfiles) {
        molecular.types <- union(molecular.types, ifelse(length(grep("rna", Biobase::annotation(eSet)) > 0), "rna", Biobase::annotation(eSet)))
      }
    }
    common.molecular.cells <- list()
    for (molecular.type in molecular.types) {
      common.molecular.cells[[molecular.type]] <- intersectList(lapply(pSets, 
        function (pSet) { return(unionList(sapply(pSet@molecularProfiles, 
          function(eSet) {
            if (length(grep(molecular.type, Biobase::annotation(eSet))) > 0) {
              intersect(Biobase::pData(eSet)$cellid, common.cells)
          }
        })))
      }))
    }
    
    for(i in 1:length(pSets)){
      pSets[[i]] <- subsetTo(pSet=pSets[[i]], drugs=common.drugs, cells=common.cells, exps=expMatch, molecular.data.cells=common.molecular.cells)
    }
    return(pSets)
  }
}

