#' Plots a Circos diagram from an adjacency matrix. Uses the Circlize chordDiagram function. Is called by VDJ_clonotype_clusters_circos(), VDJ_alpha_beta_Vgene_circos() and VDJ_VJ_usage_circos() functions or works on its own when supplied with an adjacency matrix.
#' @param Adj_matrix Adjacency matrix to be plotted. Rownames and Colnames correspond to genes to be matched and entries determine the weight of the connection between the genes (eg. number of clonotypes expressing these two genes).
#' @param group Named list of genes, with list elements corresponding to group-names, and element names being the gene-names. Is generated by VDJ_VJ_usage and VDJ_alpha_beta_Vgene_circos.
#' @param grid.col Named list of genes, with list elements corresponding to color and element names being gene-names. If not supplied it is generated randomly within the function. Is also generated by VDJ_VJ_usage and VDJ_alpha_beta_Vgene_circos.
#' @param label.threshold Genes are only labeled if the count is larger then the label.threshold. By default all label.threshold = 0 (all genes are labeled).
#' @param axis Option to choose the count axis for each gene. "default", "percent" or "max"  possible. Default: "max".
#' @param c.count Show clonotype or cell count on Circos plot.
#' @return Returns the Circos plot from input of other functions. Do not run as standalone
#' @export
#' @examples
#' \dontrun{
#' VDJ_circos() #Do not run as standalone. Called by other circos functions
#'}

VDJ_circos <- function(Adj_matrix, group, grid.col, label.threshold, axis, c.count){

  CELL_META <- NULL
  df <- NULL
  df1 <- NULL


  circlize::circos.clear()
  circlize::circos.par(points.overflow.warning=FALSE)
  if(missing(axis)){axis <- "max"}
  if(missing(label.threshold)){label.threshold <- 0}
  if(missing(grid.col)){grid.col <- stats::setNames(grDevices::rainbow(length(union(rownames(Adj_matrix), colnames(Adj_matrix)))),sample(union(rownames(Adj_matrix), colnames(Adj_matrix))))}
  if(missing(c.count)){c.count <- T}

  df <- reshape2::melt(Adj_matrix)

  df1 <- data.frame(Var = c(rownames(Adj_matrix), colnames(Adj_matrix)), value= c(rowSums(Adj_matrix), colSums(Adj_matrix)))
  df1 <- df1[order(df1$value),] #does not really work if group argument is used at the same time...

  circlize::chordDiagram(df,
               link.sort = T,
               link.decreasing = T,
               order = df1$Var,
               group = group,
               annotationTrack = c("grid"),
               preAllocateTracks = list(track.height = circlize::mm_h(5), track.margin = c(circlize::mm_h(5), 0)),
               grid.col = grid.col)
  #circos.info()

  #Add labels to circos plot(
  circlize::circos.track(track.index = 1, panel.fun = function(x, y) {
    if(circlize::get.cell.meta.data("xrange")>label.threshold){
      circlize::circos.text(CELL_META$xcenter, CELL_META$ylim[1], CELL_META$sector.index,
                facing = "clockwise", niceFacing = TRUE, adj = c(0, 0.6), cex=0.5)
    }else{
      circlize::circos.text(CELL_META$xcenter, CELL_META$ylim[1], CELL_META$sector.index,
                  facing = "clockwise", niceFacing = TRUE, adj = c(0, 0.6), cex=0.5, col = "white")
    }
  }, bg.border = NA) # here set bg.border to NA is important

  #Add axis
  if(axis=="default"){
    for(si in circlize::get.all.sector.index()) {
      circlize::circos.axis(h = "top", labels.cex = 0.3, sector.index = si, track.index = 2)
    }
  }
  if(axis=="percent"){
    circlize::circos.track(track.index = 1, panel.fun = function(x, y) {
      xlim = circlize::get.cell.meta.data("xlim")
      ylim = circlize::get.cell.meta.data("ylim")
      sector.name = circlize::get.cell.meta.data("sector.index")
      xplot = circlize::get.cell.meta.data("xplot")

      circlize::circos.lines(xlim, c(min(ylim), min(ylim)), lty = 3) # dotted line
      by = ifelse(abs(xplot[2] - xplot[1]) > 30, 0.2, 0.5)
      for(p in seq(by, 1, by = by)) {
        circlize::circos.text(p*(xlim[2] - xlim[1]) + xlim[1], min(ylim) + 0.1,
                    paste0(p*100, "%"), cex = 0.3, adj = c(0.5, 0), niceFacing = TRUE)
      }
    }, bg.border = NA)
  }
  if(axis=="max"){
    for(si in circlize::get.all.sector.index()) {
      circlize::circos.axis(h = "top", labels =  FALSE, sector.index = si, track.index = 2)
    }
    circlize::circos.track(track.index = 2, panel.fun = function(x, y) {
      xlim = circlize::get.cell.meta.data("xlim")
      ylim = circlize::get.cell.meta.data("ylim")
      sector.name = circlize::get.cell.meta.data("sector.index")
      xrange = circlize::get.cell.meta.data("xrange")

      #circos.lines(xlim, c(min(ylim), min(ylim)), lty = 1)
      if(c.count){
        circlize::circos.text(mean(xlim), max(ylim) + 0.5,
                    round(xrange,0), cex = 0.5, adj = c(0.5, 0), niceFacing = TRUE)
      }
    }, bg.border = NA)
  }

  return(plot)
}

