# This function loads R packages required by RSurvey. If a required
# package is unavailable on the local computer an attempt is made to
# acquire the package from CRAN using an existing network connection.

LoadPackages <- function() {

  ## Additional functions (subroutines)

  # Install missing packages from CRAN mirror
  InstallPackages <- function() {
    tkconfigure(tt, cursor="watch")
    idx <- which(cran.mirrors$Name %in% as.character(tclvalue(repo.var)))
    repo <- cran.mirrors$URL[idx]
    contriburl <- contrib.url(repos=repo, type=getOption("pkgType"))
    cran.pkgs <- available.packages(contriburl)
    idxs <- which(missing.pkgs %in% cran.pkgs)
    available.pkgs <- missing.pkgs[idxs]
    unavailable.pkgs <- missing.pkgs[!idxs]
    if (length(unavailable.pkgs) > 0) {
      msg <- paste0("The following package(s) are unavailable on this ",
                    "CRAN mirror:\n\n",
                    paste(paste0("\'", unavailable.pkgs, "\'"), collapse=", "),
                    "\n\nWould you like to try a different CRAN mirror?")
      ans <- tkmessageBox(icon="question", message=msg, title="CRAN",
                          type="yesno", parent=tt)
      if (tolower(substr(ans, 1, 1)) == "y") {
        tkconfigure(tt, cursor="arrow")
        return()
      }
    }
    if (length(available.pkgs) > 0) {
      tclServiceMode(FALSE)
      install.packages(available.pkgs, repos=repo, quiet=TRUE)
      tclServiceMode(TRUE)
    }
    tclvalue(tt.done.var) <- 1
  }

  ## Main program

  if (!require("tcltk"))
    stop("tcltk required")

  # Establish required and suggested packages
  require.pkgs <- c("sp", "rgeos", "rgl", "MBA")
  suggest.pkgs <- c("rgdal", "tripack", "colorspace", "dichromat")
  pkgs <- c(require.pkgs, suggest.pkgs)

  # Account for missing packages

  is.pkg.missing <- !pkgs %in% .packages(all.available=TRUE)
  if (any(is.pkg.missing)) {
    missing.pkgs <- pkgs[is.pkg.missing]
    cran.mirrors <- getCRANmirrors(all=FALSE, local.only=FALSE)
    default.repo <- getOption("repos")
    idx <- which(sub("/$", "", cran.mirrors$URL) %in%
                 sub("/$", "", default.repo["CRAN"]))
    idx <- if (length(idx) > 0) idx[1] else 1
    repo.var <- tclVar(cran.mirrors$Name[idx])
    rlogo.var <- tclVar()
    tt.done.var <- tclVar(0)

    # Open GUI
    tclServiceMode(FALSE)
    tt <- tktoplevel()
    tktitle(tt) <- "Missing Packages"
    tkwm.resizable(tt, 0, 0)

    # Frame 0, yes and no buttons
    frame0 <- tkframe(tt, relief="flat")
    frame0.but.2 <- ttkbutton(frame0, width=12, text="Yes", default="active",
                              command=InstallPackages)
    frame0.but.3 <- ttkbutton(frame0, width=12, text="No",
                              command=function() tclvalue(tt.done.var) <- 1)
    tkgrid("x", frame0.but.2, frame0.but.3, sticky="se", pady=10)
    tkgrid.columnconfigure(frame0, 0, weight=1)
    tkgrid.configure(frame0.but.2, padx=c(10, 2))
    tkgrid.configure(frame0.but.3, padx=c(2, 10))
    tkpack(frame0, fill="x", side="bottom", anchor="e")

    # Frame 1, message and mirror selection
    frame1 <- tkframe(tt, relief="flat", background="white")
    if ("RSurvey" %in% .packages(all.available=TRUE))
      f <- system.file("images/rlogo.gif", package="RSurvey")
    else
      f <- file.path(getwd(), "inst", "images", "rlogo.gif")
    tkimage.create("photo", rlogo.var, format="GIF", file=f)
    txt <- paste("The following package(s) used by RSurvey are missing:\n",
                 paste(paste0("\'", missing.pkgs, "\'"), collapse=", "), "",
                 "Some features will not be available without these packages.",
                 "Install these packages from CRAN?", sep="\n")
    frame1.lab.1.1 <- ttklabel(frame1, image=rlogo.var, background="white")
    frame1.lab.1.2 <- ttklabel(frame1, text=txt, justify="left",
                               background="white")
    frame1.lab.2.2 <- ttklabel(frame1, text="Set CRAN mirror",
                               justify="left", background="white")
    frame1.box.2.3 <- ttkcombobox(frame1, textvariable=repo.var,
                                  values=cran.mirrors$Name, state="readonly")
    tkgrid(frame1.lab.1.1, frame1.lab.1.2, "x", pady=c(30, 20))
    tkgrid("x", frame1.lab.2.2, frame1.box.2.3, pady=c( 0, 30))
    tkgrid.configure(frame1.lab.1.1, padx=c(40, 20), sticky="n")
    tkgrid.configure(frame1.lab.1.2, padx=c( 0, 40), columnspan=2)
    tkgrid.configure(frame1.lab.2.2, padx=c( 0,  4), sticky="e")
    tkgrid.configure(frame1.box.2.3, padx=c( 0, 40), sticky="w")
    tkpack(frame1)

    # Binds events
    tclServiceMode(TRUE)
    tkbind(tt, "<Return>", InstallPackages)
    tkbind(tt, "<Key-space>", InstallPackages)

    # GUI control
    tkfocus(tt)
    tkgrab(tt)
    tkwait.variable(tt.done.var)
    tclServiceMode(FALSE)
    tkgrab.release(tt)
    tkdestroy(tt)
    tclServiceMode(TRUE)
  }

  # Load packages into current session
  for (pkg in pkgs) {
    is.pkg.suggested <- pkg %in% suggest.pkgs
    is.pkg.loaded <- suppressWarnings(require(pkg, character.only=TRUE,
                                      warn.conflicts=!is.pkg.suggested,
                                      quietly=is.pkg.suggested))
    if (is.pkg.loaded)
      next
    if (is.pkg.suggested)
      warning(paste("unable to load suggested package:", pkg))
    else
      stop(paste("unable to load required package:", pkg))
  }

  # Warn if Tktable is unavailable
  tcl.pkg <- tryCatch(tcl("package", "require", "Tktable"), error=identity)
  if (inherits(tcl.pkg, "error")) {
    msg <- paste("Tcl package Tktable is missing and is strongly recommended",
                 "for full functionality of RSurvey.\n\n ",
                 "http://tktable.sourceforge.net")
    tkmessageBox(icon="warning", message=msg, title="Missing Tktable",
                 type="ok")
  }
}
