% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/util.R
\name{genSeeds}
\alias{genSeeds}
\alias{gen_seeds}
\title{Generate random seeds}
\usage{
genSeeds(design = 1L, iseed = NULL, arrayID = NULL, old.seeds = NULL)

gen_seeds(...)
}
\arguments{
\item{design}{design matrix that requires a unique seed per condition, or
a number indicating the number of seeds to generate. Default generates one
number}

\item{iseed}{the initial \code{set.seed} number used to generate a sequence
of independent seeds according to the L'Ecuyer-CMRG (2002) method. This
is recommended whenever quality random number generation is required
across similar (if not identical) simulation jobs
(e.g., see \code{\link{runArraySimulation}}). If \code{arrayID} is not
specified then this will return a list of the associated seed for the
full \code{design}}

\item{arrayID}{(optional) single integer input corresponding to the specific
row in the \code{design} object when using the \code{iseed} input.
This is used in functions such as \code{\link{runArraySimulation}}
to pull out the specific seed rather than manage a complete list, and
is therefore more memory efficient}

\item{old.seeds}{(optional) vector or matrix of last seeds used in
previous simulations to avoid repeating the same seed on a subsequent run.
Note that this approach should be used sparingly as seeds set more frequently
are more likely to correlate, and therefore provide less optimal random
number behaviour (e.g., if performing a simulation on two runs to achieve
5000 * 2 = 10,000 replications this is likely reasonable,
but for simulations with 100 * 2 = 200 replications this is more
likely to be sub-optimal).
Length must be equal to the number of rows in \code{design}}

\item{...}{does nothing}
}
\description{
Generate seeds to be passed to \code{runSimulation}'s \code{seed} input. Values
are sampled from 1 to 2147483647, or are generated using L'Ecuyer-CMRG's (2002)
method (returning either a list if \code{arrayID} is omitted, or the specific
row value from this list if \code{arrayID} is included).
}
\examples{

# generate 1 seed (default)
genSeeds()

# generate 5 unique seeds
genSeeds(5)

# generate from nrow(design)
design <- createDesign(factorA=c(1,2,3),
                       factorB=letters[1:3])
seeds <- genSeeds(design)
seeds

# construct new seeds that are independent from original (use this sparingly)
newseeds <- genSeeds(design, old.seeds=seeds)
newseeds

# can be done in batches too
newseeds2 <- genSeeds(design, old.seeds=cbind(seeds, newseeds))
cbind(seeds, newseeds, newseeds2) # all unique

############
# generate seeds for runArraySimulation()
(iseed <- genSeeds())  # initial seed
seed_list <- genSeeds(design, iseed=iseed)
seed_list

# expand number of unique seeds given iseed (e.g., in case more replications
# are required at a later date)
seed_list_tmp <- genSeeds(nrow(design)*2, iseed=iseed)
str(seed_list_tmp) # first 9 seeds identical to seed_list

# more usefully for HPC, extract only the seed associated with an arrayID
arraySeed.15 <- genSeeds(nrow(design)*2, iseed=iseed, arrayID=15)
arraySeed.15

}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
