% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mergeRaster.R
\docType{methods}
\name{mergeRaster}
\alias{mergeRaster}
\alias{mergeRaster,list-method}
\title{Merge split raster tiles into a single raster layer.}
\usage{
mergeRaster(x)

\S4method{mergeRaster}{list}(x)
}
\arguments{
\item{x}{A list of split raster tiles (i.e., from \code{splitRaster}).}
}
\value{
A \code{RasterLayer} object.
}
\description{
Recombine the split tiles from \code{splitRaster} into a single RasterLayer.
}
\details{
\code{mergeRaster} differs from \code{merge} in how overlapping tile
regions are handled: \code{merge} retains the values of the first raster in
the list. This has the consequence of retaining the values from the buffered
region in the first tile in place of the values from the neighbouring tile.
On the otherhand, \code{mergeRaster} retains the values of the tile region,
over the values in any buffered regions. This is useful for reducing edge
effects when performing raster operations involving contagious processes.
To use the average of cell values, or do another computation, use
\code{\link[raster]{mosaic}}.
}
\examples{
require(raster)

# an example with dimensions:
# nrow = 77
# ncol = 101
# nlayers = 3
b <- brick(system.file("external/rlogo.grd", package = "raster"))
r <- b[[1]] # use first layer only
nx <- 3
ny <- 4

tmpdir <- file.path(tempdir(), "splitRaster-example")

y0 <- splitRaster(r, nx, ny, path = file.path(tmpdir, "y0")) # no buffer

# buffer: 10 pixels along both axes
y1 <- splitRaster(r, nx, ny, c(10, 10), path = file.path(tmpdir, "y1"))

# buffer: half the width and length of each tile
y2 <- splitRaster(r, nx, ny, c(0.5, 0.5), path = file.path(tmpdir, "y2"))

# parallel cropping
if (interactive()) {
  n <- pmin(parallel::detectCores(), 4) # use up to 4 cores
  beginCluster(n)
  y3 <- splitRaster(r, nx, ny, c(0.7, 0.7), path = file.path(tmpdir, "y3"))
  endCluster()
}

# the original raster:
if (interactive()) plot(r) # may require a call to `dev()` if using RStudio

# the split raster:
layout(mat = matrix(seq_len(nx*ny), ncol = nx, nrow = ny))
plotOrder <- c(4, 8, 12, 3, 7, 11, 2, 6, 10, 1, 5, 9)
if (interactive()) invisible(lapply(y0[plotOrder], plot))

# can be recombined using `raster::merge`
m0 <- do.call(merge, y0)
all.equal(m0, r) ## TRUE

m1 <- do.call(merge, y1)
all.equal(m1, r) ## TRUE

m2 <- do.call(merge, y2)
all.equal(m2, r) ## TRUE

# or recombine using SpaDES::mergeRaster
n0 <- mergeRaster(y0)
all.equal(n0, r) ## TRUE

n1 <- mergeRaster(y1)
all.equal(n1, r) ## TRUE

n2 <- mergeRaster(y2)
all.equal(n2, r) ## TRUE

unlink(tmpdir, recursive = TRUE)
}
\author{
Yong Luo and Alex Chubaty
}
\seealso{
\code{\link[raster]{merge}}, \code{\link[raster]{mosaic}}
}

