\name{tam.wle}

\alias{tam.wle}
\alias{tam.mml.wle}
\alias{tam.jml.WLE}

\title{
Weighted Likelihood Estimation and Maximum Likelihood Estimation of
Person Parameters
}

\description{
Compute the weighted likelihood estimator (Warm, 1989) 
for objects of classes \code{tam}, \code{tam.mml} and \code{tam.jml}, 
respectively.
}

\usage{
tam.wle(tamobj, ...)

tam.mml.wle(tamobj, score.resp=NULL, WLE=TRUE, adj=0.3, Msteps=20, convM=1e-04)

tam.jml.WLE(tamobj, resp, resp.ind, A, B, nstud, nitems, maxK, convM, 
    PersonScores, theta, xsi, Msteps, WLE=FALSE)
}

\arguments{
  \item{tamobj}{
     An object generated by \code{tam.mml} or \code{tam.jml}
}
\item{score.resp}{An optional data frame for which WLEs or MLEs
	should be calculated. In case of the default \code{NULL},
	\code{resp} from \code{tamobj} (i.e. \code{tamobj$resp}) is chosen. 
	Note that items in \code{score.resp} must be the same (and in the
	same order) as in \code{tamobj$resp}.
}
  \item{WLE}{
     A logical indicating whether the weighted likelihood estimate
     (WLE, \code{WLE=TRUE}) or the maximum likelihood estimate (MLE, \code{WLE=FALSE})
     should be used.
}
  \item{adj}{
     Adjustment in WLE estimation for extreme scores (i.e. all or none
     items were correctly solved)
}
  \item{Msteps}{
     Maximum number of iterations
}
  \item{convM}{
     Convergence criterion
  }
\item{resp}{Data frame with item responses (only for \code{tam.jml.WLE})
}
\item{resp.ind}{Data frame with response indicators (only for \code{tam.jml.WLE})
}
\item{A}{Design matrix \eqn{A} (applies only to \code{tam.jml.WLE})
}
\item{B}{Design matrix \eqn{B} (applies only to \code{tam.jml.WLE})
}  
\item{nstud}{Number of persons (applies only to \code{tam.jml.WLE})
}
\item{nitems}{Number of items (applies only to \code{tam.jml.WLE})
}
\item{maxK}{Maximum item score (applies only to \code{tam.jml.WLE})
}
\item{PersonScores}{A vector containing the sufficient statistics for the 
	person parameters (applies only to \code{tam.jml.WLE})
}
\item{theta}{Initial \eqn{\theta} estimate (applies only to \code{tam.jml.WLE})
}
\item{xsi}{Parameter vector \eqn{\xi} (applies only to \code{tam.jml.WLE})
}
\item{\dots}{Further arguments to be passed}
}

%\details{
%%  ~~ If necessary, more details than the description above ~~
%}

\value{
For \code{tam.wle.mml}, it is a data frame with following columns:
\item{pid}{Person identifier}
\item{PersonScores}{Score of each person}
\item{PersonMax}{Maximum score of each person}
\item{theta}{Weighted likelihood estimate (WLE) or MLE}
\item{error}{Standard error of the WLE or MLE}
\item{WLE.rel}{WLE reliability (same value for all persons)} \cr

For \code{tam.jml.WLE}, it is a list with following entries:
\item{theta}{Weighted likelihood estimate (WLE) or MLE}
\item{errorWLE}{Standard error of the WLE or MLE}
\item{meanChangeWLE}{Mean change between updated and previous ability
    estimates from last iteration}
}

\references{
Warm, T. A. (1989). Weighted likelihood estimation of ability in item 
response theory. \emph{Psychometrika}, \bold{54}, 427-450.
}

%\author{
%%  ~~who you are~~
%}

%\note{
%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

%\seealso{
%% ~~objects to See Also as \code{\link{help}}, ~~~
%}

\examples{
#############################################################################
# EXAMPLE 1: 1PL model, sim.rasch data
#############################################################################
data(sim.rasch)
# estimate Rasch model
mod1 <- tam.mml(resp=sim.rasch) 
# WLE estimation
wle1 <- tam.wle( mod1 )
## ----
## WLE Reliability = 0.894 

# scoring for a different dataset containing same items (first 10 persons in 
#  sim.rasch)
wle2 <- tam.wle( mod1 , score.resp=sim.rasch[1:10,])

\dontrun{
#############################################################################
# EXAMPLE 2: 3-dimensional Rasch model | data.read from sirt package
#############################################################################

data(data.read , package="sirt")
# define Q-matrix
Q <- matrix(0,12,3)
Q[ cbind( 1:12 , rep(1:3,each=4) ) ] <- 1
# redefine data: create some missings for first three cases
resp <- data.read
resp[1:2 , 5:12] <- NA
resp[3,1:4] <- NA
##   > head(resp)
##      A1 A2 A3 A4 B1 B2 B3 B4 C1 C2 C3 C4
##   2   1  1  1  1 NA NA NA NA NA NA NA NA
##   22  1  1  0  0 NA NA NA NA NA NA NA NA
##   23 NA NA NA NA  1  0  1  1  1  1  1  1
##   41  1  1  1  1  1  1  1  1  1  1  1  1
##   43  1  0  0  1  0  0  1  1  1  0  1  0
##   63  1  1  0  0  1  0  1  1  1  1  1  1

# estimate 3-dimensional Rasch model
mod <- tam.mml( resp=resp , Q=Q , control=list(snodes=1000,maxiter=50) )
summary(mod)

# WLE estimates
wmod <- tam.wle(mod , Msteps=3)
summary(mod)
##   head(round(wmod,2))
##      pid N.items PersonScores.Dim01 PersonScores.Dim02 PersonScores.Dim03
##   2    1       4                3.7                0.3                0.3
##   22   2       4                2.0                0.3                0.3
##   23   3       8                0.3                3.0                3.7
##   41   4      12                3.7                3.7                3.7
##   43   5      12                2.0                2.0                2.0
##   63   6      12                2.0                3.0                3.7
##      PersonMax.Dim01 PersonMax.Dim02 PersonMax.Dim03 theta.Dim01 theta.Dim02
##   2              4.0             0.6             0.6        1.06          NA
##   22             4.0             0.6             0.6       -0.96          NA
##   23             0.6             4.0             4.0          NA       -0.07
##   41             4.0             4.0             4.0        1.06        0.82
##   43             4.0             4.0             4.0       -0.96       -1.11
##   63             4.0             4.0             4.0       -0.96       -0.07
##      theta.Dim03 error.Dim01 error.Dim02 error.Dim03 WLE.rel.Dim01
##   2           NA        1.50          NA          NA          -0.1
##   22          NA        1.11          NA          NA          -0.1
##   23        0.25          NA        1.17        1.92          -0.1
##   41        0.25        1.50        1.48        1.92          -0.1
##   43       -1.93        1.11        1.10        1.14          -0.1

## (1) Note that estimated WLE reliabilities are not trustworthy in this example.
## (2) If cases do not possess any observations on dimensions, then WLEs
##     and their corresponding standard errors are set to NA.
}
}

% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{WLE}
\keyword{MLE}
\keyword{Person parameter estimation}
%% \keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
