#' Non-overlapping buffers
#'
#' Takes a set of tiles and removes all overlapping buffer areas, while conserving buffer areas along
#' the edge of the tile set.
#'
#' When processing a tiled dataset, using buffered tiles can help remove the edge effects along the
#' individual tile borders. However, overlapping buffers generally need to be removed when recombining
#' a series of tiles back into a single raster. Although this can be accomplished by using the unbuffered
#' tile extent, this will also remove the buffered areas along the edge of the tile set. Once these
#' unbuffered tiles are reassembled, the resulting raster will then be smaller than the original
#' dataset before it was tiled.
#'
#' This may not be a desirable result. The \code{NonoverlappingBuffers} function will produce a set of
#' polygons that correspond to the tile extents that conserve buffers only where they do not overlap onto
#' neighbouring tiles (i.e.: along the edge of the tile set). These polygons are useful for cropping out
#' overlapping areas from buffered tiles in order to reassemble the tiles into a single raster.
#'
#' Typically, both inputs for this function would be generated by the \code{\link{TileScheme}} function.
#'
#' Both \code{buffPolys} and \code{unbuffPolys} should represent the same set of tiles. Both require
#' 'col' and 'row' data columns indicating each cell's column and row number.
#'
#' @param buffPolys \link[sp]{SpatialPolygonsDataFrame}. A set of buffered tiles.
#' @param unbuffPolys \link[sp]{SpatialPolygonsDataFrame}. A set of unbuffered tiles.
#'
#' @return A \link[sp]{SpatialPolygonsDataFrame} corresponding to the tiles input in \code{buffPolys}
#' and \code{unbuffPolys} with all overlapping buffer areas removed.


NonoverlappingBuffers <- function(buffPolys, unbuffPolys){

  if(any(buffPolys[["col"]] != unbuffPolys[["col"]])){stop("Number of columns do not match")}
  if(any(buffPolys[["row"]] != unbuffPolys[["row"]])){stop("Number of rows do not match")}

  tilesColRow <- buffPolys@data[,c("col", "row")]

  neibrowcol <- expand.grid(c(-1,0,1), c(-1,0,1))[-5,]
  row.names(neibrowcol) <- c("topleft", "top", "topright", "left", "right", "bottomleft", "bottom", "bottomright")
  colnames(neibrowcol) <- c("col", "row")

  neibgrid <- data.frame(corner = c("topleft", "topright", "bottomright", "bottomleft"),
                         hor = c("left", "right", "right", "left"),
                         vert = c("top", "top", "bottom", "bottom"),
                         stringsAsFactors = FALSE)

  nbuffPolys <- sp::SpatialPolygons(lapply(1:nrow(tilesColRow), function(tileNum){

    tileColRow <- as.numeric(tilesColRow[tileNum,])
    buffPoly <- buffPolys[tileNum,]
    unbuffPoly <- unbuffPolys[tileNum,]

    # Get row/col of potential neighbours
    neibrowcol.tile <- as.data.frame(t(apply(neibrowcol, 1, function(r) r + tileColRow)))

    # Determine which neighbours exist
    neibrowcol.tile$exists <- utils::tail(duplicated(rbind(tilesColRow, neibrowcol.tile)),8)
    neibgrid.tile <- cbind(neibgrid, data.frame(cornerExists = neibrowcol.tile[neibgrid$corner,"exists"],
                                                horExists = neibrowcol.tile[neibgrid$hor,"exists"],
                                                vertExists = neibrowcol.tile[neibgrid$vert,"exists"]))
    # Get positions of tile sides
    dims.tile <- data.frame(buff = raster::extent(buffPoly)[],
                            unbuff = raster::extent(unbuffPoly)[],
                            diff =  raster::extent(buffPoly)[] - raster::extent(unbuffPoly)[],
                            row.names = c("left", "right", "bottom", "top"))

    polyPts <- do.call(rbind, lapply(1:4, function(x){

      # Get corner
      crn <- neibgrid.tile[x,]

      dims.crn <- dims.tile[c(crn[,"hor"], crn[,"vert"]),]

      if(crn[,"horExists"] & crn[,"vertExists"]){

        # Straight corner
        return(dims.crn[cbind(1:2, as.numeric(c(crn[,"horExists"], crn[,"vertExists"])) + 1)])

      }else{

        if(crn[,"cornerExists"]){

          if(crn[,"horExists"]){
            horPt <- dims.crn[cbind(1:2, c(2,2))]
          }else{
            horPt <- dims.crn[cbind(1:2, c(1,2))] - c(0,dims.crn[crn[,"vert"], "diff"])
          }
          if(crn[,"vertExists"]){
            vertPt <- dims.crn[cbind(1:2, c(2,2))]
          }else{
            vertPt <- dims.crn[cbind(1:2, c(2,1))] - c(dims.crn[crn[,"hor"], "diff"],0)
          }
          if(crn[,"corner"] %in% c("topleft", "bottomright")){
            return(rbind(horPt, vertPt))
          }else{
            return(rbind(vertPt, horPt))
          }

        }else{

          # Straight corner
          return(dims.crn[cbind(1:2, as.numeric(c(crn[,"horExists"], crn[,"vertExists"])) + 1)])
        }
      }
    }))

    row.names(polyPts) <- 1:nrow(polyPts)
    return(sp::Polygons(list(sp::Polygon(polyPts)), ID = as.character(row.names(unbuffPoly@data))))
  }))
  nbuffPolys <- sp::SpatialPolygonsDataFrame(nbuffPolys, unbuffPolys@data)
  raster::crs(nbuffPolys) <- raster::crs(buffPolys)

  return(nbuffPolys)}
