% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/search_templates.R
\name{search_templates}
\alias{search_templates}
\title{Search acoustic templates on test sound files}
\usage{
search_templates(X, template.rows, test.files, path = NULL, pb = TRUE, ...)
}
\arguments{
\item{X}{Object of class 'data.frame', 'selection_table' or 'extended_selection_table' (the last 2 classes are created by the function \code{\link[warbleR]{selection_table}} from the warbleR package). Must contain the following columns: 1) "sound.files": name of the .wav files, 2) "selec": unique selection identifier (within a sound file), 3) "start": start time and 4) "end": end time of selections. Columns for 'top.freq', 'bottom.freq' and 'channel' are optional. Required.}

\item{template.rows}{Numeric vector with the index of the rows from 'X' to be used as templates. If only 1 is supplied the same template will be run over all 'test.files'. Otherwise, 'template.rows' must be the same length as 'test.files'. Required.}

\item{test.files}{Character vector of length 1 with the name(s) of the test (re-recorded) file(s) in which to search for the template(s) (see argument 'template.rows').}

\item{path}{Character string containing the directory path where test (re-recorded) sound files are found.}

\item{pb}{Logical argument to control if progress bar is shown. Default is \code{TRUE}.}

\item{...}{Additional arguments to be passed to \code{\link[warbleR]{cross_correlation}} for setting cross-correlation parameters (e.g. 'wl', 'ovlp', etc).}
}
\value{
A data frame with the time, start, end, test file names, template name, maximum cross-correlation score and the time where it was detected.
}
\description{
\code{search_templates} searches acoustic templates on test (re-recorded) sound files.
}
\details{
The function takes a master sound file's reference data ('X') and finds the position of acoustics templates included as selections in 'X'. This is used to align signals found in re-recorded sound files according to a master sound file referenced in 'X'. Take a look at the package vignette for information on how to:: incorporate this into a sound degradation analysis workflow.
}
\examples{
\dontrun{
# use a temporary directory
td <- tempdir()  

# load example data from warbleR
data(list = c("Phae.long1", "Phae.long2", "Phae.long3", "Phae.long4", 
"lbh_selec_table"))

# save sound files to temporary folder
writeWave(Phae.long1, file.path(td, "Phae.long1.wav"))
writeWave(Phae.long2, file.path(td, "Phae.long2.wav"))
writeWave(Phae.long3, file.path(td, "Phae.long3.wav"))
writeWave(Phae.long4, file.path(td, "Phae.long4.wav"))

# make an extended selection table
est <- selection_table(X = lbh_selec_table, extended = TRUE, confirm.extended = FALSE, 
path = td, pb = FALSE)

# create master sound file
master.sf <- master_sound_file(X = est, file.name = "example_master", 
dest.path = td, gap.duration = 0.3)

# read master
exmp.master <- readWave(file.path(td, "example_master.wav"))

# add 1 s silence
exmp.test1 <- addsilw(wave = exmp.master, at = "start", d = 1,
output = "Wave", f = exmp.master@samp.rate)

exmp.test2 <- addsilw(wave = exmp.master, at = "start", d = 2, 
output = "Wave", f = exmp.master@samp.rate)

# create noise
ns <- noisew(f = exmp.master@samp.rate, d = duration(exmp.test2) + 1, 
output = "Wave")

# make noise exactly the same length and add noise to 2 examples
exmp.test1@left <- exmp.test1@left + (ns@left[1:length(exmp.test1@left)] * 500)
exmp.test2@left <- exmp.test2@left + (ns@left[1:length(exmp.test2@left)] * 500)

exmp.test1 <- tuneR::normalize(exmp.test1, unit = "16")
exmp.test2 <- tuneR::normalize(exmp.test2, unit = "16")

# save examples
writeWave(object = exmp.test1, filename = file.path(td, "example_test1.wav"), extensible = FALSE)
writeWave(object = exmp.test2, filename = file.path(td, "example_test2.wav"), extensible = FALSE)

# search using start marker as template
search_templates(X = master.sf, 
template.rows = which(master.sf$orig.sound.file == "start_marker"), 
test.files = c("example_test1.wav", "example_test2.wav"), path = td, pb = FALSE)

# search using end marker as template
search_templates(X = master.sf, template.rows = which(master.sf$orig.sound.file == "end_marker"), 
test.files = c("example_test1.wav", "example_test2.wav"), 
path = td, pb = FALSE)

# search using both start and end markers as template
search_templates(X = master.sf, 
template.rows = which(master.sf$orig.sound.file == "start_marker" | 
master.sf$orig.sound.file == "end_marker"), 
test.files = c("example_test1.wav", "example_test2.wav"), 
path = td, pb = FALSE)
}
}
\references{
{
Araya-Salas, M. (2020). baRulho: baRulho: quantifying habitat-induced degradation of (animal) acoustic signals in R. R package version 1.0.2
}
}
\seealso{
\code{\link{spcc_align}}; \code{\link{align_test_files}}
}
\author{
Marcelo Araya-Salas (\email{marcelo.araya@ucr.ac.cr})
}
