% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesnecformula.R
\name{bayesnecformula}
\alias{bayesnecformula}
\alias{bnf}
\title{Set up a model formula for use in
\code{\link[bayesnec:bayesnec-package]{bayesnec}}}
\usage{
bayesnecformula(formula, ...)
}
\arguments{
\item{formula}{Either a \code{\link[base]{character}} string defining an
R formula or an actual \code{\link[stats]{formula}} object. See details.}

\item{...}{Unused.}
}
\value{
An object of class \code{\link{bayesnecformula}} and
\code{\link[stats]{formula}}.
}
\description{
Set up a model formula for use in the
\code{\link[bayesnec:bayesnec-package]{bayesnec}} package allowing to
define linear and non-linear (potentially multilevel) concentration-response
models.
}
\details{
See \code{methods(class = "bayesnecformula")} for an overview of
available methods.

\bold{General formula syntax}

The \code{formula} argument accepts formulas of the following syntax:

\code{response | aterms ~ crf(x, model) + glterms}

\bold{The population-level term: \code{crf}}

\code{\link[bayesnec:bayesnec-package]{bayesnec}} uses a special internal
term called \code{crf} which sets the concentration-response equation
to be evaluated based on some \code{x} predictor. The equation itself is
defined by the argument \code{"model"}: a \code{\link[base]{character}}
vector containing a specific model, a concatenation of specific models,
or a single string defining a particular group of models
(or group of equations, see \code{\link{models}}). Internally
this argument is substituted by an actual \code{\link[brms]{brmsformula}}
which is then passed onto \code{\link[brms]{brm}} for model fitting.

\bold{Group-level terms: \code{glterms}}

The user has three options to define group-level effects in a
\code{\link{bayesnecformula}}: 1) a general "offset" group-level effect
defined by the term \code{ogl} (as in e.g. \code{ogl(group_variable)}). This
adds an additional population-level parameter \code{ogl} to the model defined
by \code{crf}, analogously to what an intercept-only group-level effect
would be in a classic linear model. 2) a group-level effect applied to all
parameters in a model at once. This is done by the special term \code{pgl},
(as in e.g. \code{pgl(group_variable)}) which comes in handy so the user
does not need to know the internal syntax and name of each parameter in the
model. 3) a more classic approach where the user can specify which
specific parameters --- NB: that requires prior knowledge on the model
structure and parameter names --- to vary according to a grouping variable
(as in e.g. \code{(bot | group_variable)}). \code{\link{bayesnecformula}}
will ignore this term should the parameter not exist in the specified model
or model suite. For example, the parameter \code{bot} exists in model
\code{"nec4param"} but not in \code{"nec3param"}, so if the user specifies
\code{model = "nec"} in \code{crf}, the term \code{(bot | group_variable)}
will be dropped in models where that parameter does not exist.

\bold{Further brms terms (largely untested)}

Currently \code{\link{bayesnecformula}} is quite agnostic about additional
terms that are valid for a \code{\link[brms]{brmsformula}}. These are
\code{aterms} and \code{pterms} (see \code{?\link[brms]{brmsformula}}).
The only capability that \code{\link{bayesnecformula}} does not allow is
the addition of \code{pterms} outside of the term \code{crf}. Although
\code{pterms} can be passed to predictor \code{x} within \code{crf}, we
strongly discourage the user because those functionalities have not
been tested yet. If this is extremely important to your research, please
raise an issue on GitHub and we will consider.
Currently, the only two \code{aterms} that have validated behaviour are:
\enumerate{
\item \code{trials()} which is essential in binomially-distributed data, e.g.
\code{y | trials(trials_variable)}, and 2) weights, e.g.
\code{y | weights(weights_variable)}, following \pkg{brms} formula syntax.
Please note that \pkg{brms} does not implement design weights as in other
standard \pkg{base} function. From their help page, \pkg{brms} "takes the
weights literally, which means that an observation with weight 2 receives 2
times more weight than an observation with weight 1. It also means that
using a weight of 2 is equivalent to adding the corresponding observation
twice to the data frame." Other \code{aterms} might be added, though we
cannot attest to their functionality within
\code{\link[bayesnec:bayesnec-package]{bayesnec}}, i.e. checks will
be done outside via \code{\link[brms]{brm}}.
}

\strong{NB:} \code{aterms} other than \code{trials()} and \code{weights()} are
currently omitted from \code{\link{model.frame}} output. If you need other
\code{aterms} as part of that output please raise an issue on our GitHub
page.

\bold{Validation of formula}
Please note that the function only checks for the input nature of the
\code{formula} argument and adds a new class. This function \strong{does not}
perform any validation on the model nor checks on its adequacy to work with
other functions in the package. For that please refer to the function
\code{\link{check_formula}} which requires the dataset associated with the
formula.
}
\examples{
library(bayesnec)

bayesnecformula(y ~ crf(x, "nec3param"))
# or use shot alias bnf
bayesnecformula(y ~ crf(x, "nec3param")) == bnf(y ~ crf(x, "nec3param"))
bnf(y | trials(tr) ~ crf(sqrt(x), "nec3param"))
bnf(y | trials(tr) ~ crf(x, "nec3param") + ogl(group_1) + pgl(group_2))
bnf(y | trials(tr) ~ crf(x, "nec3param") + (nec + top | group_1))

\donttest{
# complex transformations are not advisable because
# they are passed directly to Stan via brms
# and are likely to fail -- transform your variable beforehand!
try(bnf(y | trials(tr) ~ crf(scale(x, scale = TRUE), "nec3param")))
}
}
\seealso{
\code{\link{check_formula}},
\code{\link{model.frame}},
\code{\link{models}},
\code{\link{show_params}},
\code{\link{make_brmsformula}}
}
