% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesfactor_restricted.R
\name{bayesfactor_restricted}
\alias{bayesfactor_restricted}
\alias{bf_restricted}
\alias{bayesfactor_restricted.stanreg}
\alias{bayesfactor_restricted.brmsfit}
\alias{bayesfactor_restricted.blavaan}
\alias{bayesfactor_restricted.emmGrid}
\alias{as.logical.bayesfactor_restricted}
\title{Bayes Factors (BF) for Order Restricted Models}
\usage{
bayesfactor_restricted(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  ...
)

bf_restricted(posterior, hypothesis, prior = NULL, verbose = TRUE, ...)

\method{bayesfactor_restricted}{stanreg}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  ...
)

\method{bayesfactor_restricted}{brmsfit}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  ...
)

\method{bayesfactor_restricted}{blavaan}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  ...
)

\method{bayesfactor_restricted}{emmGrid}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  ...
)

\method{as.logical}{bayesfactor_restricted}(x, which = c("posterior", "prior"), ...)
}
\arguments{
\item{posterior}{A \code{stanreg} / \code{brmsfit} object, \code{emmGrid} or a data frame - representing
a posterior distribution(s) from (see Details).}

\item{hypothesis}{A character vector specifying the restrictions as logical conditions (see examples below).}

\item{prior}{An object representing a prior distribution (see Details).}

\item{verbose}{Toggle off warnings.}

\item{...}{Currently not used.}

\item{effects}{Should results for fixed effects, random effects or both be
returned? Only applies to mixed models. May be abbreviated.}

\item{component}{Should results for all parameters, parameters for the
conditional model or the zero-inflated part of the model be returned? May
be abbreviated. Only applies to \pkg{brms}-models.}

\item{x}{An object of class \code{bayesfactor_restricted}}

\item{which}{Should the logical matrix be of the posterior or prior distribution(s)?}
}
\value{
A data frame containing the (log) Bayes factor representing evidence
\emph{against} the un-restricted model (Use \code{as.numeric()} to extract the
non-log Bayes factors; see examples). (A \code{bool_results} attribute contains
the results for each sample, indicating if they are included or not in the
hypothesized restriction.)
}
\description{
This method computes Bayes factors for comparing a model with an order restrictions on its parameters
with the fully unrestricted model. \emph{Note that this method should only be used for confirmatory analyses}.
\cr \cr
The \verb{bf_*} function is an alias of the main function.
\cr \cr
\strong{For more info, in particular on specifying correct priors for factors with more than 2 levels,
see \href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.}
}
\details{
This method is used to compute Bayes factors for order-restricted models vs un-restricted
models by setting an order restriction on the prior and posterior distributions
(\cite{Morey & Wagenmakers, 2013}).
\cr\cr
(Though it is possible to use \code{bayesfactor_restricted()} to test interval restrictions,
it is more suitable for testing order restrictions; see examples).
}
\section{Setting the correct \code{prior}}{

For the computation of Bayes factors, the model priors must be proper priors
(at the very least they should be \emph{not flat}, and it is preferable that
they be \emph{informative}); As the priors for the alternative get wider, the
likelihood of the null value(s) increases, to the extreme that for completely
flat priors the null is infinitely more favorable than the alternative (this
is called \emph{the Jeffreys-Lindley-Bartlett paradox}). Thus, you should
only ever try (or want) to compute a Bayes factor when you have an informed
prior.
\cr\cr
(Note that by default, \code{brms::brm()} uses flat priors for fixed-effects;
See example below.)
\cr\cr
It is important to provide the correct \code{prior} for meaningful results.
\itemize{
\item When \code{posterior} is a numerical vector, \code{prior} should also be a numerical vector.
\item When \code{posterior} is a \code{data.frame}, \code{prior} should also be a \code{data.frame}, with matching column order.
\item When \code{posterior} is a \code{stanreg}, \code{brmsfit} or other supported Bayesian model:
\itemize{
\item \code{prior} can be set to \code{NULL}, in which case prior samples are drawn internally.
\item \code{prior} can also be a model equivalent to \code{posterior} but with samples from
the priors \emph{only}. See \code{\link[=unupdate]{unupdate()}}.
\item \strong{Note:} When \code{posterior} is a \code{brmsfit_multiple} model, \code{prior} \strong{must} be provided.
}
\item When \code{posterior} is an \code{emmGrid} / \code{emm_list} object:
\itemize{
\item \code{prior} should also be an \code{emmGrid} / \code{emm_list} object equivalent to \code{posterior} but
created with a model of priors samples \emph{only}. See \code{\link[=unupdate]{unupdate()}}.
\item \code{prior} can also be the original (posterior) \emph{model}. If so, the function will try to
update the \code{emmGrid} / \code{emm_list} to use the \code{\link[=unupdate]{unupdate()}}d prior-model.
(\emph{This cannot be done for \code{brmsfit} models.})
\item \strong{Note}: When the \code{emmGrid} has undergone any transformations (\code{"log"}, \code{"response"}, etc.),
or \code{regrid}ing, then \code{prior} must be an \code{emmGrid} object, as stated above.
}
}
}

\section{Interpreting Bayes Factors}{

A Bayes factor greater than 1 can be interpreted as evidence against the
null, at which one convention is that a Bayes factor greater than 3 can be
considered as "substantial" evidence against the null (and vice versa, a
Bayes factor smaller than 1/3 indicates substantial evidence in favor of the
null-model) (\cite{Wetzels et al. 2011}).
}

\examples{
set.seed(444)
library(bayestestR)
prior <- data.frame(
  A = rnorm(500),
  B = rnorm(500),
  C = rnorm(500)
)

posterior <- data.frame(
  A = rnorm(500, .4, 0.7),
  B = rnorm(500, -.2, 0.4),
  C = rnorm(500, 0, 0.5)
)

hyps <- c(
  "A > B & B > C",
  "A > B & A > C",
  "C > A"
)


(b <- bayesfactor_restricted(posterior, hypothesis = hyps, prior = prior))

bool <- as.logical(b, which = "posterior")
head(bool)

\dontshow{if (require("see") && require("patchwork")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

see::plots(
  plot(estimate_density(posterior)),
  # distribution **conditional** on the restrictions
  plot(estimate_density(posterior[bool[, hyps[1]], ])) + ggplot2::ggtitle(hyps[1]),
  plot(estimate_density(posterior[bool[, hyps[2]], ])) + ggplot2::ggtitle(hyps[2]),
  plot(estimate_density(posterior[bool[, hyps[3]], ])) + ggplot2::ggtitle(hyps[3]),
  guides = "collect"
)
\dontshow{\}) # examplesIf}
\dontshow{if (require("rstanarm")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
# rstanarm models
# ---------------
data("mtcars")

fit_stan <- rstanarm::stan_glm(mpg ~ wt + cyl + am,
  data = mtcars, refresh = 0
)
hyps <- c(
  "am > 0 & cyl < 0",
  "cyl < 0",
  "wt - cyl > 0"
)

bayesfactor_restricted(fit_stan, hypothesis = hyps)
}
\dontshow{\}) # examplesIf}
\dontshow{if (require("rstanarm") && require("emmeans")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
# emmGrid objects
# ---------------
# replicating http://bayesfactor.blogspot.com/2015/01/multiple-comparisons-with-bayesfactor-2.html
data("disgust")
contrasts(disgust$condition) <- contr.equalprior_pairs # see vignette
fit_model <- rstanarm::stan_glm(score ~ condition, data = disgust, family = gaussian())

em_condition <- emmeans::emmeans(fit_model, ~condition, data = disgust)
hyps <- c("lemon < control & control < sulfur")

bayesfactor_restricted(em_condition, prior = fit_model, hypothesis = hyps)
# > # Bayes Factor (Order-Restriction)
# >
# >                          Hypothesis P(Prior) P(Posterior)   BF
# >  lemon < control & control < sulfur     0.17         0.75 4.49
# > ---
# > Bayes factors for the restricted model vs. the un-restricted model.
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Morey, R. D., & Wagenmakers, E. J. (2014). Simple relation between Bayesian order-restricted and
point-null hypothesis tests. Statistics & Probability Letters, 92, 121-124.
\item Morey, R. D., & Rouder, J. N. (2011). Bayes factor approaches for testing interval null hypotheses.
Psychological methods, 16(4), 406.
\item Morey, R. D. (Jan, 2015). Multiple Comparisons with BayesFactor, Part 2 – order restrictions.
Retrieved from https://richarddmorey.org/category/order-restrictions/.
}
}
