% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OTC.R
\name{OTC}
\alias{OTC}
\title{Find the optimal testing configuration for standard group testing algorithms}
\usage{
OTC(algorithm, p = NULL, probabilities = NULL, Se = 0.99, Sp = 0.99,
  group.sz, obj.fn = c("ET", "MAR"), weights = NULL, alpha = NULL)
}
\arguments{
\item{algorithm}{character string defining the group testing algorithm to be used.
Non-informative testing options include two-stage ("\kbd{D2}") and three-stage
("\kbd{D3}") hierarchical, and square array testing with and without master pooling
("\kbd{A2M}" and "\kbd{A2}", respectively). Informative testing options include
two-stage ("\kbd{ID2}") and three-stage ("\kbd{ID3}") hierarchical, and square
array testing without master pooling ("\kbd{IA2}").}

\item{p}{overall probability of disease that will be used to generate either a
homogeneous or heterogeneous vector/matrix of individual probabilities,
depending on the algorithm specified}

\item{probabilities}{a vector of individual probabilities, which is homogeneous for
non-informative testing algorithms and heterogeneous for informative testing algorithms}

\item{Se}{the sensitivity of the diagnostic test}

\item{Sp}{the specificity of the diagnostic test}

\item{group.sz}{a single group size for which to calculate operating characteristics
(for two-stage hierarchical and array testing) or to find the optimal testing
configuration over all possible configurations (for three-stage hierarchical testing),
or a range of group sizes over which to find the optimal testing configuration}

\item{obj.fn}{a list of objective functions which are minimized to find the
optimal testing configuration. Available options include "\kbd{ET}" (the expected
number of tests per individual), "\kbd{MAR}" (the expected number of tests divided
by the expected number of correct classifications, described in Malinovsky et al. (2016)),
\url{http://www.chrisbilder.com/grouptesting/HBTM/} for additional details.}

\item{weights}{a matrix of up to six sets of weights for the GR function. Each set of
weights is specified by a row of the matrix.}

\item{alpha}{a shape parameter for the beta distribution that specifies the degree of
heterogeneity for the generated probability vector (for informative testing only)}
}
\value{
A list containing:
\item{prob}{the probability of infection, as specified by the user}
\item{alpha}{level of heterogeneity for the generated probability vector
(for informative testing only)}
\item{Se}{the sensitivity of the diagnostic test}
\item{Sp}{the specificity of the diagnostic test}
\item{opt.ET, opt.MAR, opt.GR}{a list for each objective function specified
by the user, containing:
\describe{
\item{OTC}{a list specifying the optimal testing configuration, which may include:
\describe{
\item{Stage1}{pool sizes for the first stage of testing, if applicable}
\item{Stage2}{pool sizes for the second stage of testing, if applicable}
\item{Block.sz}{the block size/initial group size for informative Dorfman testing,
which is not tested}
\item{pool.szs}{pool sizes for the first stage of testing for informative Dorfman
testing}
\item{Array.dim}{the row/column size for array testing}
\item{Array.sz}{the array size for array testing}}}
\item{p.vec}{the sorted vector of individual probabilities, if applicable}
\item{p.mat}{the sorted matrix of individual probabilities in gradient arrangement,
if applicable}
\item{ET}{the expected testing expenditure for the OTC}
\item{value}{the value of the objective function per individual}
\item{PSe}{the overall pooling sensitivity for the algorithm}
\item{PSp}{the overall pooling specificity for the algorithm}
\item{PPPV}{the overall pooling positive predictive value for the algorithm}
\item{PNPV}{the overall pooling negative predictive value for the algorithm}}}
}
\description{
Find the optimal testing configuration (OTC) for standard group testing
algorithms, including hierarchical and array-based algorithms, and calculate the
associated operating characteristics.
}
\details{
This function finds the optimal testing configuration and computes the
associated operating characteristics for standard group testing algorithms,
as described in Hitt et al. (2018) at \url{http://www.chrisbilder.com/grouptesting/HBTM/}.

Available algorithms include two- and three-stage hierarchical testing, and
array testing with and without master pooling. Both non-informative and informative
group testing settings are allowed for each algorithm, with one exception. Only
non-informative array testing with master pooling is available, because no informative
group testing algorithms have been proposed for array testing with master pooling.
Operating characteristics calculated include the expected testing expenditure and
accuracy measures, including pooling sensitivity, pooling specificity, pooling
positive predictive value, and pooling negative predictive value for each individual.
}
\note{
Either \kbd{p} or \kbd{probabilities} should be specified, but not both.
}
\examples{
# Find the optimal testing configuration for non-informative
#   two-stage hierarchical (Dorfman) testing
# This example takes less than one second to run
OTC(algorithm="D2", p=0.05, Se=0.99, Sp=0.99, group.sz=2:100,
obj.fn=c("ET", "MAR"))

# Find the optimal testing configuration for informative
#   two-stage hierarchical (Dorfman) testing, implemented
#   via the pool-specific optimal Dorfman (PSOD) method
#   described in McMahan et al. (2012a)
# This example takes approximately 16 minutes to run
\dontrun{
set.seed(52613)
OTC(algorithm="ID2", p=0.01, Se=0.95, Sp=0.95, group.sz=3:50,
obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 0.5, 0.5), nrow=3, ncol=2, byrow=TRUE),
alpha=0.5)}

# Find the optimal testing configuration over all possible
#   testing configurations for a specified group size for
#   non-informative three-stage hierarchical testing
# This example takes approximately 1 second to run
\dontrun{
OTC(algorithm="D3", p=0.001, Se=0.95, Sp=0.95, group.sz=24,
obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1), nrow=1, ncol=2, byrow=TRUE))
}

# Find the optimal testing configuration for non-informative
#   three-stage hierarchical testing
# This example takes approximately 20 seconds to run
\dontrun{
OTC(algorithm="D3", p=0.06, Se=0.90, Sp=0.90, group.sz=3:30,
obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 100, 100), nrow=3, ncol=2, byrow=TRUE))}

# Find the optimal testing configuration over all possible
#   configurations for a specified group size, given a heterogeneous
#   vector of probabilities
# This example takes less than 1 second to run
\dontrun{
OTC(algorithm="ID3",
probabilities=c(0.012, 0.014, 0.011, 0.012, 0.010, 0.015),
Se=0.99, Sp=0.99, group.sz=6, obj.fn=c("ET","MAR","GR1"),
weights=matrix(data=c(1, 1), nrow=1, ncol=2, byrow=TRUE), alpha=0.5)}

# Calculate the operating characteristics for a specified array size
#   for non-informative array testing without master pooling
# This example takes less than 1 second to run
\dontrun{OTC(algorithm="A2", p=0.005, Se=0.95, Sp=0.95, group.sz=15, obj.fn=c("ET", "MAR"))}

# Find the optimal testing configuration for informative
#   array testing without master pooling
# This example takes approximately 30 seconds to run
\dontrun{
set.seed(1002)
OTC(algorithm="IA2", p=0.03, Se=0.95, Sp=0.95, group.sz=3:20,
obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 100, 100), nrow=3, ncol=2, byrow=TRUE),
alpha=2)}

# Find the optimal testing configuration for non-informative
#   array testing with master pooling
# This example takes approximately 15 seconds to run
\dontrun{
OTC(algorithm="A2M", p=0.02, Se=0.90, Sp=0.90, group.sz=3:20, obj.fn=c("ET", "MAR", "GR"),
weights=matrix(data=c(1, 1, 10, 10, 0.5, 0.5, 2, 2, 100, 100, 10, 100), nrow=6, ncol=2, byrow=TRUE))}

}
\references{
\emph{Graff, L.E. & Roeloffs, R. (1972)}. Group testing in the presence of test error; an extension of
the Dorfman procedure. \emph{Technometrics, 14, 113-122}.

\emph{Malinovsky, Y.; Albert, P.S. & Roy, A. (2016)}. Reader reaction: A note on the evaluation of
group testing algorithms in the presence of misclassification. \emph{Biometrics, 72, 299-302}.
}
\seealso{
\code{\link{NI.Dorf}} for non-informative two-stage (Dorfman) testing, \code{\link{Inf.Dorf}} for
informative two-stage (Dorfman) testing, \code{\link{NI.D3}} for non-informative three-stage
hierarchical testing, \code{\link{Inf.D3}} for informative three-stage hierarchical testing,
\code{\link{NI.Array}} for non-informative array testing, \code{\link{Inf.Array}} for informative
array testing, and \code{\link{NI.A2M}} for non-informative array testing with master pooling

\url{http://chrisbilder.com/grouptesting/HBTM/}

Other Optimal Testing Configuration functions: \code{\link{Inf.Array}},
  \code{\link{Inf.D3}}, \code{\link{Inf.Dorf}},
  \code{\link{NI.A2M}}, \code{\link{NI.Array}},
  \code{\link{NI.D3}}, \code{\link{NI.Dorf}}
}
\author{
Brianna Hitt
}
