% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chronochrt_plot.R
\name{plot_chronochrt}
\alias{plot_chronochrt}
\title{Plot a chronological chart}
\usage{
plot_chronochrt(
  data,
  labels_text = NULL,
  labels_image = NULL,
  year_lim = NULL,
  axis_title = "Years",
  minimal = FALSE,
  size_text = 6,
  height_image = 2,
  size_line = 0.5,
  fill_chron = "white",
  color_chron = "black",
  color_label = "black",
  line_break = 10,
  filename = NULL,
  plot_dim,
  background = NULL,
  ...
)
}
\arguments{
\item{data}{A data set with chronological data.}

\item{labels_text}{A data set containing the text labels.}

\item{labels_image}{A data set containing the image labels.}

\item{year_lim}{A numeric vector of length 2 to define the lower and upper
limit of the chronological chart.}

\item{axis_title}{A character string with the axis label of the vertical
axis. Default is \code{"Years"}.}

\item{minimal}{Should chrons be optically separated by vertical lines? If
\code{TRUE} only vertical lines between around the chronological columns
will be drawn.}

\item{size_text}{Font size of the names of the chronological units in mm. All
other text elements will be scaled accordingly. The default is \code{6} mm.}

\item{height_image}{The absolute height of the image labels in cm. The
default is  \code{2} cm.}

\item{size_line}{Thickness of the line in mm. The default is \code{0.5} mm.}

\item{fill_chron}{Fill colour of the chronological units. The default is
\code{"white"}. See the colour specification section of
\code{\link[graphics]{par}} for how to specify colours in R.}

\item{color_chron}{Line (border) colour of the chronological units. The
default is \code{"black"}. See the colour specification section of
\code{\link[graphics]{par}} for how to specify colours in R.}

\item{color_label}{Colour of the text labels. The default is \code{"black"}.
See the colour specification section of \code{\link[graphics]{par}} for
how to specify colours in R.}

\item{line_break}{Line length of the section labels in characters. Text will
be wrapped at the blank closest to the specified number of characters.
Default is \code{10} characters.}

\item{filename}{A character string with the filename or path. If specified,
the plot will be saved in the given location. The file format is
automatically recognised from the file extension. The most common file
types are supported, e.g. \code{.tiff}, \code{.png}, \code{.jpg},
\code{.eps}, and \code{.pdf}. To export as \code{.svg} installation of the
package \pkg{svglite} is required. See \code{\link[ggplot2]{ggsave}} for
more details about the supported file formats.}

\item{plot_dim}{Dimensions of the plot as a vector in the format
\code{c(width, height, units)}. Supported units are "cm", "mm", in". For
example, \code{plot_dim = c(5,5,"cm")} will produce a plot of the
dimensions 5 x 5 cm. If unspecified, the standard values of the respective
graphic device are used.}

\item{background}{Optional specifications for the background of the
chronological chart as vector in the format \code{c(background colour,
linetype of grid lines)} to overwrite the default behaviour of
\code{\link{theme_chronochrt}}. Any valid colour and line type
specifications are accepted, e.g. \code{c("grey90", "dotted")} (these are
the default values of \code{\link{theme_chronochrt}}. See the sections
"colour specification" and "line type specification" in \code{\link[graphics]{par}} for
how to specify colours and line types in R.}

\item{...}{Additional arguments passed to \code{\link[ggplot2]{ggsave}} to
enhance the saved plot like \code{dpi} to specify its resolution. See
\code{\link[ggplot2]{ggsave}} for detailed information.}
}
\value{
A ggplot2 object with the chronological chart.
}
\description{
This function converts a chronological data set into a chronological chart.
It provides basic features for the export of the plot and for its
customisation.
}
\details{
This function is wrapper around various functions for an quick and convenient
way to draw chronological charts. It relies on the common data
structure of ChronochRt (see \code{vignette("ChronochRt")} for details). For
full customisation use the respective geoms to build your own plot.

It is assumed that the majority of the text labels will be placed on the
right side of each column. Therefore they are right aligned to prevent them
from running outside the plotting area. Vertically, it will be placed
centred on the \code{year} given. Text in labels can be wrapped by inserting
\code{"\n"} (without blanks).
}
\examples{

# Create Example data
chrons <- data.frame(region = c("A", "B", "B", "B", "A"),
                     name = c("a", "a", "1", "2", "b"),
                     start = c(-100, -100, -100, "0/50", "50_100"),
                     end = c("50_100", 150, "0/50", 150, 200),
                     level = c(1, 1, 2, 2, 1),
                     add = FALSE)

# Plot with default options
plot_chronochrt(chrons)

# Add labels
labels <- data.frame(region = "A",
                     year = -50,
                     position = 0.5,
                     label = "Event")

images <- data.frame(region = "B",
                     year = 100,
                     position = 0.5,
                     image_path = "https://www.r-project.org/logo/Rlogo.png")

plot_chronochrt(chrons, labels, images)

# Customise plot
plot_chronochrt(chrons, axis_title = "BC/AD", year_lim = c(-50,100),
                fill_chron = "black", color_chron = "white", size_line = 5)
plot_chronochrt(chrons, labels, images, color_label = "red", size_text = 5, height_image = 4)

# Export plot
\dontrun{
  plot_chronochrt(chrons, filename = "chronology.jpg", plot_dim = c(10, 15, "cm"))

  # with additional parameters
  plot_chronochrt(chrons, filename = "chronology.jpg", plot_dim = c(10, 15, "cm"), dpi = 300)

}

# Additional customisation with ggplot2
plot_chronochrt(chrons) + ggplot2::theme_bw()
}
