% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/risk_ratio.R
\name{calc_riskRatio_gev}
\alias{calc_riskRatio_gev}
\title{Compute risk ratio and uncertainty based on generalized extreme value model fit to block maxima or minima}
\usage{
calc_riskRatio_gev(returnValue, y1, y2, x1 = NULL, x2 = x1,
  locationFun1 = NULL, locationFun2 = locationFun1, scaleFun1 = NULL,
  scaleFun2 = scaleFun1, shapeFun1 = NULL, shapeFun2 = shapeFun1,
  nReplicates1 = 1, nReplicates2 = 1, replicateIndex1 = NULL,
  replicateIndex2 = NULL, weights1 = NULL, weights2 = NULL,
  xNew1 = NULL, xNew2 = NULL, maxes = TRUE, scaling1 = 1,
  scaling2 = 1, ciLevel = 0.9, ciType, bootSE, bootControl = NULL,
  lrtControl = NULL, optimArgs = NULL, initial1 = NULL, initial2 = NULL,
  logScale1 = NULL, logScale2 = NULL, getReturnCalcs = FALSE,
  getParams = FALSE, getFit = FALSE)
}
\arguments{
\item{returnValue}{numeric value giving the value for which the risk ratio should be calculated, where the resulting period will be the average number of blocks until the value is exceeded and the probability the probability of exceeding the value in any single block.}

\item{y1}{a numeric vector of observed maxima or minima values for the first dataset. See \code{Details} for how the values of \code{y1} should be ordered if there are multiple replicates and the values of \code{x1} are identical for all replicates.}

\item{y2}{a numeric vector of observed maxima or minima values for the second dataset. Analogous to \code{y1}.}

\item{x1}{a data frame, or object that can be converted to a data frame with columns corresponding to covariate/predictor/feature variables and each row containing the values of the variable for the corresponding observed maximum/minimum. The number of rows should either equal the length of \code{y1} or (if there is more than one replicate) it can optionally equal the number of observations in a single replicate, in which case the values will be assumed to be the same for all replicates.}

\item{x2}{analogous to \code{x1} but for the second dataset}

\item{locationFun1}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the location parameter using columns from \code{x1} for the first dataset. \code{x1} must be supplied if this is anything other than NULL or ~1.}

\item{locationFun2}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the location parameter using columns from \code{x2} for the second dataset. \code{x2} must be supplied if this is anything other than NULL or ~1.}

\item{scaleFun1}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the log of the scale parameter using columns from \code{x1} for the first dataset.  \code{x1} must be supplied if this is anything other than NULL or ~1.}

\item{scaleFun2}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the log of the scale parameter using columns from \code{x2} for the second dataset.  \code{x2} must be supplied if this is anything other than NULL or ~1.}

\item{shapeFun1}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the shape parameter using columns from \code{x1} for the first dataset.  \code{x1} must be supplied if this is anything other than NULL or ~1.}

\item{shapeFun2}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the shape parameter using columns from \code{x2} for the first dataset.  \code{x2} must be supplied if this is anything other than NULL or ~1.}

\item{nReplicates1}{numeric value indicating the number of replicates for the first dataset.}

\item{nReplicates2}{numeric value indicating the number of replicates for the second dataset.}

\item{replicateIndex1}{numeric vector providing the index of the replicate corresponding to each element of \code{y1}. Used (and therefore required) only when using bootstrapping with the resampling by replicates based on the \code{by} element of \code{bootControl}.}

\item{replicateIndex2}{numeric vector providing the index of the replicate corresponding to each element of \code{y2}. Analogous to \code{replicateIndex1}.}

\item{weights1}{a vector providing the weights for each observation in the first dataset. When there is only one replicate or the weights do not vary by replicate, a vector of length equal to the number of observations. When weights vary by replicate, this should be of equal length to \code{y}. Likelihood contribution of each observation is multiplied by the corresponding weight.}

\item{weights2}{a vector providing the weights for each observation in the second dataset. Analogous to \code{weights1}.}

\item{xNew1}{object of the same form as \code{x1}, providing covariate/predictor/feature values for which one desires log risk ratios.}

\item{xNew2}{object of the same form as \code{x2}, providing covariate/predictor/feature values for which log risk ratios are desired. Must provide the same number of covariate sets as \code{xNew1} as the risk ratio is based on contrasting return probabilities under \code{xNew1} and \code{xNew2}.}

\item{maxes}{logical indicating whether analysis is for block maxima (TRUE) or block minima (FALSE); in the latter case, the function works with the negative of the values, changing the sign of the resulting location parameters}

\item{scaling1}{positive-valued scalar used to scale the data values of the first dataset for more robust optimization performance. When multiplied by the values, it should produce values with magnitude around 1.}

\item{scaling2}{positive-valued scalar used to scale the data values of the second dataset for more robust optimization performance. When multiplied by the values, it should produce values with magnitude around 1.}

\item{ciLevel}{statistical confidence level for confidence intervals; in repeated experimentation, this proportion of confidence intervals should contain the true risk ratio. Note that if only one endpoint of the resulting interval is used, for example the lower bound, then the effective confidence level increases by half of one minus \code{ciLevel}. For example, a two-sided 0.90 confidence interval corresponds to a one-sided 0.95 confidence interval.}

\item{ciType}{character vector indicating which type of confidence intervals to compute. See \code{Details}.}

\item{bootSE}{logical indicating whether to use the bootstrap to estimate the standard error of the risk ratio}

\item{bootControl}{a list of control parameters for the bootstrapping. See \code{Details}.}

\item{lrtControl}{list containing a single component, \code{bounds}, which sets the range inside which the algorithm searches for the endpoints of the likelihood ratio-based confidence interval. This avoids numerical issues with endpoints converging to zero and infinity. If an endpoint is not found within the interval, it is set to \code{NA}.}

\item{optimArgs}{a list with named components matching exactly any arguments that the user wishes to pass to \code{optim}. See \code{help(optim)} for details. Of particular note, \code{'method'} can be used to choose the optimization method used for maximizing the log-likelihood to fit the model and \code{'control=list(maxit=VALUE)'} for a user-chosen VALUE can be used to increase the number of iterations if the optimization is converging slowly.}

\item{initial1}{a list with components named \code{'location'}, \code{'scale'}, and \code{'shape'} providing initial parameter values for the first dataset, intended for use in speeding up or enabling optimization when the default initial values are resulting in failure of the optimization; note that use of \code{scaling1}, \code{logScale1} and \code{.normalizeX = TRUE} cause numerical changes in some of the parameters.}

\item{initial2}{a list with components named \code{'location'}, \code{'scale'}, and \code{'shape'} providing initial parameter values for the second dataset, intended for use in speeding up or enabling optimization when the default initial values are resulting in failure of the optimization; note that use of \code{scaling2}, \code{logScale2} and \code{.normalizeX = TRUE} cause numerical changes in some of the parameters.}

\item{logScale1}{logical indicating whether optimization for the scale parameter should be done on the log scale for the first dataset. By default this is FALSE when the scale is not a function of covariates and TRUE when the scale is a function of covariates (to ensure the scale is positive regardless of the regression coefficients).}

\item{logScale2}{logical indicating whether optimization for the scale parameter should be done on the log scale for the second dataset. By default this is FALSE when the scale is not a function of covariates and TRUE when the scale is a function of covariates (to ensure the scale is positive regardless of the regression coefficients).}

\item{getReturnCalcs}{logical indicating whether to return the estimated return values/probabilities/periods from the fitted models.}

\item{getParams}{logical indicating whether to return the fitted parameter values and their standard errors for the fitted models; WARNING: parameter values for models with covariates for the scale parameter must interpreted based on the value of \code{logScale}.}

\item{getFit}{logical indicating whether to return the full fitted models (potentially useful for model evaluation and for understanding optimization problems); note that estimated parameters in the fit object for nonstationary models will not generally match the MLE provided when \code{getParams} is \code{TRUE} because covariates are normalized before fitting and the fit object is based on the normalized covariates. Similarly, parameters will not match if \code{scaling} is not 1.}
}
\value{
The primary outputs of this function are as follows: the log of the risk ratio and standard error of that log risk ratio (\code{logRiskRatio} and \code{se_logRiskRatio}) as well the risk ratio itself (\code{riskRatio}). The standard error is based on the usual MLE asymptotics using a delta-method-based approximation. If requested via \code{ciType}, confidence intervals will be returned, as discussed in \code{Details}.
}
\description{
Compute risk ratio and uncertainty by fitting generalized extreme value model, designed specifically for climate data, to exceedance-only data, using the point process approach. The risk ratio is the ratio of the probability of exceedance of a pre-specified value under the model fit to the first dataset to the probability under the model fit to the second dataset. Default standard errors are based on the usual MLE asymptotics using a delta-method-based approximation, but standard errors based on the nonparametric bootstrap and on a likelihood ratio procedure can also be computed.
}
\details{
See \code{\link{fit_gev}} for more details on fitting the block maxima model for each dataset, including details on blocking and replication. Also see \code{\link{fit_gev}} for information on the \code{bootControl} argument.

\code{ciType} can include one or more of the following: \code{'delta'}, \code{'lrt'}, \code{'boot_norm'}, \code{'boot_perc'}, \code{'boot_basic'}, \code{'boot_stud'}, \code{'boot_bca'}. \code{'delta'} uses the delta method to compute an asymptotic interval based on the standard error of the log risk ratio. \code{'lrt'} inverts a likelihood-ratio test. Bootstrap-based options are the normal-based interval using the bootstrap standard error (\code{'boot_norm'}), the percentile bootstrap (\code{'boot_perc'}), the basic bootstrap (\code{'boot_basic'}), the bootstrap-t (\code{'boot_stud'}), and the bootstrap BCA method (\code{'boot_bca'}). See Paciorek et al. for more details. 

See \code{\link{fit_pot}} for information on the \code{bootControl} argument.
}
\section{Optimization failures}{


It is not uncommon for maximization of the log-likelihood to fail for extreme value models. Please see the help information for \code{fit_gev}. Also note that if the probability in the denominator of the risk ratio is near 1, one may achieve better numerical performance by swapping the two datasets and computing the risk ratio for the probability under dataset 2 relative to the probability under dataset 1.
}
\examples{
data(Fort, package = 'extRemes')
FortMax <- aggregate(Prec ~ year, data = Fort, max)
earlyYears <- 1900:1929
lateYears <- 1970:1999
earlyPeriod <- which(FortMax$year \%in\% earlyYears)
latePeriod <- which(FortMax$year \%in\% lateYears)
# contrast late period with early period, assuming a nonstationary fit
# within each time period and finding RR based on midpoint of each period
\dontrun{
out <- calc_riskRatio_gev(returnValue = 3,
                   y1 = FortMax$Prec[earlyPeriod], y2 = FortMax$Prec[latePeriod],
                   x1 = data.frame(years = earlyYears), x2 = data.frame(years = lateYears),
                   locationFun1 = ~years, locationFun2 = ~years,
                   xNew1 = data.frame(years = mean(earlyYears)),
                   xNew2 = data.frame(years = mean(lateYears)))
}
}
\author{
Christopher J. Paciorek
}
\references{
Paciorek, C.J., D.A. Stone, and M.F. Wehner. 2018. Quantifying uncertainty in the attribution of human influence on severe weather. Weather and Climate Extremes 20:69-80. arXiv preprint <https://arxiv.org/abs/1706.03388>.

Jeon S., C.J. Paciorek, and M.F. Wehner. 2016. Quantile-based bias correction and uncertainty quantification of extreme event attribution statements. Weather and Climate Extremes 12: 24-32. <DOI:10.1016/j.wace.2016.02.001>. arXiv preprint: <http://arxiv.org/abs/1602.04139>.
}

