## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup,message=FALSE, warning=FALSE---------------------------------------
library(clusterMI)

## ----message=FALSE------------------------------------------------------------
require(stargazer)
set.seed(123456)
data(wine)
stargazer(wine, type = "text")
table(wine$cult)

## -----------------------------------------------------------------------------
ref <- wine$cult # "True" partition
nb.clust <- 3 # Number of clusters
wine.na <- wine
wine.na$cult <- NULL # Remove the reference partition
wine.na <- as.matrix(wine.na)
wine.na[sample(seq(length(wine.na)), size = ceiling(length(wine.na)/3))] <- NA

## -----------------------------------------------------------------------------
# proportion of missing values
colMeans(is.na(wine.na))

# proportion of incomplete individuals
mean(apply(is.na(wine.na), 1, any))

## ----warning=FALSE------------------------------------------------------------
m <- 20 # Number of imputed datasets
res.imp.JM <- imputedata(data.na = wine.na,
                         nb.clust = nb.clust,
                         m = m)

## ----warning=FALSE, eval=FALSE------------------------------------------------
#  res.imp <- imputedata(data.na = wine.na,
#                           method = "JM-DP",
#                           nb.clust = nb.clust,
#                           m = m)

## ----warning=FALSE------------------------------------------------------------
maxit <- 20 # Number of iterations for FCS imputation, should be larger in practice
res.imp.FCS <- imputedata(data.na = wine.na,
                      method = "FCS-homo",
                      nb.clust = nb.clust,
                      maxit = maxit,
                      m = m,
                      verbose = FALSE)

## ----eval=FALSE---------------------------------------------------------------
#  imputedata(data.na = wine.na,
#             method = "FCS-homo",
#             nb.clust = nb.clust,
#             maxit = maxit,
#             m = m,
#             method.mice = "norm")

## ----conv, fig.height = 7, fig.width = 7, fig.align = "center", results='hide'----
choosemaxit(res.imp.FCS)

## ----eval = FALSE-------------------------------------------------------------
#  res.imp <- imputedata(data.na = wine.na,
#                        method = "FCS-homo",
#                        nb.clust = nb.clust,
#                        maxit = 100,
#                        m = m,
#                        verbose = TRUE)
#  choosemaxit(res.imp, layout= c(2,2))

## ----varselecho, eval=FALSE---------------------------------------------------
#  nnodes <- 2
#  # Number of CPU cores used for parallel computation.
#  # Use parallel::detectCores() to choose an appropriate number
#  
#  # variable selection to impute the "alco" variable
#  B <- 50 # number of bootstrap subsets, should be increased in practice
#  res.varsel <- varselbest(res.imputedata = res.imp.FCS, B = B, listvar = "alco",
#                           nnodes = nnodes, graph = FALSE)
#  
#  res.varsel$predictormatrix["alco", ]

## ----varsel, eval=TRUE, echo=FALSE--------------------------------------------
nnodes <- 2
# Number of CPU cores used for parallel computation.
# Use parallel::detectCores() to choose an appropriate number

# variable selection to impute the "alco" variable
B <- 50 # number of bootstrap subsets, should be increased in practice
res.varsel <- list(predictormatrix = structure(c(0, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0), dim = c(13L, 13L), dimnames = list(
    c("alco", "malic", "ash", "alca", "mg", "phe", "fla", "nfla", 
    "pro", "col", "hue", "ratio", "prol"), c("alco", "malic", 
    "ash", "alca", "mg", "phe", "fla", "nfla", "pro", "col", 
    "hue", "ratio", "prol"))), res.varsel = list(alco = list(
    res = list(garde = c(malic = 23, ash = 22, alca = 20, mg = 21, 
    phe = 20, fla = 21, nfla = 21, pro = 20, col = 19, hue = 21, 
    ratio = 20, prol = 22), effectif = c(malic = 6, ash = 8, 
    alca = 9, mg = 7, phe = 2, fla = 7, nfla = 3, pro = 9, col = 18, 
    hue = 15, ratio = 12, prol = 8), proportion = c(malic = 0.260869565217391, 
    ash = 0.363636363636364, alca = 0.45, mg = 0.333333333333333, 
    phe = 0.1, fla = 0.333333333333333, nfla = 0.142857142857143, 
    pro = 0.45, col = 0.947368421052632, hue = 0.714285714285714, 
    ratio = 0.6, prol = 0.363636363636364), selection = c("ash", 
    "alca", "mg", "fla", "pro", "col", "hue", "ratio", "prol"
    ), failure = c(malic = 0, ash = 0, alca = 0, mg = 0, phe = 0, 
    fla = 0, nfla = 0, pro = 0, col = 0, hue = 0, ratio = 0, 
    prol = 0), listvarblock = list(`1` = structure(list(V1 = 1L, 
        V2 = 11L, V3 = 5L, V4 = 12L, V5 = 3L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 2L, V2 = 8L, V3 = 9L, V4 = 4L, 
            V5 = 7L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 3L, V2 = 7L, V3 = 6L, V4 = 1L, 
            V5 = 8L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 11L, V2 = 5L, V3 = 12L, V4 = 10L, 
            V5 = 9L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 11L, V2 = 1L, V3 = 7L, V4 = 6L, 
            V5 = 4L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 9L, V2 = 10L, V3 = 12L, V4 = 5L, 
            V5 = 2L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 7L, V2 = 1L, V3 = 2L, V4 = 10L, 
            V5 = 5L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 11L, V2 = 3L, V3 = 4L, V4 = 6L, 
            V5 = 12L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 2L, V2 = 5L, V3 = 10L, V4 = 12L, 
            V5 = 6L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 8L, V2 = 3L, V3 = 1L, V4 = 9L, 
            V5 = 4L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 11L, V2 = 4L, V3 = 3L, V4 = 7L, 
            V5 = 1L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 8L, V2 = 10L, V3 = 6L, V4 = 12L, 
            V5 = 2L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 6L, V2 = 8L, V3 = 4L, V4 = 5L, 
            V5 = 7L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 12L, V2 = 1L, V3 = 3L, V4 = 2L, 
            V5 = 9L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 2L, V2 = 7L, V3 = 1L, V4 = 6L, 
            V5 = 10L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 12L, V2 = 8L, V3 = 3L, V4 = 4L, 
            V5 = 9L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 11L, V2 = 7L, V3 = 9L, V4 = 4L, 
            V5 = 10L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 12L, V2 = 8L, V3 = 6L, V4 = 2L, 
            V5 = 5L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 9L, V2 = 6L, V3 = 8L, V4 = 5L, 
            V5 = 12L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 10L, V2 = 4L, V3 = 11L, V4 = 1L, 
            V5 = 7L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 1L, V2 = 7L, V3 = 4L, V4 = 9L, 
            V5 = 6L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 3L, V2 = 10L, V3 = 12L, V4 = 5L, 
            V5 = 2L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 2L, V2 = 11L, V3 = 3L, V4 = 8L, 
            V5 = 6L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 1L, V2 = 4L, V3 = 5L, V4 = 10L, 
            V5 = 7L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 2L, V2 = 1L, V3 = 9L, V4 = 12L, 
            V5 = 11L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 10L, V2 = 8L, V3 = 7L, V4 = 6L, 
            V5 = 4L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 7L, V2 = 8L, V3 = 6L, V4 = 10L, 
            V5 = 11L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 2L, V2 = 3L, V3 = 9L, V4 = 5L, 
            V5 = 12L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 3L, V2 = 9L, V3 = 10L, V4 = 8L, 
            V5 = 6L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 11L, V2 = 4L, V3 = 1L, V4 = 2L, 
            V5 = 12L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 1L, V2 = 7L, V3 = 8L, V4 = 10L, 
            V5 = 5L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 12L, V2 = 3L, V3 = 9L, V4 = 2L, 
            V5 = 4L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 2L, V2 = 1L, V3 = 8L, V4 = 6L, 
            V5 = 11L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 3L, V2 = 12L, V3 = 9L, V4 = 10L, 
            V5 = 5L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 3L, V2 = 8L, V3 = 10L, V4 = 7L, 
            V5 = 2L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 6L, V2 = 4L, V3 = 12L, V4 = 11L, 
            V5 = 1L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 3L, V2 = 2L, V3 = 11L, V4 = 1L, 
            V5 = 4L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 6L, V2 = 10L, V3 = 5L, V4 = 12L, 
            V5 = 9L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 9L, V2 = 3L, V3 = 10L, V4 = 4L, 
            V5 = 12L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 8L, V2 = 1L, V3 = 11L, V4 = 7L, 
            V5 = 5L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 4L, V2 = 5L, V3 = 2L, V4 = 3L, 
            V5 = 12L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 11L, V2 = 7L, V3 = 10L, V4 = 8L, 
            V5 = 1L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 5L, V2 = 4L, V3 = 11L, V4 = 6L, 
            V5 = 9L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 2L, V2 = 12L, V3 = 1L, V4 = 7L, 
            V5 = 10L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 6L, V2 = 5L, V3 = 2L, V4 = 4L, 
            V5 = 7L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 8L, V2 = 9L, V3 = 3L, V4 = 11L, 
            V5 = 1L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 2L, V2 = 1L, V3 = 6L, V4 = 3L, 
            V5 = 7L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 11L, V2 = 5L, V3 = 8L, V4 = 12L, 
            V5 = 4L), row.names = 2L, class = "data.frame"), 
        `1` = structure(list(V1 = 11L, V2 = 9L, V3 = 1L, V4 = 6L, 
            V5 = 8L), row.names = 1L, class = "data.frame"), 
        `2` = structure(list(V1 = 10L, V2 = 3L, V3 = 2L, V4 = 7L, 
            V5 = 5L), row.names = 2L, class = "data.frame"))), 
    res.detail = list(list(res.select = list(SousSelect = c(V1 = 1, 
    V2 = 0, V3 = 1, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 1), garde = c(V1 = 1, V2 = 0, V3 = 1, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 1, V8 = 1, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 1, V8 = 1, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 1, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 1, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 1, V7 = 1, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 1, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 1, V7 = 0, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 1, V6 = 1, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 1, V7 = 0, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 1, V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 1, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 1), garde = c(V1 = 0, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 1, V6 = 1, V7 = 1, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 1, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 1, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 1, V8 = 0, V9 = 1, V10 = 1, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 1, V7 = 0, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 1, V7 = 0, V8 = 1, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 1, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 1, V5 = 0, V6 = 1, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 1, V7 = 1, V8 = 0, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 1, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 1), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 1, V6 = 0, V7 = 1, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 0, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 1, V7 = 1, V8 = 1, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 1, V11 = 1, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 1, V8 = 1, V9 = 0, V10 = 1, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 1, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 1, 
    V2 = 1, V3 = 0, V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 1), garde = c(V1 = 1, V2 = 1, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 1, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 1, V8 = 1, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 1, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 1), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 1, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 1, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 0, V7 = 1, V8 = 1, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 0, V6 = 1, V7 = 0, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 1, V7 = 0, V8 = 0, V9 = 1, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 1, 
    V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 1, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 1, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 1, V4 = 1, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 1), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 1, V5 = 1, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 1, V11 = 1, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 0, V7 = 1, V8 = 1, V9 = 0, V10 = 1, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 1, V6 = 1, V7 = 0, V8 = 0, V9 = 1, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 1, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 1), garde = c(V1 = 1, V2 = 1, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 0, V7 = 1, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 1, V7 = 1, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 0, 
    V4 = 1, V5 = 1, V6 = 1, V7 = 1, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 1, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 1, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 1, V3 = 1, V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 1, V8 = 0, V9 = 0, V10 = 0, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 0, V4 = 1, V5 = 0, V6 = 0, V7 = 0, V8 = 1, V9 = 0, 
    V10 = 0, V11 = 1, V12 = 1), garde = c(V1 = 0, V2 = 0, V3 = 0, 
    V4 = 1, V5 = 1, V6 = 0, V7 = 0, V8 = 1, V9 = 0, V10 = 0, 
    V11 = 1, V12 = 1), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 1, 
    V2 = 0, V3 = 0, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 1, 
    V10 = 0, V11 = 0, V12 = 0), garde = c(V1 = 1, V2 = 0, V3 = 0, 
    V4 = 0, V5 = 0, V6 = 1, V7 = 0, V8 = 1, V9 = 1, V10 = 0, 
    V11 = 1, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118), list(res.select = list(SousSelect = c(V1 = 0, 
    V2 = 0, V3 = 1, V4 = 0, V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, 
    V10 = 1, V11 = 0, V12 = 0), garde = c(V1 = 0, V2 = 1, V3 = 1, 
    V4 = 0, V5 = 1, V6 = 0, V7 = 1, V8 = 0, V9 = 0, V10 = 1, 
    V11 = 0, V12 = 0), failure = c(V1 = 0, V2 = 0, V3 = 0, V4 = 0, 
    V5 = 0, V6 = 0, V7 = 0, V8 = 0, V9 = 0, V10 = 0, V11 = 0, 
    V12 = 0)), qui = 1:118)), call = list(nnodes = 2, X = structure(c(1.71, 
    NA, NA, NA, 1.87, 2.15, 1.35, NA, 1.73, 1.87, NA, 1.92, NA, 
    3.1, 3.8, NA, 1.6, 1.81, NA, NA, 1.9, 1.5, NA, 1.83, 1.53, 
    1.65, NA, 1.71, NA, NA, 3.98, NA, 4.04, 1.68, NA, NA, 1.67, 
    1.7, 1.97, 0.94, NA, 1.36, NA, NA, 1.21, NA, NA, 1.51, 1.67, 
    1.09, 1.88, NA, 3.87, NA, NA, 1.13, NA, NA, 1.61, NA, NA, 
    1.51, NA, NA, NA, 1.41, NA, 2.08, NA, NA, NA, 1.29, NA, 2.68, 
    1.39, NA, NA, 2.4, 4.43, NA, 4.31, 2.16, 2.13, 4.3, 1.35, 
    2.99, 3.55, 1.24, NA, 5.51, NA, 2.81, 2.56, NA, NA, 3.88, 
    4.61, 3.24, 2.67, NA, 5.19, 4.12, 3.03, 1.68, 1.67, NA, NA, 
    3.45, 2.76, NA, 2.58, 4.6, 2.39, NA, 3.91, NA, 2.59, 4.1, 
    2.43, 2.14, 2.67, NA, 2.45, 2.61, NA, 2.41, 2.39, 2.38, 2.7, 
    2.72, 2.62, 2.56, 2.65, NA, 2.52, 2.61, NA, NA, NA, NA, 2.36, 
    NA, 2.7, 2.55, 2.51, NA, 2.12, 2.59, 2.29, 2.1, 2.44, 2.12, 
    2.04, NA, NA, 2.3, 2.68, 1.36, NA, NA, 2.16, 2.53, 2.56, 
    NA, 1.75, 2.67, 2.6, 2.3, NA, NA, 2.4, 2, NA, 2.51, 2.32, 
    2.58, NA, 2.3, 2.32, NA, 2.26, 2.28, 2.74, 1.98, NA, 1.7, 
    NA, NA, 2.28, 1.94, 2.7, 2.92, 2.5, 2.2, 1.99, 2.42, NA, 
    2.13, 2.39, 2.17, NA, 2.38, NA, 2.4, 2.36, 2.25, 2.54, 2.64, 
    2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.48, NA, 2.48, 2.28, 2.32, 
    2.38, NA, NA, 2.64, 2.54, NA, 2.35, 2.3, NA, 2.69, NA, 2.28, 
    NA, 2.48, 2.26, 2.37, 2.74, 15.6, NA, NA, 21, NA, 17.6, 16, 
    16, 11.4, NA, NA, NA, 20, 15.2, 18.6, NA, 17.8, 20, 16.1, 
    17, NA, NA, 19.1, NA, NA, NA, 13.2, 16.2, 18.8, NA, NA, 17, 
    NA, NA, 12.4, NA, NA, 16.3, 16.8, 10.6, NA, 16.8, 19, 19, 
    18.1, NA, 16.8, NA, 30, 21, NA, 18, NA, 19, NA, 24, 22.5, 
    18, 22.8, NA, NA, 22, NA, 18, 21.5, 16, 18, 17.5, 18.5, 20.5, 
    22.5, 19, 20, NA, NA, 21.5, 20.8, NA, NA, 21.5, 21, 21, 28.5, 
    22, NA, 20, 21.5, NA, NA, 25, NA, 21, NA, 23.5, 20, NA, NA, 
    NA, 22, 18.5, 22, 19.5, NA, 25, NA, NA, NA, 18.5, NA, 22.5, 
    24.5, 25, 19.5, NA, 23, 20, NA, 24.5, 127, 100, 101, NA, 
    96, 121, NA, 89, 91, 102, 112, 120, NA, 116, 102, NA, 95, 
    NA, 93, NA, 107, 101, 106, 104, NA, NA, NA, 117, NA, 101, 
    NA, 107, 111, 101, 92, 111, NA, 118, 102, 88, 101, 100, 87, 
    104, 98, NA, 151, 86, 139, 101, NA, 112, NA, 86, NA, 78, 
    85, NA, 90, 70, 81, NA, NA, NA, 134, 85, NA, 97, 88, 85, 
    84, 92, 94, 103, 84, 85, 86, 96, 102, 86, NA, 85, 92, 80, 
    122, NA, 106, NA, 89, NA, 101, NA, 89, 97, NA, 112, NA, 92, 
    NA, 98, NA, 89, 97, 98, 89, NA, 106, 106, 90, 88, 105, 112, 
    86, 91, 102, NA, 120, NA, 2.8, 2.65, 2.8, 2.8, 2.5, 2.6, 
    2.98, 2.6, 3.1, NA, 2.85, 2.8, NA, NA, 2.41, 2.61, 2.48, 
    2.53, NA, 2.4, 2.95, 3, NA, 2.42, 2.95, 2.45, NA, 3.15, 2.45, 
    3.25, 2.64, 3, 2.85, 3.1, 2.72, 3.88, NA, 3.2, 3, 1.98, 2.05, 
    2.02, 3.5, 1.89, NA, 2.53, NA, 2.95, NA, NA, NA, NA, 2.83, 
    NA, 2.2, NA, 1.65, NA, NA, 2.2, 1.6, 1.45, 1.38, 3.02, NA, 
    2.55, NA, NA, NA, 2.2, NA, 2.36, 2.74, 1.75, 2.56, 2.46, 
    NA, 2.9, NA, NA, 2.86, NA, 2.13, 2.1, 1.51, NA, 1.7, NA, 
    1.38, NA, NA, NA, 1.4, NA, 2, 1.38, 1.7, 1.93, 1.48, NA, 
    NA, 1.8, 1.9, NA, NA, 1.83, NA, 1.39, 1.35, NA, 1.55, NA, 
    1.39, 1.68, NA, NA, 1.65, 2.05, NA, 2.76, NA, NA, 2.52, 2.51, 
    NA, 2.76, NA, 3.64, 2.91, 3.14, 3.4, NA, 2.41, 2.88, 2.37, 
    2.61, 2.94, NA, 2.97, 3.25, 3.19, 2.69, NA, 2.43, 3.04, 3.29, 
    2.68, 3.56, 2.63, 3, 2.65, NA, NA, 3.74, 2.9, NA, 3.23, 0.57, 
    NA, 1.41, NA, 1.75, 2.65, 1.3, NA, 2.86, 2.89, 2.14, 1.57, 
    NA, NA, 2.26, 2.53, 1.58, 1.59, 2.21, 1.69, NA, NA, 1.25, 
    1.46, NA, 0.99, NA, 2.99, 2.17, 1.36, 1.92, 1.76, NA, 2.92, 
    2.03, 2.29, 2.17, 1.6, NA, NA, NA, 3.03, 2.65, 2.24, 1.75, 
    NA, NA, 1.2, 0.58, 0.47, NA, 0.6, 0.5, NA, 0.52, 0.8, NA, 
    0.65, 0.76, 1.36, 0.83, 0.63, NA, 0.58, 1.31, 1.1, NA, 0.6, 
    NA, 0.68, 0.47, 0.84, 0.96, NA, 0.7, NA, 0.69, 0.68, 0.76, 
    0.28, NA, 0.3, 0.39, NA, 0.31, 0.22, 0.29, 0.43, 0.29, 0.3, 
    NA, NA, 0.17, 0.25, 0.27, 0.26, NA, 0.34, 0.27, NA, 0.29, 
    NA, 0.42, NA, NA, 0.2, 0.34, NA, NA, 0.32, 0.28, 0.3, 0.21, 
    0.17, 0.32, 0.21, 0.26, NA, 0.28, NA, 0.53, NA, 0.45, 0.37, 
    NA, NA, NA, 0.21, 0.13, 0.34, 0.43, 0.43, 0.3, NA, 0.4, 0.61, 
    0.22, NA, NA, NA, 0.5, NA, 0.17, NA, NA, NA, NA, 0.29, NA, 
    0.48, 0.39, 0.29, NA, NA, 0.52, 0.3, 0.32, 0.43, 0.3, NA, 
    NA, NA, NA, NA, NA, 0.17, 0.6, 0.53, 0.63, 0.53, 0.53, NA, 
    NA, NA, NA, 0.47, 0.45, NA, 0.61, NA, NA, 0.63, 0.53, 0.52, 
    0.5, 0.6, 0.4, 0.41, NA, 0.39, NA, 0.48, NA, 0.43, 0.43, 
    0.53, NA, 2.29, 1.28, 2.81, 1.82, 1.98, 1.25, 1.85, 1.81, 
    NA, 2.96, 1.46, NA, NA, 1.66, NA, 1.69, 1.46, 1.66, 1.45, 
    1.35, 1.76, 2.38, 1.95, NA, NA, 1.44, NA, NA, NA, NA, NA, 
    2.03, NA, 2.14, NA, 1.87, 1.62, NA, 1.66, 0.42, NA, NA, 1.87, 
    1.03, NA, NA, 2.5, 1.87, NA, NA, 1.15, NA, 1.95, 1.43, NA, 
    NA, 1.62, NA, 1.56, 1.38, 1.64, 1.63, NA, 1.35, 1.56, 1.77, 
    2.81, 1.4, 1.35, NA, 1.63, 2.08, 2.49, NA, 1.04, NA, NA, 
    1.83, 1.71, NA, 2.91, 1.35, NA, NA, 0.94, NA, 0.84, 1.25, 
    0.8, 1.1, NA, 0.75, NA, 0.55, NA, 1.14, 0.86, 1.25, 1.26, 
    NA, 1.55, 1.56, 1.14, 2.7, 2.29, NA, NA, 0.94, NA, 1.15, 
    1.54, 1.11, 0.64, 1.24, 1.41, 1.35, 1.46, 1.35, 5.64, 4.38, 
    5.68, NA, NA, 5.05, 7.22, 5.6, NA, 7.5, NA, NA, NA, 5.1, 
    4.5, NA, 3.93, 3.52, 4.8, 3.95, 4.5, 5.7, NA, NA, 5.4, 4.25, 
    5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 6.1, 7.2, NA, 5.85, 
    NA, NA, NA, 3.27, 5.75, 4.45, 2.95, 4.6, 3.17, 2.85, 3.38, 
    NA, 3.21, 3.8, NA, NA, 2.5, 3.9, NA, 4.8, 3.05, 2.45, 1.74, 
    2.4, 3.6, 3.05, 3.25, NA, 2.9, NA, NA, NA, 2.94, 3.3, 2.7, 
    NA, NA, 2.9, 1.9, NA, 3.25, NA, NA, 2.8, NA, NA, NA, NA, 
    NA, 5, 5.45, NA, 5, 4.92, NA, NA, 4.35, 4.4, 8.21, NA, 8.42, 
    NA, 10.52, 7.9, NA, 7.5, 13, NA, NA, 5.58, NA, NA, 6.62, 
    NA, 8.5, NA, 9.7, 7.3, NA, 9.3, 9.2, 1.04, 1.05, NA, NA, 
    1.02, NA, NA, 1.15, 1.25, 1.2, 1.28, NA, 1.13, NA, NA, 1.11, 
    1.09, 1.12, NA, 1.02, NA, 1.19, 1.09, NA, 1.25, NA, NA, 0.95, 
    0.91, 0.88, 0.82, NA, 0.87, NA, 1.12, NA, NA, 0.94, 1.07, 
    NA, 1.25, 0.98, 1.22, NA, NA, 1.02, 1.28, 1.36, 1.31, NA, 
    1.23, 0.96, 1.19, 1.38, NA, 1.31, 0.84, NA, NA, 1.07, 1.08, 
    1.05, NA, NA, NA, NA, 1.42, 1.27, NA, 1.04, NA, 0.86, NA, 
    NA, 0.93, 1.71, 0.95, 0.8, 0.92, 0.73, NA, 0.86, 0.97, 0.79, 
    NA, 0.74, 0.78, 0.75, 0.75, 0.82, NA, NA, NA, 0.89, NA, 0.65, 
    0.54, 0.55, 0.48, 0.56, 0.6, 0.57, 0.67, 0.57, NA, 0.96, 
    0.87, 0.68, 0.7, 0.78, 0.74, 0.67, NA, NA, 0.7, 0.59, 0.6, 
    0.61, 3.92, 3.4, 3.17, 2.93, 3.58, 3.58, 3.55, 2.9, NA, 3, 
    NA, 2.65, NA, 3.36, 3.52, 4, 3.63, NA, NA, 2.77, NA, 2.71, 
    2.88, 2.87, 3, NA, NA, 3.38, 3, NA, 3, 3.35, 3.33, 3.33, 
    NA, 3.26, 3.2, NA, 2.84, 1.82, NA, 1.59, 2.87, NA, 2.3, NA, 
    NA, NA, 3.5, 3.13, 2.14, 2.52, NA, NA, 3.14, 2.72, 2.01, 
    NA, NA, 3.21, NA, 2.65, NA, NA, NA, 2.74, 2.83, 2.96, 2.77, 
    3.57, 2.42, 3.02, 3.26, 2.5, 3.19, 2.87, 3.33, 3.39, 3.12, 
    NA, 3.64, NA, NA, NA, NA, 1.42, 1.29, 1.51, 1.27, 1.69, 2.15, 
    NA, NA, NA, 2.05, 2, NA, 1.62, 1.47, 1.51, 1.48, 1.64, 1.73, 
    1.96, 1.78, NA, NA, 1.75, 1.68, 1.75, NA, NA, NA, NA, NA, 
    1.56, 1.62, 1.6, 1065, 1050, NA, 735, 1290, 1295, 1045, 1320, 
    NA, 1547, 1310, 1280, NA, 845, 770, 1035, 1015, 845, 1195, 
    1285, 915, 1285, NA, NA, 1235, NA, 760, 795, 1035, 1095, 
    680, 885, 1080, 985, 1150, 1190, 1060, 970, 1270, NA, NA, 
    NA, NA, NA, NA, NA, 718, 410, NA, 886, NA, NA, 463, 278, 
    714, NA, 515, NA, NA, 625, 480, 450, 495, 345, 625, 428, 
    406, NA, 562, 672, NA, 312, 680, NA, 385, NA, NA, NA, 365, 
    380, NA, 378, 466, 580, NA, NA, 600, NA, 720, 515, 590, 600, 
    NA, 520, 550, NA, NA, NA, 480, 675, NA, 480, 880, 660, NA, 
    680, 570, NA, 615, NA, NA, NA, 470, NA, NA, 835, 840, NA), dim = c(118L, 
    12L), dimnames = list(NULL, c("V1", "V2", "V3", "V4", "V5", 
    "V6", "V7", "V8", "V9", "V10", "V11", "V12"))), Y = c(14.23, 
    13.2, 13.16, 13.24, 14.39, 14.06, 13.86, 13.75, 14.75, 14.38, 
    13.63, 14.3, 13.83, 13.64, 12.93, 13.71, 12.85, 13.5, 13.39, 
    13.3, 13.87, 13.73, 13.58, 13.68, 13.76, 13.05, 14.22, 13.56, 
    13.41, 13.88, 13.24, 13.05, 14.21, 13.9, 13.05, 13.82, 13.74, 
    14.22, 13.29, 12.37, 12.33, 12.64, 12.37, 12.17, 12.37, 13.34, 
    12.21, 13.86, 12.99, 11.96, 11.66, 11.84, 12.7, 12, 12.72, 
    12.08, 13.05, 11.84, 12.16, 12.08, 12.08, 12, 12.69, 11.62, 
    11.81, 12.29, 12.29, 12.08, 12.6, 12.51, 12.72, 12.22, 11.61, 
    11.76, 12.08, 11.03, 11.82, 11.45, 12.42, 13.05, 11.87, 12.07, 
    11.79, 12.04, 12.86, 12.88, 12.7, 12.51, 12.25, 12.53, 12.84, 
    12.93, 13.36, 13.52, 13.62, 12.25, 12.87, 13.32, 12.79, 13.23, 
    13.17, 13.84, 12.45, 14.34, 13.48, 13.69, 12.85, 12.96, 13.78, 
    13.73, 13.58, 13.4, 12.77, 14.16, 13.4, 13.27, 13.17, 14.13
    ), B = 50, path.outfile = NULL, methods = "knockoff", sizeblock = 5, 
        printflag = FALSE, r = c(alco = 0.3), seed = 1234567, 
        stepwise.arg = list(
            scale = 0, direction = "both", trace = 1), nb.clust = 3, 
        modelNames = NULL))), proportion = structure(c(0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.260869565217391, 0, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 0.363636363636364, 1, 0, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 0.45, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
0.333333333333333, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0.1, 1, 
1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0.333333333333333, 1, 1, 1, 
1, 1, 0, 1, 1, 1, 1, 1, 1, 0.142857142857143, 1, 1, 1, 1, 1, 
1, 0, 1, 1, 1, 1, 1, 0.45, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 
0.947368421052632, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0.714285714285714, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0.6, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 0, 1, 0.363636363636364, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 0), dim = c(13L, 13L), dimnames = list(c("alco", "malic", 
"ash", "alca", "mg", "phe", "fla", "nfla", "pro", "col", "hue", 
"ratio", "prol"), c("alco", "malic", "ash", "alca", "mg", "phe", 
"fla", "nfla", "pro", "col", "hue", "ratio", "prol"))), call = list(
    data.na = NULL, res.imputedata = list(res.imp = list(structure(list(
        alco = c(14.23, 13.2, 13.16, 13.82, 13.24, 14.3, 14.39, 
        14.06, 13.24, 13.86, 12.79, 11.45, 13.75, 14.75, 14.38, 
        13.63, 14.3, 13.83, 13.68, 13.64, 13.64, 12.93, 13.71, 
        12.85, 13.5, 13.52, 13.39, 13.3, 13.87, 11.87, 13.73, 
        13.58, 13.68, 13.76, 12.79, 12.85, 13.05, 13.05, 12.64, 
        14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 14.21, 13.24, 
        13.9, 12.33, 14.23, 13.05, 13.71, 13.82, 14.16, 13.74, 
        13.69, 14.22, 13.29, 14.23, 12.37, 12.33, 12.64, 12.42, 
        12.37, 12.17, 12.37, 12.37, 12.77, 13.34, 12.21, 14.23, 
        13.86, 12.33, 12.99, 11.96, 11.66, 11.66, 11.84, 13.48, 
        12.7, 12, 12.72, 12.08, 13.05, 11.84, 11.79, 12.16, 12, 
        12.04, 12.08, 12.08, 12, 12.69, 11.66, 11.62, 11.03, 
        11.81, 12.29, 13.71, 12.29, 12.08, 12.6, 13.62, 12.04, 
        12.51, 13.2, 13.52, 12.72, 12.22, 11.61, 12.42, 11.66, 
        11.76, 12, 12.08, 11.03, 11.82, 12.96, 12.6, 12.08, 11.45, 
        13.39, 12.42, 13.05, 11.87, 12.07, 13.29, 11.79, 13.52, 
        12.04, 12.86, 12.88, 12.69, 12.7, 12.51, 14.22, 12.25, 
        12.53, 13.17, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 
        12.04, 12.53, 12.87, 13.32, 13.27, 12.85, 12.79, 13.62, 
        13.23, 13.29, 13.17, 13.84, 12.45, 14.34, 13.48, 13.63, 
        13.69, 12.85, 12.96, 13.78, 13.73, 13.74, 13.88, 13.58, 
        13.4, 13.62, 12.77, 14.16, 13.75, 13.4, 13.27, 13.17, 
        14.13), malic = c(1.71, 2.56, 1.67, 1.95, 4.3, 3.74, 
        1.87, 2.15, 1.21, 1.35, 2.16, 1.48, 3.12, 1.73, 1.87, 
        3.12, 1.92, 1.51, 1.67, 3.1, 1.63, 3.8, 3.12, 1.6, 1.81, 
        4.04, 1.9, 2.68, 1.9, 4.31, 1.5, 1.53, 1.83, 1.53, 1.81, 
        1.6, 1.48, 1.65, 1.5, 1.97, 1.71, 2.68, 3.74, 3.98, 1.48, 
        4.04, 1.53, 1.68, 1.53, 3.87, 4.31, 1.65, 2.13, 1.9, 
        1.67, 1.36, 1.7, 1.97, 1.97, 0.94, 1.67, 1.36, 1.25, 
        1.73, 1.92, 1.21, 1.01, 1.17, 2.68, 1.35, 2.15, 1.51, 
        1.66, 1.67, 1.09, 1.88, 0.9, 2.56, 0.99, 3.87, 1.73, 
        1.5, 1.13, 3.88, 1.6, 0.98, 1.61, 1.67, 1.92, 2.55, 1.7, 
        1.51, 2.16, 1.67, 1.73, 0.9, 1.66, 1.41, 1.81, 1.5, 2.08, 
        2.68, 2.39, 1.72, 1.67, 2.55, 1.73, 1.9, 1.29, 1.87, 
        3.74, 1.7, 2.68, 0.74, 1.39, 1.5, 1.72, 1.61, 3.43, 1.13, 
        2.4, 2.05, 4.43, 2.67, 4.31, 2.16, 1.53, 2.13, 1.35, 
        4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 4.43, 4.43, 5.51, 
        3.59, 3.88, 2.81, 2.56, 1.29, 1.68, 3.88, 2.31, 5.04, 
        4.61, 3.24, 1.95, 3.12, 2.67, 1.9, 1.24, 1.29, 5.19, 
        4.12, 3.03, 1.68, 1.67, 3.83, 2.59, 2.76, 3.45, 2.76, 
        1.39, 2.99, 3.37, 2.58, 4.6, 1.68, 2.39, 4.1, 5.65, 3.91, 
        3.59, 2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.36, 
        2.5, 2.45, 2.61, 2.4, 2.61, 2.3, 2.74, 2.41, 2.39, 2.38, 
        2.7, 2.72, 2.62, 2.48, 2.56, 2.28, 2.65, 2.68, 2.52, 
        2.61, 3.22, 2.43, 2.64, 2.12, 2.54, 2.24, 2.36, 2.48, 
        2.7, 2.65, 2.4, 1.75, 2.55, 2.1, 2.51, 2.16, 2.12, 2.59, 
        2.29, 2.1, 2.44, 2.14, 2.12, 2.4, 2.27, 2.04, 2.65, 2.6, 
        2.2, 2.38, 2.35, 2.3, 2.68, 2.5, 1.36, 1.99, 2.53, 1.92, 
        2.16, 2.53, 2.56, 2.32, 1.92, 2.32, 1.75, 2.21, 2.67, 
        2.24, 2.6, 2.3, 2, 1.71, 1.94, 2.4, 2.4, 2, 2.53, 2.51, 
        2.32, 2.58, 2.24, 2.16, 2.62, 2.1, 2.3, 2.32, 1.88, 2.26, 
        2.22, 2.28, 2.2, 2.74, 1.98, 2.54, 1.99, 1.7, 1.7, 2.46, 
        1.88, 2.65, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 
        2.92, 2.5, 2.5, 2.2, 1.99, 2.5, 2.28, 2, 2.42, 3.23, 
        1.7, 2.13, 2.39, 2.17, 2.29, 2.12, 1.88, 2.38, 2.2, 2.4, 
        2.4, 2.36, 2.25, 2.26, 2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 
        2.72, 2.35, 2.2, 2.12, 1.94, 2.48, 2.58, 2.36, 2.62, 
        2.48, 2.22, 2.28, 1.99, 2.32, 2.38, 1.71, 2.5, 2.64, 
        2.38, 2.54, 2.42, 2.35, 2.3, 2.4, 2.6, 2.35, 2.69, 2.59, 
        2.32, 2.28, 1.7, 2.61, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 
        16, 24, 16.8, 21, 15.2, 16.3, 17.6, 14, 16, 18, 16.8, 
        16, 11.4, 15.2, 16, 19, 20, 15.5, 15.2, 16, 18.6, 21, 
        17.8, 20, 25, 16.1, 17, 16, 20, 15.5, 19.1, 16.8, 17, 
        19, 20.5, 15.5, 18.5, 15.5, 13.2, 16.2, 18.8, 19.5, 20.5, 
        17, 17.8, 16, 16, 18.8, 17.4, 12.4, 18, 16, 12.4, 16.8, 
        20.5, 16.3, 16.8, 16, 10.6, 19.6, 16.8, 18, 19, 19, 18.1, 
        18, 19.6, 18.5, 16.8, 20.4, 16.8, 24, 30, 21, 16.8, 16, 
        18, 14.8, 22.5, 19, 19.6, 24, 22.5, 18, 18, 22.8, 19, 
        21.6, 18, 19.5, 22, 22, 18, 18, 19, 21.5, 16, 18, 18, 
        17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 
        18.5, 21.6, 22, 16.8, 21.5, 20.8, 21, 16, 22.8, 23.5, 
        17.8, 19.5, 21.5, 21, 21, 21.5, 28.5, 18, 22, 24, 20, 
        24, 21.5, 20, 24, 19.5, 25, 19.5, 24, 21, 21, 23.5, 20, 
        24, 21, 20, 24.5, 22, 21.5, 24, 22, 21.6, 18.5, 20, 22, 
        19.5, 16.2, 25, 24, 23.5, 21, 20, 18.5, 20.4, 22.5, 23, 
        19.5, 24.5, 25, 22.5, 19.5, 21, 20.5, 23, 20, 21.5, 24.5
        ), mg = c(127, 100, 101, 113, 101, 112, 96, 121, 97, 
        118, 91, 95, 89, 91, 102, 112, 120, 96, 106, 116, 126, 
        102, 97, 95, 95, 124, 93, 97, 107, 96, 101, 106, 104, 
        96, 110, 110, 110, 118, 93, 97, 117, 93, 101, 103, 107, 
        111, 102, 101, 101, 108, 92, 110, 111, 115, 117, 89, 
        118, 102, 108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 
        84, 151, 96, 86, 87, 139, 101, 87, 88, 112, 98, 100, 
        86, 86, 78, 85, 84, 99, 90, 88, 84, 70, 81, 78, 124, 
        88, 84, 122, 134, 85, 88, 101, 97, 88, 88, 86, 85, 90, 
        80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 151, 86, 
        87, 96, 112, 102, 86, 70, 85, 117, 92, 88, 80, 122, 88, 
        98, 106, 102, 94, 89, 102, 88, 101, 88, 89, 97, 97, 112, 
        102, 86, 84, 92, 113, 123, 107, 116, 98, 103, 112, 89, 
        97, 98, 89, 91, 70, 106, 106, 90, 88, 106, 88, 105, 112, 
        96, 86, 91, 95, 102, 97, 120, 106), phe = c(2.8, 2.65, 
        2.8, 3.85, 2.8, 3.52, 2.5, 2.6, 3.3, 2.98, 2.95, 2.2, 
        2.6, 3.1, 3.52, 2.85, 2.8, 3.25, 3.3, 2.45, 3, 2.41, 
        2.61, 2.48, 2.53, 2.2, 2.8, 2.4, 2.95, 2.65, 3, 2.8, 
        2.42, 2.95, 2.64, 2.7, 2.6, 2.45, 2.65, 2.8, 3.15, 2.45, 
        3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 
        3.88, 3.02, 2.75, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 
        1.48, 3.5, 1.89, 2.6, 2.2, 2.11, 2.53, 1.65, 1.1, 2.95, 
        1.88, 1.65, 1.41, 1.48, 2.74, 1.48, 1.9, 2.83, 2.2, 2.2, 
        1.68, 1.65, 2, 2.2, 1.48, 1.92, 1.95, 2.2, 1.6, 1.45, 
        1.38, 2.45, 3.02, 2.65, 1.25, 2.55, 3.52, 2.95, 2.05, 
        2, 2.56, 2.5, 2.2, 1.65, 1.65, 2.11, 2.36, 2.74, 3.18, 
        2.55, 1.75, 1.65, 2.56, 2.46, 2.22, 2, 2.02, 2, 2.9, 
        3.18, 1.48, 1.38, 2.86, 2.8, 2.74, 2.13, 2.22, 2.1, 1.51, 
        1.92, 1.25, 1.7, 2.05, 1.62, 1.38, 2.3, 1.62, 1.65, 0.98, 
        1.4, 1.25, 2, 1.38, 1.55, 0.98, 1.7, 1.93, 1.41, 1.4, 
        1.48, 2.2, 1.48, 1.48, 1.98, 1.8, 1.9, 2.1, 1.92, 2.3, 
        1.83, 0.98, 1.39, 1.35, 1.65, 1.7, 1.48, 1.55, 1.48, 
        1.25, 1.39, 1.68, 1.48, 1.65, 1.48, 1.65, 2.05), fla = c(2.92, 
        2.76, 3.29, 3.49, 2.17, 3.39, 2.52, 2.51, 2.98, 3.74, 
        3.32, 2.43, 2.76, 2.99, 3.64, 2.91, 3.14, 3.4, 3.4, 2.68, 
        3.67, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.27, 2.97, 
        2.33, 3.25, 3.19, 2.69, 3.17, 2.99, 2.98, 2.53, 2.43, 
        2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 2.92, 
        2.92, 3.54, 2.89, 2.99, 3.74, 2.79, 2.9, 2.78, 2.92, 
        3.23, 3.67, 0.57, 3.15, 1.41, 1.79, 3.25, 1.75, 2.65, 
        2.41, 2, 1.3, 1.64, 1.02, 2.86, 2.27, 2.89, 2.14, 1.57, 
        2.68, 0.57, 1.85, 2.33, 2.26, 2.53, 1.58, 1.59, 2.21, 
        2.63, 1.69, 1.61, 1.69, 1.61, 1.69, 1.25, 1.46, 2.61, 
        3.39, 2.27, 0.99, 2.89, 3.74, 2.99, 2.17, 1.36, 2.11, 
        1.64, 1.92, 1.84, 2.03, 1.76, 2.03, 2.92, 2.68, 2.27, 
        2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.2, 1.64, 2.92, 3.4, 
        1.25, 0.68, 3.03, 2.65, 3.15, 2.24, 1.75, 1.75, 1.46, 
        1.57, 0.55, 1.2, 0.58, 0.66, 0.47, 1.57, 0.52, 0.6, 0.5, 
        0.76, 0.52, 0.8, 0.69, 0.55, 0.34, 0.65, 0.76, 1.39, 
        1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.8, 0.58, 1.31, 
        1.1, 0.92, 1.1, 0.6, 1.36, 0.68, 0.47, 0.92, 0.66, 0.84, 
        0.96, 0.49, 0.6, 0.7, 0.61, 0.58, 0.69, 0.68, 0.76), 
        nfla = c(0.28, 0.13, 0.3, 0.24, 0.39, 0.32, 0.37, 0.31, 
        0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.27, 
        0.39, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.34, 0.3, 
        0.34, 0.27, 0.28, 0.26, 0.29, 0.3, 0.42, 0.39, 0.29, 
        0.26, 0.3, 0.31, 0.28, 0.2, 0.34, 0.26, 0.32, 0.32, 0.28, 
        0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 
        0.21, 0.2, 0.26, 0.24, 0.32, 0.28, 0.24, 0.53, 0.32, 
        0.24, 0.45, 0.37, 0.26, 0.27, 0.61, 0.22, 0.39, 0.22, 
        0.42, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 
        0.32, 0.4, 0.61, 0.22, 0.3, 0.22, 0.4, 0.34, 0.4, 0.29, 
        0.5, 0.2, 0.25, 0.17, 0.32, 0.28, 0.22, 0.24, 0.25, 0.3, 
        0.29, 0.34, 0.37, 0.26, 0.66, 0.32, 0.48, 0.39, 0.29, 
        0.24, 0.26, 0.34, 0.42, 0.32, 0.52, 0.3, 0.34, 0.27, 
        0.26, 0.32, 0.43, 0.43, 0.3, 0.24, 0.32, 0.39, 0.29, 
        0.4, 0.27, 0.39, 0.34, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 
        0.58, 0.53, 0.53, 0.6, 0.43, 0.3, 0.58, 0.43, 0.4, 0.47, 
        0.45, 0.4, 0.22, 0.29, 0.26, 0.61, 0.53, 0.43, 0.61, 
        0.63, 0.53, 0.52, 0.58, 0.5, 0.6, 0.4, 0.41, 0.63, 0.43, 
        0.4, 0.39, 0.53, 0.53, 0.48, 0.6, 0.6, 0.43, 0.43, 0.53, 
        0.43), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 
        1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 2.03, 2.96, 1.46, 
        1.46, 1.86, 1.86, 1.66, 2.1, 1.54, 1.69, 1.46, 1.66, 
        1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.86, 1.85, 
        1.54, 1.86, 1.36, 1.44, 1.37, 1.14, 2.81, 1.42, 1.69, 
        1.38, 2.03, 1.35, 2.19, 2.14, 2.38, 2.08, 2.1, 2.29, 
        1.87, 1.68, 1.62, 2.45, 2.7, 1.66, 2.91, 0.42, 1.35, 
        1.62, 0.73, 1.87, 1.03, 1.77, 1.95, 1.04, 0.73, 2.5, 
        1.46, 1.87, 1.03, 2.81, 1.04, 1.15, 1.46, 0.84, 2.76, 
        1.95, 1.43, 1.45, 1.46, 1.62, 1.35, 1.46, 1.56, 1.34, 
        1.35, 1.38, 1.64, 1.63, 1.15, 1.99, 1.35, 3.28, 1.56, 
        1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.04, 1.42, 
        0.55, 1.63, 2.08, 2.49, 1.9, 1.14, 1.42, 2.08, 1.04, 
        1.95, 1.46, 1.61, 0.83, 1.87, 1.83, 1.86, 1.71, 0.83, 
        2.91, 1.35, 1.77, 2.5, 1.9, 1.55, 0.94, 1.62, 0.83, 0.84, 
        1.25, 0.94, 0.8, 1.1, 0.88, 1.64, 0.75, 1.14, 0.55, 0.42, 
        1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 
        1.35, 1.25, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.9, 0.94, 
        0.94, 1.56, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 
        1.24, 1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 4.38, 
        5.68, 7.8, 4.9, 6.75, 6.3, 5.05, 5.4, 7.22, 5.75, 3.9, 
        5.6, 9.2, 7.5, 5.24, 8.42, 4.5, 8.7, 5.1, 5.65, 4.5, 
        4.6, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 8.7, 
        5.58, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 
        5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 4.28, 8.9, 7.2, 4.8, 
        4.9, 6.3, 5.85, 6.25, 7.65, 5.05, 6.8, 3.4, 3.27, 5.75, 
        3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 7.7, 3.38, 
        3.74, 4.7, 3.21, 3.8, 4.6, 2.6, 6.8, 3.4, 2.5, 3.9, 3.21, 
        4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 
        2.15, 3.25, 2.6, 4.38, 2.9, 4.7, 5.1, 4.6, 5.1, 2.8, 
        2.6, 2.94, 2.7, 3.4, 3.3, 2.7, 2.8, 2.9, 2.5, 3.7, 3.08, 
        2.9, 1.9, 3.21, 2.7, 3.4, 1.28, 3.25, 3.38, 4.5, 5.6, 
        2.8, 3.21, 3.94, 4.6, 2.12, 2.7, 3.21, 5.58, 3.52, 5, 
        5.45, 7.1, 5.6, 5, 5.7, 4.92, 6.25, 3.94, 4.35, 4.4, 
        8.21, 4, 4.9, 5.75, 8.42, 9.4, 1.74, 4.8, 7.1, 10.52, 
        7.6, 7.9, 9.2, 7.5, 13, 7.9, 7.65, 5.7, 5.58, 5.58, 7.7, 
        6.62, 10.68, 10.26, 5.4, 8.5, 4.8, 4.28, 9.7, 7.7, 7.3, 
        7.1, 9.3, 9.2), hue = c(1.04, 1.05, 1.08, 0.86, 0.92, 
        1.05, 1.02, 0.98, 1.08, 1.18, 1.25, 1.15, 1.15, 1.25, 
        1.2, 1.28, 1.19, 1.13, 1.23, 1.02, 1.09, 0.98, 1.11, 
        1.09, 1.12, 1.13, 1.08, 1.02, 0.89, 1.04, 1.19, 1.09, 
        1.19, 1.25, 1.1, 0.94, 1.28, 0.86, 1.18, 1.15, 0.95, 
        0.91, 0.88, 0.82, 1.07, 0.87, 0.86, 0.86, 1.07, 1.12, 
        1.12, 1.71, 1.28, 1.28, 0.9, 1.04, 0.94, 1.07, 0.89, 
        0.87, 1.25, 0.98, 1.23, 1.22, 0.98, 1.1, 1.06, 1.12, 
        1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 1.1, 1.23, 1.19, 
        0.96, 1.06, 1.19, 1.38, 1.23, 1.31, 0.84, 1.1, 1.23, 
        1.12, 0.91, 1.09, 1.07, 1.08, 1.05, 1.13, 1.15, 1.23, 
        1.12, 0.92, 1.23, 1.07, 1.42, 1.27, 1.19, 0.8, 1.19, 
        1.04, 1.06, 1, 1.12, 0.86, 0.89, 0.82, 0.9, 1.13, 1.1, 
        0.93, 1.71, 0.95, 1.05, 0.7, 0.95, 0.8, 1.25, 0.92, 0.73, 
        1.71, 0.86, 0.69, 0.97, 0.89, 0.79, 0.84, 0.74, 0.66, 
        0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.65, 0.6, 0.96, 
        0.89, 0.67, 0.65, 0.6, 0.6, 0.54, 0.55, 0.67, 0.94, 0.48, 
        0.61, 0.56, 0.78, 0.6, 0.57, 0.67, 0.57, 0.75, 0.56, 
        0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.61, 0.74, 0.67, 
        0.906, 0.96, 0.56, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
        3.4, 3.17, 2.96, 2.93, 3.3, 3.58, 3.58, 2.85, 3.55, 3.17, 
        2.82, 2.9, 2.5, 3, 4, 2.65, 3.69, 2.82, 3.36, 3.71, 3.52, 
        4, 3.63, 3.64, 3.2, 2.84, 2.77, 3.71, 2.14, 2.71, 2.88, 
        2.87, 3, 2.87, 3, 3.14, 2.87, 2.69, 3.5, 3.38, 3, 3.26, 
        3, 3.35, 3.33, 3.17, 3.33, 2.75, 3.1, 3.12, 3.5, 3.26, 
        2.93, 3.2, 3.03, 2.77, 2.84, 2.87, 1.82, 3.71, 1.59, 
        2.46, 2.87, 3.52, 2.3, 3.18, 3.48, 3.48, 2.82, 1.82, 
        3.26, 2.78, 3.5, 3.13, 2.14, 2.48, 2.52, 3.33, 3.17, 
        2.78, 3.14, 2.72, 2.01, 3.57, 2.84, 3.39, 3.21, 2.75, 
        3.21, 2.78, 2.65, 2.82, 3.3, 3, 2.63, 1.78, 2.74, 3.71, 
        2.83, 2.96, 2.77, 3.38, 2.65, 3.57, 3.3, 3.17, 2.42, 
        3.02, 3.26, 2.81, 2.78, 2.5, 2.96, 3.19, 2.87, 3.33, 
        2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 1.68, 3.64, 3.14, 
        2.84, 3.21, 2.78, 2.01, 2.46, 1.42, 1.36, 1.29, 1.51, 
        1.58, 1.27, 1.69, 1.82, 2.15, 1.56, 2.01, 2.14, 2.05, 
        2, 1.58, 1.33, 1.75, 1.62, 1.33, 2.87, 1.47, 1.33, 1.51, 
        1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 2.77, 1.36, 
        1.75, 1.68, 1.75, 1.96, 2, 3.12, 1.78, 2.77, 1.69, 1.51, 
        1.27, 1.27, 1.56, 1.62, 1.6), prol = c(1065, 1050, 1035, 
        1065, 735, 1450, 1290, 1295, 1045, 1045, 410, 1280, 1320, 
        1450, 1547, 1310, 1280, 1120, 1680, 845, 760, 770, 1035, 
        1015, 845, 880, 1195, 1285, 915, 1035, 1285, 1150, 1150, 
        1235, 660, 920, 880, 880, 714, 760, 795, 1035, 1095, 
        680, 885, 1080, 1065, 985, 1060, 1280, 1150, 1265, 1190, 
        1235, 1060, 1120, 970, 1270, 1285, 312, 685, 1035, 630, 
        480, 428, 885, 502, 345, 380, 718, 870, 410, 472, 735, 
        886, 428, 392, 680, 1150, 463, 278, 714, 495, 515, 680, 
        680, 880, 562, 680, 625, 480, 450, 495, 615, 345, 625, 
        625, 428, 660, 406, 502, 562, 438, 415, 672, 625, 510, 
        480, 312, 680, 562, 325, 685, 434, 385, 580, 580, 625, 
        372, 564, 345, 1080, 365, 380, 675, 378, 1035, 466, 438, 
        580, 415, 1120, 560, 600, 580, 695, 720, 515, 580, 590, 
        600, 502, 520, 550, 840, 830, 415, 415, 615, 550, 500, 
        480, 425, 675, 640, 428, 480, 880, 660, 406, 510, 680, 
        570, 1060, 615, 365, 695, 685, 590, 1035, 510, 470, 1035, 
        740, 625, 835, 840, 428)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.16, 13.24, 
        13.82, 14.39, 14.06, 13.4, 13.86, 14.38, 13.05, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 13.82, 13.64, 13.86, 
        12.93, 13.71, 12.85, 13.5, 12.37, 13.39, 13.3, 13.87, 
        13.87, 13.73, 13.58, 13.68, 13.76, 13.05, 11.76, 12.84, 
        13.05, 12.04, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 13.58, 13.9, 12.93, 13.29, 13.05, 13.05, 13.82, 
        13.64, 13.74, 14.16, 14.22, 13.29, 14.06, 12.37, 12.33, 
        12.64, 12.08, 12.37, 12.17, 12.37, 13.2, 13.36, 13.34, 
        12.21, 11.82, 13.86, 12.72, 12.99, 11.96, 11.66, 12.21, 
        11.84, 13.3, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 13.69, 
        12.16, 11.45, 11.96, 12.08, 12.08, 12, 12.69, 11.96, 
        11.62, 12.17, 11.81, 12.29, 12.88, 12.29, 12.08, 12.6, 
        13.05, 11.79, 12.51, 12.21, 11.79, 12.72, 12.22, 11.61, 
        12.6, 12.99, 11.76, 12.99, 12.08, 11.03, 11.82, 12.7, 
        12.29, 12, 11.45, 12.45, 12.42, 13.05, 11.87, 12.07, 
        12.22, 11.79, 12.72, 12.04, 12.86, 12.88, 12, 12.7, 12.51, 
        12.85, 12.25, 12.53, 11.79, 12.84, 12.93, 13.36, 13.52, 
        13.62, 12.25, 12.37, 12.6, 12.87, 13.32, 12.42, 12.72, 
        12.79, 12.72, 13.23, 13.05, 13.17, 13.84, 12.45, 14.34, 
        13.48, 12.79, 13.69, 12.85, 12.96, 13.78, 13.73, 12.45, 
        14.22, 13.58, 13.4, 12.08, 12.77, 14.16, 13.29, 13.4, 
        13.27, 13.17, 14.13), malic = c(1.71, 1.83, 0.99, 1.95, 
        3.37, 1.95, 1.87, 2.15, 1.13, 1.35, 2.16, 1.48, 4.31, 
        1.73, 1.87, 1.29, 1.92, 1.24, 3.45, 3.1, 1.63, 3.8, 1.7, 
        1.6, 1.81, 2.31, 2.56, 2.56, 1.9, 1.13, 1.5, 2.15, 1.83, 
        1.53, 1.68, 0.74, 0.99, 1.65, 1.5, 1.53, 1.71, 1.73, 
        3.98, 3.98, 1.66, 4.04, 1.29, 1.68, 1.87, 1.97, 1.6, 
        1.65, 4.43, 1.9, 1.67, 1.13, 1.7, 1.97, 1.9, 0.94, 3.12, 
        1.36, 1.25, 1.51, 1.67, 1.21, 1.01, 1.17, 1.5, 1.17, 
        1.61, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 4.6, 0.99, 3.87, 
        1.41, 1.87, 1.13, 4.31, 1.61, 0.98, 1.61, 1.67, 2.39, 
        0.74, 1.72, 1.51, 1.83, 2.08, 1.92, 1.5, 1.65, 1.41, 
        1.6, 1.87, 2.08, 4.31, 2.99, 1.72, 0.94, 2.55, 1.73, 
        2.08, 1.29, 5.19, 3.74, 1.36, 2.68, 0.74, 1.39, 1.41, 
        4.04, 1.61, 3.43, 1.41, 2.4, 2.05, 4.43, 4.04, 4.31, 
        2.16, 1.53, 2.13, 3.12, 4.3, 1.35, 2.99, 2.31, 3.55, 
        1.24, 5.04, 4.61, 5.51, 3.59, 3.24, 2.81, 2.56, 1.95, 
        3.87, 3.88, 4.61, 5.04, 4.61, 3.24, 3.24, 3.12, 2.67, 
        1.9, 3.83, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 
        2.15, 5.19, 3.45, 2.76, 1.68, 2.59, 3.37, 2.58, 4.6, 
        4.04, 2.39, 4.61, 5.65, 3.91, 2.76, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.4, 2.6, 2.45, 2.61, 1.82, 2.48, 2.3, 
        2.32, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 
        2.61, 2.65, 2.51, 2.52, 2.61, 3.22, 2.4, 2.52, 2.28, 
        2.3, 2.65, 2.36, 2.44, 2.7, 2.65, 2.36, 1.88, 2.55, 2.1, 
        2.51, 2.2, 2.12, 2.59, 2.29, 2.1, 2.44, 2.24, 2.12, 2.4, 
        2.27, 2.04, 2.59, 2.37, 2.36, 2.1, 2.92, 2.3, 2.68, 2.5, 
        1.36, 2.24, 2.1, 1.92, 2.16, 2.53, 2.56, 2.52, 1.92, 
        1.92, 1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.65, 1.71, 1.75, 
        2.29, 2.4, 2, 1.71, 2.51, 2.32, 2.58, 2.24, 2.21, 2.62, 
        2.3, 2.3, 2.32, 2.19, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 
        1.82, 2.13, 1.7, 2.2, 2.46, 1.88, 2.3, 2.27, 2.12, 2.28, 
        1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.62, 
        1.71, 2, 2.42, 3.23, 2.35, 2.13, 2.39, 2.17, 2.29, 2.32, 
        2.4, 2.38, 2.48, 2.4, 2.4, 2.36, 2.25, 2.5, 2.54, 2.64, 
        2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.36, 1.75, 2.48, 
        2.4, 2.36, 2.62, 2.48, 1.75, 2.28, 1.99, 2.32, 2.38, 
        2.36, 3.23, 2.64, 2.38, 2.54, 2.62, 2.35, 2.3, 2.4, 2.6, 
        2.35, 2.69, 2.68, 2.32, 2.28, 2.32, 2.45, 2.48, 2.26, 
        2.37, 2.74), alca = c(15.6, 13.2, 19, 16.8, 21, 15.2, 
        16.8, 17.6, 14, 16, 12.4, 16.8, 16, 11.4, 14, 21, 19, 
        20, 17.6, 15.2, 16, 18.6, 21.5, 17.8, 20, 23, 16.1, 17, 
        18, 15.6, 16.8, 19.1, 19.5, 16, 19, 20.5, 15.5, 24, 15.5, 
        13.2, 16.2, 18.8, 15.2, 19, 17, 18.5, 16, 16, 18.8, 17.4, 
        12.4, 17.5, 18.8, 18, 10.6, 20.5, 16.3, 16.8, 16, 10.6, 
        20.5, 16.8, 18, 19, 19, 18.1, 19, 19.6, 16.8, 16.8, 20.4, 
        21.5, 24, 30, 21, 17.8, 16, 18, 14.8, 25, 19, 16.8, 24, 
        22.5, 18, 18, 22.8, 23.5, 21.6, 18, 18.5, 22, 20.8, 18, 
        18, 19, 21.5, 16, 10.6, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 19.5, 25, 24, 21.5, 21.5, 
        20.8, 19, 16, 19.5, 22, 18.5, 25, 21.5, 21, 21, 21.5, 
        28.5, 22, 22, 20, 20, 24, 21.5, 21, 28.5, 21.5, 25, 19.5, 
        25, 21, 20, 23.5, 20, 30, 21, 20, 24, 18.1, 21.5, 24, 
        22, 19.5, 18.5, 20, 22, 19.5, 22, 25, 17, 22, 24, 25, 
        18.5, 20, 22.5, 23, 19.5, 24.5, 25, 24.5, 19.5, 20, 20.5, 
        23, 20, 23, 24.5), mg = c(127, 100, 101, 113, 117, 112, 
        96, 121, 97, 89, 102, 112, 89, 91, 102, 112, 120, 113, 
        124, 116, 126, 102, 105, 95, 102, 124, 93, 107, 107, 
        96, 101, 106, 104, 127, 110, 118, 110, 124, 101, 124, 
        117, 112, 101, 87, 107, 111, 102, 101, 112, 108, 92, 
        112, 111, 115, 99, 111, 118, 102, 108, 88, 101, 100, 
        94, 87, 104, 98, 78, 78, 106, 151, 100, 86, 87, 139, 
        101, 122, 96, 112, 97, 98, 86, 106, 78, 85, 121, 99, 
        90, 88, 84, 70, 81, 88, 84, 88, 88, 89, 134, 85, 88, 
        94, 97, 88, 87, 86, 85, 90, 80, 84, 92, 94, 107, 88, 
        103, 88, 84, 85, 86, 122, 80, 87, 96, 124, 102, 86, 96, 
        85, 98, 92, 88, 80, 122, 102, 98, 106, 102, 94, 89, 87, 
        88, 101, 88, 89, 97, 92, 112, 102, 90, 87, 92, 113, 123, 
        107, 116, 98, 103, 94, 89, 97, 98, 89, 98, 100, 106, 
        106, 90, 88, 98, 88, 105, 112, 96, 86, 91, 95, 102, 106, 
        120, 80), phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 2.95, 2.5, 
        2.6, 3, 2.98, 2.95, 2.2, 2.6, 3.1, 3.52, 2.85, 2.8, 3.1, 
        3.3, 3.15, 3, 2.41, 2.61, 2.48, 2.53, 2.2, 2.55, 2.4, 
        2.95, 2.65, 3, 2.8, 2.42, 2.95, 1.41, 2.7, 2.6, 2.45, 
        2.46, 3.1, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
        2.75, 2.88, 2.72, 2.45, 3.88, 2.65, 2.95, 2.96, 3.2, 
        3, 3.4, 1.98, 2.05, 2.02, 1.92, 3.5, 1.89, 2.7, 2, 2.11, 
        2.53, 2.55, 1.1, 2.95, 1.88, 2.65, 1.41, 2.3, 2.45, 2.2, 
        1.9, 2.83, 2.11, 2.2, 1.98, 1.65, 2.53, 2.2, 1.75, 1.92, 
        1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.45, 1.1, 2.55, 
        3.52, 2.86, 2.4, 1.41, 2.56, 2.5, 2.2, 2.5, 1.65, 2.2, 
        2.36, 2.74, 3.18, 2.55, 1.75, 1.65, 2.56, 2.46, 2.05, 
        2, 1.48, 2, 2.9, 3.18, 2.22, 1.48, 2.86, 2.45, 2.74, 
        2.13, 2.22, 2.1, 1.51, 1.93, 1.9, 1.7, 1.9, 1.62, 1.38, 
        1.62, 1.62, 1.62, 0.98, 1.4, 1.1, 2, 1.38, 1.25, 0.98, 
        1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.92, 1.48, 2.3, 1.8, 
        1.9, 1.9, 2.53, 2.3, 1.83, 1.55, 1.39, 1.35, 1.55, 1.7, 
        1.48, 1.55, 1.4, 1.25, 1.39, 1.68, 1.6, 2.2, 1.39, 1.65, 
        2.05), fla = c(2.88, 2.76, 3, 3.49, 2.64, 3.39, 2.52, 
        2.51, 2.98, 3.32, 3.32, 2.43, 2.76, 3.14, 3.64, 2.91, 
        3.14, 3.4, 3.67, 2.78, 3.03, 2.41, 2.88, 2.37, 2.61, 
        2.68, 2.94, 2.64, 2.97, 2.33, 3.25, 3.19, 2.69, 3.29, 
        1.75, 2.98, 2.61, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 
        2.63, 3, 2.65, 3.17, 3.4, 2.92, 3.54, 2.99, 2.99, 3.74, 
        2.79, 2.9, 2.78, 3.54, 3.23, 3.67, 0.57, 1.25, 1.41, 
        1.79, 3, 1.75, 2.65, 2.27, 2, 1.3, 2.52, 1.02, 2.86, 
        0.8, 2.89, 2.14, 1.57, 2.65, 2.03, 1.85, 2.53, 2.26, 
        2.53, 1.58, 1.59, 2.21, 2.63, 1.69, 1.61, 1.69, 1.64, 
        1.25, 1.25, 1.46, 2.52, 2.65, 2.27, 0.99, 2.68, 3.39, 
        2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 
        2.65, 2.92, 3.54, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 
        2.09, 1.39, 1.64, 2.97, 3.74, 1.57, 0.69, 3.03, 2.65, 
        3.15, 2.24, 2.26, 1.75, 1.79, 0.96, 1.75, 1.2, 0.58, 
        0.66, 0.47, 1.39, 0.7, 0.6, 0.5, 0.58, 0.52, 0.8, 0.68, 
        0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
        0.58, 0.63, 0.68, 0.58, 1.31, 1.1, 0.92, 0.92, 0.6, 1.36, 
        0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.66, 0.7, 
        0.61, 1.36, 0.69, 0.68, 0.76), nfla = c(0.28, 0.21, 0.3, 
        0.24, 0.39, 0.21, 0.37, 0.31, 0.29, 0.22, 0.22, 0.26, 
        0.29, 0.43, 0.29, 0.3, 0.26, 0.21, 0.32, 0.17, 0.24, 
        0.25, 0.27, 0.26, 0.32, 0.26, 0.34, 0.27, 0.31, 0.26, 
        0.29, 0.26, 0.42, 0.17, 0.29, 0.26, 0.28, 0.26, 0.28, 
        0.2, 0.34, 0.21, 0.21, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
        0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.22, 
        0.24, 0.28, 0.26, 0.53, 0.32, 0.42, 0.45, 0.37, 0.26, 
        0.27, 0.29, 0.24, 0.43, 0.37, 0.63, 0.21, 0.13, 0.34, 
        0.24, 0.43, 0.35, 0.43, 0.3, 0.24, 0.4, 0.61, 0.22, 0.3, 
        0.42, 0.26, 0.48, 0.3, 0.4, 0.5, 0.45, 0.25, 0.17, 0.32, 
        0.37, 0.26, 0.24, 0.13, 0.21, 0.29, 0.34, 0.37, 0.27, 
        0.66, 0.3, 0.48, 0.39, 0.29, 0.24, 0.26, 0.53, 0.42, 
        0.26, 0.52, 0.3, 0.3, 0.32, 0.25, 0.32, 0.22, 0.43, 0.3, 
        0.37, 0.24, 0.39, 0.43, 0.4, 0.61, 0.43, 0.52, 0.27, 
        0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.63, 
        0.29, 0.53, 0.43, 0.43, 0.4, 0.47, 0.45, 0.26, 0.22, 
        0.32, 0.26, 0.61, 0.53, 0.53, 0.53, 0.63, 0.53, 0.52, 
        0.45, 0.5, 0.6, 0.4, 0.41, 0.5, 0.43, 0.4, 0.39, 0.63, 
        0.53, 0.48, 0.6, 0.45, 0.43, 0.43, 0.53, 0.53), pro = c(2.29, 
        1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
        2.38, 1.57, 1.81, 2.45, 2.96, 1.46, 2.38, 1.87, 1.86, 
        1.66, 2.1, 1.36, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
        1.76, 1.98, 2.38, 1.95, 1.69, 2.7, 1.54, 1.86, 1.36, 
        1.44, 1.37, 2.1, 1.87, 1.9, 1.97, 1.46, 2.03, 1.95, 2.19, 
        2.14, 2.38, 2.08, 3.28, 2.29, 1.87, 1.68, 1.62, 2.45, 
        2.96, 1.66, 2.08, 0.42, 0.84, 1.31, 0.73, 1.87, 1.03, 
        2.38, 1.43, 1.04, 1.25, 2.5, 1.46, 1.87, 1.03, 1.61, 
        1.56, 1.15, 1.46, 1.14, 2.76, 1.95, 1.43, 0.73, 1.31, 
        1.62, 1.41, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 1.63, 
        1.1, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 
        1.31, 1.42, 1.42, 1.42, 1.1, 1.63, 2.08, 2.49, 1.66, 
        1.9, 2.38, 1.61, 1.04, 2.49, 1.25, 1.61, 0.83, 1.87, 
        1.83, 2.08, 1.71, 1.3, 2.91, 1.35, 1.77, 1.98, 1.9, 1.81, 
        0.94, 1.11, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.64, 
        0.75, 1.25, 0.55, 1.87, 1.14, 1.3, 0.68, 0.86, 1.25, 
        1.14, 1.25, 1.26, 1.56, 1.35, 0.73, 1.55, 1.56, 1.14, 
        2.7, 2.29, 1.04, 1.35, 1.64, 0.94, 0.94, 1.15, 1.46, 
        0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
        1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.9, 6.75, 
        4.8, 5.05, 5.64, 7.22, 5.75, 3.94, 5.6, 4, 7.5, 3.94, 
        7.5, 5.7, 8.7, 5.1, 5.65, 4.5, 4.6, 3.93, 3.52, 3.58, 
        4.8, 3.95, 4.5, 4.7, 5.7, 5.68, 6.1, 5.4, 4.2, 5.1, 4.6, 
        4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 
        6.1, 4.6, 8.9, 7.2, 5.68, 4.7, 6.3, 5.85, 6.25, 7.1, 
        4.36, 6.8, 4.36, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 
        4.68, 3.17, 2.85, 4.8, 3.38, 3.74, 1.74, 3.21, 3.8, 4.6, 
        4.5, 7.1, 4.25, 2.5, 3.9, 2.8, 4.8, 3.05, 2.62, 2.45, 
        2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 5.05, 
        2.9, 3.21, 5.3, 4.68, 2.6, 2.8, 3.25, 2.94, 2.7, 3.4, 
        3.3, 2.7, 3.93, 2.9, 4.38, 3.4, 3.08, 2.9, 1.9, 4.68, 
        2.4, 3.4, 1.28, 3.25, 3.21, 3.27, 3.05, 2.8, 2.15, 3.94, 
        2.9, 2.12, 3.74, 4.45, 5.7, 4.38, 5, 5.45, 7.1, 6.75, 
        5, 5.7, 4.92, 5.58, 6.62, 4.35, 4.4, 8.21, 4, 4.9, 5.75, 
        8.42, 9.4, 2.8, 5.58, 7.1, 10.52, 7.6, 7.9, 9.2, 7.5, 
        13, 10.26, 7.65, 5.75, 5.58, 5.75, 6.3, 6.62, 10.68, 
        10.26, 7.1, 8.5, 4.36, 4.28, 9.7, 7.7, 7.3, 7.9, 9.3, 
        9.2), hue = c(1.04, 1.05, 1.19, 0.86, 0.87, 1.05, 1.02, 
        1.71, 1.08, 1.25, 1.25, 1.13, 1.15, 1.25, 1.2, 1.28, 
        0.89, 1.13, 1.23, 1.09, 1.09, 0.7, 1.11, 1.09, 1.12, 
        1.13, 1.25, 1.02, 1.09, 1.04, 1.19, 1.09, 0.79, 1.25, 
        1.1, 0.86, 0.98, 0.86, 1.18, 1.2, 0.95, 0.91, 0.88, 0.82, 
        1.07, 0.87, 1.71, 1.25, 1.07, 1.12, 1.12, 0.93, 1.25, 
        0.89, 1.05, 1.1, 0.94, 1.07, 0.89, 0.82, 1.25, 0.98, 
        1.23, 1.22, 1.07, 1.36, 1.1, 1.12, 1.02, 1.28, 0.906, 
        1.36, 0.98, 1.31, 1.05, 1.23, 1.19, 0.96, 1.06, 1.19, 
        1.38, 1, 1.31, 0.84, 1.28, 1.23, 0.9, 1.12, 0.86, 1.07, 
        1.08, 1.05, 0.96, 1.15, 0.8, 0.8, 0.75, 0.8, 1.09, 1.42, 
        1.27, 0.82, 0.8, 0.69, 1.04, 0.96, 1, 0.79, 0.86, 1.42, 
        1.1, 0.9, 1.02, 1.1, 0.93, 1.71, 0.95, 1.05, 0.7, 1.28, 
        0.8, 1.25, 0.92, 0.73, 0.86, 0.86, 0.69, 0.97, 0.89, 
        0.79, 1.1, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
        0.81, 0.59, 0.56, 0.82, 0.89, 0.81, 0.65, 0.6, 0.6, 0.54, 
        0.55, 0.55, 0.89, 0.48, 0.61, 0.56, 0.74, 0.6, 0.57, 
        0.67, 0.57, 0.81, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 
        0.85, 0.54, 0.74, 0.67, 0.6, 0.59, 0.56, 0.64, 0.7, 0.59, 
        0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 2.83, 2.93, 2.85, 
        3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3, 3, 3, 2.65, 
        2.71, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 2.78, 3.2, 3.21, 
        2.77, 3.57, 3.1, 2.71, 2.88, 2.87, 3, 2.87, 2.96, 2.82, 
        4, 2.69, 4, 3.38, 3, 3.3, 3, 3.35, 3.33, 2.87, 3.33, 
        2.75, 3.1, 2.78, 3.33, 3.26, 2.93, 3.2, 3.03, 3.1, 2.84, 
        2.87, 1.82, 2.96, 1.59, 2.46, 2.87, 3.48, 2.3, 3.18, 
        3.48, 1.69, 3.38, 1.82, 3.33, 2.14, 3.5, 3.13, 2.14, 
        2.48, 2.52, 1.69, 2.72, 3.03, 3.14, 2.72, 2.01, 2.87, 
        3.21, 3.02, 3.21, 2.75, 3.21, 2.12, 2.65, 2.75, 3.3, 
        2.3, 2.63, 1.47, 2.74, 2.88, 2.83, 2.96, 2.77, 3.38, 
        3.21, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 
        2.5, 3.58, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 
        3.69, 3.12, 2.05, 3.64, 3.36, 2.84, 3, 2.78, 2.48, 2.87, 
        1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 
        1.42, 1.51, 1.59, 2.05, 2, 1.75, 1.33, 1.82, 1.62, 1.33, 
        2.63, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 
        1.78, 1.58, 2.12, 1.62, 1.75, 1.68, 1.75, 1.48, 1.58, 
        1.59, 1.55, 1.51, 2.05, 1.33, 1.82, 1.47, 1.56, 1.62, 
        1.6), prol = c(1065, 1050, 1060, 1285, 735, 1450, 1290, 
        1295, 1045, 1045, 1547, 1280, 1320, 1060, 1547, 1310, 
        1280, 1080, 1680, 845, 1290, 770, 1035, 1015, 845, 880, 
        1195, 1285, 915, 1035, 1285, 915, 915, 1235, 625, 920, 
        880, 480, 1235, 760, 795, 1035, 1095, 680, 885, 1080, 
        1065, 985, 1060, 1095, 1150, 1265, 1190, 1285, 1060, 
        1120, 970, 1270, 1285, 590, 660, 660, 630, 640, 392, 
        770, 502, 345, 480, 718, 870, 410, 472, 365, 886, 1150, 
        392, 735, 680, 463, 278, 714, 472, 515, 718, 885, 564, 
        562, 680, 625, 480, 450, 495, 562, 345, 378, 625, 428, 
        660, 406, 625, 562, 438, 415, 672, 480, 510, 378, 312, 
        680, 562, 325, 410, 434, 385, 450, 325, 660, 372, 564, 
        515, 1095, 365, 380, 685, 378, 640, 466, 672, 580, 1150, 
        570, 560, 600, 463, 695, 720, 515, 580, 590, 600, 392, 
        520, 550, 372, 830, 415, 372, 720, 550, 500, 480, 425, 
        675, 640, 425, 480, 880, 660, 675, 463, 680, 570, 915, 
        615, 345, 695, 685, 640, 735, 510, 470, 550, 740, 502, 
        835, 840, 510)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 13.9, 13.24, 
        14.21, 14.39, 14.06, 13.29, 13.86, 13.24, 13.4, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 14.13, 13.64, 14.16, 
        12.93, 13.71, 12.85, 13.5, 13.76, 13.39, 13.3, 13.87, 
        13.05, 13.73, 13.58, 13.68, 13.76, 13.83, 12.85, 13.05, 
        13.05, 13.62, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 13.17, 13.9, 13.68, 13.24, 13.05, 13.84, 13.82, 
        13.27, 13.74, 13.36, 14.22, 13.29, 13.9, 12.37, 12.33, 
        12.64, 12.99, 12.37, 12.17, 12.37, 13.05, 13.27, 13.34, 
        12.21, 13.34, 13.86, 11.81, 12.99, 11.96, 11.66, 12.33, 
        11.84, 13.05, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 11.84, 
        12.16, 11.61, 12.93, 12.08, 12.08, 12, 12.69, 13.05, 
        11.62, 12.08, 11.81, 12.29, 13.29, 12.29, 12.08, 12.6, 
        13.68, 11.84, 12.51, 12.85, 12, 12.72, 12.22, 11.61, 
        13.05, 12, 11.76, 11.03, 12.08, 11.03, 11.82, 13.58, 
        13.05, 11.82, 11.45, 12.37, 12.42, 13.05, 11.87, 12.07, 
        13.71, 11.79, 13.86, 12.04, 12.86, 12.88, 13.05, 12.7, 
        12.51, 12.93, 12.25, 12.53, 13.71, 12.84, 12.93, 13.36, 
        13.52, 13.62, 12.25, 12.37, 11.76, 12.87, 13.32, 12.84, 
        12.21, 12.79, 13.34, 13.23, 13.4, 13.17, 13.84, 12.45, 
        14.34, 13.48, 14.22, 13.69, 12.85, 12.96, 13.78, 13.73, 
        13.4, 13.82, 13.58, 13.4, 12.37, 12.77, 14.16, 13.64, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.05, 4.61, 
        1.95, 3.59, 1.67, 1.87, 2.15, 2.16, 1.35, 2.16, 1.48, 
        1.67, 1.73, 1.87, 1.53, 1.92, 1.73, 1.35, 3.1, 1.63, 
        3.8, 0.74, 1.6, 1.81, 3.91, 1.21, 4.12, 1.9, 1.83, 1.5, 
        3.37, 1.83, 1.53, 5.65, 1.9, 1.21, 1.65, 1.5, 1.35, 1.71, 
        3.59, 5.65, 3.98, 1.5, 4.04, 3.98, 1.68, 3.12, 1.71, 
        1.41, 1.65, 1.92, 1.9, 1.67, 3.43, 1.7, 1.97, 1.68, 0.94, 
        2.16, 1.36, 1.25, 1.87, 1.41, 1.21, 1.01, 1.17, 4.6, 
        1.65, 3.91, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 1.63, 
        0.99, 3.87, 1.83, 1.63, 1.13, 2.58, 4.31, 0.98, 1.61, 
        1.67, 2.68, 1.61, 1.35, 1.51, 3.74, 2.16, 1.87, 1.73, 
        1.68, 1.41, 3.24, 2.08, 2.08, 3.55, 4.43, 1.72, 4.31, 
        2.55, 1.73, 2.15, 1.29, 1.5, 3.74, 1.73, 2.68, 0.74, 
        1.39, 1.48, 1.73, 1.61, 3.43, 1.61, 2.4, 2.05, 4.43, 
        3.03, 4.31, 2.16, 1.53, 2.13, 4.1, 4.3, 1.35, 2.99, 2.31, 
        3.55, 1.24, 1.51, 3.45, 5.51, 3.59, 3.03, 2.81, 2.56, 
        4.61, 2.67, 3.88, 2.31, 5.04, 4.61, 3.24, 1.9, 3.12, 
        2.67, 1.9, 3.1, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 
        5.51, 3.03, 3.45, 2.76, 2.31, 3.37, 3.37, 2.58, 4.6, 
        2.15, 2.39, 5.51, 5.65, 3.91, 5.19, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.72, 2.62, 2.45, 2.61, 2.2, 2.38, 2.3, 
        2.1, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.29, 
        2.65, 1.94, 2.52, 2.61, 3.22, 2.1, 2.65, 2.4, 1.99, 2.67, 
        2.36, 2.38, 2.7, 2.65, 2.32, 2.46, 2.55, 2.1, 2.51, 2.65, 
        2.12, 2.59, 2.29, 2.1, 2.44, 2.62, 2.12, 2.4, 2.27, 2.04, 
        2.2, 2.65, 2.27, 2.44, 2.3, 2.3, 2.68, 2.5, 1.36, 1.71, 
        2.12, 1.92, 2.16, 2.53, 2.56, 2.2, 1.92, 2.53, 1.75, 
        2.21, 2.67, 2.24, 2.6, 2.3, 2, 1.71, 2.38, 2.14, 2.4, 
        2, 2.28, 2.51, 2.32, 2.58, 2.24, 2.48, 2.62, 2.38, 2.3, 
        2.32, 1.92, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.7, 2.74, 
        1.7, 2.32, 2.46, 1.88, 2.38, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.65, 2.17, 
        2, 2.42, 3.23, 2.54, 2.13, 2.39, 2.17, 2.29, 2.72, 2.74, 
        2.38, 2.42, 2.4, 2.4, 2.36, 2.25, 2.39, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.45, 2.35, 2.48, 2.4, 
        2.36, 2.62, 2.48, 2.1, 2.28, 2.19, 2.32, 2.38, 2.61, 
        2.72, 2.64, 2.38, 2.54, 2.39, 2.35, 2.3, 2.62, 2.6, 2.14, 
        2.69, 2.26, 2.32, 2.28, 2.26, 2.41, 2.48, 2.26, 2.37, 
        2.74), alca = c(15.6, 16.8, 18, 16.8, 21, 15.2, 18, 17.6, 
        14, 16, 16, 16.8, 16, 11.4, 16, 15.6, 21, 20, 13.2, 15.2, 
        16, 18.6, 18.6, 17.8, 20, 25, 16.1, 17, 13.2, 16, 13.2, 
        19.1, 21, 18.8, 19, 20.5, 15.5, 18, 15.5, 13.2, 16.2, 
        18.8, 19, 15.6, 17, 15.5, 16, 16, 18.8, 17.4, 12.4, 16, 
        15.5, 17, 16, 20.5, 16.3, 16.8, 16, 10.6, 22.5, 16.8, 
        18, 19, 19, 18.1, 17, 19.6, 22.8, 16.8, 20.4, 16, 24, 
        30, 21, 17.8, 16, 18, 14.8, 18, 19, 19, 24, 22.5, 18, 
        18, 22.8, 19.5, 21.6, 18.6, 24, 22, 22, 18, 18, 19, 21.5, 
        16, 16.1, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 
        19, 20, 19.5, 16.2, 23.5, 19, 22, 21.5, 20.8, 19, 16, 
        18.5, 20, 19.5, 21, 21.5, 21, 21, 21.5, 28.5, 19.5, 22, 
        24, 20, 24, 21.5, 21, 20, 18.5, 25, 19.5, 25, 21, 25, 
        23.5, 20, 24.5, 21, 20, 19.5, 16.2, 21.5, 24, 22, 19.1, 
        18.5, 20, 22, 19.5, 21.5, 25, 18.5, 20, 20.5, 24, 18.5, 
        21, 22.5, 23, 19.5, 24.5, 25, 22, 19.5, 19.5, 20.5, 23, 
        20, 22.8, 24.5), mg = c(127, 100, 101, 113, 151, 112, 
        96, 121, 97, 86, 100, 92, 89, 91, 102, 112, 120, 101, 
        98, 116, 126, 102, 86, 95, 122, 124, 93, 101, 107, 96, 
        101, 106, 104, 107, 110, 96, 110, 95, 110, 98, 117, 100, 
        101, 87, 107, 111, 102, 101, 101, 108, 92, 123, 111, 
        115, 122, 117, 118, 102, 108, 88, 101, 100, 94, 87, 104, 
        98, 78, 78, 88, 151, 89, 86, 87, 139, 101, 90, 86, 112, 
        110, 87, 86, 86, 78, 85, 94, 99, 90, 88, 84, 70, 81, 
        92, 102, 88, 97, 87, 134, 85, 88, 104, 97, 88, 104, 86, 
        85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 
        110, 110, 87, 96, 111, 102, 86, 94, 85, 107, 92, 88, 
        80, 122, 88, 98, 106, 88, 94, 89, 95, 88, 101, 95, 89, 
        97, 78, 112, 102, 98, 103, 92, 113, 123, 127, 116, 98, 
        103, 98, 89, 97, 98, 89, 89, 88, 106, 106, 90, 88, 101, 
        88, 105, 112, 96, 86, 91, 95, 102, 110, 120, 126), phe = c(2.8, 
        2.65, 2.8, 3.85, 2.8, 2.8, 2.5, 2.6, 3, 2.98, 2.95, 2.2, 
        2.6, 3.1, 3, 2.85, 2.8, 2.5, 3.3, 2.6, 3, 2.41, 2.61, 
        2.48, 2.53, 2.53, 2.86, 2.4, 2.95, 2.65, 3, 2.95, 2.42, 
        2.95, 3, 2.7, 2.6, 2.45, 2.85, 2.95, 3.15, 2.45, 3.25, 
        2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 
        2.8, 2.65, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2, 3.5, 
        1.89, 2.8, 2.6, 2.11, 2.53, 2.9, 1.1, 2.95, 1.88, 1.75, 
        1.7, 1.9, 2.56, 2.13, 1.9, 2.83, 2.83, 2.2, 1.9, 1.65, 
        2.48, 2.2, 2, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
        3.02, 2.56, 1.35, 2.55, 3.52, 2.75, 2.05, 2.13, 2.56, 
        2.5, 2.2, 2.05, 1.65, 1.92, 2.36, 2.74, 3.18, 2.55, 1.75, 
        2.65, 2.56, 2.46, 1.88, 2, 1.7, 2, 2.9, 3.18, 2.5, 1.4, 
        2.86, 2.74, 2.74, 2.13, 2.22, 2.1, 1.51, 1.88, 0.98, 
        1.7, 1.9, 1.62, 1.38, 1.35, 1.62, 1.51, 1.38, 1.4, 2.05, 
        2, 1.38, 1.9, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 
        1.92, 1.48, 1.45, 1.8, 1.9, 1.93, 2.02, 2.3, 1.83, 1.9, 
        1.39, 1.35, 1.4, 1.7, 1.48, 1.55, 1.55, 1.25, 1.39, 1.68, 
        1.55, 1.38, 2.05, 1.65, 2.05), fla = c(3.49, 2.76, 2.51, 
        3.49, 2.61, 3.39, 2.52, 2.51, 2.98, 3.54, 3.32, 2.43, 
        2.76, 2.98, 3.64, 2.91, 3.14, 3.4, 2.78, 2.89, 3.49, 
        2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.63, 2.97, 2.33, 
        3.25, 3.19, 2.69, 2.52, 2.98, 2.98, 3.15, 2.43, 2.64, 
        3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 2.99, 2.92, 
        3.54, 1.85, 2.99, 3.74, 2.79, 2.9, 2.78, 3.64, 3.23, 
        3.67, 0.57, 2.65, 1.41, 1.79, 2.91, 1.75, 2.65, 2.27, 
        2, 1.3, 3.64, 1.02, 2.86, 1.69, 2.89, 2.14, 1.57, 2.63, 
        1.57, 1.85, 2.64, 2.26, 2.53, 1.58, 1.59, 2.21, 2.27, 
        1.69, 1.61, 1.69, 1.57, 1.41, 1.25, 1.46, 3.4, 2.68, 
        2.27, 0.99, 2.65, 2.9, 2.99, 2.17, 1.36, 2.11, 1.64, 
        1.92, 1.84, 2.03, 1.76, 1.64, 2.92, 2.33, 2.27, 2.03, 
        2.01, 2.29, 2.17, 1.6, 2.09, 0.68, 1.64, 2.68, 3.49, 
        1.61, 0.58, 3.03, 2.65, 3.15, 2.24, 2.17, 1.75, 0.92, 
        0.65, 0.5, 1.2, 0.58, 0.66, 0.47, 0.58, 0.96, 0.6, 0.5, 
        0.61, 0.52, 0.8, 0.96, 0.55, 0.34, 0.65, 0.76, 1.39, 
        1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.68, 0.58, 1.31, 
        1.1, 0.92, 0.55, 0.6, 0.58, 0.68, 0.47, 0.92, 0.66, 0.84, 
        0.96, 0.49, 0.5, 0.7, 0.61, 0.68, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.28, 0.3, 0.24, 0.39, 0.2, 0.3, 0.31, 
            0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.32, 
            0.29, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.45, 0.43, 
            0.34, 0.27, 0.22, 0.26, 0.29, 0.13, 0.42, 0.47, 0.29, 
            0.26, 0.29, 0.34, 0.28, 0.2, 0.34, 0.17, 0.22, 0.32, 
            0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
            0.39, 0.21, 0.2, 0.26, 0.26, 0.39, 0.28, 0.25, 0.53, 
            0.32, 0.31, 0.45, 0.37, 0.26, 0.27, 0.45, 0.32, 0.47, 
            0.61, 0.45, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 
            0.3, 0.39, 0.4, 0.61, 0.22, 0.3, 0.52, 0.37, 0.32, 
            0.37, 0.45, 0.5, 0.34, 0.25, 0.17, 0.32, 0.53, 0.39, 
            0.24, 0.37, 0.22, 0.29, 0.34, 0.37, 0.25, 0.66, 0.3, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.53, 0.42, 0.34, 0.52, 
            0.3, 0.26, 0.5, 0.17, 0.32, 0.32, 0.43, 0.3, 0.17, 
            0.29, 0.39, 0.3, 0.4, 0.34, 0.4, 0.5, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.5, 0.53, 
            0.53, 0.66, 0.43, 0.4, 0.47, 0.45, 0.61, 0.22, 0.47, 
            0.26, 0.61, 0.53, 0.48, 0.63, 0.63, 0.53, 0.52, 0.63, 
            0.5, 0.6, 0.4, 0.41, 0.5, 0.43, 0.4, 0.39, 0.43, 
            0.4, 0.48, 0.45, 0.3, 0.43, 0.43, 0.53, 0.53), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 1.82, 2.96, 1.46, 1.92, 1.97, 1.86, 
            1.66, 2.1, 1.28, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
            1.76, 1.98, 2.38, 1.95, 1.98, 1.45, 1.54, 1.86, 1.36, 
            1.44, 1.37, 3.28, 2.1, 1.42, 1.87, 2.76, 2.03, 1.69, 
            2.19, 2.14, 2.38, 2.08, 1.31, 2.29, 1.87, 1.68, 1.62, 
            2.45, 1.36, 1.66, 1.85, 0.42, 1.04, 0.94, 0.73, 1.87, 
            1.03, 1.46, 1.56, 1.04, 0.75, 2.5, 1.46, 1.87, 1.03, 
            1.25, 1.03, 1.15, 1.46, 1.34, 2.76, 1.95, 1.43, 1.42, 
            1.46, 1.62, 1.04, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 0.8, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
            1.4, 1.35, 1.31, 1.42, 1.42, 1.42, 1.25, 1.63, 2.08, 
            2.49, 1.04, 1.63, 2.29, 1.9, 1.04, 1.14, 1.46, 1.61, 
            0.83, 1.87, 1.83, 1.85, 1.71, 0.64, 2.91, 1.35, 1.77, 
            1.66, 1.9, 1.35, 0.94, 0.42, 0.83, 0.84, 1.25, 0.94, 
            0.8, 1.1, 0.88, 0.75, 0.75, 0.83, 0.55, 0.55, 1.14, 
            1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.35, 
            0.55, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 0.55, 0.83, 
            0.94, 0.75, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 
            1.24, 1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 
            4.38, 5.68, 7.8, 4.4, 6.75, 7.22, 5.05, 2.6, 7.22, 
            5.75, 4.9, 5.6, 3.7, 7.5, 4.8, 8.42, 5.04, 8.7, 5.1, 
            5.65, 4.5, 4.8, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 
            4.7, 5.7, 4.28, 7.1, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            8.9, 8.9, 7.2, 7.22, 4.7, 6.3, 5.85, 6.25, 6.3, 3.58, 
            6.8, 2.9, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 
            4.68, 3.17, 2.85, 7.1, 3.38, 3.74, 2.7, 3.21, 3.8, 
            4.6, 2.9, 8.21, 3.08, 2.5, 3.9, 3.25, 4.8, 3.05, 
            2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 
            3.25, 2.6, 6.25, 2.9, 2.45, 4.5, 4.68, 3.21, 2.8, 
            4.5, 2.94, 2.7, 3.4, 3.3, 2.7, 4.45, 2.9, 2.9, 4.7, 
            3.08, 2.9, 1.9, 3.4, 2.8, 3.4, 1.28, 3.25, 2.94, 
            1.74, 5.85, 2.8, 2.5, 3.94, 4.68, 2.12, 3.25, 7.1, 
            4.7, 9.2, 5, 5.45, 7.1, 4.9, 5, 5.7, 4.92, 5.6, 7.1, 
            4.35, 4.4, 8.21, 4, 4.9, 4, 8.42, 9.4, 2.45, 9.2, 
            7.1, 10.52, 7.6, 7.9, 10.26, 7.5, 13, 7.1, 7.65, 
            4.36, 5.58, 7.22, 7.8, 6.62, 10.68, 10.26, 7.5, 8.5, 
            8.9, 3.95, 9.7, 7.7, 7.3, 8.42, 9.3, 9.2), hue = c(1.04, 
            1.05, 1.04, 0.86, 1.02, 1.05, 1.02, 0.95, 1.08, 1.23, 
            1.25, 1.22, 1.15, 1.25, 1.2, 1.28, 0.86, 1.13, 1.23, 
            1.09, 1.09, 0.93, 1.11, 1.09, 1.12, 1.13, 1.15, 1.02, 
            0.93, 1.04, 1.19, 1.09, 0.95, 1.25, 1.1, 0.89, 1.15, 
            0.95, 1.18, 1.42, 0.95, 0.91, 0.88, 0.82, 0.98, 0.87, 
            1.07, 1.28, 1.07, 1.12, 1.12, 1.31, 1.25, 0.91, 0.98, 
            1.07, 0.94, 1.07, 0.89, 1.04, 1.25, 0.98, 1.23, 1.22, 
            0.93, 1.23, 0.88, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.31, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 
            1.23, 1.31, 0.84, 1.12, 1.23, 0.69, 1.05, 1.04, 1.07, 
            1.08, 1.05, 1.28, 1.15, 0.93, 1.12, 1.04, 0.86, 0.97, 
            1.42, 1.27, 0.75, 0.8, 0.87, 1.04, 0.88, 1, 1.23, 
            0.86, 1.36, 1.05, 0.9, 1.1, 1.1, 0.93, 1.71, 0.95, 
            1.27, 0.7, 0.91, 0.8, 1.25, 0.92, 0.73, 1.07, 0.86, 
            0.69, 0.97, 0.89, 0.79, 0.73, 0.74, 0.66, 0.78, 0.75, 
            0.73, 0.75, 0.82, 0.81, 0.81, 0.73, 0.87, 0.89, 0.78, 
            0.65, 0.6, 0.81, 0.54, 0.55, 0.82, 0.79, 0.48, 0.61, 
            0.56, 0.56, 0.6, 0.57, 0.67, 0.57, 0.87, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.59, 0.74, 0.67, 1.02, 
            0.73, 0.67, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
            3.4, 3.17, 3.1, 2.93, 3.38, 3.58, 3.58, 2.85, 3.55, 
            3.17, 2.82, 2.9, 2.87, 3, 2.71, 2.65, 3.64, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.48, 3.2, 2.82, 2.77, 
            3.52, 3.05, 2.71, 2.88, 2.87, 3, 2.87, 3.52, 3.92, 
            3.58, 2.69, 3.69, 3.38, 3, 3.2, 3, 3.35, 3.33, 3.12, 
            3.33, 2.75, 3.1, 2.42, 3.52, 3.26, 2.93, 3.2, 3.03, 
            3.39, 2.84, 2.87, 1.82, 2.85, 1.59, 2.46, 2.87, 2.82, 
            2.3, 3.18, 3.48, 1.59, 3.17, 1.82, 3.03, 2.93, 3.5, 
            3.13, 2.14, 2.48, 2.52, 3.33, 2.78, 2.81, 3.14, 2.72, 
            2.01, 2.75, 3.36, 2.5, 3.21, 2.75, 3.21, 2.72, 2.65, 
            2.52, 3.3, 3.21, 2.63, 2.05, 2.74, 2.78, 2.83, 2.96, 
            2.77, 3.38, 2.42, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 
            2.81, 2.78, 2.5, 2.77, 3.19, 2.87, 3.33, 2.96, 2.12, 
            3.05, 3.39, 3.69, 3.12, 1.6, 3.64, 3.1, 2.84, 3, 
            2.78, 2.82, 1.75, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 
            1.69, 1.82, 2.15, 1.62, 2.15, 1.62, 2.05, 2, 1.51, 
            1.33, 1.51, 1.62, 1.33, 2.9, 1.47, 1.33, 1.51, 1.55, 
            1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 1.62, 1.62, 1.75, 
            1.68, 1.75, 1.27, 1.33, 1.51, 1.55, 1.55, 1.82, 1.48, 
            1.69, 1.33, 1.56, 1.62, 1.6), prol = c(1065, 1050, 
            1035, 1045, 735, 1450, 1290, 1295, 1045, 1045, 1265, 
            1280, 1320, 1050, 1547, 1310, 1280, 680, 1680, 845, 
            1285, 770, 1035, 1015, 845, 630, 1195, 1285, 915, 
            1035, 1285, 1050, 880, 1235, 1310, 920, 880, 1045, 
            680, 760, 795, 1035, 1095, 680, 885, 1080, 1065, 
            985, 1060, 1285, 1150, 1265, 1190, 680, 1060, 1120, 
            970, 1270, 1285, 675, 428, 718, 630, 564, 580, 845, 
            502, 600, 392, 718, 870, 410, 472, 600, 886, 625, 
            392, 480, 770, 463, 278, 714, 434, 515, 680, 920, 
            463, 562, 680, 625, 480, 450, 495, 714, 345, 463, 
            625, 428, 660, 406, 550, 562, 438, 415, 672, 550, 
            510, 415, 312, 680, 562, 325, 392, 434, 385, 415, 
            480, 714, 372, 564, 312, 502, 365, 380, 695, 378, 
            740, 466, 1050, 580, 672, 438, 560, 600, 466, 695, 
            720, 515, 580, 590, 600, 345, 520, 550, 372, 830, 
            415, 502, 915, 550, 500, 480, 425, 675, 640, 392, 
            480, 880, 660, 480, 870, 680, 570, 515, 615, 695, 
            695, 685, 562, 680, 510, 470, 1035, 740, 680, 835, 
            840, 510)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.16, 13.24, 
        14.75, 14.39, 14.06, 12.04, 13.86, 13.24, 14.23, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 13.63, 13.64, 14.34, 
        12.93, 13.71, 12.85, 13.5, 13.76, 13.39, 13.3, 13.87, 
        13.71, 13.73, 13.58, 13.68, 13.76, 13.68, 12.04, 13.05, 
        13.05, 12.25, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 13.73, 13.9, 13.05, 14.38, 13.05, 13.4, 13.82, 
        13.39, 13.74, 13.64, 14.22, 13.29, 13.88, 12.37, 12.33, 
        12.64, 11.84, 12.37, 12.17, 12.37, 13.05, 12.21, 13.34, 
        12.21, 12.08, 13.86, 11.79, 12.99, 11.96, 11.66, 12.29, 
        11.84, 13.52, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 13.05, 
        12.16, 12.37, 12.07, 12.08, 12.08, 12, 12.69, 12.37, 
        11.62, 11.03, 11.81, 12.29, 12.25, 12.29, 12.08, 12.6, 
        12.51, 12.37, 12.51, 12.21, 11.82, 12.72, 12.22, 11.61, 
        12.22, 12.7, 11.76, 12.08, 12.08, 11.03, 11.82, 12.25, 
        12.37, 12.08, 11.45, 14.38, 12.42, 13.05, 11.87, 12.07, 
        13.4, 11.79, 11.96, 12.04, 12.86, 12.88, 13.41, 12.7, 
        12.51, 12.85, 12.25, 12.53, 13.87, 12.84, 12.93, 13.36, 
        13.52, 13.62, 12.25, 13.24, 12.37, 12.87, 13.32, 12.25, 
        12.84, 12.79, 12.04, 13.23, 13.05, 13.17, 13.84, 12.45, 
        14.34, 13.48, 13.39, 13.69, 12.85, 12.96, 13.78, 13.73, 
        14.3, 13.88, 13.58, 13.4, 13.52, 12.77, 14.16, 13.9, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.15, 4.3, 
        1.95, 2.67, 2.68, 1.87, 2.15, 1.17, 1.35, 2.16, 1.48, 
        0.99, 1.73, 1.87, 1.73, 1.92, 1.61, 0.9, 3.1, 1.63, 3.8, 
        1.73, 1.6, 1.81, 2.15, 1.39, 1.13, 1.9, 1.67, 1.5, 2.58, 
        1.83, 1.53, 4.04, 0.9, 2.15, 1.65, 1.5, 1.39, 1.71, 0.99, 
        1.92, 3.98, 2.58, 4.04, 1.67, 1.68, 1.71, 1.88, 0.99, 
        1.65, 2.68, 1.9, 1.67, 2.15, 1.7, 1.97, 4.6, 0.94, 3.74, 
        1.36, 1.25, 3.59, 0.74, 1.21, 1.01, 1.17, 3.83, 0.9, 
        1.61, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 0.9, 0.99, 3.87, 
        1.17, 1.36, 1.13, 1.24, 1.73, 0.98, 1.61, 1.67, 1.95, 
        1.65, 1.25, 1.51, 1.48, 1.7, 1.17, 1.13, 1.72, 1.41, 
        0.94, 1.13, 2.08, 4.3, 4.1, 1.72, 3.12, 2.55, 1.73, 2.16, 
        1.29, 1.41, 3.74, 1.53, 2.68, 0.74, 1.39, 0.99, 0.9, 
        1.61, 3.43, 0.94, 2.4, 2.05, 4.43, 2.39, 4.31, 2.16, 
        1.53, 2.13, 1.63, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 
        3.03, 1.68, 5.51, 3.59, 1.51, 2.81, 2.56, 3.24, 1.09, 
        3.88, 3.83, 5.04, 4.61, 3.24, 3.03, 3.12, 2.67, 1.9, 
        2.81, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 4.1, 
        1.5, 3.45, 2.76, 3.03, 3.43, 3.37, 2.58, 4.6, 3.43, 2.39, 
        5.65, 5.65, 3.91, 3.24, 2.59, 4.1), ash = c(2.43, 2.14, 
        2.67, 2.5, 2.28, 2.29, 2.45, 2.61, 2.17, 2.28, 2.3, 2.72, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.27, 
        2.65, 2.68, 2.52, 2.61, 3.22, 2.13, 2.7, 2.7, 2.59, 2.12, 
        2.36, 2.46, 2.7, 2.65, 2.28, 1.82, 2.55, 2.1, 2.51, 2.62, 
        2.12, 2.59, 2.29, 2.1, 2.44, 2.43, 2.12, 2.4, 2.27, 2.04, 
        2.64, 2.4, 2.1, 2.51, 2.67, 2.3, 2.68, 2.5, 1.36, 2.24, 
        2.7, 1.92, 2.16, 2.53, 2.56, 1.82, 1.92, 2.52, 1.75, 
        2.21, 2.67, 2.24, 2.6, 2.3, 1.92, 1.71, 1.98, 2.68, 2.4, 
        2, 2.44, 2.51, 2.32, 2.58, 2.24, 2.74, 2.62, 2.65, 2.3, 
        2.32, 2.29, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2, 2.36, 
        1.7, 2.32, 2.46, 1.88, 2.35, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.54, 1.36, 
        2, 2.42, 3.23, 2.32, 2.13, 2.39, 2.17, 2.29, 2.67, 2.32, 
        2.38, 2.7, 2.4, 2.4, 2.36, 2.25, 2.61, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.38, 2.39, 2.48, 1.82, 
        2.36, 2.62, 2.48, 1.98, 2.28, 2.64, 2.32, 2.38, 2.46, 
        2.3, 2.64, 2.38, 2.54, 2.12, 2.35, 2.3, 2.62, 2.6, 2.24, 
        2.69, 2.6, 2.32, 2.28, 2.3, 2.35, 2.48, 2.26, 2.37, 2.74
        ), alca = c(15.6, 21, 19, 16.8, 21, 15.2, 16.8, 17.6, 
        14, 16, 18, 16.8, 16, 11.4, 16, 20.4, 16.3, 20, 20, 15.2, 
        16, 18.6, 21, 17.8, 20, 21, 16.1, 17, 19, 16.8, 16.1, 
        19.1, 19.5, 19.5, 19, 20.5, 15.5, 21.5, 15.5, 13.2, 16.2, 
        18.8, 12.4, 20, 17, 20.5, 16, 16, 18.8, 17.4, 12.4, 16.8, 
        16, 19.6, 15.5, 20.5, 16.3, 16.8, 21, 10.6, 23.5, 16.8, 
        18, 19, 19, 18.1, 16.8, 19.6, 19, 16.8, 20.4, 18, 24, 
        30, 21, 21.5, 16, 18, 14.8, 19.5, 19, 17, 24, 22.5, 18, 
        18, 22.8, 22.5, 21.6, 19.6, 19.5, 22, 22.5, 18, 18, 19, 
        21.5, 16, 16, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 
        22.5, 19, 20, 19.5, 16, 21, 20, 20, 21.5, 20.8, 23.5, 
        16, 17, 19, 15.5, 24.5, 21.5, 21, 21, 21.5, 28.5, 20, 
        22, 24, 20, 24, 21.5, 18, 30, 30, 25, 19.5, 20.5, 21, 
        23, 23.5, 20, 24.5, 21, 20, 25, 19, 21.5, 24, 22, 18.8, 
        18.5, 20, 22, 19.5, 24.5, 25, 21, 25, 22, 22.5, 18.5, 
        24, 22.5, 23, 19.5, 24.5, 25, 23.5, 19.5, 21.6, 20.5, 
        23, 20, 22, 24.5), mg = c(127, 100, 101, 113, 97, 112, 
        96, 121, 97, 88, 96, 101, 89, 91, 102, 112, 120, 106, 
        124, 116, 126, 102, 112, 95, 94, 124, 93, 112, 107, 96, 
        101, 106, 104, 101, 110, 121, 110, 88, 104, 87, 117, 
        110, 101, 102, 107, 111, 102, 101, 95, 108, 92, 116, 
        111, 115, 104, 139, 118, 102, 108, 88, 101, 100, 94, 
        87, 104, 98, 78, 78, 85, 151, 104, 86, 87, 139, 101, 
        89, 92, 112, 89, 102, 86, 107, 78, 85, 108, 99, 90, 88, 
        84, 70, 81, 85, 89, 88, 108, 92, 134, 85, 88, 87, 97, 
        88, 89, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 
        84, 85, 86, 100, 96, 87, 96, 101, 102, 86, 101, 85, 102, 
        92, 88, 80, 122, 110, 98, 106, 81, 94, 89, 98, 88, 101, 
        86, 89, 97, 116, 112, 102, 101, 85, 92, 113, 123, 112, 
        116, 98, 103, 70, 89, 97, 98, 89, 97, 107, 106, 106, 
        90, 88, 94, 88, 105, 112, 96, 86, 91, 95, 102, 100, 120, 
        107), phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.15, 2.5, 2.6, 
        2.8, 2.98, 2.95, 2.2, 2.6, 3.1, 2.88, 2.85, 2.8, 2.85, 
        3.3, 3, 3, 2.41, 2.61, 2.48, 2.53, 2.11, 2.6, 2.4, 2.95, 
        2.65, 3, 3.25, 2.42, 2.95, 2.42, 2.7, 2.6, 2.45, 2.96, 
        3.18, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
        2.88, 2.72, 2.45, 3.88, 3, 2.45, 2.96, 3.2, 3, 3.4, 1.98, 
        2.05, 2.02, 1.9, 3.5, 1.89, 2.6, 2.6, 2.11, 2.53, 3.1, 
        1.1, 2.95, 1.88, 2.02, 2.02, 1.68, 2.2, 2.2, 1.9, 2.83, 
        2.61, 2.2, 1.55, 1.65, 2.2, 2.2, 1.48, 1.92, 1.95, 2.2, 
        1.6, 1.45, 1.38, 2.45, 3.02, 2.2, 1.65, 2.55, 3.52, 2.8, 
        2.6, 1.75, 2.56, 2.5, 2.2, 2.56, 1.65, 2.53, 2.36, 2.74, 
        3.18, 2.55, 1.75, 2.83, 2.56, 2.46, 2, 2, 2.8, 2, 2.9, 
        3.18, 1.55, 1.89, 2.86, 2.95, 2.74, 2.13, 2.22, 2.1, 
        1.51, 1.98, 1.65, 1.7, 2.3, 1.62, 1.38, 1.8, 1.62, 1.39, 
        1.39, 1.4, 1.25, 2, 1.38, 0.98, 0.98, 1.7, 1.93, 1.41, 
        1.4, 1.48, 2.2, 1.48, 1.48, 1.68, 1.8, 1.9, 1.65, 1.45, 
        2.3, 1.83, 1.39, 1.39, 1.35, 1.39, 1.7, 1.48, 1.55, 1.62, 
        1.25, 1.39, 1.68, 1.92, 2.02, 1.7, 1.65, 2.05), fla = c(3.67, 
        2.76, 3.04, 3.49, 2.68, 3.39, 2.52, 2.51, 2.98, 3.19, 
        3.32, 2.43, 2.76, 2.97, 3.64, 2.91, 3.14, 3.4, 3.64, 
        3.56, 3.64, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.63, 
        2.97, 2.33, 3.25, 3.19, 2.69, 3.03, 2.9, 2.98, 1.64, 
        2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 
        3.64, 2.92, 3.54, 2.97, 2.99, 3.74, 2.79, 2.9, 2.78, 
        2.88, 3.23, 3.67, 0.57, 2.61, 1.41, 1.79, 2.98, 1.75, 
        2.65, 2.9, 2, 1.3, 3.49, 1.02, 2.86, 1.69, 2.89, 2.14, 
        1.57, 2.27, 1.79, 1.85, 2.14, 2.26, 2.53, 1.58, 1.59, 
        2.21, 2.69, 1.69, 1.61, 1.69, 1.3, 1.6, 1.25, 1.46, 2.68, 
        2.76, 2.27, 0.99, 2.24, 3.39, 2.99, 2.17, 1.36, 2.11, 
        1.64, 1.92, 1.84, 2.03, 1.76, 2.29, 2.92, 2.98, 2.27, 
        2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.75, 1.64, 2.65, 
        2.78, 1.79, 1.57, 3.03, 2.65, 3.15, 2.24, 1.61, 1.75, 
        1.1, 0.99, 0.69, 1.2, 0.58, 0.66, 0.47, 1.39, 0.47, 0.6, 
        0.5, 0.66, 0.52, 0.8, 1.02, 0.55, 0.34, 0.65, 0.76, 1.39, 
        1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 1.39, 0.58, 1.31, 
        1.1, 0.92, 1.41, 0.6, 0.57, 0.68, 0.47, 0.92, 0.66, 0.84, 
        0.96, 0.49, 0.63, 0.7, 0.61, 1.41, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.24, 0.3, 0.24, 0.39, 0.22, 0.34, 
            0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 
            0.3, 0.2, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.39, 
            0.27, 0.34, 0.27, 0.3, 0.26, 0.29, 0.2, 0.42, 0.43, 
            0.29, 0.26, 0.28, 0.26, 0.28, 0.2, 0.34, 0.22, 0.21, 
            0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 
            0.32, 0.39, 0.21, 0.2, 0.26, 0.3, 0.32, 0.28, 0.32, 
            0.53, 0.32, 0.3, 0.45, 0.37, 0.26, 0.27, 0.52, 0.21, 
            0.37, 0.39, 0.3, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 
            0.43, 0.3, 0.32, 0.4, 0.61, 0.22, 0.3, 0.3, 0.42, 
            0.26, 0.5, 0.26, 0.5, 0.3, 0.25, 0.17, 0.32, 0.35, 
            0.21, 0.24, 0.32, 0.2, 0.29, 0.34, 0.37, 0.3, 0.66, 
            0.22, 0.48, 0.39, 0.29, 0.24, 0.26, 0.22, 0.42, 0.3, 
            0.52, 0.3, 0.34, 0.39, 0.34, 0.32, 0.37, 0.43, 0.3, 
            0.28, 0.43, 0.39, 0.35, 0.4, 0.26, 0.61, 0.53, 0.27, 
            0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.41, 
            0.27, 0.43, 0.4, 0.43, 0.4, 0.47, 0.45, 0.32, 0.22, 
            0.3, 0.26, 0.61, 0.53, 0.63, 0.4, 0.63, 0.53, 0.52, 
            0.43, 0.5, 0.6, 0.4, 0.41, 0.43, 0.43, 0.4, 0.39, 
            0.5, 0.52, 0.48, 0.47, 0.63, 0.43, 0.43, 0.53, 0.43
            ), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 
            1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 2.14, 2.96, 1.46, 
            1.87, 2.38, 1.86, 1.66, 2.1, 1.56, 1.69, 1.46, 1.66, 
            1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.18, 2.18, 
            1.54, 1.86, 1.36, 1.44, 1.37, 2.29, 2.38, 2.03, 1.82, 
            2.38, 2.03, 1.37, 2.19, 2.14, 2.38, 2.08, 2.03, 2.29, 
            1.87, 1.68, 1.62, 2.45, 1.46, 1.66, 2.08, 0.42, 1.85, 
            1.55, 0.73, 1.87, 1.03, 3.28, 2.1, 1.04, 1.63, 2.5, 
            1.46, 1.87, 1.03, 2.38, 1.46, 1.15, 1.46, 1.28, 2.76, 
            1.95, 1.43, 1.37, 1.56, 1.62, 1.1, 1.46, 1.56, 1.34, 
            1.35, 1.38, 1.64, 1.63, 1.46, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.26, 1.42, 
            1.43, 1.63, 2.08, 2.49, 1.43, 1.63, 1.63, 1.46, 1.04, 
            1.54, 1.31, 1.61, 0.83, 1.87, 1.83, 1.46, 1.71, 1.64, 
            2.91, 1.35, 1.77, 1.77, 1.9, 1.04, 0.94, 0.94, 0.83, 
            0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 0.75, 0.75, 1.25, 
            0.55, 0.68, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 
            1.26, 1.56, 1.38, 0.55, 1.55, 1.56, 1.14, 2.7, 2.29, 
            1.04, 1.61, 1.56, 0.94, 1.35, 1.15, 1.46, 0.97, 1.54, 
            1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
            ), col = c(5.64, 4.38, 5.68, 7.8, 4.9, 6.75, 5.24, 
            5.05, 3.8, 7.22, 5.75, 9.4, 5.6, 5, 7.5, 5.85, 5.58, 
            4.9, 8.7, 5.1, 5.65, 4.5, 2.95, 3.93, 3.52, 3.58, 
            4.8, 3.95, 4.5, 4.7, 5.7, 6.25, 5.45, 5.4, 4.2, 5.1, 
            4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 
            5.24, 4.9, 6.1, 7.5, 8.9, 7.2, 7.22, 4.25, 6.3, 5.85, 
            6.25, 3.52, 6.25, 6.8, 4.5, 3.27, 5.75, 3.8, 4.45, 
            2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 3.17, 3.38, 3.74, 
            3.58, 3.21, 3.8, 4.6, 3.3, 7.6, 2.8, 2.5, 3.9, 4.45, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 4.9, 2.9, 3.8, 3.27, 2.7, 
            4.5, 2.8, 3.7, 2.94, 2.7, 3.4, 3.3, 2.7, 2.9, 2.9, 
            3.38, 1.28, 3.08, 2.9, 1.9, 2.8, 4.45, 3.4, 1.28, 
            3.25, 6.13, 3.7, 4.6, 2.8, 2.7, 3.94, 5.1, 2.12, 
            1.28, 5.24, 6.8, 7.1, 5, 5.45, 7.1, 7.5, 5, 5.7, 
            4.92, 7.2, 4.8, 4.35, 4.4, 8.21, 4, 4.9, 7.5, 8.42, 
            9.4, 3.58, 8.5, 7.1, 10.52, 7.6, 7.9, 10.26, 7.5, 
            13, 9.3, 7.65, 6.8, 5.58, 6.1, 4, 6.62, 10.68, 10.26, 
            9.3, 8.5, 8.7, 5.43, 9.7, 7.7, 7.3, 6.75, 9.3, 9.2
            ), hue = c(1.04, 1.05, 1.23, 0.86, 1.22, 1.05, 1.02, 
            1.22, 1.08, 1.23, 1.25, 0.92, 1.15, 1.25, 1.2, 1.28, 
            1.25, 1.13, 1.23, 0.79, 1.09, 0.7, 1.11, 1.09, 1.12, 
            1.13, 1.36, 1.02, 1.23, 1.04, 1.19, 1.09, 1.38, 1.25, 
            1.1, 1.28, 0.8, 1.02, 1.18, 1.12, 0.95, 0.91, 0.88, 
            0.82, 0.82, 0.87, 1.38, 1.18, 1.07, 1.12, 1.12, 1.06, 
            1.25, 0.96, 1.05, 0.73, 0.94, 1.07, 0.89, 0.6, 1.25, 
            0.98, 1.23, 1.22, 1, 1.12, 1.15, 1.12, 1.02, 1.28, 
            0.906, 1.36, 0.98, 1.31, 1.07, 1.23, 1.19, 0.96, 
            1.06, 1.19, 1.38, 1.09, 1.31, 0.84, 0.87, 1.23, 1.05, 
            1.23, 1.28, 1.07, 1.08, 1.05, 1.23, 1.15, 0.92, 1.25, 
            0.78, 1.02, 1.18, 1.42, 1.27, 0.7, 0.8, 0.87, 1.04, 
            0.906, 1, 1.27, 0.86, 0.93, 0.79, 0.9, 0.9, 1.1, 
            0.93, 1.71, 0.95, 1.28, 0.7, 1.25, 0.8, 1.22, 0.92, 
            0.73, 0.8, 0.86, 0.69, 0.97, 0.89, 0.79, 0.87, 0.74, 
            0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.87, 0.68, 
            0.87, 0.89, 0.75, 0.65, 0.6, 0.85, 0.54, 0.55, 0.67, 
            0.7, 0.48, 0.61, 0.56, 0.66, 0.6, 0.57, 0.67, 0.57, 
            1.04, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.6, 
            0.74, 0.67, 0.67, 0.75, 0.54, 0.64, 0.7, 0.59, 0.6, 
            0.61), ratio = c(3.92, 3.4, 3.17, 3.18, 2.93, 3.4, 
            3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.92, 3, 
            2.84, 2.65, 2.84, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 
            2.83, 3.2, 2.77, 2.77, 2.81, 3.13, 2.71, 2.88, 2.87, 
            3, 2.87, 3.17, 2.63, 2.81, 2.69, 3.18, 3.38, 3, 3, 
            3, 3.35, 3.33, 2.9, 3.33, 2.75, 3.1, 2.87, 3.12, 
            3.26, 2.93, 3.2, 3.03, 3.2, 2.84, 2.87, 1.82, 3.18, 
            1.59, 2.46, 2.87, 2.12, 2.3, 3.18, 3.48, 3.3, 2.78, 
            1.82, 2.84, 2.96, 3.5, 3.13, 2.14, 2.48, 2.52, 2.12, 
            2.3, 3.63, 3.14, 2.72, 2.01, 3.33, 2.87, 2.75, 3.21, 
            2.75, 3.21, 3.21, 2.65, 2.75, 3.3, 3.2, 2.63, 1.33, 
            2.74, 3, 2.83, 2.96, 2.77, 3.38, 2.14, 3.57, 3.3, 
            3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.12, 3.19, 
            2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.01, 
            3.64, 3.64, 2.84, 2.48, 2.78, 2.87, 1.69, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.82, 2.05, 
            1.59, 2.05, 2, 2.15, 1.33, 1.48, 1.62, 1.33, 2.63, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 2.15, 2.15, 1.75, 1.68, 1.75, 1.96, 1.75, 1.6, 
            1.59, 1.62, 2.05, 1.58, 1.51, 1.29, 1.56, 1.62, 1.6
            ), prol = c(1065, 1050, 1285, 1680, 735, 1450, 1290, 
            1295, 1045, 1045, 1320, 1280, 1320, 1547, 1547, 1310, 
            1280, 1190, 1680, 845, 1280, 770, 1035, 1015, 845, 
            1280, 1195, 1285, 915, 1035, 1285, 1270, 1095, 1235, 
            1060, 920, 880, 770, 770, 760, 795, 1035, 1095, 680, 
            885, 1080, 1065, 985, 1060, 1280, 1150, 1265, 1190, 
            1080, 1060, 1120, 970, 1270, 1285, 415, 1045, 770, 
            630, 392, 550, 1045, 502, 345, 278, 718, 870, 410, 
            472, 1065, 886, 625, 392, 880, 1050, 463, 278, 714, 
            378, 515, 1035, 1060, 835, 562, 680, 625, 480, 450, 
            495, 434, 345, 372, 625, 428, 660, 406, 870, 562, 
            438, 415, 672, 680, 510, 564, 312, 680, 562, 325, 
            630, 434, 385, 870, 415, 770, 372, 564, 406, 1280, 
            365, 380, 495, 378, 970, 466, 415, 580, 735, 520, 
            560, 600, 580, 695, 720, 515, 580, 590, 600, 472, 
            520, 550, 886, 830, 415, 425, 562, 550, 500, 480, 
            425, 675, 640, 372, 480, 880, 660, 640, 680, 680, 
            570, 365, 615, 278, 695, 685, 675, 685, 510, 470, 
            560, 740, 510, 835, 840, 685)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 14.38, 13.24, 13.29, 14.39, 14.06, 12.93, 
        13.86, 13.05, 14.38, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.39, 13.64, 14.39, 12.93, 13.71, 12.85, 13.5, 
        13.83, 13.39, 13.3, 13.87, 13.83, 13.73, 13.58, 13.68, 
        13.76, 14.23, 13.41, 13.3, 13.05, 12.21, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.2, 13.9, 14.38, 
        14.22, 13.05, 14.3, 13.82, 13.4, 13.74, 14.34, 14.22, 
        13.29, 13.29, 12.37, 12.33, 12.64, 11.87, 12.37, 12.17, 
        12.37, 12.08, 13.52, 13.34, 12.21, 13.4, 13.86, 12.72, 
        12.99, 11.96, 11.66, 12.72, 11.84, 12.37, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 13.86, 12.16, 12.99, 11.76, 12.08, 
        12.08, 12, 12.69, 12.29, 11.62, 12.08, 11.81, 12.29, 
        11.81, 12.29, 12.08, 12.6, 12.37, 13.62, 12.51, 12.72, 
        12.08, 12.72, 12.22, 11.61, 13.86, 12.69, 11.76, 11.82, 
        12.08, 11.03, 11.82, 11.03, 11.84, 12.08, 11.45, 13.32, 
        12.42, 13.05, 11.87, 12.07, 12, 11.79, 12.08, 12.04, 
        12.86, 12.88, 12.25, 12.7, 12.51, 13.56, 12.25, 12.53, 
        12.53, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.29, 
        11.82, 12.87, 13.32, 12.72, 13.73, 12.79, 11.76, 13.23, 
        13.24, 13.17, 13.84, 12.45, 14.34, 13.48, 14.23, 13.69, 
        12.85, 12.96, 13.78, 13.73, 14.39, 14.22, 13.58, 13.4, 
        12.29, 12.77, 14.16, 13.74, 13.4, 13.27, 13.17, 14.13
        ), malic = c(1.71, 0.98, 1.17, 1.95, 1.95, 0.74, 1.87, 
        2.15, 1.73, 1.35, 2.16, 1.48, 1.25, 1.73, 1.87, 1.53, 
        1.92, 1.39, 1.01, 3.1, 1.63, 3.8, 0.99, 1.6, 1.81, 2.4, 
        1.21, 1.67, 1.9, 2.99, 1.5, 2.13, 1.83, 1.53, 3.37, 1.17, 
        1.29, 1.65, 1.5, 3.59, 1.71, 1.68, 1.83, 3.98, 1.68, 
        4.04, 1.5, 1.68, 1.63, 1.01, 1.13, 1.65, 1.9, 1.9, 1.67, 
        3.74, 1.7, 1.97, 1.67, 0.94, 1.95, 1.36, 1.25, 1.21, 
        1.39, 1.21, 1.01, 1.17, 3.74, 1.39, 1.73, 1.51, 1.66, 
        1.67, 1.09, 1.88, 0.9, 1.71, 0.99, 3.87, 1.51, 1.92, 
        1.13, 3.59, 1.51, 0.98, 1.61, 1.67, 1.72, 1.01, 1.24, 
        1.51, 3.98, 1.97, 1.97, 1.24, 4.31, 1.41, 1.13, 1.21, 
        2.08, 1.35, 1.29, 1.72, 2.55, 2.55, 1.73, 3.74, 1.29, 
        4.3, 3.74, 2.81, 2.68, 0.74, 1.39, 1.48, 2.31, 1.61, 
        3.43, 3.37, 2.4, 2.05, 4.43, 4.43, 4.31, 2.16, 1.53, 
        2.13, 2.31, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 5.19, 
        3.59, 5.51, 3.59, 5.04, 2.81, 2.56, 5.19, 4.6, 3.88, 
        4.1, 5.04, 4.61, 3.24, 0.98, 3.12, 2.67, 1.9, 2.81, 1.29, 
        5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 4.31, 5.04, 3.45, 
        2.76, 1.29, 3.8, 3.37, 2.58, 4.6, 4.43, 2.39, 3.74, 5.65, 
        3.91, 5.19, 2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 
        2.41, 2.74, 2.45, 2.61, 2.5, 2.36, 2.3, 2.42, 2.41, 2.39, 
        2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.64, 2.65, 2.62, 
        2.52, 2.61, 3.22, 2.62, 2.64, 2.61, 2.32, 2.64, 2.36, 
        2.26, 2.7, 2.65, 2.54, 2.28, 2.55, 2.1, 2.51, 2.35, 2.12, 
        2.59, 2.29, 2.1, 2.44, 2.42, 2.12, 2.4, 2.27, 2.04, 2.55, 
        2.19, 2.38, 2.37, 2.64, 2.3, 2.68, 2.5, 1.36, 2.26, 1.7, 
        1.92, 2.16, 2.53, 2.56, 2.28, 1.92, 2.28, 1.75, 2.21, 
        2.67, 2.24, 2.6, 2.3, 2, 1.71, 2.24, 2.56, 2.4, 2, 2.3, 
        2.51, 2.32, 2.58, 2.24, 2.4, 2.62, 2.68, 2.3, 2.32, 2.3, 
        2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.2, 2.28, 1.7, 2.35, 
        2.46, 1.88, 2.29, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 
        2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.28, 1.88, 2, 2.42, 
        3.23, 2.21, 2.13, 2.39, 2.17, 2.29, 3.23, 2.38, 2.38, 
        2.62, 2.4, 2.4, 2.36, 2.25, 2.67, 2.54, 2.64, 2.19, 2.61, 
        2.7, 2.35, 2.72, 2.35, 2.2, 2.64, 1.88, 2.48, 2.7, 2.36, 
        2.62, 2.48, 2.22, 2.28, 2.7, 2.32, 2.38, 2.48, 2.74, 
        2.64, 2.38, 2.54, 2.35, 2.35, 2.3, 2.59, 2.6, 2.39, 2.69, 
        2.74, 2.32, 2.28, 2.64, 2.19, 2.48, 2.26, 2.37, 2.74), 
            alca = c(15.6, 12.4, 20.5, 16.8, 21, 15.2, 18, 17.6, 
            14, 16, 16, 16.8, 16, 11.4, 16.8, 19.5, 19.6, 20, 
            12.4, 15.2, 16, 18.6, 18, 17.8, 20, 22, 16.1, 17, 
            21.5, 18, 16.8, 19.1, 20.8, 19, 19, 20.5, 15.5, 19.5, 
            15.5, 13.2, 16.2, 18.8, 16.2, 24, 17, 18, 16, 14, 
            18.8, 17.4, 12.4, 18, 12.4, 18.8, 18, 20.5, 16.3, 
            16.8, 16.8, 10.6, 22.5, 16.8, 18, 19, 19, 18.1, 22.5, 
            19.6, 25, 16.8, 20.4, 25, 24, 30, 21, 20, 16, 18, 
            14.8, 25, 19, 18.5, 24, 22.5, 18, 18, 22.8, 23, 21.6, 
            15.2, 21, 22, 25, 18, 18, 19, 21.5, 16, 16, 18, 17.5, 
            18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 
            21.6, 24, 21, 20.4, 21.5, 20.8, 22.5, 16, 20.4, 18, 
            22, 18, 21.5, 21, 21, 21.5, 28.5, 19.5, 22, 19.5, 
            20, 24, 21.5, 18, 22, 22, 25, 19.5, 23.5, 21, 23, 
            23.5, 20, 19.5, 21, 20, 19, 18, 21.5, 24, 22, 16.2, 
            18.5, 20, 22, 19.5, 19.1, 25, 19, 19, 24, 22.5, 18.5, 
            18.5, 22.5, 23, 19.5, 24.5, 25, 22.5, 19.5, 20, 20.5, 
            23, 20, 14, 24.5), mg = c(127, 100, 101, 113, 110, 
            112, 96, 121, 97, 97, 97, 103, 89, 91, 102, 112, 
            120, 97, 111, 116, 126, 102, 89, 95, 96, 124, 93, 
            112, 107, 96, 101, 106, 104, 89, 110, 93, 110, 104, 
            93, 120, 117, 106, 101, 121, 107, 111, 102, 101, 
            94, 108, 92, 88, 111, 115, 106, 96, 118, 102, 108, 
            88, 101, 100, 94, 87, 104, 98, 78, 78, 110, 151, 
            78, 86, 87, 139, 101, 96, 89, 112, 87, 99, 86, 92, 
            78, 85, 80, 99, 90, 88, 84, 70, 81, 88, 86, 88, 90, 
            89, 134, 85, 88, 92, 97, 88, 97, 86, 85, 90, 80, 
            84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 106, 106, 
            87, 96, 118, 102, 86, 112, 85, 106, 92, 88, 80, 122, 
            78, 98, 106, 86, 94, 89, 86, 88, 101, 101, 89, 97, 
            92, 112, 102, 89, 121, 92, 113, 123, 87, 116, 98, 
            103, 96, 89, 97, 98, 89, 102, 88, 106, 106, 90, 88, 
            89, 88, 105, 112, 96, 86, 91, 95, 102, 96, 120, 111
            ), phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 2.95, 2.5, 
            2.6, 3.3, 2.98, 2.95, 2.2, 2.6, 3.1, 3.5, 2.85, 2.8, 
            2.98, 3.3, 3.1, 3, 2.41, 2.61, 2.48, 2.53, 2.65, 
            2.65, 2.4, 2.95, 2.65, 3, 3.2, 2.42, 2.95, 2.85, 
            2.7, 2.6, 2.45, 2.64, 3.4, 3.15, 2.45, 3.25, 2.64, 
            3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 
            2.2, 2.6, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 1.89, 
            3.5, 1.89, 2.8, 2.05, 2.11, 2.53, 3.5, 1.1, 2.95, 
            1.88, 2.22, 2.02, 2.48, 2.41, 2.2, 1.9, 2.83, 2.36, 
            2.2, 1.48, 1.65, 2.45, 2.2, 1.92, 1.92, 1.95, 2.2, 
            1.6, 1.45, 1.38, 2.45, 3.02, 2.36, 2, 2.55, 3.52, 
            3.4, 2.5, 2.11, 2.56, 2.5, 2.2, 1.7, 1.65, 1.65, 
            2.36, 2.74, 3.18, 2.55, 1.75, 1.9, 2.56, 2.46, 1.38, 
            2, 2, 2, 2.9, 3.18, 2.95, 2.6, 2.86, 2, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.35, 1.4, 1.7, 1.48, 1.62, 1.38, 
            1.38, 1.62, 1.48, 1.8, 1.4, 1.98, 2, 1.38, 1.38, 
            0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.7, 1.48, 
            1.7, 1.8, 1.9, 1.1, 1.62, 2.3, 1.83, 1.48, 1.39, 
            1.35, 1.41, 1.7, 1.48, 1.55, 2.05, 1.25, 1.39, 1.68, 
            1.68, 1.93, 1.45, 1.65, 2.05), fla = c(3.17, 2.76, 
            3.29, 3.49, 2.37, 3.39, 2.52, 2.51, 2.98, 3, 3.32, 
            2.43, 2.76, 2.65, 3.64, 2.91, 3.14, 3.4, 3.19, 2.88, 
            3.67, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.26, 2.97, 
            2.33, 3.25, 3.19, 2.69, 3.14, 2.63, 2.98, 2.65, 2.43, 
            2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 
            2.94, 2.92, 3.54, 2.51, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.39, 3.23, 3.67, 0.57, 1.3, 1.41, 1.79, 3.29, 1.75, 
            2.65, 2.17, 2, 1.3, 3.49, 1.02, 2.86, 1.61, 2.89, 
            2.14, 1.57, 2.24, 1.64, 1.85, 2.79, 2.26, 2.53, 1.58, 
            1.59, 2.21, 2.17, 1.69, 1.61, 1.69, 2.43, 1.76, 1.25, 
            1.46, 2.69, 2.9, 2.27, 0.99, 2.26, 3.74, 2.99, 2.17, 
            1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 2.61, 2.92, 
            2.65, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 0.68, 
            1.64, 2.97, 3.29, 2.64, 1.69, 3.03, 2.65, 3.15, 2.24, 
            1.36, 1.75, 0.83, 0.47, 0.83, 1.2, 0.58, 0.66, 0.47, 
            0.49, 0.68, 0.6, 0.5, 0.7, 0.52, 0.8, 0.57, 0.55, 
            0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 0.58, 
            0.63, 1.59, 0.58, 1.31, 1.1, 0.92, 1.25, 0.6, 0.34, 
            0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.6, 0.7, 
            0.61, 1.84, 0.69, 0.68, 0.76), nfla = c(0.28, 0.32, 
            0.3, 0.24, 0.39, 0.32, 0.22, 0.31, 0.29, 0.22, 0.22, 
            0.26, 0.29, 0.43, 0.29, 0.3, 0.22, 0.28, 0.32, 0.17, 
            0.24, 0.25, 0.27, 0.26, 0.32, 0.52, 0.34, 0.27, 0.32, 
            0.26, 0.29, 0.24, 0.42, 0.3, 0.29, 0.26, 0.21, 0.29, 
            0.28, 0.2, 0.34, 0.24, 0.28, 0.32, 0.28, 0.3, 0.27, 
            0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 
            0.26, 0.32, 0.26, 0.28, 0.3, 0.53, 0.32, 0.26, 0.45, 
            0.37, 0.26, 0.27, 0.6, 0.24, 0.4, 0.17, 0.61, 0.21, 
            0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.3, 0.4, 
            0.61, 0.22, 0.3, 0.35, 0.48, 0.34, 0.3, 0.5, 0.5, 
            0.29, 0.25, 0.17, 0.32, 0.43, 0.32, 0.24, 0.52, 0.39, 
            0.29, 0.34, 0.37, 0.37, 0.66, 0.29, 0.48, 0.39, 0.29, 
            0.24, 0.26, 0.26, 0.42, 0.3, 0.52, 0.3, 0.28, 0.22, 
            0.29, 0.32, 0.43, 0.43, 0.3, 0.17, 0.39, 0.39, 0.25, 
            0.4, 0.61, 0.53, 0.6, 0.27, 0.17, 0.6, 0.63, 0.53, 
            0.63, 0.58, 0.53, 0.53, 0.53, 0.63, 0.63, 0.29, 0.43, 
            0.4, 0.47, 0.45, 0.25, 0.22, 0.48, 0.26, 0.61, 0.53, 
            0.43, 0.61, 0.63, 0.53, 0.52, 0.6, 0.5, 0.6, 0.4, 
            0.41, 0.6, 0.43, 0.4, 0.39, 0.52, 0.53, 0.48, 0.53, 
            0.45, 0.43, 0.43, 0.53, 0.6), pro = c(2.29, 1.28, 
            2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 
            1.57, 1.81, 1.44, 2.96, 1.46, 2.91, 2.45, 1.86, 1.66, 
            2.1, 0.73, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 
            1.98, 2.38, 1.95, 1.86, 1.66, 1.54, 1.86, 1.36, 1.44, 
            1.37, 2.45, 2.1, 2.81, 1.66, 1.04, 2.03, 1.04, 2.19, 
            2.14, 2.38, 2.08, 1.46, 2.29, 1.87, 1.68, 1.62, 2.45, 
            2.96, 1.66, 2.91, 0.42, 0.86, 1.63, 0.73, 1.87, 1.03, 
            1.04, 1.35, 1.04, 0.97, 2.5, 1.46, 1.87, 1.03, 1.04, 
            1.24, 1.15, 1.46, 1.55, 2.76, 1.95, 1.43, 1.46, 1.35, 
            1.62, 1.26, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 1.63, 
            1.46, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 
            1.35, 1.31, 1.42, 1.28, 1.42, 1.87, 1.63, 2.08, 2.49, 
            1.56, 1.04, 1.1, 1.56, 1.04, 1.14, 1.57, 1.61, 0.83, 
            1.87, 1.83, 2.81, 1.71, 2.14, 2.91, 1.35, 1.77, 1.46, 
            1.9, 1.66, 0.94, 0.83, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 0.68, 0.75, 1.25, 0.55, 1.35, 1.14, 1.3, 
            0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.46, 0.86, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.46, 0.75, 0.94, 
            0.86, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 
            1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 4.38, 
            5.68, 7.8, 2.6, 6.75, 4.7, 5.05, 4.25, 7.22, 5.75, 
            5.4, 5.6, 6.3, 7.5, 4.7, 9.2, 4.68, 8.7, 5.1, 5.65, 
            4.5, 3.7, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 
            5.7, 6.3, 4.35, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 
            6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 5.45, 
            8.9, 7.2, 5, 4.6, 6.3, 5.85, 6.25, 8.42, 4.9, 6.8, 
            4.25, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 
            3.17, 2.85, 9.2, 3.38, 3.74, 2.62, 3.21, 3.8, 4.6, 
            2.12, 4.38, 2.7, 2.5, 3.9, 2.9, 4.8, 3.05, 2.62, 
            2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 
            2.6, 6.8, 2.9, 4.5, 3.25, 1.28, 5.05, 2.8, 7.1, 2.94, 
            2.7, 3.4, 3.3, 2.7, 3.08, 2.9, 4.6, 2.62, 3.08, 2.9, 
            1.9, 3.9, 2.8, 3.4, 1.28, 3.25, 5.24, 3.27, 6.75, 
            2.8, 2.7, 3.94, 2.6, 2.12, 5.75, 7.65, 6.62, 9.2, 
            5, 5.45, 7.1, 7.1, 5, 5.7, 4.92, 7.1, 7.2, 4.35, 
            4.4, 8.21, 4, 4.9, 5, 8.42, 9.4, 10.26, 8.9, 7.1, 
            10.52, 7.6, 7.9, 7.7, 7.5, 13, 10.26, 7.65, 5.58, 
            5.58, 5.7, 8.9, 6.62, 10.68, 10.26, 9.7, 8.5, 7.22, 
            10.52, 9.7, 7.7, 7.3, 7.7, 9.3, 9.2), hue = c(1.04, 
            1.05, 1.04, 0.86, 1.2, 1.05, 1.02, 0.89, 1.08, 1.25, 
            1.25, 1.05, 1.15, 1.25, 1.2, 1.28, 0.95, 1.13, 1.23, 
            1.25, 1.09, 0.82, 1.11, 1.09, 1.12, 1.13, 1.09, 1.02, 
            1.02, 1.04, 1.19, 1.09, 0.98, 1.25, 1.1, 1.31, 1.05, 
            1.07, 1.18, 0.7, 0.95, 0.91, 0.88, 0.82, 0.88, 0.87, 
            1.38, 1.05, 1.07, 1.12, 1.12, 1.42, 1.28, 0.98, 1.1, 
            0.94, 0.94, 1.07, 0.89, 0.69, 1.25, 0.98, 1.23, 1.22, 
            0.98, 1.23, 1.08, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.15, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 
            1.25, 1.31, 0.84, 0.86, 1.23, 1.42, 1.38, 1.38, 1.07, 
            1.08, 1.05, 0.98, 1.15, 1.06, 1.09, 0.57, 0.89, 1.23, 
            1.42, 1.27, 1.25, 0.8, 1.19, 1.04, 0.95, 1, 0.81, 
            0.86, 0.74, 1.2, 0.9, 0.75, 1.1, 0.93, 1.71, 0.95, 
            1.06, 0.7, 0.8, 0.8, 0.93, 0.92, 0.73, 0.73, 0.86, 
            0.69, 0.97, 0.89, 0.79, 0.61, 0.74, 0.66, 0.78, 0.75, 
            0.73, 0.75, 0.82, 0.81, 0.64, 0.87, 0.7, 0.89, 0.74, 
            0.65, 0.6, 0.7, 0.54, 0.55, 0.75, 0.57, 0.48, 0.61, 
            0.56, 0.66, 0.6, 0.57, 0.67, 0.57, 0.55, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.68, 0.74, 0.67, 0.65, 
            0.57, 0.7, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
            3.4, 3.17, 2.96, 2.93, 3.21, 3.58, 3.58, 2.85, 3.55, 
            3.17, 2.82, 2.9, 3.05, 3, 2.84, 2.65, 3.3, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.58, 3.2, 2.85, 2.77, 
            3.63, 2.69, 2.71, 2.88, 2.87, 3, 2.87, 2.75, 3.21, 
            2.72, 2.69, 3.36, 3.38, 3, 3.38, 3, 3.35, 3.33, 3.39, 
            3.33, 2.75, 3.1, 3.33, 3.5, 3.26, 2.93, 3.2, 3.03, 
            3, 2.84, 2.87, 1.82, 2.77, 1.59, 2.46, 2.87, 2.87, 
            2.3, 3.18, 3.48, 2.77, 3.58, 1.82, 3.36, 2.78, 3.5, 
            3.13, 2.14, 2.48, 2.52, 2.78, 2.82, 3.48, 3.14, 2.72, 
            2.01, 3.3, 2.71, 3, 3.21, 2.75, 3.21, 2.83, 2.65, 
            2.3, 3.3, 2.88, 2.63, 1.75, 2.74, 3.1, 2.83, 2.96, 
            2.77, 3.38, 1.64, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 
            2.81, 2.78, 2.5, 2.87, 3.19, 2.87, 3.33, 2.96, 2.12, 
            3.05, 3.39, 3.69, 3.12, 2.05, 3.64, 3.17, 2.84, 3.33, 
            2.78, 2.77, 1.51, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 
            1.69, 1.82, 2.15, 1.27, 1.47, 1.33, 2.05, 2, 1.33, 
            1.33, 1.82, 1.62, 1.33, 1.64, 1.47, 1.33, 1.51, 1.55, 
            1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 2.77, 1.42, 1.75, 
            1.68, 1.75, 1.55, 1.62, 1.27, 1.78, 1.36, 1.62, 1.68, 
            1.58, 2.01, 1.56, 1.62, 1.6), prol = c(1065, 1050, 
            1190, 1050, 735, 1450, 1290, 1295, 1045, 1045, 1285, 
            1280, 1320, 1270, 1547, 1310, 1280, 1095, 1680, 845, 
            1285, 770, 1035, 1015, 845, 1150, 1195, 1285, 915, 
            1035, 1285, 915, 675, 1235, 1190, 920, 880, 970, 
            406, 760, 795, 1035, 1095, 680, 885, 1080, 1065, 
            985, 1060, 795, 1150, 1265, 1190, 1285, 1060, 1120, 
            970, 1270, 1285, 564, 385, 480, 630, 415, 685, 830, 
            502, 425, 415, 718, 870, 410, 472, 425, 886, 564, 
            392, 830, 1265, 463, 278, 714, 720, 515, 590, 1035, 
            510, 562, 680, 625, 480, 450, 495, 870, 345, 1035, 
            625, 428, 660, 406, 564, 562, 438, 415, 672, 278, 
            510, 470, 312, 680, 562, 325, 695, 434, 385, 425, 
            470, 880, 372, 564, 680, 1035, 365, 380, 470, 378, 
            345, 466, 550, 580, 625, 580, 560, 600, 463, 695, 
            720, 515, 580, 590, 600, 480, 520, 550, 385, 830, 
            415, 845, 1015, 550, 500, 480, 425, 675, 640, 740, 
            480, 880, 660, 1065, 510, 680, 570, 885, 615, 480, 
            695, 685, 1015, 886, 510, 470, 970, 740, 870, 835, 
            840, 870)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.38, 13.24, 
        14.38, 14.39, 14.06, 13.78, 13.86, 13.16, 14.75, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 13.86, 13.64, 13.88, 
        12.93, 13.71, 12.85, 13.5, 13.41, 13.39, 13.3, 13.87, 
        13.24, 13.73, 13.58, 13.68, 13.76, 13.17, 14.22, 13.23, 
        13.05, 12.84, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 12.93, 13.9, 14.22, 14.34, 13.05, 14.06, 13.82, 
        12.93, 13.74, 13.58, 14.22, 13.29, 14.3, 12.37, 12.33, 
        12.64, 11.96, 12.37, 12.17, 12.37, 13.69, 12.37, 13.34, 
        12.21, 13.86, 13.86, 11.62, 12.99, 11.96, 11.66, 12.6, 
        11.84, 11.76, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.7, 
        12.16, 11.66, 12.29, 12.08, 12.08, 12, 12.69, 12.37, 
        11.62, 12.08, 11.81, 12.29, 13.73, 12.29, 12.08, 12.6, 
        12.37, 13.16, 12.51, 13.73, 12.08, 12.72, 12.22, 11.61, 
        12.6, 12, 11.76, 12.51, 12.08, 11.03, 11.82, 13.05, 11.87, 
        12.08, 11.45, 13.86, 12.42, 13.05, 11.87, 12.07, 13.3, 
        11.79, 11.79, 12.04, 12.86, 12.88, 12.45, 12.7, 12.51, 
        13.24, 12.25, 12.53, 12.08, 12.84, 12.93, 13.36, 13.52, 
        13.62, 12.25, 11.96, 13.34, 12.87, 13.32, 13.56, 13.62, 
        12.79, 13.34, 13.23, 13.27, 13.17, 13.84, 12.45, 14.34, 
        13.48, 13.2, 13.69, 12.85, 12.96, 13.78, 13.73, 14.13, 
        13.86, 13.58, 13.4, 13.36, 12.77, 14.16, 13.4, 13.4, 
        13.27, 13.17, 14.13), malic = c(1.71, 1.51, 1.35, 1.95, 
        1.67, 2.99, 1.87, 2.15, 1.39, 1.35, 2.16, 1.48, 0.74, 
        1.73, 1.87, 1.67, 1.92, 1.97, 0.99, 3.1, 1.63, 3.8, 1.6, 
        1.6, 1.81, 3.45, 1.73, 1.68, 1.9, 4.31, 1.5, 1.68, 1.83, 
        1.53, 1.68, 0.98, 2.67, 1.65, 1.5, 1.68, 1.71, 1.68, 
        1.39, 3.98, 3.74, 4.04, 0.99, 1.68, 1.9, 0.9, 1.17, 1.65, 
        1.92, 1.9, 1.67, 1.41, 1.7, 1.97, 1.53, 0.94, 1.51, 1.36, 
        1.25, 1.41, 1.61, 1.21, 1.01, 1.17, 2.31, 1.92, 1.9, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.37, 0.99, 3.87, 
        1.73, 2.39, 1.13, 1.67, 3.03, 0.98, 1.61, 1.67, 0.98, 
        1.5, 1.9, 1.51, 1.51, 1.51, 2.68, 1.83, 2.55, 1.41, 1.87, 
        0.9, 2.08, 2.13, 3.12, 1.72, 1.41, 2.55, 1.73, 1.87, 
        1.29, 1.48, 3.74, 2.68, 2.68, 0.74, 1.39, 1.73, 1.29, 
        1.61, 3.43, 1.29, 2.4, 2.05, 4.43, 1.92, 4.31, 2.16, 
        1.53, 2.13, 1.67, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 
        3.98, 2.39, 5.51, 3.59, 4.12, 2.81, 2.56, 2.39, 3.24, 
        3.88, 4.61, 5.04, 4.61, 3.24, 4.1, 3.12, 2.67, 1.9, 2.39, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 1.67, 4.12, 
        3.45, 2.76, 2.59, 3.59, 3.37, 2.58, 4.6, 2.55, 2.39, 
        2.76, 5.65, 3.91, 2.58, 2.59, 4.1), ash = c(2.43, 2.14, 
        2.67, 2.5, 2.28, 2.7, 2.45, 2.61, 2.39, 2.26, 2.3, 2.48, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.6, 2.65, 
        2.17, 2.52, 2.61, 3.22, 2.74, 2.54, 2.04, 2.3, 2.3, 2.36, 
        2.32, 2.7, 2.65, 2.2, 2.32, 2.55, 2.1, 2.51, 2.41, 2.12, 
        2.59, 2.29, 2.1, 2.44, 2.42, 2.12, 2.4, 2.27, 2.04, 2.3, 
        2.62, 2.26, 2.12, 2.14, 2.3, 2.68, 2.5, 1.36, 1.94, 2.37, 
        1.92, 2.16, 2.53, 2.56, 2.28, 1.92, 2.56, 1.75, 2.21, 
        2.67, 2.24, 2.6, 2.3, 2.36, 1.71, 1.92, 2.04, 2.4, 2, 
        2.38, 2.51, 2.32, 2.58, 2.24, 2.52, 2.62, 2.4, 2.3, 2.32, 
        2.1, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.1, 1.94, 1.7, 
        2.42, 2.46, 1.88, 2.2, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 
        2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.32, 1.7, 2, 2.42, 
        3.23, 2.4, 2.13, 2.39, 2.17, 2.29, 2.36, 2.17, 2.38, 
        2.7, 2.4, 2.4, 2.36, 2.25, 2.67, 2.54, 2.64, 2.19, 2.61, 
        2.7, 2.35, 2.72, 2.35, 2.2, 2.29, 2, 2.48, 2.61, 2.36, 
        2.62, 2.48, 1.7, 2.28, 2.26, 2.32, 2.38, 2.42, 2.2, 2.64, 
        2.38, 2.54, 2.4, 2.35, 2.3, 2.51, 2.6, 2.04, 2.69, 3.22, 
        2.32, 2.28, 1.92, 2.68, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 
        17.4, 19, 16.8, 21, 15.2, 11.4, 17.6, 14, 16, 18.8, 16.8, 
        16, 11.4, 16.3, 17, 16.8, 20, 12.4, 15.2, 16, 18.6, 19.5, 
        17.8, 20, 22.5, 16.1, 17, 16.8, 16, 16.3, 19.1, 21.5, 
        21, 19, 20.5, 15.5, 21.5, 15.5, 13.2, 16.2, 18.8, 18, 
        19.5, 17, 17.5, 16, 19.1, 18.8, 17.4, 12.4, 16, 18.1, 
        19.5, 16.8, 20.5, 16.3, 16.8, 18.1, 10.6, 19, 16.8, 18, 
        19, 19, 18.1, 18, 19.6, 20, 16.8, 20.4, 20, 24, 30, 21, 
        15.2, 16, 18, 14.8, 22.5, 19, 18.8, 24, 22.5, 18, 18, 
        22.8, 22.8, 21.6, 18, 22, 22, 21, 18, 18, 19, 21.5, 16, 
        16.2, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 
        20, 19.5, 24.5, 20, 22.8, 22, 21.5, 20.8, 15.2, 16, 21, 
        18.5, 30, 19, 21.5, 21, 21, 21.5, 28.5, 18.5, 22, 24, 
        20, 24, 21.5, 20, 20, 19, 25, 19.5, 21.5, 21, 14.8, 23.5, 
        20, 19, 21, 20, 22.5, 24, 21.5, 24, 22, 16, 18.5, 20, 
        22, 19.5, 21, 25, 21.5, 18.8, 24, 22.5, 18.5, 16, 22.5, 
        23, 19.5, 24.5, 25, 16.8, 19.5, 22.5, 20.5, 23, 20, 19, 
        24.5), mg = c(127, 100, 101, 113, 107, 112, 96, 121, 
        97, 92, 127, 120, 89, 91, 102, 112, 120, 93, 103, 116, 
        126, 102, 101, 95, 89, 124, 93, 105, 107, 96, 101, 106, 
        104, 89, 110, 111, 110, 88, 113, 101, 117, 101, 101, 
        151, 107, 111, 102, 101, 101, 108, 92, 92, 111, 115, 
        115, 102, 118, 102, 108, 88, 101, 100, 94, 87, 104, 98, 
        78, 78, 112, 151, 89, 86, 87, 139, 101, 70, 94, 112, 
        88, 151, 86, 102, 78, 85, 102, 99, 90, 88, 84, 70, 81, 
        89, 84, 88, 102, 108, 134, 85, 88, 101, 97, 88, 102, 
        86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 
        86, 85, 102, 87, 96, 88, 102, 86, 134, 85, 112, 92, 88, 
        80, 122, 112, 98, 106, 98, 94, 89, 105, 88, 101, 85, 
        89, 97, 91, 112, 102, 107, 116, 92, 113, 123, 89, 116, 
        98, 103, 112, 89, 97, 98, 89, 98, 97, 106, 106, 90, 88, 
        104, 88, 105, 112, 96, 86, 91, 95, 102, 102, 120, 120
        ), phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.25, 2.5, 2.6, 
        3.2, 2.98, 2.95, 2.2, 2.6, 3.1, 3.85, 2.85, 2.8, 3.18, 
        3.3, 2.96, 3, 2.41, 2.61, 2.48, 2.53, 2.55, 2.85, 2.4, 
        2.95, 2.65, 3, 2.45, 2.42, 2.95, 2.53, 2.7, 2.6, 2.45, 
        2.95, 2.65, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
        2.75, 2.88, 2.72, 2.45, 3.88, 2.86, 2.95, 2.96, 3.2, 
        3, 3.4, 1.98, 2.05, 2.02, 1.92, 3.5, 1.89, 2.96, 1.65, 
        2.11, 2.53, 3.4, 1.1, 2.95, 1.88, 3.5, 2.5, 1.51, 2, 
        2.53, 1.9, 2.83, 2.53, 2.2, 1.45, 1.65, 2.4, 2.2, 1.51, 
        1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.61, 1.4, 
        2.55, 3.52, 3, 2.56, 2, 2.56, 2.5, 2.2, 2.53, 1.65, 1.95, 
        2.36, 2.74, 3.18, 2.55, 1.75, 2.5, 2.56, 2.46, 1.95, 
        2, 2.5, 2, 2.9, 3.18, 3, 2.3, 2.86, 2.65, 2.74, 2.13, 
        2.22, 2.1, 1.51, 2.2, 1.1, 1.7, 0.98, 1.62, 1.38, 1.88, 
        1.62, 1.83, 1.62, 1.4, 0.98, 2, 1.38, 1.48, 0.98, 1.7, 
        1.93, 1.41, 1.4, 1.48, 2.2, 2.1, 1.48, 2.05, 1.8, 1.9, 
        1.9, 1.7, 2.3, 1.83, 1.48, 1.39, 1.35, 1.9, 1.7, 1.48, 
        1.55, 1.88, 1.25, 1.39, 1.68, 1.8, 1.62, 1.88, 1.65, 
        2.05), fla = c(2.89, 2.76, 2.78, 3.49, 2.33, 3.39, 2.52, 
        2.51, 2.98, 3.14, 3.32, 2.43, 2.76, 2.98, 3.64, 2.91, 
        3.14, 3.4, 3.67, 2.98, 3.49, 2.41, 2.88, 2.37, 2.61, 
        2.68, 2.94, 3, 2.97, 2.33, 3.25, 3.19, 2.69, 3.49, 2.37, 
        2.98, 2.64, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
        3, 2.65, 3.17, 3.4, 2.92, 3.54, 2.33, 2.99, 3.74, 2.79, 
        2.9, 2.78, 2.89, 3.23, 3.67, 0.57, 2.03, 1.41, 1.79, 
        2.41, 1.75, 2.65, 1.61, 2, 1.3, 3.67, 1.02, 2.86, 0.8, 
        2.89, 2.14, 1.57, 1.69, 1.57, 1.85, 2.68, 2.26, 2.53, 
        1.58, 1.59, 2.21, 2.79, 1.69, 1.61, 1.69, 1.92, 1.79, 
        1.25, 1.46, 3, 2.33, 2.27, 0.99, 2.17, 3.67, 2.99, 2.17, 
        1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 2.01, 2.92, 
        2.65, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.1, 1.64, 
        2.33, 3.67, 3.15, 1.76, 3.03, 2.65, 3.15, 2.24, 1.6, 
        1.75, 1.69, 1.3, 0.47, 1.2, 0.58, 0.66, 0.47, 0.66, 0.66, 
        0.6, 0.5, 0.92, 0.52, 0.8, 0.58, 0.55, 0.34, 0.65, 0.76, 
        1.39, 1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.68, 0.58, 
        1.31, 1.1, 0.92, 1.31, 0.6, 0.58, 0.68, 0.47, 0.92, 0.66, 
        0.84, 0.96, 0.49, 0.58, 0.7, 0.61, 0.6, 0.69, 0.68, 0.76
        ), nfla = c(0.28, 0.25, 0.3, 0.24, 0.39, 0.22, 0.17, 
        0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 
        0.17, 0.26, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.3, 
        0.26, 0.34, 0.27, 0.28, 0.26, 0.29, 0.32, 0.42, 0.29, 
        0.29, 0.26, 0.32, 0.61, 0.28, 0.2, 0.34, 0.32, 0.28, 
        0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 
        0.32, 0.39, 0.21, 0.2, 0.26, 0.39, 0.32, 0.28, 0.39, 
        0.53, 0.32, 0.5, 0.45, 0.37, 0.26, 0.27, 0.43, 0.17, 
        0.58, 0.61, 0.45, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 
        0.43, 0.3, 0.29, 0.4, 0.61, 0.22, 0.3, 0.4, 0.6, 0.26, 
        0.39, 0.61, 0.5, 0.29, 0.25, 0.17, 0.32, 0.4, 0.27, 0.24, 
        0.21, 0.17, 0.29, 0.34, 0.37, 0.34, 0.66, 0.39, 0.48, 
        0.39, 0.29, 0.24, 0.26, 0.5, 0.42, 0.39, 0.52, 0.3, 0.31, 
        0.48, 0.3, 0.32, 0.4, 0.43, 0.3, 0.39, 0.26, 0.39, 0.26, 
        0.4, 0.5, 0.26, 0.53, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 
        0.58, 0.53, 0.53, 0.29, 0.53, 0.63, 0.45, 0.43, 0.4, 
        0.47, 0.45, 0.53, 0.22, 0.53, 0.26, 0.61, 0.53, 0.63, 
        0.43, 0.63, 0.53, 0.52, 0.53, 0.5, 0.6, 0.4, 0.41, 0.43, 
        0.43, 0.4, 0.39, 0.63, 0.4, 0.48, 0.53, 0.45, 0.43, 0.43, 
        0.53, 0.45), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 
        1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 2.03, 2.96, 
        1.46, 1.85, 1.4, 1.86, 1.66, 2.1, 0.94, 1.69, 1.46, 1.66, 
        1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.76, 1.66, 
        1.54, 1.86, 1.36, 1.44, 1.37, 1.66, 1.86, 1.46, 1.71, 
        1.43, 2.03, 1.98, 2.19, 2.14, 2.38, 2.08, 1.4, 2.29, 
        1.87, 1.68, 1.62, 2.45, 1.81, 1.66, 1.69, 0.42, 1.98, 
        1.63, 0.73, 1.87, 1.03, 1.46, 1.35, 1.04, 1.14, 2.5, 
        1.46, 1.87, 1.03, 2.5, 1.83, 1.15, 1.46, 1.04, 2.76, 
        1.95, 1.43, 1.03, 1.03, 1.62, 1.35, 1.46, 1.56, 1.34, 
        1.35, 1.38, 1.64, 1.63, 0.83, 1.99, 1.35, 3.28, 1.56, 
        1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 2.08, 1.42, 
        1.87, 1.63, 2.08, 2.49, 1.45, 1.87, 1.46, 1.64, 1.04, 
        1.95, 1.11, 1.61, 0.83, 1.87, 1.83, 2.7, 1.71, 0.97, 
        2.91, 1.35, 1.77, 1.46, 1.9, 1.14, 0.94, 1.54, 0.83, 
        0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.06, 0.75, 0.68, 0.55, 
        0.55, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 
        1.56, 0.94, 1.04, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 
        1.04, 1.15, 0.94, 0.94, 1.15, 1.46, 0.97, 1.54, 1.11, 
        0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 
        4.38, 5.68, 7.8, 5.4, 6.75, 5.64, 5.05, 5.58, 7.22, 5.75, 
        7.7, 5.6, 7.8, 7.5, 5.7, 8.42, 3.21, 8.7, 5.1, 5.65, 
        4.5, 4.38, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 
        5.58, 2.7, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
        4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 6.25, 8.9, 7.2, 
        6.75, 5, 6.3, 5.85, 6.25, 8.5, 4.2, 6.8, 5.6, 3.27, 5.75, 
        3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 7.5, 3.38, 
        3.74, 5.75, 3.21, 3.8, 4.6, 2.7, 5.85, 1.28, 2.5, 3.9, 
        3.8, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
        3.05, 2.15, 3.25, 2.6, 8.21, 2.9, 3.05, 3.4, 4.68, 4.6, 
        2.8, 5.1, 2.94, 2.7, 3.4, 3.3, 2.7, 1.9, 2.9, 2.45, 2.15, 
        3.08, 2.9, 1.9, 2.4, 3.52, 3.4, 1.28, 3.25, 3.74, 3.8, 
        4.36, 2.8, 2.6, 3.94, 1.9, 2.12, 3.08, 5.1, 7.3, 5.45, 
        5, 5.45, 7.1, 4, 5, 5.7, 4.92, 6.8, 4.4, 4.35, 4.4, 8.21, 
        4, 4.9, 5.24, 8.42, 9.4, 5.58, 3.17, 7.1, 10.52, 7.6, 
        7.9, 9.2, 7.5, 13, 7.5, 7.65, 7.1, 5.58, 4.9, 8.42, 6.62, 
        10.68, 10.26, 9.4, 8.5, 7.5, 6.25, 9.7, 7.7, 7.3, 10.68, 
        9.3, 9.2), hue = c(1.04, 1.05, 1.02, 0.86, 1.23, 1.05, 
        1.02, 0.98, 1.08, 1.09, 1.25, 0.79, 1.15, 1.25, 1.2, 
        1.28, 1.02, 1.13, 1.23, 0.73, 1.09, 0.84, 1.11, 1.09, 
        1.12, 1.13, 1.08, 1.02, 1.25, 1.04, 1.19, 1.09, 1.12, 
        1.25, 1.1, 1.04, 0.98, 1.09, 1.18, 0.92, 0.95, 0.91, 
        0.88, 0.82, 1.19, 0.87, 1.07, 1.1, 1.07, 1.12, 1.12, 
        1.07, 0.88, 1.12, 0.97, 1.07, 0.94, 1.07, 0.89, 1.27, 
        1.25, 0.98, 1.23, 1.22, 0.84, 1.15, 1.19, 1.12, 1.02, 
        1.28, 0.906, 1.36, 0.98, 1.31, 1.13, 1.23, 1.19, 0.96, 
        1.06, 1.19, 1.38, 1.05, 1.31, 0.84, 0.79, 1.23, 0.8, 
        0.93, 1.31, 1.07, 1.08, 1.05, 0.8, 1.15, 0.8, 0.98, 0.73, 
        0.97, 1.71, 1.42, 1.27, 0.48, 0.8, 0.93, 1.04, 1.09, 
        1, 1.12, 0.86, 1.31, 1.05, 0.9, 0.96, 1.1, 0.93, 1.71, 
        0.95, 1.12, 0.7, 1.31, 0.8, 1.71, 0.92, 0.73, 0.82, 0.86, 
        0.69, 0.97, 0.89, 0.79, 0.96, 0.74, 0.66, 0.78, 0.75, 
        0.73, 0.75, 0.82, 0.81, 0.82, 0.68, 0.56, 0.89, 0.96, 
        0.65, 0.6, 0.57, 0.54, 0.55, 0.74, 0.82, 0.48, 0.61, 
        0.56, 0.66, 0.6, 0.57, 0.67, 0.57, 0.98, 0.56, 0.96, 
        0.87, 0.68, 0.7, 0.78, 0.85, 0.7, 0.74, 0.67, 0.67, 0.66, 
        0.75, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
        3.17, 3.2, 2.93, 3.55, 3.58, 3.58, 2.85, 3.55, 3.17, 
        2.82, 2.9, 3.58, 3, 2.75, 2.65, 3.71, 2.82, 3.36, 3.71, 
        3.52, 4, 3.63, 2.71, 3.2, 3.02, 2.77, 3.05, 2.87, 2.71, 
        2.88, 2.87, 3, 2.87, 3.52, 3.13, 3, 2.69, 3.5, 3.38, 
        3, 3.92, 3, 3.35, 3.33, 3.33, 3.33, 2.75, 3.1, 2.93, 
        2.87, 3.26, 2.93, 3.2, 3.03, 3.4, 2.84, 2.87, 1.82, 2.72, 
        1.59, 2.46, 2.87, 2.77, 2.3, 3.18, 3.48, 2.72, 3.92, 
        1.82, 3.12, 2.05, 3.5, 3.13, 2.14, 2.48, 2.52, 1.36, 
        2.85, 3.63, 3.14, 2.72, 2.01, 2.71, 3.1, 2.46, 3.21, 
        2.75, 3.21, 2.93, 2.65, 2.48, 3.3, 3.03, 2.63, 1.33, 
        2.74, 3.5, 2.83, 2.96, 2.77, 3.38, 3.19, 3.57, 3.3, 3.17, 
        2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.57, 3.19, 2.87, 
        3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.87, 3.64, 
        3.26, 2.84, 3.63, 2.78, 2.72, 2.72, 1.42, 1.36, 1.29, 
        1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.55, 1.36, 1.78, 
        2.05, 2, 1.75, 1.33, 1.27, 1.62, 1.33, 2.72, 1.47, 1.33, 
        1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 1.75, 
        1.68, 1.75, 1.68, 1.75, 2, 1.55, 1.6, 1.78, 1.58, 1.82, 
        2, 1.96, 1.96, 1.56, 1.62, 1.6), prol = c(1065, 1050, 
        1285, 1280, 735, 1450, 1290, 1295, 1045, 1045, 845, 1280, 
        1320, 1295, 1547, 1310, 1280, 1150, 1680, 845, 1190, 
        770, 1035, 1015, 845, 1015, 1195, 1285, 915, 1035, 1285, 
        1045, 600, 1235, 1060, 920, 880, 740, 680, 760, 795, 
        1035, 1095, 680, 885, 1080, 1065, 985, 1060, 1280, 1150, 
        1265, 1190, 1320, 1060, 1120, 970, 1270, 1285, 278, 378, 
        1035, 630, 378, 1120, 1320, 502, 550, 735, 718, 870, 
        410, 472, 434, 886, 625, 392, 428, 675, 463, 278, 714, 
        434, 515, 434, 1035, 463, 562, 680, 625, 480, 450, 495, 
        740, 345, 392, 625, 428, 660, 406, 680, 562, 438, 415, 
        672, 312, 510, 380, 312, 680, 562, 325, 870, 434, 385, 
        672, 564, 970, 372, 564, 835, 885, 365, 380, 880, 378, 
        615, 466, 580, 580, 770, 845, 560, 600, 428, 695, 720, 
        515, 580, 590, 600, 770, 520, 550, 845, 830, 415, 630, 
        434, 550, 500, 480, 425, 675, 640, 434, 480, 880, 660, 
        695, 718, 680, 570, 886, 615, 830, 695, 685, 845, 835, 
        510, 470, 515, 740, 600, 835, 840, 1035)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 13.68, 13.24, 13.68, 14.39, 14.06, 13.29, 
        13.86, 13.16, 13.32, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 13.23, 13.64, 14.3, 12.93, 13.71, 12.85, 13.5, 
        11.84, 13.39, 13.3, 13.87, 12.85, 13.73, 13.58, 13.68, 
        13.76, 13.4, 11.81, 12.04, 13.05, 13.05, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 14.16, 13.9, 13.83, 
        13.68, 13.05, 12.85, 13.82, 13.2, 13.74, 13.17, 14.22, 
        13.29, 14.38, 12.37, 12.33, 12.64, 12.51, 12.37, 12.17, 
        12.37, 13.05, 11.62, 13.34, 12.21, 12.7, 13.86, 11.76, 
        12.99, 11.96, 11.66, 11.76, 11.84, 13.86, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 12, 12.16, 12.08, 13.36, 12.08, 
        12.08, 12, 12.69, 11.45, 11.62, 12.08, 11.81, 12.29, 
        12.85, 12.29, 12.08, 12.6, 12.21, 12.08, 12.51, 12.7, 
        12.69, 12.72, 12.22, 11.61, 12.21, 12.99, 11.76, 12, 
        12.08, 11.03, 11.82, 11.96, 13.86, 11.45, 11.45, 14.21, 
        12.42, 13.05, 11.87, 12.07, 12.6, 11.79, 11.76, 12.04, 
        12.86, 12.88, 13.16, 12.7, 12.51, 12.25, 12.25, 12.53, 
        12.04, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 13.05, 
        12.69, 12.87, 13.32, 13.05, 12.33, 12.79, 13.41, 13.23, 
        12.96, 13.17, 13.84, 12.45, 14.34, 13.48, 14.75, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.56, 13.75, 13.58, 13.4, 
        12.84, 12.77, 14.16, 13.84, 13.4, 13.27, 13.17, 14.13
        ), malic = c(1.71, 1.67, 1.17, 1.95, 1.83, 1.87, 1.87, 
        2.15, 4.31, 1.35, 2.16, 1.48, 1.87, 1.73, 1.87, 1.87, 
        1.92, 1.5, 0.74, 3.1, 1.63, 3.8, 1.67, 1.6, 1.81, 4.61, 
        1.29, 2.55, 1.9, 1.67, 1.5, 1.51, 1.83, 1.53, 1.35, 1.21, 
        1.97, 1.65, 1.5, 0.94, 1.71, 0.74, 1.73, 3.98, 1.73, 
        4.04, 1.67, 1.68, 1.51, 1.67, 2.68, 1.65, 1.6, 1.9, 1.67, 
        1.87, 1.7, 1.97, 1.71, 0.94, 1.25, 1.36, 1.25, 3.37, 
        1.01, 1.21, 1.01, 1.17, 2.55, 1.97, 1.67, 1.51, 1.66, 
        1.67, 1.09, 1.88, 0.9, 2.99, 0.99, 3.87, 1.83, 2.68, 
        1.13, 4.31, 1.36, 0.98, 1.61, 1.67, 1.13, 2.55, 1.53, 
        1.51, 1.35, 1.92, 1.67, 3.91, 2.31, 1.41, 2.05, 4.12, 
        2.08, 1.21, 5.51, 1.72, 1.87, 2.55, 1.73, 2.15, 1.29, 
        1.68, 3.74, 3.87, 2.68, 0.74, 1.39, 1.51, 1.83, 1.61, 
        3.43, 2.4, 2.4, 2.05, 4.43, 1.97, 4.31, 2.16, 1.53, 2.13, 
        3.1, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 5.51, 3.83, 5.51, 
        3.59, 3.24, 2.81, 2.56, 2.76, 2.99, 3.88, 2.31, 5.04, 
        4.61, 3.24, 1.6, 3.12, 2.67, 1.9, 3.12, 1.29, 5.19, 4.12, 
        3.03, 1.68, 1.67, 3.83, 1.65, 3.12, 3.45, 2.76, 1.51, 
        1.6, 3.37, 2.58, 4.6, 1.39, 2.39, 1.51, 5.65, 3.91, 3.1, 
        2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.27, 2.68, 
        2.45, 2.61, 2.53, 2.65, 2.3, 2.28, 2.41, 2.39, 2.38, 
        2.7, 2.72, 2.62, 2.48, 2.56, 2.38, 2.65, 2.3, 2.52, 2.61, 
        3.22, 2.32, 2.69, 3.22, 1.94, 2.7, 2.36, 2.37, 2.7, 2.65, 
        2.14, 1.7, 2.55, 2.1, 2.51, 2.4, 2.12, 2.59, 2.29, 2.1, 
        2.44, 2.48, 2.12, 2.4, 2.27, 2.04, 2.69, 2.29, 2.51, 
        2.25, 2.61, 2.3, 2.68, 2.5, 1.36, 1.99, 1.88, 1.92, 2.16, 
        2.53, 2.56, 2.28, 1.92, 2.29, 1.75, 2.21, 2.67, 2.24, 
        2.6, 2.3, 2.6, 1.71, 2.5, 2.32, 2.4, 2, 2.45, 2.51, 2.32, 
        2.58, 2.24, 2.36, 2.62, 2.3, 2.3, 2.32, 2.24, 2.26, 2.22, 
        2.28, 2.2, 2.74, 1.98, 2.17, 2.4, 1.7, 1.82, 2.46, 1.88, 
        2.3, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 
        2.5, 2.2, 1.99, 2.38, 2.28, 2, 2.42, 3.23, 2.7, 2.13, 
        2.39, 2.17, 2.29, 3.22, 2.22, 2.38, 1.88, 2.4, 2.4, 2.36, 
        2.25, 2.6, 2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 
        2.2, 2.61, 2.4, 2.48, 2, 2.36, 2.62, 2.48, 1.75, 2.28, 
        2.28, 2.32, 2.38, 2.61, 2.68, 2.64, 2.38, 2.54, 2.56, 
        2.35, 2.3, 2.54, 2.6, 2.54, 2.69, 2.27, 2.32, 2.28, 2.58, 
        2.61, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 13.2, 20, 
        16.8, 21, 15.2, 17, 17.6, 14, 16, 13.2, 16.8, 16, 11.4, 
        16.1, 20, 17.8, 20, 15.5, 15.2, 16, 18.6, 15.2, 17.8, 
        20, 23, 16.1, 17, 23, 17.4, 16.2, 19.1, 15.2, 20.5, 19, 
        20.5, 15.5, 19.5, 15.5, 13.2, 16.2, 18.8, 18.5, 16.8, 
        17, 17.4, 16, 16.8, 18.8, 17.4, 12.4, 25, 15.5, 17.8, 
        18, 20.5, 16.3, 16.8, 17.6, 10.6, 19, 16.8, 18, 19, 19, 
        18.1, 22.5, 19.6, 18, 16.8, 20.4, 20, 24, 30, 21, 20, 
        16, 18, 14.8, 22, 19, 19.5, 24, 22.5, 18, 18, 22.8, 23, 
        21.6, 18.6, 21, 22, 20, 18, 18, 19, 21.5, 16, 16.8, 18, 
        17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 
        19.5, 23.5, 23, 20, 21.5, 20.8, 21.5, 16, 19.5, 24.5, 
        18, 22, 21.5, 21, 21, 21.5, 28.5, 20, 22, 15.2, 20, 24, 
        21.5, 23, 21.5, 16, 25, 19.5, 23, 21, 20.5, 23.5, 20, 
        19.5, 21, 20, 23.5, 19, 21.5, 24, 22, 19, 18.5, 20, 22, 
        19.5, 21.5, 25, 23, 19, 25, 23, 18.5, 18, 22.5, 23, 19.5, 
        24.5, 25, 21.5, 19.5, 20.5, 20.5, 23, 20, 20, 24.5), 
            mg = c(127, 100, 101, 113, 90, 112, 96, 121, 97, 
            80, 97, 102, 89, 91, 102, 112, 120, 98, 116, 116, 
            126, 102, 97, 95, 85, 124, 93, 111, 107, 96, 101, 
            106, 104, 98, 110, 98, 110, 93, 87, 116, 117, 115, 
            101, 126, 107, 111, 102, 101, 104, 108, 92, 102, 
            111, 115, 106, 112, 118, 102, 108, 88, 101, 100, 
            94, 87, 104, 98, 78, 78, 84, 151, 122, 86, 87, 139, 
            101, 86, 92, 112, 101, 86, 86, 85, 78, 85, 116, 99, 
            90, 88, 84, 70, 81, 87, 97, 88, 88, 111, 134, 85, 
            88, 102, 97, 88, 112, 86, 85, 90, 80, 84, 92, 94, 
            107, 88, 103, 88, 84, 85, 86, 89, 84, 87, 96, 139, 
            102, 86, 127, 85, 97, 92, 88, 80, 122, 84, 98, 106, 
            106, 94, 89, 104, 88, 101, 97, 89, 97, 86, 112, 102, 
            96, 88, 92, 113, 123, 112, 116, 98, 103, 89, 89, 
            97, 98, 89, 105, 106, 106, 106, 90, 88, 110, 88, 
            105, 112, 96, 86, 91, 95, 102, 116, 120, 123), phe = c(2.8, 
            2.65, 2.8, 3.85, 2.8, 2.98, 2.5, 2.6, 3, 2.98, 2.95, 
            2.2, 2.6, 3.1, 2.88, 2.85, 2.8, 3, 3.3, 2.7, 3, 2.41, 
            2.61, 2.48, 2.53, 2.22, 3, 2.4, 2.95, 2.65, 3, 2.8, 
            2.42, 2.95, 2.02, 2.7, 2.6, 2.45, 2.6, 2.74, 3.15, 
            2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 
            2.72, 2.45, 3.88, 2.95, 3, 2.96, 3.2, 3, 3.4, 1.98, 
            2.05, 2.02, 2.2, 3.5, 1.89, 2.53, 2.1, 2.11, 2.53, 
            2.88, 1.1, 2.95, 1.88, 2.56, 2.2, 1.92, 1.65, 2.8, 
            1.9, 2.83, 2.2, 2.2, 1.62, 1.65, 1.89, 2.2, 2, 1.92, 
            1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.46, 1.38, 
            2.55, 3.52, 2.72, 2.56, 1.95, 2.56, 2.5, 2.2, 2.48, 
            1.65, 2.2, 2.36, 2.74, 3.18, 2.55, 1.75, 2.1, 2.56, 
            2.46, 1.4, 2, 2.02, 2, 2.9, 3.18, 2.45, 2.53, 2.86, 
            2.7, 2.74, 2.13, 2.22, 2.1, 1.51, 2.05, 1.39, 1.7, 
            1.68, 1.62, 1.38, 1.65, 1.62, 1.98, 1.1, 1.4, 1.39, 
            2, 1.38, 1.38, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
            2.2, 1.95, 1.48, 1.83, 1.8, 1.9, 2.22, 1.93, 2.3, 
            1.83, 1.48, 1.39, 1.35, 1.39, 1.7, 1.48, 1.55, 1.6, 
            1.25, 1.39, 1.68, 1.35, 1.48, 1.51, 1.65, 2.05), 
            fla = c(2.9, 2.76, 2.97, 3.49, 2.52, 3.39, 2.52, 
            2.51, 2.98, 3.67, 3.32, 2.43, 2.76, 3.03, 3.64, 2.91, 
            3.14, 3.4, 3.49, 2.86, 3.29, 2.41, 2.88, 2.37, 2.61, 
            2.68, 2.94, 2.17, 2.97, 2.33, 3.25, 3.19, 2.69, 3.64, 
            2.65, 2.98, 2.69, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 
            2.63, 3, 2.65, 3.17, 3.23, 2.92, 3.54, 3.19, 2.99, 
            3.74, 2.79, 2.9, 2.78, 2.91, 3.23, 3.67, 0.57, 1.61, 
            1.41, 1.79, 3.19, 1.75, 2.65, 1.64, 2, 1.3, 2.9, 
            1.02, 2.86, 1.28, 2.89, 2.14, 1.57, 1.3, 2.68, 1.85, 
            2.43, 2.26, 2.53, 1.58, 1.59, 2.21, 2.17, 1.69, 1.61, 
            1.69, 2.09, 1.1, 1.25, 1.46, 2.89, 3.23, 2.27, 0.99, 
            1.92, 3.17, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
            2.03, 1.76, 2.09, 2.92, 2.79, 2.27, 2.03, 2.01, 2.29, 
            2.17, 1.6, 2.09, 1.59, 1.64, 2.17, 3.49, 2.01, 3.15, 
            3.03, 2.65, 3.15, 2.24, 1.75, 1.75, 1.1, 1.46, 0.66, 
            1.2, 0.58, 0.66, 0.47, 0.96, 0.6, 0.6, 0.5, 0.55, 
            0.52, 0.8, 0.66, 0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 
            1.36, 1.28, 0.83, 0.58, 0.63, 0.47, 0.58, 1.31, 1.1, 
            0.92, 0.99, 0.6, 0.66, 0.68, 0.47, 0.92, 0.66, 0.84, 
            0.96, 0.49, 0.49, 0.7, 0.61, 0.34, 0.69, 0.68, 0.76
            ), nfla = c(0.28, 0.39, 0.3, 0.24, 0.39, 0.31, 0.22, 
            0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 
            0.32, 0.24, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.29, 
            0.29, 0.34, 0.27, 0.43, 0.26, 0.29, 0.34, 0.42, 0.26, 
            0.29, 0.26, 0.3, 0.2, 0.28, 0.2, 0.34, 0.22, 0.27, 
            0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 
            0.32, 0.39, 0.21, 0.2, 0.26, 0.3, 0.34, 0.28, 0.3, 
            0.53, 0.32, 0.37, 0.45, 0.37, 0.26, 0.27, 0.53, 0.39, 
            0.29, 0.4, 0.3, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 
            0.43, 0.3, 0.37, 0.4, 0.61, 0.22, 0.3, 0.43, 0.34, 
            0.32, 0.53, 0.37, 0.5, 0.32, 0.25, 0.17, 0.32, 0.42, 
            0.32, 0.24, 0.25, 0.31, 0.29, 0.34, 0.37, 0.43, 0.66, 
            0.24, 0.48, 0.39, 0.29, 0.24, 0.26, 0.43, 0.42, 0.26, 
            0.52, 0.3, 0.27, 0.26, 0.26, 0.32, 0.32, 0.43, 0.3, 
            0.32, 0.3, 0.39, 0.43, 0.4, 0.43, 0.28, 0.43, 0.27, 
            0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.5, 
            0.63, 0.4, 0.53, 0.43, 0.4, 0.47, 0.45, 0.24, 0.22, 
            0.26, 0.26, 0.61, 0.53, 0.6, 0.63, 0.63, 0.53, 0.52, 
            0.45, 0.5, 0.6, 0.4, 0.41, 0.43, 0.43, 0.4, 0.39, 
            0.27, 0.43, 0.48, 0.61, 0.52, 0.43, 0.43, 0.53, 0.61
            ), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 
            1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.66, 2.96, 1.46, 
            1.81, 1.86, 1.86, 1.66, 2.1, 1.37, 1.69, 1.46, 1.66, 
            1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.76, 2.18, 
            1.54, 1.86, 1.36, 1.44, 1.37, 2.81, 2.81, 1.83, 2.49, 
            1.82, 2.03, 1.71, 2.19, 2.14, 2.38, 2.08, 1.97, 2.29, 
            1.87, 1.68, 1.62, 2.45, 2.18, 1.66, 2.91, 0.42, 1.11, 
            1.66, 0.73, 1.87, 1.03, 2.45, 1.35, 1.04, 1.1, 2.5, 
            1.46, 1.87, 1.03, 2.08, 0.55, 1.15, 1.46, 1.63, 2.76, 
            1.95, 1.43, 1.43, 2.29, 1.62, 2.7, 1.46, 1.56, 1.34, 
            1.35, 1.38, 1.64, 1.63, 1.87, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.42, 1.42, 
            1.9, 1.63, 2.08, 2.49, 1.63, 1.56, 1.99, 1.42, 1.04, 
            2.5, 0.64, 1.61, 0.83, 1.87, 1.83, 1.97, 1.71, 2.29, 
            2.91, 1.35, 1.77, 1.28, 1.9, 1.46, 0.94, 1.66, 0.83, 
            0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.35, 0.75, 1.35, 
            0.55, 0.94, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 
            1.26, 1.56, 1.56, 1.83, 1.55, 1.56, 1.14, 2.7, 2.29, 
            1.04, 0.97, 1.14, 0.94, 0.64, 1.15, 1.46, 0.97, 1.54, 
            1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
            ), col = c(5.64, 4.38, 5.68, 7.8, 3.3, 6.75, 4.8, 
            5.05, 7.22, 7.22, 5.75, 5.7, 5.6, 5.24, 7.5, 4.9, 
            7.3, 4.28, 8.7, 5.1, 5.65, 4.5, 5.1, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.64, 7.1, 5.4, 4.2, 
            5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 
            5.04, 5.24, 4.9, 6.1, 5, 8.9, 7.2, 5.65, 4.5, 6.3, 
            5.85, 6.25, 6.1, 4.28, 6.8, 3.7, 3.27, 5.75, 3.8, 
            4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 5.58, 3.38, 
            3.74, 2.7, 3.21, 3.8, 4.6, 4.5, 7.1, 3.74, 2.5, 3.9, 
            3.8, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 
            3.6, 3.05, 2.15, 3.25, 2.6, 5.4, 2.9, 2.7, 2.95, 
            2.62, 2.9, 2.8, 3.58, 2.94, 2.7, 3.4, 3.3, 2.7, 3.74, 
            2.9, 4.45, 5.1, 3.08, 2.9, 1.9, 2.95, 3.08, 3.4, 
            1.28, 3.25, 8.21, 4.5, 2.8, 2.8, 3.58, 3.94, 3.05, 
            2.12, 3.8, 3.05, 5.45, 9.4, 5, 5.45, 7.1, 5.7, 5, 
            5.7, 4.92, 5.64, 5, 4.35, 4.4, 8.21, 4, 4.9, 6.13, 
            8.42, 9.4, 5, 7.7, 7.1, 10.52, 7.6, 7.9, 10.68, 7.5, 
            13, 7.1, 7.65, 8.9, 5.58, 6.75, 7.3, 6.62, 10.68, 
            10.26, 10.26, 8.5, 5.45, 6.13, 9.7, 7.7, 7.3, 6.8, 
            9.3, 9.2), hue = c(1.04, 1.05, 0.89, 0.86, 1.23, 
            1.05, 1.02, 1.28, 1.08, 0.95, 1.25, 1.04, 1.15, 1.25, 
            1.2, 1.28, 0.98, 1.13, 1.23, 1.19, 1.09, 1.02, 1.11, 
            1.09, 1.12, 1.13, 1.2, 1.02, 0.86, 1.04, 1.19, 1.09, 
            1.04, 1.25, 1.1, 1.19, 1.25, 1.09, 1.18, 1.05, 0.95, 
            0.91, 0.88, 0.82, 1.31, 0.87, 1.13, 1.19, 1.07, 1.12, 
            1.12, 1.28, 1.28, 1.08, 0.86, 0.95, 0.94, 1.07, 0.89, 
            1.08, 1.25, 0.98, 1.23, 1.22, 1.1, 1.15, 0.6, 1.12, 
            1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 0.89, 1.23, 
            1.19, 0.96, 1.06, 1.19, 1.38, 1.12, 1.31, 0.84, 1.25, 
            1.23, 1.22, 0.88, 0.8, 1.07, 1.08, 1.05, 0.98, 1.15, 
            0.91, 0.91, 0.79, 1.12, 1.25, 1.42, 1.27, 1.12, 0.8, 
            1.19, 1.04, 1.22, 1, 1.06, 0.86, 1.15, 1.22, 0.9, 
            1.12, 1.1, 0.93, 1.71, 0.95, 1.25, 0.7, 1.07, 0.8, 
            1.27, 0.92, 0.73, 1.28, 0.86, 0.69, 0.97, 0.89, 0.79, 
            0.906, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
            0.81, 0.89, 0.89, 0.81, 0.89, 0.89, 0.65, 0.6, 0.78, 
            0.54, 0.55, 0.56, 0.75, 0.48, 0.61, 0.56, 0.65, 0.6, 
            0.57, 0.67, 0.57, 0.78, 0.56, 0.96, 0.87, 0.68, 0.7, 
            0.78, 0.85, 0.61, 0.74, 0.67, 0.87, 0.7, 0.56, 0.64, 
            0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 
            2.87, 2.93, 3.17, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
            2.9, 2.75, 3, 3.38, 2.65, 3.26, 2.82, 3.36, 3.71, 
            3.52, 4, 3.63, 3.58, 3.2, 2.82, 2.77, 2.93, 2.69, 
            2.71, 2.88, 2.87, 3, 2.87, 3.38, 2.75, 3, 2.69, 2.87, 
            3.38, 3, 3.92, 3, 3.35, 3.33, 3, 3.33, 2.75, 3.1, 
            3.58, 2.83, 3.26, 2.93, 3.2, 3.03, 3.52, 2.84, 2.87, 
            1.82, 2.77, 1.59, 2.46, 2.87, 2.77, 2.3, 3.18, 3.48, 
            2.48, 3.21, 1.82, 2.52, 3.02, 3.5, 3.13, 2.14, 2.48, 
            2.52, 1.78, 2.87, 2.77, 3.14, 2.72, 2.01, 3.36, 2.77, 
            2.74, 3.21, 2.75, 3.21, 2.14, 2.65, 2.48, 3.3, 2.83, 
            2.63, 2.14, 2.74, 3.3, 2.83, 2.96, 2.77, 3.38, 2.87, 
            3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 
            3.48, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 
            3.12, 2.69, 3.64, 2.87, 2.84, 3, 2.78, 2.74, 2.3, 
            1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 
            1.58, 1.6, 1.58, 2.05, 2, 1.29, 1.33, 1.62, 1.62, 
            1.33, 3.17, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 
            1.96, 1.78, 1.58, 1.6, 1.33, 1.75, 1.68, 1.75, 1.51, 
            1.51, 1.56, 1.62, 1.59, 1.75, 1.58, 1.62, 1.48, 1.56, 
            1.62, 1.6), prol = c(1065, 1050, 985, 1285, 735, 
            1450, 1290, 1295, 1045, 1045, 1150, 1280, 1320, 1285, 
            1547, 1310, 1280, 1295, 1680, 845, 1290, 770, 1035, 
            1015, 845, 845, 1195, 1285, 915, 1035, 1285, 1095, 
            1045, 1235, 1065, 920, 880, 970, 1035, 760, 795, 
            1035, 1095, 680, 885, 1080, 1065, 985, 1060, 1310, 
            1150, 1265, 1190, 1547, 1060, 1120, 970, 1270, 1285, 
            378, 675, 680, 630, 740, 845, 870, 502, 425, 415, 
            718, 870, 410, 472, 680, 886, 434, 392, 1320, 695, 
            463, 278, 714, 510, 515, 615, 600, 510, 562, 680, 
            625, 480, 450, 495, 434, 345, 380, 625, 428, 660, 
            406, 515, 562, 438, 415, 672, 740, 510, 450, 312, 
            680, 562, 325, 760, 434, 385, 720, 450, 562, 372, 
            564, 550, 1680, 365, 380, 680, 378, 680, 466, 380, 
            580, 580, 365, 560, 600, 406, 695, 720, 515, 580, 
            590, 600, 450, 520, 550, 415, 830, 415, 502, 564, 
            550, 500, 480, 425, 675, 640, 378, 480, 880, 660, 
            428, 718, 680, 570, 480, 615, 425, 695, 685, 680, 
            590, 510, 470, 740, 740, 720, 835, 840, 880)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 14.34, 13.24, 14.38, 14.39, 14.06, 14.39, 
        13.86, 13.74, 13.39, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.34, 13.64, 13.73, 12.93, 13.71, 12.85, 13.5, 
        13.24, 13.39, 13.3, 13.87, 13.39, 13.73, 13.58, 13.68, 
        13.76, 13.75, 13.69, 11.76, 13.05, 13.86, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.58, 13.9, 14.13, 
        14.34, 13.05, 12.25, 13.82, 14.39, 13.74, 13.3, 14.22, 
        13.29, 14.38, 12.37, 12.33, 12.64, 11.62, 12.37, 12.17, 
        12.37, 13.24, 12.08, 13.34, 12.21, 11.03, 13.86, 12.04, 
        12.99, 11.96, 11.66, 12, 11.84, 12.22, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 12.6, 12.16, 12.04, 11.82, 12.08, 
        12.08, 12, 12.69, 12.64, 11.62, 12.6, 11.81, 12.29, 13.5, 
        12.29, 12.08, 12.6, 12.69, 11.84, 12.51, 11.79, 11.81, 
        12.72, 12.22, 11.61, 13.05, 11.62, 11.76, 12.33, 12.08, 
        11.03, 11.82, 11.82, 11.03, 12.21, 11.45, 13.76, 12.42, 
        13.05, 11.87, 12.07, 13.24, 11.79, 12.6, 12.04, 12.86, 
        12.88, 12.17, 12.7, 12.51, 13.4, 12.25, 12.53, 13.83, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.22, 12.07, 
        12.87, 13.32, 14.38, 11.03, 12.79, 12.51, 13.23, 13.05, 
        13.17, 13.84, 12.45, 14.34, 13.48, 13.87, 13.69, 12.85, 
        12.96, 13.78, 13.73, 13.05, 13.76, 13.58, 13.4, 12, 12.77, 
        14.16, 14.23, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
        1.48, 1.66, 1.95, 1.67, 1.5, 1.87, 2.15, 2.16, 1.35, 
        2.16, 1.48, 2.99, 1.73, 1.87, 1.67, 1.92, 0.94, 2.13, 
        3.1, 1.63, 3.8, 1.7, 1.6, 1.81, 1.53, 2.16, 1.67, 1.9, 
        2.76, 1.5, 1.61, 1.83, 1.53, 3.45, 3.45, 3.91, 1.65, 
        1.5, 3.91, 1.71, 3.43, 2.15, 3.98, 1.5, 4.04, 1.53, 1.68, 
        1.51, 1.48, 2.56, 1.65, 1.61, 1.9, 1.67, 2.55, 1.7, 1.97, 
        2.59, 0.94, 1.35, 1.36, 1.25, 0.98, 3.91, 1.21, 1.01, 
        1.17, 3.91, 1.63, 1.72, 1.51, 1.66, 1.67, 1.09, 1.88, 
        0.9, 0.99, 0.99, 3.87, 1.9, 1.67, 1.13, 3.43, 5.04, 0.98, 
        1.61, 1.67, 0.98, 3.88, 1.73, 1.51, 1.5, 3.88, 1.9, 3.45, 
        4.6, 1.41, 1.01, 1.29, 2.08, 2.31, 4.6, 1.72, 1.63, 2.55, 
        1.73, 1.51, 1.29, 1.29, 3.74, 0.98, 2.68, 0.74, 1.39, 
        1.13, 1.66, 1.61, 3.43, 1.9, 2.4, 2.05, 4.43, 3.91, 4.31, 
        2.16, 1.53, 2.13, 2.59, 4.3, 1.35, 2.99, 2.31, 3.55, 
        1.24, 3.43, 4.43, 5.51, 3.59, 0.94, 2.81, 2.56, 1.51, 
        3.83, 3.88, 4.1, 5.04, 4.61, 3.24, 1.87, 3.12, 2.67, 
        1.9, 2.31, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 
        2.67, 3.87, 3.45, 2.76, 2.16, 1.67, 3.37, 2.58, 4.6, 
        3.83, 2.39, 5.65, 5.65, 3.91, 3.74, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.5, 2.53, 2.45, 2.61, 2.1, 1.88, 2.3, 
        2.43, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 
        2.67, 2.65, 2.42, 2.52, 2.61, 3.22, 2.72, 2.72, 2.17, 
        2.36, 2.43, 2.36, 2.35, 2.7, 2.65, 2.59, 2.28, 2.55, 
        2.1, 2.51, 1.92, 2.12, 2.59, 2.29, 2.1, 2.44, 2.29, 2.12, 
        2.4, 2.27, 2.04, 1.92, 2.41, 2.1, 2.6, 2.3, 2.3, 2.68, 
        2.5, 1.36, 2.17, 2.28, 1.92, 2.16, 2.53, 2.56, 2.4, 1.92, 
        1.82, 1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.04, 1.71, 1.75, 
        1.98, 2.4, 2, 2.38, 2.51, 2.32, 2.58, 2.24, 2.16, 2.62, 
        2.62, 2.3, 2.32, 2.7, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 
        2.28, 2.2, 1.7, 2.04, 2.46, 1.88, 2.52, 2.27, 2.12, 2.28, 
        1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.32, 
        1.94, 2, 2.42, 3.23, 2.36, 2.13, 2.39, 2.17, 2.29, 2.74, 
        2.17, 2.38, 2.62, 2.4, 2.4, 2.36, 2.25, 2.17, 2.54, 2.64, 
        2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.4, 2.52, 2.48, 
        2.64, 2.36, 2.62, 2.48, 2.1, 2.28, 2.3, 2.32, 2.38, 2.24, 
        2.48, 2.64, 2.38, 2.54, 2, 2.35, 2.3, 2.24, 2.6, 2.36, 
        2.69, 2.61, 2.32, 2.28, 2.38, 2.56, 2.48, 2.26, 2.37, 
        2.74), alca = c(15.6, 17.4, 18, 16.8, 21, 15.2, 16, 17.6, 
        14, 16, 16.8, 16.8, 16, 11.4, 12.4, 23, 20.4, 20, 15.2, 
        15.2, 16, 18.6, 22, 17.8, 20, 28.5, 16.1, 17, 19.1, 19, 
        17.5, 19.1, 19, 21, 19, 20.5, 15.5, 21, 15.5, 13.2, 16.2, 
        18.8, 19, 16, 17, 22.8, 16, 18.8, 18.8, 17.4, 12.4, 16, 
        20, 16, 19.1, 20.5, 16.3, 16.8, 18.8, 10.6, 19.5, 16.8, 
        18, 19, 19, 18.1, 20.5, 19.6, 16, 16.8, 20.4, 25, 24, 
        30, 21, 16, 16, 18, 14.8, 18.5, 19, 19, 24, 22.5, 18, 
        18, 22.8, 24, 21.6, 19.5, 20, 22, 21.5, 18, 18, 19, 21.5, 
        16, 16.8, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 
        19, 20, 19.5, 19.5, 24, 21.5, 20, 21.5, 20.8, 22, 16, 
        18, 24, 21.5, 21, 21.5, 21, 21, 21.5, 28.5, 18, 22, 24, 
        20, 24, 21.5, 16.2, 20, 21.5, 25, 19.5, 28.5, 21, 21, 
        23.5, 20, 23.5, 21, 20, 19, 24, 21.5, 24, 22, 20, 18.5, 
        20, 22, 19.5, 16.2, 25, 24, 25, 23.5, 20.5, 18.5, 19, 
        22.5, 23, 19.5, 24.5, 25, 22, 19.5, 18.5, 20.5, 23, 20, 
        23, 24.5), mg = c(127, 100, 101, 113, 112, 112, 96, 121, 
        97, 112, 97, 105, 89, 91, 102, 112, 120, 101, 116, 116, 
        126, 102, 86, 95, 85, 124, 93, 102, 107, 96, 101, 106, 
        104, 101, 110, 127, 110, 123, 94, 112, 117, 112, 101, 
        78, 107, 111, 102, 101, 102, 108, 92, 102, 111, 115, 
        151, 123, 118, 102, 108, 88, 101, 100, 94, 87, 104, 98, 
        78, 78, 80, 151, 97, 86, 87, 139, 101, 78, 70, 112, 85, 
        106, 86, 96, 78, 85, 115, 99, 90, 88, 84, 70, 81, 86, 
        87, 88, 98, 85, 134, 85, 88, 102, 97, 88, 120, 86, 85, 
        90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 87, 
        86, 87, 96, 120, 102, 86, 99, 85, 96, 92, 88, 80, 122, 
        78, 98, 106, 87, 94, 89, 88, 88, 101, 102, 89, 97, 84, 
        112, 102, 85, 91, 92, 113, 123, 107, 116, 98, 103, 106, 
        89, 97, 98, 89, 102, 106, 106, 106, 90, 88, 103, 88, 
        105, 112, 96, 86, 91, 95, 102, 101, 120, 127), phe = c(2.8, 
        2.65, 2.8, 3.85, 2.8, 3.25, 2.5, 2.6, 3.2, 2.98, 2.95, 
        2.2, 2.6, 3.1, 3, 2.85, 2.8, 2.72, 3.3, 2.83, 3, 2.41, 
        2.61, 2.48, 2.53, 2.8, 2.86, 2.4, 2.95, 2.65, 3, 3.18, 
        2.42, 2.95, 3.5, 2.7, 2.6, 2.45, 2.74, 3.85, 3.15, 2.45, 
        3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 
        3.88, 2.61, 3.85, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 
        1.41, 3.5, 1.89, 2.2, 2.8, 2.11, 2.53, 3, 1.1, 2.95, 
        1.88, 2.72, 1.65, 2.5, 1.7, 1.95, 1.9, 2.83, 2.41, 2.2, 
        1.88, 1.65, 2.56, 2.2, 2.2, 1.92, 1.95, 2.2, 1.6, 1.45, 
        1.38, 2.45, 3.02, 2.8, 1.92, 2.55, 3.52, 2.95, 2.2, 1.39, 
        2.56, 2.5, 2.2, 2.1, 1.65, 1.89, 2.36, 2.74, 3.18, 2.55, 
        1.75, 2.05, 2.56, 2.46, 1.38, 2, 2.05, 2, 2.9, 3.18, 
        2.85, 1.39, 2.86, 3.02, 2.74, 2.13, 2.22, 2.1, 1.51, 
        1.45, 1.25, 1.7, 1.38, 1.62, 1.38, 2.3, 1.62, 1.25, 1.48, 
        1.4, 1.51, 2, 1.38, 1.9, 0.98, 1.7, 1.93, 1.41, 1.4, 
        1.48, 2.2, 2.05, 1.48, 1.89, 1.8, 1.9, 1.41, 1.48, 2.3, 
        1.83, 1.48, 1.39, 1.35, 1.38, 1.7, 1.48, 1.55, 1.8, 1.25, 
        1.39, 1.68, 2.3, 1.48, 1.38, 1.65, 2.05), fla = c(3.64, 
        2.76, 3.67, 3.49, 2.65, 3.39, 2.52, 2.51, 2.98, 3.17, 
        3.32, 2.43, 2.76, 3.04, 3.64, 2.91, 3.14, 3.4, 3.67, 
        2.92, 3.39, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 1.85, 
        2.97, 2.33, 3.25, 3.19, 2.69, 2.27, 3.19, 2.98, 2.65, 
        2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 
        3.67, 2.92, 3.54, 2.79, 2.99, 3.74, 2.79, 2.9, 2.78, 
        2.68, 3.23, 3.67, 0.57, 2, 1.41, 1.79, 2.91, 1.75, 2.65, 
        2.65, 2, 1.3, 3.64, 1.02, 2.86, 1.3, 2.89, 2.14, 1.57, 
        1.75, 1.84, 1.85, 2.33, 2.26, 2.53, 1.58, 1.59, 2.21, 
        2.63, 1.69, 1.61, 1.69, 1.84, 1.69, 1.25, 1.46, 2.68, 
        2.68, 2.27, 0.99, 2.63, 2.9, 2.99, 2.17, 1.36, 2.11, 
        1.64, 1.92, 1.84, 2.03, 1.76, 2.21, 2.92, 2.33, 2.27, 
        2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.31, 1.64, 2.79, 
        3.64, 2.29, 0.47, 3.03, 2.65, 3.15, 2.24, 2.43, 1.75, 
        1.1, 0.65, 0.65, 1.2, 0.58, 0.66, 0.47, 0.76, 0.68, 0.6, 
        0.5, 0.68, 0.52, 0.8, 1.2, 0.55, 0.34, 0.65, 0.76, 1.39, 
        1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.52, 0.58, 1.31, 
        1.1, 0.92, 0.57, 0.6, 1.36, 0.68, 0.47, 0.92, 0.66, 0.84, 
        0.96, 0.49, 0.7, 0.7, 0.61, 1.2, 0.69, 0.68, 0.76), nfla = c(0.28, 
        0.39, 0.3, 0.24, 0.39, 0.29, 0.66, 0.31, 0.29, 0.22, 
        0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.22, 0.24, 0.32, 
        0.17, 0.24, 0.25, 0.27, 0.26, 0.17, 0.3, 0.34, 0.27, 
        0.34, 0.26, 0.29, 0.26, 0.42, 0.45, 0.29, 0.26, 0.21, 
        0.32, 0.28, 0.2, 0.34, 0.28, 0.29, 0.32, 0.28, 0.3, 0.27, 
        0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 
        0.26, 0.28, 0.29, 0.28, 0.43, 0.53, 0.32, 0.34, 0.45, 
        0.37, 0.26, 0.27, 0.4, 0.28, 0.4, 0.32, 0.24, 0.21, 0.13, 
        0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.17, 0.4, 0.61, 0.22, 
        0.3, 0.29, 0.4, 0.32, 0.26, 0.22, 0.5, 0.32, 0.25, 0.17, 
        0.32, 0.17, 0.22, 0.24, 0.32, 0.32, 0.29, 0.34, 0.37, 
        0.4, 0.66, 0.26, 0.48, 0.39, 0.29, 0.24, 0.26, 0.35, 
        0.42, 0.3, 0.52, 0.3, 0.39, 0.22, 0.3, 0.32, 0.24, 0.43, 
        0.3, 0.26, 0.43, 0.39, 0.22, 0.4, 0.35, 0.63, 0.4, 0.27, 
        0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.53, 
        0.39, 0.53, 0.29, 0.43, 0.4, 0.47, 0.45, 0.53, 0.22, 
        0.61, 0.26, 0.61, 0.53, 0.5, 0.53, 0.63, 0.53, 0.52, 
        0.53, 0.5, 0.6, 0.4, 0.41, 0.53, 0.43, 0.4, 0.39, 0.63, 
        0.5, 0.48, 0.53, 0.53, 0.43, 0.43, 0.53, 0.4), pro = c(2.29, 
        1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
        2.38, 1.57, 1.81, 1.82, 2.96, 1.46, 2.38, 1.92, 1.86, 
        1.66, 2.1, 2.29, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
        1.76, 1.98, 2.38, 1.95, 1.95, 1.63, 1.54, 1.86, 1.36, 
        1.44, 1.37, 1.71, 1.43, 1.14, 2.49, 1.98, 2.03, 1.31, 
        2.19, 2.14, 2.38, 2.08, 1.28, 2.29, 1.87, 1.68, 1.62, 
        2.45, 2.38, 1.66, 1.97, 0.42, 1.4, 1.25, 0.73, 1.87, 
        1.03, 1.98, 1.43, 1.04, 0.68, 2.5, 1.46, 1.87, 1.03, 
        1.45, 1.04, 1.15, 1.46, 1.42, 2.76, 1.95, 1.43, 2.08, 
        1.4, 1.62, 1.44, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
        1.63, 1.38, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
        1.4, 1.35, 1.31, 1.42, 1.4, 1.42, 1.04, 1.63, 2.08, 2.49, 
        1.24, 1.04, 1.98, 1.63, 1.04, 1.63, 1.38, 1.61, 0.83, 
        1.87, 1.83, 2.14, 1.71, 1.56, 2.91, 1.35, 1.77, 1.42, 
        1.9, 1.35, 0.94, 1.35, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 
        0.88, 1.56, 0.75, 1.63, 0.55, 1.55, 1.14, 1.3, 0.68, 
        0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.42, 1.46, 1.55, 
        1.56, 1.14, 2.7, 2.29, 1.04, 1.25, 0.8, 0.94, 1.25, 1.15, 
        1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 
        1.35, 1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.2, 
        6.75, 5.45, 5.05, 5.7, 7.22, 5.75, 5.24, 5.6, 5.6, 7.5, 
        3.3, 9.2, 4.7, 8.7, 5.1, 5.65, 4.5, 5.75, 3.93, 3.52, 
        3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 3.3, 4.28, 5.4, 4.2, 
        5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 
        5.24, 4.9, 6.1, 10.68, 8.9, 7.2, 4.6, 7.22, 6.3, 5.85, 
        6.25, 9.2, 4.9, 6.8, 3.95, 3.27, 5.75, 3.8, 4.45, 2.95, 
        4.6, 5.3, 4.68, 3.17, 2.85, 4.25, 3.38, 3.74, 2.62, 3.21, 
        3.8, 4.6, 3.52, 2.5, 3.93, 2.5, 3.9, 3.4, 4.8, 3.05, 
        2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 
        2.6, 5.75, 2.9, 2.7, 2.7, 4.68, 3.05, 2.8, 4.45, 2.94, 
        2.7, 3.4, 3.3, 2.7, 2.94, 2.9, 3.8, 2.95, 3.08, 2.9, 
        1.9, 2.6, 3.38, 3.4, 1.28, 3.25, 4.8, 2.8, 8.42, 2.8, 
        4.45, 3.94, 3.4, 2.12, 5.3, 9.3, 8.42, 7.1, 5, 5.45, 
        7.1, 6.25, 5, 5.7, 4.92, 4.35, 10.26, 4.35, 4.4, 8.21, 
        4, 4.9, 8.7, 8.42, 9.4, 2.9, 7.1, 7.1, 10.52, 7.6, 7.9, 
        9.7, 7.5, 13, 6.75, 7.65, 7.1, 5.58, 5.7, 10.26, 6.62, 
        10.68, 10.26, 6.3, 8.5, 3.3, 4.9, 9.7, 7.7, 7.3, 10.26, 
        9.3, 9.2), hue = c(1.04, 1.05, 1.25, 0.86, 0.84, 1.05, 
        1.02, 0.6, 1.08, 1.02, 1.25, 1.12, 1.15, 1.25, 1.2, 1.28, 
        0.98, 1.13, 1.23, 0.88, 1.09, 1.27, 1.11, 1.09, 1.12, 
        1.13, 1.02, 1.02, 1.27, 1.04, 1.19, 1.09, 1.23, 1.25, 
        1.1, 1.12, 1.27, 1.02, 1.18, 0.89, 0.95, 0.91, 0.88, 
        0.82, 0.86, 0.87, 1.13, 1.22, 1.07, 1.12, 1.12, 1.22, 
        1.22, 1.25, 1.05, 1.07, 0.94, 1.07, 0.89, 1.04, 1.25, 
        0.98, 1.23, 1.22, 0.67, 1.07, 1.28, 1.12, 1.02, 1.28, 
        0.906, 1.36, 0.98, 1.31, 1.23, 1.23, 1.19, 0.96, 1.06, 
        1.19, 1.38, 0.95, 1.31, 0.84, 1.12, 1.23, 1.22, 1.02, 
        1.25, 1.07, 1.08, 1.05, 1, 1.15, 1.28, 0.92, 0.59, 0.86, 
        1.05, 1.42, 1.27, 0.78, 0.8, 1.12, 1.04, 0.7, 1, 1.12, 
        0.86, 1.28, 0.78, 0.9, 0.96, 1.1, 0.93, 1.71, 0.95, 1.1, 
        0.7, 0.95, 0.8, 1.28, 0.92, 0.73, 1.09, 0.86, 0.69, 0.97, 
        0.89, 0.79, 0.78, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 
        0.82, 0.81, 0.86, 1.31, 0.96, 0.89, 0.81, 0.65, 0.6, 
        0.57, 0.54, 0.55, 0.59, 0.73, 0.48, 0.61, 0.56, 0.73, 
        0.6, 0.57, 0.67, 0.57, 0.7, 0.56, 0.96, 0.87, 0.68, 0.7, 
        0.78, 0.85, 0.61, 0.74, 0.67, 0.7, 0.7, 0.48, 0.64, 0.7, 
        0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 3.4, 2.93, 
        3.71, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3, 3, 
        3.71, 2.65, 2.84, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.35, 
        3.2, 2.9, 2.77, 3.38, 3.63, 2.71, 2.88, 2.87, 3, 2.87, 
        3.35, 3.35, 3.13, 2.69, 3.26, 3.38, 3, 3.4, 3, 3.35, 
        3.33, 2.88, 3.33, 2.75, 3.1, 3.48, 3.64, 3.26, 2.93, 
        3.2, 3.03, 2.05, 2.84, 2.87, 1.82, 3.02, 1.59, 2.46, 
        2.87, 2.75, 2.3, 3.18, 3.48, 2.15, 3.12, 1.82, 3.14, 
        2.75, 3.5, 3.13, 2.14, 2.48, 2.52, 1.82, 2.96, 4, 3.14, 
        2.72, 2.01, 2.88, 3.33, 2.3, 3.21, 2.75, 3.21, 2.14, 
        2.65, 2.14, 3.3, 3.64, 2.63, 1.36, 2.74, 3.3, 2.83, 2.96, 
        2.77, 3.38, 3.03, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 
        2.81, 2.78, 2.5, 2.5, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 
        3.39, 3.69, 3.12, 1.68, 3.64, 2.9, 2.84, 2.96, 2.78, 
        2.46, 1.75, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
        1.82, 2.15, 2.05, 1.64, 1.69, 2.05, 2, 1.36, 1.33, 1.51, 
        1.62, 1.33, 2.78, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
        1.73, 1.96, 1.78, 1.58, 1.62, 1.68, 1.75, 1.68, 1.75, 
        1.68, 1.51, 1.27, 1.58, 1.6, 1.75, 1.68, 1.56, 1.58, 
        1.56, 1.62, 1.6), prol = c(1065, 1050, 1290, 1120, 735, 
        1450, 1290, 1295, 1045, 1045, 1320, 1280, 1320, 1680, 
        1547, 1310, 1280, 1060, 1680, 845, 1095, 770, 1035, 1015, 
        845, 425, 1195, 1285, 915, 1035, 1285, 770, 1015, 1235, 
        1310, 920, 880, 580, 718, 760, 795, 1035, 1095, 680, 
        885, 1080, 1065, 985, 1060, 1120, 1150, 1265, 1190, 1290, 
        1060, 1120, 970, 1270, 1285, 466, 985, 845, 630, 625, 
        1045, 1035, 502, 392, 510, 718, 870, 410, 472, 560, 886, 
        685, 392, 680, 480, 463, 278, 714, 830, 515, 985, 450, 
        372, 562, 680, 625, 480, 450, 495, 1080, 345, 434, 625, 
        428, 660, 406, 672, 562, 438, 415, 672, 510, 510, 466, 
        312, 680, 562, 325, 915, 434, 385, 718, 495, 675, 372, 
        564, 510, 1265, 365, 380, 415, 378, 378, 466, 502, 580, 
        1050, 415, 560, 600, 625, 695, 720, 515, 580, 590, 600, 
        740, 520, 550, 480, 830, 415, 740, 470, 550, 500, 480, 
        425, 675, 640, 680, 480, 880, 660, 312, 392, 680, 570, 
        886, 615, 365, 695, 685, 625, 885, 510, 470, 1035, 740, 
        870, 835, 840, 365)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.75, 13.24, 
        13.5, 14.39, 14.06, 14.38, 13.86, 12.85, 13.73, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 14.75, 13.64, 13.75, 
        12.93, 13.71, 12.85, 13.5, 13.39, 13.39, 13.3, 13.87, 
        12.79, 13.73, 13.58, 13.68, 13.76, 12.85, 13.68, 13.69, 
        13.05, 12.29, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 14.39, 13.9, 13.9, 12.85, 13.05, 13.48, 13.82, 
        13.83, 13.74, 14.21, 14.22, 13.29, 14.75, 12.37, 12.33, 
        12.64, 12, 12.37, 12.17, 12.37, 12.7, 11.03, 13.34, 12.21, 
        12.72, 13.86, 12.22, 12.99, 11.96, 11.66, 11.96, 11.84, 
        12.64, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.08, 12.16, 
        12.64, 12.85, 12.08, 12.08, 12, 12.69, 11.61, 11.62, 
        12.37, 11.81, 12.29, 13.86, 12.29, 12.08, 12.6, 12.29, 
        11.87, 12.51, 11.82, 12.22, 12.72, 12.22, 11.61, 11.84, 
        12.08, 11.76, 12.08, 12.08, 11.03, 11.82, 13.75, 12.21, 
        11.84, 11.45, 14.38, 12.42, 13.05, 11.87, 12.07, 13.16, 
        11.79, 13.17, 12.04, 12.86, 12.88, 12.77, 12.7, 12.51, 
        13.05, 12.25, 12.53, 13.24, 12.84, 12.93, 13.36, 13.52, 
        13.62, 12.25, 12.25, 13.62, 12.87, 13.32, 12.93, 12.77, 
        12.79, 12.96, 13.23, 12.99, 13.17, 13.84, 12.45, 14.34, 
        13.48, 13.2, 13.69, 12.85, 12.96, 13.78, 13.73, 13.05, 
        13.41, 13.58, 13.4, 12.37, 12.77, 14.16, 13.3, 13.4, 
        13.27, 13.17, 14.13), malic = c(1.71, 1.61, 1.29, 1.95, 
        4.3, 1.53, 1.87, 2.15, 0.94, 1.35, 2.16, 1.48, 1.5, 1.73, 
        1.87, 1.29, 1.92, 1.71, 1.25, 3.1, 1.63, 3.8, 1.67, 1.6, 
        1.81, 1.97, 1.87, 1.01, 1.9, 1.5, 1.5, 1.71, 1.83, 1.53, 
        1.6, 1.61, 1.53, 1.65, 1.5, 1.5, 1.71, 1.36, 1.5, 3.98, 
        1.73, 4.04, 2.55, 1.68, 2.08, 1.35, 1.17, 1.65, 4.3, 
        1.9, 1.67, 4.04, 1.7, 1.97, 1.24, 0.94, 2.13, 1.36, 1.25, 
        3.87, 4.3, 1.21, 1.01, 1.17, 4.12, 0.94, 1.68, 1.51, 
        1.66, 1.67, 1.09, 1.88, 0.9, 3.12, 0.99, 3.87, 1.61, 
        4.31, 1.13, 3.8, 1.9, 0.98, 1.61, 1.67, 1.67, 1.87, 1.13, 
        1.51, 3.12, 1.48, 0.94, 1.09, 2.81, 1.41, 1.61, 1.53, 
        2.08, 1.13, 1.9, 1.72, 2.55, 2.55, 1.73, 1.92, 1.29, 
        1.48, 3.74, 1.63, 2.68, 0.74, 1.39, 1.65, 2.13, 1.61, 
        3.43, 1.81, 2.4, 2.05, 4.43, 3.74, 4.31, 2.16, 1.53, 
        2.13, 5.04, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 4.43, 
        2.39, 5.51, 3.59, 3.8, 2.81, 2.56, 3.24, 2.81, 3.88, 
        5.51, 5.04, 4.61, 3.24, 1.39, 3.12, 2.67, 1.9, 3.45, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 4.12, 3.43, 
        3.45, 2.76, 3.24, 1.21, 3.37, 2.58, 4.6, 3.98, 2.39, 
        5.51, 5.65, 3.91, 4.31, 2.59, 4.1), ash = c(2.43, 2.14, 
        2.67, 2.5, 2.7, 2.61, 2.45, 2.61, 2.5, 2.59, 2.3, 2.74, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.28, 
        2.65, 2.74, 2.52, 2.61, 3.22, 2.41, 2.5, 2.67, 2.7, 2.7, 
        2.36, 2.44, 2.7, 2.65, 2.67, 2.35, 2.55, 2.1, 2.51, 2.48, 
        2.12, 2.59, 2.29, 2.1, 2.44, 2.3, 2.12, 2.4, 2.27, 2.04, 
        2.74, 2.38, 2.44, 2.7, 2.67, 2.3, 2.68, 2.5, 1.36, 2.59, 
        2.28, 1.92, 2.16, 2.53, 2.56, 2.22, 1.92, 2.54, 1.75, 
        2.21, 2.67, 2.24, 2.6, 2.3, 2.5, 1.71, 2.16, 2.5, 2.4, 
        2, 2.36, 2.51, 2.32, 2.58, 2.24, 2.69, 2.62, 2.19, 2.3, 
        2.32, 2.61, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.54, 
        2.61, 1.7, 2.51, 2.46, 1.88, 2.5, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.4, 1.75, 
        2, 2.42, 3.23, 1.94, 2.13, 2.39, 2.17, 2.29, 2.3, 2.36, 
        2.38, 2.5, 2.4, 2.4, 2.36, 2.25, 2.69, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.24, 2.68, 2.48, 2.17, 
        2.36, 2.62, 2.48, 1.82, 2.28, 2.58, 2.32, 2.38, 2.26, 
        2.43, 2.64, 2.38, 2.54, 2.59, 2.35, 2.3, 2.21, 2.6, 2.16, 
        2.69, 2.37, 2.32, 2.28, 2.51, 2.28, 2.48, 2.26, 2.37, 
        2.74), alca = c(15.6, 16.8, 21, 16.8, 21, 15.2, 15.5, 
        17.6, 14, 16, 16, 16.8, 16, 11.4, 13.2, 16, 15.5, 20, 
        10.6, 15.2, 16, 18.6, 15.6, 17.8, 20, 22, 16.1, 17, 16, 
        21.5, 16, 19.1, 18.8, 16, 19, 20.5, 15.5, 18, 15.5, 13.2, 
        16.2, 18.8, 16, 19.5, 17, 18, 16, 12.4, 18.8, 17.4, 12.4, 
        20.8, 13.2, 16, 17, 20.5, 16.3, 16.8, 13.2, 10.6, 30, 
        16.8, 18, 19, 19, 18.1, 15.2, 19.6, 19, 16.8, 20.4, 24, 
        24, 30, 21, 23.5, 16, 18, 14.8, 24, 19, 22.5, 24, 22.5, 
        18, 18, 22.8, 20, 21.6, 21.6, 21.5, 22, 21, 18, 18, 19, 
        21.5, 16, 16.8, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 
        22.5, 19, 20, 19.5, 16.8, 24, 25, 20.5, 21.5, 20.8, 19.6, 
        16, 16, 19.5, 20.8, 18, 21.5, 21, 21, 21.5, 28.5, 21, 
        22, 15.6, 20, 24, 21.5, 21, 21, 22.8, 25, 19.5, 22.8, 
        21, 24.5, 23.5, 20, 22, 21, 20, 22, 19.5, 21.5, 24, 22, 
        16, 18.5, 20, 22, 19.5, 18, 25, 25, 18, 19, 22.8, 18.5, 
        23.5, 22.5, 23, 19.5, 24.5, 25, 22, 19.5, 22.5, 20.5, 
        23, 20, 21, 24.5), mg = c(127, 100, 101, 113, 96, 112, 
        96, 121, 97, 89, 120, 102, 89, 91, 102, 112, 120, 108, 
        88, 116, 126, 102, 112, 95, 118, 124, 93, 110, 107, 96, 
        101, 106, 104, 88, 110, 111, 110, 80, 85, 106, 117, 116, 
        101, 99, 107, 111, 102, 101, 101, 108, 92, 111, 111, 
        115, 124, 124, 118, 102, 108, 88, 101, 100, 94, 87, 104, 
        98, 78, 78, 101, 151, 85, 86, 87, 139, 101, 113, 88, 
        112, 91, 121, 86, 102, 78, 85, 87, 99, 90, 88, 84, 70, 
        81, 112, 86, 88, 106, 106, 134, 85, 88, 88, 97, 88, 88, 
        86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 
        86, 102, 90, 87, 96, 151, 102, 86, 113, 85, 102, 92, 
        88, 80, 122, 101, 98, 106, 101, 94, 89, 92, 88, 101, 
        88, 89, 97, 88, 112, 102, 80, 110, 92, 113, 123, 102, 
        116, 98, 103, 106, 89, 97, 98, 89, 94, 113, 106, 106, 
        90, 88, 91, 88, 105, 112, 96, 86, 91, 95, 102, 89, 120, 
        89), phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.15, 2.5, 2.6, 
        3, 2.98, 2.95, 2.2, 2.6, 3.1, 3.88, 2.85, 2.8, 2.64, 
        3.3, 3.25, 3, 2.41, 2.61, 2.48, 2.53, 1.38, 2.8, 2.4, 
        2.95, 2.65, 3, 2.8, 2.42, 2.95, 2.53, 2.7, 2.6, 2.45, 
        2.86, 2.75, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
        2.75, 2.88, 2.72, 2.45, 3.88, 3.1, 2.64, 2.96, 3.2, 3, 
        3.4, 1.98, 2.05, 2.02, 2.65, 3.5, 1.89, 2.41, 2.46, 2.11, 
        2.53, 2.5, 1.1, 2.95, 1.88, 2.2, 2.22, 1.98, 2.9, 1.95, 
        1.9, 2.83, 2.45, 2.2, 1.48, 1.65, 2.56, 2.2, 1.7, 1.92, 
        1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.11, 1.39, 2.55, 
        3.52, 2.8, 2.42, 1.62, 2.56, 2.5, 2.2, 2.45, 1.65, 2.02, 
        2.36, 2.74, 3.18, 2.55, 1.75, 1.41, 2.56, 2.46, 1.83, 
        2, 2.05, 2, 2.9, 3.18, 2.55, 1.68, 2.86, 2.7, 2.74, 2.13, 
        2.22, 2.1, 1.51, 1.55, 1.88, 1.7, 1.4, 1.62, 1.38, 1.92, 
        1.62, 1.38, 1.4, 1.4, 1.25, 2, 1.38, 1.35, 0.98, 1.7, 
        1.93, 1.41, 1.4, 1.48, 2.2, 2.3, 1.48, 1.55, 1.8, 1.9, 
        1.93, 0.98, 2.3, 1.83, 1.48, 1.39, 1.35, 1.39, 1.7, 1.48, 
        1.55, 1.88, 1.25, 1.39, 1.68, 1.55, 2.56, 1.45, 1.65, 
        2.05), fla = c(2.61, 2.76, 2.76, 3.49, 2.53, 3.39, 2.52, 
        2.51, 2.98, 3.17, 3.32, 2.43, 2.76, 2.86, 3.64, 2.91, 
        3.14, 3.4, 2.91, 3.25, 3.04, 2.41, 2.88, 2.37, 2.61, 
        2.68, 2.94, 2.69, 2.97, 2.33, 3.25, 3.19, 2.69, 3.32, 
        2.37, 2.98, 2.76, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 
        2.63, 3, 2.65, 3.17, 2.78, 2.92, 3.54, 2.79, 2.99, 3.74, 
        2.79, 2.9, 2.78, 2.92, 3.23, 3.67, 0.57, 2, 1.41, 1.79, 
        2.99, 1.75, 2.65, 2.29, 2, 1.3, 2.98, 1.02, 2.86, 1.69, 
        2.89, 2.14, 1.57, 2.69, 1.75, 1.85, 2.17, 2.26, 2.53, 
        1.58, 1.59, 2.21, 2.29, 1.69, 1.61, 1.69, 2.11, 2.01, 
        1.25, 1.46, 2.68, 2.65, 2.27, 0.99, 2.51, 3.49, 2.99, 
        2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 2.09, 
        2.92, 2.76, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 
        1.2, 1.64, 2.86, 3.25, 1.92, 1.41, 3.03, 2.65, 3.15, 
        2.24, 1.75, 1.75, 1.84, 1.36, 1.57, 1.2, 0.58, 0.66, 
        0.47, 0.92, 0.66, 0.6, 0.5, 0.49, 0.52, 0.8, 0.63, 0.55, 
        0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 0.58, 
        0.63, 1.2, 0.58, 1.31, 1.1, 0.92, 0.68, 0.6, 0.47, 0.68, 
        0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.58, 0.7, 0.61, 
        1.3, 0.69, 0.68, 0.76), nfla = c(0.28, 0.26, 0.3, 0.24, 
        0.39, 0.34, 0.39, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 
        0.43, 0.29, 0.3, 0.24, 0.31, 0.32, 0.17, 0.24, 0.25, 
        0.27, 0.26, 0.39, 0.27, 0.34, 0.27, 0.42, 0.26, 0.29, 
        0.28, 0.42, 0.32, 0.29, 0.26, 0.17, 0.24, 0.28, 0.2, 
        0.34, 0.26, 0.32, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
        0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.24, 
        0.32, 0.28, 0.43, 0.53, 0.32, 0.29, 0.45, 0.37, 0.26, 
        0.27, 0.53, 0.32, 0.32, 0.27, 0.26, 0.21, 0.13, 0.34, 
        0.24, 0.43, 0.35, 0.43, 0.3, 0.39, 0.4, 0.61, 0.22, 0.3, 
        0.42, 0.27, 0.43, 0.29, 0.28, 0.5, 0.32, 0.25, 0.17, 
        0.32, 0.17, 0.25, 0.24, 0.29, 0.24, 0.29, 0.34, 0.37, 
        0.43, 0.66, 0.24, 0.48, 0.39, 0.29, 0.24, 0.26, 0.39, 
        0.42, 0.29, 0.52, 0.3, 0.32, 0.53, 0.22, 0.32, 0.24, 
        0.43, 0.3, 0.32, 0.22, 0.39, 0.39, 0.4, 0.27, 0.17, 0.29, 
        0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 
        0.5, 0.4, 0.63, 0.48, 0.43, 0.4, 0.47, 0.45, 0.37, 0.22, 
        0.32, 0.26, 0.61, 0.53, 0.6, 0.53, 0.63, 0.53, 0.52, 
        0.48, 0.5, 0.6, 0.4, 0.41, 0.53, 0.43, 0.4, 0.39, 0.53, 
        0.4, 0.48, 0.52, 0.43, 0.43, 0.43, 0.53, 0.53), pro = c(2.29, 
        1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
        2.38, 1.57, 1.81, 1.95, 2.96, 1.46, 1.97, 2.96, 1.86, 
        1.66, 2.1, 2.08, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
        1.76, 1.98, 2.38, 1.95, 1.45, 1.98, 1.54, 1.86, 1.36, 
        1.44, 1.37, 1.45, 3.28, 2.38, 2.81, 1.97, 2.03, 1.63, 
        2.19, 2.14, 2.38, 2.08, 1.38, 2.29, 1.87, 1.68, 1.62, 
        2.45, 2.76, 1.66, 2.96, 0.42, 1.85, 1.43, 0.73, 1.87, 
        1.03, 2.18, 1.98, 1.04, 1.14, 2.5, 1.46, 1.87, 1.03, 
        2.91, 2.19, 1.15, 1.46, 2.29, 2.76, 1.95, 1.43, 2.91, 
        1.04, 1.62, 1.35, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
        1.63, 1.63, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
        1.4, 1.35, 1.31, 1.42, 1.35, 1.42, 2.19, 1.63, 2.08, 
        2.49, 1.82, 1.77, 1.99, 1.87, 1.04, 1.95, 1.04, 1.61, 
        0.83, 1.87, 1.83, 2.96, 1.71, 1.03, 2.91, 1.35, 1.77, 
        3.28, 1.9, 1.04, 0.94, 1.28, 0.83, 0.84, 1.25, 0.94, 
        0.8, 1.1, 0.88, 1.3, 0.75, 1.3, 0.55, 1.54, 1.14, 1.3, 
        0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 0.86, 0.88, 
        1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.06, 1.25, 0.94, 
        0.73, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 
        1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 4.38, 5.68, 
        7.8, 3.38, 6.75, 5.7, 5.05, 3.95, 7.22, 5.75, 8.21, 5.6, 
        7.22, 7.5, 5.43, 8.7, 7.3, 8.7, 5.1, 5.65, 4.5, 4.45, 
        3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 4.35, 4.8, 
        5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 
        4.36, 5.04, 5.24, 4.9, 6.1, 7.2, 8.9, 7.2, 4.6, 5.04, 
        6.3, 5.85, 6.25, 6.62, 5.68, 6.8, 8.7, 3.27, 5.75, 3.8, 
        4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 5.05, 3.38, 3.74, 
        4.35, 3.21, 3.8, 4.6, 2.94, 3.25, 3.08, 2.5, 3.9, 3.05, 
        4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 
        2.15, 3.25, 2.6, 7.2, 2.9, 5, 2.5, 2.62, 5.3, 2.8, 5.1, 
        2.94, 2.7, 3.4, 3.3, 2.7, 2.9, 2.9, 3.8, 2.62, 3.08, 
        2.9, 1.9, 2.62, 5.6, 3.4, 1.28, 3.25, 10.26, 2.7, 5.6, 
        2.8, 4.45, 3.94, 2.9, 2.12, 3.4, 4.6, 5.6, 8.7, 5, 5.45, 
        7.1, 8.5, 5, 5.7, 4.92, 7.2, 6.8, 4.35, 4.4, 8.21, 4, 
        4.9, 5, 8.42, 9.4, 7.3, 7.5, 7.1, 10.52, 7.6, 7.9, 5.45, 
        7.5, 13, 6.62, 7.65, 5.6, 5.58, 5.43, 10.68, 6.62, 10.68, 
        10.26, 7.5, 8.5, 6.1, 5, 9.7, 7.7, 7.3, 7.9, 9.3, 9.2
        ), hue = c(1.04, 1.05, 1.19, 0.86, 1.12, 1.05, 1.02, 
        1.18, 1.08, 0.98, 1.25, 0.7, 1.15, 1.25, 1.2, 1.28, 0.78, 
        1.13, 1.23, 0.92, 1.09, 0.66, 1.11, 1.09, 1.12, 1.13, 
        1.19, 1.02, 0.8, 1.04, 1.19, 1.09, 1.19, 1.25, 1.1, 1.19, 
        0.86, 1.09, 1.18, 0.95, 0.95, 0.91, 0.88, 0.82, 0.86, 
        0.87, 1.25, 0.95, 1.07, 1.12, 1.12, 1.09, 0.95, 0.69, 
        1.04, 0.67, 0.94, 1.07, 0.89, 0.73, 1.25, 0.98, 1.23, 
        1.22, 0.61, 1.38, 0.95, 1.12, 1.02, 1.28, 0.906, 1.36, 
        0.98, 1.31, 1.42, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 
        0.89, 1.31, 0.84, 1.1, 1.23, 1.1, 1.08, 1.04, 1.07, 1.08, 
        1.05, 0.67, 1.15, 1.05, 1.09, 0.57, 1.04, 1.2, 1.42, 
        1.27, 0.75, 0.8, 1.13, 1.04, 1.2, 1, 1.07, 0.86, 1.05, 
        1.02, 0.9, 1.12, 1.1, 0.93, 1.71, 0.95, 1.1, 0.7, 1.25, 
        0.8, 0.7, 0.92, 0.73, 0.87, 0.86, 0.69, 0.97, 0.89, 0.79, 
        0.98, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 
        0.6, 0.96, 0.79, 0.89, 1.23, 0.65, 0.6, 0.64, 0.54, 0.55, 
        0.65, 0.64, 0.48, 0.61, 0.56, 0.79, 0.6, 0.57, 0.67, 
        0.57, 0.66, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 
        0.7, 0.74, 0.67, 0.68, 0.87, 0.64, 0.64, 0.7, 0.59, 0.6, 
        0.61), ratio = c(3.92, 3.4, 3.17, 2.9, 2.93, 3.21, 3.58, 
        3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.33, 3, 2.87, 2.65, 
        4, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 2.63, 3.2, 3.3, 2.77, 
        3.2, 2.75, 2.71, 2.88, 2.87, 3, 2.87, 2.77, 3.39, 2.46, 
        2.69, 3.58, 3.38, 3, 3.26, 3, 3.35, 3.33, 4, 3.33, 2.75, 
        3.1, 3.69, 3.26, 3.26, 2.93, 3.2, 3.03, 3.69, 2.84, 2.87, 
        1.82, 3.05, 1.59, 2.46, 2.87, 2.15, 2.3, 3.18, 3.48, 
        1.69, 2.74, 1.82, 2.75, 3.57, 3.5, 3.13, 2.14, 2.48, 
        2.52, 2.42, 2.72, 3.57, 3.14, 2.72, 2.01, 3.33, 2.14, 
        3.18, 3.21, 2.75, 3.21, 3.58, 2.65, 3.33, 3.3, 2.63, 
        2.63, 1.82, 2.74, 3, 2.83, 2.96, 2.77, 3.38, 2.42, 3.57, 
        3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.57, 3.19, 
        2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.12, 
        3.64, 2.82, 2.84, 3.03, 2.78, 3, 3.05, 1.42, 1.36, 1.29, 
        1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.82, 1.64, 2.05, 
        2.05, 2, 1.33, 1.33, 1.75, 1.62, 1.33, 1.64, 1.47, 1.33, 
        1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 1.33, 
        1.75, 1.75, 1.68, 1.75, 1.36, 1.48, 1.64, 1.82, 1.47, 
        1.75, 1.96, 1.55, 1.51, 1.56, 1.62, 1.6), prol = c(1065, 
        1050, 685, 760, 735, 1450, 1290, 1295, 1045, 1045, 1235, 
        1280, 1320, 1547, 1547, 1310, 1280, 1280, 1680, 845, 
        920, 770, 1035, 1015, 845, 1195, 1195, 1285, 915, 1035, 
        1285, 1050, 1015, 1235, 1045, 920, 880, 870, 830, 760, 
        795, 1035, 1095, 680, 885, 1080, 1065, 985, 1060, 1285, 
        1150, 1265, 1190, 880, 1060, 1120, 970, 1270, 1285, 415, 
        590, 510, 630, 770, 562, 685, 502, 714, 392, 718, 870, 
        410, 472, 1080, 886, 520, 392, 463, 680, 463, 278, 714, 
        714, 515, 550, 425, 410, 562, 680, 625, 480, 450, 495, 
        325, 345, 434, 625, 428, 660, 406, 466, 562, 438, 415, 
        672, 672, 510, 428, 312, 680, 562, 325, 570, 434, 385, 
        406, 345, 760, 372, 564, 415, 1285, 365, 380, 1035, 378, 
        415, 466, 345, 580, 760, 392, 560, 600, 438, 695, 720, 
        515, 580, 590, 600, 564, 520, 550, 675, 830, 415, 562, 
        1035, 550, 500, 480, 425, 675, 640, 835, 480, 880, 660, 
        325, 570, 680, 570, 428, 615, 480, 695, 685, 510, 714, 
        510, 470, 685, 740, 680, 835, 840, 415)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 13.16, 13.24, 13.84, 14.39, 14.06, 13.32, 
        13.86, 13.63, 12.45, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.21, 13.64, 13.62, 12.93, 13.71, 12.85, 13.5, 
        14.22, 13.39, 13.3, 13.87, 14.22, 13.73, 13.58, 13.68, 
        13.76, 13.5, 13.58, 13.48, 13.05, 12.69, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.71, 13.9, 13.68, 
        13.32, 13.05, 14.3, 13.82, 13.68, 13.74, 14.21, 14.22, 
        13.29, 13.58, 12.37, 12.33, 12.64, 12.29, 12.37, 12.17, 
        12.37, 12.08, 11.81, 13.34, 12.21, 12.85, 13.86, 11.76, 
        12.99, 11.96, 11.66, 11.79, 11.84, 12.77, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 13.4, 12.16, 12.7, 11.61, 12.08, 
        12.08, 12, 12.69, 12.08, 11.62, 12.84, 11.81, 12.29, 
        11.81, 12.29, 12.08, 12.6, 12.51, 12.37, 12.51, 12.69, 
        12.29, 12.72, 12.22, 11.61, 11.82, 11.79, 11.76, 12, 
        12.08, 11.03, 11.82, 11.61, 11.87, 12.51, 11.45, 13.83, 
        12.42, 13.05, 11.87, 12.07, 12.08, 11.79, 12.64, 12.04, 
        12.86, 12.88, 12.6, 12.7, 12.51, 13.05, 12.25, 12.53, 
        13.4, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.79, 
        12.42, 12.87, 13.32, 13.34, 11.03, 12.79, 13.52, 13.23, 
        13.34, 13.17, 13.84, 12.45, 14.34, 13.48, 13.2, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.82, 13.5, 13.58, 13.4, 
        12.79, 12.77, 14.16, 13.4, 13.4, 13.27, 13.17, 14.13), 
            malic = c(1.71, 1.67, 1.97, 1.95, 4.1, 1.39, 1.87, 
            2.15, 1.41, 1.35, 2.16, 1.48, 1.68, 1.73, 1.87, 1.65, 
            1.92, 1.83, 1.53, 3.1, 1.63, 3.8, 1.73, 1.6, 1.81, 
            1.39, 1.35, 1.71, 1.9, 1.66, 1.5, 1.5, 1.83, 1.53, 
            1.67, 1.35, 1.7, 1.65, 1.5, 1.97, 1.71, 1.81, 3.59, 
            3.98, 2.08, 4.04, 1.25, 1.68, 1.97, 1.41, 0.9, 1.65, 
            1.67, 1.9, 1.67, 2.76, 1.7, 1.97, 3.87, 0.94, 1.25, 
            1.36, 1.25, 1.5, 2.15, 1.21, 1.01, 1.17, 1.88, 1.65, 
            1.13, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 1.71, 0.99, 
            3.87, 1.17, 1.81, 1.13, 1.53, 1.73, 0.98, 1.61, 1.67, 
            1.13, 1.68, 1.09, 1.51, 2.08, 0.99, 1.35, 1.21, 3.55, 
            1.41, 4.6, 0.9, 2.08, 0.74, 2.59, 1.72, 2.55, 2.55, 
            1.73, 1.25, 1.29, 1.39, 3.74, 1.09, 2.68, 0.74, 1.39, 
            1.65, 3.74, 1.61, 3.43, 0.94, 2.4, 2.05, 4.43, 1.67, 
            4.31, 2.16, 1.53, 2.13, 2.67, 4.3, 1.35, 2.99, 2.31, 
            3.55, 1.24, 1.9, 3.8, 5.51, 3.59, 4.04, 2.81, 2.56, 
            2.15, 2.4, 3.88, 2.58, 5.04, 4.61, 3.24, 1.67, 3.12, 
            2.67, 1.9, 2.76, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 
            3.83, 3.87, 1.67, 3.45, 2.76, 3.37, 1.21, 3.37, 2.58, 
            4.6, 2.55, 2.39, 4.1, 5.65, 3.91, 4.61, 2.59, 4.1
            ), ash = c(2.43, 2.14, 2.67, 2.5, 2.74, 2.59, 2.45, 
            2.61, 2.17, 2.3, 2.3, 2.38, 2.41, 2.39, 2.38, 2.7, 
            2.72, 2.62, 2.48, 2.56, 2.61, 2.65, 2.36, 2.52, 2.61, 
            3.22, 2, 2.26, 2.19, 2.35, 2.65, 2.36, 2.39, 2.7, 
            2.65, 2.14, 2.58, 2.55, 2.1, 2.51, 2.35, 2.12, 2.59, 
            2.29, 2.1, 2.44, 1.94, 2.12, 2.4, 2.27, 2.04, 2.3, 
            2.45, 2.43, 2.46, 2.61, 2.3, 2.68, 2.5, 1.36, 2.32, 
            2.36, 1.92, 2.16, 2.53, 2.56, 2.2, 1.92, 1.98, 1.75, 
            2.21, 2.67, 2.24, 2.6, 2.3, 2.2, 1.71, 2.56, 2.3, 
            2.4, 2, 2.35, 2.51, 2.32, 2.58, 2.24, 2.28, 2.62, 
            1.82, 2.3, 2.32, 2.1, 2.26, 2.22, 2.28, 2.2, 2.74, 
            1.98, 2.39, 2.04, 1.7, 1.7, 2.46, 1.88, 2.68, 2.27, 
            2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 
            2.2, 1.99, 2.16, 1.98, 2, 2.42, 3.23, 2.64, 2.13, 
            2.39, 2.17, 2.29, 2.37, 2.67, 2.38, 1.98, 2.4, 2.4, 
            2.36, 2.25, 2.39, 2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 
            2.72, 2.35, 2.2, 2.37, 2.24, 2.48, 2.17, 2.36, 2.62, 
            2.48, 2.12, 2.28, 2.17, 2.32, 2.38, 2.56, 2.36, 2.64, 
            2.38, 2.54, 2.12, 2.35, 2.3, 2.43, 2.6, 2.35, 2.69, 
            2.6, 2.32, 2.28, 2.35, 3.23, 2.48, 2.26, 2.37, 2.74
            ), alca = c(15.6, 16.3, 16.3, 16.8, 21, 15.2, 13.2, 
            17.6, 14, 16, 15.5, 16.8, 16, 11.4, 16, 16.8, 15.2, 
            20, 16.1, 15.2, 16, 18.6, 16.2, 17.8, 20, 21.5, 16.1, 
            17, 17.4, 16, 10.6, 19.1, 20.5, 18, 19, 20.5, 15.5, 
            20.5, 15.5, 13.2, 16.2, 18.8, 17.8, 25, 17, 17.6, 
            16, 18.8, 18.8, 17.4, 12.4, 10.6, 16.8, 16.8, 11.4, 
            20.5, 16.3, 16.8, 16.2, 10.6, 16.8, 16.8, 18, 19, 
            19, 18.1, 20, 19.6, 20.8, 16.8, 20.4, 20, 24, 30, 
            21, 17.6, 16, 18, 14.8, 20, 19, 20.5, 24, 22.5, 18, 
            18, 22.8, 18.1, 21.6, 19.6, 18, 22, 23, 18, 18, 19, 
            21.5, 16, 23, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
            19, 22.5, 19, 20, 19.5, 21, 21, 19, 21.5, 21.5, 20.8, 
            20, 16, 19.5, 21, 15.6, 25, 21.5, 21, 21, 21.5, 28.5, 
            19.5, 22, 19, 20, 24, 21.5, 16, 24, 19, 25, 19.5, 
            25, 21, 24, 23.5, 20, 23.5, 21, 20, 22.5, 22, 21.5, 
            24, 22, 21, 18.5, 20, 22, 19.5, 23, 25, 14.8, 18.6, 
            18.1, 18.5, 18.5, 19.5, 22.5, 23, 19.5, 24.5, 25, 
            21, 19.5, 24.5, 20.5, 23, 20, 20.5, 24.5), mg = c(127, 
            100, 101, 113, 85, 112, 96, 121, 97, 98, 113, 110, 
            89, 91, 102, 112, 120, 98, 121, 116, 126, 102, 112, 
            95, 106, 124, 93, 112, 107, 96, 101, 106, 104, 126, 
            110, 102, 110, 118, 101, 98, 117, 102, 101, 81, 107, 
            111, 102, 101, 107, 108, 92, 116, 111, 115, 134, 
            124, 118, 102, 108, 88, 101, 100, 94, 87, 104, 98, 
            78, 78, 84, 151, 80, 86, 87, 139, 101, 86, 78, 112, 
            88, 102, 86, 151, 78, 85, 78, 99, 90, 88, 84, 70, 
            81, 89, 96, 88, 92, 86, 134, 85, 88, 87, 97, 88, 
            97, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 
            84, 85, 86, 101, 100, 87, 96, 112, 102, 86, 92, 85, 
            84, 92, 88, 80, 122, 96, 98, 106, 87, 94, 89, 106, 
            88, 101, 88, 89, 97, 90, 112, 102, 110, 89, 92, 113, 
            123, 111, 116, 98, 103, 89, 89, 97, 98, 89, 86, 89, 
            106, 106, 90, 88, 88, 88, 105, 112, 96, 86, 91, 95, 
            102, 100, 120, 88), phe = c(2.8, 2.65, 2.8, 3.85, 
            2.8, 3.1, 2.5, 2.6, 2.55, 2.98, 2.95, 2.2, 2.6, 3.1, 
            3.25, 2.85, 2.8, 3.15, 3.3, 3.3, 3, 2.41, 2.61, 2.48, 
            2.53, 2.53, 2.75, 2.4, 2.95, 2.65, 3, 2.74, 2.42, 
            2.95, 2.45, 2.7, 2.6, 2.45, 2.8, 2.72, 3.15, 2.45, 
            3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 
            2.45, 3.88, 3.18, 2.83, 2.96, 3.2, 3, 3.4, 1.98, 
            2.05, 2.02, 2.2, 3.5, 1.89, 2.45, 2.74, 2.11, 2.53, 
            3.85, 1.1, 2.95, 1.88, 2.8, 2.13, 1.38, 2.55, 1.83, 
            1.9, 2.83, 2.56, 2.2, 1.35, 1.65, 2.53, 2.2, 2.2, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.45, 
            1.98, 2.55, 3.52, 2.36, 2.65, 1.89, 2.56, 2.5, 2.2, 
            2.56, 1.65, 2.22, 2.36, 2.74, 3.18, 2.55, 1.75, 1.75, 
            2.56, 2.46, 2.46, 2, 1.7, 2, 2.9, 3.18, 2.05, 2.55, 
            2.86, 2.5, 2.74, 2.13, 2.22, 2.1, 1.51, 1.25, 1.9, 
            1.7, 1.38, 1.62, 1.38, 1.83, 1.62, 1.62, 1.62, 1.4, 
            1.39, 2, 1.38, 1.9, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
            2.2, 2.2, 1.48, 1.38, 1.8, 1.9, 2.56, 1.38, 2.3, 
            1.83, 1.38, 1.39, 1.35, 1.55, 1.7, 1.48, 1.55, 2.22, 
            1.25, 1.39, 1.68, 1.45, 1.41, 1.48, 1.65, 2.05), 
            fla = c(2.78, 2.76, 3.4, 3.49, 2.24, 3.39, 2.52, 
            2.51, 2.98, 3.29, 3.32, 2.43, 2.76, 2.92, 3.64, 2.91, 
            3.14, 3.4, 2.98, 3, 3.67, 2.41, 2.88, 2.37, 2.61, 
            2.68, 2.94, 2.86, 2.97, 2.33, 3.25, 3.19, 2.69, 3, 
            2.99, 2.98, 2.61, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 
            2.63, 3, 2.65, 3.17, 3.32, 2.92, 3.54, 3.25, 2.99, 
            3.74, 2.79, 2.9, 2.78, 3, 3.23, 3.67, 0.57, 1.6, 
            1.41, 1.79, 2.61, 1.75, 2.65, 2.76, 2, 1.3, 3.64, 
            1.02, 2.86, 1.57, 2.89, 2.14, 1.57, 2.27, 1.36, 1.85, 
            2.63, 2.26, 2.53, 1.58, 1.59, 2.21, 2.68, 1.69, 1.61, 
            1.69, 2.53, 1.6, 1.25, 1.46, 2.76, 2.65, 2.27, 0.99, 
            2.65, 3.14, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
            2.03, 1.76, 2.29, 2.92, 2.52, 2.27, 2.03, 2.01, 2.29, 
            2.17, 1.6, 2.09, 1.28, 1.64, 2.65, 3.17, 1.69, 1.85, 
            3.03, 2.65, 3.15, 2.24, 2.11, 1.75, 0.7, 0.63, 1.2, 
            1.2, 0.58, 0.66, 0.47, 1.02, 0.69, 0.6, 0.5, 0.58, 
            0.52, 0.8, 1.02, 0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 
            1.36, 1.28, 0.83, 0.58, 0.63, 1.36, 0.58, 1.31, 1.1, 
            0.92, 1.59, 0.6, 0.61, 0.68, 0.47, 0.92, 0.66, 0.84, 
            0.96, 0.49, 0.47, 0.7, 0.61, 0.47, 0.69, 0.68, 0.76
            ), nfla = c(0.28, 0.22, 0.3, 0.24, 0.39, 0.3, 0.26, 
            0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 
            0.3, 0.3, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.29, 
            0.27, 0.34, 0.27, 0.21, 0.26, 0.29, 0.27, 0.42, 0.3, 
            0.29, 0.26, 0.31, 0.29, 0.28, 0.2, 0.34, 0.32, 0.22, 
            0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 
            0.32, 0.39, 0.21, 0.2, 0.26, 0.24, 0.24, 0.28, 0.39, 
            0.53, 0.32, 0.3, 0.45, 0.37, 0.26, 0.27, 0.61, 0.17, 
            0.45, 0.32, 0.26, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 
            0.43, 0.3, 0.34, 0.4, 0.61, 0.22, 0.3, 0.35, 0.4, 
            0.32, 0.2, 0.43, 0.5, 0.43, 0.25, 0.17, 0.32, 0.4, 
            0.27, 0.24, 0.21, 0.32, 0.29, 0.34, 0.37, 0.6, 0.66, 
            0.22, 0.48, 0.39, 0.29, 0.24, 0.26, 0.66, 0.42, 0.26, 
            0.52, 0.3, 0.29, 0.3, 0.5, 0.32, 0.5, 0.43, 0.3, 
            0.25, 0.27, 0.39, 0.37, 0.4, 0.26, 0.4, 0.52, 0.27, 
            0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.43, 
            0.63, 0.63, 0.26, 0.43, 0.4, 0.47, 0.45, 0.37, 0.22, 
            0.37, 0.26, 0.61, 0.53, 0.58, 0.43, 0.63, 0.53, 0.52, 
            0.58, 0.5, 0.6, 0.4, 0.41, 0.43, 0.43, 0.4, 0.39, 
            0.43, 0.41, 0.48, 0.53, 0.53, 0.43, 0.43, 0.53, 0.53
            ), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 
            1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.77, 2.96, 1.46, 
            2.38, 1.77, 1.86, 1.66, 2.1, 2.38, 1.69, 1.46, 1.66, 
            1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.35, 1.95, 
            1.54, 1.86, 1.36, 1.44, 1.37, 1.9, 1.92, 2.29, 1.86, 
            1.87, 2.03, 1.38, 2.19, 2.14, 2.38, 2.08, 1.92, 2.29, 
            1.87, 1.68, 1.62, 2.45, 1.42, 1.66, 1.87, 0.42, 1.71, 
            1.36, 0.73, 1.87, 1.03, 2.08, 2.81, 1.04, 0.83, 2.5, 
            1.46, 1.87, 1.03, 2.29, 1.42, 1.15, 1.46, 0.75, 2.76, 
            1.95, 1.43, 1.41, 1.46, 1.62, 1.04, 1.46, 1.56, 1.34, 
            1.35, 1.38, 1.64, 1.63, 1.46, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.95, 1.42, 
            1.35, 1.63, 2.08, 2.49, 1.98, 1.57, 1.97, 1.38, 1.04, 
            2.7, 1.03, 1.61, 0.83, 1.87, 1.83, 2.96, 1.71, 1.56, 
            2.91, 1.35, 1.77, 1.38, 1.9, 1.35, 0.94, 1.14, 0.83, 
            0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.25, 0.75, 0.42, 
            0.55, 1.14, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 
            1.26, 1.56, 1.63, 0.42, 1.55, 1.56, 1.14, 2.7, 2.29, 
            1.04, 1.95, 0.42, 0.94, 1.03, 1.15, 1.46, 0.97, 1.54, 
            1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
            ), col = c(5.64, 4.38, 5.68, 7.8, 4.9, 6.75, 4.7, 
            5.05, 5.4, 7.22, 5.75, 4.2, 5.6, 6.13, 7.5, 6.13, 
            10.68, 4.38, 8.7, 5.1, 5.65, 4.5, 4.5, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 9.4, 5.7, 5.4, 4.2, 
            5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 
            5.04, 5.24, 4.9, 6.1, 6.3, 8.9, 7.2, 7.1, 5, 6.3, 
            5.85, 6.25, 4.4, 5.7, 6.8, 4.8, 3.27, 5.75, 3.8, 
            4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 4.28, 3.38, 
            3.74, 3.9, 3.21, 3.8, 4.6, 3.25, 7.6, 4.5, 2.5, 3.9, 
            2.4, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 
            3.6, 3.05, 2.15, 3.25, 2.6, 2.8, 2.9, 8.21, 2.85, 
            5.1, 3.38, 2.8, 2.62, 2.94, 2.7, 3.4, 3.3, 2.7, 2.45, 
            2.9, 5.04, 2.9, 3.08, 2.9, 1.9, 4.68, 3.08, 3.4, 
            1.28, 3.25, 4.8, 1.28, 4.4, 2.8, 3.9, 3.94, 4.5, 
            2.12, 4.92, 7.65, 5.75, 8.7, 5, 5.45, 7.1, 6.1, 5, 
            5.7, 4.92, 2.6, 5.4, 4.35, 4.4, 8.21, 4, 4.9, 4.4, 
            8.42, 9.4, 2.4, 6.62, 7.1, 10.52, 7.6, 7.9, 7.9, 
            7.5, 13, 6.25, 7.65, 6.8, 5.58, 6.1, 9.4, 6.62, 10.68, 
            10.26, 6.25, 8.5, 4.25, 2.6, 9.7, 7.7, 7.3, 7.65, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.23, 0.86, 0.82, 
            1.05, 1.02, 1.09, 1.08, 1.19, 1.25, 1.19, 1.15, 1.25, 
            1.2, 1.28, 0.87, 1.13, 1.23, 1.02, 1.09, 0.906, 1.11, 
            1.09, 1.12, 1.13, 1.09, 1.02, 1.36, 1.04, 1.19, 1.09, 
            1.23, 1.25, 1.1, 1.1, 0.89, 1.02, 1.18, 0.84, 0.95, 
            0.91, 0.88, 0.82, 1.31, 0.87, 1.2, 1.1, 1.07, 1.12, 
            1.12, 0.89, 1.25, 0.89, 1.02, 0.93, 0.94, 1.07, 0.89, 
            0.89, 1.25, 0.98, 1.23, 1.22, 0.82, 1.25, 1.07, 1.12, 
            1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 1.11, 1.23, 
            1.19, 0.96, 1.06, 1.19, 1.38, 0.75, 1.31, 0.84, 0.98, 
            1.23, 1.02, 1.18, 1.19, 1.07, 1.08, 1.05, 0.96, 1.15, 
            1.09, 1.23, 0.84, 0.86, 0.73, 1.42, 1.27, 1.02, 0.8, 
            1.09, 1.04, 0.86, 1, 1.2, 0.86, 1.42, 0.95, 0.9, 
            1.25, 1.1, 0.93, 1.71, 0.95, 1.1, 0.7, 1.23, 0.8, 
            1.23, 0.92, 0.73, 1.2, 0.86, 0.69, 0.97, 0.89, 0.79, 
            0.56, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 
            0.67, 0.92, 0.56, 0.89, 0.96, 0.65, 0.6, 0.67, 0.54, 
            0.55, 0.67, 0.73, 0.48, 0.61, 0.56, 0.78, 0.6, 0.57, 
            0.67, 0.57, 1.36, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 
            0.85, 0.64, 0.74, 0.67, 0.96, 0.87, 0.57, 0.64, 0.7, 
            0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 3.17, 
            2.93, 3.14, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 
            3.3, 3, 2.87, 2.65, 3.36, 2.82, 3.36, 3.71, 3.52, 
            4, 3.63, 3.2, 3.2, 2.74, 2.77, 3.35, 3.55, 2.71, 
            2.88, 2.87, 3, 2.87, 3.33, 2.71, 3.39, 2.69, 2.84, 
            3.38, 3, 3.17, 3, 3.35, 3.33, 3.33, 3.33, 2.75, 3.1, 
            3.1, 3.48, 3.26, 2.93, 3.2, 3.03, 3.33, 2.84, 2.87, 
            1.82, 2.5, 1.59, 2.46, 2.87, 3.52, 2.3, 3.18, 3.48, 
            2.88, 3.36, 1.82, 3.2, 2.96, 3.5, 3.13, 2.14, 2.48, 
            2.52, 1.69, 3.57, 4, 3.14, 2.72, 2.01, 2.93, 3, 3.58, 
            3.21, 2.75, 3.21, 3.33, 2.65, 2.65, 3.3, 2.87, 2.63, 
            2.78, 2.74, 3.17, 2.83, 2.96, 2.77, 3.38, 3.02, 3.57, 
            3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.96, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            2.69, 3.64, 3.58, 2.84, 2.69, 2.78, 2.14, 1.73, 1.42, 
            1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 2.14, 
            1.59, 1.33, 2.05, 2, 1.29, 1.33, 1.33, 1.62, 1.33, 
            3.3, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 
            1.78, 1.58, 1.64, 2, 1.75, 1.68, 1.75, 1.58, 1.56, 
            1.82, 1.73, 2.15, 1.29, 1.68, 1.56, 1.58, 1.56, 1.62, 
            1.6), prol = c(1065, 1050, 1547, 915, 735, 1450, 
            1290, 1295, 1045, 1045, 1045, 1280, 1320, 1060, 1547, 
            1310, 1280, 1080, 1680, 845, 1285, 770, 1035, 1015, 
            845, 1280, 1195, 1285, 915, 1035, 1285, 1235, 885, 
            1235, 1065, 920, 880, 1195, 463, 760, 795, 1035, 
            1095, 680, 885, 1080, 1065, 985, 1060, 1235, 1150, 
            1265, 1190, 1190, 1060, 1120, 970, 1270, 1285, 680, 
            680, 660, 630, 470, 480, 685, 502, 680, 570, 718, 
            870, 410, 472, 1065, 886, 520, 392, 660, 740, 463, 
            278, 714, 480, 515, 886, 1065, 510, 562, 680, 625, 
            480, 450, 495, 450, 345, 880, 625, 428, 660, 406, 
            480, 562, 438, 415, 672, 515, 510, 835, 312, 680, 
            562, 325, 675, 434, 385, 660, 372, 830, 372, 564, 
            472, 1680, 365, 380, 735, 378, 886, 466, 550, 580, 
            510, 680, 560, 600, 835, 695, 720, 515, 580, 590, 
            600, 630, 520, 550, 770, 830, 415, 870, 680, 550, 
            500, 480, 425, 675, 640, 675, 480, 880, 660, 718, 
            840, 680, 570, 520, 615, 680, 695, 685, 880, 615, 
            510, 470, 880, 740, 720, 835, 840, 675)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 13.88, 13.24, 14.3, 14.39, 14.06, 13.62, 
        13.86, 13.86, 13.88, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.34, 13.64, 13.62, 12.93, 13.71, 12.85, 13.5, 
        12.37, 13.39, 13.3, 13.87, 13.86, 13.73, 13.58, 13.68, 
        13.76, 13.36, 12.21, 13.39, 13.05, 12.93, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.69, 13.9, 13.71, 
        14.06, 13.05, 14.34, 13.82, 13.83, 13.74, 13.88, 14.22, 
        13.29, 13.74, 12.37, 12.33, 12.64, 11.61, 12.37, 12.17, 
        12.37, 11.84, 13.17, 13.34, 12.21, 13.41, 13.86, 12.86, 
        12.99, 11.96, 11.66, 13.4, 11.84, 12.21, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 13.36, 12.16, 11.84, 11.76, 12.08, 
        12.08, 12, 12.69, 11.79, 11.62, 12.69, 11.81, 12.29, 
        11.84, 12.29, 12.08, 12.6, 13.73, 11.61, 12.51, 12.07, 
        12.6, 12.72, 12.22, 11.61, 13.5, 12.08, 11.76, 12.45, 
        12.08, 11.03, 11.82, 12.17, 12.17, 12.6, 11.45, 14.75, 
        12.42, 13.05, 11.87, 12.07, 13.39, 11.79, 11.84, 12.04, 
        12.86, 12.88, 13.39, 12.7, 12.51, 13.41, 12.25, 12.53, 
        11.84, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 13.58, 
        11.62, 12.87, 13.32, 11.79, 12.29, 12.79, 12.85, 13.23, 
        12.6, 13.17, 13.84, 12.45, 14.34, 13.48, 13.9, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.58, 14.22, 13.58, 13.4, 
        12.85, 12.77, 14.16, 13.62, 13.4, 13.27, 13.17, 14.13
        ), malic = c(1.71, 4.04, 1.09, 1.95, 3.88, 1.5, 1.87, 
        2.15, 1.6, 1.35, 2.16, 1.48, 1.87, 1.73, 1.87, 1.97, 
        1.92, 1.61, 1.5, 3.1, 1.63, 3.8, 1.67, 1.6, 1.81, 0.94, 
        3.74, 3.8, 1.9, 1.09, 1.5, 1.36, 1.83, 1.53, 1.71, 1.63, 
        1.71, 1.65, 1.5, 2.4, 1.71, 1.9, 1.39, 3.98, 1.5, 4.04, 
        2.4, 1.68, 3.88, 5.51, 1.51, 1.65, 1.87, 1.9, 1.67, 5.04, 
        1.7, 1.97, 2.31, 0.94, 1.21, 1.36, 1.25, 2.08, 1.5, 1.21, 
        1.01, 1.17, 3.12, 0.9, 3.59, 1.51, 1.66, 1.67, 1.09, 
        1.88, 0.9, 2.08, 0.99, 3.87, 1.71, 3.91, 1.13, 1.68, 
        2.05, 0.98, 1.61, 1.67, 0.99, 1.35, 1.88, 1.51, 1.13, 
        1.01, 1.9, 1.5, 2.15, 1.41, 1.35, 1.65, 2.08, 1.61, 3.45, 
        1.72, 2.13, 2.55, 1.73, 1.88, 1.29, 1.87, 3.74, 3.12, 
        2.68, 0.74, 1.39, 0.99, 1.51, 1.61, 3.43, 1.87, 2.4, 
        2.05, 4.43, 2.99, 4.31, 2.16, 1.53, 2.13, 1.53, 4.3, 
        1.35, 2.99, 2.31, 3.55, 1.24, 5.19, 4.31, 5.51, 3.59, 
        3.83, 2.81, 2.56, 3.43, 3.98, 3.88, 4.61, 5.04, 4.61, 
        3.24, 1.73, 3.12, 2.67, 1.9, 3.91, 1.29, 5.19, 4.12, 
        3.03, 1.68, 1.67, 3.83, 2.31, 3.45, 3.45, 2.76, 2.59, 
        4.43, 3.37, 2.58, 4.6, 5.19, 2.39, 4.3, 5.65, 3.91, 2.39, 
        2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.2, 2.72, 
        2.45, 2.61, 2.28, 2.67, 2.3, 2.74, 2.41, 2.39, 2.38, 
        2.7, 2.72, 2.62, 2.48, 2.56, 2.4, 2.65, 2.74, 2.52, 2.61, 
        3.22, 2.19, 2.38, 2.74, 2.3, 2.2, 2.36, 2.72, 2.7, 2.65, 
        2.32, 2.24, 2.55, 2.1, 2.51, 2.38, 2.12, 2.59, 2.29, 
        2.1, 2.44, 2.52, 2.12, 2.4, 2.27, 2.04, 2.68, 2.6, 2.68, 
        2.38, 2.45, 2.3, 2.68, 2.5, 1.36, 2.28, 2.12, 1.92, 2.16, 
        2.53, 2.56, 2.7, 1.92, 2.14, 1.75, 2.21, 2.67, 2.24, 
        2.6, 2.3, 1.98, 1.71, 2.6, 2.16, 2.4, 2, 2.5, 2.51, 2.32, 
        2.58, 2.24, 2.29, 2.62, 2.36, 2.3, 2.32, 2.52, 2.26, 
        2.22, 2.28, 2.2, 2.74, 1.98, 2.56, 2.1, 1.7, 1.88, 2.46, 
        1.88, 2.12, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 
        2.92, 2.5, 2.5, 2.2, 1.99, 2.61, 2.14, 2, 2.42, 3.23, 
        2.32, 2.13, 2.39, 2.17, 2.29, 2.72, 2.22, 2.38, 2.1, 
        2.4, 2.4, 2.36, 2.25, 2.5, 2.54, 2.64, 2.19, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.51, 2.1, 2.48, 2.24, 2.36, 2.62, 
        2.48, 1.98, 2.28, 2.52, 2.32, 2.38, 2.36, 3.22, 2.64, 
        2.38, 2.54, 2.12, 2.35, 2.3, 2.24, 2.6, 2.54, 2.69, 2.38, 
        2.32, 2.28, 2.04, 2.61, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 
        19.5, 21, 16.8, 21, 15.2, 19.6, 17.6, 14, 16, 17, 16.8, 
        16, 11.4, 13.2, 19.6, 19.5, 20, 12.4, 15.2, 16, 18.6, 
        19.5, 17.8, 20, 24, 16.1, 17, 19, 16.8, 16.3, 19.1, 24, 
        16.8, 19, 20.5, 15.5, 22, 15.5, 13.2, 16.2, 18.8, 13.2, 
        20, 17, 21, 16, 18, 18.8, 17.4, 12.4, 16.8, 13.2, 20, 
        20, 20.5, 16.3, 16.8, 15.2, 10.6, 20.5, 16.8, 18, 19, 
        19, 18.1, 17, 19.6, 22.5, 16.8, 20.4, 22.5, 24, 30, 21, 
        17.8, 16, 18, 14.8, 20, 19, 22, 24, 22.5, 18, 18, 22.8, 
        25, 21.6, 17.8, 21, 22, 16, 18, 18, 19, 21.5, 16, 15.6, 
        18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 
        19.5, 20, 23.5, 16, 24.5, 21.5, 20.8, 21, 16, 22.5, 14, 
        16, 24.5, 21.5, 21, 21, 21.5, 28.5, 18, 22, 19, 20, 24, 
        21.5, 19, 22.5, 17, 25, 19.5, 30, 21, 24.5, 23.5, 20, 
        20.5, 21, 20, 22, 18, 21.5, 24, 22, 18, 18.5, 20, 22, 
        19.5, 16, 25, 24, 20, 22.8, 21, 18.5, 22, 22.5, 23, 19.5, 
        24.5, 25, 22.5, 19.5, 20.5, 20.5, 23, 20, 20, 24.5), 
            mg = c(127, 100, 101, 113, 106, 112, 96, 121, 97, 
            120, 97, 122, 89, 91, 102, 112, 120, 122, 123, 116, 
            126, 102, 110, 95, 110, 124, 93, 88, 107, 96, 101, 
            106, 104, 111, 110, 101, 110, 116, 85, 106, 117, 
            94, 101, 101, 107, 111, 102, 101, 120, 108, 92, 139, 
            111, 115, 123, 120, 118, 102, 108, 88, 101, 100, 
            94, 87, 104, 98, 78, 78, 98, 151, 111, 86, 87, 139, 
            101, 90, 88, 112, 85, 111, 86, 90, 78, 85, 117, 99, 
            90, 88, 84, 70, 81, 85, 121, 88, 86, 104, 134, 85, 
            88, 88, 97, 88, 101, 86, 85, 90, 80, 84, 92, 94, 
            107, 88, 103, 88, 84, 85, 86, 89, 88, 87, 96, 88, 
            102, 86, 101, 85, 89, 92, 88, 80, 122, 88, 98, 106, 
            81, 94, 89, 87, 88, 101, 86, 89, 97, 102, 112, 102, 
            106, 97, 92, 113, 123, 101, 116, 98, 103, 107, 89, 
            97, 98, 89, 100, 81, 106, 106, 90, 88, 104, 88, 105, 
            112, 96, 86, 91, 95, 102, 102, 120, 96), phe = c(2.8, 
            2.65, 2.8, 3.85, 2.8, 2.85, 2.5, 2.6, 3.52, 2.98, 
            2.95, 2.2, 2.6, 3.1, 3.1, 2.85, 2.8, 2.8, 3.3, 3.18, 
            3, 2.41, 2.61, 2.48, 2.53, 2.11, 2.65, 2.4, 2.95, 
            2.65, 3, 3.02, 2.42, 2.95, 1.9, 2.7, 2.6, 2.45, 2.72, 
            3.25, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
            2.75, 2.88, 2.72, 2.45, 3.88, 2.53, 2.2, 2.96, 3.2, 
            3, 3.4, 1.98, 2.05, 2.02, 2.56, 3.5, 1.89, 3.02, 
            2.53, 2.11, 2.53, 2.6, 1.1, 2.95, 1.88, 2.56, 1.7, 
            2.55, 3.18, 1.51, 1.9, 2.83, 2.8, 2.2, 2, 1.65, 2.46, 
            2.2, 1.55, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
            3.02, 2.55, 1.25, 2.55, 3.52, 2.95, 2.5, 2.05, 2.56, 
            2.5, 2.2, 2.56, 1.65, 1.9, 2.36, 2.74, 3.18, 2.55, 
            1.75, 2.42, 2.56, 2.46, 1.75, 2, 1.6, 2, 2.9, 3.18, 
            1.4, 1.88, 2.86, 2.8, 2.74, 2.13, 2.22, 2.1, 1.51, 
            1.7, 1.9, 1.7, 1.88, 1.62, 1.38, 1.88, 1.62, 1.25, 
            1.48, 1.4, 1.35, 2, 1.38, 1.4, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.65, 1.48, 1.51, 1.8, 1.9, 1.7, 
            1.38, 2.3, 1.83, 1.48, 1.39, 1.35, 1.68, 1.7, 1.48, 
            1.55, 1.9, 1.25, 1.39, 1.68, 1.7, 1.7, 0.98, 1.65, 
            2.05), fla = c(2.98, 2.76, 2.91, 3.49, 2.17, 3.39, 
            2.52, 2.51, 2.98, 2.88, 3.32, 2.43, 2.76, 2.98, 3.64, 
            2.91, 3.14, 3.4, 3.03, 3.32, 3.32, 2.41, 2.88, 2.37, 
            2.61, 2.68, 2.94, 2.65, 2.97, 2.33, 3.25, 3.19, 2.69, 
            2.98, 1.36, 2.98, 2.41, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 2.78, 2.92, 3.54, 2.69, 
            2.99, 3.74, 2.79, 2.9, 2.78, 3.19, 3.23, 3.67, 0.57, 
            2.27, 1.41, 1.79, 3.14, 1.75, 2.65, 2.89, 2, 1.3, 
            2.98, 1.02, 2.86, 1.39, 2.89, 2.14, 1.57, 2.14, 1.6, 
            1.85, 2.63, 2.26, 2.53, 1.58, 1.59, 2.21, 1.92, 1.69, 
            1.61, 1.69, 2.37, 1.64, 1.25, 1.46, 2.68, 2.52, 2.27, 
            0.99, 1.69, 3.23, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 
            1.84, 2.03, 1.76, 2.29, 2.92, 2.33, 2.27, 2.03, 2.01, 
            2.29, 2.17, 1.6, 2.09, 0.61, 1.64, 2.76, 3.23, 1.39, 
            0.68, 3.03, 2.65, 3.15, 2.24, 2.24, 1.75, 1.02, 0.5, 
            1.39, 1.2, 0.58, 0.66, 0.47, 1.02, 0.69, 0.6, 0.5, 
            0.52, 0.52, 0.8, 0.66, 0.55, 0.34, 0.65, 0.76, 1.39, 
            1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.65, 0.58, 1.31, 
            1.1, 0.92, 1.1, 0.6, 0.58, 0.68, 0.47, 0.92, 0.66, 
            0.84, 0.96, 0.49, 0.66, 0.7, 0.61, 0.66, 0.69, 0.68, 
            0.76), nfla = c(0.28, 0.26, 0.3, 0.24, 0.39, 0.34, 
            0.43, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 
            0.3, 0.32, 0.2, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 
            0.32, 0.32, 0.34, 0.27, 0.32, 0.26, 0.29, 0.26, 0.42, 
            0.3, 0.29, 0.26, 0.26, 0.3, 0.28, 0.2, 0.34, 0.28, 
            0.21, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 
            0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.43, 0.26, 0.28, 
            0.24, 0.53, 0.32, 0.34, 0.45, 0.37, 0.26, 0.27, 0.3, 
            0.22, 0.28, 0.39, 0.43, 0.21, 0.13, 0.34, 0.24, 0.43, 
            0.35, 0.43, 0.3, 0.43, 0.4, 0.61, 0.22, 0.3, 0.39, 
            0.3, 0.61, 0.32, 0.4, 0.5, 0.3, 0.25, 0.17, 0.32, 
            0.43, 0.61, 0.24, 0.39, 0.22, 0.29, 0.34, 0.37, 0.29, 
            0.66, 0.17, 0.48, 0.39, 0.29, 0.24, 0.26, 0.34, 0.42, 
            0.26, 0.52, 0.3, 0.29, 0.34, 0.42, 0.32, 0.66, 0.43, 
            0.3, 0.39, 0.26, 0.39, 0.39, 0.4, 0.43, 0.29, 0.53, 
            0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 
            0.61, 0.43, 0.42, 0.37, 0.43, 0.4, 0.47, 0.45, 0.27, 
            0.22, 0.32, 0.26, 0.61, 0.53, 0.5, 0.58, 0.63, 0.53, 
            0.52, 0.3, 0.5, 0.6, 0.4, 0.41, 0.53, 0.43, 0.4, 
            0.39, 0.41, 0.53, 0.48, 0.53, 0.63, 0.43, 0.43, 0.53, 
            0.52), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 
            1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.63, 2.96, 
            1.46, 1.99, 2.49, 1.86, 1.66, 2.1, 1.95, 1.69, 1.46, 
            1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.56, 
            1.63, 1.54, 1.86, 1.36, 1.44, 1.37, 2.29, 1.92, 1.37, 
            1.87, 1.77, 2.03, 2.03, 2.19, 2.14, 2.38, 2.08, 1.56, 
            2.29, 1.87, 1.68, 1.62, 2.45, 1.63, 1.66, 2.81, 0.42, 
            1.54, 1.56, 0.73, 1.87, 1.03, 1.98, 1.92, 1.04, 1.36, 
            2.5, 1.46, 1.87, 1.03, 2.49, 1.04, 1.15, 1.46, 1.35, 
            2.76, 1.95, 1.43, 1.95, 1.37, 1.62, 1.46, 1.46, 1.56, 
            1.34, 1.35, 1.38, 1.64, 1.63, 1.63, 1.99, 1.35, 3.28, 
            1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.15, 
            1.42, 1.61, 1.63, 2.08, 2.49, 1.69, 1.03, 1.95, 1.35, 
            1.04, 1.87, 1.35, 1.61, 0.83, 1.87, 1.83, 1.92, 1.71, 
            0.55, 2.91, 1.35, 1.77, 1.25, 1.9, 0.83, 0.94, 0.68, 
            0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 0.64, 0.75, 
            1.45, 0.55, 1.25, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 
            1.25, 1.26, 1.56, 1.14, 1.95, 1.55, 1.56, 1.14, 2.7, 
            2.29, 1.04, 1.35, 1.54, 0.94, 1.35, 1.15, 1.46, 0.97, 
            1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 
            1.35), col = c(5.64, 4.38, 5.68, 7.8, 6.13, 6.75, 
            7.9, 5.05, 7.9, 7.22, 5.75, 7.8, 5.6, 7.22, 7.5, 
            3.4, 4, 3.6, 8.7, 5.1, 5.65, 4.5, 3.6, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 4.7, 3.3, 5.4, 4.2, 
            5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 
            5.04, 5.24, 4.9, 6.1, 7.3, 8.9, 7.2, 9.3, 8.21, 6.3, 
            5.85, 6.25, 8.21, 5.43, 6.8, 4.7, 3.27, 5.75, 3.8, 
            4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 7.1, 3.38, 
            3.74, 2.6, 3.21, 3.8, 4.6, 3.9, 2.9, 3.25, 2.5, 3.9, 
            2.62, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 
            3.6, 3.05, 2.15, 3.25, 2.6, 5.04, 2.9, 2.7, 2.45, 
            3.05, 4.45, 2.8, 1.9, 2.94, 2.7, 3.4, 3.3, 2.7, 2.6, 
            2.9, 2.95, 3.93, 3.08, 2.9, 1.9, 2.8, 2.4, 3.4, 1.28, 
            3.25, 4, 3.8, 4.28, 2.8, 1.74, 3.94, 2.8, 2.12, 3.4, 
            5.4, 8.9, 8.9, 5, 5.45, 7.1, 10.68, 5, 5.7, 4.92, 
            7.1, 7.5, 4.35, 4.4, 8.21, 4, 4.9, 5, 8.42, 9.4, 
            3.25, 4, 7.1, 10.52, 7.6, 7.9, 7.1, 7.5, 13, 8.21, 
            7.65, 6.62, 5.58, 9.2, 6.25, 6.62, 10.68, 10.26, 
            7.1, 8.5, 7.9, 5.7, 9.7, 7.7, 7.3, 7.5, 9.3, 9.2), 
            hue = c(1.04, 1.05, 1.2, 0.86, 0.79, 1.05, 1.02, 
            1.12, 1.08, 0.95, 1.25, 1.05, 1.15, 1.25, 1.2, 1.28, 
            0.86, 1.13, 1.23, 0.82, 1.09, 0.57, 1.11, 1.09, 1.12, 
            1.13, 0.75, 1.02, 1.04, 1.04, 1.19, 1.09, 1.27, 1.25, 
            1.1, 1.42, 1.38, 1.25, 1.18, 0.95, 0.95, 0.91, 0.88, 
            0.82, 0.75, 0.87, 0.73, 0.95, 1.07, 1.12, 1.12, 0.96, 
            0.86, 1.12, 1.31, 1.05, 0.94, 1.07, 0.89, 0.82, 1.25, 
            0.98, 1.23, 1.22, 0.98, 0.87, 1.05, 1.12, 1.02, 1.28, 
            0.906, 1.36, 0.98, 1.31, 1.08, 1.23, 1.19, 0.96, 
            1.06, 1.19, 1.38, 0.93, 1.31, 0.84, 0.96, 1.23, 1.08, 
            1.31, 1.13, 1.07, 1.08, 1.05, 0.67, 1.15, 1.1, 0.95, 
            1.12, 0.81, 0.97, 1.42, 1.27, 1.36, 0.8, 1.28, 1.04, 
            0.69, 1, 1.31, 0.86, 1.13, 1.09, 0.9, 0.67, 1.1, 
            0.93, 1.71, 0.95, 1.38, 0.7, 1.27, 0.8, 0.9, 0.92, 
            0.73, 1.04, 0.86, 0.69, 0.97, 0.89, 0.79, 1.19, 0.74, 
            0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.85, 0.7, 
            0.96, 0.89, 0.7, 0.65, 0.6, 0.73, 0.54, 0.55, 0.84, 
            0.79, 0.48, 0.61, 0.56, 0.87, 0.6, 0.57, 0.67, 0.57, 
            1.09, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.64, 
            0.74, 0.67, 0.73, 0.48, 0.74, 0.64, 0.7, 0.59, 0.6, 
            0.61), ratio = c(3.92, 3.4, 3.17, 2.78, 2.93, 2.84, 
            3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 2.87, 3, 
            2.65, 2.65, 2.87, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 
            4, 3.2, 2.77, 2.77, 3.69, 3.57, 2.71, 2.88, 2.87, 
            3, 2.87, 3, 2.77, 3.17, 2.69, 2.83, 3.38, 3, 4, 3, 
            3.35, 3.33, 2.87, 3.33, 2.75, 3.1, 3.39, 3.21, 3.26, 
            2.93, 3.2, 3.03, 3.21, 2.84, 2.87, 1.82, 2.96, 1.59, 
            2.46, 2.87, 3.55, 2.3, 3.18, 3.48, 2.85, 3, 1.82, 
            3.5, 2.82, 3.5, 3.13, 2.14, 2.48, 2.52, 2.65, 3.38, 
            3.14, 3.14, 2.72, 2.01, 2.63, 3, 2.81, 3.21, 2.75, 
            3.21, 3.13, 2.65, 3.12, 3.3, 3.92, 2.63, 1.59, 2.74, 
            2.87, 2.83, 2.96, 2.77, 3.38, 3.48, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.69, 3.19, 2.87, 
            3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.12, 3.64, 
            3.26, 2.84, 2.78, 2.78, 2.82, 3.12, 1.42, 1.36, 1.29, 
            1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.33, 1.58, 2.48, 
            2.05, 2, 1.82, 1.33, 1.82, 1.62, 1.33, 3.39, 1.47, 
            1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 
            1.29, 1.58, 1.75, 1.68, 1.75, 1.56, 1.56, 1.6, 1.82, 
            1.33, 1.78, 1.51, 1.96, 2, 1.56, 1.62, 1.6), prol = c(1065, 
            1050, 870, 1190, 735, 1450, 1290, 1295, 1045, 1045, 
            1285, 1280, 1320, 1320, 1547, 1310, 1280, 885, 1680, 
            845, 1280, 770, 1035, 1015, 845, 870, 1195, 1285, 
            915, 1035, 1285, 1065, 885, 1235, 562, 920, 880, 
            714, 1120, 760, 795, 1035, 1095, 680, 885, 1080, 
            1065, 985, 1060, 1265, 1150, 1265, 1190, 1195, 1060, 
            1120, 970, 1270, 1285, 564, 885, 685, 630, 410, 630, 
            550, 502, 714, 372, 718, 870, 410, 472, 580, 886, 
            564, 392, 660, 480, 463, 278, 714, 580, 515, 438, 
            480, 410, 562, 680, 625, 480, 450, 495, 680, 345, 
            415, 625, 428, 660, 406, 600, 562, 438, 415, 672, 
            564, 510, 830, 312, 680, 562, 325, 672, 434, 385, 
            672, 415, 886, 372, 564, 695, 1450, 365, 380, 434, 
            378, 985, 466, 615, 580, 660, 410, 560, 600, 415, 
            695, 720, 515, 580, 590, 600, 515, 520, 550, 434, 
            830, 415, 515, 463, 550, 500, 480, 425, 675, 640, 
            463, 480, 880, 660, 680, 406, 680, 570, 985, 615, 
            438, 695, 685, 660, 640, 510, 470, 660, 740, 515, 
            835, 840, 672)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 13.74, 13.24, 
        13.23, 14.39, 14.06, 13.34, 13.86, 14.16, 13.34, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 14.75, 13.64, 14.22, 
        12.93, 13.71, 12.85, 13.5, 12.93, 13.39, 13.3, 13.87, 
        13.34, 13.73, 13.58, 13.68, 13.76, 13.68, 13.48, 12.96, 
        13.05, 12.07, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 13.4, 13.9, 13.05, 13.63, 13.05, 14.23, 13.82, 
        13.17, 13.74, 12.93, 14.22, 13.29, 13.63, 12.37, 12.33, 
        12.64, 11.84, 12.37, 12.17, 12.37, 13.24, 12.08, 13.34, 
        12.21, 12.53, 13.86, 11.84, 12.99, 11.96, 11.66, 12.37, 
        11.84, 12.08, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.22, 
        12.16, 12, 11.66, 12.08, 12.08, 12, 12.69, 12.08, 11.62, 
        11.84, 11.81, 12.29, 13.83, 12.29, 12.08, 12.6, 13.62, 
        11.61, 12.51, 11.76, 12.17, 12.72, 12.22, 11.61, 12.87, 
        12.72, 11.76, 12, 12.08, 11.03, 11.82, 12.51, 11.66, 
        11.03, 11.45, 13.29, 12.42, 13.05, 11.87, 12.07, 11.96, 
        11.79, 12.85, 12.04, 12.86, 12.88, 13.71, 12.7, 12.51, 
        12.99, 12.25, 12.53, 13.24, 12.84, 12.93, 13.36, 13.52, 
        13.62, 12.25, 12.96, 12.07, 12.87, 13.32, 13.64, 11.66, 
        12.79, 14.16, 13.23, 13.24, 13.17, 13.84, 12.45, 14.34, 
        13.48, 14.13, 13.69, 12.85, 12.96, 13.78, 13.73, 14.22, 
        13.29, 13.58, 13.4, 13.05, 12.77, 14.16, 13.17, 13.4, 
        13.27, 13.17, 14.13), malic = c(1.71, 1.29, 3.37, 1.95, 
        2.39, 2.4, 1.87, 2.15, 1.6, 1.35, 2.16, 1.48, 5.04, 1.73, 
        1.87, 1.81, 1.92, 1.97, 1.53, 3.1, 1.63, 3.8, 1.61, 1.6, 
        1.81, 1.71, 1.65, 1.71, 1.9, 1.35, 1.5, 4.31, 1.83, 1.53, 
        3.8, 2.31, 2.99, 1.65, 1.5, 1.21, 1.71, 2.56, 1.48, 3.98, 
        1.71, 4.04, 1.67, 1.68, 1.73, 4.43, 0.94, 1.65, 1.67, 
        1.9, 1.67, 1.6, 1.7, 1.97, 2.67, 0.94, 1.51, 1.36, 1.25, 
        1.9, 1.72, 1.21, 1.01, 1.17, 1.25, 1.68, 1.65, 1.51, 
        1.66, 1.67, 1.09, 1.88, 0.9, 1.67, 0.99, 3.87, 0.9, 1.88, 
        1.13, 3.43, 1.25, 0.98, 1.61, 1.67, 1.72, 3.87, 1.63, 
        1.51, 1.73, 1.61, 1.81, 1.68, 1.7, 1.41, 1.63, 1.21, 
        2.08, 2.16, 1.81, 1.72, 4.04, 2.55, 1.73, 1.36, 1.29, 
        3.03, 3.74, 1.29, 2.68, 0.74, 1.39, 1.68, 2.05, 1.61, 
        3.43, 1.09, 2.4, 2.05, 4.43, 2.76, 4.31, 2.16, 1.53, 
        2.13, 3.45, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 2.67, 
        2.67, 5.51, 3.59, 1.68, 2.81, 2.56, 1.35, 4.61, 3.88, 
        3.37, 5.04, 4.61, 3.24, 3.45, 3.12, 2.67, 1.9, 3.37, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 1.29, 4.31, 
        3.45, 2.76, 3.12, 1.83, 3.37, 2.58, 4.6, 5.19, 2.39, 
        2.13, 5.65, 3.91, 2.58, 2.59, 4.1), ash = c(2.43, 2.14, 
        2.67, 2.5, 2.26, 2.27, 2.45, 2.61, 2.2, 2.37, 2.3, 2.7, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.5, 2.65, 
        2.3, 2.52, 2.61, 3.22, 2.25, 2.59, 2.51, 2.74, 2.48, 
        2.36, 2.29, 2.7, 2.65, 2.65, 2.2, 2.55, 2.1, 2.51, 2.65, 
        2.12, 2.59, 2.29, 2.1, 2.44, 2.29, 2.12, 2.4, 2.27, 2.04, 
        2.7, 2.61, 2.39, 2.65, 2.74, 2.3, 2.68, 2.5, 1.36, 2.58, 
        2.32, 1.92, 2.16, 2.53, 2.56, 2.29, 1.92, 2.4, 1.75, 
        2.21, 2.67, 2.24, 2.6, 2.3, 1.92, 1.71, 2.29, 2.58, 2.4, 
        2, 1.71, 2.51, 2.32, 2.58, 2.24, 2.13, 2.62, 2.74, 2.3, 
        2.32, 2.3, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.5, 2.32, 
        1.7, 1.98, 2.46, 1.88, 2, 2.27, 2.12, 2.28, 1.94, 2.7, 
        1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.58, 1.71, 2, 
        2.42, 3.23, 2, 2.13, 2.39, 2.17, 2.29, 2.72, 2.42, 2.38, 
        2.69, 2.4, 2.4, 2.36, 2.25, 2.74, 2.54, 2.64, 2.19, 2.61, 
        2.7, 2.35, 2.72, 2.35, 2.2, 2.27, 2, 2.48, 1.82, 2.36, 
        2.62, 2.48, 2.12, 2.28, 2.54, 2.32, 2.38, 2.26, 2.59, 
        2.64, 2.38, 2.54, 2.35, 2.35, 2.3, 2.36, 2.6, 2.32, 2.69, 
        2.27, 2.32, 2.28, 2.62, 1.99, 2.48, 2.26, 2.37, 2.74), 
            alca = c(15.6, 16, 22.5, 16.8, 21, 15.2, 17.5, 17.6, 
            14, 16, 18.8, 16.8, 16, 11.4, 16.1, 13.2, 18.5, 20, 
            14, 15.2, 16, 18.6, 16.1, 17.8, 20, 15.2, 16.1, 17, 
            19.1, 16, 16.3, 19.1, 15.5, 19, 19, 20.5, 15.5, 21.5, 
            15.5, 13.2, 16.2, 18.8, 17, 20.4, 17, 19.5, 16, 15.5, 
            18.8, 17.4, 12.4, 16.3, 16.3, 19.5, 17.4, 20.5, 16.3, 
            16.8, 16.3, 10.6, 24, 16.8, 18, 19, 19, 18.1, 15.6, 
            19.6, 16.8, 16.8, 20.4, 20, 24, 30, 21, 15.5, 16, 
            18, 14.8, 25, 19, 16, 24, 22.5, 18, 18, 22.8, 24, 
            21.6, 15.6, 19.5, 22, 20, 18, 18, 19, 21.5, 16, 16.8, 
            18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 
            20, 19.5, 14.8, 24, 22.5, 18, 21.5, 20.8, 18, 16, 
            16.8, 20, 17.4, 24, 21.5, 21, 21, 21.5, 28.5, 20.4, 
            22, 22.5, 20, 24, 21.5, 20.5, 18, 22, 25, 19.5, 21.5, 
            21, 21, 23.5, 20, 28.5, 21, 20, 20, 19.5, 21.5, 24, 
            22, 15.2, 18.5, 20, 22, 19.5, 19.5, 25, 22.5, 20, 
            19.5, 21.5, 18.5, 20.4, 22.5, 23, 19.5, 24.5, 25, 
            25, 19.5, 24, 20.5, 23, 20, 20.5, 24.5), mg = c(127, 
            100, 101, 113, 104, 112, 96, 121, 97, 101, 98, 111, 
            89, 91, 102, 112, 120, 96, 121, 116, 126, 102, 105, 
            95, 97, 124, 93, 102, 107, 96, 101, 106, 104, 107, 
            110, 89, 110, 101, 112, 89, 117, 98, 101, 91, 107, 
            111, 102, 101, 122, 108, 92, 117, 111, 115, 108, 
            89, 118, 102, 108, 88, 101, 100, 94, 87, 104, 98, 
            78, 78, 112, 151, 105, 86, 87, 139, 101, 90, 91, 
            112, 94, 92, 86, 84, 78, 85, 86, 99, 90, 88, 84, 
            70, 81, 92, 88, 88, 107, 80, 134, 85, 88, 106, 97, 
            88, 91, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 
            88, 84, 85, 86, 84, 101, 87, 96, 124, 102, 86, 88, 
            85, 112, 92, 88, 80, 122, 103, 98, 106, 116, 94, 
            89, 85, 88, 101, 86, 89, 97, 88, 112, 102, 91, 85, 
            92, 113, 123, 97, 116, 98, 103, 85, 89, 97, 98, 89, 
            80, 80, 106, 106, 90, 88, 101, 88, 105, 112, 96, 
            86, 91, 95, 102, 123, 120, 99), phe = c(2.8, 2.65, 
            2.8, 3.85, 2.8, 3.52, 2.5, 2.6, 2.95, 2.98, 2.95, 
            2.2, 2.6, 3.1, 3, 2.85, 2.8, 3.1, 3.3, 2.56, 3, 2.41, 
            2.61, 2.48, 2.53, 2.6, 2.65, 2.4, 2.95, 2.65, 3, 
            2.72, 2.42, 2.95, 2, 2.7, 2.6, 2.45, 2.6, 3, 3.15, 
            2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 
            2.72, 2.45, 3.88, 3.18, 2.45, 2.96, 3.2, 3, 3.4, 
            1.98, 2.05, 2.02, 2, 3.5, 1.89, 2.2, 3.4, 2.11, 2.53, 
            3.18, 1.1, 2.95, 1.88, 2.05, 1.83, 2.2, 1.38, 2.2, 
            1.9, 2.83, 2.36, 2.2, 1.45, 1.65, 2.56, 2.2, 1.93, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.8, 
            1.7, 2.55, 3.52, 3.2, 2.22, 1.51, 2.56, 2.5, 2.2, 
            2.2, 1.65, 1.83, 2.36, 2.74, 3.18, 2.55, 1.75, 1.65, 
            2.56, 2.46, 2, 2, 2.5, 2, 2.9, 3.18, 1.9, 1.8, 2.86, 
            2.98, 2.74, 2.13, 2.22, 2.1, 1.51, 0.98, 1.25, 1.7, 
            0.98, 1.62, 1.38, 1.38, 1.62, 1.9, 1.39, 1.4, 1.38, 
            2, 1.38, 1.38, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
            2.2, 1.38, 1.48, 1.98, 1.8, 1.9, 2.2, 1.4, 2.3, 1.83, 
            1.65, 1.39, 1.35, 0.98, 1.7, 1.48, 1.55, 1.48, 1.25, 
            1.39, 1.68, 1.65, 1.65, 1.65, 1.65, 2.05), fla = c(2.78, 
            2.76, 2.98, 3.49, 2.14, 3.39, 2.52, 2.51, 2.98, 3.04, 
            3.32, 2.43, 2.76, 2.51, 3.64, 2.91, 3.14, 3.4, 3.17, 
            3.15, 3, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.11, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.32, 2.41, 2.98, 2.41, 
            2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 
            3.17, 3.67, 2.92, 3.54, 2.88, 2.99, 3.74, 2.79, 2.9, 
            2.78, 3.23, 3.23, 3.67, 0.57, 2.26, 1.41, 1.79, 3.54, 
            1.75, 2.65, 3, 2, 1.3, 3.56, 1.02, 2.86, 1.46, 2.89, 
            2.14, 1.57, 0.68, 2.01, 1.85, 2.43, 2.26, 2.53, 1.58, 
            1.59, 2.21, 2.53, 1.69, 1.61, 1.69, 1.84, 0.92, 1.25, 
            1.46, 2.94, 2.92, 2.27, 0.99, 1.92, 3.67, 2.99, 2.17, 
            1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 2.65, 2.92, 
            2.61, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.79, 
            1.64, 2.37, 3.17, 1.3, 0.6, 3.03, 2.65, 3.15, 2.24, 
            1.61, 1.75, 0.92, 0.58, 0.5, 1.2, 0.58, 0.66, 0.47, 
            1.36, 0.76, 0.6, 0.5, 0.76, 0.52, 0.8, 0.58, 0.55, 
            0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 0.58, 
            0.63, 0.57, 0.58, 1.31, 1.1, 0.92, 1.28, 0.6, 0.61, 
            0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.52, 0.7, 
            0.61, 0.58, 0.69, 0.68, 0.76), nfla = c(0.28, 0.24, 
            0.3, 0.24, 0.39, 0.34, 0.39, 0.31, 0.29, 0.22, 0.22, 
            0.26, 0.29, 0.43, 0.29, 0.3, 0.22, 0.32, 0.32, 0.17, 
            0.24, 0.25, 0.27, 0.26, 0.34, 0.43, 0.34, 0.27, 0.24, 
            0.26, 0.29, 0.28, 0.42, 0.26, 0.29, 0.26, 0.27, 0.29, 
            0.28, 0.2, 0.34, 0.32, 0.3, 0.32, 0.28, 0.3, 0.27, 
            0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 
            0.26, 0.34, 0.29, 0.28, 0.2, 0.53, 0.32, 0.26, 0.45, 
            0.37, 0.26, 0.27, 0.29, 0.26, 0.5, 0.28, 0.52, 0.21, 
            0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.24, 0.4, 
            0.61, 0.22, 0.3, 0.26, 0.4, 0.48, 0.47, 0.27, 0.5, 
            0.43, 0.25, 0.17, 0.32, 0.53, 0.32, 0.24, 0.26, 0.26, 
            0.29, 0.34, 0.37, 0.25, 0.66, 0.3, 0.48, 0.39, 0.29, 
            0.24, 0.26, 0.39, 0.42, 0.24, 0.52, 0.3, 0.4, 0.66, 
            0.3, 0.32, 0.4, 0.43, 0.3, 0.6, 0.32, 0.39, 0.39, 
            0.4, 0.41, 0.3, 0.37, 0.27, 0.17, 0.6, 0.63, 0.53, 
            0.63, 0.58, 0.53, 0.53, 0.45, 0.58, 0.66, 0.6, 0.43, 
            0.4, 0.47, 0.45, 0.29, 0.22, 0.4, 0.26, 0.61, 0.53, 
            0.43, 0.27, 0.63, 0.53, 0.52, 0.45, 0.5, 0.6, 0.4, 
            0.41, 0.58, 0.43, 0.4, 0.39, 0.5, 0.3, 0.48, 0.24, 
            0.61, 0.43, 0.43, 0.53, 0.4), pro = c(2.29, 1.28, 
            2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 
            1.57, 1.81, 2.29, 2.96, 1.46, 2.96, 2.08, 1.86, 1.66, 
            2.1, 2.29, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 
            1.98, 2.38, 1.95, 1.57, 1.97, 1.54, 1.86, 1.36, 1.44, 
            1.37, 1.46, 1.61, 2.29, 2.49, 1.04, 2.03, 2.08, 2.19, 
            2.14, 2.38, 2.08, 1.35, 2.29, 1.87, 1.68, 1.62, 2.45, 
            1.25, 1.66, 2.18, 0.42, 1.03, 1.9, 0.73, 1.87, 1.03, 
            1.34, 2.49, 1.04, 0.94, 2.5, 1.46, 1.87, 1.03, 1.66, 
            1.04, 1.15, 1.46, 0.86, 2.76, 1.95, 1.43, 1.46, 1.03, 
            1.62, 1.63, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 1.63, 
            1.3, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 
            1.35, 1.31, 1.42, 1.11, 1.42, 1.46, 1.63, 2.08, 2.49, 
            1.4, 1.25, 1.36, 0.75, 1.04, 2.5, 0.83, 1.61, 0.83, 
            1.87, 1.83, 2.91, 1.71, 0.55, 2.91, 1.35, 1.77, 1.31, 
            1.9, 1.9, 0.94, 0.68, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 0.94, 0.75, 0.94, 0.55, 1.41, 1.14, 1.3, 
            0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.41, 0.73, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.26, 1.25, 0.94, 
            0.84, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 
            1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 4.38, 
            5.68, 7.8, 3.95, 6.75, 7.9, 5.05, 5.58, 7.22, 5.75, 
            4.8, 5.6, 5.75, 7.5, 5.75, 9.2, 5.85, 8.7, 5.1, 5.65, 
            4.5, 3.74, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 
            5.7, 4.7, 5.85, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 
            6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 5.45, 
            8.9, 7.2, 7.1, 4.7, 6.3, 5.85, 6.25, 6.3, 4.8, 6.8, 
            4.38, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 
            3.17, 2.85, 7.5, 3.38, 3.74, 3.74, 3.21, 3.8, 4.6, 
            4.5, 4.7, 3.52, 2.5, 3.9, 3.8, 4.8, 3.05, 2.62, 2.45, 
            2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 
            4.38, 2.9, 2.12, 3.6, 3.7, 3.4, 2.8, 3.8, 2.94, 2.7, 
            3.4, 3.3, 2.7, 2.5, 2.9, 4.6, 3.58, 3.08, 2.9, 1.9, 
            2.5, 3.4, 3.4, 1.28, 3.25, 7.22, 3.8, 3.3, 2.8, 3.4, 
            3.94, 3.7, 2.12, 4.5, 4.6, 8.21, 7.1, 5, 5.45, 7.1, 
            4.8, 5, 5.7, 4.92, 5.68, 7.5, 4.35, 4.4, 8.21, 4, 
            4.9, 4.36, 8.42, 9.4, 3.93, 4.4, 7.1, 10.52, 7.6, 
            7.9, 4, 7.5, 13, 7.5, 7.65, 4.28, 5.58, 4.8, 4, 6.62, 
            10.68, 10.26, 9.2, 8.5, 5.75, 5.43, 9.7, 7.7, 7.3, 
            9.4, 9.3, 9.2), hue = c(1.04, 1.05, 0.89, 0.86, 0.95, 
            1.05, 1.02, 0.93, 1.08, 1.04, 1.25, 1.19, 1.15, 1.25, 
            1.2, 1.28, 1.09, 1.13, 1.23, 0.48, 1.09, 1.19, 1.11, 
            1.09, 1.12, 1.13, 1.25, 1.02, 1.25, 1.04, 1.19, 1.09, 
            0.87, 1.25, 1.1, 0.8, 1.02, 0.86, 1.18, 0.87, 0.95, 
            0.91, 0.88, 0.82, 0.93, 0.87, 0.69, 1.02, 1.07, 1.12, 
            1.12, 1.05, 0.86, 0.8, 1.09, 1.28, 0.94, 1.07, 0.89, 
            0.78, 1.25, 0.98, 1.23, 1.22, 1.23, 1.25, 1.04, 1.12, 
            1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 1.2, 1.23, 1.19, 
            0.96, 1.06, 1.19, 1.38, 1.04, 1.31, 0.84, 1.27, 1.23, 
            1.06, 1.19, 1.25, 1.07, 1.08, 1.05, 1.02, 1.15, 1.31, 
            1.18, 0.93, 1.07, 1.09, 1.42, 1.27, 0.94, 0.8, 1.04, 
            1.04, 1.19, 1, 1.23, 0.86, 1.05, 0.75, 0.9, 0.86, 
            1.1, 0.93, 1.71, 0.95, 1.12, 0.7, 1.15, 0.8, 1.2, 
            0.92, 0.73, 1.09, 0.86, 0.69, 0.97, 0.89, 0.79, 1.12, 
            0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 1.19, 
            0.78, 0.54, 0.89, 0.61, 0.65, 0.6, 0.7, 0.54, 0.55, 
            0.85, 0.86, 0.48, 0.61, 0.56, 0.78, 0.6, 0.57, 0.67, 
            0.57, 0.86, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 
            0.7, 0.74, 0.67, 0.67, 0.73, 0.89, 0.64, 0.7, 0.59, 
            0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 3.39, 2.93, 
            3.14, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.58, 
            3, 2.82, 2.65, 3.52, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 
            2.77, 3.2, 3.36, 2.77, 3.17, 2.82, 2.71, 2.88, 2.87, 
            3, 2.87, 2.81, 2.14, 3.39, 2.69, 3, 3.38, 3, 3.71, 
            3, 3.35, 3.33, 2.96, 3.33, 2.75, 3.1, 3, 3, 3.26, 
            2.93, 3.2, 3.03, 3.33, 2.84, 2.87, 1.82, 3.21, 1.59, 
            2.46, 2.87, 2.52, 2.3, 3.18, 3.48, 2.87, 2.75, 1.82, 
            2.81, 2.46, 3.5, 3.13, 2.14, 2.48, 2.52, 2.01, 3.02, 
            3.13, 3.14, 2.72, 2.01, 2.46, 2.83, 3.39, 3.21, 2.75, 
            3.21, 3.12, 2.65, 2.5, 3.3, 3.33, 2.63, 1.75, 2.74, 
            3.38, 2.83, 2.96, 2.77, 3.38, 2.93, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.9, 3.19, 2.87, 
            3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 1.47, 3.64, 
            2.87, 2.84, 3.02, 2.78, 2.96, 3.12, 1.42, 1.36, 1.29, 
            1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.62, 1.75, 2.87, 
            2.05, 2, 1.58, 1.33, 2, 1.62, 1.33, 1.82, 1.47, 1.33, 
            1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 2.87, 
            1.96, 1.75, 1.68, 1.75, 1.33, 1.56, 1.33, 1.36, 1.96, 
            1.58, 1.33, 1.96, 1.55, 1.56, 1.62, 1.6), prol = c(1065, 
            1050, 1015, 795, 735, 1450, 1290, 1295, 1045, 1045, 
            845, 1280, 1320, 1320, 1547, 1310, 1280, 1060, 1680, 
            845, 1320, 770, 1035, 1015, 845, 1280, 1195, 1285, 
            915, 1035, 1285, 1320, 1295, 1235, 1285, 920, 880, 
            1285, 718, 760, 795, 1035, 1095, 680, 885, 1080, 
            1065, 985, 1060, 1265, 1150, 1265, 1190, 695, 1060, 
            1120, 970, 1270, 1285, 415, 515, 515, 630, 640, 660, 
            550, 502, 495, 463, 718, 870, 410, 472, 680, 886, 
            510, 392, 406, 718, 463, 278, 714, 365, 515, 600, 
            740, 720, 562, 680, 625, 480, 450, 495, 845, 345, 
            564, 625, 428, 660, 406, 845, 562, 438, 415, 672, 
            480, 510, 630, 312, 680, 562, 325, 325, 434, 385, 
            562, 564, 1065, 372, 564, 463, 1280, 365, 380, 410, 
            378, 562, 466, 480, 580, 845, 695, 560, 600, 510, 
            695, 720, 515, 580, 590, 600, 600, 520, 550, 312, 
            830, 415, 672, 365, 550, 500, 480, 425, 675, 640, 
            600, 480, 880, 660, 570, 385, 680, 570, 515, 615, 
            740, 695, 685, 625, 680, 510, 470, 590, 740, 515, 
            835, 840, 502)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 13.63, 13.24, 
        13.74, 14.39, 14.06, 11.84, 13.86, 14.22, 13.83, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 13.63, 13.64, 13.73, 
        12.93, 13.71, 12.85, 13.5, 12.84, 13.39, 13.3, 13.87, 
        12.79, 13.73, 13.58, 13.68, 13.76, 12.96, 12.93, 12.93, 
        13.05, 13.52, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 12.96, 13.9, 14.21, 13.86, 13.05, 13.78, 13.82, 
        13.75, 13.74, 14.22, 14.22, 13.29, 13.17, 12.37, 12.33, 
        12.64, 11.81, 12.37, 12.17, 12.37, 12.93, 12.93, 13.34, 
        12.21, 12.08, 13.86, 12.6, 12.99, 11.96, 11.66, 11.76, 
        11.84, 14.22, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.08, 
        12.16, 11.79, 11.62, 12.08, 12.08, 12, 12.69, 13.05, 
        11.62, 11.82, 11.81, 12.29, 12.99, 12.29, 12.08, 12.6, 
        12.33, 12.17, 12.51, 12.08, 12.17, 12.72, 12.22, 11.61, 
        13.24, 12.29, 11.76, 11.45, 12.08, 11.03, 11.82, 12.93, 
        12.22, 12.6, 11.45, 13.3, 12.42, 13.05, 11.87, 12.07, 
        13.05, 11.79, 12.07, 12.04, 12.86, 12.88, 12.29, 12.7, 
        12.51, 12.84, 12.25, 12.53, 13.05, 12.84, 12.93, 13.36, 
        13.52, 13.62, 12.25, 11.61, 11.45, 12.87, 13.32, 13.48, 
        12.72, 12.79, 13.78, 13.23, 12.88, 13.17, 13.84, 12.45, 
        14.34, 13.48, 13.2, 13.69, 12.85, 12.96, 13.78, 13.73, 
        13.82, 13.3, 13.58, 13.4, 11.03, 12.77, 14.16, 13.73, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 1.67, 1.67, 
        1.95, 2.39, 1.6, 1.87, 2.15, 0.99, 1.35, 2.16, 1.48, 
        0.99, 1.73, 1.87, 1.92, 1.92, 1.83, 1.35, 3.1, 1.63, 
        3.8, 2.16, 1.6, 1.81, 0.94, 1.67, 1.65, 1.9, 1.29, 1.5, 
        1.65, 1.83, 1.53, 3.55, 1.29, 1.5, 1.65, 1.5, 0.98, 1.71, 
        1.35, 4.3, 3.98, 1.65, 4.04, 3.37, 1.68, 1.65, 1.97, 
        0.9, 1.65, 2.39, 1.9, 1.67, 2.55, 1.7, 1.97, 4.31, 0.94, 
        3.87, 1.36, 1.25, 1.17, 3.37, 1.21, 1.01, 1.17, 1.72, 
        1.51, 2.4, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.8, 0.99, 
        3.87, 2.16, 0.99, 1.13, 1.09, 4.3, 0.98, 1.61, 1.67, 
        3.74, 1.5, 1.87, 1.51, 0.99, 2.16, 1.29, 1.01, 3.03, 
        1.41, 1.48, 1.87, 2.08, 1.36, 3.59, 1.72, 1.29, 2.55, 
        1.73, 2.15, 1.29, 1.13, 3.74, 1.6, 2.68, 0.74, 1.39, 
        1.01, 4.3, 1.61, 3.43, 0.94, 2.4, 2.05, 4.43, 3.45, 4.31, 
        2.16, 1.53, 2.13, 1.53, 4.3, 1.35, 2.99, 2.31, 3.55, 
        1.24, 3.91, 4.04, 5.51, 3.59, 3.88, 2.81, 2.56, 4.61, 
        4.61, 3.88, 4.1, 5.04, 4.61, 3.24, 3.45, 3.12, 2.67, 
        1.9, 1.9, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 3.24, 
        3.03, 3.45, 2.76, 1.9, 3.91, 3.37, 2.58, 4.6, 4.43, 2.39, 
        1.51, 5.65, 3.91, 4.12, 2.59, 4.1), ash = c(2.43, 2.14, 
        2.67, 2.5, 2.36, 2.61, 2.45, 2.61, 2.24, 2.43, 2.3, 2.17, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.7, 2.65, 
        2.35, 2.52, 2.61, 3.22, 2.35, 2.35, 2.39, 2.36, 2.35, 
        2.36, 2.29, 2.7, 2.65, 2.28, 2.5, 2.55, 2.1, 2.51, 2.32, 
        2.12, 2.59, 2.29, 2.1, 2.44, 2.28, 2.12, 2.4, 2.27, 2.04, 
        2.7, 2.38, 2.6, 1.98, 2.69, 2.3, 2.68, 2.5, 1.36, 2.69, 
        2.13, 1.92, 2.16, 2.53, 2.56, 1.98, 1.92, 2.42, 1.75, 
        2.21, 2.67, 2.24, 2.6, 2.3, 1.94, 1.71, 2.21, 2.3, 2.4, 
        2, 2.3, 2.51, 2.32, 2.58, 2.24, 2, 2.62, 2.62, 2.3, 2.32, 
        2.16, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 1.71, 2, 1.7, 
        2.28, 2.46, 1.88, 2.56, 2.27, 2.12, 2.28, 1.94, 2.7, 
        1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.22, 2, 2, 2.42, 
        3.23, 2.51, 2.13, 2.39, 2.17, 2.29, 2.26, 2.6, 2.38, 
        2.61, 2.4, 2.4, 2.36, 2.25, 2.27, 2.54, 2.64, 2.19, 2.61, 
        2.7, 2.35, 2.72, 2.35, 2.2, 2.12, 2.29, 2.48, 2.45, 2.36, 
        2.62, 2.48, 2, 2.28, 2.3, 2.32, 2.38, 2.58, 2.61, 2.64, 
        2.38, 2.54, 2.38, 2.35, 2.3, 2.2, 2.6, 2.37, 2.69, 2.64, 
        2.32, 2.28, 2.5, 2.62, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 
        16.8, 25, 16.8, 21, 15.2, 16.3, 17.6, 14, 16, 18, 16.8, 
        16, 11.4, 16.8, 16.2, 15.5, 20, 12.4, 15.2, 16, 18.6, 
        15.6, 17.8, 20, 23, 16.1, 17, 18, 13.2, 19.1, 19.1, 17.5, 
        16.3, 19, 20.5, 15.5, 21, 15.5, 13.2, 16.2, 18.8, 15.5, 
        18.1, 17, 20, 16, 16.8, 18.8, 17.4, 12.4, 20, 17, 19, 
        10.6, 20.5, 16.3, 16.8, 16.2, 10.6, 25, 16.8, 18, 19, 
        19, 18.1, 19.1, 19.6, 13.2, 16.8, 20.4, 19, 24, 30, 21, 
        20, 16, 18, 14.8, 22, 19, 21, 24, 22.5, 18, 18, 22.8, 
        21.6, 21.6, 20, 24, 22, 23, 18, 18, 19, 21.5, 16, 10.6, 
        18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 
        19.5, 21, 23.5, 19.5, 18.6, 21.5, 20.8, 19.5, 16, 18.5, 
        22, 19.5, 22, 21.5, 21, 21, 21.5, 28.5, 25, 22, 18, 20, 
        24, 21.5, 17.5, 19, 21.5, 25, 19.5, 25, 21, 21, 23.5, 
        20, 22, 21, 20, 20, 19, 21.5, 24, 22, 18, 18.5, 20, 22, 
        19.5, 14.8, 25, 25, 19, 19, 18.5, 18.5, 16.8, 22.5, 23, 
        19.5, 24.5, 25, 22, 19.5, 21, 20.5, 23, 20, 17, 24.5), 
            mg = c(127, 100, 101, 113, 111, 112, 96, 121, 97, 
            139, 112, 102, 89, 91, 102, 112, 120, 102, 105, 116, 
            126, 102, 100, 95, 91, 124, 93, 101, 107, 96, 101, 
            106, 104, 106, 110, 99, 110, 100, 96, 123, 117, 86, 
            101, 102, 107, 111, 102, 101, 112, 108, 92, 106, 
            111, 115, 99, 106, 118, 102, 108, 88, 101, 100, 94, 
            87, 104, 98, 78, 78, 81, 151, 96, 86, 87, 139, 101, 
            94, 86, 112, 111, 112, 86, 88, 78, 85, 110, 99, 90, 
            88, 84, 70, 81, 78, 85, 88, 95, 103, 134, 85, 88, 
            97, 97, 88, 101, 86, 85, 90, 80, 84, 92, 94, 107, 
            88, 103, 88, 84, 85, 86, 97, 101, 87, 96, 107, 102, 
            86, 116, 85, 98, 92, 88, 80, 122, 102, 98, 106, 84, 
            94, 89, 98, 88, 101, 88, 89, 97, 96, 112, 102, 139, 
            92, 92, 113, 123, 105, 116, 98, 103, 102, 89, 97, 
            98, 89, 102, 97, 106, 106, 90, 88, 106, 88, 105, 
            112, 96, 86, 91, 95, 102, 116, 120, 98), phe = c(2.8, 
            2.65, 2.8, 3.85, 2.8, 3.1, 2.5, 2.6, 2.95, 2.98, 
            2.95, 2.2, 2.6, 3.1, 3.3, 2.85, 2.8, 2.95, 3.3, 3.15, 
            3, 2.41, 2.61, 2.48, 2.53, 1.65, 3, 2.4, 2.95, 2.65, 
            3, 2.95, 2.42, 2.95, 2.85, 2.7, 2.6, 2.45, 2.95, 
            2.8, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
            2.75, 2.88, 2.72, 2.45, 3.88, 2.53, 2.95, 2.96, 3.2, 
            3, 3.4, 1.98, 2.05, 2.02, 1.6, 3.5, 1.89, 2.45, 2.48, 
            2.11, 2.53, 3.88, 1.1, 2.95, 1.88, 2.05, 2.2, 1.88, 
            2.41, 2.6, 1.9, 2.83, 2.6, 2.2, 1.7, 1.65, 2.22, 
            2.2, 1.9, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
            3.02, 2.48, 1.1, 2.55, 3.52, 2.74, 2.56, 1.48, 2.56, 
            2.5, 2.2, 1.8, 1.65, 2.1, 2.36, 2.74, 3.18, 2.55, 
            1.75, 2.11, 2.56, 2.46, 1.89, 2, 2.56, 2, 2.9, 3.18, 
            1.65, 1.51, 2.86, 2.55, 2.74, 2.13, 2.22, 2.1, 1.51, 
            1.7, 1.62, 1.7, 1.45, 1.62, 1.38, 1.48, 1.62, 1.39, 
            1.39, 1.4, 1.25, 2, 1.38, 1.4, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.4, 1.48, 1.93, 1.8, 1.9, 1.93, 
            1.7, 2.3, 1.83, 1.62, 1.39, 1.35, 1.4, 1.7, 1.48, 
            1.55, 1.68, 1.25, 1.39, 1.68, 2, 1.41, 1.62, 1.65, 
            2.05), fla = c(3.19, 2.76, 3.54, 3.49, 2.65, 3.39, 
            2.52, 2.51, 2.98, 3.32, 3.32, 2.43, 2.76, 2.79, 3.64, 
            2.91, 3.14, 3.4, 3.39, 3.39, 3.49, 2.41, 2.88, 2.37, 
            2.61, 2.68, 2.94, 2.76, 2.97, 2.33, 3.25, 3.19, 2.69, 
            3.4, 2.94, 2.98, 2.51, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.17, 2.92, 3.54, 2.24, 
            2.99, 3.74, 2.79, 2.9, 2.78, 3, 3.23, 3.67, 0.57, 
            1.61, 1.41, 1.79, 3.64, 1.75, 2.65, 2.43, 2, 1.3, 
            3.74, 1.02, 2.86, 2.03, 2.89, 2.14, 1.57, 2.01, 2.43, 
            1.85, 2.76, 2.26, 2.53, 1.58, 1.59, 2.21, 2, 1.69, 
            1.61, 1.69, 1.69, 1.69, 1.25, 1.46, 2.65, 3.56, 2.27, 
            0.99, 2.43, 3.17, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 
            1.84, 2.03, 1.76, 1.61, 2.92, 2.92, 2.27, 2.03, 2.01, 
            2.29, 2.17, 1.6, 2.09, 1.85, 1.64, 2.99, 3.64, 1.79, 
            1.02, 3.03, 2.65, 3.15, 2.24, 2.43, 1.75, 1.69, 0.63, 
            1.36, 1.2, 0.58, 0.66, 0.47, 0.58, 0.68, 0.6, 0.5, 
            0.83, 0.52, 0.8, 0.7, 0.55, 0.34, 0.65, 0.76, 1.39, 
            1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.47, 0.58, 1.31, 
            1.1, 0.92, 1.1, 0.6, 1.39, 0.68, 0.47, 0.92, 0.66, 
            0.84, 0.96, 0.49, 0.68, 0.7, 0.61, 0.57, 0.69, 0.68, 
            0.76), nfla = c(0.28, 0.27, 0.3, 0.24, 0.39, 0.3, 
            0.4, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 
            0.3, 0.39, 0.21, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 
            0.26, 0.2, 0.34, 0.27, 0.34, 0.26, 0.29, 0.29, 0.42, 
            0.29, 0.29, 0.26, 0.31, 0.26, 0.28, 0.2, 0.34, 0.2, 
            0.29, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 
            0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.43, 0.29, 0.28, 
            0.43, 0.53, 0.32, 0.32, 0.45, 0.37, 0.26, 0.27, 0.43, 
            0.17, 0.4, 0.4, 0.32, 0.21, 0.13, 0.34, 0.24, 0.43, 
            0.35, 0.43, 0.3, 0.29, 0.4, 0.61, 0.22, 0.3, 0.39, 
            0.43, 0.43, 0.41, 0.45, 0.5, 0.26, 0.25, 0.17, 0.32, 
            0.5, 0.3, 0.24, 0.26, 0.26, 0.29, 0.34, 0.37, 0.43, 
            0.66, 0.2, 0.48, 0.39, 0.29, 0.24, 0.26, 0.4, 0.42, 
            0.43, 0.52, 0.3, 0.43, 0.27, 0.34, 0.32, 0.43, 0.43, 
            0.3, 0.29, 0.24, 0.39, 0.34, 0.4, 0.58, 0.17, 0.43, 
            0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 
            0.53, 0.61, 0.45, 0.43, 0.43, 0.4, 0.47, 0.45, 0.34, 
            0.22, 0.48, 0.26, 0.61, 0.53, 0.53, 0.53, 0.63, 0.53, 
            0.52, 0.6, 0.5, 0.6, 0.4, 0.41, 0.48, 0.43, 0.4, 
            0.39, 0.43, 0.3, 0.48, 0.39, 0.53, 0.43, 0.43, 0.53, 
            0.52), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 
            1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.34, 2.96, 
            1.46, 2.81, 1.86, 1.86, 1.66, 2.1, 1.95, 1.69, 1.46, 
            1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.91, 
            1.85, 1.54, 1.86, 1.36, 1.44, 1.37, 1.04, 2.29, 1.35, 
            1.81, 1.95, 2.03, 1.76, 2.19, 2.14, 2.38, 2.08, 0.64, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.38, 1.66, 1.69, 0.42, 
            2.76, 1.66, 0.73, 1.87, 1.03, 2.45, 1.15, 1.04, 0.42, 
            2.5, 1.46, 1.87, 1.03, 1.87, 1.54, 1.15, 1.46, 1.95, 
            2.76, 1.95, 1.43, 1.42, 0.83, 1.62, 1.04, 1.46, 1.56, 
            1.34, 1.35, 1.38, 1.64, 1.63, 1.15, 1.99, 1.35, 3.28, 
            1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.56, 
            1.42, 1.46, 1.63, 2.08, 2.49, 1.45, 1.34, 1.92, 2.29, 
            1.04, 2.49, 1.14, 1.61, 0.83, 1.87, 1.83, 2.18, 1.71, 
            0.84, 2.91, 1.35, 1.77, 1.69, 1.9, 1.41, 0.94, 1.35, 
            0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.63, 0.75, 
            0.94, 0.55, 1.1, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 
            1.25, 1.26, 1.56, 1.15, 1.99, 1.55, 1.56, 1.14, 2.7, 
            2.29, 1.04, 0.73, 1.54, 0.94, 0.86, 1.15, 1.46, 0.97, 
            1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 
            1.35), col = c(5.64, 4.38, 5.68, 7.8, 5.43, 6.75, 
            7.8, 5.05, 5.3, 7.22, 5.75, 6.75, 5.6, 3.95, 7.5, 
            8.21, 7.7, 2.95, 8.7, 5.1, 5.65, 4.5, 4.36, 3.93, 
            3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.75, 5.45, 
            5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 
            4.36, 5.04, 5.24, 4.9, 6.1, 7.65, 8.9, 7.2, 7.22, 
            5.04, 6.3, 5.85, 6.25, 7.8, 5.1, 6.8, 5.68, 3.27, 
            5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 
            5.7, 3.38, 3.74, 5.7, 3.21, 3.8, 4.6, 3.27, 10.52, 
            3.05, 2.5, 3.9, 2.15, 4.8, 3.05, 2.62, 2.45, 2.6, 
            2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 5.7, 
            2.9, 2.45, 4.6, 2.7, 5.1, 2.8, 2.9, 2.94, 2.7, 3.4, 
            3.3, 2.7, 3.25, 2.9, 2.85, 2.85, 3.08, 2.9, 1.9, 
            2.7, 3.95, 3.4, 1.28, 3.25, 2.6, 1.28, 5.64, 2.8, 
            4.6, 3.94, 3.9, 2.12, 2.62, 4.35, 7.5, 7.22, 5, 5.45, 
            7.1, 4.28, 5, 5.7, 4.92, 5.1, 7.3, 4.35, 4.4, 8.21, 
            4, 4.9, 5.85, 8.42, 9.4, 4.6, 9.7, 7.1, 10.52, 7.6, 
            7.9, 8.42, 7.5, 13, 7.65, 7.65, 7.5, 5.58, 7.6, 7.7, 
            6.62, 10.68, 10.26, 7.1, 8.5, 3.25, 4, 9.7, 7.7, 
            7.3, 9.4, 9.3, 9.2), hue = c(1.04, 1.05, 1.1, 0.86, 
            0.69, 1.05, 1.02, 1.12, 1.08, 1.23, 1.25, 0.96, 1.15, 
            1.25, 1.2, 1.28, 1.23, 1.13, 1.23, 1.07, 1.09, 1.12, 
            1.11, 1.09, 1.12, 1.13, 1.23, 1.02, 1.13, 1.04, 1.19, 
            1.09, 0.69, 1.25, 1.1, 1.25, 1.22, 1.15, 1.18, 1.25, 
            0.95, 0.91, 0.88, 0.82, 0.86, 0.87, 0.87, 1.27, 1.07, 
            1.12, 1.12, 1.31, 1.13, 1.06, 1.05, 0.97, 0.94, 1.07, 
            0.89, 0.75, 1.25, 0.98, 1.23, 1.22, 1.28, 1.09, 0.84, 
            1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 1.22, 
            1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 1.19, 1.31, 0.84, 
            0.95, 1.23, 0.86, 1.05, 0.7, 1.07, 1.08, 1.05, 1.22, 
            1.15, 1.2, 1.19, 0.78, 1.08, 1.19, 1.42, 1.27, 0.8, 
            0.8, 1.19, 1.04, 1.04, 1, 0.84, 0.86, 1.22, 0.9, 
            0.9, 1.12, 1.1, 0.93, 1.71, 0.95, 1.25, 0.7, 1.71, 
            0.8, 1.28, 0.92, 0.73, 1.31, 0.86, 0.69, 0.97, 0.89, 
            0.79, 0.92, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
            0.81, 0.78, 0.74, 0.73, 0.89, 0.66, 0.65, 0.6, 0.57, 
            0.54, 0.55, 0.48, 0.89, 0.48, 0.61, 0.56, 0.61, 0.6, 
            0.57, 0.67, 0.57, 0.89, 0.56, 0.96, 0.87, 0.68, 0.7, 
            0.78, 0.85, 0.57, 0.74, 0.67, 0.73, 0.89, 0.67, 0.64, 
            0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 
            3.19, 2.93, 2.78, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
            2.9, 2.87, 3, 3.3, 2.65, 3.69, 2.82, 3.36, 3.71, 
            3.52, 4, 3.63, 2.71, 3.2, 2.93, 2.77, 2.87, 3.17, 
            2.71, 2.88, 2.87, 3, 2.87, 3.5, 2.42, 2.42, 2.69, 
            3.58, 3.38, 3, 3.38, 3, 3.35, 3.33, 3.26, 3.33, 2.75, 
            3.1, 3.13, 3.21, 3.26, 2.93, 3.2, 3.03, 3.21, 2.84, 
            2.87, 1.82, 3.3, 1.59, 2.46, 2.87, 3.02, 2.3, 3.18, 
            3.48, 2.75, 3, 1.82, 3, 2.96, 3.5, 3.13, 2.14, 2.48, 
            2.52, 1.82, 2.96, 2.84, 3.14, 2.72, 2.01, 3.13, 2.52, 
            3.13, 3.21, 2.75, 3.21, 2.85, 2.65, 2.87, 3.3, 3, 
            2.63, 1.33, 2.74, 3.71, 2.83, 2.96, 2.77, 3.38, 2.87, 
            3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 
            2.5, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 
            3.12, 2.46, 3.64, 2.78, 2.84, 3.3, 2.78, 3.48, 3.33, 
            1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 
            1.82, 1.62, 1.82, 2.05, 2, 1.78, 1.33, 1.29, 1.62, 
            1.33, 3.21, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 
            1.96, 1.78, 1.58, 1.78, 1.64, 1.75, 1.68, 1.75, 1.6, 
            1.82, 1.75, 1.96, 1.59, 2.01, 1.6, 1.59, 1.75, 1.56, 
            1.62, 1.6), prol = c(1065, 1050, 1285, 1095, 735, 
            1450, 1290, 1295, 1045, 1045, 1060, 1280, 1320, 1290, 
            1547, 1310, 1280, 1045, 1680, 845, 1060, 770, 1035, 
            1015, 845, 480, 1195, 1285, 915, 1035, 1285, 1045, 
            985, 1235, 1285, 920, 880, 463, 660, 760, 795, 1035, 
            1095, 680, 885, 1080, 1065, 985, 1060, 1235, 1150, 
            1265, 1190, 885, 1060, 1120, 970, 1270, 1285, 580, 
            718, 463, 630, 880, 406, 795, 502, 680, 735, 718, 
            870, 410, 472, 560, 886, 312, 392, 735, 1045, 463, 
            278, 714, 830, 515, 550, 428, 415, 562, 680, 625, 
            480, 450, 495, 1035, 345, 438, 625, 428, 660, 406, 
            515, 562, 438, 415, 672, 630, 510, 392, 312, 680, 
            562, 325, 740, 434, 385, 312, 720, 463, 372, 564, 
            1035, 1450, 365, 380, 735, 378, 880, 466, 550, 580, 
            640, 570, 560, 600, 510, 695, 720, 515, 580, 590, 
            600, 425, 520, 550, 625, 830, 415, 365, 480, 550, 
            500, 480, 425, 675, 640, 640, 480, 880, 660, 480, 
            550, 680, 570, 845, 615, 495, 695, 685, 625, 463, 
            510, 470, 345, 740, 520, 835, 840, 463)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 13.76, 13.24, 13.76, 14.39, 14.06, 12.25, 
        13.86, 13.05, 14.21, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.3, 13.64, 13.83, 12.93, 13.71, 12.85, 13.5, 
        13.74, 13.39, 13.3, 13.87, 12.37, 13.73, 13.58, 13.68, 
        13.76, 13.32, 12.25, 13.05, 13.05, 12.86, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 12.45, 13.9, 13.17, 
        13.16, 13.05, 14.13, 13.82, 12.7, 13.74, 13.16, 14.22, 
        13.29, 14.22, 12.37, 12.33, 12.64, 11.79, 12.37, 12.17, 
        12.37, 13.78, 12.6, 13.34, 12.21, 12.16, 13.86, 11.76, 
        12.99, 11.96, 11.66, 12.16, 11.84, 14.16, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 12.16, 12.16, 12.22, 12.33, 12.08, 
        12.08, 12, 12.69, 12.08, 11.62, 12.22, 11.81, 12.29, 
        11.81, 12.29, 12.08, 12.6, 12.37, 12.53, 12.51, 12.93, 
        12.6, 12.72, 12.22, 11.61, 12, 12.29, 11.76, 12.22, 12.08, 
        11.03, 11.82, 11.84, 12.29, 11.76, 11.45, 14.75, 12.42, 
        13.05, 11.87, 12.07, 12.51, 11.79, 12.08, 12.04, 12.86, 
        12.88, 12.08, 12.7, 12.51, 12.86, 12.25, 12.53, 12.77, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.29, 12.04, 
        12.87, 13.32, 12.51, 12.72, 12.79, 13.86, 13.23, 14.39, 
        13.17, 13.84, 12.45, 14.34, 13.48, 13.3, 13.69, 12.85, 
        12.96, 13.78, 13.73, 12.45, 13.74, 13.58, 13.4, 12.72, 
        12.77, 14.16, 13.71, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
        1.87, 3.74, 1.95, 1.17, 3.55, 1.87, 2.15, 1.68, 1.35, 
        2.16, 1.48, 1.67, 1.73, 1.87, 2.08, 1.92, 1.41, 1.87, 
        3.1, 1.63, 3.8, 1.73, 1.6, 1.81, 1.67, 1.73, 1.9, 1.9, 
        1.53, 1.5, 1.01, 1.83, 1.53, 1.67, 2.31, 1.83, 1.65, 
        1.5, 1.5, 1.71, 1.39, 1.51, 3.98, 2.68, 4.04, 1.51, 1.68, 
        1.09, 1.35, 1.67, 1.65, 0.74, 1.9, 1.67, 3.37, 1.7, 1.97, 
        1.51, 0.94, 1.97, 1.36, 1.25, 1.13, 2.4, 1.21, 1.01, 
        1.17, 1.68, 2.55, 1.13, 1.51, 1.66, 1.67, 1.09, 1.88, 
        0.9, 1.53, 0.99, 3.87, 1.73, 1.9, 1.13, 3.55, 1.92, 0.98, 
        1.61, 1.67, 1.17, 1.67, 1.73, 1.51, 1.17, 1.01, 1.35, 
        1.73, 1.29, 1.41, 1.71, 1.53, 2.08, 1.87, 2.68, 1.72, 
        3.87, 2.55, 1.73, 1.71, 1.29, 3.98, 3.74, 1.68, 2.68, 
        0.74, 1.39, 1.66, 1.68, 1.61, 3.43, 1.13, 2.4, 2.05, 
        4.43, 1.39, 4.31, 2.16, 1.53, 2.13, 1.67, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 2.15, 3.88, 5.51, 3.59, 5.19, 
        2.81, 2.56, 3.8, 1.24, 3.88, 3.24, 5.04, 4.61, 3.24, 
        1.24, 3.12, 2.67, 1.9, 1.35, 1.29, 5.19, 4.12, 3.03, 
        1.68, 1.67, 3.83, 1.13, 3.83, 3.45, 2.76, 3.74, 2.99, 
        3.37, 2.58, 4.6, 1.24, 2.39, 2.31, 5.65, 3.91, 3.74, 
        2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.4, 2.7, 
        2.45, 2.61, 2.3, 2.55, 2.3, 2.62, 2.41, 2.39, 2.38, 2.7, 
        2.72, 2.62, 2.48, 2.56, 2.51, 2.65, 2.36, 2.52, 2.61, 
        3.22, 2, 2.39, 2.68, 1.82, 2.48, 2.36, 2.67, 2.7, 2.65, 
        2.38, 2.72, 2.55, 2.1, 2.51, 1.92, 2.12, 2.59, 2.29, 
        2.1, 2.44, 2.17, 2.12, 2.4, 2.27, 2.04, 2.72, 2.4, 2.58, 
        2, 2.65, 2.3, 2.68, 2.5, 1.36, 2.42, 2.37, 1.92, 2.16, 
        2.53, 2.56, 2.55, 1.92, 2.3, 1.75, 2.21, 2.67, 2.24, 
        2.6, 2.3, 1.98, 1.71, 1.71, 2.29, 2.4, 2, 2.38, 2.51, 
        2.32, 2.58, 2.24, 2.28, 2.62, 2.74, 2.3, 2.32, 2.7, 2.26, 
        2.22, 2.28, 2.2, 2.74, 1.98, 2.2, 2.12, 1.7, 2.2, 2.46, 
        1.88, 2.61, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 
        2.92, 2.5, 2.5, 2.2, 1.99, 2.68, 2.35, 2, 2.42, 3.23, 
        3.23, 2.13, 2.39, 2.17, 2.29, 2.67, 2.61, 2.38, 2.22, 
        2.4, 2.4, 2.36, 2.25, 2.48, 2.54, 2.64, 2.19, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.65, 2.32, 2.48, 2.7, 2.36, 2.62, 
        2.48, 2.6, 2.28, 2.38, 2.32, 2.38, 2.56, 2.6, 2.64, 2.38, 
        2.54, 2.37, 2.35, 2.3, 2.61, 2.6, 2.54, 2.69, 2.72, 2.32, 
        2.28, 2.48, 2.7, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 
        16, 21.5, 16.8, 21, 15.2, 15.6, 17.6, 14, 16, 14.8, 16.8, 
        16, 11.4, 16.8, 20.8, 20, 20, 19.5, 15.2, 16, 18.6, 18.8, 
        17.8, 20, 24, 16.1, 17, 17, 19.5, 16, 19.1, 20, 20, 19, 
        20.5, 15.5, 20, 15.5, 13.2, 16.2, 18.8, 15.5, 19, 17, 
        15.2, 16, 16, 18.8, 17.4, 12.4, 21, 16.1, 19.5, 12.4, 
        20.5, 16.3, 16.8, 16.8, 10.6, 18, 16.8, 18, 19, 19, 18.1, 
        19, 19.6, 21.5, 16.8, 20.4, 18.1, 24, 30, 21, 19.6, 16, 
        18, 14.8, 25, 19, 19.5, 24, 22.5, 18, 18, 22.8, 20, 21.6, 
        20, 20, 22, 19.5, 18, 18, 19, 21.5, 16, 16, 18, 17.5, 
        18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 19.5, 
        20, 23.5, 18.5, 21.5, 20.8, 28.5, 16, 24.5, 19.5, 20.5, 
        24, 21.5, 21, 21, 21.5, 28.5, 22, 22, 18, 20, 24, 21.5, 
        19.5, 21, 22, 25, 19.5, 24, 21, 20.5, 23.5, 20, 28.5, 
        21, 20, 25, 28.5, 21.5, 24, 22, 20.5, 18.5, 20, 22, 19.5, 
        23, 25, 22, 20, 19.5, 22.5, 18.5, 20, 22.5, 23, 19.5, 
        24.5, 25, 25, 19.5, 18.6, 20.5, 23, 20, 22.5, 24.5), 
            mg = c(127, 100, 101, 113, 92, 112, 96, 121, 97, 
            97, 88, 120, 89, 91, 102, 112, 120, 100, 120, 116, 
            126, 102, 98, 95, 84, 124, 93, 106, 107, 96, 101, 
            106, 104, 110, 110, 88, 110, 101, 100, 102, 117, 
            100, 101, 102, 107, 111, 102, 101, 117, 108, 92, 
            134, 111, 115, 96, 112, 118, 102, 108, 88, 101, 100, 
            94, 87, 104, 98, 78, 78, 89, 151, 88, 86, 87, 139, 
            101, 89, 88, 112, 88, 105, 86, 89, 78, 85, 104, 99, 
            90, 88, 84, 70, 81, 80, 88, 88, 102, 88, 134, 85, 
            88, 86, 97, 88, 88, 86, 85, 90, 80, 84, 92, 94, 107, 
            88, 103, 88, 84, 85, 86, 97, 70, 87, 96, 115, 102, 
            86, 85, 85, 101, 92, 88, 80, 122, 89, 98, 106, 80, 
            94, 89, 110, 88, 101, 105, 89, 97, 100, 112, 102, 
            103, 106, 92, 113, 123, 96, 116, 98, 103, 87, 89, 
            97, 98, 89, 98, 97, 106, 106, 90, 88, 85, 88, 105, 
            112, 96, 86, 91, 95, 102, 81, 120, 100), phe = c(2.8, 
            2.65, 2.8, 3.85, 2.8, 2.95, 2.5, 2.6, 3.1, 2.98, 
            2.95, 2.2, 2.6, 3.1, 3.52, 2.85, 2.8, 3.18, 3.3, 
            3.1, 3, 2.41, 2.61, 2.48, 2.53, 2.46, 2.86, 2.4, 
            2.95, 2.65, 3, 2.55, 2.42, 2.95, 2.6, 2.7, 2.6, 2.45, 
            2.42, 2.6, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 
            3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.46, 2.8, 2.96, 
            3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.2, 3.5, 1.89, 2.74, 
            2.8, 2.11, 2.53, 3.52, 1.1, 2.95, 1.88, 2.02, 1.65, 
            1.9, 2.95, 1.92, 1.9, 2.83, 2.55, 2.2, 1.4, 1.65, 
            2.02, 2.2, 2.22, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 
            2.45, 3.02, 1.65, 1.51, 2.55, 3.52, 2.85, 2.74, 1.35, 
            2.56, 2.5, 2.2, 2.72, 1.65, 1.83, 2.36, 2.74, 3.18, 
            2.55, 1.75, 2.56, 2.56, 2.46, 1.75, 2, 1.1, 2, 2.9, 
            3.18, 2.53, 0.98, 2.86, 2.6, 2.74, 2.13, 2.22, 2.1, 
            1.51, 1.9, 1.38, 1.7, 1.35, 1.62, 1.38, 1.62, 1.62, 
            1.39, 1.25, 1.4, 1.38, 2, 1.38, 0.98, 0.98, 1.7, 
            1.93, 1.41, 1.4, 1.48, 2.2, 1.35, 1.48, 2.2, 1.8, 
            1.9, 1.7, 1.9, 2.3, 1.83, 1.4, 1.39, 1.35, 1.55, 
            1.7, 1.48, 1.55, 2, 1.25, 1.39, 1.68, 1.35, 1.65, 
            1.8, 1.65, 2.05), fla = c(3.56, 2.76, 2.52, 3.49, 
            2.27, 3.39, 2.52, 2.51, 2.98, 3.32, 3.32, 2.43, 2.76, 
            3.32, 3.64, 2.91, 3.14, 3.4, 3.25, 3.54, 3.49, 2.41, 
            2.88, 2.37, 2.61, 2.68, 2.94, 2.27, 2.97, 2.33, 3.25, 
            3.19, 2.69, 3.67, 2.98, 2.98, 2.27, 2.43, 2.64, 3.04, 
            3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 3.25, 2.92, 
            3.54, 2.17, 2.99, 3.74, 2.79, 2.9, 2.78, 3.56, 3.23, 
            3.67, 0.57, 2.03, 1.41, 1.79, 2.78, 1.75, 2.65, 2.88, 
            2, 1.3, 3.67, 1.02, 2.86, 1.57, 2.89, 2.14, 1.57, 
            2.43, 1.69, 1.85, 2.53, 2.26, 2.53, 1.58, 1.59, 2.21, 
            2.21, 1.69, 1.61, 1.69, 2.64, 1.75, 1.25, 1.46, 2.76, 
            2.52, 2.27, 0.99, 2.43, 3.67, 2.99, 2.17, 1.36, 2.11, 
            1.64, 1.92, 1.84, 2.03, 1.76, 1.84, 2.92, 2.79, 2.27, 
            2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 0.68, 1.64, 2.89, 
            3.74, 2.01, 0.61, 3.03, 2.65, 3.15, 2.24, 2.01, 1.75, 
            1.41, 0.92, 0.63, 1.2, 0.58, 0.66, 0.47, 0.58, 0.47, 
            0.6, 0.5, 0.47, 0.52, 0.8, 0.6, 0.55, 0.34, 0.65, 
            0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.52, 
            0.58, 1.31, 1.1, 0.92, 1.76, 0.6, 1.39, 0.68, 0.47, 
            0.92, 0.66, 0.84, 0.96, 0.49, 0.61, 0.7, 0.61, 0.52, 
            0.69, 0.68, 0.76), nfla = c(0.28, 0.24, 0.3, 0.24, 
            0.39, 0.21, 0.4, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 
            0.43, 0.29, 0.3, 0.22, 0.28, 0.32, 0.17, 0.24, 0.25, 
            0.27, 0.26, 0.4, 0.43, 0.34, 0.27, 0.34, 0.26, 0.29, 
            0.22, 0.42, 0.28, 0.29, 0.26, 0.37, 0.17, 0.28, 0.2, 
            0.34, 0.24, 0.24, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.24, 
            0.21, 0.28, 0.2, 0.53, 0.32, 0.26, 0.45, 0.37, 0.26, 
            0.27, 0.53, 0.22, 0.43, 0.39, 0.6, 0.21, 0.13, 0.34, 
            0.24, 0.43, 0.35, 0.43, 0.3, 0.34, 0.4, 0.61, 0.22, 
            0.3, 0.5, 0.43, 0.53, 0.43, 0.4, 0.5, 0.52, 0.25, 
            0.17, 0.32, 0.34, 0.32, 0.24, 0.24, 0.27, 0.29, 0.34, 
            0.37, 0.41, 0.66, 0.26, 0.48, 0.39, 0.29, 0.24, 0.26, 
            0.26, 0.42, 0.4, 0.52, 0.3, 0.42, 0.53, 0.17, 0.32, 
            0.32, 0.43, 0.3, 0.24, 0.3, 0.39, 0.48, 0.4, 0.66, 
            0.28, 0.53, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 
            0.53, 0.53, 0.58, 0.63, 0.61, 0.52, 0.43, 0.4, 0.47, 
            0.45, 0.39, 0.22, 0.48, 0.26, 0.61, 0.53, 0.45, 0.43, 
            0.63, 0.53, 0.52, 0.58, 0.5, 0.6, 0.4, 0.41, 0.52, 
            0.43, 0.4, 0.39, 0.58, 0.53, 0.48, 0.6, 0.43, 0.43, 
            0.43, 0.53, 0.43), pro = c(2.29, 1.28, 2.81, 2.18, 
            1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 
            1.95, 2.96, 1.46, 1.87, 2.76, 1.86, 1.66, 2.1, 1.57, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 1.43, 1.87, 1.54, 1.86, 1.36, 1.44, 1.37, 1.92, 
            2.29, 1.87, 1.86, 2.03, 2.03, 1.56, 2.19, 2.14, 2.38, 
            2.08, 1.63, 2.29, 1.87, 1.68, 1.62, 2.45, 2.08, 1.66, 
            1.86, 0.42, 1.28, 1.14, 0.73, 1.87, 1.03, 1.99, 1.87, 
            1.04, 1.35, 2.5, 1.46, 1.87, 1.03, 1.87, 1.56, 1.15, 
            1.46, 1.35, 2.76, 1.95, 1.43, 1.98, 1.98, 1.62, 1.15, 
            1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 1.63, 0.73, 1.99, 
            1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 
            1.42, 1.46, 1.42, 1.4, 1.63, 2.08, 2.49, 3.28, 1.44, 
            1.42, 2.19, 1.04, 3.28, 1.35, 1.61, 0.83, 1.87, 1.83, 
            2.96, 1.71, 0.94, 2.91, 1.35, 1.77, 1.46, 1.9, 1.46, 
            0.94, 1.11, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 
            0.83, 0.75, 0.64, 0.55, 0.88, 1.14, 1.3, 0.68, 0.86, 
            1.25, 1.14, 1.25, 1.26, 1.56, 1.56, 0.83, 1.55, 1.56, 
            1.14, 2.7, 2.29, 1.04, 1.25, 1.25, 0.94, 1.46, 1.15, 
            1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 
            1.35, 1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 
            3.17, 6.75, 5.75, 5.05, 5.68, 7.22, 5.75, 5.4, 5.6, 
            5.65, 7.5, 5.3, 5.24, 3.94, 8.7, 5.1, 5.65, 4.5, 
            3.94, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 
            3.7, 5.85, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
            4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 6.62, 8.9, 
            7.2, 5.1, 5.64, 6.3, 5.85, 6.25, 7.7, 6.75, 6.8, 
            5.7, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 
            3.17, 2.85, 4.28, 3.38, 3.74, 5.7, 3.21, 3.8, 4.6, 
            6.25, 9.3, 3.25, 2.5, 3.9, 3.25, 4.8, 3.05, 2.62, 
            2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 
            2.6, 5.24, 2.9, 3.3, 2.8, 3.74, 3.9, 2.8, 5.6, 2.94, 
            2.7, 3.4, 3.3, 2.7, 3.4, 2.9, 4.5, 2.6, 3.08, 2.9, 
            1.9, 2.85, 1.28, 3.4, 1.28, 3.25, 4.5, 3.05, 5.64, 
            2.8, 3.38, 3.94, 2.85, 2.12, 5.58, 9.4, 5.64, 6.3, 
            5, 5.45, 7.1, 6.13, 5, 5.7, 4.92, 3.95, 4.8, 4.35, 
            4.4, 8.21, 4, 4.9, 6.3, 8.42, 9.4, 3.25, 6.25, 7.1, 
            10.52, 7.6, 7.9, 6.62, 7.5, 13, 6.75, 7.65, 5.1, 
            5.58, 4.28, 7.8, 6.62, 10.68, 10.26, 7.2, 8.5, 5.1, 
            5.6, 9.7, 7.7, 7.3, 9.4, 9.3, 9.2), hue = c(1.04, 
            1.05, 0.98, 0.86, 0.89, 1.05, 1.02, 0.87, 1.08, 0.98, 
            1.25, 1.05, 1.15, 1.25, 1.2, 1.28, 1.05, 1.13, 1.23, 
            0.84, 1.09, 0.92, 1.11, 1.09, 1.12, 1.13, 1.25, 1.02, 
            1.04, 1.04, 1.19, 1.09, 1.19, 1.25, 1.1, 1.25, 0.96, 
            0.8, 1.18, 0.94, 0.95, 0.91, 0.88, 0.82, 0.94, 0.87, 
            1.1, 0.93, 1.07, 1.12, 1.12, 1.36, 1.22, 0.98, 0.94, 
            0.98, 0.94, 1.07, 0.89, 0.56, 1.25, 0.98, 1.23, 1.22, 
            0.98, 1.08, 0.86, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 0.69, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 
            1.25, 1.31, 0.84, 0.98, 1.23, 1.25, 1.23, 1.13, 1.07, 
            1.08, 1.05, 1.05, 1.15, 1.19, 1.2, 0.84, 0.69, 1.07, 
            1.42, 1.27, 0.87, 0.8, 0.98, 1.04, 1, 1, 0.95, 0.86, 
            1.31, 1.25, 0.9, 0.9, 1.1, 0.93, 1.71, 0.95, 0.89, 
            0.7, 1.04, 0.8, 1.28, 0.92, 0.73, 1.1, 0.86, 0.69, 
            0.97, 0.89, 0.79, 0.82, 0.74, 0.66, 0.78, 0.75, 0.73, 
            0.75, 0.82, 0.81, 0.82, 0.81, 0.6, 0.89, 0.7, 0.65, 
            0.6, 0.61, 0.54, 0.55, 0.87, 0.67, 0.48, 0.61, 0.56, 
            0.85, 0.6, 0.57, 0.67, 0.57, 0.98, 0.56, 0.96, 0.87, 
            0.68, 0.7, 0.78, 0.85, 0.61, 0.74, 0.67, 0.74, 0.6, 
            0.56, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
            3.4, 3.17, 2.81, 2.93, 2.84, 3.58, 3.58, 2.85, 3.55, 
            3.17, 2.82, 2.9, 3.1, 3, 2.81, 2.65, 3, 2.82, 3.36, 
            3.71, 3.52, 4, 3.63, 2.87, 3.2, 3.18, 2.77, 3.33, 
            3.14, 2.71, 2.88, 2.87, 3, 2.87, 3.57, 3.02, 2.69, 
            2.69, 3.19, 3.38, 3, 4, 3, 3.35, 3.33, 3.2, 3.33, 
            2.75, 3.1, 2.65, 2.75, 3.26, 2.93, 3.2, 3.03, 3.5, 
            2.84, 2.87, 1.82, 2.78, 1.59, 2.46, 2.87, 3.13, 2.3, 
            3.18, 3.48, 2.14, 3.21, 1.82, 3.17, 2.48, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.77, 2.87, 3, 3.14, 2.72, 2.01, 
            3.17, 3.39, 2.87, 3.21, 2.75, 3.21, 3.58, 2.65, 2.42, 
            3.3, 3.39, 2.63, 1.82, 2.74, 3.38, 2.83, 2.96, 2.77, 
            3.38, 2.01, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 
            2.78, 2.5, 2.3, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 
            3.39, 3.69, 3.12, 1.69, 3.64, 2.71, 2.84, 3.33, 2.78, 
            2.85, 2, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
            1.82, 2.15, 2.05, 1.82, 1.75, 2.05, 2, 1.48, 1.33, 
            2, 1.62, 1.33, 2.01, 1.47, 1.33, 1.51, 1.55, 1.48, 
            1.64, 1.73, 1.96, 1.78, 1.58, 2.46, 1.48, 1.75, 1.68, 
            1.75, 1.55, 1.33, 1.56, 1.56, 1.58, 1.51, 1.33, 1.58, 
            1.33, 1.56, 1.62, 1.6), prol = c(1065, 1050, 640, 
            1280, 735, 1450, 1290, 1295, 1045, 1045, 1285, 1280, 
            1320, 845, 1547, 1310, 1280, 680, 1680, 845, 1045, 
            770, 1035, 1015, 845, 1235, 1195, 1285, 915, 1035, 
            1285, 1150, 1450, 1235, 1080, 920, 880, 562, 680, 
            760, 795, 1035, 1095, 680, 885, 1080, 1065, 985, 
            1060, 760, 1150, 1265, 1190, 1285, 1060, 1120, 970, 
            1270, 1285, 372, 425, 1060, 630, 570, 1015, 695, 
            502, 680, 372, 718, 870, 410, 472, 1270, 886, 560, 
            392, 365, 770, 463, 278, 714, 500, 515, 714, 714, 
            580, 562, 680, 625, 480, 450, 495, 570, 345, 463, 
            625, 428, 660, 406, 562, 562, 438, 415, 672, 550, 
            510, 415, 312, 680, 562, 325, 510, 434, 385, 835, 
            385, 428, 372, 564, 562, 1450, 365, 380, 580, 378, 
            502, 466, 480, 580, 695, 625, 560, 600, 625, 695, 
            720, 515, 580, 590, 600, 434, 520, 550, 480, 830, 
            415, 510, 278, 550, 500, 480, 425, 675, 640, 480, 
            480, 880, 660, 410, 278, 680, 570, 580, 615, 380, 
            695, 685, 795, 695, 510, 470, 680, 740, 515, 835, 
            840, 740)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 13.23, 13.24, 
        13.74, 14.39, 14.06, 13.86, 13.86, 12.37, 13.48, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 13.68, 13.64, 13.9, 
        12.93, 13.71, 12.85, 13.5, 14.75, 13.39, 13.3, 13.87, 
        13.64, 13.73, 13.58, 13.68, 13.76, 11.62, 14.06, 13.64, 
        13.05, 12.88, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 12.85, 13.9, 13.64, 13.76, 13.05, 13.82, 13.82, 
        13.86, 13.74, 13.64, 14.22, 13.29, 14.3, 12.37, 12.33, 
        12.64, 11.96, 12.37, 12.17, 12.37, 12, 11.76, 13.34, 
        12.21, 12.64, 13.86, 12.7, 12.99, 11.96, 11.66, 11.76, 
        11.84, 13.88, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.08, 
        12.16, 13.05, 11.82, 12.08, 12.08, 12, 12.69, 12.16, 
        11.62, 12.21, 11.81, 12.29, 12.08, 12.29, 12.08, 12.6, 
        11.84, 11.87, 12.51, 12.69, 11.45, 12.72, 12.22, 11.61, 
        11.79, 12.07, 11.76, 13.24, 12.08, 11.03, 11.82, 11.03, 
        12.93, 12.7, 11.45, 14.38, 12.42, 13.05, 11.87, 12.07, 
        13.84, 11.79, 11.76, 12.04, 12.86, 12.88, 12.25, 12.7, 
        12.51, 12.84, 12.25, 12.53, 12.87, 12.84, 12.93, 13.36, 
        13.52, 13.62, 12.25, 13.24, 12.93, 12.87, 13.32, 13.24, 
        12.69, 12.79, 12.99, 13.23, 13.24, 13.17, 13.84, 12.45, 
        14.34, 13.48, 12.21, 13.69, 12.85, 12.96, 13.78, 13.73, 
        13.41, 13.27, 13.58, 13.4, 13.24, 12.77, 14.16, 13.3, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 0.9, 1.35, 
        1.95, 4.12, 1.72, 1.87, 2.15, 1.73, 1.35, 2.16, 1.48, 
        2.55, 1.73, 1.87, 2.16, 1.92, 1.63, 2.55, 3.1, 1.63, 
        3.8, 1.5, 1.6, 1.81, 1.68, 1.51, 1.6, 1.9, 0.98, 1.5, 
        1.5, 1.83, 1.53, 1.41, 1.51, 1.71, 1.65, 1.5, 3.45, 1.71, 
        1.68, 4.31, 3.98, 1.71, 4.04, 1.9, 1.68, 2.08, 1.17, 
        1.9, 1.65, 1.65, 1.9, 1.67, 2.67, 1.7, 1.97, 1.9, 0.94, 
        1.51, 1.36, 1.25, 0.99, 1.51, 1.21, 1.01, 1.17, 1.61, 
        1.9, 1.73, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 0.99, 0.99, 
        3.87, 1.5, 1.6, 1.13, 1.29, 2.59, 0.98, 1.61, 1.67, 2.31, 
        3.59, 1.71, 1.51, 3.45, 1.97, 1.21, 2.56, 1.7, 1.41, 
        0.9, 1.17, 2.08, 0.74, 1.24, 1.72, 1.66, 2.55, 1.73, 
        2.4, 1.29, 1.88, 3.74, 1.41, 2.68, 0.74, 1.39, 1.5, 2.39, 
        1.61, 3.43, 3.59, 2.4, 2.05, 4.43, 1.92, 4.31, 2.16, 
        1.53, 2.13, 1.17, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 
        1.35, 5.65, 5.51, 3.59, 4.12, 2.81, 2.56, 4.6, 5.19, 
        3.88, 4.61, 5.04, 4.61, 3.24, 2.67, 3.12, 2.67, 1.9, 
        1.9, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 1.66, 
        1.83, 3.45, 2.76, 1.29, 3.59, 3.37, 2.58, 4.6, 2.56, 
        2.39, 1.7, 5.65, 3.91, 3.45, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.28, 2.67, 2.45, 2.61, 2.38, 2.19, 
        2.3, 2.48, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 
        2.69, 2.65, 2.32, 2.52, 2.61, 3.22, 2.21, 2.6, 2.32, 
        2, 2.56, 2.36, 2.62, 2.7, 2.65, 2.37, 2.35, 2.55, 2.1, 
        2.51, 2.38, 2.12, 2.59, 2.29, 2.1, 2.44, 2.64, 2.12, 
        2.4, 2.27, 2.04, 2.6, 2.27, 2.7, 2.42, 2.32, 2.3, 2.68, 
        2.5, 1.36, 2.65, 2.7, 1.92, 2.16, 2.53, 2.56, 1.71, 1.92, 
        2.5, 1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.51, 1.71, 2.3, 
        1.75, 2.4, 2, 2.7, 2.51, 2.32, 2.58, 2.24, 2.5, 2.62, 
        1.7, 2.3, 2.32, 2.3, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 
        2.64, 2.7, 1.7, 2.28, 2.46, 1.88, 2.26, 2.27, 2.12, 2.28, 
        1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2, 
        2, 2, 2.42, 3.23, 2.69, 2.13, 2.39, 2.17, 2.29, 2.12, 
        2.28, 2.38, 2.38, 2.4, 2.4, 2.36, 2.25, 2.24, 2.54, 2.64, 
        2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.27, 2.5, 2.48, 
        2.56, 2.36, 2.62, 2.48, 2.04, 2.28, 2.1, 2.32, 2.38, 
        2.6, 2.36, 2.64, 2.38, 2.54, 2.68, 2.35, 2.3, 2.53, 2.6, 
        2.1, 2.69, 2.44, 2.32, 2.28, 2.64, 2.25, 2.48, 2.26, 
        2.37, 2.74), alca = c(15.6, 16.1, 12.4, 16.8, 21, 15.2, 
        14.8, 17.6, 14, 16, 18, 16.8, 16, 11.4, 16.8, 17, 16.8, 
        20, 16.3, 15.2, 16, 18.6, 17.4, 17.8, 20, 18, 16.1, 17, 
        14.8, 10.6, 16.8, 19.1, 14.8, 16, 19, 20.5, 15.5, 18.8, 
        15.5, 13.2, 16.2, 18.8, 17.4, 21, 17, 19.1, 16, 20.5, 
        18.8, 17.4, 12.4, 21.5, 20.5, 16.8, 19, 20.5, 16.3, 16.8, 
        16.1, 10.6, 22, 16.8, 18, 19, 19, 18.1, 16.8, 19.6, 20, 
        16.8, 20.4, 22.5, 24, 30, 21, 24, 16, 18, 14.8, 23, 19, 
        28.5, 24, 22.5, 18, 18, 22.8, 22, 21.6, 22, 22, 22, 22.8, 
        18, 18, 19, 21.5, 16, 16.2, 18, 17.5, 18.5, 21, 19.5, 
        20.5, 22, 19, 22.5, 19, 20, 19.5, 21, 20.5, 18.8, 20, 
        21.5, 20.8, 18, 16, 21.5, 18, 19, 25, 21.5, 21, 21, 21.5, 
        28.5, 21, 22, 18, 20, 24, 21.5, 24, 21, 21.5, 25, 19.5, 
        22, 21, 30, 23.5, 20, 25, 21, 20, 22, 22.5, 21.5, 24, 
        22, 17.5, 18.5, 20, 22, 19.5, 24, 25, 22, 19.6, 19.5, 
        24.5, 18.5, 21.5, 22.5, 23, 19.5, 24.5, 25, 21, 19.5, 
        21.5, 20.5, 23, 20, 19.5, 24.5), mg = c(127, 100, 101, 
        113, 88, 112, 96, 121, 97, 104, 97, 126, 89, 91, 102, 
        112, 120, 112, 111, 116, 126, 102, 121, 95, 115, 124, 
        93, 95, 107, 96, 101, 106, 104, 120, 110, 97, 110, 88, 
        108, 92, 117, 106, 101, 85, 107, 111, 102, 101, 115, 
        108, 92, 112, 111, 115, 151, 102, 118, 102, 108, 88, 
        101, 100, 94, 87, 104, 98, 78, 78, 104, 151, 112, 86, 
        87, 139, 101, 92, 89, 112, 85, 70, 86, 88, 78, 85, 112, 
        99, 90, 88, 84, 70, 81, 101, 85, 88, 88, 88, 134, 85, 
        88, 70, 97, 88, 101, 86, 85, 90, 80, 84, 92, 94, 107, 
        88, 103, 88, 84, 85, 86, 90, 110, 87, 96, 107, 102, 86, 
        85, 85, 108, 92, 88, 80, 122, 108, 98, 106, 102, 94, 
        89, 96, 88, 101, 121, 89, 97, 94, 112, 102, 92, 100, 
        92, 113, 123, 110, 116, 98, 103, 89, 89, 97, 98, 89, 
        97, 88, 106, 106, 90, 88, 134, 88, 105, 112, 96, 86, 
        91, 95, 102, 103, 120, 107), phe = c(2.8, 2.65, 2.8, 
        3.85, 2.8, 3, 2.5, 2.6, 2.95, 2.98, 2.95, 2.2, 2.6, 3.1, 
        2.95, 2.85, 2.8, 2.72, 3.3, 3.85, 3, 2.41, 2.61, 2.48, 
        2.53, 2.2, 3.1, 2.4, 2.95, 2.65, 3, 3.1, 2.42, 2.95, 
        2.5, 2.7, 2.6, 2.45, 3.18, 3, 3.15, 2.45, 3.25, 2.64, 
        3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 3, 
        2.95, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.56, 3.5, 
        1.89, 2.41, 1.6, 2.11, 2.53, 2.8, 1.1, 2.95, 1.88, 1.95, 
        1.4, 1.45, 2.2, 1.38, 1.9, 2.83, 2.2, 2.2, 1.4, 1.65, 
        2.2, 2.2, 1.7, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
        3.02, 2, 1.25, 2.55, 3.52, 2.83, 2.53, 1.48, 2.56, 2.5, 
        2.2, 2.22, 1.65, 1.75, 2.36, 2.74, 3.18, 2.55, 1.75, 
        2.13, 2.56, 2.46, 2.53, 2, 1.4, 2, 2.9, 3.18, 1.4, 2.42, 
        2.86, 2.6, 2.74, 2.13, 2.22, 2.1, 1.51, 1.8, 1.9, 1.7, 
        1.55, 1.62, 1.38, 0.98, 1.62, 1.65, 1.38, 1.4, 1.4, 2, 
        1.38, 1.25, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.38, 
        1.48, 1.51, 1.8, 1.9, 1.68, 1.9, 2.3, 1.83, 1.9, 1.39, 
        1.35, 1.25, 1.7, 1.48, 1.55, 2, 1.25, 1.39, 1.68, 2.53, 
        1.48, 1.8, 1.65, 2.05), fla = c(2.89, 2.76, 3.39, 3.49, 
        1.79, 3.39, 2.52, 2.51, 2.98, 3.03, 3.32, 2.43, 2.76, 
        3.23, 3.64, 2.91, 3.14, 3.4, 2.52, 3.64, 3.67, 2.41, 
        2.88, 2.37, 2.61, 2.68, 2.94, 2.94, 2.97, 2.33, 3.25, 
        3.19, 2.69, 3.04, 2.53, 2.98, 2.69, 2.43, 2.64, 3.04, 
        3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 3.64, 2.92, 3.54, 
        2.92, 2.99, 3.74, 2.79, 2.9, 2.78, 3.74, 3.23, 3.67, 
        0.57, 1.75, 1.41, 1.79, 2.98, 1.75, 2.65, 1.25, 2, 1.3, 
        2.91, 1.02, 2.86, 1.76, 2.89, 2.14, 1.57, 1.3, 1.25, 
        1.85, 2.11, 2.26, 2.53, 1.58, 1.59, 2.21, 1.6, 1.69, 
        1.61, 1.69, 2.65, 1.69, 1.25, 1.46, 2.63, 2.51, 2.27, 
        0.99, 2.14, 3.64, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 
        1.84, 2.03, 1.76, 2.21, 2.92, 1.85, 2.27, 2.03, 2.01, 
        2.29, 2.17, 1.6, 2.09, 0.66, 1.64, 2.43, 3.39, 1.57, 
        2.14, 3.03, 2.65, 3.15, 2.24, 1.6, 1.75, 0.68, 1.02, 
        1.25, 1.2, 0.58, 0.66, 0.47, 0.55, 0.83, 0.6, 0.5, 1.2, 
        0.52, 0.8, 0.92, 0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 
        1.36, 1.28, 0.83, 0.58, 0.63, 0.58, 0.58, 1.31, 1.1, 
        0.92, 1.41, 0.6, 0.5, 0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 
        0.49, 0.66, 0.7, 0.61, 0.58, 0.69, 0.68, 0.76), nfla = c(0.28, 
        0.21, 0.3, 0.24, 0.39, 0.28, 0.32, 0.31, 0.29, 0.22, 
        0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.2, 0.24, 0.32, 0.17, 
        0.24, 0.25, 0.27, 0.26, 0.22, 0.58, 0.34, 0.27, 0.32, 
        0.26, 0.29, 0.34, 0.42, 0.28, 0.29, 0.26, 0.34, 0.3, 
        0.28, 0.2, 0.34, 0.27, 0.32, 0.32, 0.28, 0.3, 0.27, 0.21, 
        0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 
        0.24, 0.34, 0.28, 0.47, 0.53, 0.32, 0.17, 0.45, 0.37, 
        0.26, 0.27, 0.63, 0.22, 0.3, 0.29, 0.24, 0.21, 0.13, 
        0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.66, 0.4, 0.61, 0.22, 
        0.3, 0.52, 0.43, 0.17, 0.34, 0.3, 0.5, 0.27, 0.25, 0.17, 
        0.32, 0.5, 0.29, 0.24, 0.4, 0.17, 0.29, 0.34, 0.37, 0.27, 
        0.66, 0.28, 0.48, 0.39, 0.29, 0.24, 0.26, 0.32, 0.42, 
        0.32, 0.52, 0.3, 0.24, 0.32, 0.26, 0.32, 0.27, 0.43, 
        0.3, 0.29, 0.34, 0.39, 0.34, 0.4, 0.29, 0.4, 0.53, 0.27, 
        0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.6, 0.39, 
        0.53, 0.52, 0.43, 0.4, 0.47, 0.45, 0.34, 0.22, 0.32, 
        0.26, 0.61, 0.53, 0.6, 0.63, 0.63, 0.53, 0.52, 0.58, 
        0.5, 0.6, 0.4, 0.41, 0.45, 0.43, 0.4, 0.39, 0.53, 0.5, 
        0.48, 0.52, 0.53, 0.43, 0.43, 0.53, 0.45), pro = c(2.29, 
        1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
        2.38, 1.57, 1.81, 1.87, 2.96, 1.46, 1.81, 2.38, 1.86, 
        1.66, 2.1, 2.18, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
        1.76, 1.98, 2.38, 1.95, 1.71, 1.35, 1.54, 1.86, 1.36, 
        1.44, 1.37, 1.83, 2.91, 1.25, 2.1, 1.97, 2.03, 1.44, 
        2.19, 2.14, 2.38, 2.08, 1.98, 2.29, 1.87, 1.68, 1.62, 
        2.45, 1.95, 1.66, 2.08, 0.42, 1.56, 1.35, 0.73, 1.87, 
        1.03, 1.76, 1.46, 1.04, 1.56, 2.5, 1.46, 1.87, 1.03, 
        1.35, 1.98, 1.15, 1.46, 1.56, 2.76, 1.95, 1.43, 1.87, 
        1.46, 1.62, 1.71, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
        1.63, 1.46, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
        1.4, 1.35, 1.31, 1.42, 1.77, 1.42, 1.82, 1.63, 2.08, 
        2.49, 1.83, 1.77, 1.35, 2.5, 1.04, 1.86, 1.71, 1.61, 
        0.83, 1.87, 1.83, 1.46, 1.71, 2.7, 2.91, 1.35, 1.77, 
        1.87, 1.9, 1.66, 0.94, 1.56, 0.83, 0.84, 1.25, 0.94, 
        0.8, 1.1, 0.88, 1.41, 0.75, 1.77, 0.55, 1.55, 1.14, 1.3, 
        0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.82, 0.94, 
        1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.77, 1.25, 0.94, 
        1.35, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 
        1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 4.38, 5.68, 
        7.8, 5.43, 6.75, 4.8, 5.05, 5.64, 7.22, 5.75, 3.95, 5.6, 
        5.4, 7.5, 5.1, 8.5, 5.1, 8.7, 5.1, 5.65, 4.5, 2.6, 3.93, 
        3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5, 4.9, 5.4, 4.2, 
        5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 
        5.24, 4.9, 6.1, 8.9, 8.9, 7.2, 7.8, 4.8, 6.3, 5.85, 6.25, 
        6.75, 4.35, 6.8, 10.68, 3.27, 5.75, 3.8, 4.45, 2.95, 
        4.6, 5.3, 4.68, 3.17, 2.85, 9.7, 3.38, 3.74, 2.8, 3.21, 
        3.8, 4.6, 6.13, 7.5, 2.7, 2.5, 3.9, 2.8, 4.8, 3.05, 2.62, 
        2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 
        2.8, 2.9, 2.9, 3.25, 3.93, 5.1, 2.8, 5.75, 2.94, 2.7, 
        3.4, 3.3, 2.7, 3.25, 2.9, 5.3, 3.8, 3.08, 2.9, 1.9, 3.8, 
        3.25, 3.4, 1.28, 3.25, 3.8, 2.8, 5.85, 2.8, 4.9, 3.94, 
        2.95, 2.12, 3.6, 7.6, 7.1, 7.7, 5, 5.45, 7.1, 5.43, 5, 
        5.7, 4.92, 5.04, 7.6, 4.35, 4.4, 8.21, 4, 4.9, 5, 8.42, 
        9.4, 6.13, 8.42, 7.1, 10.52, 7.6, 7.9, 7.1, 7.5, 13, 
        8.42, 7.65, 7.7, 5.58, 7.1, 7.6, 6.62, 10.68, 10.26, 
        7.6, 8.5, 5.43, 6.62, 9.7, 7.7, 7.3, 7.6, 9.3, 9.2), 
            hue = c(1.04, 1.05, 1.25, 0.86, 0.94, 1.05, 1.02, 
            0.8, 1.08, 0.82, 1.25, 1.07, 1.15, 1.25, 1.2, 1.28, 
            0.86, 1.13, 1.23, 1.25, 1.09, 1.19, 1.11, 1.09, 1.12, 
            1.13, 1.15, 1.02, 0.91, 1.04, 1.19, 1.09, 0.89, 1.25, 
            1.1, 1.08, 0.9, 0.88, 1.18, 0.87, 0.95, 0.91, 0.88, 
            0.82, 1.23, 0.87, 1.19, 1.12, 1.07, 1.12, 1.12, 1.23, 
            1.23, 0.94, 1.23, 1.02, 0.94, 1.07, 0.89, 0.56, 1.25, 
            0.98, 1.23, 1.22, 0.95, 1.31, 1.25, 1.12, 1.02, 1.28, 
            0.906, 1.36, 0.98, 1.31, 1.71, 1.23, 1.19, 0.96, 
            1.06, 1.19, 1.38, 1.42, 1.31, 0.84, 1.31, 1.23, 1.42, 
            1.04, 1.28, 1.07, 1.08, 1.05, 1.12, 1.15, 1, 1.2, 
            1.2, 0.96, 1.25, 1.42, 1.27, 0.84, 0.8, 1.06, 1.04, 
            0.86, 1, 1.23, 0.86, 1.23, 0.8, 0.9, 1.13, 1.1, 0.93, 
            1.71, 0.95, 0.89, 0.7, 1.09, 0.8, 0.96, 0.92, 0.73, 
            1.2, 0.86, 0.69, 0.97, 0.89, 0.79, 0.78, 0.74, 0.66, 
            0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.73, 0.84, 0.906, 
            0.89, 0.92, 0.65, 0.6, 0.7, 0.54, 0.55, 0.56, 0.7, 
            0.48, 0.61, 0.56, 0.6, 0.6, 0.57, 0.67, 0.57, 0.906, 
            0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.67, 0.74, 
            0.67, 0.906, 0.82, 0.65, 0.64, 0.7, 0.59, 0.6, 0.61
            ), ratio = c(3.92, 3.4, 3.17, 3.33, 2.93, 3.58, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3, 3, 2.87, 2.65, 
            3.64, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.36, 3.2, 
            2.85, 2.77, 2.84, 2.84, 2.71, 2.88, 2.87, 3, 2.87, 
            2.84, 3.03, 3.52, 2.69, 2.87, 3.38, 3, 3.2, 3, 3.35, 
            3.33, 3, 3.33, 2.75, 3.1, 2.75, 2.3, 3.26, 2.93, 
            3.2, 3.03, 2.85, 2.84, 2.87, 1.82, 2.42, 1.59, 2.46, 
            2.87, 2.46, 2.3, 3.18, 3.48, 3.48, 4, 1.82, 2.84, 
            2.42, 3.5, 3.13, 2.14, 2.48, 2.52, 2.5, 3.1, 3.58, 
            3.14, 2.72, 2.01, 2.42, 3.58, 2.87, 3.21, 2.75, 3.21, 
            3.39, 2.65, 2.75, 3.3, 2.75, 2.63, 2.01, 2.74, 3.92, 
            2.83, 2.96, 2.77, 3.38, 1.29, 3.57, 3.3, 3.17, 2.42, 
            3.02, 3.26, 2.81, 2.78, 2.5, 2.87, 3.19, 2.87, 3.33, 
            2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 3.14, 3.64, 2.5, 
            2.84, 2.9, 2.78, 2.93, 1.58, 1.42, 1.36, 1.29, 1.51, 
            1.58, 1.27, 1.69, 1.82, 2.15, 1.47, 1.82, 1.33, 2.05, 
            2, 2.12, 1.33, 1.82, 1.62, 1.33, 1.29, 1.47, 1.33, 
            1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 1.82, 
            1.51, 1.75, 1.68, 1.75, 1.62, 1.82, 1.78, 1.62, 1.51, 
            1.56, 1.51, 1.51, 1.68, 1.56, 1.62, 1.6), prol = c(1065, 
            1050, 1280, 1065, 735, 1450, 1290, 1295, 1045, 1045, 
            886, 1280, 1320, 1310, 1547, 1310, 1280, 760, 1680, 
            845, 845, 770, 1035, 1015, 845, 1080, 1195, 1285, 
            915, 1035, 1285, 1035, 1045, 1235, 278, 920, 880, 
            560, 795, 760, 795, 1035, 1095, 680, 885, 1080, 1065, 
            985, 1060, 1060, 1150, 1265, 1190, 1065, 1060, 1120, 
            970, 1270, 1285, 770, 500, 406, 630, 510, 640, 550, 
            502, 470, 392, 718, 870, 410, 472, 1035, 886, 434, 
            392, 835, 740, 463, 278, 714, 410, 515, 835, 625, 
            415, 562, 680, 625, 480, 450, 495, 625, 345, 495, 
            625, 428, 660, 406, 365, 562, 438, 415, 672, 495, 
            510, 470, 312, 680, 562, 325, 886, 434, 385, 502, 
            625, 385, 372, 564, 580, 1035, 365, 380, 835, 378, 
            660, 466, 466, 580, 470, 550, 560, 600, 472, 695, 
            720, 515, 580, 590, 600, 406, 520, 550, 685, 830, 
            415, 450, 406, 550, 500, 480, 425, 675, 640, 770, 
            480, 880, 660, 720, 680, 680, 570, 880, 615, 500, 
            695, 685, 562, 770, 510, 470, 406, 740, 886, 835, 
            840, 470)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.34, 13.24, 
        14.3, 14.39, 14.06, 13.58, 13.86, 13.29, 13.05, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 14.3, 13.64, 14.3, 
        12.93, 13.71, 12.85, 13.5, 14.21, 13.39, 13.3, 13.87, 
        12.6, 13.73, 13.58, 13.68, 13.76, 11.76, 12.45, 12.25, 
        13.05, 12.7, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 13.32, 13.9, 13.83, 14.38, 13.05, 13.48, 13.82, 
        12.79, 13.74, 12.25, 14.22, 13.29, 13.48, 12.37, 12.33, 
        12.64, 12.7, 12.37, 12.17, 12.37, 12.79, 13.05, 13.34, 
        12.21, 13.27, 13.86, 12.51, 12.99, 11.96, 11.66, 13.52, 
        11.84, 14.38, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 11.66, 
        12.16, 12.08, 12.21, 12.08, 12.08, 12, 12.69, 11.82, 
        11.62, 12.22, 11.81, 12.29, 12.25, 12.29, 12.08, 12.6, 
        11.84, 11.87, 12.51, 12.42, 12, 12.72, 12.22, 11.61, 
        12.7, 12.42, 11.76, 11.84, 12.08, 11.03, 11.82, 12.7, 
        12.29, 12.29, 11.45, 13.29, 12.42, 13.05, 11.87, 12.07, 
        11.84, 11.79, 12.22, 12.04, 12.86, 12.88, 12, 12.7, 12.51, 
        13.83, 12.25, 12.53, 13.86, 12.84, 12.93, 13.36, 13.52, 
        13.62, 12.25, 12.29, 11.66, 12.87, 13.32, 12.25, 12.84, 
        12.79, 12, 13.23, 12.85, 13.17, 13.84, 12.45, 14.34, 
        13.48, 13.48, 13.69, 12.85, 12.96, 13.78, 13.73, 14.23, 
        14.3, 13.58, 13.4, 12, 12.77, 14.16, 14.22, 13.4, 13.27, 
        13.17, 14.13), malic = c(1.71, 0.9, 1.6, 1.95, 3.12, 
        2.16, 1.87, 2.15, 1.68, 1.35, 2.16, 1.48, 1.87, 1.73, 
        1.87, 1.13, 1.92, 1.97, 1.88, 3.1, 1.63, 3.8, 0.9, 1.6, 
        1.81, 4.1, 1.73, 1.51, 1.9, 1.13, 1.5, 2.55, 1.83, 1.53, 
        1.17, 1.09, 1.17, 1.65, 1.5, 4.31, 1.71, 4.3, 1.66, 3.98, 
        1.41, 4.04, 1.29, 1.68, 1.51, 1.13, 0.98, 1.65, 1.67, 
        1.9, 1.67, 1.39, 1.7, 1.97, 1.53, 0.94, 2.08, 1.36, 1.25, 
        1.17, 1.95, 1.21, 1.01, 1.17, 2.39, 1.71, 1.81, 1.51, 
        1.66, 1.67, 1.09, 1.88, 0.9, 0.9, 0.99, 3.87, 1.88, 1.9, 
        1.13, 1.71, 3.37, 0.98, 1.61, 1.67, 1.51, 1.5, 3.87, 
        1.51, 3.03, 1.88, 1.6, 1.67, 2.81, 1.41, 1.17, 1.35, 
        2.08, 4.3, 1.67, 1.72, 1.65, 2.55, 1.73, 1.51, 1.29, 
        1.13, 3.74, 4.3, 2.68, 0.74, 1.39, 1.88, 1.21, 1.61, 
        3.43, 1.9, 2.4, 2.05, 4.43, 3.03, 4.31, 2.16, 1.53, 2.13, 
        1.09, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 3.12, 1.35, 
        5.51, 3.59, 3.1, 2.81, 2.56, 4.12, 2.99, 3.88, 5.65, 
        5.04, 4.61, 3.24, 1.9, 3.12, 2.67, 1.9, 2.05, 1.29, 5.19, 
        4.12, 3.03, 1.68, 1.67, 3.83, 4.6, 2.31, 3.45, 2.76, 
        1.29, 3.37, 3.37, 2.58, 4.6, 5.51, 2.39, 1.67, 5.65, 
        3.91, 3.55, 2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 
        2.26, 2.45, 2.45, 2.61, 2.38, 2.3, 2.3, 2.26, 2.41, 2.39, 
        2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.32, 2.65, 2.72, 
        2.52, 2.61, 3.22, 2.5, 2.32, 2.65, 2.14, 2.32, 2.36, 
        2.26, 2.7, 2.65, 2.44, 2.35, 2.55, 2.1, 2.51, 2.3, 2.12, 
        2.59, 2.29, 2.1, 2.44, 2.21, 2.12, 2.4, 2.27, 2.04, 2.2, 
        2.35, 2.65, 2.17, 2.67, 2.3, 2.68, 2.5, 1.36, 2.38, 2.65, 
        1.92, 2.16, 2.53, 2.56, 2.12, 1.92, 2.74, 1.75, 2.21, 
        2.67, 2.24, 2.6, 2.3, 2.48, 1.71, 2.42, 2.42, 2.4, 2, 
        2.55, 2.51, 2.32, 2.58, 2.24, 2.5, 2.62, 2.38, 2.3, 2.32, 
        2.56, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.24, 1.82, 
        1.7, 2.32, 2.46, 1.88, 2.26, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.58, 2.1, 
        2, 2.42, 3.23, 2.61, 2.13, 2.39, 2.17, 2.29, 2.74, 2.35, 
        2.38, 2.61, 2.4, 2.4, 2.36, 2.25, 2.28, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.72, 2.28, 2.48, 2.7, 
        2.36, 2.62, 2.48, 2, 2.28, 2.35, 2.32, 2.38, 2.32, 2.74, 
        2.64, 2.38, 2.54, 2.48, 2.35, 2.3, 2.25, 2.6, 2.17, 2.69, 
        2.7, 2.32, 2.28, 2.37, 2.28, 2.48, 2.26, 2.37, 2.74), 
            alca = c(15.6, 19.5, 17.8, 16.8, 21, 15.2, 16.8, 
            17.6, 14, 16, 19.5, 16.8, 16, 11.4, 16.1, 18.1, 20.5, 
            20, 16, 15.2, 16, 18.6, 19.1, 17.8, 20, 21, 16.1, 
            17, 16, 17, 16, 19.1, 18.8, 21.5, 19, 20.5, 15.5, 
            16, 15.5, 13.2, 16.2, 18.8, 16, 16.8, 17, 19.6, 16, 
            16, 18.8, 17.4, 12.4, 17, 16.1, 19.5, 16, 20.5, 16.3, 
            16.8, 16.1, 10.6, 19, 16.8, 18, 19, 19, 18.1, 22.5, 
            19.6, 22, 16.8, 20.4, 18, 24, 30, 21, 25, 16, 18, 
            14.8, 20, 19, 25, 24, 22.5, 18, 18, 22.8, 30, 21.6, 
            20.5, 23, 22, 18, 18, 18, 19, 21.5, 16, 19.5, 18, 
            17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 
            19.5, 15.5, 22, 18.5, 18.6, 21.5, 20.8, 21, 16, 20, 
            20.5, 21.6, 21, 21.5, 21, 21, 21.5, 28.5, 19.1, 22, 
            23, 20, 24, 21.5, 21.5, 21, 21, 25, 19.5, 25, 21, 
            22.5, 23.5, 20, 20, 21, 20, 22.5, 20, 21.5, 24, 22, 
            21.5, 18.5, 20, 22, 19.5, 21, 25, 22.5, 16.3, 16.8, 
            30, 18.5, 19, 22.5, 23, 19.5, 24.5, 25, 21, 19.5, 
            20, 20.5, 23, 20, 21, 24.5), mg = c(127, 100, 101, 
            113, 103, 112, 96, 121, 97, 120, 124, 116, 89, 91, 
            102, 112, 120, 94, 101, 116, 126, 102, 86, 95, 86, 
            124, 93, 89, 107, 96, 101, 106, 104, 101, 110, 98, 
            110, 87, 85, 124, 117, 95, 101, 124, 107, 111, 102, 
            101, 92, 108, 92, 95, 111, 115, 106, 101, 118, 102, 
            108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 88, 151, 
            102, 86, 87, 139, 101, 88, 70, 112, 107, 101, 86, 
            102, 78, 85, 88, 99, 90, 88, 84, 70, 81, 90, 111, 
            88, 96, 102, 134, 85, 88, 102, 97, 88, 94, 86, 85, 
            90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 
            88, 88, 87, 96, 126, 102, 86, 124, 85, 101, 92, 88, 
            80, 122, 108, 98, 106, 94, 94, 89, 103, 88, 101, 
            106, 89, 97, 70, 112, 102, 85, 101, 92, 113, 123, 
            93, 116, 98, 103, 124, 89, 97, 98, 89, 100, 134, 
            106, 106, 90, 88, 112, 88, 105, 112, 96, 86, 91, 
            95, 102, 116, 120, 98), phe = c(2.8, 2.65, 2.8, 3.85, 
            2.8, 3.15, 2.5, 2.6, 3.15, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.15, 2.85, 2.8, 3, 3.3, 3, 3, 2.41, 2.61, 2.48, 
            2.53, 2.8, 2.45, 2.4, 2.95, 2.65, 3, 3, 2.42, 2.95, 
            2.65, 2.7, 2.6, 2.45, 3.1, 3.1, 3.15, 2.45, 3.25, 
            2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 
            3.88, 2.85, 3, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 
            2.4, 3.5, 1.89, 3, 1.9, 2.11, 2.53, 3.5, 1.1, 2.95, 
            1.88, 1.9, 1.65, 1.51, 2.2, 2.2, 1.9, 2.83, 3.02, 
            2.2, 1.51, 1.65, 2.86, 2.2, 1.83, 1.92, 1.95, 2.2, 
            1.6, 1.45, 1.38, 2.45, 3.02, 2.13, 1.25, 2.55, 3.52, 
            2.45, 1.75, 2, 2.56, 2.5, 2.2, 2, 1.65, 1.83, 2.36, 
            2.74, 3.18, 2.55, 1.75, 2.86, 2.56, 2.46, 1.1, 2, 
            2.56, 2, 2.9, 3.18, 1.68, 1.38, 2.86, 2.8, 2.74, 
            2.13, 2.22, 2.1, 1.51, 2.56, 1.9, 1.7, 1.25, 1.62, 
            1.38, 1.7, 1.62, 0.98, 1.68, 1.4, 2.05, 2, 1.38, 
            1.48, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.93, 
            1.48, 1.48, 1.8, 1.9, 1.8, 1.39, 2.3, 1.83, 1.48, 
            1.39, 1.35, 1.38, 1.7, 1.48, 1.55, 1.65, 1.25, 1.39, 
            1.68, 1.48, 1.38, 0.98, 1.65, 2.05), fla = c(2.99, 
            2.76, 3.29, 3.49, 2.43, 3.39, 2.52, 2.51, 2.98, 3.17, 
            3.32, 2.43, 2.76, 3.19, 3.64, 2.91, 3.14, 3.4, 2.92, 
            2.88, 3.25, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 1.85, 
            2.97, 2.33, 3.25, 3.19, 2.69, 2.98, 2.64, 2.98, 2.64, 
            2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 
            3.17, 3.4, 2.92, 3.54, 3.23, 2.99, 3.74, 2.79, 2.9, 
            2.78, 3.25, 3.23, 3.67, 0.57, 2.09, 1.41, 1.79, 2.92, 
            1.75, 2.65, 1.75, 2, 1.3, 3.64, 1.02, 2.86, 1.31, 
            2.89, 2.14, 1.57, 2.17, 1.75, 1.85, 2.65, 2.26, 2.53, 
            1.58, 1.59, 2.21, 2.64, 1.69, 1.61, 1.69, 2.43, 2.24, 
            1.25, 1.46, 2.91, 2.89, 2.27, 0.99, 2.65, 3.29, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 2.53, 
            2.92, 2.94, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 
            1.25, 1.64, 2.86, 2.9, 1.58, 0.83, 3.03, 2.65, 3.15, 
            2.24, 2.17, 1.75, 1.59, 1.36, 0.7, 1.2, 0.58, 0.66, 
            0.47, 1.59, 0.68, 0.6, 0.5, 1.39, 0.52, 0.8, 0.68, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.83, 0.58, 1.31, 1.1, 0.92, 1.57, 0.6, 
            0.5, 0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.47, 
            0.7, 0.61, 0.57, 0.69, 0.68, 0.76), nfla = c(0.28, 
            0.24, 0.3, 0.24, 0.39, 0.34, 0.21, 0.31, 0.29, 0.22, 
            0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.42, 0.22, 0.32, 
            0.17, 0.24, 0.25, 0.27, 0.26, 0.24, 0.37, 0.34, 0.27, 
            0.13, 0.26, 0.29, 0.26, 0.42, 0.22, 0.29, 0.26, 0.29, 
            0.29, 0.28, 0.2, 0.34, 0.27, 0.26, 0.32, 0.28, 0.3, 
            0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 
            0.2, 0.26, 0.2, 0.25, 0.28, 0.52, 0.53, 0.32, 0.26, 
            0.45, 0.37, 0.26, 0.27, 0.53, 0.24, 0.35, 0.4, 0.43, 
            0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.39, 
            0.4, 0.61, 0.22, 0.3, 0.5, 0.43, 0.26, 0.39, 0.4, 
            0.5, 0.22, 0.25, 0.17, 0.32, 0.3, 0.22, 0.24, 0.26, 
            0.26, 0.29, 0.34, 0.37, 0.26, 0.66, 0.26, 0.48, 0.39, 
            0.29, 0.24, 0.26, 0.48, 0.42, 0.28, 0.52, 0.3, 0.52, 
            0.66, 0.2, 0.32, 0.28, 0.43, 0.3, 0.22, 0.32, 0.39, 
            0.4, 0.4, 0.26, 0.21, 0.29, 0.27, 0.17, 0.6, 0.63, 
            0.53, 0.63, 0.58, 0.53, 0.53, 0.47, 0.43, 0.58, 0.52, 
            0.43, 0.4, 0.47, 0.45, 0.5, 0.22, 0.34, 0.26, 0.61, 
            0.53, 0.22, 0.41, 0.63, 0.53, 0.52, 0.53, 0.5, 0.6, 
            0.4, 0.41, 0.47, 0.43, 0.4, 0.39, 0.4, 0.53, 0.48, 
            0.45, 0.63, 0.43, 0.43, 0.53, 0.52), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 2.1, 2.96, 1.46, 2.5, 1.66, 1.86, 
            1.66, 2.1, 1.98, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
            1.76, 1.98, 2.38, 1.95, 2.38, 1.95, 1.54, 1.86, 1.36, 
            1.44, 1.37, 2.14, 1.45, 1.68, 1.66, 1.87, 2.03, 1.55, 
            2.19, 2.14, 2.38, 2.08, 2.81, 2.29, 1.87, 1.68, 1.62, 
            2.45, 1.99, 1.66, 2.5, 0.42, 1.46, 1.42, 0.73, 1.87, 
            1.03, 1.46, 1.1, 1.04, 0.88, 2.5, 1.46, 1.87, 1.03, 
            1.77, 1.46, 1.15, 1.46, 1.4, 2.76, 1.95, 1.43, 1.46, 
            1.15, 1.62, 0.97, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 1.64, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
            1.4, 1.35, 1.31, 1.42, 1.35, 1.42, 1.87, 1.63, 2.08, 
            2.49, 1.56, 1.31, 1.55, 1.46, 1.04, 1.04, 1.98, 1.61, 
            0.83, 1.87, 1.83, 1.86, 1.71, 1.46, 2.91, 1.35, 1.77, 
            2.03, 1.9, 1.56, 0.94, 0.83, 0.83, 0.84, 1.25, 0.94, 
            0.8, 1.1, 0.88, 0.86, 0.75, 1.45, 0.55, 1.63, 1.14, 
            1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.87, 
            0.8, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.46, 1.87, 
            0.94, 0.86, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 
            1.24, 1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 
            4.38, 5.68, 7.8, 5.65, 6.75, 8.42, 5.05, 4.92, 7.22, 
            5.75, 4.4, 5.6, 5.7, 7.5, 3.95, 7.1, 1.28, 8.7, 5.1, 
            5.65, 4.5, 5.75, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 
            4.7, 5.7, 3.17, 5.75, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            5.24, 8.9, 7.2, 5.1, 6.25, 6.3, 5.85, 6.25, 8.7, 
            4.5, 6.8, 5, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 
            4.68, 3.17, 2.85, 5.7, 3.38, 3.74, 2.8, 3.21, 3.8, 
            4.6, 3.9, 9.3, 3.52, 2.5, 3.9, 2.62, 4.8, 3.05, 2.62, 
            2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 
            2.6, 4.6, 2.9, 2.6, 3.4, 2.7, 4.9, 2.8, 3.6, 2.94, 
            2.7, 3.4, 3.3, 2.7, 2.6, 2.9, 4.68, 3.05, 3.08, 2.9, 
            1.9, 4.68, 3.94, 3.4, 1.28, 3.25, 5.75, 3.4, 4.35, 
            2.8, 1.74, 3.94, 3.4, 2.12, 2.94, 4.2, 4.35, 3.17, 
            5, 5.45, 7.1, 5.1, 5, 5.7, 4.92, 7.5, 5.4, 4.35, 
            4.4, 8.21, 4, 4.9, 5.85, 8.42, 9.4, 3.74, 4.7, 7.1, 
            10.52, 7.6, 7.9, 7.5, 7.5, 13, 7.7, 7.65, 5.1, 5.58, 
            5.45, 5.75, 6.62, 10.68, 10.26, 5.58, 8.5, 2.7, 4.6, 
            9.7, 7.7, 7.3, 5.58, 9.3, 9.2), hue = c(1.04, 1.05, 
            1.04, 0.86, 0.92, 1.05, 1.02, 0.98, 1.08, 0.98, 1.25, 
            1.23, 1.15, 1.25, 1.2, 1.28, 0.8, 1.13, 1.23, 0.87, 
            1.09, 0.66, 1.11, 1.09, 1.12, 1.13, 1.15, 1.02, 0.87, 
            1.04, 1.19, 1.09, 1.04, 1.25, 1.1, 1.71, 1.22, 0.8, 
            1.18, 0.86, 0.95, 0.91, 0.88, 0.82, 1.23, 0.87, 1.12, 
            1.04, 1.07, 1.12, 1.12, 1.12, 0.88, 0.96, 0.84, 0.95, 
            0.94, 1.07, 0.89, 0.92, 1.25, 0.98, 1.23, 1.22, 0.95, 
            1.19, 0.96, 1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 
            1.31, 1.23, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 1.09, 
            1.31, 0.84, 0.82, 1.23, 0.8, 0.95, 0.97, 1.07, 1.08, 
            1.05, 0.86, 1.15, 0.93, 0.86, 0.7, 1.31, 1.07, 1.42, 
            1.27, 0.73, 0.8, 1.27, 1.04, 0.89, 1, 1.09, 0.86, 
            0.89, 1.09, 0.9, 1.02, 1.1, 0.93, 1.71, 0.95, 1.13, 
            0.7, 0.98, 0.8, 0.91, 0.92, 0.73, 0.97, 0.86, 0.69, 
            0.97, 0.89, 0.79, 0.84, 0.74, 0.66, 0.78, 0.75, 0.73, 
            0.75, 0.82, 0.81, 0.7, 0.61, 0.92, 0.89, 0.906, 0.65, 
            0.6, 0.73, 0.54, 0.55, 0.74, 1.19, 0.48, 0.61, 0.56, 
            0.7, 0.6, 0.57, 0.67, 0.57, 0.75, 0.56, 0.96, 0.87, 
            0.68, 0.7, 0.78, 0.85, 0.55, 0.74, 0.67, 0.906, 0.48, 
            0.59, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
            3.4, 3.17, 3.21, 2.93, 3.4, 3.58, 3.58, 2.85, 3.55, 
            3.17, 2.82, 2.9, 3.17, 3, 2.85, 2.65, 2.84, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.4, 3.2, 4, 2.77, 3, 
            3.17, 2.71, 2.88, 2.87, 3, 2.87, 2.78, 2.93, 3, 2.69, 
            3.26, 3.38, 3, 3.71, 3, 3.35, 3.33, 3, 3.33, 2.75, 
            3.1, 3.33, 4, 3.26, 2.93, 3.2, 3.03, 2.14, 2.84, 
            2.87, 1.82, 2.42, 1.59, 2.46, 2.87, 2.42, 2.3, 3.18, 
            3.48, 1.36, 3.33, 1.82, 2.72, 2.52, 3.5, 3.13, 2.14, 
            2.48, 2.52, 1.64, 3.3, 3.33, 3.14, 2.72, 2.01, 3.02, 
            3, 2.93, 3.21, 2.75, 3.21, 2.74, 2.65, 3.55, 3.3, 
            3, 2.63, 2.65, 2.74, 2.84, 2.83, 2.96, 2.77, 3.38, 
            1.59, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 
            2.5, 3.33, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 
            3.69, 3.12, 1.82, 3.64, 3.26, 2.84, 2.93, 2.78, 2.78, 
            3.03, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 
            2.15, 1.62, 1.29, 2, 2.05, 2, 1.58, 1.33, 1.73, 1.62, 
            1.33, 3.33, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 
            1.96, 1.78, 1.58, 1.69, 1.51, 1.75, 1.68, 1.75, 1.55, 
            1.55, 1.73, 2, 1.58, 1.82, 1.55, 2, 1.62, 1.56, 1.62, 
            1.6), prol = c(1065, 1050, 970, 845, 735, 1450, 1290, 
            1295, 1045, 1045, 886, 1280, 1320, 1295, 1547, 1310, 
            1280, 1095, 1680, 845, 1150, 770, 1035, 1015, 845, 
            1095, 1195, 1285, 915, 1035, 1285, 1285, 1065, 1235, 
            1035, 920, 880, 770, 450, 760, 795, 1035, 1095, 680, 
            885, 1080, 1065, 985, 1060, 915, 1150, 1265, 1190, 
            1120, 1060, 1120, 970, 1270, 1285, 880, 680, 760, 
            630, 520, 675, 1050, 502, 680, 735, 718, 870, 410, 
            472, 840, 886, 580, 392, 1050, 1450, 463, 278, 714, 
            562, 515, 515, 735, 600, 562, 680, 625, 480, 450, 
            495, 450, 345, 480, 625, 428, 660, 406, 428, 562, 
            438, 415, 672, 438, 510, 480, 312, 680, 562, 325, 
            463, 434, 385, 470, 495, 714, 372, 564, 625, 770, 
            365, 380, 502, 378, 880, 466, 640, 580, 630, 520, 
            560, 600, 495, 695, 720, 515, 580, 590, 600, 600, 
            520, 550, 380, 830, 415, 278, 675, 550, 500, 480, 
            425, 675, 640, 470, 480, 880, 660, 1285, 428, 680, 
            570, 675, 615, 714, 695, 685, 1015, 680, 510, 470, 
            680, 740, 680, 835, 840, 1035)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 14.38, 13.24, 13.63, 14.39, 14.06, 14.23, 
        13.86, 13.05, 14.75, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.3, 13.64, 13.88, 12.93, 13.71, 12.85, 13.5, 
        13.48, 13.39, 13.3, 13.87, 13.68, 13.73, 13.58, 13.68, 
        13.76, 13.9, 13.3, 13.5, 13.05, 12.64, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.73, 13.9, 13.76, 
        13.82, 13.05, 11.61, 13.82, 13.86, 13.74, 13.27, 14.22, 
        13.29, 13.82, 12.37, 12.33, 12.64, 11.76, 12.37, 12.17, 
        12.37, 13.68, 13.62, 13.34, 12.21, 13.24, 13.86, 12.08, 
        12.99, 11.96, 11.66, 12.07, 11.84, 12.37, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 12.51, 12.16, 12.64, 12.87, 12.08, 
        12.08, 12, 12.69, 13.05, 11.62, 12.08, 11.81, 12.29, 
        13.74, 12.29, 12.08, 12.6, 12.93, 12.96, 12.51, 13.4, 
        11.84, 12.72, 12.22, 11.61, 11.84, 13.86, 11.76, 12.6, 
        12.08, 11.03, 11.82, 12.87, 11.82, 12.04, 11.45, 13.75, 
        12.42, 13.05, 11.87, 12.07, 12.51, 11.79, 12.37, 12.04, 
        12.86, 12.88, 12.64, 12.7, 12.51, 13.68, 12.25, 12.53, 
        12.37, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.37, 
        12, 12.87, 13.32, 13.84, 12.21, 12.79, 13.5, 13.23, 13.5, 
        13.17, 13.84, 12.45, 14.34, 13.48, 13.36, 13.69, 12.85, 
        12.96, 13.78, 13.73, 14.39, 13.39, 13.58, 13.4, 12.25, 
        12.77, 14.16, 12.45, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
        1.25, 1.53, 1.95, 1.5, 1.61, 1.87, 2.15, 2.55, 1.35, 
        2.16, 1.48, 3.74, 1.73, 1.87, 2.67, 1.92, 1.9, 0.9, 3.1, 
        1.63, 3.8, 1.61, 1.6, 1.81, 5.19, 1.53, 3.98, 1.9, 1.67, 
        1.5, 0.98, 1.83, 1.53, 1.7, 1.9, 3.98, 1.65, 1.5, 3.24, 
        1.71, 1.17, 1.5, 3.98, 1.21, 4.04, 1.48, 1.68, 1.21, 
        1.95, 1.29, 1.65, 1.68, 1.9, 1.67, 1.29, 1.7, 1.97, 3.87, 
        0.94, 2.58, 1.36, 1.25, 1.9, 4.3, 1.21, 1.01, 1.17, 2.58, 
        1.67, 3.98, 1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 2.15, 
        0.99, 3.87, 2.08, 1.17, 1.13, 1.29, 4.43, 0.98, 1.61, 
        1.67, 1.41, 2.99, 4.31, 1.51, 1.25, 1.09, 1.21, 2.55, 
        5.04, 1.41, 2.16, 1.61, 2.08, 3.24, 1.51, 1.72, 1.9, 
        2.55, 1.73, 1.5, 1.29, 1.73, 3.74, 1.01, 2.68, 0.74, 
        1.39, 1.39, 1.83, 1.61, 3.43, 1.41, 2.4, 2.05, 4.43, 
        2.76, 4.31, 2.16, 1.53, 2.13, 1.87, 4.3, 1.35, 2.99, 
        2.31, 3.55, 1.24, 2.58, 3.8, 5.51, 3.59, 4.61, 2.81, 
        2.56, 2.56, 3.1, 3.88, 3.83, 5.04, 4.61, 3.24, 1.5, 3.12, 
        2.67, 1.9, 1.41, 1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 
        3.83, 1.35, 1.01, 3.45, 2.76, 2.05, 1.81, 3.37, 2.58, 
        4.6, 1.7, 2.39, 4.6, 5.65, 3.91, 2.15, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.62, 2.48, 2.45, 2.61, 2.25, 2.22, 
        2.3, 2.32, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 
        2.38, 2.65, 2.58, 2.52, 2.61, 3.22, 2.7, 2.62, 2.14, 
        2.58, 2.39, 2.36, 2.51, 2.7, 2.65, 2.62, 2.21, 2.55, 
        2.1, 2.51, 2.3, 2.12, 2.59, 2.29, 2.1, 2.44, 2.39, 2.12, 
        2.4, 2.27, 2.04, 2.39, 2.4, 2.41, 2.21, 2.44, 2.3, 2.68, 
        2.5, 1.36, 2.36, 2.2, 1.92, 2.16, 2.53, 2.56, 2.12, 1.92, 
        2.32, 1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.88, 1.71, 2.2, 
        1.99, 2.4, 2, 2.32, 2.51, 2.32, 2.58, 2.24, 2.1, 2.62, 
        2.68, 2.3, 2.32, 2.5, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 
        1.75, 2.04, 1.7, 2.1, 2.46, 1.88, 2.19, 2.27, 2.12, 2.28, 
        1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 3.22, 
        1.94, 2, 2.42, 3.23, 2.61, 2.13, 2.39, 2.17, 2.29, 2.59, 
        2.56, 2.38, 2.21, 2.4, 2.4, 2.36, 2.25, 2.48, 2.54, 2.64, 
        2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.53, 2.26, 2.48, 
        2.16, 2.36, 2.62, 2.48, 2.12, 2.28, 2.2, 2.32, 2.38, 
        2.48, 2.3, 2.64, 2.38, 2.54, 2.12, 2.35, 2.3, 2.51, 2.6, 
        2.55, 2.69, 3.22, 2.32, 2.28, 2.61, 2.7, 2.48, 2.26, 
        2.37, 2.74), alca = c(15.6, 16, 22.5, 16.8, 21, 15.2, 
        16, 17.6, 14, 16, 20.5, 16.8, 16, 11.4, 12.4, 17.4, 22, 
        20, 15.2, 15.2, 16, 18.6, 21, 17.8, 20, 18.6, 16.1, 17, 
        16.8, 17, 17.4, 19.1, 21.5, 22, 19, 20.5, 15.5, 20.5, 
        15.5, 13.2, 16.2, 18.8, 18.6, 22, 17, 17.4, 16, 17.6, 
        18.8, 17.4, 12.4, 15.5, 15.2, 18, 17.6, 20.5, 16.3, 16.8, 
        17.6, 10.6, 18.6, 16.8, 18, 19, 19, 18.1, 18.6, 19.6, 
        19, 16.8, 20.4, 24, 24, 30, 21, 17.4, 16, 18, 14.8, 21.5, 
        19, 22.5, 24, 22.5, 18, 18, 22.8, 22, 21.6, 16.1, 21.5, 
        22, 22, 18, 18, 19, 21.5, 16, 11.4, 18, 17.5, 18.5, 21, 
        19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 20, 20, 21.5, 
        22.5, 21.5, 20.8, 23.5, 16, 18, 22, 28.5, 22, 21.5, 21, 
        21, 21.5, 28.5, 23, 22, 17.8, 20, 24, 21.5, 17.8, 18, 
        21.5, 25, 19.5, 20, 21, 25, 23.5, 20, 16.8, 21, 20, 18.8, 
        19.5, 21.5, 24, 22, 18.8, 18.5, 20, 22, 19.5, 21, 25, 
        21.6, 18.6, 24.5, 18.1, 18.5, 22.5, 22.5, 23, 19.5, 24.5, 
        25, 24.5, 19.5, 18.8, 20.5, 23, 20, 18, 24.5), mg = c(127, 
        100, 101, 113, 100, 112, 96, 121, 97, 87, 104, 101, 89, 
        91, 102, 112, 120, 104, 106, 116, 126, 102, 84, 95, 80, 
        124, 93, 101, 107, 96, 101, 106, 104, 116, 110, 113, 
        110, 105, 104, 78, 117, 106, 101, 92, 107, 111, 102, 
        101, 104, 108, 92, 112, 111, 115, 98, 108, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 80, 151, 
        101, 86, 87, 139, 101, 113, 98, 112, 120, 112, 86, 100, 
        78, 85, 112, 99, 90, 88, 84, 70, 81, 84, 88, 88, 78, 
        105, 134, 85, 88, 115, 97, 88, 84, 86, 85, 90, 80, 84, 
        92, 94, 107, 88, 103, 88, 84, 85, 86, 86, 78, 87, 96, 
        120, 102, 86, 111, 85, 107, 92, 88, 80, 122, 87, 98, 
        106, 100, 94, 89, 78, 88, 101, 85, 89, 97, 99, 112, 102, 
        85, 122, 92, 113, 123, 98, 116, 98, 103, 88, 89, 97, 
        98, 89, 120, 102, 106, 106, 90, 88, 98, 88, 105, 112, 
        96, 86, 91, 95, 102, 102, 120, 96), phe = c(2.8, 2.65, 
        2.8, 3.85, 2.8, 2.8, 2.5, 2.6, 3.25, 2.98, 2.95, 2.2, 
        2.6, 3.1, 3.2, 2.85, 2.8, 3.5, 3.3, 2.95, 3, 2.41, 2.61, 
        2.48, 2.53, 2.2, 2.86, 2.4, 2.95, 2.65, 3, 3.25, 2.42, 
        2.95, 2.53, 2.7, 2.6, 2.45, 2.41, 3.2, 3.15, 2.45, 3.25, 
        2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 
        2.74, 2.5, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2, 3.5, 
        1.89, 2.98, 2.56, 2.11, 2.53, 2.2, 1.1, 2.95, 1.88, 2.2, 
        2.05, 2.05, 2.74, 1.88, 1.9, 2.83, 2.53, 2.2, 2.3, 1.65, 
        2.65, 2.2, 2.2, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
        3.02, 2.42, 1.48, 2.55, 3.52, 3.3, 2.55, 2, 2.56, 2.5, 
        2.2, 2.45, 1.65, 2.5, 2.36, 2.74, 3.18, 2.55, 1.75, 2.11, 
        2.56, 2.46, 2.2, 2, 1.48, 2, 2.9, 3.18, 2.6, 2.22, 2.86, 
        2.98, 2.74, 2.13, 2.22, 2.1, 1.51, 2.3, 1.39, 1.7, 1.48, 
        1.62, 1.38, 1.48, 1.62, 1.98, 1.4, 1.4, 1.38, 2, 1.38, 
        0.98, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.75, 1.48, 
        1.1, 1.8, 1.9, 1.65, 1.75, 2.3, 1.83, 1.55, 1.39, 1.35, 
        1.38, 1.7, 1.48, 1.55, 1.83, 1.25, 1.39, 1.68, 1.48, 
        1.98, 0.98, 1.65, 2.05), fla = c(2.99, 2.76, 2.99, 3.49, 
        2.33, 3.39, 2.52, 2.51, 2.98, 3.32, 3.32, 2.43, 2.76, 
        2.9, 3.64, 2.91, 3.14, 3.4, 3.49, 2.88, 3.67, 2.41, 2.88, 
        2.37, 2.61, 2.68, 2.94, 2.69, 2.97, 2.33, 3.25, 3.19, 
        2.69, 3.64, 2.37, 2.98, 2.63, 2.43, 2.64, 3.04, 3.29, 
        2.68, 3.56, 2.63, 3, 2.65, 3.17, 3.56, 2.92, 3.54, 2.37, 
        2.99, 3.74, 2.79, 2.9, 2.78, 3.32, 3.23, 3.67, 0.57, 
        1.84, 1.41, 1.79, 2.94, 1.75, 2.65, 2.27, 2, 1.3, 2.27, 
        1.02, 2.86, 1.57, 2.89, 2.14, 1.57, 2.53, 0.8, 1.85, 
        2.69, 2.26, 2.53, 1.58, 1.59, 2.21, 2.11, 1.69, 1.61, 
        1.69, 2, 0.76, 1.25, 1.46, 2.63, 2.88, 2.27, 0.99, 1.64, 
        3.19, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 
        1.76, 1.6, 2.92, 2.51, 2.27, 2.03, 2.01, 2.29, 2.17, 
        1.6, 2.09, 0.49, 1.64, 2.91, 3.49, 2.26, 1.57, 3.03, 
        2.65, 3.15, 2.24, 2.24, 1.75, 1.75, 1.75, 1.39, 1.2, 
        0.58, 0.66, 0.47, 0.66, 0.58, 0.6, 0.5, 1.59, 0.52, 0.8, 
        1.02, 0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 
        0.83, 0.58, 0.63, 0.52, 0.58, 1.31, 1.1, 0.92, 1.1, 0.6, 
        0.55, 0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.55, 
        0.7, 0.61, 0.84, 0.69, 0.68, 0.76), nfla = c(0.28, 0.32, 
        0.3, 0.24, 0.39, 0.32, 0.29, 0.31, 0.29, 0.22, 0.22, 
        0.26, 0.29, 0.43, 0.29, 0.3, 0.28, 0.32, 0.32, 0.17, 
        0.24, 0.25, 0.27, 0.26, 0.13, 0.4, 0.34, 0.27, 0.24, 
        0.26, 0.29, 0.28, 0.42, 0.34, 0.29, 0.26, 0.34, 0.21, 
        0.28, 0.2, 0.34, 0.21, 0.28, 0.32, 0.28, 0.3, 0.27, 0.21, 
        0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 
        0.28, 0.39, 0.28, 0.41, 0.53, 0.32, 0.29, 0.45, 0.37, 
        0.26, 0.27, 0.52, 0.45, 0.39, 0.26, 0.53, 0.21, 0.13, 
        0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.28, 0.4, 0.61, 0.22, 
        0.3, 0.37, 0.42, 0.5, 0.42, 0.63, 0.5, 0.5, 0.25, 0.17, 
        0.32, 0.63, 0.47, 0.24, 0.26, 0.27, 0.29, 0.34, 0.37, 
        0.43, 0.66, 0.4, 0.48, 0.39, 0.29, 0.24, 0.26, 0.43, 
        0.42, 0.29, 0.52, 0.3, 0.66, 0.63, 0.43, 0.32, 0.45, 
        0.43, 0.3, 0.32, 0.3, 0.39, 0.29, 0.4, 0.26, 0.3, 0.42, 
        0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 
        0.41, 0.4, 0.63, 0.61, 0.43, 0.4, 0.47, 0.45, 0.17, 0.22, 
        0.37, 0.26, 0.61, 0.53, 0.53, 0.5, 0.63, 0.53, 0.52, 
        0.4, 0.5, 0.6, 0.4, 0.41, 0.63, 0.43, 0.4, 0.39, 0.39, 
        0.45, 0.48, 0.41, 0.53, 0.43, 0.43, 0.53, 0.39), pro = c(2.29, 
        1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
        2.38, 1.57, 1.81, 1.92, 2.96, 1.46, 2.5, 2.03, 1.86, 
        1.66, 2.1, 1.46, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
        1.76, 1.98, 2.38, 1.95, 1.87, 2.38, 1.54, 1.86, 1.36, 
        1.44, 1.37, 1.9, 1.95, 1.76, 2.96, 1.85, 2.03, 1.35, 
        2.19, 2.14, 2.38, 2.08, 1.37, 2.29, 1.87, 1.68, 1.62, 
        2.45, 2.38, 1.66, 2.08, 0.42, 1.66, 1.35, 0.73, 1.87, 
        1.03, 1.71, 1.63, 1.04, 0.73, 2.5, 1.46, 1.87, 1.03, 
        1.85, 1.81, 1.15, 1.46, 1.3, 2.76, 1.95, 1.43, 1.85, 
        0.64, 1.62, 1.62, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
        1.63, 1.63, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
        1.4, 1.35, 1.31, 1.42, 1.04, 1.42, 1.35, 1.63, 2.08, 
        2.49, 2.1, 1.77, 1.1, 1.14, 1.04, 1.14, 1.63, 1.61, 0.83, 
        1.87, 1.83, 1.87, 1.71, 0.83, 2.91, 1.35, 1.77, 1.81, 
        1.9, 2.29, 0.94, 1.64, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 
        0.88, 0.42, 0.75, 1.35, 0.55, 1.46, 1.14, 1.3, 0.68, 
        0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 0.94, 1.46, 1.55, 
        1.56, 1.14, 2.7, 2.29, 1.04, 0.94, 1.35, 0.94, 1.46, 
        1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 
        1.41, 1.35, 1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 
        3.58, 6.75, 4.5, 5.05, 6.3, 7.22, 5.75, 7.3, 5.6, 3.3, 
        7.5, 6.25, 7.6, 5.45, 8.7, 5.1, 5.65, 4.5, 4.38, 3.93, 
        3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 6.13, 5.4, 5.4, 
        4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 
        5.04, 5.24, 4.9, 6.1, 6.75, 8.9, 7.2, 3.17, 5.1, 6.3, 
        5.85, 6.25, 6.3, 4.4, 6.8, 5.65, 3.27, 5.75, 3.8, 4.45, 
        2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 4.7, 3.38, 3.74, 2.9, 
        3.21, 3.8, 4.6, 5.04, 8.21, 3.93, 2.5, 3.9, 1.28, 4.8, 
        3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 
        3.25, 2.6, 5.64, 2.9, 2.9, 2.9, 2.4, 4.68, 2.8, 4.6, 
        2.94, 2.7, 3.4, 3.3, 2.7, 3.25, 2.9, 7.2, 4.68, 3.08, 
        2.9, 1.9, 3.93, 3.6, 3.4, 1.28, 3.25, 6.8, 3.8, 5.1, 
        2.8, 4.6, 3.94, 2.95, 2.12, 4.6, 3.7, 7.5, 8.5, 5, 5.45, 
        7.1, 5.43, 5, 5.7, 4.92, 7.5, 7.65, 4.35, 4.4, 8.21, 
        4, 4.9, 8.5, 8.42, 9.4, 5.75, 6.62, 7.1, 10.52, 7.6, 
        7.9, 7.1, 7.5, 13, 7.65, 7.65, 10.52, 5.58, 4.8, 7.3, 
        6.62, 10.68, 10.26, 9.4, 8.5, 3.3, 4, 9.7, 7.7, 7.3, 
        5.7, 9.3, 9.2), hue = c(1.04, 1.05, 1.09, 0.86, 1.07, 
        1.05, 1.02, 0.8, 1.08, 1.12, 1.25, 0.86, 1.15, 1.25, 
        1.2, 1.28, 0.87, 1.13, 1.23, 0.92, 1.09, 1.08, 1.11, 
        1.09, 1.12, 1.13, 1.09, 1.02, 1.2, 1.04, 1.19, 1.09, 
        0.95, 1.25, 1.1, 0.92, 0.9, 0.86, 1.18, 0.75, 0.95, 0.91, 
        0.88, 0.82, 1.25, 0.87, 1.25, 1.07, 1.07, 1.12, 1.12, 
        1.22, 1.11, 1.15, 0.84, 1.42, 0.94, 1.07, 0.89, 0.98, 
        1.25, 0.98, 1.23, 1.22, 0.82, 1.07, 1.04, 1.12, 1.02, 
        1.28, 0.906, 1.36, 0.98, 1.31, 1.27, 1.23, 1.19, 0.96, 
        1.06, 1.19, 1.38, 1.22, 1.31, 0.84, 0.96, 1.23, 1.19, 
        1.07, 1.05, 1.07, 1.08, 1.05, 1.38, 1.15, 0.91, 1.02, 
        0.65, 1.36, 1.22, 1.42, 1.27, 0.75, 0.8, 0.96, 1.04, 
        1.02, 1, 1.42, 0.86, 1.22, 1.09, 0.9, 0.96, 1.1, 0.93, 
        1.71, 0.95, 0.87, 0.7, 1.13, 0.8, 1.25, 0.92, 0.73, 1.22, 
        0.86, 0.69, 0.97, 0.89, 0.79, 0.92, 0.74, 0.66, 0.78, 
        0.75, 0.73, 0.75, 0.82, 0.81, 0.74, 0.54, 0.6, 0.89, 
        0.55, 0.65, 0.6, 0.57, 0.54, 0.55, 0.54, 0.61, 0.48, 
        0.61, 0.56, 0.68, 0.6, 0.57, 0.67, 0.57, 0.68, 0.56, 
        0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.64, 0.74, 0.67, 
        0.84, 0.67, 0.67, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
        3.4, 3.17, 2.01, 2.93, 2.84, 3.58, 3.58, 2.85, 3.55, 
        3.17, 2.82, 2.9, 3.3, 3, 3.92, 2.65, 3, 2.82, 3.36, 3.71, 
        3.52, 4, 3.63, 3.69, 3.2, 2.84, 2.77, 2.87, 2.75, 2.71, 
        2.88, 2.87, 3, 2.87, 2.96, 2.65, 2.83, 2.69, 2.71, 3.38, 
        3, 3.71, 3, 3.35, 3.33, 3.57, 3.33, 2.75, 3.1, 2.48, 
        2.84, 3.26, 2.93, 3.2, 3.03, 2.5, 2.84, 2.87, 1.82, 2.52, 
        1.59, 2.46, 2.87, 2.93, 2.3, 3.18, 3.48, 2.01, 2.82, 
        1.82, 3.52, 2.46, 3.5, 3.13, 2.14, 2.48, 2.52, 1.58, 
        3.21, 2.65, 3.14, 2.72, 2.01, 2.85, 3.17, 2.78, 3.21, 
        2.75, 3.21, 1.78, 2.65, 2.46, 3.3, 3.3, 2.63, 1.73, 2.74, 
        3.55, 2.83, 2.96, 2.77, 3.38, 3.33, 3.57, 3.3, 3.17, 
        2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.19, 3.19, 2.87, 
        3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.3, 3.64, 
        3.17, 2.84, 3.19, 2.78, 2.93, 3.17, 1.42, 1.36, 1.29, 
        1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.29, 2.15, 1.59, 
        2.05, 2, 1.68, 1.33, 1.33, 1.62, 1.33, 2.96, 1.47, 1.33, 
        1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 1.58, 
        1.27, 1.75, 1.68, 1.75, 1.33, 1.58, 1.33, 1.82, 2.14, 
        2.15, 1.33, 1.6, 1.48, 1.56, 1.62, 1.6), prol = c(1065, 
        1050, 695, 1095, 735, 1450, 1290, 1295, 1045, 1045, 1050, 
        1280, 1320, 1680, 1547, 1310, 1280, 1045, 1680, 845, 
        1320, 770, 1035, 1015, 845, 1680, 1195, 1285, 915, 1035, 
        1285, 1285, 740, 1235, 920, 920, 880, 680, 1120, 760, 
        795, 1035, 1095, 680, 885, 1080, 1065, 985, 1060, 1680, 
        1150, 1265, 1190, 1680, 1060, 1120, 970, 1270, 1285, 
        312, 1120, 714, 630, 564, 510, 680, 502, 672, 345, 718, 
        870, 410, 472, 434, 886, 672, 392, 378, 685, 463, 278, 
        714, 312, 515, 570, 795, 472, 562, 680, 625, 480, 450, 
        495, 886, 345, 392, 625, 428, 660, 406, 590, 562, 438, 
        415, 672, 835, 510, 480, 312, 680, 562, 325, 680, 434, 
        385, 470, 480, 1095, 372, 564, 502, 1265, 365, 380, 520, 
        378, 680, 466, 615, 580, 718, 630, 560, 600, 640, 695, 
        720, 515, 580, 590, 600, 600, 520, 550, 660, 830, 415, 
        830, 560, 550, 500, 480, 425, 675, 640, 680, 480, 880, 
        660, 325, 438, 680, 570, 378, 615, 886, 695, 685, 830, 
        714, 510, 470, 1050, 740, 434, 835, 840, 735)), row.names = c(NA, 
        -178L), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 13.29, 13.24, 13.82, 14.39, 14.06, 13.74, 
        13.86, 13.56, 13.2, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 13.82, 13.64, 13.3, 12.93, 13.71, 12.85, 13.5, 
        13.68, 13.39, 13.3, 13.87, 14.3, 13.73, 13.58, 13.68, 
        13.76, 11.03, 13.86, 13.27, 13.05, 14.3, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.39, 13.9, 14.34, 
        13.88, 13.05, 13.05, 13.82, 14.06, 13.74, 14.34, 14.22, 
        13.29, 13.29, 12.37, 12.33, 12.64, 12.22, 12.37, 12.17, 
        12.37, 12.29, 11.62, 13.34, 12.21, 12.08, 13.86, 13.34, 
        12.99, 11.96, 11.66, 12.08, 11.84, 12.29, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 12.29, 12.16, 12.29, 11.84, 12.08, 
        12.08, 12, 12.69, 13.36, 11.62, 12.16, 11.81, 12.29, 
        12.93, 12.29, 12.08, 12.6, 12.72, 12.64, 12.51, 12.93, 
        12.29, 12.72, 12.22, 11.61, 12.88, 11.79, 11.76, 13.86, 
        12.08, 11.03, 11.82, 14.3, 13.05, 12.29, 11.45, 13.36, 
        12.42, 13.05, 11.87, 12.07, 13.88, 11.79, 12.22, 12.04, 
        12.86, 12.88, 13.4, 12.7, 12.51, 13.4, 12.25, 12.53, 
        13.27, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.7, 
        11.76, 12.87, 13.32, 13.84, 12.51, 12.79, 13.88, 13.23, 
        14.75, 13.17, 13.84, 12.45, 14.34, 13.48, 13.73, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.4, 13.17, 13.58, 13.4, 
        12.22, 12.77, 14.16, 12.25, 13.4, 13.27, 13.17, 14.13
        ), malic = c(1.71, 1.5, 0.74, 1.95, 1.67, 3.1, 1.87, 
        2.15, 1.25, 1.35, 2.16, 1.48, 1.17, 1.73, 1.87, 1.81, 
        1.92, 1.63, 1.67, 3.1, 1.63, 3.8, 1.25, 1.6, 1.81, 1.97, 
        1.68, 1.66, 1.9, 1.71, 1.5, 0.98, 1.83, 1.53, 1.81, 1.01, 
        0.9, 1.65, 1.5, 1.81, 1.71, 1.83, 1.35, 3.98, 1.17, 4.04, 
        1.61, 1.68, 1.68, 1.73, 1.51, 1.65, 2.67, 1.9, 1.67, 
        1.5, 1.7, 1.97, 1.29, 0.94, 1.73, 1.36, 1.25, 2.67, 1.81, 
        1.21, 1.01, 1.17, 5.51, 1.5, 2.08, 1.51, 1.66, 1.67, 
        1.09, 1.88, 0.9, 2.39, 0.99, 3.87, 0.94, 1.6, 1.13, 1.39, 
        2.99, 0.98, 1.61, 1.67, 2.56, 1.7, 2.16, 1.51, 0.9, 1.73, 
        1.67, 0.94, 1.36, 1.41, 1.13, 1.73, 2.08, 1.6, 3.98, 
        1.72, 1.09, 2.55, 1.73, 1.29, 1.29, 1.41, 3.74, 1.29, 
        2.68, 0.74, 1.39, 0.9, 1.67, 1.61, 3.43, 1.25, 2.4, 2.05, 
        4.43, 1.39, 4.31, 2.16, 1.53, 2.13, 3.12, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 5.04, 3.83, 5.51, 3.59, 2.58, 
        2.81, 2.56, 2.58, 2.59, 3.88, 3.03, 5.04, 4.61, 3.24, 
        1.71, 3.12, 2.67, 1.9, 2.67, 1.29, 5.19, 4.12, 3.03, 
        1.68, 1.67, 3.83, 3.24, 1.68, 3.45, 2.76, 3.59, 1.9, 
        3.37, 2.58, 4.6, 3.12, 2.39, 1.67, 5.65, 3.91, 3.87, 
        2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.29, 2.32, 
        2.45, 2.61, 2.25, 2.1, 2.3, 2.12, 2.41, 2.39, 2.38, 2.7, 
        2.72, 2.62, 2.48, 2.56, 2.62, 2.65, 2.29, 2.52, 2.61, 
        3.22, 2.29, 2.36, 2.7, 2.28, 2.52, 2.36, 2.92, 2.7, 2.65, 
        2.51, 2.53, 2.55, 2.1, 2.51, 2.67, 2.12, 2.59, 2.29, 
        2.1, 2.44, 2.38, 2.12, 2.4, 2.27, 2.04, 3.23, 2.27, 2.32, 
        2.4, 2.36, 2.3, 2.68, 2.5, 1.36, 2.64, 1.71, 1.92, 2.16, 
        2.53, 2.56, 2.67, 1.92, 2.64, 1.75, 2.21, 2.67, 2.24, 
        2.6, 2.3, 2.21, 1.71, 2.22, 2, 2.4, 2, 2.61, 2.51, 2.32, 
        2.58, 2.24, 2.19, 2.62, 2.36, 2.3, 2.32, 2.27, 2.26, 
        2.22, 2.28, 2.2, 2.74, 1.98, 2.64, 1.75, 1.7, 2.04, 2.46, 
        1.88, 2.7, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 
        2.5, 2.5, 2.2, 1.99, 2.29, 1.7, 2, 2.42, 3.23, 2.64, 
        2.13, 2.39, 2.17, 2.29, 2.48, 2.4, 2.38, 2.67, 2.4, 2.4, 
        2.36, 2.25, 2.48, 2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 
        2.72, 2.35, 2.2, 2.2, 2.92, 2.48, 2.26, 2.36, 2.62, 2.48, 
        2.3, 2.28, 2.64, 2.32, 2.38, 2.2, 2.53, 2.64, 2.38, 2.54, 
        2.62, 2.35, 2.3, 2.41, 2.6, 2.19, 2.69, 2.62, 2.32, 2.28, 
        1.71, 3.22, 2.48, 2.26, 2.37, 2.74), alca = c(15.6, 16.1, 
        19, 16.8, 21, 15.2, 16, 17.6, 14, 16, 14, 16.8, 16, 11.4, 
        19.6, 18.8, 16, 20, 15.5, 15.2, 16, 18.6, 15.5, 17.8, 
        20, 22.5, 16.1, 17, 17.6, 16.3, 21.5, 19.1, 22, 19.6, 
        19, 20.5, 15.5, 19.1, 15.5, 13.2, 16.2, 18.8, 16, 22, 
        17, 18.8, 16, 17, 18.8, 17.4, 12.4, 18, 10.6, 16, 16, 
        20.5, 16.3, 16.8, 18.8, 10.6, 19, 16.8, 18, 19, 19, 18.1, 
        30, 19.6, 18, 16.8, 20.4, 21, 24, 30, 21, 18, 16, 18, 
        14.8, 22, 19, 20, 24, 22.5, 18, 18, 22.8, 23.5, 21.6, 
        21.5, 24, 22, 20.5, 18, 18, 19, 21.5, 16, 19.5, 18, 17.5, 
        18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 18, 
        18.6, 21, 25, 21.5, 20.8, 11.4, 16, 21.6, 22, 21, 22, 
        21.5, 21, 21, 21.5, 28.5, 23, 22, 25, 20, 24, 21.5, 21, 
        20, 20.5, 25, 19.5, 20, 21, 22, 23.5, 20, 22, 21, 20, 
        19, 18, 21.5, 24, 22, 18.5, 18.5, 20, 22, 19.5, 22, 25, 
        24.5, 18.5, 24, 24.5, 18.5, 20, 22.5, 23, 19.5, 24.5, 
        25, 25, 19.5, 19, 20.5, 23, 20, 22, 24.5), mg = c(127, 
        100, 101, 113, 89, 112, 96, 121, 97, 106, 108, 103, 89, 
        91, 102, 112, 120, 108, 120, 116, 126, 102, 122, 95, 
        92, 124, 93, 105, 107, 96, 101, 106, 104, 107, 110, 139, 
        110, 98, 106, 105, 117, 102, 101, 81, 107, 111, 102, 
        101, 101, 108, 92, 95, 111, 115, 115, 116, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 88, 151, 
        102, 86, 87, 139, 101, 107, 98, 112, 87, 89, 86, 98, 
        78, 85, 101, 99, 90, 88, 84, 70, 81, 78, 94, 88, 116, 
        78, 134, 85, 88, 117, 97, 88, 96, 86, 85, 90, 80, 84, 
        92, 94, 107, 88, 103, 88, 84, 85, 86, 91, 85, 87, 96, 
        101, 102, 86, 116, 85, 88, 92, 88, 80, 122, 86, 98, 106, 
        106, 94, 89, 84, 88, 101, 88, 89, 97, 88, 112, 102, 97, 
        85, 92, 113, 123, 89, 116, 98, 103, 104, 89, 97, 98, 
        89, 102, 98, 106, 106, 90, 88, 96, 88, 105, 112, 96, 
        86, 91, 95, 102, 94, 120, 88), phe = c(2.8, 2.65, 2.8, 
        3.85, 2.8, 2.88, 2.5, 2.6, 2.85, 2.98, 2.95, 2.2, 2.6, 
        3.1, 3.88, 2.85, 2.8, 3.3, 3.3, 2.95, 3, 2.41, 2.61, 
        2.48, 2.53, 2, 2.86, 2.4, 2.95, 2.65, 3, 3, 2.42, 2.95, 
        1.92, 2.7, 2.6, 2.45, 2.8, 2.8, 3.15, 2.45, 3.25, 2.64, 
        3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.64, 
        2.8, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.45, 3.5, 
        1.89, 1.65, 2.2, 2.11, 2.53, 2.88, 1.1, 2.95, 1.88, 3.1, 
        2.2, 2.36, 2, 2.53, 1.9, 2.83, 2.2, 2.2, 1.41, 1.65, 
        2.45, 2.2, 2.36, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
        3.02, 2.46, 1.38, 2.55, 3.52, 3.3, 2.95, 1.9, 2.56, 2.5, 
        2.2, 2.6, 1.65, 1.4, 2.36, 2.74, 3.18, 2.55, 1.75, 2.2, 
        2.56, 2.46, 2, 2, 1.4, 2, 2.9, 3.18, 2.45, 1.35, 2.86, 
        2.41, 2.74, 2.13, 2.22, 2.1, 1.51, 0.98, 1.38, 1.7, 1.25, 
        1.62, 1.38, 1.48, 1.62, 1.38, 1.39, 1.4, 1.4, 2, 1.38, 
        1.25, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.62, 1.48, 
        1.62, 1.8, 1.9, 2.2, 1.38, 2.3, 1.83, 1.38, 1.39, 1.35, 
        1.38, 1.7, 1.48, 1.55, 2.3, 1.25, 1.39, 1.68, 1.25, 1.65, 
        1.51, 1.65, 2.05), fla = c(2.88, 2.76, 3.67, 3.49, 2.29, 
        3.39, 2.52, 2.51, 2.98, 2.97, 3.32, 2.43, 2.76, 2.68, 
        3.64, 2.91, 3.14, 3.4, 3.67, 2.91, 2.99, 2.41, 2.88, 
        2.37, 2.61, 2.68, 2.94, 2.17, 2.97, 2.33, 3.25, 3.19, 
        2.69, 2.98, 1.69, 2.98, 3.39, 2.43, 2.64, 3.04, 3.29, 
        2.68, 3.56, 2.63, 3, 2.65, 3.17, 3.19, 2.92, 3.54, 2.91, 
        2.99, 3.74, 2.79, 2.9, 2.78, 2.98, 3.23, 3.67, 0.57, 
        3.03, 1.41, 1.79, 3.56, 1.75, 2.65, 2.86, 2, 1.3, 2.51, 
        1.02, 2.86, 1.57, 2.89, 2.14, 1.57, 1.64, 2.11, 1.85, 
        2.29, 2.26, 2.53, 1.58, 1.59, 2.21, 2.53, 1.69, 1.61, 
        1.69, 2.26, 1.75, 1.25, 1.46, 3.15, 2.99, 2.27, 0.99, 
        2.17, 3.67, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
        2.03, 1.76, 2.21, 2.92, 2.24, 2.27, 2.03, 2.01, 2.29, 
        2.17, 1.6, 2.09, 0.66, 1.64, 2.76, 3.29, 2.26, 0.69, 
        3.03, 2.65, 3.15, 2.24, 1.64, 1.75, 1.41, 0.47, 0.47, 
        1.2, 0.58, 0.66, 0.47, 0.66, 0.7, 0.6, 0.5, 0.47, 0.52, 
        0.8, 1.36, 0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 
        1.28, 0.83, 0.58, 0.63, 0.83, 0.58, 1.31, 1.1, 0.92, 
        0.5, 0.6, 0.47, 0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 
        0.52, 0.7, 0.61, 0.5, 0.69, 0.68, 0.76), nfla = c(0.28, 
        0.3, 0.3, 0.24, 0.39, 0.28, 0.26, 0.31, 0.29, 0.22, 0.22, 
        0.26, 0.29, 0.43, 0.29, 0.3, 0.3, 0.25, 0.32, 0.17, 0.24, 
        0.25, 0.27, 0.26, 0.3, 0.24, 0.34, 0.27, 0.26, 0.26, 
        0.29, 0.26, 0.42, 0.31, 0.29, 0.26, 0.32, 0.53, 0.28, 
        0.2, 0.34, 0.22, 0.17, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
        0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.34, 
        0.37, 0.28, 0.26, 0.53, 0.32, 0.32, 0.45, 0.37, 0.26, 
        0.27, 0.35, 0.32, 0.53, 0.24, 0.61, 0.21, 0.13, 0.34, 
        0.24, 0.43, 0.35, 0.43, 0.3, 0.29, 0.4, 0.61, 0.22, 0.3, 
        0.42, 0.37, 0.35, 0.3, 0.17, 0.5, 0.5, 0.25, 0.17, 0.32, 
        0.58, 0.29, 0.24, 0.26, 0.27, 0.29, 0.34, 0.37, 0.29, 
        0.66, 0.26, 0.48, 0.39, 0.29, 0.24, 0.26, 0.37, 0.42, 
        0.4, 0.52, 0.3, 0.26, 0.3, 0.17, 0.32, 0.24, 0.43, 0.3, 
        0.27, 0.29, 0.39, 0.39, 0.4, 0.24, 0.6, 0.5, 0.27, 0.17, 
        0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.63, 0.45, 
        0.5, 0.34, 0.43, 0.4, 0.47, 0.45, 0.4, 0.22, 0.58, 0.26, 
        0.61, 0.53, 0.66, 0.28, 0.63, 0.53, 0.52, 0.26, 0.5, 
        0.6, 0.4, 0.41, 0.26, 0.43, 0.4, 0.39, 0.4, 0.29, 0.48, 
        0.52, 0.4, 0.43, 0.43, 0.53, 0.43), pro = c(2.29, 1.28, 
        2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 
        1.57, 1.81, 1.62, 2.96, 1.46, 1.95, 1.95, 1.86, 1.66, 
        2.1, 1.35, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 
        1.98, 2.38, 1.95, 1.66, 2.1, 1.54, 1.86, 1.36, 1.44, 
        1.37, 1.76, 1.04, 1.14, 1.77, 2.29, 2.03, 1.28, 2.19, 
        2.14, 2.38, 2.08, 1.46, 2.29, 1.87, 1.68, 1.62, 2.45, 
        2.45, 1.66, 1.83, 0.42, 1.04, 0.94, 0.73, 1.87, 1.03, 
        1.56, 2.96, 1.04, 1.43, 2.5, 1.46, 1.87, 1.03, 1.95, 
        1.98, 1.15, 1.46, 1.14, 2.76, 1.95, 1.43, 2.38, 1.46, 
        1.62, 1.43, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 1.63, 
        1.46, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 
        1.35, 1.31, 1.42, 1.38, 1.42, 1.57, 1.63, 2.08, 2.49, 
        1.83, 1.87, 1.44, 1.77, 1.04, 1.35, 1.35, 1.61, 0.83, 
        1.87, 1.83, 2.08, 1.71, 1.46, 2.91, 1.35, 1.77, 1.35, 
        1.9, 1.24, 0.94, 1.15, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 
        0.88, 0.55, 0.75, 0.94, 0.55, 1.1, 1.14, 1.3, 0.68, 0.86, 
        1.25, 1.14, 1.25, 1.26, 1.56, 0.55, 1.03, 1.55, 1.56, 
        1.14, 2.7, 2.29, 1.04, 1.34, 1.25, 0.94, 1.46, 1.15, 
        1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 
        1.35, 1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 3.74, 
        6.75, 5.1, 5.05, 5.05, 7.22, 5.75, 3.74, 5.6, 6.62, 7.5, 
        5.1, 8.42, 5.05, 8.7, 5.1, 5.65, 4.5, 3.6, 3.93, 3.52, 
        3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 7.7, 6.1, 5.4, 4.2, 5.1, 
        4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 
        4.9, 6.1, 6.8, 8.9, 7.2, 4.8, 8.7, 6.3, 5.85, 6.25, 5.45, 
        7.22, 6.8, 2.15, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 
        4.68, 3.17, 2.85, 5.68, 3.38, 3.74, 4.36, 3.21, 3.8, 
        4.6, 3.58, 3.9, 3.6, 2.5, 3.9, 1.74, 4.8, 3.05, 2.62, 
        2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 
        7.5, 2.9, 2.7, 3.4, 2.8, 2.12, 2.8, 2.4, 2.94, 2.7, 3.4, 
        3.3, 2.7, 3.25, 2.9, 2.15, 5.65, 3.08, 2.9, 1.9, 1.74, 
        2.9, 3.4, 1.28, 3.25, 4.2, 4.36, 6.3, 2.8, 3.52, 3.94, 
        2.4, 2.12, 3.38, 4.8, 7.1, 9.4, 5, 5.45, 7.1, 3.17, 5, 
        5.7, 4.92, 6.8, 6.8, 4.35, 4.4, 8.21, 4, 4.9, 6.8, 8.42, 
        9.4, 5.1, 9.2, 7.1, 10.52, 7.6, 7.9, 7.3, 7.5, 13, 5.43, 
        7.65, 7.9, 5.58, 4, 10.26, 6.62, 10.68, 10.26, 4, 8.5, 
        1.74, 8.7, 9.7, 7.7, 7.3, 9.7, 9.3, 9.2), hue = c(1.04, 
        1.05, 1.23, 0.86, 1.1, 1.05, 1.02, 0.86, 1.08, 1.02, 
        1.25, 1, 1.15, 1.25, 1.2, 1.28, 0.89, 1.13, 1.23, 1.07, 
        1.09, 1.09, 1.11, 1.09, 1.12, 1.13, 1.09, 1.02, 0.95, 
        1.04, 1.19, 1.09, 1.07, 1.25, 1.1, 1.05, 1.22, 1.15, 
        1.18, 1.23, 0.95, 0.91, 0.88, 0.82, 1.15, 0.87, 1.11, 
        0.97, 1.07, 1.12, 1.12, 1.38, 1.09, 0.89, 0.86, 1.07, 
        0.94, 1.07, 0.89, 1.25, 1.25, 0.98, 1.23, 1.22, 1.19, 
        1.19, 1.05, 1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 
        1.07, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 1.23, 1.31, 
        0.84, 0.86, 1.23, 1.08, 1.25, 1.09, 1.07, 1.08, 1.05, 
        0.8, 1.15, 1.13, 1.04, 0.67, 0.9, 1.71, 1.42, 1.27, 1.12, 
        0.8, 1.15, 1.04, 1.15, 1, 0.91, 0.86, 1.28, 0.82, 0.9, 
        0.98, 1.1, 0.93, 1.71, 0.95, 1.12, 0.7, 1.09, 0.8, 1.28, 
        0.92, 0.73, 1.19, 0.86, 0.69, 0.97, 0.89, 0.79, 1.02, 
        0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.87, 
        0.67, 0.81, 0.89, 0.906, 0.65, 0.6, 0.78, 0.54, 0.55, 
        0.54, 0.78, 0.48, 0.61, 0.56, 0.61, 0.6, 0.57, 0.67, 
        0.57, 0.56, 0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 
        0.7, 0.74, 0.67, 0.89, 0.85, 0.54, 0.64, 0.7, 0.59, 0.6, 
        0.61), ratio = c(3.92, 3.4, 3.17, 3.13, 2.93, 2.87, 3.58, 
        3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 2.93, 3, 3.33, 2.65, 
        2.83, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.36, 3.2, 3.33, 
        2.77, 4, 3.58, 2.71, 2.88, 2.87, 3, 2.87, 2.83, 3.3, 
        2.87, 2.69, 3.64, 3.38, 3, 3.64, 3, 3.35, 3.33, 3, 3.33, 
        2.75, 3.1, 3.3, 3.2, 3.26, 2.93, 3.2, 3.03, 3.39, 2.84, 
        2.87, 1.82, 3.26, 1.59, 2.46, 2.87, 3.3, 2.3, 3.18, 3.48, 
        1.29, 3, 1.82, 2.87, 3.48, 3.5, 3.13, 2.14, 2.48, 2.52, 
        2.77, 3.33, 3.39, 3.14, 2.72, 2.01, 3.17, 3.17, 3.02, 
        3.21, 2.75, 3.21, 2.3, 2.65, 2.77, 3.3, 2.87, 2.63, 1.51, 
        2.74, 3.5, 2.83, 2.96, 2.77, 3.38, 2.52, 3.57, 3.3, 3.17, 
        2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.14, 3.19, 2.87, 
        3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 1.75, 3.64, 
        2.9, 2.84, 2.78, 2.78, 2.77, 2.5, 1.42, 1.36, 1.29, 1.51, 
        1.58, 1.27, 1.69, 1.82, 2.15, 1.58, 1.62, 1.78, 2.05, 
        2, 2, 1.33, 1.48, 1.62, 1.33, 2.14, 1.47, 1.33, 1.51, 
        1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 1.58, 2.12, 
        1.75, 1.68, 1.75, 1.36, 1.51, 1.82, 1.58, 2.65, 1.62, 
        1.62, 1.96, 1.58, 1.56, 1.62, 1.6), prol = c(1065, 1050, 
        1320, 970, 735, 1450, 1290, 1295, 1045, 1045, 1045, 1280, 
        1320, 1080, 1547, 1310, 1280, 915, 1680, 845, 1450, 770, 
        1035, 1015, 845, 1120, 1195, 1285, 915, 1035, 1285, 795, 
        1285, 1235, 500, 920, 880, 885, 1035, 760, 795, 1035, 
        1095, 680, 885, 1080, 1065, 985, 1060, 1120, 1150, 1265, 
        1190, 1270, 1060, 1120, 970, 1270, 1285, 510, 1015, 625, 
        630, 345, 365, 1320, 502, 410, 562, 718, 870, 410, 472, 
        378, 886, 466, 392, 450, 502, 463, 278, 714, 365, 515, 
        480, 680, 640, 562, 680, 625, 480, 450, 495, 1035, 345, 
        345, 625, 428, 660, 406, 434, 562, 438, 415, 672, 470, 
        510, 570, 312, 680, 562, 325, 415, 434, 385, 562, 580, 
        1280, 372, 564, 410, 915, 365, 380, 680, 378, 1035, 466, 
        660, 580, 840, 580, 560, 600, 625, 695, 720, 515, 580, 
        590, 600, 500, 520, 550, 835, 830, 415, 570, 720, 550, 
        500, 480, 425, 675, 640, 680, 480, 880, 660, 615, 520, 
        680, 570, 406, 615, 550, 695, 685, 365, 830, 510, 470, 
        660, 740, 550, 835, 840, 630)), row.names = c(NA, -178L
        ), class = "data.frame"), structure(list(alco = c(14.23, 
        13.2, 13.16, 14.75, 13.24, 13.86, 14.39, 14.06, 13.4, 
        13.86, 13.58, 13.58, 13.75, 14.75, 14.38, 13.63, 14.3, 
        13.83, 14.3, 13.64, 13.64, 12.93, 13.71, 12.85, 13.5, 
        13.29, 13.39, 13.3, 13.87, 13.4, 13.73, 13.58, 13.68, 
        13.76, 13.17, 13.23, 13.17, 13.05, 13.41, 14.22, 13.56, 
        13.41, 13.88, 13.24, 13.05, 14.21, 13.56, 13.9, 14.16, 
        14.3, 13.05, 13.05, 13.82, 13.23, 13.74, 14.39, 14.22, 
        13.29, 14.34, 12.37, 12.33, 12.64, 11.76, 12.37, 12.17, 
        12.37, 13.62, 13.52, 13.34, 12.21, 14.39, 13.86, 12.07, 
        12.99, 11.96, 11.66, 12.42, 11.84, 13.86, 12.7, 12, 12.72, 
        12.08, 13.05, 11.84, 12.29, 12.16, 11.76, 12.99, 12.08, 
        12.08, 12, 12.69, 11.62, 11.62, 12.07, 11.81, 12.29, 
        13.36, 12.29, 12.08, 12.6, 12.51, 12.08, 12.51, 12.51, 
        12.08, 12.72, 12.22, 11.61, 12, 12.29, 11.76, 12.08, 
        12.08, 11.03, 11.82, 13.4, 12.08, 11.62, 11.45, 13.78, 
        12.42, 13.05, 11.87, 12.07, 11.84, 11.79, 13.52, 12.04, 
        12.86, 12.88, 13.3, 12.7, 12.51, 14.39, 12.25, 12.53, 
        12.17, 12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.51, 
        11.45, 12.87, 13.32, 13.73, 12.37, 12.79, 11.45, 13.23, 
        12.72, 13.17, 13.84, 12.45, 14.34, 13.48, 13.73, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.05, 13.69, 13.58, 13.4, 
        12.08, 12.77, 14.16, 13.05, 13.4, 13.27, 13.17, 14.13
        ), malic = c(1.71, 1.83, 3.8, 1.95, 0.94, 0.98, 1.87, 
        2.15, 1.21, 1.35, 2.16, 1.48, 1.71, 1.73, 1.87, 3.12, 
        1.92, 1.41, 0.99, 3.1, 1.63, 3.8, 3.1, 1.6, 1.81, 5.04, 
        1.65, 1.72, 1.9, 0.74, 1.5, 1.5, 1.83, 1.53, 1.7, 1.68, 
        1.53, 1.65, 1.5, 1.67, 1.71, 3.12, 2.31, 3.98, 0.94, 
        4.04, 1.01, 1.68, 1.67, 1.71, 1.63, 1.65, 1.9, 1.9, 1.67, 
        1.6, 1.7, 1.97, 1.36, 0.94, 0.94, 1.36, 1.25, 1.13, 2.4, 
        1.21, 1.01, 1.17, 2.13, 1.87, 2.4, 1.51, 1.66, 1.67, 
        1.09, 1.88, 0.9, 1.5, 0.99, 3.87, 1.65, 2.4, 1.13, 3.74, 
        1.24, 0.98, 1.61, 1.67, 1.67, 1.66, 2.13, 1.51, 1.5, 
        1.41, 1.29, 2.16, 5.51, 1.41, 1.83, 1.73, 2.08, 3.8, 
        1.9, 1.72, 3.8, 2.55, 1.73, 2.56, 1.29, 1.01, 3.74, 2.4, 
        2.68, 0.74, 1.39, 1.87, 1.67, 1.61, 3.43, 1.67, 2.4, 
        2.05, 4.43, 3.74, 4.31, 2.16, 1.53, 2.13, 3.74, 4.3, 
        1.35, 2.99, 2.31, 3.55, 1.24, 4.1, 3.83, 5.51, 3.59, 
        3.88, 2.81, 2.56, 3.45, 4.6, 3.88, 2.81, 5.04, 4.61, 
        3.24, 3.24, 3.12, 2.67, 1.9, 3.83, 1.29, 5.19, 4.12, 
        3.03, 1.68, 1.67, 3.83, 2.05, 2.15, 3.45, 2.76, 3.91, 
        3.88, 3.37, 2.58, 4.6, 2.55, 2.39, 4.12, 5.65, 3.91, 
        3.91, 2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.61, 
        2.38, 2.45, 2.61, 1.75, 2.53, 2.3, 2.36, 2.41, 2.39, 
        2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.14, 2.65, 2.72, 
        2.52, 2.61, 3.22, 2.35, 2.53, 2.5, 2, 2.43, 2.36, 2.14, 
        2.7, 2.65, 2.32, 2.26, 2.55, 2.1, 2.51, 2.2, 2.12, 2.59, 
        2.29, 2.1, 2.44, 2.3, 2.12, 2.4, 2.27, 2.04, 2.64, 2.16, 
        2.3, 1.92, 2.44, 2.3, 2.68, 2.5, 1.36, 1.94, 1.98, 1.92, 
        2.16, 2.53, 2.56, 1.92, 1.92, 2.74, 1.75, 2.21, 2.67, 
        2.24, 2.6, 2.3, 1.92, 1.71, 2.04, 2.3, 2.4, 2, 2.13, 
        2.51, 2.32, 2.58, 2.24, 2.7, 2.62, 2.48, 2.3, 2.32, 2.17, 
        2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.38, 2.29, 1.7, 2, 
        2.46, 1.88, 2.5, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 
        2.92, 2.5, 2.5, 2.2, 1.99, 2.4, 1.94, 2, 2.42, 3.23, 
        2.13, 2.13, 2.39, 2.17, 2.29, 2.6, 2.44, 2.38, 2.48, 
        2.4, 2.4, 2.36, 2.25, 2.12, 2.54, 2.64, 2.19, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.26, 2.22, 2.48, 2.52, 2.36, 
        2.62, 2.48, 2.52, 2.28, 1.98, 2.32, 2.38, 3.23, 2.74, 
        2.64, 2.38, 2.54, 2.21, 2.35, 2.3, 2.61, 2.6, 1.88, 2.69, 
        2.72, 2.32, 2.28, 2.64, 2.4, 2.48, 2.26, 2.37, 2.74), 
            alca = c(15.6, 15.5, 24, 16.8, 21, 15.2, 19.1, 17.6, 
            14, 16, 19, 16.8, 16, 11.4, 10.6, 19.5, 16, 20, 15.2, 
            15.2, 16, 18.6, 20, 17.8, 20, 25, 16.1, 17, 16.8, 
            16.8, 14.8, 19.1, 11.4, 17.8, 19, 20.5, 15.5, 22, 
            15.5, 13.2, 16.2, 18.8, 15.2, 20.5, 17, 16, 16, 16, 
            18.8, 17.4, 12.4, 16.8, 10.6, 19.5, 17, 20.5, 16.3, 
            16.8, 16, 10.6, 18, 16.8, 18, 19, 19, 18.1, 16.8, 
            19.6, 21, 16.8, 20.4, 22, 24, 30, 21, 19.5, 16, 18, 
            14.8, 18, 19, 16, 24, 22.5, 18, 18, 22.8, 18, 21.6, 
            21.5, 24.5, 22, 22, 18, 18, 19, 21.5, 16, 16, 18, 
            17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 
            19.5, 21, 25, 24, 25, 21.5, 20.8, 19, 16, 18.8, 21, 
            21.5, 18.5, 21.5, 21, 21, 21.5, 28.5, 18.1, 22, 21, 
            20, 24, 21.5, 18, 20.8, 28.5, 25, 19.5, 28.5, 21, 
            24, 23.5, 20, 22.8, 21, 20, 25, 18.5, 21.5, 24, 22, 
            22.5, 18.5, 20, 22, 19.5, 25, 25, 28.5, 18, 20, 22.8, 
            18.5, 21, 22.5, 23, 19.5, 24.5, 25, 21, 19.5, 24, 
            20.5, 23, 20, 22, 24.5), mg = c(127, 100, 101, 113, 
            96, 112, 96, 121, 97, 110, 102, 106, 89, 91, 102, 
            112, 120, 113, 98, 116, 126, 102, 123, 95, 108, 124, 
            93, 89, 107, 96, 101, 106, 104, 106, 110, 113, 110, 
            90, 106, 85, 117, 108, 101, 92, 107, 111, 102, 101, 
            88, 108, 92, 120, 111, 115, 107, 118, 118, 102, 108, 
            88, 101, 100, 94, 87, 104, 98, 78, 78, 94, 151, 90, 
            86, 87, 139, 101, 90, 78, 112, 117, 70, 86, 88, 78, 
            85, 90, 99, 90, 88, 84, 70, 81, 89, 86, 88, 87, 107, 
            134, 85, 88, 90, 97, 88, 87, 86, 85, 90, 80, 84, 
            92, 94, 107, 88, 103, 88, 84, 85, 86, 112, 81, 87, 
            96, 103, 102, 86, 111, 85, 116, 92, 88, 80, 122, 
            88, 98, 106, 78, 94, 89, 102, 88, 101, 85, 89, 97, 
            85, 112, 102, 89, 90, 92, 113, 123, 95, 116, 98, 
            103, 96, 89, 97, 98, 89, 88, 88, 106, 106, 90, 88, 
            89, 88, 105, 112, 96, 86, 91, 95, 102, 107, 120, 
            102), phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.2, 2.5, 
            2.6, 2.8, 2.98, 2.95, 2.2, 2.6, 3.1, 3.1, 2.85, 2.8, 
            2.74, 3.3, 2.45, 3, 2.41, 2.61, 2.48, 2.53, 2.1, 
            3.18, 2.4, 2.95, 2.65, 3, 2.95, 2.42, 2.95, 2.64, 
            2.7, 2.6, 2.45, 2.41, 3.1, 3.15, 2.45, 3.25, 2.64, 
            3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 
            2.45, 3.52, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 
            1.9, 3.5, 1.89, 2.5, 2.45, 2.11, 2.53, 2.3, 1.1, 
            2.95, 1.88, 2.53, 3.18, 2.2, 2.48, 1.93, 1.9, 2.83, 
            2.11, 2.2, 1.93, 1.65, 2.95, 2.2, 2.13, 1.92, 1.95, 
            2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 1.48, 1.4, 2.55, 
            3.52, 2.45, 1.48, 1.9, 2.56, 2.5, 2.2, 2, 1.65, 1.83, 
            2.36, 2.74, 3.18, 2.55, 1.75, 2.11, 2.56, 2.46, 1.83, 
            2, 1.6, 2, 2.9, 3.18, 2.3, 1.88, 2.86, 2.9, 2.74, 
            2.13, 2.22, 2.1, 1.51, 1.8, 1.39, 1.7, 2.02, 1.62, 
            1.38, 0.98, 1.62, 1.25, 1.68, 1.4, 1.38, 2, 1.38, 
            2.05, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 2.2, 
            1.48, 1.39, 1.8, 1.9, 1.95, 1.39, 2.3, 1.83, 1.48, 
            1.39, 1.35, 1.39, 1.7, 1.48, 1.55, 1.55, 1.25, 1.39, 
            1.68, 1.35, 1.41, 1.8, 1.65, 2.05), fla = c(3.04, 
            2.76, 2.65, 3.49, 3.23, 3.39, 2.52, 2.51, 2.98, 3.03, 
            3.32, 2.43, 2.76, 3.17, 3.64, 2.91, 3.14, 3.4, 3.03, 
            2.76, 3.32, 2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.65, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.32, 2.21, 2.98, 2.21, 
            2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 
            3.17, 2.89, 2.92, 3.54, 2.76, 2.99, 3.74, 2.79, 2.9, 
            2.78, 3.25, 3.23, 3.67, 0.57, 2.09, 1.41, 1.79, 3.4, 
            1.75, 2.65, 2.01, 2, 1.3, 2.9, 1.02, 2.86, 0.92, 
            2.89, 2.14, 1.57, 2.27, 1.79, 1.85, 2.69, 2.26, 2.53, 
            1.58, 1.59, 2.21, 2.27, 1.69, 1.61, 1.69, 2.26, 1.36, 
            1.25, 1.46, 2.17, 2.76, 2.27, 0.99, 2.27, 3.74, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 2.01, 
            2.92, 2.76, 2.27, 2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 
            0.84, 1.64, 2.51, 2.91, 1.64, 0.6, 3.03, 2.65, 3.15, 
            2.24, 2.65, 1.75, 1.69, 0.68, 0.65, 1.2, 0.58, 0.66, 
            0.47, 0.34, 0.96, 0.6, 0.5, 0.61, 0.52, 0.8, 0.65, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.68, 0.58, 1.31, 1.1, 0.92, 1.58, 0.6, 
            0.61, 0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.58, 
            0.7, 0.61, 0.58, 0.69, 0.68, 0.76), nfla = c(0.28, 
            0.39, 0.3, 0.24, 0.39, 0.32, 0.29, 0.31, 0.29, 0.22, 
            0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.32, 0.42, 0.32, 
            0.17, 0.24, 0.25, 0.27, 0.26, 0.24, 0.26, 0.34, 0.27, 
            0.27, 0.26, 0.29, 0.27, 0.42, 0.27, 0.29, 0.26, 0.34, 
            0.25, 0.28, 0.2, 0.34, 0.27, 0.17, 0.32, 0.28, 0.3, 
            0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 
            0.2, 0.26, 0.21, 0.34, 0.28, 0.3, 0.53, 0.32, 0.34, 
            0.45, 0.37, 0.26, 0.27, 0.42, 0.32, 0.45, 0.22, 0.37, 
            0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.29, 
            0.4, 0.61, 0.22, 0.3, 0.45, 0.32, 0.32, 0.29, 0.53, 
            0.5, 0.41, 0.25, 0.17, 0.32, 0.61, 0.29, 0.24, 0.26, 
            0.39, 0.29, 0.34, 0.37, 0.45, 0.66, 0.52, 0.48, 0.39, 
            0.29, 0.24, 0.26, 0.43, 0.42, 0.32, 0.52, 0.3, 0.2, 
            0.26, 0.26, 0.32, 0.26, 0.43, 0.3, 0.27, 0.32, 0.39, 
            0.32, 0.4, 0.48, 0.17, 0.47, 0.27, 0.17, 0.6, 0.63, 
            0.53, 0.63, 0.58, 0.53, 0.53, 0.3, 0.43, 0.4, 0.37, 
            0.43, 0.4, 0.47, 0.45, 0.43, 0.22, 0.27, 0.26, 0.61, 
            0.53, 0.47, 0.61, 0.63, 0.53, 0.52, 0.43, 0.5, 0.6, 
            0.4, 0.41, 0.53, 0.43, 0.4, 0.39, 0.32, 0.27, 0.48, 
            0.39, 0.63, 0.43, 0.43, 0.53, 0.48), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 1.97, 2.96, 1.46, 2.38, 2.96, 1.86, 
            1.66, 2.1, 1.66, 1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 
            1.76, 1.98, 2.38, 1.95, 1.77, 1.9, 1.54, 1.86, 1.36, 
            1.44, 1.37, 2.08, 2.29, 1.26, 1.87, 1.45, 2.03, 2.08, 
            2.19, 2.14, 2.38, 2.08, 1.85, 2.29, 1.87, 1.68, 1.62, 
            2.45, 2.1, 1.66, 2.7, 0.42, 1.95, 1.24, 0.73, 1.87, 
            1.03, 1.28, 1.04, 1.04, 0.97, 2.5, 1.46, 1.87, 1.03, 
            2.38, 0.83, 1.15, 1.46, 1.76, 2.76, 1.95, 1.43, 1.45, 
            1.24, 1.62, 1.46, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 0.73, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 
            1.4, 1.35, 1.31, 1.42, 1.99, 1.42, 1.76, 1.63, 2.08, 
            2.49, 1.95, 1.31, 1.43, 1.81, 1.04, 1.54, 1.35, 1.61, 
            0.83, 1.87, 1.83, 1.35, 1.71, 0.94, 2.91, 1.35, 1.77, 
            1.46, 1.9, 1.42, 0.94, 0.94, 0.83, 0.84, 1.25, 0.94, 
            0.8, 1.1, 0.88, 0.55, 0.75, 0.42, 0.55, 1.25, 1.14, 
            1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 0.73, 
            0.97, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.98, 1.06, 
            0.94, 1.25, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 
            1.24, 1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 
            4.38, 5.68, 7.8, 7.2, 6.75, 5.05, 5.05, 5.4, 7.22, 
            5.75, 7.7, 5.6, 8.5, 7.5, 5, 7.3, 6.3, 8.7, 5.1, 
            5.65, 4.5, 5.1, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 
            4.7, 5.7, 6.75, 8.9, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            7.1, 8.9, 7.2, 4.36, 8.7, 6.3, 5.85, 6.25, 4.9, 5.24, 
            6.8, 3.17, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 
            4.68, 3.17, 2.85, 8.42, 3.38, 3.74, 2.15, 3.21, 3.8, 
            4.6, 4.38, 7.8, 3.52, 2.5, 3.9, 3.25, 4.8, 3.05, 
            2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 
            3.25, 2.6, 4.4, 2.9, 2.45, 1.28, 3.8, 2.9, 2.8, 3.08, 
            2.94, 2.7, 3.4, 3.3, 2.7, 3.05, 2.9, 2.45, 2.8, 3.08, 
            2.9, 1.9, 2.62, 3.7, 3.4, 1.28, 3.25, 3.3, 3.38, 
            5.24, 2.8, 2.8, 3.94, 3.25, 2.12, 7.2, 3.58, 10.26, 
            4.4, 5, 5.45, 7.1, 6.62, 5, 5.7, 4.92, 5, 8.5, 4.35, 
            4.4, 8.21, 4, 4.9, 5.7, 8.42, 9.4, 3.93, 5.7, 7.1, 
            10.52, 7.6, 7.9, 10.26, 7.5, 13, 4.8, 7.65, 6.25, 
            5.58, 10.68, 5.58, 6.62, 10.68, 10.26, 9.4, 8.5, 
            4.7, 2.8, 9.7, 7.7, 7.3, 9.3, 9.3, 9.2), hue = c(1.04, 
            1.05, 0.84, 0.86, 1.07, 1.05, 1.02, 1.11, 1.08, 1.22, 
            1.25, 1.08, 1.15, 1.25, 1.2, 1.28, 1.04, 1.13, 1.23, 
            1.08, 1.09, 0.9, 1.11, 1.09, 1.12, 1.13, 1.1, 1.02, 
            1.22, 1.04, 1.19, 1.09, 0.8, 1.25, 1.1, 0.88, 0.8, 
            0.88, 1.18, 1.1, 0.95, 0.91, 0.88, 0.82, 1.07, 0.87, 
            1.23, 1.04, 1.07, 1.12, 1.12, 1.25, 0.86, 1.2, 1.02, 
            0.906, 0.94, 1.07, 0.89, 0.906, 1.25, 0.98, 1.23, 
            1.22, 1.27, 1.1, 1.08, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 0.87, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 
            0.89, 1.31, 0.84, 1.06, 1.23, 1.1, 1.08, 0.89, 1.07, 
            1.08, 1.05, 1.13, 1.15, 1.1, 1.25, 0.92, 1.31, 1, 
            1.42, 1.27, 1.23, 0.8, 0.87, 1.04, 0.69, 1, 0.96, 
            0.86, 1, 0.81, 0.9, 1.18, 1.1, 0.93, 1.71, 0.95, 
            1.1, 0.7, 0.86, 0.8, 1.12, 0.92, 0.73, 0.8, 0.86, 
            0.69, 0.97, 0.89, 0.79, 1, 0.74, 0.66, 0.78, 0.75, 
            0.73, 0.75, 0.82, 0.81, 0.84, 0.61, 0.98, 0.89, 0.74, 
            0.65, 0.6, 0.906, 0.54, 0.55, 0.82, 1.19, 0.48, 0.61, 
            0.56, 0.87, 0.6, 0.57, 0.67, 0.57, 1.25, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.66, 0.74, 0.67, 1.23, 
            0.9, 0.54, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
            3.4, 3.17, 3.71, 2.93, 2.84, 3.58, 3.58, 2.85, 3.55, 
            3.17, 2.82, 2.9, 3.17, 3, 2.78, 2.65, 2.83, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 2.71, 3.2, 3, 2.77, 2.71, 
            3.17, 2.71, 2.88, 2.87, 3, 2.87, 3.26, 2.93, 3.57, 
            2.69, 2.84, 3.38, 3, 3.69, 3, 3.35, 3.33, 3.71, 3.33, 
            2.75, 3.1, 2.3, 2.87, 3.26, 2.93, 3.2, 3.03, 3.5, 
            2.84, 2.87, 1.82, 3.14, 1.59, 2.46, 2.87, 3.57, 2.3, 
            3.18, 3.48, 2.77, 2.87, 1.82, 2.83, 1.59, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.72, 2.81, 2.63, 3.14, 2.72, 2.01, 
            3.14, 2.77, 3.05, 3.21, 2.75, 3.21, 2.75, 2.65, 2.72, 
            3.3, 2.87, 2.63, 1.78, 2.74, 3.5, 2.83, 2.96, 2.77, 
            3.38, 2.96, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 
            2.78, 2.5, 3.14, 3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 
            3.39, 3.69, 3.12, 2.15, 3.64, 3.03, 2.84, 2.82, 2.78, 
            2.05, 2.96, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
            1.82, 2.15, 1.58, 1.75, 1.58, 2.05, 2, 2.12, 1.33, 
            1.51, 1.62, 1.33, 2.72, 1.47, 1.33, 1.51, 1.55, 1.48, 
            1.64, 1.73, 1.96, 1.78, 1.58, 2.46, 1.68, 1.75, 1.68, 
            1.75, 1.73, 1.96, 1.82, 1.27, 1.58, 1.36, 1.56, 1.82, 
            1.69, 1.56, 1.62, 1.6), prol = c(1065, 1050, 740, 
            1045, 735, 1450, 1290, 1295, 1045, 1045, 1015, 1280, 
            1320, 1450, 1547, 1310, 1280, 1310, 1680, 845, 1065, 
            770, 1035, 1015, 845, 770, 1195, 1285, 915, 1035, 
            1285, 770, 1190, 1235, 870, 920, 880, 660, 880, 760, 
            795, 1035, 1095, 680, 885, 1080, 1065, 985, 1060, 
            1045, 1150, 1265, 1190, 1285, 1060, 1120, 970, 1270, 
            1285, 380, 410, 660, 630, 660, 880, 1310, 502, 1035, 
            1035, 718, 870, 410, 472, 502, 886, 415, 392, 680, 
            1120, 463, 278, 714, 392, 515, 886, 500, 740, 562, 
            680, 625, 480, 450, 495, 562, 345, 590, 625, 428, 
            660, 406, 1035, 562, 438, 415, 672, 660, 510, 378, 
            312, 680, 562, 325, 718, 434, 385, 466, 466, 550, 
            372, 564, 615, 1195, 365, 380, 625, 378, 1290, 466, 
            680, 580, 1015, 630, 560, 600, 312, 695, 720, 515, 
            580, 590, 600, 450, 520, 550, 428, 830, 415, 625, 
            695, 550, 500, 480, 425, 675, 640, 562, 480, 880, 
            660, 580, 562, 680, 570, 735, 615, 615, 695, 685, 
            718, 660, 510, 470, 345, 740, 428, 835, 840, 1035
            )), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.39, 13.24, 
        13.83, 14.39, 14.06, 14.21, 13.86, 13.17, 13.16, 13.75, 
        14.75, 14.38, 13.63, 14.3, 13.83, 14.38, 13.64, 13.73, 
        12.93, 13.71, 12.85, 13.5, 14.22, 13.39, 13.3, 13.87, 
        14.16, 13.73, 13.58, 13.68, 13.76, 13.39, 13.29, 13.84, 
        13.05, 13.48, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 
        14.21, 13.75, 13.9, 14.39, 13.76, 13.05, 14.75, 13.82, 
        13.17, 13.74, 14.39, 14.22, 13.29, 14.34, 12.37, 12.33, 
        12.64, 11.82, 12.37, 12.17, 12.37, 12.85, 12, 13.34, 
        12.21, 12.64, 13.86, 12.72, 12.99, 11.96, 11.66, 11.66, 
        11.84, 12.96, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.08, 
        12.16, 12.33, 12.07, 12.08, 12.08, 12, 12.69, 12.6, 11.62, 
        13.52, 11.81, 12.29, 13.24, 12.29, 12.08, 12.6, 12.29, 
        12.29, 12.51, 12.08, 11.82, 12.72, 12.22, 11.61, 12.07, 
        12.64, 11.76, 12.08, 12.08, 11.03, 11.82, 11.82, 12.37, 
        12.69, 11.45, 14.39, 12.42, 13.05, 11.87, 12.07, 12.07, 
        11.79, 12.6, 12.04, 12.86, 12.88, 13.58, 12.7, 12.51, 
        12.64, 12.25, 12.53, 12.7, 12.84, 12.93, 13.36, 13.52, 
        13.62, 12.25, 12.86, 12.42, 12.87, 13.32, 13.24, 12.16, 
        12.79, 12.87, 13.23, 12.93, 13.17, 13.84, 12.45, 14.34, 
        13.48, 13.23, 13.69, 12.85, 12.96, 13.78, 13.73, 13.27, 
        13.63, 13.58, 13.4, 12.6, 12.77, 14.16, 13.86, 13.4, 
        13.27, 13.17, 14.13), malic = c(1.71, 1.73, 1.83, 1.95, 
        2.55, 1.7, 1.87, 2.15, 1.67, 1.35, 2.16, 1.48, 1.01, 
        1.73, 1.87, 1.61, 1.92, 0.9, 1.01, 3.1, 1.63, 3.8, 1.83, 
        1.6, 1.81, 3.91, 1.53, 3.1, 1.9, 0.94, 1.5, 1.6, 1.83, 
        1.53, 1.92, 1.21, 1.35, 1.65, 1.5, 3.74, 1.71, 1.9, 2.15, 
        3.98, 0.99, 4.04, 1.92, 1.68, 1.53, 1.17, 1.17, 1.65, 
        3.37, 1.9, 1.67, 1.73, 1.7, 1.97, 2.39, 0.94, 3.74, 1.36, 
        1.25, 1.67, 3.88, 1.21, 1.01, 1.17, 5.65, 0.74, 1.53, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 4.3, 0.99, 3.87, 1.53, 
        3.45, 1.13, 2.55, 1.53, 0.98, 1.61, 1.67, 2.13, 1.13, 
        1.88, 1.51, 2.76, 2.68, 2.55, 2.76, 4.1, 1.41, 3.12, 
        1.71, 2.08, 3.24, 5.65, 1.72, 1.53, 2.55, 1.73, 1.7, 
        1.29, 3.88, 3.74, 4.12, 2.68, 0.74, 1.39, 1.68, 2.16, 
        1.61, 3.43, 3.59, 2.4, 2.05, 4.43, 1.67, 4.31, 2.16, 
        1.53, 2.13, 5.51, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 
        1.65, 3.98, 5.51, 3.59, 2.99, 2.81, 2.56, 3.91, 3.8, 
        3.88, 4.1, 5.04, 4.61, 3.24, 3.12, 3.12, 2.67, 1.9, 4.43, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 3.74, 4.04, 
        3.45, 2.76, 3.43, 1.67, 3.37, 2.58, 4.6, 1.9, 2.39, 3.24, 
        5.65, 3.91, 3.59, 2.59, 4.1), ash = c(2.43, 2.14, 2.67, 
        2.5, 2.26, 2.21, 2.45, 2.61, 2.52, 2.32, 2.3, 2.68, 2.41, 
        2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.1, 2.65, 2.68, 
        2.52, 2.61, 3.22, 2.54, 2.35, 2.36, 2.28, 2.7, 2.36, 
        2.4, 2.7, 2.65, 2.54, 2.38, 2.55, 2.1, 2.51, 2.25, 2.12, 
        2.59, 2.29, 2.1, 2.44, 2.21, 2.12, 2.4, 2.27, 2.04, 2.92, 
        2.48, 2, 2.5, 2.69, 2.3, 2.68, 2.5, 1.36, 1.82, 1.98, 
        1.92, 2.16, 2.53, 2.56, 2.51, 1.92, 2.27, 1.75, 2.21, 
        2.67, 2.24, 2.6, 2.3, 2.38, 1.71, 2.3, 2.58, 2.4, 2, 
        2.72, 2.51, 2.32, 2.58, 2.24, 2.04, 2.62, 2.36, 2.3, 
        2.32, 2.35, 2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.28, 
        2.28, 1.7, 2.36, 2.46, 1.88, 2.46, 2.27, 2.12, 2.28, 
        1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.24, 
        1.94, 2, 2.42, 3.23, 2, 2.13, 2.39, 2.17, 2.29, 2.38, 
        2.4, 2.38, 2.4, 2.4, 2.4, 2.36, 2.25, 2.51, 2.54, 2.64, 
        2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.72, 2.4, 2.48, 
        2.6, 2.36, 2.62, 2.48, 2.17, 2.28, 2.32, 2.32, 2.38, 
        2.2, 2.5, 2.64, 2.38, 2.54, 2.53, 2.35, 2.3, 2.27, 2.6, 
        2.19, 2.69, 2.61, 2.32, 2.28, 2.92, 2.48, 2.48, 2.26, 
        2.37, 2.74), alca = c(15.6, 15.2, 16.8, 16.8, 21, 15.2, 
        16.2, 17.6, 14, 16, 16.3, 16.8, 16, 11.4, 12.4, 20, 16, 
        20, 14, 15.2, 16, 18.6, 19.6, 17.8, 20, 21.5, 16.1, 17, 
        14, 16, 20, 19.1, 16.8, 20.5, 19, 20.5, 15.5, 18, 15.5, 
        13.2, 16.2, 18.8, 19.1, 20.5, 17, 19.5, 16, 16.8, 18.8, 
        17.4, 12.4, 16.8, 13.2, 15.5, 15.2, 20.5, 16.3, 16.8, 
        15.5, 10.6, 18, 16.8, 18, 19, 19, 18.1, 22.5, 19.6, 21.5, 
        16.8, 20.4, 18.5, 24, 30, 21, 24.5, 16, 18, 14.8, 19, 
        19, 24, 24, 22.5, 18, 18, 22.8, 23, 21.6, 21, 18.5, 22, 
        21.5, 18, 18, 19, 21.5, 16, 20, 18, 17.5, 18.5, 21, 19.5, 
        20.5, 22, 19, 22.5, 19, 20, 19.5, 16, 24.5, 25, 20, 21.5, 
        20.8, 19.5, 16, 21.5, 21, 20.5, 17.8, 21.5, 21, 21, 21.5, 
        28.5, 19, 22, 16, 20, 24, 21.5, 21.6, 21, 22, 25, 19.5, 
        21.5, 21, 18, 23.5, 20, 28.5, 21, 20, 25, 21.6, 21.5, 
        24, 22, 16.8, 18.5, 20, 22, 19.5, 18.6, 25, 18.6, 19.6, 
        24, 25, 18.5, 16, 22.5, 23, 19.5, 24.5, 25, 28.5, 19.5, 
        21, 20.5, 23, 20, 23.5, 24.5), mg = c(127, 100, 101, 
        113, 100, 112, 96, 121, 97, 108, 111, 101, 89, 91, 102, 
        112, 120, 117, 112, 116, 126, 102, 126, 95, 101, 124, 
        93, 139, 107, 96, 101, 106, 104, 116, 110, 110, 110, 
        107, 108, 111, 117, 111, 101, 113, 107, 111, 102, 101, 
        101, 108, 92, 124, 111, 115, 95, 110, 118, 102, 108, 
        88, 101, 100, 94, 87, 104, 98, 78, 78, 112, 151, 88, 
        86, 87, 139, 101, 84, 103, 112, 94, 97, 86, 105, 78, 
        85, 99, 99, 90, 88, 84, 70, 81, 87, 97, 88, 101, 88, 
        134, 85, 88, 97, 97, 88, 110, 86, 85, 90, 80, 84, 92, 
        94, 107, 88, 103, 88, 84, 85, 86, 85, 102, 87, 96, 93, 
        102, 86, 93, 85, 88, 92, 88, 80, 122, 88, 98, 106, 86, 
        94, 89, 112, 88, 101, 85, 89, 97, 86, 112, 102, 111, 
        111, 92, 113, 123, 95, 116, 98, 103, 106, 89, 97, 98, 
        89, 120, 95, 106, 106, 90, 88, 86, 88, 105, 112, 96, 
        86, 91, 95, 102, 112, 120, 85), phe = c(2.8, 2.65, 2.8, 
        3.85, 2.8, 3.1, 2.5, 2.6, 2.86, 2.98, 2.95, 2.2, 2.6, 
        3.1, 3.5, 2.85, 2.8, 2.95, 3.3, 2.8, 3, 2.41, 2.61, 2.48, 
        2.53, 2.61, 2.61, 2.4, 2.95, 2.65, 3, 3.18, 2.42, 2.95, 
        2.9, 2.7, 2.6, 2.45, 2.61, 3, 3.15, 2.45, 3.25, 2.64, 
        3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.4, 
        2.95, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 1.95, 3.5, 
        1.89, 2.45, 2.56, 2.11, 2.53, 2.8, 1.1, 2.95, 1.88, 2.45, 
        2.13, 1.88, 2.2, 2, 1.9, 2.83, 2.53, 2.2, 1.1, 1.65, 
        2.56, 2.2, 1.7, 1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 
        3.02, 2.95, 1.38, 2.55, 3.52, 3, 2.53, 2.1, 2.56, 2.5, 
        2.2, 2.2, 1.65, 1.51, 2.36, 2.74, 3.18, 2.55, 1.75, 1.83, 
        2.56, 2.46, 1.1, 2, 1.38, 2, 2.9, 3.18, 2.64, 0.98, 2.86, 
        2.05, 2.74, 2.13, 2.22, 2.1, 1.51, 1.9, 1.7, 1.7, 1.39, 
        1.62, 1.38, 1.95, 1.62, 0.98, 1.48, 1.4, 1.9, 2, 1.38, 
        1.7, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.6, 1.48, 
        1.7, 1.8, 1.9, 2.13, 1.51, 2.3, 1.83, 1.38, 1.39, 1.35, 
        1.68, 1.7, 1.48, 1.55, 1.75, 1.25, 1.39, 1.68, 1.65, 
        1.68, 1.35, 1.65, 2.05), fla = c(3.25, 2.76, 3.56, 3.49, 
        2.11, 3.39, 2.52, 2.51, 2.98, 3.56, 3.32, 2.43, 2.76, 
        2.65, 3.64, 2.91, 3.14, 3.4, 2.97, 2.91, 3.4, 2.41, 2.88, 
        2.37, 2.61, 2.68, 2.94, 3.15, 2.97, 2.33, 3.25, 3.19, 
        2.69, 3, 2.92, 2.98, 2.76, 2.43, 2.64, 3.04, 3.29, 2.68, 
        3.56, 2.63, 3, 2.65, 3.17, 3.03, 2.92, 3.54, 2.9, 2.99, 
        3.74, 2.79, 2.9, 2.78, 3.39, 3.23, 3.67, 0.57, 1.69, 
        1.41, 1.79, 2.97, 1.75, 2.65, 2.21, 2, 1.3, 3.74, 1.02, 
        2.86, 1.58, 2.89, 2.14, 1.57, 2.65, 1.31, 1.85, 2.14, 
        2.26, 2.53, 1.58, 1.59, 2.21, 2.11, 1.69, 1.61, 1.69, 
        2.29, 1.79, 1.25, 1.46, 3.15, 2.17, 2.27, 0.99, 2.11, 
        2.52, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 
        1.76, 2.26, 2.92, 2.63, 2.27, 2.03, 2.01, 2.29, 2.17, 
        1.6, 2.09, 0.47, 1.64, 2.92, 3.4, 1.92, 0.68, 3.03, 2.65, 
        3.15, 2.24, 1.57, 1.75, 2.24, 1.59, 0.84, 1.2, 0.58, 
        0.66, 0.47, 0.57, 0.96, 0.6, 0.5, 0.83, 0.52, 0.8, 1.59, 
        0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
        0.58, 0.63, 1.46, 0.58, 1.31, 1.1, 0.92, 0.66, 0.6, 0.47, 
        0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.6, 0.7, 0.61, 
        0.66, 0.69, 0.68, 0.76), nfla = c(0.28, 0.22, 0.3, 0.24, 
        0.39, 0.27, 0.43, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 
        0.43, 0.29, 0.3, 0.2, 0.39, 0.32, 0.17, 0.24, 0.25, 0.27, 
        0.26, 0.3, 0.42, 0.34, 0.27, 0.22, 0.26, 0.29, 0.26, 
        0.42, 0.27, 0.29, 0.26, 0.27, 0.25, 0.28, 0.2, 0.34, 
        0.24, 0.21, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 
        0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.29, 0.22, 
        0.28, 0.17, 0.53, 0.32, 0.24, 0.45, 0.37, 0.26, 0.27, 
        0.52, 0.28, 0.5, 0.28, 0.3, 0.21, 0.13, 0.34, 0.24, 0.43, 
        0.35, 0.43, 0.3, 0.32, 0.4, 0.61, 0.22, 0.3, 0.35, 0.29, 
        0.17, 0.43, 0.29, 0.5, 0.42, 0.25, 0.17, 0.32, 0.48, 
        0.24, 0.24, 0.27, 0.29, 0.29, 0.34, 0.37, 0.32, 0.66, 
        0.43, 0.48, 0.39, 0.29, 0.24, 0.26, 0.27, 0.42, 0.32, 
        0.52, 0.3, 0.24, 0.29, 0.4, 0.32, 0.29, 0.43, 0.3, 0.4, 
        0.29, 0.39, 0.66, 0.4, 0.5, 0.2, 0.5, 0.27, 0.17, 0.6, 
        0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.4, 0.53, 0.4, 0.53, 
        0.43, 0.4, 0.47, 0.45, 0.5, 0.22, 0.39, 0.26, 0.61, 0.53, 
        0.63, 0.43, 0.63, 0.53, 0.52, 0.35, 0.5, 0.6, 0.4, 0.41, 
        0.6, 0.43, 0.4, 0.39, 0.53, 0.43, 0.48, 0.63, 0.63, 0.43, 
        0.43, 0.53, 0.63), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 
        1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.99, 
        2.96, 1.46, 1.87, 2.96, 1.86, 1.66, 2.1, 1.35, 1.69, 
        1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 
        1.36, 1.98, 1.54, 1.86, 1.36, 1.44, 1.37, 2.76, 2.76, 
        1.15, 2.76, 1.66, 2.03, 0.73, 2.19, 2.14, 2.38, 2.08, 
        1.35, 2.29, 1.87, 1.68, 1.62, 2.45, 2.18, 1.66, 2.08, 
        0.42, 1.46, 1.38, 0.73, 1.87, 1.03, 2.08, 1.66, 1.04, 
        0.97, 2.5, 1.46, 1.87, 1.03, 0.73, 1.46, 1.15, 1.46, 
        1.35, 2.76, 1.95, 1.43, 1.76, 1.35, 1.62, 1.35, 1.46, 
        1.56, 1.34, 1.35, 1.38, 1.64, 1.63, 1.56, 1.99, 1.35, 
        3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 
        1.44, 1.42, 1.14, 1.63, 2.08, 2.49, 1.38, 1.86, 1.46, 
        1.77, 1.04, 1.98, 1.46, 1.61, 0.83, 1.87, 1.83, 1.85, 
        1.71, 1.46, 2.91, 1.35, 1.77, 1.45, 1.9, 1.04, 0.94, 
        1.42, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 0.64, 0.75, 
        1.46, 0.55, 0.8, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 1.25, 
        1.26, 1.56, 1.35, 1.1, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 
        0.75, 1.56, 0.94, 1.15, 1.15, 1.46, 0.97, 1.54, 1.11, 
        0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35), col = c(5.64, 
        4.38, 5.68, 7.8, 5.65, 6.75, 5.75, 5.05, 7.7, 7.22, 5.75, 
        5.1, 5.6, 10.26, 7.5, 3.05, 7.8, 7.9, 8.7, 5.1, 5.65, 
        4.5, 2.15, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 
        5.6, 6.13, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
        4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 7.5, 8.9, 7.2, 
        7.6, 5.1, 6.3, 5.85, 6.25, 7.8, 5, 6.8, 7.7, 3.27, 5.75, 
        3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 7.1, 3.38, 
        3.74, 1.9, 3.21, 3.8, 4.6, 2.9, 7.5, 2.8, 2.5, 3.9, 2.8, 
        4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 
        2.15, 3.25, 2.6, 2.9, 2.9, 4.5, 2.62, 2.5, 6.3, 2.8, 
        4.9, 2.94, 2.7, 3.4, 3.3, 2.7, 2.5, 2.9, 4.45, 2.62, 
        3.08, 2.9, 1.9, 3.05, 2.7, 3.4, 1.28, 3.25, 6.62, 2.7, 
        7.5, 2.8, 4.6, 3.94, 2.15, 2.12, 3.25, 5.65, 5.45, 10.26, 
        5, 5.45, 7.1, 5.7, 5, 5.7, 4.92, 7.7, 10.26, 4.35, 4.4, 
        8.21, 4, 4.9, 4.35, 8.42, 9.4, 1.9, 10.52, 7.1, 10.52, 
        7.6, 7.9, 10.26, 7.5, 13, 7.1, 7.65, 2.6, 5.58, 5.43, 
        10.26, 6.62, 10.68, 10.26, 6.8, 8.5, 7.3, 4.8, 9.7, 7.7, 
        7.3, 5.45, 9.3, 9.2), hue = c(1.04, 1.05, 1.09, 0.86, 
        0.86, 1.05, 1.02, 0.89, 1.08, 0.95, 1.25, 1.13, 1.15, 
        1.25, 1.2, 1.28, 0.89, 1.13, 1.23, 0.9, 1.09, 0.906, 
        1.11, 1.09, 1.12, 1.13, 1.27, 1.02, 1.04, 1.04, 1.19, 
        1.09, 1.07, 1.25, 1.1, 1.05, 1.05, 0.97, 1.18, 0.84, 
        0.95, 0.91, 0.88, 0.82, 1.11, 0.87, 1.09, 1.19, 1.07, 
        1.12, 1.12, 0.79, 1.23, 0.94, 0.89, 1, 0.94, 1.07, 0.89, 
        0.86, 1.25, 0.98, 1.23, 1.22, 1.25, 0.8, 1.04, 1.12, 
        1.02, 1.28, 0.906, 1.36, 0.98, 1.31, 1.28, 1.23, 1.19, 
        0.96, 1.06, 1.19, 1.38, 0.79, 1.31, 0.84, 1.19, 1.23, 
        0.69, 1.38, 1.1, 1.07, 1.08, 1.05, 1.05, 1.15, 1.05, 
        1.04, 0.87, 1.07, 1.23, 1.42, 1.27, 0.8, 0.8, 0.86, 1.04, 
        1.05, 1, 0.86, 0.86, 0.86, 1.12, 0.9, 1.23, 1.1, 0.93, 
        1.71, 0.95, 1.23, 0.7, 1.38, 0.8, 0.92, 0.92, 0.73, 1.08, 
        0.86, 0.69, 0.97, 0.89, 0.79, 0.96, 0.74, 0.66, 0.78, 
        0.75, 0.73, 0.75, 0.82, 0.81, 0.6, 0.82, 0.61, 0.89, 
        0.96, 0.65, 0.6, 0.56, 0.54, 0.55, 0.56, 1.13, 0.48, 
        0.61, 0.56, 0.61, 0.6, 0.57, 0.67, 0.57, 0.89, 0.56, 
        0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.64, 0.74, 0.67, 
        0.78, 0.89, 0.67, 0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 
        3.4, 3.17, 2.75, 2.93, 3.19, 3.58, 3.58, 2.85, 3.55, 
        3.17, 2.82, 2.9, 2.93, 3, 3.26, 2.65, 3.58, 2.82, 3.36, 
        3.71, 3.52, 4, 3.63, 3.3, 3.2, 2.84, 2.77, 3.69, 2.87, 
        2.71, 2.88, 2.87, 3, 2.87, 3.17, 2.87, 3.4, 2.69, 2.84, 
        3.38, 3, 3.26, 3, 3.35, 3.33, 4, 3.33, 2.75, 3.1, 3.02, 
        3.38, 3.26, 2.93, 3.2, 3.03, 3.05, 2.84, 2.87, 1.82, 
        2.14, 1.59, 2.46, 2.87, 3.3, 2.3, 3.18, 3.48, 2.65, 4, 
        1.82, 2.87, 2.78, 3.5, 3.13, 2.14, 2.48, 2.52, 2.77, 
        3.03, 3.26, 3.14, 2.72, 2.01, 3.03, 2.81, 2.87, 3.21, 
        2.75, 3.21, 2.87, 2.65, 2.82, 3.3, 3.05, 2.63, 2.77, 
        2.74, 2.87, 2.83, 2.96, 2.77, 3.38, 2.14, 3.57, 3.3, 
        3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.63, 3.19, 
        2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 1.27, 
        3.64, 2.87, 2.84, 2.87, 2.78, 2.65, 3.12, 1.42, 1.36, 
        1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.6, 1.68, 
        1.59, 2.05, 2, 1.59, 1.33, 1.59, 1.62, 1.33, 3.17, 1.47, 
        1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 
        2.05, 1.33, 1.75, 1.68, 1.75, 1.73, 1.62, 1.82, 1.6, 
        1.56, 1.78, 1.96, 1.48, 1.82, 1.56, 1.62, 1.6), prol = c(1065, 
        1050, 1320, 1310, 735, 1450, 1290, 1295, 1045, 1045, 
        1060, 1280, 1320, 1280, 1547, 1310, 1280, 1190, 1680, 
        845, 1050, 770, 1035, 1015, 845, 1190, 1195, 1285, 915, 
        1035, 1285, 845, 1285, 1235, 970, 920, 880, 1035, 1035, 
        760, 795, 1035, 1095, 680, 885, 1080, 1065, 985, 1060, 
        1280, 1150, 1265, 1190, 1065, 1060, 1120, 970, 1270, 
        1285, 562, 564, 520, 630, 515, 660, 720, 502, 472, 562, 
        718, 870, 410, 472, 795, 886, 470, 392, 480, 740, 463, 
        278, 714, 560, 515, 515, 550, 564, 562, 680, 625, 480, 
        450, 495, 562, 345, 562, 625, 428, 660, 406, 472, 562, 
        438, 415, 672, 562, 510, 472, 312, 680, 562, 325, 615, 
        434, 385, 410, 372, 450, 372, 564, 680, 1265, 365, 380, 
        680, 378, 562, 466, 564, 580, 1095, 502, 560, 600, 590, 
        695, 720, 515, 580, 590, 600, 770, 520, 550, 434, 830, 
        415, 372, 560, 550, 500, 480, 425, 675, 640, 510, 480, 
        880, 660, 695, 870, 680, 570, 325, 615, 463, 695, 685, 
        562, 570, 510, 470, 1050, 740, 550, 835, 840, 410)), row.names = c(NA, 
        -178L), class = "data.frame")), res.conv = structure(c(0.00313838593076876, 
    0.253322393184982, 0.0562098152788901, 0.234016711415312, 
    0.240639525310072, 0.199804579641206, 0.249640387321862, 
    0.292668560028394, 0.211856491662472, 0.00819887694586096, 
    0.0117186819724589, 0.00129559737109677, 0.0219283616605123, 
    0.020193509538226, 0.194790220999689, 0.0253669704541279, 
    0.00273444295090807, 0.00701808738584665, 0.00708003102938139, 
    0.226530924590098, 0.0103468912300088, 0.257667012625652, 
    0.0131541559998879, 0.0192432993390795, 0.0245689559845072, 
    0.259308921264739, 0.00893754127459996, 0.00886797902388601, 
    0.247227886857966, NA, 0.00331598153150732, 0.18705262714498, 
    0.026603320196102, 0.00585439741884315, 0.0124100138567878, 
    0.0316777402259306, 0.000514586113730367, 0.0274738497186967, 
    0.209958850964376, 0.00609215320144656, 0.0153943603976094, 
    0.013849108710276, 0.0192916619034996, 0.0188689636563801, 
    0.00858200450346639, 0.00267398590129897, 0.0314504914892749, 
    0.0154674786764182, 0.00450351213228692, NA, 0.00186499018390939, 
    0.0601816848865689, 0.0117796130146782, 0.011492208728687, 
    NA, 0.0392829625697147, 0.00133481098744819, 0.00428248467106998, 
    0.0438264933225768, NA, 0.0732498171817024, 0.00616110699116423, 
    0.011713034604881, 0.0169548544291527, 0.0146396643991983, 
    0.00377526114262564, 0.0122612018188745, 0.0603194860181543, 
    NA, NA, 0.00258527860763537, 0.0116400419401329, 0.0129208530292205, 
    0.0324649911449701, NA, 0.0137678494929091, 0.002754070017762, 
    0.00185240002460797, 0.0749777995174497, NA, 0.20982054442009, 
    0.00310067785930431, 0.00949707286599839, 0.0182731063030496, 
    0.00331108197581929, 0.0117548571338847, 0.0221547673092988, 
    0.0325285471242315, NA, NA, 0.0032462044067625, 0.00192271493992216, 
    0.0392598337843288, 0.0309355502318104, NA, 0.0114701923716319, 
    0.00118386888218286, 0.00485186765639367, 0.0153153410870997, 
    NA, 0.00094163301498425, 0.0117147714383694, 0.00976094187313365, 
    0.0094950769465697, 0.0252814148269123, 0.000754587680449373, 
    0.0134711392240477, 0.0362262335533237, NA, NA, 0.0022639009905561, 
    0.0304044037858913, 0.00983393394722621, 0.0424421071733603, 
    NA, 0.0613985369869622, 0.00166469260928216, 0.00524342666879698, 
    NA, NA, 0.0312811576061835, 0.00326897012409308, 0.00432136233778966, 
    NA, 0.0205757170080526, 0.252772940608592, 0.0221608856259166, 
    0.0603862227292128, NA, NA, 0.0139465740660017, 0.0297360793621588, 
    0.0174891428482706, 0.04197110463478, NA, 0.0427397412630165, 
    0.00114892484964386, 0.00463600059092153, NA, NA, 0.0357430554532056, 
    0.00372312199678486, 0.0169412527724659, NA, 0.00439617700554495, 
    0.258944736484694, 0.0151817006828155, 0.00299176744594524, 
    NA, NA, 0.00489250611749427, 0.0264864850440421, 0.00967761677456238, 
    0.0232362562371492, NA, 0.0393440404027667, 0.00157676963800426, 
    0.000365738850994135, NA, NA, 0.0200335644889163, 0.00339283137822732, 
    0.00924286842477656, NA, 0.0146804126045095, 0.244305066462735, 
    0.0139765648399033, 0.000707216896951617, NA, NA, 0.0443555643805498, 
    0.0319942538228809, 0.0247696692381125, 0.00500472667242822, 
    NA, 0.031456697033813, 5.93256583044844e-05, 0.00705356466135106, 
    NA, NA, 0.00793617323934232, 0.0167464855269179, 0.00702167045664215, 
    NA, 0.00310922715408216, 0.233785523518927, 0.0237006351326242, 
    0.000481561041660011, NA, NA, 0.075881950522974, 0.0432373821479529, 
    0.0298366581992794, 0.00335014529888995, NA, 0.00812951983021508, 
    0.00612222499298668, 0.00967848646572976, NA, NA, 0.0103176031899321, 
    0.0056588062261895, 0.00698380773001035, NA, 0.0109419426993335, 
    0.0353823903225083, 0.00711142340510531, 0.0045552091826138, 
    NA, NA, 0.0692147516454442, 0.0216872607037789, 0.00800374751157056, 
    0.0079333384745064, NA, 0.00260040437601283, 4.96951514979248e-05, 
    0.00161501212301269, NA, NA, 0.0252357306560278, 0.0130437326803467, 
    0.00128879796716533, NA, 0.000612447359611561, 0.133792487878668, 
    0.0231045185102777, 0.00250197935774939, NA, NA, 0.0822524510955946, 
    0.0111900863231691, 0.0111910304844987, 0.00834540319978944, 
    NA, 0.0104733607772268, 0.00322943508243595, 0.0136644239258291, 
    NA, NA, 0.00811676562002155, 0.00398070825012766, 0.00318183569585349, 
    NA, 0.00228222376309144, 0.0502051196406763, 0.0145634299888556, 
    0.0198667239934392, NA, NA, 0.0693083177264788, 0.00945937532977552, 
    0.0225078993462222, 0.00295020828039764, NA, 0.00572593774600956, 
    0.00232538114813944, 0.00194516231887995, NA, NA, 0.0175765150676814, 
    0.00417271043070166, 0.00360986237275192, NA, 0.00206521269550258, 
    0.0058811987738162, 0.014383714587167, 0.0137435991528991, 
    NA, NA, 0.0853370355743777, 0.0183768133239169, 0.0109852054255572, 
    0.00518661856903441, NA, 0.0181682164683362, 0.00124326253297804, 
    0.0143026734133602, NA, NA, 0.00348780674697364, 0.00384519789878244, 
    0.00301528814891274, NA, 0.00138118972198948, 0.000977614793096245, 
    0.0173526084420556, 0.00654484641620116, NA, NA, 0.0732725361713605, 
    0.065390688267647, 0.0202265449265574, 0.00449169531534596, 
    NA, 0.01620340667872, 0.00118053743448513, 0.0024380413340107, 
    NA, NA, 0.00301875076274036, 0.0218531587410212, 0.00728989506447017, 
    NA, 0.00188851761038629, 0.00845605114708854, 0.0119780564668176, 
    0.00427434817321393, NA, NA, 0.0755472970489406, 0.0108280002571349, 
    0.0202342428262721, 0.00365979938497432, NA, 0.000488076471555845, 
    0.00240663433236371, 9.68716216872378e-05, NA, NA, 0.00262587598667799, 
    0.0124335219199166, 0.0315880360342037, NA, 0.00817736909130509, 
    0.00622301276727425, 0.00869025367132447, 0.00296008173173946, 
    NA, NA, 0.103039392768817, NA, NA, 0.0126346907004813, NA, 
    0.00191057641408641, 0.000239709031149095, 0.00258996240656809, 
    NA, NA, NA, 0.0026497236798, 0.00623359723184821, NA, 0.00905939605142927, 
    NA, 0.0170903969989367, 0.00603140043344697, NA, NA, 0.0562393968555665, 
    NA, NA, 0.0422242838136767, NA, 0.00738121551099479, 0.00504750862498272, 
    0.0113930303425897, NA, NA, NA, 0.00112617318946029, 0.0158318438612537, 
    NA, 0.0338030431235047, NA, 0.00351070876619114, 0.0079571979047129, 
    NA, NA, 0.0633521817705255, NA, NA, 0.00107673313692665, 
    NA, 0.00303938630255229, 0.00382175337913643, 0.221571655462496, 
    NA, NA, NA, 0.00379016942950422, 0.0280895512119376, NA, 
    0.0128413061218411, NA, 0.00389254855432399, 0.00761000441394832, 
    NA, NA, 0.0894518676050929, NA, NA, 0.00554508730279442, 
    NA, 0.0115031668079426, 0.0251351368699211, 0.0115517136561352, 
    NA, NA, 0.0888796657372985, 0.053926079658429, 0.00827459897840117, 
    0.165791383039701, 0.0994931346678016, 0.170907517584044, 
    0.0822877051075708, 0.0943915402467134, 0.154590236202639, 
    0.000545240687610624, 0.057830985658862, 0.021855226627311, 
    0.0360590456834154, 0.00420321241087438, 0.108752465901594, 
    0.0153584994860101, 0.0245146084348133, 0.0120290414561614, 
    0.0460452698159704, 0.102676631937945, 0.294669374849365, 
    0.074775248908893, 0.0239535783176485, 0.00723497667888831, 
    0.00337074633790054, 0.118343686494459, 0.0528939514046845, 
    0.0317059800462094, 0.183028353329273, NA, 0.164543634935867, 
    0.121783582561286, 0.0188929207052437, 0.00608876192149313, 
    0.00996236368392686, 0.0423659353408028, 0.0154158550018219, 
    0.0129159261610649, 0.111139951640155, 0.00226841308598821, 
    0.0037159448979892, 0.0305118416042238, 0.0347749231379265, 
    0.0615540901579689, 0.0155109423924196, 0.0831877052552968, 
    0.0005345846509452, 0.00233971623748713, 0.0221939318186478, 
    NA, 0.125957572702289, 0.0393662781610135, 0.0117960933747594, 
    0.00117235595312489, NA, 0.00739658623691684, 0.0964406026661419, 
    0.00942681476024607, 0.0139979403851173, NA, 0.118805839764348, 
    0.0361820425335243, 0.0290171979215809, 0.0171532780050054, 
    0.0167558341181619, 0.0873343305796651, 0.0310762655734509, 
    0.0366507545117407, NA, NA, 0.158386789042992, 0.0460452221887022, 
    0.0290338240251715, 0.0166659541106959, NA, 0.0399152578378151, 
    0.121809686195689, 0.0391574359551237, 0.101905456816095, 
    NA, 0.00237714549078038, 0.0106140111545166, 0.042529699534885, 
    0.0138063996582656, 0.00118710017402921, 0.120791817549902, 
    0.0466562296953476, 0.0511679481899411, NA, NA, 0.0846901268526503, 
    0.0359641117617063, 0.0197109006890322, 0.00483160377257583, 
    NA, 0.02332239492339, 0.133559489471864, 0.0223720030611143, 
    0.0214424047632921, NA, 0.0158799945769749, 0.00241606933431719, 
    0.0106453102514227, 0.018931614956462, 0.00499623562153212, 
    0.0636687516383271, 0.0523730526776316, 0.115299482234932, 
    NA, NA, 0.0702322018172313, 0.0309254450483821, 0.0033694110986122, 
    0.00955666288694805, NA, 0.0223386561556799, 0.136037883252289, 
    0.00644986749421167, NA, NA, 0.0085056781159246, 0.00438530487460734, 
    0.00441909211438853, NA, 0.0140393225970533, 0.0983530170072854, 
    0.0951537784413727, 0.146475793381518, NA, NA, 0.0655787123983715, 
    0.0527729352074699, 0.00650734878946474, 0.0485679904260866, 
    NA, 0.00583387043577505, 0.140904110888586, 0.0291172580829931, 
    NA, NA, 0.0261356804721731, 0.0350990969499164, 0.0195060257247679, 
    NA, 0.0208806548831155, 0.0979054625899257, 0.0889969100338275, 
    0.00235052597360682, NA, NA, 0.0728111154099499, 0.0201575394594231, 
    0.0123453234021332, 0.00121391390641269, NA, 0.00874207276880215, 
    0.0917545067362942, 0.0119575234581693, NA, NA, 0.0264216596673201, 
    0.016353098460821, 0.00311344860369409, NA, 0.0445837319142294, 
    0.0736263513719989, 0.0523750950797577, 0.0148665282894407, 
    NA, NA, 0.00294360670475109, 0.0169603898669316, 0.0124380982652032, 
    0.000846187077783939, NA, 0.0278778997099149, 0.185257726748647, 
    0.0393546654510038, NA, NA, 0.00383467213849889, 0.0253821939791103, 
    0.0153146503807706, NA, 0.0204699708778669, 0.0474386452417574, 
    0.166979688001931, 0.00289983801915375, NA, NA, 0.00302740716768532, 
    0.026095217530231, 0.0235621767880579, 0.000529069076095468, 
    NA, 0.0261876284300026, 0.138917486360841, 0.0452265094795907, 
    NA, NA, 0.0418132636723704, 0.0231550199684301, 0.0238814296983364, 
    NA, 0.00172487304832819, 0.0191721280245236, 0.146812850678365, 
    0.0175132574052545, NA, NA, 0.00903423463697866, 0.0273267972390779, 
    0.000386997155568526, 0.0108191033734682, NA, 0.0443808740230134, 
    0.127725254416959, 0.0273542147179913, NA, NA, 0.00745157061921619, 
    0.012846627970011, 0.0626449651177777, NA, 0.00567111937518451, 
    0.00846383788481516, 0.0695244903497316, 0.00521383033481178, 
    NA, NA, 0.00340111776873551, 0.0252574408398441, 0.00377164104327573, 
    0.00328125424153734, NA, 0.0118305722088407, 0.149012978577094, 
    0.0157251646412323, NA, NA, 5.49730573540699e-05, 0.0273463027165426, 
    0.0193644683652908, NA, 0.00334546633624944, 0.0188508546250845, 
    0.0230464566574275, 0.023918854076964, NA, NA, 0.0180960549617758, 
    0.0050198684643482, 0.0172725768469092, 0.0136763295058185, 
    NA, 0.00765177989695712, 0.142587182468874, 0.0039817985638241, 
    NA, NA, 0.0100688880608864, 0.0183274166844681, 0.0411652488614273, 
    NA, 0.0084761517528605, 0.116100399696216, 0.0158623378590857, 
    0.0540470102856957, NA, NA, 0.0330264223516695, 0.00871158572942438, 
    0.010596365322318, 0.00193802642861516, NA, 0.0270701910712794, 
    0.138526567710745, 0.0135873670473784, NA, NA, 0.0119076799435132, 
    0.00194958606654758, 0.074484002468716, NA, 0.00809729067218579, 
    0.0741225698210754, 0.00809199135729501, 0.0273081316451316, 
    NA, NA, 0.075104581560041, 0.00889122993606701, 0.00760090483501355, 
    0.0986882037226174, NA, 0.0150358080274095, 0.131689425182907, 
    0.0191046929732695, NA, NA, 0.0157770060072381, 0.0165106033581537, 
    0.0686513596755306, NA, 0.00199229120491684, 0.0814296587193828, 
    0.0145588190791904, 0.0236645914202777, NA, NA, 0.0505426693298916, 
    0.0339036604330461, 0.01156880374805, 0.0196200162362615, 
    NA, 0.0498821546575287, 0.159008280654056, 0.0457454586299321, 
    NA, NA, 0.0071473190915048, 0.01397939018587, 0.00938986158181732, 
    NA, 0.0153101743130556, 0.016023810009358, 0.0240856939782081, 
    0.0954860771824117, NA, NA, 0.0238895811561067, NA, NA, 0.0176183373303225, 
    NA, 0.0996194493319524, 0.165304391781137, 0.0265314653804854, 
    NA, NA, NA, 0.00221633617900187, 0.00100574640132533, NA, 
    0.00325683554190233, NA, 0.0336652562306938, 0.0751331658160044, 
    NA, NA, 0.104969262433142, NA, NA, 0.0290345756425151, NA, 
    0.0980790023939219, 0.17079716632004, 0.0243289706243895, 
    NA, NA, NA, 0.0471541613971595, 0.0125401049525576, NA, 0.019790179892199, 
    NA, 0.0648380578966316, 0.0467963697244003, NA, NA, 0.0498989924544016, 
    NA, NA, 0.0198273967400812, NA, 0.0814477510060771, 0.146124833845263, 
    0.0669777023698551, NA, NA, NA, 0.016498779102757, 0.00671934311091182, 
    NA, 0.00141341383090072, NA, 0.0556802669081577, 0.0447050636562259, 
    NA, NA, 0.107351603844881, NA, NA, 0.0277420904488023, NA, 
    0.0509631332824339, 0.232182677173753, 0.011996788308877, 
    NA, NA, 0.00355887183884009, 0.00336432612706764, 0.00949445157561673, 
    0.00156642642067428, 0.00141282469185647, 0.0017626824706035, 
    0.00212271928698911, 0.00381300499325338, 0.00301930351580176, 
    0.00132139182604319, 0.00340923424236341, 0.000490712975591781, 
    0.00269137080361732, 0.00199083935893088, 0.00489951027931961, 
    0.00101691537415553, 0.00164403956613462, 0.00136038691918587, 
    0.00261858538083416, 0.00319469662340904, 0.000504921492733157, 
    0.0037095636569073, 0.00972452347354711, 0.00268923664752371, 
    0.00107215782690224, 0.00455590444788557, 0.0007042698842544, 
    0.00274477544778606, 0.00230959773026, NA, 0.000602630892397728, 
    0.0033018366667314, 0.00239072585400438, 0.00485809231974573, 
    0.00155486125003642, 0.00372516603654976, 0.00500361282946878, 
    0.000809492860588053, 0.00227792633103215, 0.00205243140323465, 
    0.000352599907028506, 0.00447613682733064, 0.00844019314436279, 
    0.00301175173357884, 0.00142460943114664, 0.010382058648524, 
    0.00381523993938482, 0.00275085033277788, 0.00389558231749874, 
    NA, 0.000492710626972575, 0.00249104669074095, 0.00349682983159263, 
    0.00369961117739776, NA, 0.00475773487552174, 0.00437311926752713, 
    0.00117997842295209, 0.00244478017001744, NA, 0.00114023168685404, 
    0.00215065754933417, 0.00536459103410431, 0.000697482634971532, 
    0.00365482968055234, 0.0130015020172947, 0.00322741242360423, 
    0.00458534237039643, NA, NA, 0.000943275542944069, 0.000539454491348821, 
    0.00481322425018486, 0.00470787210254292, NA, 0.00321629547510837, 
    0.00526065014490877, 0.00384183819394326, 0.00391902552467201, 
    NA, 0.00516330385975057, 0.00331906144043721, 0.00533654273724736, 
    0.00245815483788657, 0.0100411495254016, 0.0130135048667421, 
    0.0061429816452575, 0.00508964996713167, NA, NA, 0.00125212878271836, 
    0.000556380900674286, 0.00564499590220334, 0.00494944830151397, 
    NA, 0.00647760379996935, 0.00555199668907884, 0.00295320279602567, 
    0.00341151190210287, NA, 0.000762614308911761, 0.00304095410810672, 
    0.00452940884074839, 0.00427848102815072, 0.00562250381202316, 
    0.00890993034750004, 0.0025838099798839, 0.00710401577003834, 
    NA, NA, 0.00108541879417191, 0.00464965220422213, 0.00517866344373562, 
    0.00466156060901083, NA, 0.00438920092014971, 0.00587544719784323, 
    0.00137475596334376, NA, NA, 0.0011868857283754, 0.00251923060071144, 
    0.00185597788463273, NA, 0.00124403013122226, 0.00160690881021891, 
    0.00310143980900082, 0.0103082232541762, NA, NA, 0.00450772517928606, 
    0.00555085119030033, 0.00441854200186272, 0.00362493967182462, 
    NA, 0.00591619743822169, 0.00710337730695979, 0.000875188483407317, 
    NA, NA, 1.58243591539509e-05, 0.00120809411317977, 0.00510781725484907, 
    NA, 0.00146784998114626, 0.00331160081149622, 0.00275535581701469, 
    0.0087009781320486, NA, NA, 0.00118972704906767, 0.00354115705348692, 
    0.00355493363010195, 0.00294927923510124, NA, 0.00501171956296208, 
    0.0048127441526005, 0.00113118235419498, NA, NA, 4.06228156078395e-05, 
    0.00118548036864033, 0.00469419620670881, NA, 0.00263092048208773, 
    0.00472601594408306, 0.000808738764381052, 0.00560123926763224, 
    NA, NA, 0.00626311501073098, 0.00278454533346418, 0.00139491054560963, 
    0.000457060450698698, NA, 0.0045142380607307, 0.00781290404693537, 
    0.00390749843637666, NA, NA, 0.00130140997472862, 0.00244379474778261, 
    0.00136612993454596, NA, 0.00270317925178699, 0.00163906886398543, 
    0.00246050149281073, 0.00596098870691233, NA, NA, 0.00847682242222525, 
    0.00419969123600782, 0.00272056351488, 0.00267430818817555, 
    NA, 0.00272551641105805, 0.00861894184072162, 0.00348540147512428, 
    NA, NA, 0.00143616449222193, 0.00145695507020719, 0.00159863056795332, 
    NA, 0.00387721314562057, 2.70153596035739e-05, 0.00134696751422503, 
    0.0109540577852882, NA, NA, 0.0079270016666803, 0.00599683164539877, 
    0.00139383243204617, 0.000745179606572305, NA, 0.00578445852817096, 
    0.00643619285280471, 0.00271080861733557, NA, NA, 0.000261776700269613, 
    0.000847052750182026, 0.00172397066868661, NA, 0.00386765635722365, 
    0.000181891093873901, 0.00184033519732852, 0.00787344021013476, 
    NA, NA, 0.00982257054438222, 0.00966797223052327, 0.00171939365717899, 
    0.00108274861174883, NA, 0.00467388335363518, 0.00783907337048473, 
    0.00343822348669494, NA, NA, 0.00224320756758707, 0.00148137487927272, 
    0.00244178691619649, NA, 0.00252430787723592, 0.000783619247758782, 
    0.00238087342561071, 0.00564384148565321, NA, NA, 0.0111418089955926, 
    0.00876096295164234, 0.00269226407520243, 0.000847252205070309, 
    NA, 0.00506773372304097, 0.00695596908977116, 0.00644641381905812, 
    NA, NA, 0.00296839984036116, 0.00151623959983284, 0.00704935787245033, 
    NA, 0.00549550710361599, 0.000399614451638498, 0.00160112627346562, 
    0.00480535968415209, NA, NA, 0.00874712039109974, 0.00558179921748916, 
    0.00234239881559688, 0.000720071137812562, NA, 0.000652686910751652, 
    0.0115007559253894, 0.00608292631284047, NA, NA, 0.0006921443740114, 
    0.00198668707673696, 0.00473896045223396, NA, 0.00306093924250762, 
    0.00319350929440011, 0.00132833168865544, 0.00782649853434404, 
    NA, NA, 0.011506714131129, 0.00509353899472201, 0.00503311477087529, 
    0.000406262697425348, NA, 0.00098275847782539, 0.0072697096134856, 
    0.00525884859737817, NA, NA, 0.000199689485755165, 0.00410165886679084, 
    0.00586059485061748, NA, 0.00664683752212799, 0.00510690278526757, 
    0.000979173273598939, 0.00908705832736234, NA, NA, 0.0110681261685333, 
    0.005399056701971, 0.00313769066571871, 0.000445389353821239, 
    NA, 0.000948241567666946, 0.00653800537204158, 0.00743821408524563, 
    NA, NA, 0.00214044661268117, 0.00437978138751204, 0.00504517180281674, 
    NA, 0.00336657355184295, 0.0016827858726875, 0.00453207913101773, 
    0.00568731650287599, NA, NA, 0.0154005198795233, NA, NA, 
    0.00109772749645419, NA, 0.0017220006016137, 0.0102727742216417, 
    0.00561482686176471, NA, NA, NA, 0.00792447029928913, 0.00255260648449928, 
    NA, 0.00234171593754919, NA, 0.00270004306507719, 0.00509424315412316, 
    NA, NA, 0.0119594646406271, NA, NA, 0.00146766161028366, 
    NA, 0.00235121381158718, 0.00902938059771394, 0.0106192547497873, 
    NA, NA, NA, 0.0073739567993663, 0.00601794026736226, NA, 
    0.00358372380977258, NA, 0.00224420213694176, 0.00515527391394052, 
    NA, NA, 0.0132364856041806, NA, NA, 0.00233941485062189, 
    NA, 0.000855763594944587, 0.00627668930023343, 0.00535260964446676, 
    NA, NA, NA, 0.010486507337777, 0.00428788698014721, NA, 0.00235196474228362, 
    NA, 0.00289778373738402, 0.00604594426992494, NA, NA, 0.0123349148588657, 
    NA, NA, 0.00227402083489437, NA, 0.00244282644286937, 0.0063243556978641, 
    0.00180131341960771, NA, NA, 0.186845433916513, 2.67820064769184, 
    0.377042445371348, 2.19753249401054, 2.43299170138981, 2.02235670537643, 
    2.61182094326487, 2.08952456707343, 2.01561585545874, 0.176778855506301, 
    0.0675604181435914, 0.121244777910134, 0.184048896805501, 
    0.25787072638465, 2.20928090198083, 0.193779753268655, 0.209085644429412, 
    0.00130335947220912, 0.0774888490861144, 2.57113687861599, 
    0.115736998162591, 2.68309816389011, 0.192493190215849, 0.177846540474065, 
    0.0628562837104387, 2.13258029635012, 0.201631741048651, 
    0.1914219527604, 2.0512979977535, NA, 0.0432034742619104, 
    2.16852646850792, 0.300206878988379, 0.231068096859057, 0.028934787668855, 
    0.0100274711936238, 1.17550909566767, 0.0571578237337296, 
    2.14103439954875, 0.0176705341327865, 0.514385517902765, 
    0.141097645569505, 0.205601258567498, 0.150139362782226, 
    0.0357374675646522, 0.496247201073601, 0.115147049021105, 
    0.0550920369337745, 0.0142143880611244, NA, 0.0244685987863408, 
    0.0880236799228868, 0.370526949744547, 0.160428489946667, 
    NA, 0.0468248669862808, 0.0326034568951136, 0.283991257416992, 
    0.197952771585243, NA, 0.100778880080498, 0.0174087468637856, 
    0.0051393820169347, 0.0857833768270277, 0.0330460924694028, 
    0.669557606019678, 0.255600654393617, 0.151503384583376, 
    NA, NA, 0.00218152950307881, 0.858913196658998, 0.443348514062267, 
    0.240297341687475, NA, 0.129307100850062, 0.191070870749518, 
    0.00108778946441024, 0.262387659015128, NA, 2.85066704262543, 
    0.199706152426092, 0.13358751850335, 0.179829803769899, 0.178366765483852, 
    1.03435918520549, 0.204235785425726, 0.0342810107799955, 
    NA, NA, 0.00118182728502036, 0.141744066850287, 0.841175677065835, 
    0.0385853655679456, NA, 0.289168171585457, 0.241181868079905, 
    0.00265371269726782, 0.0171688993138747, NA, 0.0881296799646151, 
    0.183642401974525, 0.164053491710754, 0.183704305304759, 
    0.0514525145698523, 0.608085383134252, 0.273239244901827, 
    0.0175789966489255, NA, NA, 0.00307911044081094, 0.366892120502852, 
    0.511917020068358, 0.00802235419893137, NA, 0.271636532007681, 
    0.0979210321645107, 0.0267512840994411, NA, NA, 0.237609331054967, 
    0.0936863456782789, 0.330371497543918, NA, 0.0265753779971754, 
    2.76680468185408, 0.116076118064523, 0.149086076537304, NA, 
    NA, 0.0123437114142314, 0.483360004624567, 0.410258335525728, 
    0.149517158513029, NA, 0.0744377747505719, 0.0233304574848033, 
    0.0288348836782584, NA, NA, 0.0827848323691466, 0.0169438800065524, 
    0.195840184104356, NA, 0.025836779447089, 2.17380189552234, 
    0.166650864717939, 0.0586437855079439, NA, NA, 0.0278640486119595, 
    0.458318698794393, 0.288903208688439, 0.30449683750411, NA, 
    0.304453089316175, 0.0606912833861579, 0.046867668597448, 
    NA, NA, 0.151812901880767, 0.092058476047912, 0.219433689456448, 
    NA, 0.0891922714120011, 2.30576724254026, 0.573052223006379, 
    0.0314968445737338, NA, NA, 0.424243243698186, 0.334912548826885, 
    0.308210706703355, 0.16598206791313, NA, 0.311838422115833, 
    0.0511130972721695, 0.330607030836711, NA, NA, 0.0200314494612971, 
    0.148898723153188, 0.17359412531352, NA, 0.0273857821454882, 
    1.82167069635482, 0.31733952575567, 0.168274594117745, NA, 
    NA, 0.330187600989916, 0.0994363523736909, 0.531244767517835, 
    0.113861454497063, NA, 0.237960617100734, 0.12157411548796, 
    0.0962791241010615, NA, NA, 0.0175826209258248, 0.0106253745112069, 
    0.242968622128816, NA, 0.0985497017873956, 0.355833417019584, 
    0.128443297051073, 0.0816215069254808, NA, NA, 0.56803536528538, 
    0.291543606206529, 0.112437858314601, 0.195496261235054, 
    NA, 0.153642912181437, 0.330649137196164, 0.0796258850469494, 
    NA, NA, 0.0942669932704616, 0.0194924496163775, 0.152166834338047, 
    NA, 0.0774208892520976, 1.24712041881307, 0.0977844451610484, 
    0.00813348624985394, NA, NA, 0.65862266807804, 0.0583365944397121, 
    0.3136298785748, 0.246584766974011, NA, 0.0579249814890344, 
    0.134485667980971, 0.174928674809683, NA, NA, 0.169603240732391, 
    0.109838046957549, 0.164964630548241, NA, 0.113806116251899, 
    0.243693650179885, 0.407474218568537, 0.00903098894263769, 
    NA, NA, 0.58649007788937, 0.272988035865071, 0.0437935066462769, 
    0.0951044146802931, NA, 0.079584690022766, 0.104813720987631, 
    0.0143278266872709, NA, NA, 0.0463108868522746, 0.115912676834341, 
    0.0930151486613067, NA, 0.0457414043172007, 0.0509362079405052, 
    0.291942361533527, 0.0736253118063689, NA, NA, 0.593190046030998, 
    0.174339333603315, 0.10048585142199, 0.254901138111189, NA, 
    0.324027576110259, 0.123617739288376, 0.0137384939682627, 
    NA, NA, 0.283698213124413, 0.0409429158734584, 0.189380462511505, 
    NA, 0.25281206425973, 0.0472185070575544, 0.108923709708822, 
    0.125515762867708, NA, NA, 0.794716405441031, 0.762173334657323, 
    0.178180640670012, 0.186517689406919, NA, 0.13429959800292, 
    0.0268809895981347, 0.0539326961701347, NA, NA, 0.107300024372624, 
    0.195821300371071, 0.150743336781691, NA, 0.0107731388694485, 
    0.0868414179744169, 0.23117193658344, 0.250848703423234, 
    NA, NA, 0.478810185285616, 0.0384812365505862, 0.00274343406489967, 
    0.0711139276996059, NA, 0.0707300688509811, 0.00637016916135242, 
    0.128461866217357, NA, NA, 0.200582061848852, 0.178218462279549, 
    0.09733539932306, NA, 0.0729318925943015, 0.401447969766574, 
    0.13481149625727, 0.00891147668506444, NA, NA, 0.440868344298471, 
    NA, NA, 0.0814162164066518, NA, 0.0342762466383831, 0.0437869021222554, 
    0.125537908409781, NA, NA, NA, 0.0952423551294898, 0.552163604683551, 
    NA, 0.00860437584306872, NA, 0.154705710141328, 0.059730296198772, 
    NA, NA, 0.365806630531553, NA, NA, 0.1212342616278, NA, 0.0915739160339374, 
    0.0549051324383658, 0.134472808203143, NA, NA, NA, 0.0173113435070892, 
    0.351013774250098, NA, 0.0706521762834927, NA, 0.364242746133858, 
    0.0197240339415538, NA, NA, 0.414413697779784, NA, NA, 0.0172738632796969, 
    NA, 0.0325904213814898, 0.0702482848014206, 1.93647053976117, 
    NA, NA, NA, 0.00677917869565901, 0.301010439576058, NA, 0.0140924949122613, 
    NA, 0.0829443722175051, 0.0258902927355753, NA, NA, 0.531272833987176, 
    NA, NA, 0.137211615081216, NA, 0.0203728044599147, 0.00597819139500348, 
    0.00279550398941337, NA, NA, 100.31251283665, 47.0322110204228, 
    14.6978350304273, 44.3112949790097, 46.3832065055623, 47.5829091120974, 
    73.7434177944596, 45.3150716326586, 53.9366839204816, 51.7096885633567, 
    73.2841707831719, 149.699369142506, 61.1004633351324, 87.0800157478301, 
    49.7760198577287, 28.7561429685047, 77.0164687765241, 70.8998639753843, 
    64.7272378101811, 36.6206469562214, 60.4041906863284, 58.3765267388093, 
    59.2137622925048, 74.1586545855849, 67.2060876594115, 43.7553300379711, 
    128.203197949356, 79.6309992100129, 41.6643854085392, NA, 
    74.9508801669049, 34.7820850692253, 79.0943777765171, 84.1846248709685, 
    58.7481209273304, 30.9750915747206, 93.4968590717171, 28.6702825417019, 
    40.2641745964662, 62.1508949168494, 66.7296853746937, 50.8857489931021, 
    55.2508336803607, 65.6557643108385, 96.2525821667659, 10.1976797666883, 
    113.041099150849, 40.0239079710091, 65.4803071600509, NA, 
    71.8537218444705, 62.686558428123, 66.6854710392242, 86.6210537521979, 
    NA, 27.9551194273424, 71.7290538321315, 96.3919274261907, 
    74.4272740032838, NA, 25.3520924896322, 66.665519962213, 
    74.2841853461928, 73.7876970204806, 87.9252404298117, 11.5482475993385, 
    94.9107767990418, 60.4478351920388, NA, NA, 73.6652474036388, 
    35.8465404309111, 61.556495024172, 59.803533855153, NA, 4.12683920380423, 
    44.3828721864692, 92.5924314352223, 73.1659622956733, NA, 
    53.8252027867626, 64.8656251719706, 82.9239470564482, 70.6610468856145, 
    87.3342322300184, 11.4727790571131, 94.0071950853426, 45.6837922053116, 
    NA, NA, 76.8759711531277, 76.3156802829349, 73.5122508841916, 
    52.4676829138709, NA, 7.10648905969857, 61.4774877169667, 
    92.2503796964723, 51.5498396852064, NA, 86.4880802165615, 
    66.3523805012704, 66.8769436952826, 58.4104868437814, 87.030275645313, 
    21.6517137837584, 74.7078138433153, 51.6728210403391, NA, 
    NA, 84.6558339155302, 52.7623671969476, 73.8996331838277, 
    50.2046215704592, NA, 90.5312513237072, 55.1254987875572, 
    90.7407333611096, NA, NA, 81.2484080007131, 60.4803186986195, 
    70.1164464457843, NA, 63.8467243998891, 40.1507283161293, 
    62.5982835938617, 62.449141231341, NA, NA, 64.0797078441223, 
    69.4647578570654, 64.9307830287174, 62.7566385747162, NA, 
    80.2930044601507, 56.861657682482, 90.1352478647063, NA, 
    NA, 77.188648807746, 73.8221466306427, 68.6954455159227, 
    NA, 68.2679443567399, 43.4851646000892, 67.0734860422063, 
    62.782980904242, NA, NA, 53.6303255888025, 86.3881536908079, 
    72.5377559273708, 55.7444937153484, NA, 67.1136873427421, 
    57.0627192428743, 76.148443693543, NA, NA, 88.7145965227095, 
    62.6690514408011, 75.1508170372747, NA, 74.1658809801476, 
    50.4641360397743, 66.8945857640663, 86.4442744580069, NA, 
    NA, 46.0528216134326, 96.7934068072861, 63.2162996210385, 
    65.1648808149423, NA, 64.1187222419481, 63.4122640799263, 
    59.0526216136569, NA, NA, 81.2909097856851, 81.8912838749024, 
    59.5227617124701, NA, 71.9762936293715, 49.2582659017563, 
    61.2426931381428, 50.7789896519337, NA, NA, 39.1516349354606, 
    80.3827474010607, 74.5721682228478, 60.9042610105557, NA, 
    69.7682408429504, 66.8228917798147, 90.8852571807068, NA, 
    NA, 80.7622862134466, 92.1950934390505, 95.7239133502158, 
    NA, 74.5942741825431, 64.2575818034803, 78.2251412843824, 
    36.6912196268321, NA, NA, 52.3238982467608, 75.8581466788377, 
    70.0773473590721, 63.0736421044957, NA, 64.7082778335074, 
    60.8536964760356, 77.8089170414666, NA, NA, 77.9218734760663, 
    69.417516597993, 81.1220515481, NA, 62.2282619948003, 53.4609361913768, 
    77.9016071759421, 70.5463499348613, NA, NA, 70.82408637863, 
    70.0505444588092, 68.5719211785448, 84.4514139782546, NA, 
    62.6802002377724, 55.1560406019097, 81.9455446184985, NA, 
    NA, 78.3594369397803, 74.7459591420946, 57.6085025175219, 
    NA, 66.5937809754079, 68.6266101515838, 72.3290282696305, 
    43.4237391753731, NA, NA, 58.6256119893054, 60.1907583602159, 
    62.3894036536701, 90.9264502114558, NA, 76.3796932745045, 
    53.0783625878964, 93.2874273082768, NA, NA, 77.1472393882656, 
    76.1273092480224, 69.68250841333, NA, 79.3426861879437, 50.0962892035341, 
    77.883398305744, 40.5656513389235, NA, NA, 53.0154525739752, 
    99.3352412432074, 65.8790580931763, 81.3854851653049, NA, 
    58.3980008771065, 57.9621914771749, 83.5349435524288, NA, 
    NA, 71.3941827945242, 83.1862739000828, 47.0679526557383, 
    NA, 71.6995554139011, 73.2919333333152, 79.6672793545877, 
    36.877403628328, NA, NA, 45.2012175217485, 19.1798212342004, 
    65.4354677296403, 77.8884948272664, NA, 67.6575541797014, 
    49.5880791181947, 81.3795477084112, NA, NA, 64.1517209001389, 
    36.2248530951778, 72.6274143400902, NA, 67.3784663388358, 
    60.3018476645691, 75.2348485417926, 41.6746313479866, NA, 
    NA, 57.4116500871724, 47.155395435611, 69.489375566172, 80.0407953440758, 
    NA, 56.7189016163495, 61.8513903918309, 69.5822675361503, 
    NA, NA, 60.2059338948305, 53.050816593496, 73.4265615281022, 
    NA, 66.820777174058, 79.0942536970836, 70.8803957412284, 
    61.6226308325489, NA, NA, 76.5188512602999, NA, NA, 52.4618534252597, 
    NA, 75.0322844948341, 56.0867303097349, 69.7711538469088, 
    NA, NA, NA, 64.9140352356612, 59.7359012653582, NA, 69.0035760455428, 
    NA, 78.2256073961323, 49.4015746268208, NA, NA, 72.2498708127855, 
    NA, NA, 51.09522662153, NA, 102.128611204286, 52.4783584783007, 
    68.1013962668503, NA, NA, NA, 66.0725147547002, 56.2460192999888, 
    NA, 70.1973047265392, NA, 69.4007634237517, 40.8865184195095, 
    NA, NA, 69.7966504465327, NA, NA, 45.3435401297076, NA, 108.753341718514, 
    63.5915480990942, 23.6089792601587, NA, NA, NA, 63.5253948654696, 
    61.5404964305506, NA, 66.0886703778699, NA, 68.3220870139361, 
    51.0863911561452, NA, NA, 74.5582579864562, NA, NA, 39.2511608290266, 
    NA, 85.3362634587984, 61.2617990356764, 37.4371287117171, 
    NA, NA, 0.00491284040864938, 0.154535575403326, 0.00431015672719418, 
    0.204011617227304, 0.183517447614391, 0.193163458714832, 
    0.168460162050916, 0.161850634594647, 0.167713891939996, 
    0.00146562898964611, 0.00565392786350684, 0.000525618867187704, 
    0.0250918574898508, 0.0108221132261557, 0.169997790154788, 
    0.00342228578648447, 0.0156294533881866, 0.00446961454723451, 
    0.00408311183928397, 0.151010344823958, 0.023380423733202, 
    0.150972127954597, 0.00145027838312319, 0.000147442439697945, 
    0.0138705082509707, 0.196914604396887, 0.00384207778359138, 
    0.0143733146325839, 0.171194811213612, NA, 0.00241824360453922, 
    0.165054224218161, 0.00725693235949079, 0.0141164072575754, 
    0.00237295526497237, 0.0101063491328819, 0.00282445048159989, 
    0.00953081008694837, 0.149902678016164, 0.00173993904771742, 
    0.00224655037195125, 0.00874957050355393, 0.00700869724933656, 
    0.00567818099227861, 0.0255433768113197, 0.00834163042043143, 
    0.00135470112280055, 0.025872720356487, 0.00319721566710583, 
    NA, 0.00274176641250381, 0.0384714325437766, 0.0083175999649077, 
    0.0148057194417473, NA, 0.00326897687850632, 0.00297903111083522, 
    0.00329416994409349, 0.0182363193192146, NA, 0.00201686879501929, 
    0.00889470608828271, 0.00140473843264739, 0.000317445526044081, 
    0.0143387210142047, 0.00479588031844289, 0.00595340195912496, 
    0.0394185049794656, NA, NA, 0.000455906621352449, 0.0466168491122563, 
    0.00417311960811556, 0.00360150664625952, NA, 0.0874972597946388, 
    0.00409618163971466, 0.00257516450466403, 0.0330926544485304, 
    NA, 0.166445951461354, 0.00168460342866235, 0.00102754570461818, 
    0.00120753798255123, 0.0217724497035158, 0.00609122857846954, 
    0.00645071817458769, 0.0299960703522903, NA, NA, 0.00383727437120925, 
    0.0170061321709945, 0.00361300890868633, 0.00216461141660905, 
    NA, 0.0707378432184548, 0.00176670296553013, 0.00818061617505502, 
    0.00117033377357818, NA, 0.000853699154805227, 0.0032914214046825, 
    0.00492820696820488, 0.000993427285122135, 0.019984991248911, 
    0.0119192073157218, 0.0246325980958777, 0.0418728455117088, 
    NA, NA, 0.0195982285362106, 0.00687716410485476, 0.00815795769990278, 
    0.00274153297942766, NA, 0.0243910451384073, 0.00138673652253876, 
    0.00909308280853873, NA, NA, 0.00428845099913621, 0.00551937020726736, 
    0.00292715629375533, NA, 0.0206437051379185, 0.137782723863329, 
    0.0146617869598248, 0.0362568210032403, NA, NA, 0.0108333398308405, 
    0.0188250490157235, 0.00393795238573659, 0.00931288261822424, 
    NA, 0.00335174816025998, 0.00328007120028048, 0.0166344326240898, 
    NA, NA, 0.00900244023336057, 0.0020734173666253, 0.000119520674501654, 
    NA, 0.0118198130561413, 0.162385993278233, 0.0103672627019979, 
    0.00732949513622623, NA, NA, 0.00905333334804075, 0.00723344000834865, 
    0.00294326361830147, 0.00957881876194482, NA, 0.00803847022944026, 
    0.000526304324834937, 0.00333520777521012, NA, NA, 0.0108790345479267, 
    0.00869525271022437, 0.00308637401093188, NA, 0.0109517348382371, 
    0.160318353429251, 0.0241692648789916, 0.0065296497476433, 
    NA, NA, 0.03872980400202, 0.00928003437623999, 0.0013624067342148, 
    0.00625106917781798, NA, 0.00174637167912145, 0.00672906413176802, 
    0.00742281998562358, NA, NA, 0.00793458847905548, 0.00942713541663393, 
    0.00137517298643158, NA, 0.0162427677124959, 0.158045622915631, 
    0.00931020045403148, 0.00558545374555589, NA, NA, 0.0459900848199174, 
    0.0116496235417008, 0.00283605358950335, 0.000830197870769927, 
    NA, 0.000959566424246076, 0.0034891592726693, 0.00237721938901722, 
    NA, NA, 0.00755964256476745, 0.00111945483504178, 0.000163075183614667, 
    NA, 0.00537031580688046, 0.0412630520354955, 0.00617941577544754, 
    0.0122683989213215, NA, NA, 0.0627668840933461, 0.0132504935504203, 
    0.00582872705365541, 0.000259834875466797, NA, 0.000686109373578036, 
    0.00422921138827113, 0.0048935454521378, NA, NA, 0.00169433287385633, 
    0.00143606257785387, 0.00923939714577283, NA, 0.00186270307168552, 
    0.0824489223442607, 0.0197514075682617, 0.00439066426768371, 
    NA, NA, 0.0482385824115707, 0.00791833607666852, 0.00231128297235979, 
    0.00143117053672682, NA, 0.0005074469691066, 0.00230279410630384, 
    0.00724014190244349, NA, NA, 0.000357093620935309, 0.00120889665682606, 
    0.0074123210907493, NA, 0.000481108838079133, 0.0445055375458995, 
    0.0105575118717498, 0.000172674873898719, NA, NA, 0.0486651647305248, 
    0.00427609246876863, 0.00174785360340193, 0.0013850788683704, 
    NA, 0.00375091688293541, 0.00339165568502492, 0.000232111919285215, 
    NA, NA, 0.000161363489690026, 0.00246446002649347, 0.0117917312738871, 
    NA, 0.00306900632481671, 0.00700137150690631, 0.0141968884336357, 
    0.00850117868778064, NA, NA, 0.0558602568107359, 0.0116889519530253, 
    0.00417579010582123, 0.00571387484977748, NA, 0.00304851518572627, 
    6.4004072952859e-05, 0.000976828951043072, NA, NA, 0.0024474977055995, 
    0.0086528559915075, 0.0111112886218832, NA, 0.008011269535235, 
    0.0142358516085201, 0.022634397060142, 0.0178097093342468, 
    NA, NA, 0.0562100855903998, 0.0394975423825875, 0.000803559618140902, 
    0.00141716548725651, NA, 0.00198345059580977, 0.00975634740581402, 
    0.00218248089386278, NA, NA, 0.00910566431181528, 0.00120284867814768, 
    0.00863529296386843, NA, 0.00218730154820079, 0.00456968141602222, 
    0.0123623169903918, 0.0155293771779905, NA, NA, 0.0316775815676349, 
    0.00515026236924756, 0.000226841475875439, 0.00319217634496533, 
    NA, 0.00513987635889111, 0.000307414841286407, 0.000102463500140812, 
    NA, NA, 0.00587973563342772, 0.00232349112826036, 0.0134546397510126, 
    NA, 0.00120796460636328, 0.00304234897607428, 0.00744133823400014, 
    0.0126087683706599, NA, NA, 0.046108796806195, NA, NA, 0.000838408177809902, 
    NA, 0.00813431762871125, 0.00184314009027734, 0.00879697251648673, 
    NA, NA, NA, 0.00804776814503646, 0.0284170501617303, NA, 
    0.00364831343037577, NA, 0.0112090652951208, 0.0169256871637179, 
    NA, NA, 0.033799022533786, NA, NA, 0.000708082257470254, 
    NA, 0.00191938794474637, 0.00324195999032362, 0.0033772757382011, 
    NA, NA, NA, 0.000779916374008999, 0.0163972247342478, NA, 
    0.00461664559134757, NA, 0.00792286557796412, 0.0132656930133657, 
    NA, NA, 0.0415521390615677, NA, NA, 0.000512853745634434, 
    NA, 0.000923397843803711, 0.00247165829190482, 0.172255583966662, 
    NA, NA, NA, 0.000804808587589903, 0.00862850153128009, NA, 
    0.00419883753199312, NA, 0.00639671391388208, 0.0269399848621988, 
    NA, NA, 0.0317854076486774, NA, NA, 0.00072903359142007, 
    NA, 0.0020230985746148, 0.000634567160069911, 0.0166874210117785, 
    NA, NA, 0.000598285095793597, 0.498698972290186, 0.0120971699011338, 
    0.52226034224098, 0.532693719768655, 0.510678069262198, 0.491360480132959, 
    0.451236238050408, 0.485953383787322, 0.00495051057003807, 
    0.0339895131908904, 0.00778427809538936, 0.089312479267915, 
    0.0423138546919209, 0.483298447084884, 0.00491693432865897, 
    0.0144751672775715, 0.0175858740499076, 0.00671840860917233, 
    0.468001823898208, 0.0283124616272925, 0.4854679797188, 0.0119345504611084, 
    0.00255982485335677, 0.0363328087707513, 0.496932873159749, 
    0.00101067653248857, 0.03068865244067, 0.466845727588078, 
    NA, 0.000930712553178586, 0.467275431747481, 0.037026714623317, 
    0.0539310451779026, 0.0100941887694468, 0.00587098440373056, 
    0.00282158852980063, 0.0484941191097019, 0.45657958859397, 
    0.0354960802931656, 0.00460450768168909, 0.000906567317401168, 
    0.0175289773688932, 0.0120676357693971, 0.0278870660543429, 
    0.0488023533272476, 0.0172955580213747, 0.04863780566599, 
    0.0182170080869487, NA, 7.2968737619588e-06, 0.149396622543368, 
    0.0421289143212957, 0.0471563512762343, NA, 0.000695454904619274, 
    0.0222318405978724, 0.00624417577992852, 0.103886126910283, 
    NA, 0.0676527831935326, 0.00685348228991173, 0.00311625739484427, 
    0.000435203161983118, 0.0290726479386895, 0.0561729902140547, 
    0.033622983728418, 0.0689391544905166, NA, NA, 0.00109047259841752, 
    0.0793772906149525, 0.0484398414132795, 0.0400211024341563, 
    NA, 0.152654365762951, 0.0289144949818164, 0.00767297189677883, 
    0.16353445477972, NA, 0.434006457685775, 0.0239585146574788, 
    0.0127456687313692, 0.000139460776357004, 0.0236139458749384, 
    0.0462562747842304, 0.0407934327603199, 0.0735776117629739, 
    NA, NA, 0.00304325235989143, 0.0170097554360658, 0.0410927562012795, 
    0.023189265862765, NA, 0.133439757184542, 0.0208725292188679, 
    0.0015622024631935, 0.0309569383661829, NA, 0.0147757176291085, 
    0.0279678505944792, 0.0248910941206943, 0.0130068523302225, 
    0.028379699934683, 0.060022434652914, 0.0613727863814415, 
    0.0980252535357378, NA, NA, 0.02222844828516, 0.0447726828201529, 
    0.0353355716179679, 0.0144094354169587, NA, 0.0628419230115301, 
    0.00314723702515274, 0.000671317481194377, NA, NA, 0.0534428875669545, 
    0.0125494534075466, 0.0170826115785978, NA, 0.0561662408716163, 
    0.471227638647503, 0.0633860469503551, 0.0837839756196107, 
    NA, NA, 0.0016815693388656, 0.0712347650345477, 0.0176446736720995, 
    0.0386340410986598, NA, 0.00764387224491494, 0.00238564834064114, 
    0.00787912019383329, NA, NA, 0.066307657657755, 0.0187668477828701, 
    0.0152661308874129, NA, 0.0422455620431955, 0.501757073117591, 
    0.0434479320564248, 0.0477361372586139, NA, NA, 0.00298474631691296, 
    0.0470134953579372, 0.0168425959606527, 0.0421211143288948, 
    NA, 0.0561800537064832, 0.00121408754469391, 0.000871514608508889, 
    NA, NA, 0.095790046790368, 0.0310069675546017, 0.023497142655926, 
    NA, 0.0481080899841931, 0.520940267527562, 0.0546510437709534, 
    0.0499707267806693, NA, NA, 0.101075469853344, 0.0642792778480852, 
    0.0174780868583352, 0.0235109179670736, NA, 0.0248478865528522, 
    0.0060082694796917, 0.0372131341291655, NA, NA, 0.0488565562299231, 
    0.0331277238728809, 0.0179755271260428, NA, 0.0618881715355496, 
    0.490214730741217, 0.0344642688575381, 0.0445119093124609, 
    NA, NA, 0.0908957472221085, 0.0471477062132389, 0.0313906474497308, 
    0.00917203515609115, NA, 0.00739460924917312, 0.00666959504444543, 
    0.0271613928528169, NA, NA, 0.07313758530922, 0.0221199409652343, 
    0.00305415673196013, NA, 0.0203437181644686, 0.0683247840416538, 
    0.0199105739956819, 0.0541334830310799, NA, NA, 0.133506683964829, 
    0.0530016972223953, 0.0256860457188433, 0.020050532817135, 
    NA, 0.0240556746610463, 0.00485134258656093, 0.0224336871116432, 
    NA, NA, 0.0339972474604517, 0.0106136758302465, 0.0489957333179032, 
    NA, 0.0227872195637992, 0.203222885125624, 0.0198041120808204, 
    0.0428880776207596, NA, NA, 0.11892430148356, 0.0456006179695378, 
    0.0319101233925415, 0.0121910508300416, NA, 0.0253502412075146, 
    0.0057932772955881, 0.0454548429227967, NA, NA, 0.0156834153135226, 
    0.00906426649742561, 0.0170133057309516, NA, 0.0143248031892517, 
    0.128637299705751, 0.0333077314550113, 0.0104295820721495, 
    NA, NA, 0.116861714703302, 0.0246496183618565, 0.0139816263393959, 
    0.0155675296945316, NA, 0.0169729208702651, 0.0179771792878182, 
    0.0391760755099008, NA, NA, 0.00732309095518777, 0.00635565550752728, 
    0.0306092914280742, NA, 0.0238732363971114, 0.0480580208391002, 
    0.0380230566745603, 0.0414315954513499, NA, NA, 0.176273625906348, 
    0.0301850600983413, 0.0322009993377154, 0.0195265802442251, 
    NA, 0.0368112959384662, 0.00141073006637892, 0.03473584312628, 
    NA, NA, 0.0113815351309897, 0.0232507470076638, 0.0262526515376352, 
    NA, 0.0393541507171321, 0.0385809674300516, 0.0457727748223802, 
    0.0575053292997434, NA, NA, 0.206287571687776, 0.0454727649868721, 
    0.0144494703368493, 0.0237515681563264, NA, 0.0539342614839181, 
    0.0243873829854462, 0.0431644600688981, NA, NA, 0.007552583471854, 
    0.0277076178274452, 0.0235666760627594, NA, 0.00605060080900519, 
    0.0569367442883992, 0.0215895820445707, 0.0377720145785598, 
    NA, NA, 0.179080669319897, 0.00964000498024611, 0.0131725001402741, 
    0.0157017745422832, NA, 0.0335807375597748, 0.001055691479624, 
    0.0264434528614297, NA, NA, 0.00424355112389609, 0.0309880918858002, 
    0.0519299835749688, NA, 0.0158678199729717, 0.0183641377644264, 
    0.0235237905611236, 0.0477155393661461, NA, NA, 0.214385194794231, 
    NA, NA, 0.00315904165197293, NA, 0.0231595768092391, 0.00314414977371368, 
    0.0621531052724399, NA, NA, NA, 0.0431492409661722, 0.0741280843797352, 
    NA, 0.0222745263404435, NA, 0.0177793473019766, 0.0288537355161987, 
    NA, NA, 0.155365566985201, NA, NA, 0.00180478386067445, NA, 
    0.012443904407272, 0.00156532574338449, 0.0325933164105448, 
    NA, NA, NA, 0.0238654225683301, 0.0644155309442958, NA, 0.0343340743412628, 
    NA, 0.0276442081510081, 0.0366428680207626, NA, NA, 0.13577598218993, 
    NA, NA, 0.00302789928679661, NA, 0.0198081270320885, 7.34371225855826e-05, 
    0.501375564858554, NA, NA, NA, 0.0234657603040243, 0.0616873001423932, 
    NA, 0.0206152442642494, NA, 0.033497038961096, 0.0523958558124555, 
    NA, NA, 0.131791394086176, NA, NA, 0.00374933969215045, NA, 
    0.0181808726050703, 0.00149182243998436, 0.0411936953450141, 
    NA, NA, 5.72098749613464e-05, 0.00355667248141021, 1.23515243314112e-05, 
    0.00332307372951575, 0.00395013291191669, 0.00320500393527694, 
    0.00386135572433698, 0.00367516835042011, 0.00372571914287196, 
    0.000593755044183091, 0.00133619732370514, 0.000405374978452343, 
    0.00122833849980064, 0.00052876494641117, 0.00290785791126111, 
    0.000195423870011233, 0.000121030762047412, 0.000286081967867259, 
    0.000341912195179381, 0.00411264124754943, 0.000185068012669483, 
    0.00375086953633763, 0.000491738130198032, 3.8387905278534e-05, 
    0.000186572571780318, 0.00321339252434221, 0.000187908886158676, 
    0.000814509732107034, 0.00252903939533137, NA, 7.0540931751886e-06, 
    0.00237708017716618, 0.000332110813886875, 0.00064685745542378, 
    7.08595335670513e-06, 0.000253104670516317, 7.16528213114069e-05, 
    0.00135716302250752, 0.0021675465788791, 0.000562694500508539, 
    0.000111583539985664, 1.71094704564209e-06, 0.000475754815466965, 
    0.000173578865741302, 9.34746437721126e-05, 0.00143580629145504, 
    0.00120620676785374, 9.21468326450037e-05, 0.000664310535163332, 
    NA, 4.13568359603527e-05, 0.000901738903276897, 0.00037984478433571, 
    0.000176047337337675, NA, 9.99556160041491e-05, 0.000256890275826559, 
    0.000555281655809998, 9.17931053484039e-05, NA, 0.00166371688709866, 
    0.000169373112648322, 0.000139943478306611, 0.000301901061138855, 
    0.000415573027568345, 0.00146167764917224, 0.00154463417350081, 
    4.52646213175459e-06, NA, NA, 3.52367010151098e-05, 0.000181078452751221, 
    0.000642536271005123, 0.000657229583424193, NA, 0.00059914131212389, 
    8.00203327724059e-05, 0.000321063369254004, 0.00132133161955145, 
    NA, 0.00301751550633386, 0.000394158073032601, 0.000307020999502029, 
    0.000233568506194437, 0.000339026591984835, 0.00127294653924513, 
    0.00113446416213038, 4.5296199184165e-05, NA, NA, 0.00033493534167962, 
    2.74828564310438e-06, 0.000792571038480698, 0.000315491221899873, 
    NA, 0.000480616208796847, 8.71200075431264e-05, 0.000395131807131554, 
    0.000392116982121527, NA, 0.000143353962676232, 0.000786177449163037, 
    0.000195831998443943, 0.000272364290525043, 0.000262736498767528, 
    0.00119545733015672, 0.00109660523628778, 0.000134658264617484, 
    NA, NA, 4.72123654200132e-05, 0.00107650959688764, 0.000618440010372644, 
    0.000201776135088454, NA, 0.000357405840840589, 0.000228882078771108, 
    0.000635368374918227, NA, NA, 0.000409841099963278, 0.000456905311272344, 
    0.000398546444616304, NA, 0.000153198179937112, 0.00241119458908892, 
    0.000568027585516134, 0.000333205519522647, NA, NA, 0.000291381017337551, 
    0.000855600121680316, 0.000485472072978267, 0.000593464464858451, 
    NA, 0.000146250461829835, 5.7897079077394e-05, 0.000254449644134288, 
    NA, NA, 0.000579490845145661, 0.00025460969929907, 0.000270658601891373, 
    NA, 0.000170142985913618, 0.00272098308112546, 0.00052949143253662, 
    0.000511290073695811, NA, NA, 1.4543360830511e-05, 0.00109815344164393, 
    0.000602262782386408, 0.00111202060261781, NA, 0.000902707912773186, 
    0.000147301267597939, 0.00048933082500352, NA, NA, 0.00113307544202787, 
    0.000603053211689928, 0.000443303991489266, NA, 0.000302041770877387, 
    0.00269862265252338, 0.000397330974401066, 0.000772938108969856, 
    NA, NA, 0.000195051424483441, 0.00136611713366866, 0.0004746044370173, 
    4.28973934847122e-05, NA, 0.000160101077749702, 5.32645256351902e-05, 
    0.000572240726401829, NA, NA, 0.000612962248323722, 0.000724202129960498, 
    0.00025905519827351, NA, 0.000388787722435555, 0.00337324241537603, 
    0.000705755011593019, 0.0003334767903056, NA, NA, 0.000471276404657888, 
    0.00147417649354787, 0.000660802874905284, 0.000193225631482951, 
    NA, 0.000509648728707428, 0.000297328974475747, 0.000426990131148697, 
    NA, NA, 0.00106532186831213, 0.000992884258193451, 0.000238650330097952, 
    NA, 0.00019348502265523, 0.000385471629746342, 0.000628571228279313, 
    0.000829002125352179, NA, NA, 0.000436944721077761, 0.00133105849511764, 
    0.000528271061903097, 7.60097603724724e-05, NA, 0.000523346583016317, 
    0.000187693497226281, 0.0004978314503567, NA, NA, 0.000855597237273125, 
    0.000826438945165327, 0.000193217554601692, NA, 0.000132920613592246, 
    0.00110085164941708, 0.000390832580300822, 0.0013607996008906, 
    NA, NA, 0.000823385495874323, 0.000969996160662492, 0.000296114087794501, 
    8.89907439228157e-05, NA, 0.000642396436053771, 5.71946570117037e-07, 
    0.000853010564901883, NA, NA, 0.000761380978431855, 0.000378242705211501, 
    0.000507540326503431, NA, 0.000247592567292574, 0.000912340305537233, 
    0.000965109575565199, 0.000532129436895303, NA, NA, 0.000809747630293507, 
    0.00087309662023152, 3.63263182751928e-05, 1.83106894246886e-05, 
    NA, 0.000857411735415617, 0.000186170215687034, 0.00103497472269676, 
    NA, NA, 0.000275598524019525, 0.000708179311362283, 0.000495030548720225, 
    NA, 0.00100470217725485, 0.000328866735843135, 0.000771816842838178, 
    0.000999691442084744, NA, NA, 0.000586689213034855, 0.00125135338428562, 
    6.10771565285626e-05, 0.000182640516889989, NA, 0.000565523765837899, 
    0.000108751168740075, 0.00105743092702441, NA, NA, 0.000200288763464843, 
    0.000202975094062872, 0.000304898310627243, NA, 0.000884121222118393, 
    0.000580703381658648, 0.000837940431690479, 0.00145144149966738, 
    NA, NA, 0.00059417645488812, 0.00063044310728778, 0.000135762353446684, 
    5.84005970925515e-05, NA, 0.000483450031341416, 1.58409007681751e-05, 
    0.000789341893845263, NA, NA, 0.000436006361430235, 0.000996055424431764, 
    0.000417599650547129, NA, 0.000588086185430721, 0.0011439906164809, 
    0.000274435488144828, 0.000934585640774198, NA, NA, 0.000565487759147808, 
    0.000252295067639089, 0.000869699250493359, 4.80047944718458e-06, 
    NA, 0.000373811127347696, 5.98517466548771e-05, 0.000147330610936848, 
    NA, NA, 0.000612731422198977, 0.000746364571905418, 0.000342495043846214, 
    NA, 0.000835523566450925, 0.000441955313212546, 0.0011527492446265, 
    0.000838376450766109, NA, NA, 0.00095820821043695, NA, NA, 
    0.000184097053081673, NA, 0.000289440530652045, 0.000206161076305135, 
    0.000359793634084946, NA, NA, NA, 0.00113462341848404, 0.000556964799217671, 
    NA, 0.000183999964409027, NA, 0.00105657522921, 0.00107110102139581, 
    NA, NA, 0.000911940556630264, NA, NA, 2.89370514838288e-05, 
    NA, 0.000382828674822576, 9.65080862613156e-05, 0.000697754582500646, 
    NA, NA, NA, 0.000829176489946022, 0.000397088303529531, NA, 
    0.000269717334913801, NA, 0.000710411593547519, 0.000428918293008835, 
    NA, NA, 0.00131483679826493, NA, NA, 2.08621260021278e-05, 
    NA, 0.000218920017516074, 5.75476554209237e-05, 0.00272092936395065, 
    NA, NA, NA, 0.000829340140537049, 0.000241168858348361, NA, 
    0.000336800595540234, NA, 0.000622413330105908, 0.00110141768629547, 
    NA, NA, 0.000958366984318775, NA, NA, 8.81143586287638e-06, 
    NA, 0.000948692324802664, 3.14234657679127e-05, 0.000105863009898833, 
    NA, NA, 0.00415132168439553, 0.0321766279608464, 0.00177922248798609, 
    0.0495536880974044, 0.0577792817606334, 0.0449050936476729, 
    0.0601981883978289, 0.0569422689584137, 0.0448169274927967, 
    0.00307406182074284, 0.00818690080503843, 0.0128368600745412, 
    0.00551911428472724, 0.00765693892630906, 0.0576274763870225, 
    0.00181739329393384, 0.00272121279695803, 0.020463408006553, 
    0.00313705140719966, 0.0486674473319142, 0.00269155513509761, 
    0.0637936871109998, 0.000450225866219441, 0.000759450438247889, 
    0.00875939835145449, 0.0392538290561015, 0.00450048210469674, 
    0.00603264060156806, 0.0403687004914108, NA, 0.000657573387287651, 
    0.0383735778998702, 0.00485818386743732, 0.00836825793961696, 
    0.00309686948429443, 0.000178011199412241, 0.00521627007404526, 
    0.00622130040965432, 0.0517651814550586, 0.00124946886215665, 
    0.00378581273461475, 0.00921618672030416, 0.0005743183697178, 
    0.00643488172168828, 0.0057366953409538, 0.00478715208106856, 
    0.00451487050680493, 0.0110923844869191, 0.00823190197842388, 
    NA, 0.00693468054622875, 0.00932702340159585, 0.000890709188975599, 
    0.00834434083337753, NA, 0.00249930707008467, 0.0117995465107018, 
    0.0018371564898038, 0.00869025576716033, NA, 0.00639948804355725, 
    0.00478772673614587, 0.000125430862497427, 0.00468019228956665, 
    0.00940300355069608, 0.00187259191224358, 0.0196374598797498, 
    0.0086312356214947, NA, NA, 0.00367079321843689, 0.000333964173267293, 
    0.00329562655020015, 0.00320205656398056, NA, 0.0168540526290872, 
    0.00396918032040179, 0.00424598738917127, 0.00606718239056765, 
    NA, 0.041900255625354, 0.0397836416538078, 0.00539857202232613, 
    0.000847180490678787, 0.0130540168877215, 0.00165575961805715, 
    0.0310323834131299, 0.0085776780239547, NA, NA, 0.00463466568510034, 
    0.00294593032249805, 0.000483615806379408, 0.00620638341535155, 
    NA, 0.0172901115414125, 0.014113149818164, 0.0011073319160745, 
    0.00551507176962246, NA, 0.00425316228265607, 0.0269772714219003, 
    0.00354311348017641, 0.00426787730478749, 0.0102161560939544, 
    0.00559917390543548, 0.0334743049959442, 0.0202539333926914, 
    NA, NA, 0.0124412043396951, 0.00548159765499934, 0.000286081962923884, 
    0.00292230484653157, NA, 0.00913752060923595, 0.00016765155479324, 
    0.000771325487514566, NA, NA, 0.0033989043567167, 0.0141573134965134, 
    0.00045280070477067, NA, 0.0097737119068114, 0.0440921869956379, 
    0.0165832516383153, 0.0171433453342832, NA, NA, 0.0029709580433224, 
    0.0122026568436, 0.00217130250074568, 0.0027501815724396, 
    NA, 0.000629859873814402, 0.00294862379336406, 0.000413336223012873, 
    NA, NA, 0.00756229376999079, 0.0146028926303123, 0.0022108729644012, 
    NA, 0.0133881853209313, 0.0562717003115, 0.0224959324610547, 
    0.0117675062794642, NA, NA, 0.0124676349549795, 0.00404479767249882, 
    0.000697234254059895, 0.00718060092883093, NA, 0.00194123108927664, 
    0.000819941491859305, 0.00254954493808174, NA, NA, 0.00769851459339938, 
    0.0252360042923873, 0.00196290763919385, NA, 0.0150033844731864, 
    0.0444402270535197, 0.0171642197396647, 0.00618431584355737, 
    NA, NA, 0.0155645527658545, 0.0065215634365745, 0.0030462356400547, 
    0.00931449751483345, NA, 0.00175645375535632, 0.00704303128674109, 
    0.00953153417640623, NA, NA, 0.00372093028710227, 0.0330740527973123, 
    0.000153373267529087, NA, 0.0120707096553995, 0.0549012168206978, 
    0.0210583484172028, 0.0261639604090729, NA, NA, 0.0122342700258868, 
    0.00217029284422976, 0.000515525409421943, 0.00963677240920547, 
    NA, 0.000202475580751608, 0.00179950346480917, 0.00419013000423097, 
    NA, NA, 0.00986385731740439, 0.0139391089211849, 0.00039356303216485, 
    NA, 0.0155503090320273, 0.00706023918350092, 0.0290946041981913, 
    0.00933274091377208, NA, NA, 0.0305953984697869, 0.00631236237744918, 
    0.00210778793285654, 0.0159033209420128, NA, 0.001771295981313, 
    0.000752181127477692, 0.000900641370129125, NA, NA, 0.00232646116638608, 
    0.00753752085369337, 0.0103531509002967, NA, 0.00419741129837867, 
    0.00738951476921197, 0.0115011471497886, 0.00101905777771622, 
    NA, NA, 0.00716114988071307, 0.00679099399289502, 0.00263942891845115, 
    0.0152981484794955, NA, 0.0101882765432516, 2.69860355787111e-05, 
    0.00974243800467479, NA, NA, 0.00559646016315882, 0.0164294812857427, 
    0.00360926653132901, NA, 0.0154280229211673, 0.0109227093146425, 
    0.00550661284164643, 0.00690708424981926, NA, NA, 0.00167731858132312, 
    0.0029552663855329, 0.00125302923882301, 0.0117047184669308, 
    NA, 0.00781675322734469, 0.00255477990913035, 0.00297290726142173, 
    NA, NA, 0.00727931905703241, 0.00888623556349512, 0.017202632432736, 
    NA, 0.0167382272330643, 0.0200546902078188, 0.015868990574856, 
    0.00358189721447168, NA, NA, 0.00655964578110935, 0.00199335009244427, 
    0.00486566129895234, 0.0164144393015044, NA, 0.0029580561129388, 
    0.000417530526185166, 0.00396214416606412, NA, NA, 0.0047845465129418, 
    0.00768167868093463, 0.00570713898928653, NA, 0.013414929424426, 
    0.00348197849427396, 0.0231983976529525, 0.0104629403568864, 
    NA, NA, 0.00849696651346005, 0.00434589190072138, 0.00273487230812165, 
    0.016542848353579, NA, 0.00125728392694472, 0.00204256348474669, 
    0.00266710549722771, NA, NA, 0.00749938194356773, 0.0189245473769182, 
    0.00361558819455484, NA, 0.01301026448886, 0.00523552538176985, 
    0.0138372699343003, 0.00747265991789424, NA, NA, 0.00734895429603011, 
    0.00100177066118417, 0.00119928674682962, 0.0159035755600927, 
    NA, 0.0337985886196933, 0.0011374913656143, 0.00197882009361678, 
    NA, NA, 0.0141857020067211, 0.0141364307697667, 0.00140699107261923, 
    NA, 0.0177933404060836, 0.015942557682916, 0.00841869804548177, 
    0.0051804343159838, NA, NA, 0.00632891521731361, NA, NA, 
    0.00627171882388135, NA, 0.0308093238607492, 0.000395059651615089, 
    0.00457402524476489, NA, NA, NA, 0.00600929984432825, 0.011606261257354, 
    NA, 0.0147857667132303, NA, 0.0121182675648406, 0.00772342241239382, 
    NA, NA, 0.00125932155273301, NA, NA, 0.0184244176846259, 
    NA, 0.0435965296408245, 0.00157469869772623, 0.00099698452657297, 
    NA, NA, NA, 0.00138726633810429, 0.0137107377200003, NA, 
    0.0171704233710602, NA, 0.0119558753801754, 0.00655839428479379, 
    NA, NA, 0.0052736428751317, NA, NA, 0.00917969794190922, 
    NA, 0.0264671894232935, 0.0015896373557542, 0.0342380087389643, 
    NA, NA, NA, 0.00303459390150794, 0.0120889769954039, NA, 
    0.0219993376276701, NA, 0.0219592644891625, 0.00746984663182087, 
    NA, NA, 0.00662785774626623, NA, NA, 0.00964624538548622, 
    NA, 0.01931537750765, 0.0039286292801785, 0.00237680075134864, 
    NA, NA, 0.1119206639979, 0.135612265848558, 0.171774936077799, 
    0.0316888967599586, 0.110030253718161, 0.0350938451120903, 
    0.0106686574819123, 0.0968318131264885, 0.10298375070419, 
    0.136386471140939, 0.05405390825683, 0.0210691452635407, 
    0.131165740394043, 0.120072811791152, 0.125747622099553, 
    0.0309777103730064, 0.00463808770641347, 0.222929749126417, 
    0.0496233520949936, 0.147308049601162, 0.010002866581983, 
    0.0757017984254693, 0.017356351950381, 0.130276743311037, 
    0.0623424838103373, 0.0637293280380354, 0.100393368278953, 
    0.00168304321995858, 0.154748617435291, NA, 0.0439169691878585, 
    0.0666711252383312, 0.183797633088082, 0.00695207986773807, 
    0.0412888550897443, 0.0895158170071467, 0.00927228572343274, 
    0.0598380985711464, 0.0768889059955312, 0.101877073171432, 
    0.708146128096728, 0.0986717085248715, 0.114801095527371, 
    0.225001489396722, 0.0655815862680525, 0.0614962901552055, 
    0.0302074868259596, 0.036350485426257, 0.00308362919886718, 
    NA, 0.0157638675320983, 0.00620288163701809, 0.0444077526913013, 
    0.0136676864650443, NA, 0.125471925495386, 0.0226647124065573, 
    0.0841365946172458, 0.273517372744089, NA, 0.412966913030093, 
    0.0508692442757286, 0.0649670092603052, 0.1562331247686, 
    0.12618337785142, 0.0749197318648875, 0.04764176046046, 0.0139526350363527, 
    NA, NA, 0.00476046987567021, 1.51044421415537, 0.0337723578656906, 
    0.0826752538733413, NA, 0.242478999395068, 0.015137257210416, 
    0.0342035908279083, 0.0246469995821032, NA, 0.124683083971943, 
    0.101475096424953, 0.198453574022595, 0.105599700915589, 
    0.109425397855675, 0.010444381000914, 0.0346679528239427, 
    0.0838126460315786, NA, NA, 0.0313119749540052, 0.088898702701699, 
    0.137005925771505, 0.165851371918688, NA, 0.123470583558936, 
    0.00217422495383397, 0.117723603730781, 0.0950812168862573, 
    NA, 0.0377557470518752, 0.0640821245647412, 0.0123443023066891, 
    0.0548811871101562, 0.0592695955013649, 0.0712729472484814, 
    0.257866110720545, 0.0586593117396128, NA, NA, 0.00297775534854488, 
    0.0905812529699182, 0.0679895581704479, 0.154213193081919, 
    NA, 0.00242775385257963, 0.0535160685015537, 0.00489748711709955, 
    NA, NA, 0.158151684009678, 0.10993934356179, 0.0698731739100236, 
    NA, 0.00308966609459852, 0.0335111181046925, 0.0886408250275072, 
    0.0898691274062017, NA, NA, 0.108379338842237, 0.0528519608706903, 
    0.117267654793539, 0.139759309708081, NA, 0.043916368044328, 
    0.0303231765092872, 0.0974275602813322, NA, NA, 0.25515299858391, 
    0.141418546784538, 0.0500949280415001, NA, 0.0397177202223112, 
    0.0596774075604708, 0.143205427691805, 0.0327893268043331, 
    NA, NA, 0.00522550107753811, 0.0376244689248236, 0.126173292721842, 
    0.0161599506234555, NA, 0.0782629593157077, 0.0146433755715247, 
    0.0566481220920829, NA, NA, 0.229979089080604, 0.27838837392336, 
    0.0265372874620125, NA, 0.314240690774916, 0.145638224672033, 
    0.0835967426051252, 0.0553087043981593, NA, NA, 0.124238393242646, 
    0.0300160049960295, 0.190661916857698, 0.290142341433287, 
    NA, 0.0210344865575051, 0.000935776468292604, 0.0908818051242817, 
    NA, NA, 0.209560643360293, 0.173349858632314, 0.204943398516866, 
    NA, 0.333697391549709, 0.104233521179595, 0.0370632736700847, 
    0.0730726670617184, NA, NA, 0.0256016210773413, 0.0489454639155101, 
    0.153095404483308, 0.390288834509564, NA, 0.0470151482339283, 
    0.000965161448774568, 0.0383054975818358, NA, NA, 0.062592156015638, 
    0.2136742682313, 0.0887277248119294, NA, 0.12978394064063, 
    0.0769755124057864, 0.0478879716285722, 0.182617320977049, 
    NA, NA, 9.72284316563388e-05, 0.0555346796244407, 0.0909770035991269, 
    0.292326827573173, NA, 0.309681591075167, 0.0369727738154761, 
    0.00568650258334518, NA, NA, 0.00591681477516573, 0.113345787754861, 
    0.107316650440899, NA, 0.068433932416938, 0.121888529503562, 
    0.19045620460978, 0.19884854019501, NA, NA, 0.148304794616065, 
    0.0793298958584695, 0.0606010002840425, 0.351693569231645, 
    NA, 0.128878722435441, 0.00430057310600943, 0.00919920605132049, 
    NA, NA, 0.109380544223413, 0.163935255444311, 0.0900334811496757, 
    NA, 0.124947273164694, 0.0577439260786833, 0.0706029827608322, 
    0.273637633014911, NA, NA, 0.0533973578012742, 0.0510043321108948, 
    0.017195309620409, 0.371499301946975, NA, 0.161186409452532, 
    0.0140982448062701, 0.0256392744291333, NA, NA, 0.0455475099465897, 
    0.263198497138187, 0.311985931913547, NA, 0.268014493740372, 
    0.0283695079816857, 0.0424710045601833, 0.248745940290831, 
    NA, NA, 0.0625466762955495, 0.00541231462983214, 0.00971296514166673, 
    0.432156165624571, NA, 0.167131585932112, 0.00690704468725441, 
    0.134721129384019, NA, NA, 0.052125058428678, 0.052046862694286, 
    0.13007001188896, NA, 0.0990278773702348, 0.0140372676779691, 
    0.0448877301452602, 0.109748788362896, NA, NA, 0.112505342607998, 
    0.0686729894661538, 0.0218657094128955, 0.7168167363648, 
    NA, 0.0819673168425018, 0.0922666932116363, 0.0385183475505575, 
    NA, NA, 0.076067852701679, 0.10722584504484, 0.0753880243709772, 
    NA, 0.0685253853450617, 0.0774676748835261, 0.0367875735766, 
    0.214853354594161, NA, NA, 0.161986347850798, 0.0230957117028393, 
    0.0573482910037792, 0.617908597865906, NA, 0.116432732707484, 
    0.00206177085604514, 0.0727105236324084, NA, NA, 0.146625851383604, 
    0.131672848299833, 0.00901230776161547, NA, 0.167706429162768, 
    0.0397365585500269, 0.0396448993705894, 0.208049282279164, 
    NA, NA, 0.265672440328255, NA, NA, 0.875357181395292, NA, 
    0.120047936755149, 0.0033883862144795, 0.183653997999082, 
    NA, NA, NA, 0.140591583548984, 0.0956441383377404, NA, 0.221319237390381, 
    NA, 0.0163387237712679, 0.210140183794134, NA, NA, 0.118513562881435, 
    NA, NA, 0.769097477424754, NA, 0.129181747229069, 0.0291701886812035, 
    0.135216702754132, NA, NA, NA, 0.0705130835850964, 0.111705766534434, 
    NA, 0.359675364438319, NA, 0.0843511735018961, 0.158052915861035, 
    NA, NA, 0.0775287980094322, NA, NA, 0.76598175014796, NA, 
    0.111696582679955, 0.0130596688157136, 0.528299029388629, 
    NA, NA, NA, 0.210010853206043, 0.0928006701239521, NA, 0.364362050773563, 
    NA, 0.157479192804783, 0.219934493656455, NA, NA, 0.0699097173741165, 
    NA, NA, 0.687686690688223, NA, 0.120413038397028, 0.00572698707550413, 
    0.156705130071431, NA, NA, 0.00185188192678078, 0.00559866613846215, 
    0.00105658329831864, 0.00799718568565736, 0.00515696562455043, 
    0.00623891485764583, 0.00612411433074395, 0.00619205950981825, 
    0.00746690878197256, 0.000847425285240637, 0.000786977236481622, 
    0.000116457190242692, 0.00418682016806916, 0.000134694297569215, 
    0.00577650731402933, 5.03255842146317e-06, 0.00155076965712522, 
    0.000244398873487797, 0.00211282135407586, 0.00655463107076055, 
    0.00313811128030535, 0.00588438955205458, 0.00253648664087657, 
    0.00118011623241511, 0.000364824868133841, 0.00852752441593251, 
    0.000858314073278923, 0.000505298204642096, 0.00810525181015742, 
    NA, 0.00157709320730354, 0.00537447849193479, 0.00378314483963268, 
    0.000356309400922344, 0.00190129597230585, 0.00021643803602585, 
    0.00023058709435366, 0.000766150683350651, 0.00619082830089926, 
    0.000917435771588592, 0.000829671035865531, 0.00304497700007229, 
    6.60096160895266e-05, 0.00230100138753891, 0.000654802480713981, 
    0.0045367654758229, 3.01521870825283e-06, 0.00246842625139794, 
    0.000732699613814624, NA, 0.00204238797055082, 0.00425183326617427, 
    0.00316296105753354, 0.000349769975446208, NA, 0.000190619154849241, 
    0.000189211466907134, 0.000237527207130915, 0.0041271815086315, 
    NA, 0.00453180929394476, 0.000551434461137062, 0.000377125046954907, 
    0.00182012768347584, 0.000250191666365137, 0.00569573186948474, 
    0.000878092736049679, 0.00211089113514429, NA, NA, 0.000489841304273536, 
    0.00604826541655121, 0.0024052130877946, 0.00135662821881228, 
    NA, 0.00154450836173884, 0.000142836592649326, 0.00034151748415334, 
    0.0036826592152992, NA, 0.00652330153942355, 0.000370575107790988, 
    0.00191683877942965, 0.000614983825728808, 0.00019735184160452, 
    0.00419733402984669, 0.000299488775134531, 0.00291389528618373, 
    NA, NA, 0.000686614560939989, 0.000800562873714392, 0.0035035447519262, 
    0.000418311946831092, NA, 0.000922451309424079, 0.000742884988250912, 
    0.000105844639474484, 0.00212306911003486, NA, 3.74791314661861e-05, 
    0.000281962023878912, 0.00126197789479614, 0.00016076489028249, 
    0.00138334294486959, 0.00355058870713523, 0.00257018794458789, 
    0.00348267262667305, NA, NA, 0.000381176935603832, 0.00223458585786388, 
    0.00480011607534965, 0.000595813974291287, NA, 0.000426702686048538, 
    0.0001376737476366, 5.55781793174099e-05, NA, NA, 0.000528888371763891, 
    0.00185968368075926, 0.000829022574079977, NA, 0.00162535981613683, 
    0.0045015548396758, 0.00262525819040018, 0.00171477410153189, 
    NA, NA, 0.00125917501121801, 0.000249055441097863, 0.00479741978961554, 
    0.00236532732568262, NA, 0.000969275784969316, 2.34698246454751e-05, 
    0.000162637714384692, NA, NA, 0.000695450280893003, 0.0012488468217272, 
    0.000646573545534731, NA, 0.00139527117402368, 0.00709131892502923, 
    0.000723960918263661, 0.00181349564031941, NA, NA, 0.00134532809013672, 
    0.00066221397184582, 0.00247729950826356, 0.00179171619845422, 
    NA, 0.00177724116199798, 0.000184902175983451, 7.99730695626201e-05, 
    NA, NA, 0.000748062025998855, 0.000940186374890824, 0.000130940218437286, 
    NA, 0.00285114542879816, 0.00874801068113795, 0.000338119740970077, 
    0.00134344709760054, NA, NA, 0.00310725271219964, 0.000309093022766486, 
    0.000842929011586887, 0.000135621738724976, NA, 0.00130350957564952, 
    0.00190018506242685, 0.00178354669731527, NA, NA, 7.16028609114883e-05, 
    0.000662957379002636, 0.000756881205620536, NA, 0.00366283711287887, 
    0.00621570076704087, 0.000708043479200114, 0.00136637839158695, 
    NA, NA, 0.00752491941371296, 0.0024676758303379, 0.00121074941167814, 
    0.000112841644094299, NA, 0.000456374377825805, 0.00135375572394376, 
    0.00107387052392943, NA, NA, 0.0020441030572295, 0.000398765425488465, 
    0.000187924743772861, NA, 0.00366453054974575, 0.00127474887033809, 
    4.63479359919903e-05, 0.00159521359894806, NA, NA, 0.00272656254563253, 
    0.00238153745639434, 0.00161287775765558, 5.28147636254324e-05, 
    NA, 0.00282707625175455, 0.00061230135208784, 0.000443207655001116, 
    NA, NA, 0.00011009409986208, 0.000687920849626969, 0.000124829469220356, 
    NA, 0.000603281205615632, 0.00151838384521001, 0.000861678306673684, 
    0.00126757131034886, NA, NA, 0.0020802489284246, 0.000934595501622828, 
    0.00147628131742032, 0.000394247921348885, NA, 0.0011227623645267, 
    0.000684330455928024, 0.00249320498993116, NA, NA, 0.000989626357101776, 
    0.000672167987820614, 0.000255569191668671, NA, 0.00259085787908988, 
    0.001583227666995, 0.00159593474410453, 0.000511415615551812, 
    NA, NA, 0.00385149669579358, 0.00200671255357763, 0.000169337078171608, 
    0.00223880146399102, NA, 0.00308423859556188, 0.000362342990488669, 
    0.0041614513550744, NA, NA, 0.000755250213572328, 0.00149627504021731, 
    3.66479819011789e-05, NA, 0.000509733186868012, 0.000943771720217256, 
    2.36536747349775e-05, 0.00375707586324055, NA, NA, 0.00375497647650323, 
    8.94396516247292e-05, 0.000836544742324035, 0.000573788772133246, 
    NA, 0.00547410687976508, 0.000332282360750743, 0.00280886249852491, 
    NA, NA, 0.000877837784036825, 0.00116736131147012, 1.6647541760305e-05, 
    NA, 0.000770250590706989, 0.000457814262501687, 0.000329069653450155, 
    0.00112048255207575, NA, NA, 0.00863594769770925, 0.000793819954772677, 
    0.000293886067564256, 0.00194486695558558, NA, 0.000527354383320479, 
    0.000562503214055753, 0.00407362210235164, NA, NA, 0.000342743062393917, 
    0.00141455098979082, 0.000457038304369003, NA, 0.00179734514699674, 
    0.00113202773319875, 0.000307799801352787, 0.00219430385889681, 
    NA, NA, 0.0038214674865071, 0.00120301995972959, 0.00107858593840786, 
    0.000913591622421724, NA, 0.00125597680831908, 0.000139366555500761, 
    0.00262144091033572, NA, NA, 0.00209102021449913, 0.00173007314437091, 
    0.000563675031136376, NA, 0.000941662556513568, 0.000332813552528642, 
    0.000103818774975071, 0.000865095771428806, NA, NA, 0.00544789066123819, 
    NA, NA, 0.000722531096748148, NA, 0.00266551461455098, 0.000288889209369948, 
    0.0069803495198273, NA, NA, NA, 0.00120421809724309, 0.00376516577552538, 
    NA, 0.00144728931597507, NA, 0.000617874412177152, 0.00420513723964626, 
    NA, NA, 0.00499102944313663, NA, NA, 0.000518240202134353, 
    NA, 0.000793417173786342, 0.000978988274565511, 0.00302124370730624, 
    NA, NA, NA, 0.0026820292360554, 0.00303217742230511, NA, 
    0.0024565725647916, NA, 0.00193412721616748, 0.00297001148095726, 
    NA, NA, 0.00553222679214419, NA, NA, 0.00031965045649167, 
    NA, 0.000913016942952574, 0.00115886461579237, 0.00905988190918331, 
    NA, NA, NA, 0.00486071311975209, 0.00203459626911694, NA, 
    0.00150569930736503, NA, 0.000632949448308231, 0.00313714371451488, 
    NA, NA, 0.00502766940250057, NA, NA, 0.000404407316339256, 
    NA, 0.00025950188588502, 0.000584841308048613, 0.00119154034818792, 
    NA, NA, 0.000346974443734695, 0.152840714654544, 0.0826875639323893, 
    0.14863007695969, 0.130965604500336, 0.146439956455929, 0.13576548234338, 
    0.152254720061948, 0.147927465995615, 0.0165748613758694, 
    0.0264095044817574, 0.00685884070745455, 0.0546157947879998, 
    0.014277438752823, 0.138314910052141, 0.0179110670547943, 
    0.0180114090396261, 0.0228859422673797, 0.0029456351798915, 
    0.154115512235045, 0.00511777934880212, 0.145174054831506, 
    0.0655577395444986, 0.00580746118377459, 0.0414495210469115, 
    0.135962218852456, 0.00173618644790644, 0.0229889664976631, 
    0.16983412445462, NA, 0.00350956219924321, 0.143042682952489, 
    0.0639608851044625, 0.0294473401788123, 0.0259813764955377, 
    0.0292512707607248, 0.00983600277336654, 0.106885610730483, 
    0.136408991472292, 0.0425514627673383, 0.00198102347552186, 
    0.00504727060395237, 0.0436943974259373, 0.000783670226205866, 
    0.0261540633689897, 0.0690506367079282, 0.00573409219214444, 
    0.0793792772683962, 0.0196539704754419, NA, 0.000523847291665642, 
    0.0457852872784191, 0.0416641262053171, 0.0387080326012279, 
    NA, 0.0225071360122251, 0.0404067700751728, 0.0055115254449239, 
    0.0573393752785037, NA, 0.0129255635347222, 0.0112960634759631, 
    0.0322867211354671, 0.000242717736454822, 0.0259672043946058, 
    0.0719791768425962, 0.0247267677108718, 0.0706887455153926, 
    NA, NA, 0.00258138529725544, 0.0628152983548035, 0.0465608047656701, 
    0.0337064953441752, NA, 0.163744058704709, 0.0439024459681643, 
    0.00181389454486044, 0.0731417366790056, NA, 0.144614343717063, 
    0.0226427918807196, 0.0231666978764007, 0.00234557801277029, 
    0.00892838177215149, 0.0502171039598839, 0.0404152647017187, 
    0.0725881084872718, NA, NA, 0.000428922788755603, 0.0130906416433486, 
    0.0472459417003807, 0.0342805204300794, NA, 0.137516897286463, 
    0.0254457977541665, 0.00342512007413536, 0.031491339581157, 
    NA, 0.00396459106652736, 0.0272704889258094, 0.0415640264176924, 
    0.0109838209291442, 0.00970118658566285, 0.0692139278872769, 
    0.054651025124234, 0.0808065561855893, NA, NA, 0.00528516089751608, 
    0.0349878353800574, 0.0609446334051693, 0.0153177815106532, 
    NA, 0.0167897641180522, 0.0095989580958266, 0.0048437790133308, 
    NA, NA, 0.00845985996832049, 0.0149806826678153, 0.0304083471223117, 
    NA, 0.0181496093299658, 0.15437411657989, 0.0566435443967513, 
    0.0407317710916505, NA, NA, 0.00217914799654931, 0.0350853934643613, 
    0.0358936589689052, 0.0438615410718426, NA, 0.00636973659239832, 
    0.0105205383898913, 0.0118900363977382, NA, NA, 0.00720975761434267, 
    0.0152141650817813, 0.0314674239274579, NA, 0.00749890480347034, 
    0.134078198500846, 0.0580359364377535, 0.039975121526796, 
    NA, NA, 0.00435424113388407, 0.0267065809397379, 0.0516058151281595, 
    0.0302540022989875, NA, 0.0460720669078732, 0.0146813210525209, 
    0.00515929139559194, NA, NA, 0.0154616699164965, 0.021432966608181, 
    0.026910937980223, NA, 0.0104438101705643, 0.137408665575473, 
    0.0450979987504137, 0.0375632971385418, NA, NA, 0.14528083779931, 
    0.0496542388027623, 0.0313930529161877, 0.010751225240748, 
    NA, 0.0341127715436692, 0.00265518795815017, 0.0228436658410759, 
    NA, NA, 0.0290575259183141, 0.0238968912695706, 0.0335834971270051, 
    NA, 0.0228062504641896, 0.131614432765515, 0.0340249953375929, 
    0.0331687876162469, NA, NA, 0.125580806403184, 0.0312374261204776, 
    0.0463412529831009, 0.00153615963309699, NA, 0.0399185511967386, 
    0.00142271430655813, 0.00667800355580616, NA, NA, 0.0279295205754725, 
    0.020476508392104, 0.0110320076234845, NA, 0.00748458063034552, 
    0.0407817628874093, 0.0367829312771448, 0.0763623172112083, 
    NA, NA, 0.139114155084708, 0.0397543280612362, 0.0588575685013579, 
    0.00245919371466677, NA, 0.0506253249297403, 0.00255084246913646, 
    0.0241565577719946, NA, NA, 0.0130235086705576, 0.0237164923526351, 
    0.0470872479625187, NA, 0.00664274870024933, 0.0853027488410885, 
    0.0538267145386701, 0.0568153230687024, NA, NA, 0.1389832567631, 
    0.0535862450999506, 0.042426987843146, 0.00499599293819345, 
    NA, 0.037697734449033, 0.00337750500032915, 0.0252935594975886, 
    NA, NA, 0.00866124365009228, 0.031480310302538, 0.0387406399025441, 
    NA, 0.00357252632571711, 0.05258690716801, 0.0480901698729247, 
    0.0313597211761538, NA, NA, 0.106198687267257, 0.0443420428373465, 
    0.0124035580870686, 0.00276778305462884, NA, 0.0362007050515168, 
    0.00994216662783767, 0.0256605601557469, NA, NA, 0.00845677599840077, 
    0.0236204115261627, 0.0391101206092483, NA, 0.00150161010854295, 
    0.0467808174917024, 0.0574734103390298, 0.0455673841099451, 
    NA, NA, 0.120874099664962, 0.0425592957320422, 0.0206698557056389, 
    0.00208813441454064, NA, 0.0509259980898352, 0.0073585247990893, 
    0.0329324909236284, NA, NA, 0.00373763399485906, 0.0129066693061807, 
    0.0300458378432502, NA, 0.00267704768471143, 0.0274724208826759, 
    0.0327828339523511, 0.0718404419337697, NA, NA, 0.0849064018590127, 
    0.102966019736565, 0.0145091561946409, 0.0220220248052788, 
    NA, 0.0306724940458318, 0.0227637585779769, 0.0217134837954224, 
    NA, NA, 0.00877265152232463, 0.0247843940058387, 0.034500326260793, 
    NA, 0.00515177336359149, 0.0307207252782439, 0.0408328431111712, 
    0.06517532362929, NA, NA, 0.0727289189855824, 0.0187462893299843, 
    0.0170609343712878, 0.00288326406285946, NA, 0.0222628177827446, 
    0.00480559194769809, 0.0252128045361405, NA, NA, 0.00634002275446222, 
    0.0097028071042306, 0.0471853711649803, NA, 0.0043030627310542, 
    0.0127398128339777, 0.039807569547134, 0.0650759975009522, 
    NA, NA, 0.100009656508662, NA, NA, 0.00302293572735683, NA, 
    0.0282200116971432, 0.00522652789816812, 0.0402001466723083, 
    NA, NA, NA, 0.0394257507867671, 0.055866150353505, NA, 0.00375900327347569, 
    NA, 0.0438375502583754, 0.0473495674487949, NA, NA, 0.0950478426850809, 
    NA, NA, 0.00366717050547317, NA, 0.00971869936950446, 0.00447773228341717, 
    0.0293637871543411, NA, NA, NA, 0.026467981047782, 0.0639033485309157, 
    NA, 0.0114584201792371, NA, 0.0412782610370148, 0.0482455407536668, 
    NA, NA, 0.0845484614491846, NA, NA, 0.000157090366203469, 
    NA, 0.0259448470863819, 0.00906121443245109, 0.165865224038203, 
    NA, NA, NA, 0.0419250696682293, 0.0637118800760614, NA, 0.0132447822188349, 
    NA, 0.0582534458652024, 0.0551426954165045, NA, NA, 0.0827967276487148, 
    NA, NA, 0.003502706856667, NA, 0.0333970079134629, 0.00953141697746914, 
    0.0697840262770176, NA, NA, 845.266216093708, 72972.8417598198, 
    2128.61051246013, 75298.848958755, 68457.603713961, 66241.9530110202, 
    59914.2092078032, 66567.7537939545, 65011.2052162964, 90.7595699355029, 
    89.8298854432715, 1447.98471054301, 3795.77476536209, 4320.76432471844, 
    62218.7217996193, 921.474282221837, 996.635807701167, 2316.67724863456, 
    567.634687066027, 71563.6846644741, 3481.13767778375, 67121.4846293606, 
    3198.44831489401, 438.904576595361, 345.277529618525, 66068.776026571, 
    49.0901968487702, 3325.52528321788, 61872.9148470037, NA, 
    244.051845882017, 67079.4353578931, 2035.71180643042, 2100.24921846185, 
    888.284081446107, 1094.21297818096, 280.431425973098, 7171.75261524866, 
    65958.9689995471, 3109.49836767407, 4752.84241265358, 876.596618324771, 
    132.991293262291, 705.44616256606, 1121.70614796843, 880.875864173323, 
    670.525285887769, 680.843911874645, 3501.11379103299, NA, 
    1485.17551161047, 11368.845725816, 1681.47421165364, 694.963774788528, 
    NA, 447.421876977875, 623.191563135078, 1226.96119963522, 
    4476.63891976463, NA, 353.698802243613, 1387.23402471533, 
    1498.2001688411, 313.121382311053, 167.449128464404, 3240.83672437119, 
    943.316119233182, 1829.00727869749, NA, NA, 1678.66417930477, 
    1171.01190697532, 592.650578720623, 389.102748133489, NA, 
    1204.50959900896, 899.337870130841, 1962.35137849398, 6140.65352260885, 
    NA, 52869.6339682762, 404.282630785859, 373.458463346857, 
    1808.8201607274, 1142.89475284875, 4536.80193864996, 1569.5982152043, 
    1070.60127936563, NA, NA, 918.968169286302, 2195.66774025742, 
    688.23689925492, 1307.55716819743, NA, 934.511551319735, 
    119.158363530726, 1503.5037903434, 935.971570087953, NA, 
    260.788995628685, 1143.13659331748, 1187.98638775099, 2654.76183029989, 
    352.772325965362, 5331.58094813062, 894.681726401728, 2381.08853354784, 
    NA, NA, 901.907566904843, 1263.57483580316, 643.967583360233, 
    620.070463330239, NA, 10856.2631474401, 97.6263522461382, 
    140.782132805623, NA, NA, 8111.32975180557, 373.369388817516, 
    791.987431854648, NA, 1007.59399628893, 57914.0865441646, 
    286.950220358299, 1342.28450836685, NA, NA, 2171.2491663404, 
    4587.53198814579, 820.102203123542, 1995.24421565327, NA, 
    5698.47947929973, 61.8661111073168, 78.5488091134809, NA, 
    NA, 3639.33628998754, 1677.90187431627, 2393.30525350883, 
    NA, 491.189006773574, 68686.99046657, 5.99271565450073, 259.75418973477, 
    NA, NA, 1413.03888724626, 3494.78727621245, 2737.68898287253, 
    434.476139991735, NA, 5655.6232879792, 63.292544118049, 17.7325657446055, 
    NA, NA, 2106.78361166882, 1249.31857492397, 1561.71357853791, 
    NA, 368.084968774906, 66795.1319390548, 1078.88714090462, 
    39.4223417202754, NA, NA, 3647.29396204183, 5795.34181438253, 
    1451.09803984178, 297.553707310654, NA, 3703.08817087856, 
    86.0274236657651, 1797.59027552882, NA, NA, 451.036221108625, 
    694.97347125262, 1801.65881502256, NA, 88.8254084758869, 
    65696.5891915194, 937.568907990539, 922.84015740183, NA, 
    NA, 4752.98446188719, 3120.6285723355, 1280.95625155378, 
    1000.75851564077, NA, 1365.61966883164, 1233.75287790485, 
    1381.0359456872, NA, NA, 2315.75090355937, 1315.28188828324, 
    1058.72431882487, NA, 882.797149762938, 1084.33883886387, 
    365.024255215039, 113.012688534573, NA, NA, 6186.58629561658, 
    5262.55398335621, 862.74212177692, 535.65400325365, NA, 1843.99278727185, 
    395.93141557989, 1018.3275033808, NA, NA, 10.0296075994031, 
    363.002696100816, 177.708251360658, NA, 585.867398088172, 
    18805.4446033358, 179.045434168636, 103.02302219, NA, NA, 
    10305.4962902897, 2991.09790969846, 900.746934150792, 1630.15764461517, 
    NA, 1580.11343254735, 111.591887549107, 4286.88896627265, 
    NA, NA, 501.090325896802, 432.823310901225, 535.149277245379, 
    NA, 921.430326610054, 1673.26991300962, 581.409886913573, 
    2100.18030599597, NA, NA, 10257.1312485657, 2353.10463865304, 
    3623.36667813366, 799.395968561705, NA, 2084.92763150542, 
    137.540371172489, 646.187521833486, NA, NA, 1936.66005920077, 
    1872.4240230827, 893.73458699605, NA, 33.2396114812233, 1655.89528541004, 
    599.066044184427, 645.222310093198, NA, NA, 10525.6622878693, 
    2576.11301459188, 6568.78365391059, 117.147009121999, NA, 
    263.459460734374, 340.232374417485, 2299.23372381521, NA, 
    NA, 309.24044302138, 2433.05877501966, 444.328049243445, 
    NA, 1476.23853007188, 319.56050963969, 94.2239001658862, 
    1726.51983950914, NA, NA, 11750.588931626, 9347.64209092198, 
    3510.66300759083, 1330.76661313586, NA, 392.609671047415, 
    1085.65429070272, 1843.08380794134, NA, NA, 536.270528850875, 
    639.878875606285, 1962.48824116899, NA, 1695.89920733064, 
    1284.20503912714, 1210.74498908599, 1133.50718882497, NA, 
    NA, 7600.85986819907, 2496.4292177961, 64.0318001743832, 
    1290.41181641633, NA, 349.605884225712, 607.186833098364, 
    190.283211230129, NA, NA, 36.819931947024, 324.673170055138, 
    18.3306181684803, NA, 675.176766081724, 99.5849058742734, 
    407.362591189101, 863.522068114979, NA, NA, 10897.9762238839, 
    NA, NA, 1315.16640827442, NA, 2043.77895999877, 972.325834895714, 
    488.067689453412, NA, NA, NA, 1365.63228842266, 744.187436880236, 
    NA, 113.667566667726, NA, 272.027877456072, 2650.15764688201, 
    NA, NA, 10621.0056628625, NA, NA, 1204.69096687901, NA, 1089.13443038936, 
    630.972367507765, 12.6948173487961, NA, NA, NA, 262.627945291369, 
    654.092837726597, NA, 78.3998949602695, NA, 457.224093448692, 
    1849.72977921761, NA, NA, 10892.2338829678, NA, NA, 1331.8492128591, 
    NA, 2663.49150488143, 55.3475679317144, 73991.0841704984, 
    NA, NA, NA, 762.300763988931, 403.050925420336, NA, 631.614842425191, 
    NA, 403.051761025611, 2478.43121140699, NA, NA, 9518.9213977003, 
    NA, NA, 246.483628924363, NA, 643.780857796134, 439.755387626037, 
    165.454757271319, NA, NA, 0.613896587557427, 0.403351890365075, 
    0.606156880845337, 0.431139417861294, 0.40119087804809, 0.406134623110972, 
    0.377020883351033, 0.390689529228014, 0.426285194993254, 
    0.629571761231137, 0.712140947493518, 0.580653995483341, 
    0.601963823669623, 0.65479885253727, 0.408765090829625, 0.630447497416091, 
    0.602803371087724, 0.648110384398019, 0.673052266029071, 
    0.422716720909208, 0.698747721192665, 0.465032671757633, 
    0.628690737321663, 0.654464972975023, 0.582213558849479, 
    0.396166814122208, 0.70272802178562, 0.687556528614039, 0.393651260345671, 
    NA, 0.664775307447157, 0.464330812445982, 0.577061207641293, 
    0.609932106810421, 0.646079419295592, 0.588437093759677, 
    0.570949673449456, 0.619910621938923, 0.431989517297207, 
    0.641641276921013, 0.585178786932273, 0.607041867807841, 
    0.672168598795907, 0.590492218643835, 0.628767635693479, 
    0.635872892649389, 0.686592347167461, 0.64625618310871, 0.658609440777699, 
    NA, 0.693740829788317, 0.599724147710326, 0.608228744515937, 
    0.664506329965923, NA, 0.620291762843744, 0.644027605374123, 
    0.721956336184883, 0.566140808785743, NA, 0.540938912145403, 
    0.667660774715691, 0.614177850384388, 0.626355384177084, 
    0.631477202158055, 0.649198959284088, 0.637023787450158, 
    0.623614449091049, NA, NA, 0.622070197342371, 0.667610456734277, 
    0.591861068445341, 0.590354378252834, NA, 0.599356301498128, 
    0.71204935126743, 0.625972865093433, 0.584824245678864, NA, 
    0.433534918905248, 0.648685523377914, 0.66847934740922, 0.641357634764998, 
    0.592083704036048, 0.634249624623469, 0.634017322704589, 
    0.610749435453726, NA, NA, 0.584956888637045, 0.686102146188724, 
    0.584114828505786, 0.63573425471706, NA, 0.624170383313257, 
    0.641181795175386, 0.702026496817789, 0.661134163394658, 
    NA, 0.62959417370134, 0.632668668152592, 0.645573902212209, 
    0.651341774461081, 0.660461685791697, 0.640224076629613, 
    0.654722134983755, 0.561653314483541, NA, NA, 0.626569393416716, 
    0.625272332106042, 0.622536880975132, 0.65918093284684, NA, 
    0.564777012186122, 0.68137194733517, 0.637756800575238, NA, 
    NA, 0.61691212607012, 0.622347858559154, 0.679641435920006, 
    NA, 0.62215206041904, 0.434434053457814, 0.683583278621022, 
    0.54772703317282, NA, NA, 0.603794421389118, 0.605347167702606, 
    0.612760307978645, 0.583518985000367, NA, 0.610512654267851, 
    0.687317521937378, 0.608350128054452, NA, NA, 0.636727317605752, 
    0.658554573369962, 0.651194553943858, NA, 0.699483737146709, 
    0.42926226073788, 0.617794934211769, 0.625098527340067, NA, 
    NA, 0.635587243914068, 0.633950016029055, 0.671648645692298, 
    0.612263178808931, NA, 0.667553008580948, 0.606578883057363, 
    0.633071466047378, NA, NA, 0.603706746077932, 0.687344840632883, 
    0.640374856609941, NA, 0.687603329978498, 0.417851138561883, 
    0.58351832532548, 0.651540289731, NA, NA, 0.575440777621723, 
    0.624801961932706, 0.600850568105658, 0.646474127007663, 
    NA, 0.60424869369968, 0.660869837326167, 0.725888226147891, 
    NA, NA, 0.608481592825549, 0.663558109852454, 0.625709764968178, 
    NA, 0.632194172037939, 0.446545965560735, 0.564338769614251, 
    0.646469556241512, NA, NA, 0.617221802159768, 0.631810948239625, 
    0.64468048925685, 0.633672061493182, NA, 0.651485187908707, 
    0.653294038736341, 0.616797286795222, NA, NA, 0.662680531515282, 
    0.627540578321216, 0.649751254130866, NA, 0.66834698862247, 
    0.59914913663116, 0.626227877188254, 0.677648657122146, NA, 
    NA, 0.532091361850327, 0.634661255897661, 0.629527397545872, 
    0.631494549418437, NA, 0.656917787771444, 0.698110410895719, 
    0.631812550684714, NA, NA, 0.640660040079991, 0.611894217073472, 
    0.605157285229401, NA, 0.707254046769918, 0.513228166079166, 
    0.641735934084092, 0.694538356458436, NA, NA, 0.581648268510516, 
    0.593167835656379, 0.643432994259852, 0.682342947387258, 
    NA, 0.633458166176441, 0.655957031272823, 0.620908436192843, 
    NA, NA, 0.649248279828786, 0.664588386561134, 0.59166935733533, 
    NA, 0.677418688369215, 0.65017422640149, 0.634131460176528, 
    0.641741737690692, NA, NA, 0.600040340902482, 0.637327927409778, 
    0.624649659674104, 0.609892996491727, NA, 0.6615667304777, 
    0.693301610393333, 0.620705412103542, NA, NA, 0.659968681245916, 
    0.68110286399172, 0.63038549490537, NA, 0.662393444670992, 
    0.624160882402803, 0.604063328084213, 0.653457095203874, 
    NA, NA, 0.525368724430673, 0.623654398013036, 0.669312055021356, 
    0.6464099096471, NA, 0.6355735364669, 0.643143964458564, 
    0.672571537544852, NA, NA, 0.62118730056271, 0.632197981624005, 
    0.660399397496839, NA, 0.68643778515492, 0.644052974778928, 
    0.686257630164181, 0.587917090207963, NA, NA, 0.608424500187685, 
    0.58795800823532, 0.674858633712504, 0.654219111400978, NA, 
    0.668818209910095, 0.697259053526253, 0.723200302940702, 
    NA, NA, 0.637931107209738, 0.658824877491778, 0.675817162125279, 
    NA, 0.697467273956335, 0.665710496637282, 0.631660199435215, 
    0.626710944087864, NA, NA, 0.625292360191307, 0.648598562045605, 
    0.621635473749918, 0.663552942061813, NA, 0.659822540874739, 
    0.640458597330305, 0.693176098962835, NA, NA, 0.641404969234885, 
    0.649549892421707, 0.575337090212482, NA, 0.599079722816283, 
    0.643117288331072, 0.661690926103957, 0.626560360762895, 
    NA, NA, 0.582076829930338, NA, NA, 0.677614359293207, NA, 
    0.640695537081685, 0.574615375554131, 0.710226790528668, 
    NA, NA, NA, 0.640031317224127, 0.609365847282733, NA, 0.645383379481963, 
    NA, 0.64822623284578, 0.664627279026217, NA, NA, 0.602829650613188, 
    NA, NA, 0.614614903157666, NA, 0.630175014762961, 0.625640980833419, 
    0.613922778898668, NA, NA, NA, 0.638149322328783, 0.593438816408914, 
    NA, 0.644029556926994, NA, 0.641205302469764, 0.655455657164092, 
    NA, NA, 0.547798316903885, NA, NA, 0.664013470120238, NA, 
    0.675657624807156, 0.649624860053511, 0.419537626825094, 
    NA, NA, NA, 0.602247682483133, 0.57700342947232, NA, 0.687802100644982, 
    NA, 0.638707858591239, 0.740079027905204, NA, NA, 0.581965705933602, 
    NA, NA, 0.6290483510257, NA, 0.657329518459069, 0.604556054267562, 
    0.635697752951617, NA, NA, 1.05478937857529, 1.34519072377548, 
    1.28824762359451, 1.15895300529511, 1.20665680852119, 1.17219185118253, 
    1.20505859270836, 1.19057140950711, 1.18556927964132, 1.18519441655264, 
    1.24831238007779, 1.2335809083304, 1.07910738847894, 1.1361887961739, 
    1.11120199691876, 1.30301634270563, 1.37133373457743, 1.42312101535485, 
    1.23418058866149, 1.19743868494124, 0.952658503575077, 1.19235204878079, 
    1.18446369538517, 1.3521406103682, 1.31865177922705, 1.23476308032791, 
    1.12024705351894, 1.26115358314026, 1.15585758594607, NA, 
    1.0958835586003, 1.18161334017574, 1.26225581367173, 1.3081676955965, 
    1.24965066497407, 1.46649937206988, 1.39666054949256, 1.10230654890521, 
    1.10640496693073, 1.39562285064334, 1.10000831971506, 1.24521265025286, 
    1.27172579646818, 1.18984304404226, 1.41117822564192, 1.32131244939689, 
    1.33347749715691, 1.3644302023334, 1.3860283160036, NA, 1.13786515485736, 
    1.12622203139586, 1.19886465968672, 1.2868815135078, NA, 
    1.33818756980399, 1.26281362344167, 1.36436039645046, 1.32851098525558, 
    NA, 1.0432253431671, 1.09919338039287, 1.21817467179184, 
    1.20111821864946, 1.30299729048732, 1.34378519031416, 1.29650491104566, 
    1.11425026808642, NA, NA, 1.16614259171701, 1.2969632615886, 
    1.3063142380882, 1.50434358067027, NA, 1.20708783520599, 
    1.02034429688725, 1.19670201360933, 1.32497219752048, NA, 
    1.16338312467713, 1.41512480970144, 1.28750253440022, 1.16236547573626, 
    1.43252006116924, 1.11305318939367, 1.18636581297603, 1.12792235922074, 
    NA, NA, 1.27341342383539, 1.29040291576007, 1.3508962448734, 
    1.55906927048572, NA, 1.20205000755104, 1.00201292562724, 
    1.47947118593017, 1.27835529754702, NA, 1.34900737128592, 
    1.32356420146482, 1.22250384073961, 1.29135602864914, 1.31626336859511, 
    1.33319739215665, 1.08701284872371, 1.14690438091366, NA, 
    NA, 1.30108282469457, 1.33204337202017, 1.25774970264965, 
    1.20244266800562, NA, 1.16223547589835, 1.09328384695854, 
    1.26239444824875, NA, NA, 1.3939138427779, 1.39654924884336, 
    1.17377056828203, NA, 1.27238152073081, 1.26284917968505, 
    1.12949312535865, 1.28895916432584, NA, NA, 1.17660256837426, 
    1.23611787403379, 1.36898885118529, 1.50336959636851, NA, 
    1.16239006909206, 1.08268970617997, 1.37233449358978, NA, 
    NA, 1.33087889155156, 1.49989995935611, 1.30575954680395, 
    NA, 1.34548577927924, 1.0879760296459, 1.07930755909886, 
    1.23191936734794, NA, NA, 1.17785916612016, 1.37869043743743, 
    1.31949260110235, 1.50629979963986, NA, 1.20120341706834, 
    1.18131293424338, 1.40634362854284, NA, NA, 1.17302968801605, 
    1.36665474145838, 1.2035680467883, NA, 1.18670453976864, 
    1.24279773649569, 1.31329384823548, 1.25318760628953, NA, 
    NA, 1.50061887593633, 1.23761069017347, 1.3371138522461, 
    1.26603684536761, NA, 1.44443540984696, 1.04575930708546, 
    1.25978659512216, NA, NA, 1.4371914230515, 1.26253581195448, 
    1.3877662137778, NA, 1.14324188368595, 1.17163931208686, 
    1.25058009296007, 1.26351343366371, NA, NA, 1.22534113531432, 
    1.28382253590999, 1.09218308264888, 1.32107669092895, NA, 
    1.19600088627774, 1.1379310586461, 1.23882294450349, NA, 
    NA, 1.30237860604105, 1.41552701512815, 1.38060224976133, 
    NA, 1.24027949824318, 1.45434037039739, 1.1940021505841, 
    1.26649049527749, NA, NA, 1.29304319245556, 1.21372265674401, 
    1.3410752077428, 1.19142673679823, NA, 1.2405609294109, 1.02634715121364, 
    1.26244359174584, NA, NA, 1.32218971520328, 1.26967815741062, 
    1.25115503803839, NA, 1.37212761815796, 1.36558447987809, 
    1.344591176864, 1.33458023607095, NA, NA, 1.31813569576491, 
    1.21297793663775, 1.35526256234014, 1.34590192023138, NA, 
    1.23570316090567, 1.14698407040662, 1.19709359877248, NA, 
    NA, 1.29254986534689, 1.23650456522942, 1.40175485053384, 
    NA, 1.27434415618616, 1.38685389540648, 1.44229567501787, 
    1.32795922002984, NA, NA, 1.20114236506095, 1.17048276062289, 
    1.22327517280591, 1.3985990145164, NA, 1.2742566312885, 1.01920059685192, 
    1.29911546188309, NA, NA, 1.1642847068135, 1.1388721540768, 
    1.16250686324563, NA, 1.38066936017745, 1.28711510655299, 
    1.33825243615935, 1.16831645707954, NA, NA, 1.09677129573948, 
    1.23977883530327, 1.53715807540486, 1.26585016950624, NA, 
    1.24251343788971, 1.08887065486216, 1.329159675624, NA, NA, 
    1.26579792540935, 1.11392411674875, 1.17735687620822, NA, 
    1.31471399893771, 1.39012030670967, 1.29801476284041, 1.19677101240234, 
    NA, NA, 1.24129956248743, 1.20026155380336, 1.33563319755105, 
    1.0758486603097, NA, 1.30203588746558, 0.943714125505137, 
    1.39449870621875, NA, NA, 1.40168513576779, 1.23661307105164, 
    1.23400583007324, NA, 1.54558085612497, 1.18748886482562, 
    1.23544800139802, 1.25252891950006, NA, NA, 1.36995109408382, 
    1.18597037377981, 1.33949502973257, 1.21542808690728, NA, 
    1.15037555585882, 1.03783019933584, 1.22212333318928, NA, 
    NA, 1.14789907972178, 1.36502115583105, 1.26314502037753, 
    NA, 1.32418476319483, 1.12212772704404, 1.12578115679821, 
    1.10864648183791, NA, NA, 1.31098137263761, NA, NA, 1.22403886504312, 
    NA, 1.12506662881475, 0.973914658843784, 1.24714013227133, 
    NA, NA, NA, 1.24778658011945, 1.32638668195368, NA, 1.40325912203921, 
    NA, 1.23587524686235, 1.27057747993579, NA, NA, 1.35937113650639, 
    NA, NA, 1.17830150566035, NA, 1.27119331170783, 1.06583631430109, 
    1.25129293317564, NA, NA, NA, 1.35676581082857, 1.28234904098861, 
    NA, 1.18144017296729, NA, 1.15759879351102, 1.29292097657026, 
    NA, NA, 1.22384280782334, NA, NA, 1.30987759316018, NA, 1.22014421023619, 
    0.914066912777639, 1.10066811570867, NA, NA, NA, 1.40273000514166, 
    1.45678106087848, NA, 1.49809372226303, NA, 1.18200719995209, 
    1.29683043691188, NA, NA, 1.28569902423238, NA, NA, 1.14020587382339, 
    NA, 1.27212608525058, 1.07245691694315, 1.17121029728638, 
    NA, NA, 0.0804235451539638, 0.0678716573346165, 0.0587553937721929, 
    0.074509248999096, 0.0857907638701938, 0.0705822108509468, 
    0.0713463281817648, 0.0739990799707663, 0.0810683116842992, 
    0.072340914069677, 0.0743610977864208, 0.0775826363489885, 
    0.0734536361399504, 0.0834732687082324, 0.0782241357249728, 
    0.0783139803460818, 0.0658591197571831, 0.0730462410318304, 
    0.0794572289102907, 0.0679152137414439, 0.0719530920156623, 
    0.0732185546362375, 0.0660656419096116, 0.0738790533411141, 
    0.0878986034406145, 0.0771689408999241, 0.0721955060278779, 
    0.0758317836987863, 0.0656665921447558, NA, 0.0821930167531016, 
    0.0975386253961395, 0.0727005000013169, 0.0678450512227363, 
    0.0718004127052723, 0.0720824845126233, 0.0670223845193508, 
    0.0779904566407375, 0.0764428286241503, 0.0672148833297536, 
    0.064865683769275, 0.0649041686896495, 0.0746075533522917, 
    0.0851328038780864, 0.0669179735760494, 0.0566797422604522, 
    0.079834147764188, 0.0702327502227076, 0.0657753714762142, 
    NA, 0.0782924711682553, 0.0674608659465523, 0.0751574467748965, 
    0.0781265124181078, NA, 0.0693323800089626, 0.0655109484006966, 
    0.0684646276873875, 0.0698321166864401, NA, 0.0769796805716992, 
    0.0699513024304664, 0.0676475412724226, 0.0698994779760624, 
    0.0679991313092217, 0.0570102799546785, 0.0696844926388879, 
    0.0639286331377465, NA, NA, 0.081271646815344, 0.0755841441704363, 
    0.0752420465489567, 0.0676374914879128, NA, 0.0725333731273408, 
    0.068512879081199, 0.0754191799855795, 0.0664304284584109, 
    NA, 0.0862164461718111, 0.0697814719518113, 0.0656876713771321, 
    0.0732126821776418, 0.0679307732116266, 0.0594749814354925, 
    0.0686548216624057, 0.0794641595266191, NA, NA, 0.0687222589208544, 
    0.0816767588544072, 0.0709924772703759, 0.0629162504738932, 
    NA, 0.0744406262214926, 0.0701087437477347, 0.0709470528535135, 
    0.0750581541754126, NA, 0.0764274879509039, 0.0663933218671489, 
    0.0611740408499472, 0.0743931418729981, 0.0670422607379074, 
    0.0603343096474501, 0.0808362663993611, 0.0778521292874039, 
    NA, NA, 0.0697364250386775, 0.0725502089244233, 0.0741213239315958, 
    0.06844572698094, NA, 0.0704230450083947, 0.0880722488001368, 
    0.0702926092683189, NA, NA, 0.0693121042981364, 0.0728360938532716, 
    0.0696243023830103, NA, 0.0954324753605086, 0.0731266854329322, 
    0.0680964802768469, 0.0653613765438291, NA, NA, 0.064516122819704, 
    0.0678332953821022, 0.0662733277115573, 0.0671549744804289, 
    NA, 0.0669289610013693, 0.0772693818143633, 0.0838157217551989, 
    NA, NA, 0.0777347532194346, 0.0725863131586294, 0.0693690953824442, 
    NA, 0.0654733348440021, 0.069859693848269, 0.0753741101595034, 
    0.0728581842211896, NA, NA, 0.0685433622073394, 0.0597313085442911, 
    0.0685318767473757, 0.0688588542076459, NA, 0.0658278713977752, 
    0.0755756695577896, 0.073752973686709, NA, NA, 0.0708469545104874, 
    0.0664091573033708, 0.0675021173900593, NA, 0.0710953767026112, 
    0.0761104977536824, 0.0752269006750837, 0.0677004145639547, 
    NA, NA, 0.0772249325842697, 0.0740622069705379, 0.0696053166984252, 
    0.068682016054793, NA, 0.064857359591081, 0.0665792812831997, 
    0.0689314896964349, NA, NA, 0.070903087563461, 0.0726939940415117, 
    0.0680154885479689, NA, 0.065819242159651, 0.0797961823669198, 
    0.0640172822466161, 0.0639942063442176, NA, NA, 0.064163794292836, 
    0.0680067757504838, 0.0729055569244585, 0.075482430228691, 
    NA, 0.0717236692978171, 0.0703336556848433, 0.0640017087382326, 
    NA, NA, 0.075584076007715, 0.0727757428214732, 0.0692335402438129, 
    NA, 0.069317727518437, 0.0718893462109052, 0.0702914367276636, 
    0.0709872627550477, NA, NA, 0.0632006747630634, 0.0620986234739044, 
    0.0778480467498753, 0.0772788987926197, NA, 0.0708740473422999, 
    0.0746233734898288, 0.0641382792503579, NA, NA, 0.0752870428932161, 
    0.0785093448006323, 0.0682249625468165, NA, 0.0694746268140931, 
    0.0652119606925167, 0.0696722515972681, 0.0678696761893098, 
    NA, NA, 0.0638863929703255, 0.0609999516427251, 0.0837555747811495, 
    0.072324561071372, NA, 0.0638398775351415, 0.0721102512097451, 
    0.0724844756674523, NA, NA, 0.0758735232744783, 0.07249122958986, 
    0.0695893360480757, NA, 0.0745658070072484, 0.072145610016223, 
    0.0736696757474735, 0.0748896422916476, NA, NA, 0.0597702002204155, 
    0.0712773055750588, 0.0671917310011768, 0.0712043700648451, 
    NA, 0.0689199098825644, 0.0658692960282695, 0.0624296529654388, 
    NA, NA, 0.0690722768418759, 0.0715474928834395, 0.0723733160323597, 
    NA, 0.0696220569665772, 0.0684233908507223, 0.0651113721484508, 
    0.0694860334480282, NA, NA, 0.0623312661762529, 0.0661903886375797, 
    0.0750043534884051, 0.0661441410828667, NA, 0.0742656220148891, 
    0.0654284733385924, 0.0612878886095178, NA, NA, 0.0651360875411508, 
    0.0738516540418087, 0.0646472786589894, NA, 0.0666968943643602, 
    0.0718592649765253, 0.0723134938384245, 0.071256423445204, 
    NA, NA, 0.0650845653790339, 0.0630918037650306, 0.0765028560661403, 
    0.0829669382872988, NA, 0.0749456375580287, 0.0698022162797097, 
    0.0644922087186173, NA, NA, 0.0780340215355805, 0.069477131689957, 
    0.066523551721785, NA, 0.0741387829803338, 0.0652699561971841, 
    0.0652717956697163, 0.0700214917294487, NA, NA, 0.0631725789192081, 
    0.0699485414548274, 0.065683446185689, 0.0804341365898728, 
    NA, 0.0798211436109721, 0.0754198282348262, 0.0673862178046673, 
    NA, NA, 0.077892645799893, 0.0775619715477234, 0.0616569365168399, 
    NA, 0.0734834832591658, 0.0743503949125669, 0.0649488734002283, 
    0.068154991917778, NA, NA, 0.0558363346842945, NA, NA, 0.0747770484156781, 
    NA, 0.0725778731517003, 0.0661444616071678, 0.0649379852831665, 
    NA, NA, NA, 0.0675434535739592, 0.0718056184870952, NA, 0.0807766813607717, 
    NA, 0.0738127204748799, 0.0602331302835741, NA, NA, 0.0634793341221553, 
    NA, NA, 0.0814444201975689, NA, 0.0648982527961644, 0.0626977529712799, 
    0.0630002408947021, NA, NA, NA, 0.0675272204509809, 0.0687827951195839, 
    NA, 0.0753517262596631, NA, 0.0736997979893049, 0.0734119682272033, 
    NA, NA, 0.0628598058994174, NA, NA, 0.0669627976225507, NA, 
    0.0653087863356197, 0.0638443086798196, 0.0692256380515312, 
    NA, NA, NA, 0.0716410901877879, 0.0663111282957018, NA, 0.0708679727656068, 
    NA, 0.0778202474455474, 0.066793659315481, NA, NA, 0.0610659846487723, 
    NA, NA, 0.0742037092496909, NA, 0.0634991537364009, 0.0695931294681503, 
    0.0823283293022709, NA, NA, 12.0850331167715, 8.41329064128682, 
    10.5265656849152, 7.60033772439623, 7.41187794890687, 7.8572039561562, 
    9.08916251841926, 8.47034161143938, 7.27519086086496, 10.3541894565755, 
    10.3215899416595, 10.5545123865893, 9.35258820709552, 9.38170003488287, 
    9.30189224534905, 9.11574808412561, 10.6182240386914, 8.98392483141278, 
    10.9687758270911, 7.15302547462224, 10.3051669281093, 9.33268172501281, 
    10.4721337508269, 8.06532225134515, 10.1390424664474, 7.72410225635787, 
    9.76153105405298, 9.74426293551128, 8.37481991665124, NA, 
    10.8378140740274, 8.37224268942668, 10.1623650816227, 9.35872738350958, 
    10.8160721558989, 10.226261191854, 10.2698024180301, 11.0588073321809, 
    9.87062858492291, 10.8485146066323, 9.39854593014672, 9.46322125355939, 
    10.6516039554206, 9.5529814552963, 10.710769286633, 8.79670223712865, 
    9.88966168724957, 11.6609024713354, 11.3671061522747, NA, 
    9.9298679748786, 10.1687671293184, 8.98196673792115, 9.29555434050403, 
    NA, 9.66748424802445, 10.2396049132602, 9.15696348314607, 
    11.5486590829786, NA, 10.2777380369754, 10.2688799161838, 
    10.2393894022275, 10.0460371003854, 12.8764539706539, 7.91696051037977, 
    11.1214716849575, 11.2285887754974, NA, NA, 10.0210718475958, 
    11.6119882046792, 11.4703315768353, 10.6449330585145, NA, 
    10.5921564138577, 9.82689655760549, 9.42747457639849, 10.3033212161269, 
    NA, 7.62416725859917, 8.94723899338883, 10.3483478431934, 
    9.8827686686452, 10.4172742520644, 9.57333176290712, 9.62936129827583, 
    10.2626679224355, NA, NA, 9.4439968117757, 10.5727986045296, 
    12.8408265953745, 10.3934484685439, NA, 10.4411285081267, 
    9.0612029318191, 10.6452139807758, 12.1203923303288, NA, 
    10.6561125243025, 10.3666542777376, 11.1735238343844, 10.42460903897, 
    10.1030560702048, 10.0157742936742, 8.20252549439877, 10.3409884190814, 
    NA, NA, 9.01448937838636, 9.64750000170394, 10.438276452978, 
    9.51251293174981, NA, 9.96328487943027, 11.3134099235229, 
    9.36244736506102, NA, NA, 9.24567087346466, 12.1447519828156, 
    9.84634135436872, NA, 9.56518197587229, 6.34479391680771, 
    9.81603756707624, 10.6277732215311, NA, NA, 8.78849361973083, 
    9.71137456171807, 11.1454091938266, 8.84899092127488, NA, 
    9.50737134026016, 9.43375608461847, 10.3624291612656, NA, 
    NA, 11.4245235882848, 12.4507192938351, 9.73114630750024, 
    NA, 9.87245762782472, 7.25577517807947, 11.0948363212434, 
    12.8023876499169, NA, NA, 9.5716196024422, 10.1258171426398, 
    10.2657631844437, 8.41382439718848, NA, 9.60797495007279, 
    10.168720407057, 11.0393133691503, NA, NA, 9.86509689486207, 
    10.4403604104563, 9.71953014086801, NA, 8.72645884587117, 
    7.37337427999477, 11.2804362885452, 10.107184193174, NA, 
    NA, 8.98524419475655, 8.80004926155059, 9.99537343671288, 
    10.7934106855271, NA, 9.40286836995587, 10.1298956137492, 
    8.34551119918475, NA, NA, 12.8848066391639, 9.48423251027693, 
    9.65299216429638, NA, 9.7342882489112, 7.48417862822541, 
    9.97927580059201, 10.0169498093667, NA, NA, 10.1822439101829, 
    10.3984518561858, 9.42333546225113, 12.2684286603874, NA, 
    9.73651230298511, 9.75560427107939, 10.081073482893, NA, 
    NA, 10.757475136933, 10.4074610413454, 9.99922554527429, 
    NA, 10.5592056321351, 10.101846484508, 10.8875676232873, 
    10.4357604524231, NA, NA, 8.96464390500877, 11.531153023007, 
    9.76817349126247, 10.2716319422746, NA, 9.77260629877678, 
    9.22610663858972, 9.47586237401125, NA, NA, 8.84259325164811, 
    10.6029166527697, 11.1773057701311, NA, 9.92692073428028, 
    11.1233264944555, 9.94794462945074, 9.83648587999178, NA, 
    NA, 10.9802199654278, 11.4732099274641, 9.07522159219909, 
    9.1805948820602, NA, 8.70702893847057, 10.0594762686432, 
    10.7232470605773, NA, NA, 10.2261169966114, 10.2904561078209, 
    9.97056118689905, NA, 10.6173225291212, 11.6417046581145, 
    10.7327404007977, 10.143671005755, NA, NA, 10.8948367747807, 
    9.63908493471945, 10.8807147625117, 11.1448046877058, NA, 
    10.3429793801704, 9.71202632446118, 11.1091379604608, NA, 
    NA, 10.9936035810179, 8.82420599505052, 9.08390821960028, 
    NA, 11.8319725838156, 10.2899629840807, 8.89727901203042, 
    9.54597827359951, NA, NA, 8.94785937954658, 11.1315213531787, 
    9.67519398698225, 10.4683345644516, NA, 9.9718495227409, 
    9.39466183399782, 9.44518746235038, NA, NA, 10.1402356336121, 
    9.24116698186673, 9.82753028108148, NA, 9.76942502701662, 
    10.7248847627316, 9.00974344090347, 11.1612927209096, NA, 
    NA, 10.101694716892, 8.76212568061853, 10.4238481435744, 
    8.83974793362048, NA, 10.3257622628732, 8.77675523057609, 
    11.8521473442288, NA, NA, 11.7050181803995, 8.7923695846182, 
    9.01772181913297, NA, 10.5574559357423, 9.93556358139435, 
    10.7824087350489, 9.98137102893379, NA, NA, 9.29793265021495, 
    10.1258417655956, 10.5246978612261, 10.8828202346536, NA, 
    10.8861582659552, 9.00308223583802, 10.3408437770095, NA, 
    NA, 9.33412693953986, 9.41820623157224, 9.91842429011009, 
    NA, 9.68664170922365, 9.52968698794079, 11.0945310741253, 
    12.4795991280366, NA, NA, 8.56782374003432, NA, NA, 9.27141959977817, 
    NA, 11.1987789231634, 9.57636838130155, 9.54912343485496, 
    NA, NA, NA, 10.1795363975533, 9.44271330479758, NA, 10.3723629893886, 
    NA, 13.0583358881417, 10.9827315141787, NA, NA, 9.61930036353485, 
    NA, NA, 9.25466808656056, NA, 9.63685967820921, 9.07566329326546, 
    10.8561865151146, NA, NA, NA, 12.998078443136, 10.0703572647601, 
    NA, 10.4383088134905, NA, 12.3845238237437, 11.5910637453792, 
    NA, NA, 9.2272770609628, NA, NA, 10.0897419806794, NA, 11.4057355475618, 
    10.0253924171302, 7.36021800966441, NA, NA, NA, 11.7879272977593, 
    9.56415304988234, NA, 10.0304527014013, NA, 9.31734883571079, 
    10.1162953530162, NA, NA, 10.0948428711006, NA, NA, 10.3348424185004, 
    NA, 9.91704797574461, 9.83382865117538, 10.5510035744098, 
    NA, NA, 93.3285678349824, 134.952165952182, 169.184029633125, 
    132.383188040811, 137.132384960162, 181.042264477096, 166.946267851292, 
    148.308839489674, 151.56438917635, 144.311394633209, 122.836899788726, 
    124.702852799168, 107.967078641891, 81.9018047293824, 154.331794812136, 
    141.505440164938, 127.152228357676, 130.221238158185, 113.613912297129, 
    123.095676740282, 145.119859307359, 147.430315831573, 115.133416094063, 
    98.8691196853405, 129.538041869688, 147.020613656007, 120.146789425976, 
    113.938657399001, 129.464670266249, NA, 119.306410579213, 
    169.736189138577, 107.718051209722, 114.210653502974, 136.973157951599, 
    150.713079110799, 117.033787311315, 160.074225727456, 158.25040689577, 
    113.894048903375, 115.555032463963, 137.409037018765, 123.984490142389, 
    124.883719340216, 126.230311407278, 199.185889227126, 121.082622601455, 
    123.369097962585, 121.042732860149, NA, 110.849598443372, 
    104.467620337184, 99.8990826787407, 111.671586066007, NA, 
    132.489931702565, 134.055443074825, 122.809309791332, 92.2692289635133, 
    NA, 159.675460849593, 128.935319578249, 121.289132416716, 
    102.888985216611, 96.4890065087061, 220.64431015852, 108.57249551506, 
    126.945581043285, NA, NA, 108.186034000224, 145.338158470743, 
    102.52004202923, 128.466760299625, NA, 163.158352059925, 
    139.268971015147, 103.890588385507, 113.549635482385, NA, 
    142.249566813035, 130.096279890521, 128.931247994682, 98.918930389982, 
    103.944394206038, 153.841985492818, 111.185993905946, 142.96375860898, 
    NA, NA, 105.410892879191, 125.919043868056, 110.085400927448, 
    121.393098553116, NA, 153.511772523435, 126.264085316741, 
    100.045795028941, 123.124570111536, NA, 111.057936700494, 
    119.761241516152, 108.638521523755, 113.997447760435, 101.712970156985, 
    148.080169816797, 121.694534345045, 134.846810319338, NA, 
    NA, 106.112755908987, 119.099676736899, 121.225351035337, 
    126.121473619542, NA, 92.6499126707379, 119.70763465519, 
    112.697468886081, NA, NA, 101.783185232465, 106.210029742234, 
    108.492441320912, NA, 111.610446411877, 142.703709254884, 
    125.033865124734, 108.225300127073, NA, NA, 114.884469658718, 
    104.471266634792, 118.891619445718, 137.085710876111, NA, 
    102.504748348838, 105.641435361326, 107.307499263182, NA, 
    NA, 113.320788132034, 119.632373000717, 109.684841064055, 
    NA, 109.936606836291, 122.528880343731, 108.617588317092, 
    109.976676967239, NA, NA, 123.306456383171, 113.176200557393, 
    107.642808395316, 122.765416649505, NA, 98.2471256859158, 
    113.461078257441, 114.256334743523, NA, NA, 112.368000371622, 
    104.187690132233, 102.718549204361, NA, 105.247576916583, 
    150.794827474933, 112.804694629823, 118.094704206303, NA, 
    NA, 142.355617977528, 101.10423869202, 109.139873841908, 
    138.482417505977, NA, 100.811842080738, 118.349129683487, 
    109.833630122235, NA, NA, 112.922699607068, 122.878315923103, 
    112.289004478667, NA, 105.565683393732, 152.896891275368, 
    117.645231366339, 143.528578836893, NA, NA, 142.453086690597, 
    109.721532361596, 100.03611987209, 121.794672204947, NA, 
    112.231127923619, 121.489190028038, 100.333307394473, NA, 
    NA, 109.511416601854, 121.665530219578, 124.926635822868, 
    NA, 102.324328266643, 149.751918259643, 103.59259006267, 
    135.899993603821, NA, NA, 137.697311196491, 107.182757247434, 
    96.8601605313458, 110.637000491136, NA, 110.593073006033, 
    112.518352507679, 107.556314621202, NA, NA, 113.813923771756, 
    116.388347560573, 123.129968398876, NA, 108.656632589217, 
    138.372544429757, 113.404414790981, 108.827498064129, NA, 
    NA, 163.752892538173, 126.179761058171, 107.176847916266, 
    117.09018430479, NA, 110.29303546479, 117.092570684544, 113.950838413947, 
    NA, NA, 125.00393258427, 108.05233021531, 119.1562052214, 
    NA, 107.921589558616, 104.036058703359, 108.390356909009, 
    134.350153009957, NA, NA, 140.629690371508, 143.154810381105, 
    102.473271513607, 122.233788394781, NA, 107.059771439547, 
    121.339040517772, 112.952599203527, NA, NA, 106.454199823955, 
    114.457496963315, 114.3876847441, NA, 103.346526032735, 112.391685799622, 
    132.175053909885, 151.333477558943, NA, NA, 120.727509173279, 
    122.388783501143, 105.244789905814, 104.180384043128, NA, 
    113.266940418184, 114.224779864827, 116.478975144705, NA, 
    NA, 103.239985871048, 117.166333093984, 124.028121072326, 
    NA, 105.969331090328, 99.3490210916312, 124.360652724695, 
    139.79145131423, NA, NA, 143.270345412225, 167.126611034453, 
    101.08192275136, 124.728977714079, NA, 110.743436856784, 
    120.275858515942, 119.705795051341, NA, NA, 108.575491573034, 
    138.305540794483, 136.462220806987, NA, 109.259237233945, 
    110.464880021329, 98.7054683373893, 138.309209076202, NA, 
    NA, 132.301138702122, 124.524442968631, 102.63766016164, 
    106.093531129854, NA, 122.002850687653, 123.137878639857, 
    127.106944684529, NA, NA, 129.076511503478, 135.129495235818, 
    139.668912528204, NA, 114.410254261367, 121.270599225527, 
    101.392328662256, 140.983984493799, NA, NA, 118.890188002419, 
    NA, NA, 133.909217146638, NA, 112.71130848585, 119.007197224667, 
    106.170930485878, NA, NA, NA, 113.391639552876, 118.474804453617, 
    NA, 102.207129673432, NA, 113.019058681383, 148.47208400214, 
    NA, NA, 121.507135878289, NA, NA, 119.839661650153, NA, 125.581778897973, 
    122.99923273493, 123.912490868612, NA, NA, NA, 112.584125820985, 
    130.250635162832, NA, 94.176922012509, NA, 115.95929212479, 
    139.260180229651, NA, NA, 119.262338317512, NA, NA, 121.846966119503, 
    NA, 123.136602732944, 114.665742647024, 184.374734917344, 
    NA, NA, NA, 116.704658158139, 122.325461150563, NA, 98.9783981740806, 
    NA, 115.331870819671, 149.142525647289, NA, NA, 111.054164687449, 
    NA, NA, 132.950076388496, NA, 101.93507223114, 114.548988743644, 
    136.408629399633, NA, NA, 0.415831929822382, 0.223829997125395, 
    0.409378900841295, 0.246098230014206, 0.221060676991088, 
    0.206350974437547, 0.264143388637129, 0.226699914578437, 
    0.243211395271215, 0.393036555078022, 0.413813819264381, 
    0.395298573154022, 0.398314760992664, 0.401398174616288, 
    0.24349759237267, 0.416640067451743, 0.400644883816712, 0.435598145205322, 
    0.432575495028536, 0.221205259266434, 0.384338529050537, 
    0.232471947919661, 0.3921909569407, 0.400908147132326, 0.370386444785262, 
    0.230260663877321, 0.401216137719502, 0.381994520867984, 
    0.243073273624161, NA, 0.421199800203061, 0.235900339599539, 
    0.368676636634323, 0.399415773022692, 0.419730961538462, 
    0.385499041600611, 0.38062785352042, 0.413293167314895, 0.248145362004036, 
    0.407889678424824, 0.407718188928642, 0.382920294412492, 
    0.431164680480748, 0.380866964822644, 0.386750225637866, 
    0.465548474995551, 0.38425349544329, 0.35624691100319, 0.403195604052627, 
    NA, 0.37625465133778, 0.345913758719953, 0.395298941507129, 
    0.407664637836374, NA, 0.41947195545327, 0.376104342200615, 
    0.407805956302593, 0.361881683458213, NA, 0.424832913429447, 
    0.3958088793176, 0.393112380617978, 0.4111167547012, 0.397755408514895, 
    0.384415623910821, 0.40344456546923, 0.357013220812732, NA, 
    NA, 0.418460533853335, 0.357755316649642, 0.412192127835389, 
    0.386452211318644, NA, 0.328137132958802, 0.389414486205254, 
    0.395158442363156, 0.386962830969977, NA, 0.251705446720694, 
    0.422145118197395, 0.382210037933811, 0.390419302062898, 
    0.414260301843006, 0.377789884917301, 0.397385931238365, 
    0.377292024585218, NA, NA, 0.418286488411268, 0.369692491740128, 
    0.402410864339641, 0.373676106295801, NA, 0.340056680831539, 
    0.42111647150739, 0.379265552238024, 0.397897817343705, NA, 
    0.39812761949183, 0.42468587944117, 0.436251000202607, 0.390969620309879, 
    0.400179864829867, 0.368360179756618, 0.411944071516545, 
    0.353810098561009, NA, NA, 0.375309737629678, 0.387214074375135, 
    0.406182289743602, 0.417396075908339, NA, 0.380623056616422, 
    0.391184583386564, 0.416462052275415, NA, NA, 0.389664471611645, 
    0.422582018506279, 0.42477250599636, NA, 0.401829016109399, 
    0.240521483307483, 0.397731260003942, 0.37327813670412, NA, 
    NA, 0.412360612952899, 0.382609570981752, 0.401733844104606, 
    0.389161820702064, NA, 0.418842864262414, 0.397853349453677, 
    0.420005401361518, NA, NA, 0.387434224960428, 0.404069345542098, 
    0.417500921820133, NA, 0.406660306246977, 0.240655810155676, 
    0.422166473568675, 0.38647749893018, NA, NA, 0.425674406205046, 
    0.391015218620612, 0.391650259926705, 0.4093621261945, NA, 
    0.395874283842015, 0.409861361373941, 0.417290221463522, 
    NA, NA, 0.398052012668334, 0.418175811549339, 0.423178100171621, 
    NA, 0.404498792241677, 0.246674008646244, 0.384395262958403, 
    0.394007659304244, NA, NA, 0.381878960674157, 0.384951644073451, 
    0.365522885526863, 0.429186164126058, NA, 0.41632739741569, 
    0.402005502210865, 0.430122455232152, NA, NA, 0.398842724991466, 
    0.402181641884212, 0.393554065746052, NA, 0.377975086725075, 
    0.24649038579539, 0.397928408938722, 0.393215606726607, NA, 
    NA, 0.395922249481307, 0.362568341361721, 0.398870788980879, 
    0.447130211168493, NA, 0.407785869758054, 0.393586222623556, 
    0.402016051662618, NA, NA, 0.383766869239298, 0.426411024271132, 
    0.406752124917383, NA, 0.381992075305353, 0.361886335668077, 
    0.387216115091867, 0.382007500586316, NA, NA, 0.371266129415049, 
    0.357362497233572, 0.409179996592349, 0.435291837893123, 
    NA, 0.399458935443996, 0.418025052593549, 0.421476736077972, 
    NA, NA, 0.405017209229413, 0.419722680005153, 0.429123691479401, 
    NA, 0.387650893696399, 0.331884987515606, 0.401874927490443, 
    0.381854532045913, NA, NA, 0.383255550273696, 0.395313140378325, 
    0.38296125837343, 0.421149993457718, NA, 0.405584826733709, 
    0.463616158677436, 0.381270756342728, NA, NA, 0.412245497592295, 
    0.41010920711795, 0.420822816518139, NA, 0.409707453212167, 
    0.368037032836628, 0.433530283229879, 0.408448736564051, 
    NA, NA, 0.406404981084398, 0.386892058648515, 0.416175391567662, 
    0.417832459952486, NA, 0.398834126040938, 0.401599510203121, 
    0.401276173019485, NA, NA, 0.42230187347534, 0.409152457029434, 
    0.407491973435114, NA, 0.368015238088767, 0.461803690985203, 
    0.419863852003178, 0.412331055247329, NA, NA, 0.368192025098114, 
    0.4084212109052, 0.391782636860471, 0.411022130705077, NA, 
    0.422643733267752, 0.39779105526299, 0.395866249574396, NA, 
    NA, 0.403315477297557, 0.402332199557034, 0.392251408638564, 
    NA, 0.406162610025427, 0.406189640753555, 0.388301621119381, 
    0.390254679631793, NA, NA, 0.34291591175842, 0.383342237948179, 
    0.388111353271646, 0.417286536696811, NA, 0.432463996064965, 
    0.417592184976461, 0.396153035455083, NA, NA, 0.376046188358302, 
    0.377459949579648, 0.399643459718874, NA, 0.379888209750878, 
    0.393242551887854, 0.37123304028773, 0.376441753361083, NA, 
    NA, 0.363513256079127, 0.418967210172098, 0.410460697976214, 
    0.428729070972293, NA, 0.410373505789828, 0.396811193920234, 
    0.39920843474503, NA, NA, 0.417553975387908, 0.38509681249502, 
    0.387789748583794, NA, 0.379429407568867, 0.408812972321742, 
    0.378042151855635, 0.384304932550941, NA, NA, 0.367054250788806, 
    NA, NA, 0.465201725643677, NA, 0.400378941429488, 0.395079357069172, 
    0.369766456974739, NA, NA, NA, 0.412417413018958, 0.36953558523784, 
    NA, 0.439650032738037, NA, 0.382034527685564, 0.396429624665597, 
    NA, NA, 0.374419488386552, NA, NA, 0.423601966978737, NA, 
    0.413446803192736, 0.387427488311658, 0.402914635005392, 
    NA, NA, NA, 0.382231420578396, 0.386289522519887, NA, 0.397309604250844, 
    NA, 0.39990022494091, 0.366008249670639, NA, NA, 0.366426263286621, 
    NA, NA, 0.398715744284917, NA, 0.408520577033106, 0.408582526785737, 
    0.226613813836646, NA, NA, NA, 0.395337774419606, 0.366708703367091, 
    NA, 0.405341498284192, NA, 0.369374908987267, 0.406027964260387, 
    NA, NA, 0.373105590962609, NA, NA, 0.423914537296094, NA, 
    0.428950957731407, 0.403043226047858, 0.412487711547242, 
    NA, NA, 0.957800023198614, 0.456706860306708, 0.891041057595394, 
    0.433602452860648, 0.417425788500503, 0.418891874558026, 
    0.427412695602428, 0.479151809544593, 0.407941847244114, 
    0.914109384013979, 0.878371568111975, 0.917083847772557, 
    0.846170212311431, 0.89032995290813, 0.460845641414043, 0.857695693496111, 
    0.925011377981866, 0.89702904830838, 0.897866987805422, 0.470874328102802, 
    0.922925920521426, 0.489653119258602, 0.914185472263295, 
    0.958943940454054, 0.91075489480203, 0.413527864120898, 0.94029448695697, 
    0.91529695859329, 0.407076166743446, NA, 0.906094426318176, 
    0.448231057332181, 0.853825185389308, 0.922441145833333, 
    0.930871260668755, 0.947955265438461, 0.962869564733676, 
    0.894485643546528, 0.49615299883186, 0.892311784875374, 0.918927107644659, 
    0.949103974280882, 0.916421571804191, 0.912506183824088, 
    0.951042680821052, 0.903551516133679, 0.952197110202334, 
    0.890241906491566, 0.852400745353273, NA, 0.91648426037911, 
    0.836268280877917, 0.927197411268908, 0.923559634710384, 
    NA, 0.93394192989528, 0.916242001088783, 0.951539184906853, 
    0.845710505459367, NA, 0.849855107224344, 0.92300350546416, 
    0.934542665089691, 0.921953333638926, 0.902562413922187, 
    0.901104272757792, 0.885400498155183, 0.882285135371875, 
    NA, NA, 0.927616243725279, 0.865812941047716, 0.942730039283602, 
    0.921354039593365, NA, 0.816537948970038, 0.937081013792328, 
    0.927466357102085, 0.789289241723247, NA, 0.435882524134702, 
    0.932995140184082, 0.919092498797983, 0.90599342964152, 0.943296232764122, 
    0.933972825076898, 0.922691963654274, 0.89139595533714, NA, 
    NA, 0.919512902166052, 0.900684121599662, 0.917415134216597, 
    0.944830128784375, NA, 0.818668612970741, 0.949666771374492, 
    0.917387374610408, 0.899875671152817, NA, 0.896725431215734, 
    0.872906158369036, 0.884458852224464, 0.910431529818496, 
    0.92224292662762, 0.887633432661819, 0.880941605740765, 0.865948780677114, 
    NA, NA, 0.915459823397708, 0.859702699076072, 0.931142931727569, 
    0.955411407910904, NA, 0.896989739657325, 0.932371295988356, 
    0.928997439620182, NA, NA, 0.893071731799287, 0.997898081625909, 
    0.910408084040642, NA, 0.91079205984799, 0.459077853083339, 
    0.885019886643888, 0.915700786405386, NA, NA, 0.950565886790411, 
    0.826125988026934, 0.952931872849804, 0.908113200411251, 
    NA, 0.957061960351979, 0.905204870533238, 0.906904884351047, 
    NA, NA, 0.929163813747371, 0.955100911969686, 0.915520373341851, 
    NA, 0.901097727945442, 0.40109366858169, 0.928462407547161, 
    0.938303343236273, NA, NA, 1.00059829875315, 0.873347065177349, 
    0.9765753563181, 0.926557957126729, NA, 0.871630159650416, 
    0.936601721696564, 0.891010334274208, NA, NA, 0.874168067715376, 
    0.901687808988764, 0.909758115518547, NA, 0.895881898653605, 
    0.475863658744374, 0.888472425487979, 0.900689997875308, 
    NA, NA, 0.883579359082397, 0.87754531816252, 0.93929744969008, 
    0.983753126976747, NA, 0.925479988715422, 0.946834957385603, 
    0.908320788355369, NA, NA, 0.929958075760987, 0.904282274864652, 
    0.908028377052723, NA, 0.969569677852154, 0.449175030652546, 
    0.938242879861058, 0.90100488149678, NA, NA, 0.874001415383623, 
    0.870236983219913, 0.908585229964737, 0.941016703639516, 
    NA, 0.886363287481038, 0.94139013541888, 0.926434141170665, 
    NA, NA, 0.932290154243867, 0.967459395608431, 0.89980445013586, 
    NA, 0.936919206340013, 0.849508077970718, 0.883340382322965, 
    0.899366011982176, NA, NA, 0.866921974032899, 0.86844594827691, 
    0.913276029776675, 0.938389919777234, NA, 0.91360964853047, 
    0.954737071123829, 0.903088721675126, NA, NA, 0.91771041571339, 
    0.937925678417954, 0.908479534956305, NA, 0.95868912180724, 
    0.735147689928031, 0.924039559804043, 0.896177498064129, 
    NA, NA, 0.866441397291847, 0.925186971981226, 0.882826526020411, 
    0.907006856631138, NA, 0.97474174528409, 0.934167551513906, 
    0.867819910896166, NA, NA, 0.966466619404316, 0.899187813416727, 
    0.952451083235088, NA, 0.95936786187829, 0.847572560160428, 
    0.904411432791927, 0.91832206860327, NA, NA, 0.823655697870868, 
    0.900986589187695, 0.931259911976473, 0.930230210489789, 
    NA, 0.898184180505824, 0.955529461287867, 0.881133787512445, 
    NA, NA, 0.921307864732225, 0.970180630946203, 0.84512508554453, 
    NA, 0.929935780772438, 0.935800636527394, 0.892760496538418, 
    0.933951818890274, NA, NA, 0.785781493396771, 0.901188519626441, 
    0.916328163646756, 0.889788228492046, NA, 0.88728150800043, 
    0.962768792721148, 0.881438822320002, NA, NA, 0.906995563720165, 
    0.959405688417156, 0.942550381538997, NA, 0.883124437844918, 
    0.865837515084782, 0.883749763367242, 0.926709302691167, 
    NA, NA, 0.804050355972873, 0.867474160906323, 0.923946379303348, 
    0.951125328700131, NA, 0.923798856177993, 0.9309550895559, 
    0.912977731573571, NA, NA, 0.942007386860175, 0.935580366644213, 
    0.907303217258791, NA, 0.922481175481867, 0.889670868386768, 
    0.901610367456755, 0.89242687760677, NA, NA, 0.792018156585923, 
    0.949262636731659, 0.943509534956305, 0.934597240733566, 
    NA, 0.937778762503348, 0.971466755181151, 0.91672211019879, 
    NA, NA, 0.94371978999465, 0.917308603607193, 0.931648453491058, 
    NA, 0.987045158186352, 0.925381702407269, 0.926221560404664, 
    0.929893474647236, NA, NA, 0.759625627702928, NA, NA, 0.947341766326818, 
    NA, 0.944538355901277, 0.957629603540262, 0.920540999007323, 
    NA, NA, NA, 0.931683355928159, 0.904797635226375, NA, 0.931535156780375, 
    NA, 0.920691051006318, 0.90786823140717, NA, NA, 0.825318125099131, 
    NA, NA, 0.958973239747456, NA, 0.944080830474687, 0.938482815274164, 
    0.960244267227885, NA, NA, NA, 0.916753123069847, 0.89366255263101, 
    NA, 0.948109341894061, NA, 0.870040140416092, 0.90671826062461, 
    NA, NA, 0.852126141910563, NA, NA, 0.977231932173878, NA, 
    0.88203371111966, 0.955258253951774, 0.461649839130841, NA, 
    NA, NA, 0.905210593691683, 0.905109660468641, NA, 0.919186437972842, 
    NA, 0.916880602750809, 0.895465301238422, NA, NA, 0.780214908779624, 
    NA, NA, 0.976697283208998, NA, 0.891013231674692, 0.921643188922217, 
    0.926291824160099, NA, NA, 0.0165452550915833, 0.0117379241604279, 
    0.0157589178860966, 0.0133899044298076, 0.0119919356400338, 
    0.0118389583169639, 0.0122016319034878, 0.0122173483772658, 
    0.0109072722723534, 0.0147215208048259, 0.0157268502712955, 
    0.014785222799606, 0.0138181581546622, 0.013161996952339, 
    0.01343330166455, 0.0147786103428407, 0.0148570906872645, 
    0.0156015868870753, 0.0155226913902265, 0.0128173802143872, 
    0.0161120661875578, 0.0119744082063716, 0.0129828389434038, 
    0.0142710932208419, 0.0148866410376124, 0.0139669287734737, 
    0.015723945677659, 0.0149942675687388, 0.0117790277679056, 
    NA, 0.0155863716106501, 0.0113231975655431, 0.0149231726099233, 
    0.0148551656477198, 0.0158209534356093, 0.0143621869593284, 
    0.0144780378743078, 0.0137631248199366, 0.0122102876592221, 
    0.0152225504180623, 0.0150848215856298, 0.0143431823745047, 
    0.014789981833946, 0.0145339990915873, 0.0150948506939378, 
    0.0140255780034572, 0.0146880616326008, 0.0148966803356418, 
    0.0134406762089346, NA, 0.0164071881709832, 0.0140479675731781, 
    0.014153762083484, 0.016060617856451, NA, 0.0143583703529392, 
    0.0144812488480214, 0.0148666126757138, 0.0153832100508187, 
    NA, 0.012873765753982, 0.0144756496117552, 0.0156793375468165, 
    0.0139772051123241, 0.0136768679521059, 0.0149368831386071, 
    0.0137090743229012, 0.0152342060211406, NA, NA, 0.0156821506238176, 
    0.0151385308137555, 0.0150927875517445, 0.0167319416070821, 
    NA, 0.0136747003745318, 0.0155643048786908, 0.0148054989334855, 
    0.0140607445072002, NA, 0.0121292210168324, 0.0154958431894343, 
    0.0146021066358975, 0.0156532039972773, 0.0151770477673132, 
    0.013125904615912, 0.0143945757318224, 0.0140322192660349, 
    NA, NA, 0.0151723995907784, 0.0151806502751447, 0.0141242576447664, 
    0.0165124850437232, NA, 0.013764393890938, 0.0139014325742016, 
    0.0146183671197716, 0.0133429922212619, NA, 0.0142449555941979, 
    0.0148376326758212, 0.0163718141320951, 0.0135723680665006, 
    0.0147833182922942, 0.0145481104011903, 0.013639580220094, 
    0.0149919924108023, NA, NA, 0.0172449382468764, 0.0142091361549114, 
    0.0137431462792373, 0.0140994011152587, NA, 0.013495456802765, 
    0.0159509468569693, 0.0151895180030644, NA, NA, 0.0153068992106036, 
    0.0139120380039656, 0.0146622287100359, NA, 0.0150162122480392, 
    0.0131071143365518, 0.0158313317125523, 0.0168691079509988, 
    NA, NA, 0.0155608125188321, 0.0135489068850108, 0.0152831335323746, 
    0.0162953532349269, NA, 0.0153576126867625, 0.0152805986916586, 
    0.0159166808949391, NA, NA, 0.0158285889427599, 0.017356351669215, 
    0.0162001436957983, NA, 0.0145813404126472, 0.0124431881093808, 
    0.0148463197024211, 0.0140400828590147, NA, NA, 0.0155990818127587, 
    0.0129799301336622, 0.014615159260285, 0.0152704090148547, 
    NA, 0.0130465914181194, 0.0157850715388659, 0.0133986031479797, 
    NA, NA, 0.0146888031086602, 0.0159553863792708, 0.0154857706203022, 
    NA, 0.0138373503513294, 0.0114965705049062, 0.015040012795325, 
    0.0145386797423115, NA, NA, 0.0158431161048689, 0.0148579612655234, 
    0.0154244992115119, 0.0154837816874394, NA, 0.0160201570967232, 
    0.0158949017412503, 0.0131870825913611, NA, NA, 0.0139841498587335, 
    0.0142388202157029, 0.0143860432741416, NA, 0.0150020846421649, 
    0.0106458744890552, 0.013127523614843, 0.0155407057624024, 
    NA, NA, 0.0144330033579983, 0.0150730397670253, 0.0155607127165604, 
    0.0159246887732639, NA, 0.0145215373589078, 0.0155200204763512, 
    0.0147229309646725, NA, NA, 0.0146458888374069, 0.0144614775648087, 
    0.0154453321216127, NA, 0.0140307069985542, 0.0152175835400554, 
    0.0139341007828304, 0.0148450320874997, NA, NA, 0.0144180767408589, 
    0.0147973375407364, 0.0150053736799224, 0.016701912850409, 
    NA, 0.0150685610044095, 0.0151028884999963, 0.0145320511402253, 
    NA, NA, 0.0151208230379446, 0.0149347906975566, 0.0154608286516854, 
    NA, 0.0156273495543158, 0.0154342733348021, 0.0134747304799188, 
    0.0151843967127062, NA, NA, 0.0140595459521752, 0.0126203459678566, 
    0.014758967341318, 0.0152346552603696, NA, 0.0148228226019465, 
    0.0158177496037392, 0.0157526358181306, NA, NA, 0.0151206919921527, 
    0.0135886049150385, 0.0146894834078735, NA, 0.0153650825999843, 
    0.0140894523974043, 0.0145882959287903, 0.0147373220212539, 
    NA, NA, 0.0149865691226417, 0.0139031816274645, 0.0164482368681912, 
    0.0152994932494719, NA, 0.0141167487241223, 0.0165499394926831, 
    0.0124614524960887, NA, NA, 0.0158772388702489, 0.0136070460389723, 
    0.0152838988793823, NA, 0.0156853243345492, 0.0141677119789656, 
    0.0133053609124957, 0.0143433113353423, NA, NA, 0.0153727477267455, 
    0.0148349393817793, 0.0154086362634942, 0.0167090650758382, 
    NA, 0.0137681967239992, 0.0151044573907852, 0.0133213470050549, 
    NA, NA, 0.0143282934862511, 0.0139236029895124, 0.0151414626286481, 
    NA, 0.0137829062996592, 0.0144020923512034, 0.014868943168865, 
    0.0140068844692759, NA, NA, 0.0140312527838444, 0.0165744710449657, 
    0.0145944611031876, 0.0158488869928582, NA, 0.0152895994573595, 
    0.0146842600965806, 0.0152666642922424, NA, NA, 0.0145756335830212, 
    0.0147241724508365, 0.0141357679798026, NA, 0.0141651267927286, 
    0.0144187192686346, 0.014834026195986, 0.014529206809674, 
    NA, NA, 0.014484445961342, 0.0145342091616249, 0.013094850680071, 
    0.0162375584398812, NA, 0.0150570309380481, 0.0150507435064697, 
    0.0158442739844425, NA, NA, 0.0141998301230605, 0.014242579374566, 
    0.0146531778509903, NA, 0.0162231811425505, 0.0156502300169225, 
    0.0130277992972243, 0.0150771518916149, NA, NA, 0.0145521313931485, 
    NA, NA, 0.014070381548105, NA, 0.0153503902987887, 0.0157482764947086, 
    0.0144791029698792, NA, NA, NA, 0.0147606044567842, 0.0158394497948992, 
    NA, 0.0150899080017569, NA, 0.0139930775924034, 0.0157111771000535, 
    NA, NA, 0.0152856891618396, NA, NA, 0.0152470539850015, NA, 
    0.0156199645331057, 0.0142953774079945, 0.0151228741259958, 
    NA, NA, NA, 0.0148678598690996, 0.0158948981880751, NA, 0.0143485537167211, 
    NA, 0.0132920533729971, 0.0142450243909957, NA, NA, 0.0141129374726605, 
    NA, NA, 0.0169103113369445, NA, 0.0153921423483468, 0.0131988341145576, 
    0.0121632266769533, NA, NA, NA, 0.0150613333855329, 0.0154201996557281, 
    NA, 0.0162217526174379, NA, 0.0134906153278918, 0.0141590450078088, 
    NA, NA, 0.0137447165909874, NA, NA, 0.0162706892584939, NA, 
    0.0142503103263777, 0.0154136750066472, 0.0147756323820971, 
    NA, NA, 0.306277175348807, 0.249453964767345, 0.335059134411395, 
    0.241273000452021, 0.216399448828939, 0.25048286241848, 0.277340581959449, 
    0.300871529804369, 0.246701280435495, 0.30124669313444, 0.277774950602612, 
    0.279983352019891, 0.288559310156631, 0.290805089226702, 
    0.240914162926196, 0.347779488412921, 0.299233969629503, 
    0.290756867211223, 0.302592569221509, 0.237462905527573, 
    0.286368500413444, 0.264449227293747, 0.280550872479949, 
    0.288789637429446, 0.298724202204031, 0.277602177129986, 
    0.300250764581328, 0.271334282766694, 0.242303503144494, 
    NA, 0.303846053048768, 0.230327570944973, 0.301913161291002, 
    0.270899018919365, 0.254918368490709, 0.286575583674972, 
    0.263579740530676, 0.296729087798905, 0.265011223039323, 
    0.290292722780313, 0.307949612085484, 0.324776241634702, 
    0.290998500718781, 0.316318271289295, 0.340845781619026, 
    0.303181238904918, 0.285732462531953, 0.267179086918207, 
    0.346673681281266, NA, 0.320839492537978, 0.27112032226183, 
    0.292214192338609, 0.260340332818939, NA, 0.309533340322921, 
    0.273148140018777, 0.311375458710297, 0.326922242654756, 
    NA, 0.27975419204734, 0.285673452407902, 0.291096097353637, 
    0.339250018542399, 0.270989068157422, 0.309235604653847, 
    0.281865762566911, 0.279185583593251, NA, NA, 0.278775324695968, 
    0.295537680821052, 0.297816559411168, 0.298943985602413, 
    NA, 0.28857603511236, 0.285187229855081, 0.283758014630776, 
    0.329838700705001, NA, 0.253836330664256, 0.25316621631867, 
    0.292740690697027, 0.292249679123006, 0.34990222916707, 0.314954078154951, 
    0.276597688547481, 0.266144001688203, NA, NA, 0.278143064399485, 
    0.296310918560219, 0.292727481277322, 0.286261044939654, 
    NA, 0.281879567160772, 0.312283659296847, 0.297567885859459, 
    0.291249667530971, NA, 0.270361460239753, 0.27332929656194, 
    0.274446900406959, 0.280778218516447, 0.331002089708343, 
    0.280309284622528, 0.27090976267228, 0.305871656810566, NA, 
    NA, 0.308508742005463, 0.282670289732957, 0.315629058170898, 
    0.287399573704156, NA, 0.304958720395688, 0.293679488957768, 
    0.279231922208483, NA, NA, 0.257716952227048, 0.26342971150033, 
    0.291855695697199, NA, 0.288650884103793, 0.238889523015724, 
    0.287944980276847, 0.31491570024077, NA, NA, 0.274951109877395, 
    0.286580643876006, 0.29674662452867, 0.318106212822208, NA, 
    0.316517498414281, 0.310952285182775, 0.29831750899855, NA, 
    NA, 0.270984281788651, 0.278843203190922, 0.33536580927269, 
    NA, 0.263860066162467, 0.245976349177201, 0.26381981365686, 
    0.3039696512764, NA, NA, 0.266660682176696, 0.290605966119952, 
    0.306671649094002, 0.272597839293363, NA, 0.309685141843434, 
    0.292524563621131, 0.269705388151175, NA, NA, 0.294878600038592, 
    0.257224606741573, 0.313524786465086, NA, 0.280902375531863, 
    0.2284044674295, 0.275511405181431, 0.303028321449714, NA, 
    NA, 0.273467766385768, 0.29216292400188, 0.314474320476597, 
    0.291538870115516, NA, 0.29693036924679, 0.292593618126212, 
    0.258498869181756, NA, NA, 0.285739289382131, 0.275722803660206, 
    0.301832288265106, NA, 0.267498284158513, 0.226916019008112, 
    0.27281658214712, 0.288667071026352, NA, NA, 0.274266904068293, 
    0.324545639487547, 0.310524974527455, 0.258512766790391, 
    NA, 0.31029622723455, 0.292288985993593, 0.311459563847555, 
    NA, NA, 0.286857920235935, 0.282161989424983, 0.290898739076155, 
    NA, 0.280401341643392, 0.298459673075052, 0.291458608779968, 
    0.267873258959981, NA, NA, 0.311037239454718, 0.310064727005205, 
    0.336045440826139, 0.281628164981482, NA, 0.257322593679083, 
    0.30861685056044, 0.264003073438607, NA, NA, 0.271961674801295, 
    0.280152770776151, 0.311182870435393, NA, 0.285582367138687, 
    0.322971705405008, 0.304333633812211, 0.303496498970169, 
    NA, NA, 0.292546488043791, 0.299997782676717, 0.279092262155908, 
    0.257108795088236, NA, 0.293906916614178, 0.264183142104808, 
    0.298812832794467, NA, NA, 0.270232627073301, 0.272273476042877, 
    0.334598930034761, NA, 0.290741655781017, 0.268256349516313, 
    0.280279209655513, 0.26082128338053, NA, NA, 0.308951781279805, 
    0.267637676424718, 0.332174050675329, 0.293825470271865, 
    NA, 0.259146701513218, 0.265332166183535, 0.331975423126156, 
    NA, NA, 0.270873085942336, 0.287865727572473, 0.315562043744514, 
    NA, 0.271562833239098, 0.267779939258158, 0.295349532970151, 
    0.280572956970606, NA, NA, 0.277944255241489, 0.297581943431101, 
    0.304521979150486, 0.282981987917281, NA, 0.28313434699273, 
    0.293555809329894, 0.258983307797072, NA, NA, 0.295659778698521, 
    0.29165902325064, 0.317856328375945, NA, 0.280512178895231, 
    0.284673472206395, 0.267314369043172, 0.267956656284952, 
    NA, NA, 0.286239016948224, 0.288626611571063, 0.268128251034891, 
    0.301075978183474, NA, 0.275552907336785, 0.270004135164414, 
    0.319631632035912, NA, NA, 0.277240287921348, 0.288809327765109, 
    0.315975549288086, NA, 0.266037822873847, 0.298732701483525, 
    0.295708623892237, 0.318472463803858, NA, NA, 0.304164074361968, 
    0.312283587879404, 0.30571869078126, 0.303567769598347, NA, 
    0.236691930254186, 0.276006379357779, 0.301748423941227, 
    NA, NA, 0.285825953718566, 0.265984327341583, 0.297712794939248, 
    NA, 0.296556694311755, 0.265668059663379, 0.284229304100712, 
    0.27081339222107, NA, NA, 0.292041770302192, NA, NA, 0.258065186238611, 
    NA, 0.251358041370913, 0.291367003219235, 0.288302871611693, 
    NA, NA, NA, 0.302144964137492, 0.30032315422049, NA, 0.278154127239553, 
    NA, 0.281183705039629, 0.27699514847512, NA, NA, 0.29093126044449, 
    NA, NA, 0.268724203070329, NA, 0.297720109672394, 0.307778504811932, 
    0.318062616533899, NA, NA, NA, 0.294211789967918, 0.294094492048968, 
    NA, 0.288741380694083, NA, 0.28435428747805, 0.265888812507278, 
    NA, NA, 0.270028607472047, NA, NA, 0.335222448251753, NA, 
    0.26867006912992, 0.284542243088634, 0.296499754800993, NA, 
    NA, NA, 0.298468840008352, 0.300580865827472, NA, 0.295306838991444, 
    NA, 0.290083747756766, 0.251882469994868, NA, NA, 0.267173543591949, 
    NA, NA, 0.299109719770428, NA, 0.251953830925629, 0.278971900955903, 
    0.273259561450394, NA, NA, 4.60146687545419, 4.42485883628501, 
    4.29624067116876, 4.80005009137285, 4.5556475552706, 4.65083437102224, 
    4.74607848303065, 5.21629471445841, 4.47578327681759, 4.43185916703606, 
    4.23101609868674, 4.53746628271272, 4.9720480236509, 5.05215304353015, 
    4.47221106998478, 4.78971897249532, 4.99291671597999, 4.69536741347931, 
    4.06292833645443, 4.72961610454604, 4.7741748635026, 4.4965314959818, 
    5.11662152016173, 4.89409746764718, 5.15483632252725, 4.42564360467248, 
    4.69161568360844, 5.22336678950318, 4.80823467065965, NA, 
    4.64254311161002, 4.84374136055892, 5.03289041450692, 4.7952399413417, 
    4.60230118720289, 4.15581999917768, 4.65122046771679, 4.66763861207145, 
    4.09890070705838, 5.24163036275838, 4.08749856323013, 4.431300267867, 
    4.87518895938994, 4.23791663015889, 5.49805527776427, 5.31265664508024, 
    4.74497250307431, 4.66342027584126, 4.65201062657692, NA, 
    4.46979674350186, 5.2848141900711, 4.67434258501858, 4.58686496408413, 
    NA, 4.24190315340879, 5.08265301262816, 4.58726533695705, 
    4.46517172269841, NA, 4.22323843982939, 5.30024321627218, 
    4.6760765237532, 4.61293370075848, 4.46289608496893, 4.91530272742056, 
    4.67036625936027, 4.76358697170522, NA, NA, 4.39423757645686, 
    3.38060483583832, 4.95609157345611, 4.32389091517097, NA, 
    5.30079067930712, 4.57819010360261, 4.43313948769753, 5.18211637625428, 
    NA, 4.56645524452193, 4.43063254465572, 5.10648253568577, 
    4.92694948794945, 4.24034555593804, 4.71526382818984, 4.48732495526847, 
    5.13270535674585, NA, NA, 4.81501751627185, 4.48906697082437, 
    4.3749660474642, 4.65179809468906, NA, 4.90435243436809, 
    5.07190389965164, 4.78854636533878, 5.01982103346092, NA, 
    4.19048547564728, 4.81028213815461, 4.82620749670859, 4.63724655244293, 
    5.052532790561, 4.38190036735826, 5.15878648364885, 4.65548899024246, 
    NA, NA, 4.69730109833155, 4.62299762911568, 4.59859971401741, 
    4.69268483746978, NA, 4.75141408745534, 4.91672266713789, 
    4.82826999489275, NA, NA, 4.85563960496899, 4.66465513945803, 
    4.62535267194277, NA, 4.71603342442428, 4.80065022516005, 
    4.67434732356484, 5.09486191981006, NA, NA, 4.79560880659395, 
    4.95554633795522, 4.39961949329382, 5.8934179311706, NA, 
    4.95569624715577, 5.75826467855952, 4.91787552329397, NA, 
    NA, 4.94944180320879, 4.74033180355002, 4.60401936623953, 
    NA, 4.83295078438569, 4.89006539006609, 4.91804171282076, 
    4.65357232260862, NA, NA, 4.58368915931888, 5.01627338488151, 
    4.74801010583895, 5.04496151131317, NA, 5.08506879488199, 
    5.15597437471253, 5.15399034527315, NA, NA, 4.70727322754608, 
    4.75115262153176, 5.14838849211127, NA, 4.38138967155307, 
    4.30619767988547, 5.02338051089822, 4.89039214145464, NA, 
    NA, 4.32535515870786, 4.89479753180488, 4.61160457727183, 
    4.6010330057451, NA, 5.00742478626158, 5.02228599161655, 
    5.75254009867574, NA, NA, 5.10130550990318, 4.17088014073645, 
    4.95196700736623, NA, 4.94883916002206, 4.49189808152208, 
    4.34953547648772, 5.32710502830503, NA, NA, 4.94796088681308, 
    4.69859598287151, 4.93887002917406, 4.41520216410172, NA, 
    4.65321473435665, 5.24396753644291, 5.13995242439771, NA, 
    NA, 5.29749335086481, 4.96686568884483, 4.7135887409121, 
    NA, 4.58840308120005, 4.56525820808405, 4.67608980895469, 
    5.01398481259194, NA, NA, 4.99806375187386, 5.19570686815993, 
    4.28804250467003, 4.39643865658287, NA, 4.51282486328665, 
    4.87050022517455, 4.96096589294752, NA, NA, 4.69910465978613, 
    5.03456393639613, 4.62204345876248, NA, 4.60062561814486, 
    4.27712411094588, 4.70562809345864, 5.12030651598477, NA, 
    NA, 4.93544983232498, 4.78132457011805, 4.95894614022852, 
    4.11661051169248, NA, 4.35011328614933, 5.14342553470866, 
    4.34376933643069, NA, NA, 4.4516595675049, 4.62343329966237, 
    4.67756685024788, NA, 4.49373589815206, 4.32184685791324, 
    4.76416752601331, 4.31486647315855, NA, NA, 4.82601611272012, 
    4.95697912957324, 4.12059588151707, 4.21421321225578, NA, 
    5.34517996158648, 4.83777726333664, 4.31191893791779, NA, 
    NA, 4.86569160443291, 4.35400336184427, 4.46812531982234, 
    NA, 4.56389135463736, 4.44050687126336, 4.45136726081035, 
    4.65241720830152, NA, NA, 4.71117029125905, 4.28434807231626, 
    4.44459828343806, 4.53232528873725, NA, 4.86903540055949, 
    4.56067890721276, 4.35751764097326, NA, NA, 4.73590103676132, 
    4.35818396990261, 4.53778044891144, NA, 4.96801004713425, 
    4.99057200198495, 4.39328405372041, 4.71893571170023, NA, 
    NA, 4.56574957470042, 4.70190441553321, 4.82330691083707, 
    3.89400757874693, NA, 4.98525203993063, 5.07422550474317, 
    4.52443589118193, NA, NA, 4.51245476123596, 4.7666650431006, 
    4.45584472717554, NA, 5.15019901813935, 4.55536377316596, 
    4.63423865101625, 4.63800494928161, NA, NA, 5.10880888002447, 
    4.21903227718745, 4.74390665155398, 3.87472109851081, NA, 
    4.57628879550865, 4.95378089105533, 4.95634414433881, NA, 
    NA, 4.15517854200107, 4.94189012670332, 4.68962639000382, 
    NA, 4.75922309046859, 4.75905536166207, 4.93488321576639, 
    4.61298998044019, NA, NA, 4.36032060663551, NA, NA, 3.55942725554449, 
    NA, 4.89491336863558, 4.5399203784617, 4.65975067628447, 
    NA, NA, NA, 5.00161081829422, 4.66428421666794, NA, 4.71550404565469, 
    NA, 4.54760823999593, 4.80933927902622, NA, NA, 4.30683359341196, 
    NA, NA, 3.75363538458762, NA, 4.56909136538297, 4.33406843333622, 
    4.8753889436289, NA, NA, NA, 4.25162372994855, 4.44719299624804, 
    NA, 4.82602570865851, NA, 4.38822721306546, 4.81486687961933, 
    NA, NA, 5.080800137731, NA, NA, 3.93263903321273, NA, 4.67565193392148, 
    5.18267942978295, 3.52838948216294, NA, NA, NA, 3.99811621408344, 
    4.70994177401189, NA, 4.79372556442654, NA, 4.34099939575727, 
    4.90540531507982, NA, NA, 4.97413889075617, NA, NA, 3.56020545676791, 
    NA, 4.11764886035313, 4.48973006695808, 4.79157806965083, 
    NA, NA, 0.0467149086299671, 0.0488379186361875, 0.0620090287456152, 
    0.0492513268758879, 0.0534177765797167, 0.0482858715329614, 
    0.0490152478709983, 0.0485228180308963, 0.0547477904984213, 
    0.0521345129801299, 0.0476624550321713, 0.0573562669607483, 
    0.0550982717395183, 0.0526032495226555, 0.046126607570455, 
    0.054215118053875, 0.0560222196119065, 0.0606198733143673, 
    0.0573432710584983, 0.0478645190365491, 0.0468279066467241, 
    0.0488908519578558, 0.0516715745887662, 0.0524442005205201, 
    0.0527853547809004, 0.0514057703700794, 0.0508858941075063, 
    0.0541294231689218, 0.0432023953303552, NA, 0.0512872754330196, 
    0.0436950667675022, 0.0494809490879017, 0.0502355512227363, 
    0.0544578795105877, 0.0548401296953212, 0.0572323914437097, 
    0.0551807206712763, 0.048462345288294, 0.0495849664503531, 
    0.0523927772661165, 0.0560276436286362, 0.0518101080458219, 
    0.0518626749159581, 0.0527403860055883, 0.0530506739888912, 
    0.0515144261397061, 0.0474172438660115, 0.0525393200806684, 
    NA, 0.0519862635885958, 0.0526507760003325, 0.0514090828763132, 
    0.0526562999652178, NA, 0.0512697991004216, 0.0543249796705755, 
    0.0547282346916192, 0.0541727837735298, NA, 0.0474822050981774, 
    0.0577184563354795, 0.0596681328750246, 0.0523299307687398, 
    0.0596916708509938, 0.045116335798739, 0.0553861068599609, 
    0.0478337816334455, NA, NA, 0.0495949072123279, 0.0472786419183812, 
    0.0519760156712005, 0.0549106315968675, NA, 0.0484284127340824, 
    0.0544542537999779, 0.0561096160848405, 0.0471420958661299, 
    NA, 0.0474630831342288, 0.0509832726387057, 0.0497030265153564, 
    0.0559106509426085, 0.0551907851360498, 0.0552175085405358, 
    0.0532087481898951, 0.0490215229690871, NA, NA, 0.0491703319104651, 
    0.0494497297661038, 0.0527197645524546, 0.055788734259456, 
    NA, 0.054881609793972, 0.058084221185212, 0.0524929080432676, 
    0.0456153892916821, NA, 0.0496730629718036, 0.0514719772514651, 
    0.0574866968937407, 0.0553647562560374, 0.0541547624711132, 
    0.0493162902222929, 0.0501579869070723, 0.0484898526857875, 
    NA, NA, 0.0557579106795963, 0.0494868442645954, 0.0501746229727503, 
    0.0552938499570305, NA, 0.0495319708400214, 0.0524618885551029, 
    0.0544465781140799, NA, NA, 0.0539465835383485, 0.0569857768671514, 
    0.055775938036954, NA, 0.0573228699528317, 0.0469567191156329, 
    0.0527468291723066, 0.0486453152811218, NA, NA, 0.049293698742096, 
    0.0605035374133397, 0.0572909015082004, 0.0556017535984431, 
    NA, 0.0481034833363538, 0.0514552920741047, 0.0601736718424894, 
    NA, NA, 0.061783836804071, 0.0506136538726296, 0.0574237803239262, 
    NA, 0.05477110087496, 0.0550502172446463, 0.0505511290956235, 
    0.0479857044606779, NA, NA, 0.0540810145433692, 0.0510053997132949, 
    0.0533581717706154, 0.0476053356889337, NA, 0.0500340665011759, 
    0.0556089935442539, 0.0556399269430621, NA, NA, 0.0544561404736855, 
    0.0561209666360926, 0.0570318615742657, NA, 0.0535936030878033, 
    0.048038878348025, 0.0580022297098568, 0.0509496497336076, 
    NA, NA, 0.04885294082397, 0.0581569981273408, 0.0654324362200758, 
    0.05744810607342, NA, 0.0567676017739276, 0.0545853360838931, 
    0.0546852335072044, NA, NA, 0.0554021773436081, 0.0502315348568262, 
    0.0535348613773867, NA, 0.0544110560824248, 0.0436450450983802, 
    0.0489568963011306, 0.0491516203459462, NA, NA, 0.0536171116429718, 
    0.0655690023668594, 0.0639010174106928, 0.0509963330813192, 
    NA, 0.0556295583326905, 0.0518095882982757, 0.0524961955030874, 
    NA, NA, 0.0547289644847475, 0.0576734678783873, 0.053303721891753, 
    NA, 0.0470146650063709, 0.0577440776667153, 0.055022804065018, 
    0.04984653466516, NA, NA, 0.0524829574644672, 0.0520599684140766, 
    0.0494981628937773, 0.055703951427512, NA, 0.0538703676315935, 
    0.0522949613672658, 0.0528560156753865, NA, NA, 0.0619844001559137, 
    0.052193163988146, 0.0513954316089263, NA, 0.0512865979763058, 
    0.0545802593816553, 0.0497373203046128, 0.0518036839604503, 
    NA, NA, 0.0487652882512244, 0.0525999522827478, 0.0500003691055061, 
    0.0516557215269531, NA, 0.0518741841068784, 0.0516097123416985, 
    0.0495077839003605, NA, NA, 0.05700540709827, 0.060656061781148, 
    0.0520802259099599, NA, 0.0494665102562466, 0.0540248916361233, 
    0.053885864018678, 0.0561190007460187, NA, NA, 0.048145802635099, 
    0.0510122557585042, 0.0559951204398185, 0.0698419166271591, 
    NA, 0.0513496445000069, 0.0510937342724832, 0.0500945396814109, 
    NA, NA, 0.0514555373132551, 0.0527026266136143, 0.0484225431555815, 
    NA, 0.0561960202533542, 0.054529723166792, 0.0504457255703098, 
    0.0564661349687251, NA, NA, 0.0484173332066176, 0.0523580604114986, 
    0.0536151044181355, 0.0549000488746285, NA, 0.0539008796118395, 
    0.0480148717864529, 0.0508801802991017, NA, NA, 0.050648058946174, 
    0.0497082115959911, 0.0516626395432037, NA, 0.0537266554817586, 
    0.0547790488860907, 0.0549562049330919, 0.0564531480501209, 
    NA, NA, 0.0449881771602632, 0.0594750088547211, 0.0545175434236616, 
    0.0505092440152514, NA, 0.0474958002688699, 0.0593023002198541, 
    0.0489362261491318, NA, NA, 0.0589923263732834, 0.0568991144564912, 
    0.0515354762771232, NA, 0.0554619463566013, 0.0567874228412237, 
    0.0506871917401193, 0.0530350602365457, NA, NA, 0.0529180948162325, 
    0.0498703758236311, 0.0564944418358631, 0.0543500250926395, 
    NA, 0.0509526771495145, 0.0520760826302081, 0.0466047487121867, 
    NA, NA, 0.0528610222043874, 0.0497097384955735, 0.0506840472261544, 
    NA, 0.0556345365345103, 0.0560218143984876, 0.0613697735744758, 
    0.054427522332346, NA, NA, 0.0521444149819887, NA, NA, 0.0544992297920285, 
    NA, 0.0534956929350008, 0.0512271198803914, 0.0447099885687979, 
    NA, NA, NA, 0.0587912083640623, 0.049597943427858, NA, 0.0547497950168112, 
    NA, 0.0586538448151931, 0.0562490175387908, NA, NA, 0.0498151955284579, 
    NA, NA, 0.0516130750358407, NA, 0.0480543530572451, 0.0573696991386399, 
    0.0525773736389884, NA, NA, NA, 0.0509358710290626, 0.0457438569167935, 
    NA, 0.0495426532905297, NA, 0.0492622713446203, 0.0510480478549852, 
    NA, NA, 0.044764226685952, NA, NA, 0.049212584488591, NA, 
    0.0446830372168757, 0.054185898419178, 0.0472395888647089, 
    NA, NA, NA, 0.0551554196917616, 0.0485652885939054, NA, 0.0562130928905898, 
    NA, 0.05132932147708, 0.0517227796537467, NA, NA, 0.0492208810330505, 
    NA, NA, 0.0505406643917784, NA, 0.05135290058855, 0.0555323854309995, 
    0.0523163460297947, NA, NA, 0.523926046639462, 0.380346108347602, 
    0.428825818216329, 0.366662726979207, 0.360896272156652, 
    0.386710292881276, 0.38419184943291, 0.397467082740728, 0.37551006714414, 
    0.520930504108287, 0.511185786516854, 0.527312453952311, 
    0.469814330827453, 0.486340223158552, 0.363574336886377, 
    0.521574730193028, 0.52447869322019, 0.491904289395289, 0.500852231251115, 
    0.383595660596668, 0.539599794189892, 0.402131689392708, 
    0.479760253764427, 0.54233296300509, 0.51781233982924, 0.379142278685734, 
    0.520445974895358, 0.511753998405758, 0.364369801817315, 
    NA, 0.519965191619719, 0.356950187265917, 0.475489061240696, 
    0.494115568544834, 0.511384691551426, 0.485602374612334, 
    0.508046928674683, 0.422208518798617, 0.376079434862767, 
    0.476502697060966, 0.524537828310869, 0.524130816127521, 
    0.483586226863925, 0.511367989917715, 0.514653246314131, 
    0.466468694815869, 0.475416258142409, 0.444653032414725, 
    0.529837915018814, NA, 0.560700723469602, 0.460763939460629, 
    0.460538288262553, 0.52516632669684, NA, 0.50954735521363, 
    0.481176530644433, 0.536496557499149, 0.485882026690945, 
    NA, 0.532373186623086, 0.504981152153377, 0.473975026118667, 
    0.492754542710458, 0.505456618355047, 0.464902176521878, 
    0.494120906193938, 0.485522515689001, NA, NA, 0.537311772132362, 
    0.468581924218104, 0.481430689363859, 0.474736052945182, 
    NA, 0.359995494382022, 0.489752297751063, 0.56136412906327, 
    0.468281227593182, NA, 0.390776036285462, 0.49859557764333, 
    0.481036385698905, 0.526870862461917, 0.536751472412926, 
    0.470913236274935, 0.487970523077602, 0.492286717923535, 
    NA, NA, 0.522311261531406, 0.504674277558772, 0.496165474156203, 
    0.506471647856753, NA, 0.370249622092403, 0.516908109580766, 
    0.536547942039234, 0.46914444188583, NA, 0.524689227895725, 
    0.52111596480478, 0.493794656198139, 0.483960630529005, 0.510957808490716, 
    0.475873144452074, 0.495029924250845, 0.484619400133057, 
    NA, NA, 0.555458230088471, 0.449513149344094, 0.474436824112821, 
    0.532571048813801, NA, 0.506381707287073, 0.567276423800399, 
    0.530895196494812, NA, NA, 0.544350129300711, 0.533698827495043, 
    0.509931310117936, NA, 0.527560720805118, 0.409337649611248, 
    0.481232080524344, 0.488724790579856, NA, NA, 0.536998784082734, 
    0.487337258347279, 0.466736757013925, 0.492721355658368, 
    NA, 0.501611948169627, 0.518930354805412, 0.559525441445968, 
    NA, NA, 0.55704114820563, 0.534360017470927, 0.480093805715264, 
    NA, 0.50194605479759, 0.40701395211145, 0.46827611065226, 
    0.469486082866589, NA, NA, 0.554372472664879, 0.477490616383832, 
    0.466784454408515, 0.487331327836096, NA, 0.52004612839575, 
    0.506428917553716, 0.513584784478666, NA, NA, 0.46120880413981, 
    0.516854459221891, 0.483097773041113, NA, 0.529067428940659, 
    0.433989595376427, 0.49397685290973, 0.510238280945033, NA, 
    NA, 0.402287780430712, 0.531010806646045, 0.532150189729943, 
    0.502764684997858, NA, 0.547975727383234, 0.553882158967743, 
    0.504106277347915, NA, NA, 0.47075967203649, 0.480547777964112, 
    0.500521871513318, NA, 0.530408372058219, 0.401924946732023, 
    0.479556714673769, 0.485792113785091, NA, NA, 0.407858734058879, 
    0.501762977868918, 0.51053107689949, 0.532245433600081, NA, 
    0.525779794971674, 0.549935081426304, 0.547640074969633, 
    NA, NA, 0.511299544567817, 0.474271275978556, 0.488518440552251, 
    NA, 0.534238891784754, 0.495104804465198, 0.458495104955654, 
    0.435386480289107, NA, NA, 0.443145045142536, 0.503445637220925, 
    0.482659339086068, 0.568303058526212, NA, 0.473154100016605, 
    0.502784257265745, 0.49922260205631, NA, NA, 0.505722306251801, 
    0.544137225612268, 0.458515630461923, NA, 0.499471116342043, 
    0.434327487240214, 0.437221006708647, 0.483566005046435, 
    NA, NA, 0.417219536444829, 0.483400764116057, 0.501823917976826, 
    0.531724724142983, NA, 0.487535102314002, 0.485378876769649, 
    0.501255351624744, NA, NA, 0.516645548421616, 0.492365088005756, 
    0.471527394436555, NA, 0.504730415222067, 0.448931603752328, 
    0.473380490397243, 0.501788773332115, NA, NA, 0.416643419158699, 
    0.488718845939323, 0.553298515514749, 0.531910000684798, 
    NA, 0.454377492145807, 0.507860396179889, 0.510702376973403, 
    NA, NA, 0.526478042837605, 0.513569555018465, 0.479228750745379, 
    NA, 0.534727189916706, 0.517629610484563, 0.461069513534219, 
    0.497270679265891, NA, NA, 0.418182181107668, 0.500781267328177, 
    0.491519908844292, 0.517712007612981, NA, 0.457057198476255, 
    0.50337533771827, 0.483092483827034, NA, NA, 0.503078197339568, 
    0.502181425631327, 0.480586544431715, NA, 0.486915617382831, 
    0.527790563588982, 0.479439006724331, 0.4708402296986, NA, 
    NA, 0.451748531230244, 0.397427835205993, 0.495895063600839, 
    0.533142174790732, NA, 0.529445546605601, 0.480725270585007, 
    0.498418851137036, NA, NA, 0.520278696161049, 0.545178022355732, 
    0.482856838238639, NA, 0.490922727961999, 0.525898252754833, 
    0.485254813119103, 0.46556900473834, NA, NA, 0.45619229045767, 
    0.488536970359449, 0.486818250706354, 0.530582182219176, 
    NA, 0.497761408956303, 0.519034882739841, 0.542420338375108, 
    NA, NA, 0.518421976993044, 0.505361379867111, 0.479591819215022, 
    NA, 0.529988216779109, 0.523513097152135, 0.472238598865945, 
    0.464809045422921, NA, NA, 0.395489614416726, NA, NA, 0.498706558654666, 
    NA, 0.508071381435037, 0.535169713106755, 0.472838611060301, 
    NA, NA, NA, 0.52108629314708, 0.476371433284924, NA, 0.505931944839136, 
    NA, 0.477364965206844, 0.479554753344034, NA, NA, 0.443275058463764, 
    NA, NA, 0.515159922033348, NA, 0.510782111765453, 0.526868754902544, 
    0.514663608376526, NA, NA, NA, 0.488218788299523, 0.459908660685092, 
    NA, 0.49907810298703, NA, 0.467081785043026, 0.486957214580256, 
    NA, NA, 0.452863459394143, NA, NA, 0.512604821008623, NA, 
    0.503132330668952, 0.50862333297318, 0.341669746293826, NA, 
    NA, NA, 0.488207479883595, 0.479450990773579, NA, 0.524572273708447, 
    NA, 0.458420433695459, 0.500159554930674, NA, NA, 0.444770422963582, 
    NA, NA, 0.51714953402201, NA, 0.500845903335117, 0.52234616476726, 
    0.459203879290461, NA, NA, 106678.756413625, 28985.480705778, 
    95053.0535451084, 29117.6039196694, 30184.5454149369, 31925.1193283309, 
    27769.5114398423, 30581.4461807962, 28349.3002754344, 92107.6273761572, 
    96554.2444738548, 92370.7928743579, 87717.004334499, 87706.5296091283, 
    27795.932568516, 94868.3939477996, 90637.5973699279, 94736.5288175187, 
    87842.8965589888, 30511.7204927031, 101541.012461088, 26562.6591656148, 
    83267.2613492898, 96210.0076819579, 110017.985820969, 29534.4911114167, 
    96073.1248328192, 108653.90423957, 31540.1114438687, NA, 
    110230.68799126, 31955.8819000288, 104822.683755999, 87846.5086403944, 
    87305.4181341832, 93091.4213167313, 98858.65416928, 95584.9175021608, 
    28552.138310136, 91210.3021941237, 96612.107088672, 95009.8118212662, 
    100842.834265379, 91923.3557879452, 92942.5432207981, 94789.9601413815, 
    104174.396567414, 101646.553291761, 91962.7141662956, NA, 
    102838.63398214, 85923.9295866445, 89849.8410263214, 88569.1350763666, 
    NA, 97000.2554364926, 94358.2648186349, 98489.8742062479, 
    86204.6426103452, NA, 96035.4556290151, 99908.8402398977, 
    96531.9389550069, 96277.0370572799, 93083.4129154694, 88591.5133892508, 
    108243.91355505, 93417.9054848425, NA, NA, 92623.0745216168, 
    92666.9545114062, 102461.312967549, 91991.8411036529, NA, 
    97247.0359129213, 104072.786230362, 90767.570506369, 78709.136939454, 
    NA, 33824.2120423286, 88215.8323799451, 91464.2684966329, 
    100416.171254498, 91906.3258316734, 88845.2243522224, 102691.902857893, 
    94786.2204918754, NA, NA, 90654.4783968038, 97266.4489116805, 
    100023.427442369, 92647.7819304012, NA, 92669.2213106926, 
    92087.5187290081, 93402.5434574789, 92317.8585018727, NA, 
    88065.3506647677, 95002.7033258846, 90291.3097869744, 102757.85996619, 
    87715.7026456291, 89219.1432344221, 97701.801577474, 100262.119173812, 
    NA, NA, 103056.859160781, 92275.7565617477, 96122.2019659391, 
    101267.918300715, NA, 92508.49837257, 95121.2925026964, 97091.6853902344, 
    NA, NA, 89604.7047451014, 92444.4455651024, 101236.706451493, 
    NA, 93890.3304135078, 43344.2949733205, 102638.583203452, 
    90076.2298521937, NA, NA, 93723.7965665216, 91763.9009433022, 
    105477.556331152, 93950.494611515, NA, 102246.86378544, 101198.064989764, 
    103985.588168541, NA, NA, 94491.4458397941, 91839.9344152936, 
    98444.1944308745, NA, 93656.3301195994, 33398.2544520009, 
    102159.221556533, 91584.1847068692, NA, NA, 96578.2146160993, 
    100953.720645767, 99533.3194756554, 97951.1665818868, NA, 
    91131.4925117936, 95635.7602585584, 96572.4637794138, NA, 
    NA, 94773.9227701012, 95176.0599442024, 91147.7135455626, 
    NA, 95648.138519421, 33170.3113446215, 96858.9195754191, 
    91155.6407816227, NA, NA, 91171.9249812734, 96699.9397157273, 
    94295.2391966183, 94261.2568279753, NA, 98144.7475821829, 
    95439.5835156096, 97018.3482738968, NA, NA, 98825.9716377476, 
    92503.1472521409, 90082.4192054294, NA, 88638.6688472999, 
    35148.4129862359, 87627.9268627455, 91969.4879577351, NA, 
    NA, 91199.7639600292, 94431.4510893234, 106457.246532186, 
    97381.5293268034, NA, 96490.7266258281, 100683.920048493, 
    90751.9873152647, NA, NA, 97048.0324571274, 102905.378539693, 
    90063.4063780568, NA, 97235.5450103179, 94144.1404251179, 
    101241.191752865, 97109.7598149372, NA, NA, 86248.8194612323, 
    90446.2060216937, 94310.2712919335, 94742.5320843616, NA, 
    96025.4745779598, 97853.7398696114, 89516.4042539731, NA, 
    NA, 97554.5983434169, 93306.7343320522, 96898.8033317728, 
    NA, 96464.1895707289, 74838.8164748109, 91547.5031076821, 
    93292.559511581, NA, NA, 84288.6382886776, 100303.916103684, 
    98791.5696925378, 93129.7272184072, NA, 97217.8950575423, 
    102403.330849479, 89728.2626875589, NA, NA, 108455.004895666, 
    93110.4329698714, 113148.083553205, NA, 94111.7894688703, 
    91808.8030891366, 92418.5604766769, 92478.7882585792, NA, 
    NA, 79649.0369435818, 86632.332427374, 92731.796590393, 95372.397397175, 
    NA, 99955.4656584832, 107840.835338965, 94675.5459714123, 
    NA, NA, 104850.855532265, 86547.7696708953, 103895.878845651, 
    NA, 94978.8798494454, 87962.9720293229, 96671.1898578481, 
    92729.3219393702, NA, NA, 89650.7116548147, 92066.3166344423, 
    82226.4433691926, 95099.2927396471, NA, 102924.955638369, 
    97675.4443393813, 95709.965146277, NA, NA, 99337.8356837303, 
    87377.6501001224, 95840.03115414, NA, 98637.1553911502, 94439.9120632678, 
    100965.323505465, 96793.1462380063, NA, NA, 86273.7925858591, 
    85862.4737088508, 101790.261275959, 86119.5978926084, NA, 
    89945.9464456045, 97862.0887026062, 103868.470572819, NA, 
    NA, 98197.989286829, 91151.6344749808, 100998.368216349, 
    NA, 98767.6036332198, 87081.0660440694, 105039.440972283, 
    94697.9849523188, NA, NA, 86350.5949039257, 92765.5276373989, 
    93478.0652835272, 87198.2214685224, NA, 102850.526832603, 
    97887.5185071365, 100595.819585134, NA, NA, 93481.708694489, 
    91799.7382678946, 92307.4699120676, NA, 94263.7901888482, 
    93294.0205732319, 98392.8281675535, 92487.2729072669, NA, 
    NA, 82104.2197109728, NA, NA, 88395.2018533087, NA, 99544.7328440664, 
    99363.7395608876, 92481.8354793384, NA, NA, NA, 99345.9618284818, 
    89174.0937144895, NA, 89739.152310873, NA, 104971.379299668, 
    91616.016447298, NA, NA, 80747.6320091487, NA, NA, 92334.8568174916, 
    NA, 94580.1385465075, 89380.8681829278, 91704.6696568338, 
    NA, NA, NA, 97794.7002014073, 94080.0163656568, NA, 97593.8785105441, 
    NA, 97512.906698118, 92882.0587575833, NA, NA, 83080.1225429885, 
    NA, NA, 96902.9145795914, NA, 79172.9949551615, 94728.6903376358, 
    29314.5107038862, NA, NA, NA, 95228.3374445706, 89056.0654740242, 
    NA, 93366.805969341, NA, 100998.436845211, 96399.1274617403, 
    NA, NA, 82390.7240069203, NA, NA, 96571.0756438947, NA, 89253.5172106296, 
    106331.327556447, 95680.6938666398, NA, NA), dim = c(20L, 
    20L, 13L, 2L), dimnames = list(c("1", "2", "3", "4", "5", 
    "6", "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", 
    "17", "18", "19", "20"), c("1", "2", "3", "4", "5", "6", 
    "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", 
    "17", "18", "19", "20"), c("alco", "malic", "ash", "alca", 
    "mg", "phe", "fla", "nfla", "pro", "col", "hue", "ratio", 
    "prol"), c("var_inter", "var_intra"))), call = list(data.na = structure(c(14.23, 
    13.2, 13.16, NA, 13.24, NA, 14.39, 14.06, NA, 13.86, NA, 
    NA, 13.75, 14.75, 14.38, 13.63, 14.3, 13.83, NA, 13.64, NA, 
    12.93, 13.71, 12.85, 13.5, NA, 13.39, 13.3, 13.87, NA, 13.73, 
    13.58, 13.68, 13.76, NA, NA, NA, 13.05, NA, 14.22, 13.56, 
    13.41, 13.88, 13.24, 13.05, 14.21, NA, 13.9, NA, NA, 13.05, 
    NA, 13.82, NA, 13.74, NA, 14.22, 13.29, NA, 12.37, 12.33, 
    12.64, NA, 12.37, 12.17, 12.37, NA, NA, 13.34, 12.21, NA, 
    13.86, NA, 12.99, 11.96, 11.66, NA, 11.84, NA, 12.7, 12, 
    12.72, 12.08, 13.05, 11.84, NA, 12.16, NA, NA, 12.08, 12.08, 
    12, 12.69, NA, 11.62, NA, 11.81, 12.29, NA, 12.29, 12.08, 
    12.6, NA, NA, 12.51, NA, NA, 12.72, 12.22, 11.61, NA, NA, 
    11.76, NA, 12.08, 11.03, 11.82, NA, NA, NA, 11.45, NA, 12.42, 
    13.05, 11.87, 12.07, NA, 11.79, NA, 12.04, 12.86, 12.88, 
    NA, 12.7, 12.51, NA, 12.25, 12.53, NA, 12.84, 12.93, 13.36, 
    13.52, 13.62, 12.25, NA, NA, 12.87, 13.32, NA, NA, 12.79, 
    NA, 13.23, NA, 13.17, 13.84, 12.45, 14.34, 13.48, NA, 13.69, 
    12.85, 12.96, 13.78, 13.73, NA, NA, 13.58, 13.4, NA, 12.77, 
    14.16, NA, 13.4, 13.27, 13.17, 14.13, 1.71, NA, NA, 1.95, 
    NA, NA, 1.87, 2.15, NA, 1.35, 2.16, 1.48, NA, 1.73, 1.87, 
    NA, 1.92, NA, NA, 3.1, 1.63, 3.8, NA, 1.6, 1.81, NA, NA, 
    NA, 1.9, NA, 1.5, NA, 1.83, 1.53, NA, NA, NA, 1.65, 1.5, 
    NA, 1.71, NA, NA, 3.98, NA, 4.04, NA, 1.68, NA, NA, NA, 1.65, 
    NA, 1.9, 1.67, NA, 1.7, 1.97, NA, 0.94, NA, 1.36, 1.25, NA, 
    NA, 1.21, 1.01, 1.17, NA, NA, NA, 1.51, 1.66, 1.67, 1.09, 
    1.88, 0.9, NA, 0.99, 3.87, NA, NA, 1.13, NA, NA, 0.98, 1.61, 
    1.67, NA, NA, NA, 1.51, NA, NA, NA, NA, NA, 1.41, NA, NA, 
    2.08, NA, NA, 1.72, NA, 2.55, 1.73, NA, 1.29, NA, 3.74, NA, 
    2.68, 0.74, 1.39, NA, NA, 1.61, 3.43, NA, 2.4, 2.05, 4.43, 
    NA, 4.31, 2.16, 1.53, 2.13, NA, 4.3, 1.35, 2.99, 2.31, 3.55, 
    1.24, NA, NA, 5.51, 3.59, NA, 2.81, 2.56, NA, NA, 3.88, NA, 
    5.04, 4.61, 3.24, NA, 3.12, 2.67, 1.9, NA, 1.29, 5.19, 4.12, 
    3.03, 1.68, 1.67, 3.83, NA, NA, 3.45, 2.76, NA, NA, 3.37, 
    2.58, 4.6, NA, 2.39, NA, 5.65, 3.91, NA, 2.59, 4.1, 2.43, 
    2.14, 2.67, 2.5, NA, NA, 2.45, 2.61, NA, NA, 2.3, NA, 2.41, 
    2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, NA, 2.65, NA, 2.52, 
    2.61, 3.22, NA, NA, NA, NA, NA, 2.36, NA, 2.7, 2.65, NA, 
    NA, 2.55, 2.1, 2.51, NA, 2.12, 2.59, 2.29, 2.1, 2.44, NA, 
    2.12, 2.4, 2.27, 2.04, NA, NA, NA, NA, NA, 2.3, 2.68, 2.5, 
    1.36, NA, NA, 1.92, 2.16, 2.53, 2.56, NA, 1.92, NA, 1.75, 
    2.21, 2.67, 2.24, 2.6, 2.3, NA, 1.71, NA, NA, 2.4, 2, NA, 
    2.51, 2.32, 2.58, 2.24, NA, 2.62, NA, 2.3, 2.32, NA, 2.26, 
    2.22, 2.28, 2.2, 2.74, 1.98, NA, NA, 1.7, NA, 2.46, 1.88, 
    NA, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 
    2.2, 1.99, NA, NA, 2, 2.42, 3.23, NA, 2.13, 2.39, 2.17, 2.29, 
    NA, NA, 2.38, NA, 2.4, 2.4, 2.36, 2.25, NA, 2.54, 2.64, 2.19, 
    2.61, 2.7, 2.35, 2.72, 2.35, 2.2, NA, NA, 2.48, NA, 2.36, 
    2.62, 2.48, NA, 2.28, NA, 2.32, 2.38, NA, NA, 2.64, 2.38, 
    2.54, NA, 2.35, 2.3, NA, 2.6, NA, 2.69, NA, 2.32, 2.28, NA, 
    NA, 2.48, 2.26, 2.37, 2.74, 15.6, NA, NA, 16.8, 21, 15.2, 
    NA, 17.6, 14, 16, NA, 16.8, 16, 11.4, NA, NA, NA, 20, NA, 
    15.2, 16, 18.6, NA, 17.8, 20, NA, 16.1, 17, NA, NA, NA, 19.1, 
    NA, NA, 19, 20.5, 15.5, NA, 15.5, 13.2, 16.2, 18.8, NA, NA, 
    17, NA, 16, NA, 18.8, 17.4, 12.4, NA, NA, NA, NA, 20.5, 16.3, 
    16.8, NA, 10.6, NA, 16.8, 18, 19, 19, 18.1, NA, 19.6, NA, 
    16.8, 20.4, NA, 24, 30, 21, NA, 16, 18, 14.8, NA, 19, NA, 
    24, 22.5, 18, 18, 22.8, NA, 21.6, NA, NA, 22, NA, 18, 18, 
    19, 21.5, 16, NA, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 
    22.5, 19, 20, 19.5, NA, NA, NA, NA, 21.5, 20.8, NA, 16, NA, 
    NA, NA, NA, 21.5, 21, 21, 21.5, 28.5, NA, 22, NA, 20, 24, 
    21.5, NA, NA, NA, 25, 19.5, NA, 21, NA, 23.5, 20, NA, 21, 
    20, NA, NA, 21.5, 24, 22, NA, 18.5, 20, 22, 19.5, NA, 25, 
    NA, NA, NA, NA, 18.5, NA, 22.5, 23, 19.5, 24.5, 25, NA, 19.5, 
    NA, 20.5, 23, 20, NA, 24.5, 127, 100, 101, 113, NA, 112, 
    96, 121, 97, NA, NA, NA, 89, 91, 102, 112, 120, NA, NA, 116, 
    126, 102, NA, 95, NA, 124, 93, NA, 107, 96, 101, 106, 104, 
    NA, 110, NA, 110, NA, NA, NA, 117, NA, 101, NA, 107, 111, 
    102, 101, NA, 108, 92, NA, 111, 115, NA, NA, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, NA, 151, NA, 86, 87, 
    139, 101, NA, NA, 112, NA, NA, 86, NA, 78, 85, NA, 99, 90, 
    88, 84, 70, 81, NA, NA, 88, NA, NA, 134, 85, 88, NA, 97, 
    88, NA, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 
    85, 86, NA, NA, 87, 96, NA, 102, 86, NA, 85, NA, 92, 88, 
    80, 122, NA, 98, 106, NA, 94, 89, NA, 88, 101, NA, 89, 97, 
    NA, 112, 102, NA, NA, 92, 113, 123, NA, 116, 98, 103, NA, 
    89, 97, 98, 89, NA, NA, 106, 106, 90, 88, NA, 88, 105, 112, 
    96, 86, 91, 95, 102, NA, 120, NA, 2.8, 2.65, 2.8, 3.85, 2.8, 
    NA, 2.5, 2.6, NA, 2.98, 2.95, 2.2, 2.6, 3.1, NA, 2.85, 2.8, 
    NA, 3.3, NA, 3, 2.41, 2.61, 2.48, 2.53, NA, NA, 2.4, 2.95, 
    2.65, 3, NA, 2.42, 2.95, NA, 2.7, 2.6, 2.45, NA, NA, 3.15, 
    2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 2.45, 
    3.88, NA, NA, 2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, NA, 3.5, 
    1.89, NA, NA, 2.11, 2.53, NA, 1.1, 2.95, 1.88, NA, NA, NA, 
    NA, NA, 1.9, 2.83, NA, 2.2, NA, 1.65, NA, 2.2, NA, 1.92, 
    1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, NA, NA, 2.55, 3.52, 
    NA, NA, NA, 2.56, 2.5, 2.2, NA, 1.65, NA, 2.36, 2.74, 3.18, 
    2.55, 1.75, NA, 2.56, 2.46, NA, 2, NA, 2, 2.9, 3.18, NA, 
    NA, 2.86, NA, 2.74, 2.13, 2.22, 2.1, 1.51, NA, NA, 1.7, NA, 
    1.62, 1.38, NA, 1.62, NA, NA, 1.4, NA, 2, 1.38, NA, 0.98, 
    1.7, 1.93, 1.41, 1.4, 1.48, 2.2, NA, 1.48, NA, 1.8, 1.9, 
    NA, NA, 2.3, 1.83, NA, 1.39, 1.35, NA, 1.7, 1.48, 1.55, NA, 
    1.25, 1.39, 1.68, NA, NA, NA, 1.65, 2.05, NA, 2.76, NA, 3.49, 
    NA, 3.39, 2.52, 2.51, 2.98, NA, 3.32, 2.43, 2.76, NA, 3.64, 
    2.91, 3.14, 3.4, NA, NA, NA, 2.41, 2.88, 2.37, 2.61, 2.68, 
    2.94, NA, 2.97, 2.33, 3.25, 3.19, 2.69, NA, NA, 2.98, NA, 
    2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, 3.17, 
    NA, 2.92, 3.54, NA, 2.99, 3.74, 2.79, 2.9, 2.78, NA, 3.23, 
    3.67, 0.57, NA, 1.41, 1.79, NA, 1.75, 2.65, NA, 2, 1.3, NA, 
    1.02, 2.86, NA, 2.89, 2.14, 1.57, NA, NA, 1.85, NA, 2.26, 
    2.53, 1.58, 1.59, 2.21, NA, 1.69, 1.61, 1.69, NA, NA, 1.25, 
    1.46, NA, NA, 2.27, 0.99, NA, NA, 2.99, 2.17, 1.36, 2.11, 
    1.64, 1.92, 1.84, 2.03, 1.76, NA, 2.92, NA, 2.27, 2.03, 2.01, 
    2.29, 2.17, 1.6, 2.09, NA, 1.64, NA, NA, NA, NA, 3.03, 2.65, 
    3.15, 2.24, NA, 1.75, NA, NA, NA, 1.2, 0.58, 0.66, 0.47, 
    NA, NA, 0.6, 0.5, NA, 0.52, 0.8, NA, 0.55, 0.34, 0.65, 0.76, 
    1.39, 1.57, 1.36, 1.28, 0.83, 0.58, 0.63, NA, 0.58, 1.31, 
    1.1, 0.92, NA, 0.6, NA, 0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 
    0.49, NA, 0.7, 0.61, NA, 0.69, 0.68, 0.76, 0.28, NA, 0.3, 
    0.24, 0.39, NA, NA, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 
    0.29, 0.3, NA, NA, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, NA, 
    NA, 0.34, 0.27, NA, 0.26, 0.29, NA, 0.42, NA, 0.29, 0.26, 
    NA, NA, 0.28, 0.2, 0.34, NA, NA, 0.32, 0.28, 0.3, 0.27, 0.21, 
    0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, NA, 
    NA, 0.28, NA, 0.53, 0.32, NA, 0.45, 0.37, 0.26, 0.27, NA, 
    NA, NA, NA, NA, 0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 
    0.3, NA, 0.4, 0.61, 0.22, 0.3, NA, NA, NA, NA, NA, 0.5, NA, 
    0.25, 0.17, 0.32, NA, NA, 0.24, NA, NA, 0.29, 0.34, 0.37, 
    NA, 0.66, NA, 0.48, 0.39, 0.29, 0.24, 0.26, NA, 0.42, NA, 
    0.52, 0.3, NA, NA, NA, 0.32, NA, 0.43, 0.3, NA, NA, 0.39, 
    NA, 0.4, NA, NA, NA, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 
    0.53, 0.53, NA, NA, NA, NA, 0.43, 0.4, 0.47, 0.45, NA, 0.22, 
    NA, 0.26, 0.61, 0.53, NA, NA, 0.63, 0.53, 0.52, NA, 0.5, 
    0.6, 0.4, 0.41, NA, 0.43, 0.4, 0.39, NA, NA, 0.48, NA, NA, 
    0.43, 0.43, 0.53, NA, 2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 
    1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 1.81, NA, 2.96, 1.46, 
    NA, NA, 1.86, 1.66, 2.1, NA, 1.69, 1.46, 1.66, 1.92, 1.45, 
    1.35, 1.76, 1.98, 2.38, 1.95, NA, NA, 1.54, 1.86, 1.36, 1.44, 
    1.37, NA, NA, NA, NA, NA, 2.03, NA, 2.19, 2.14, 2.38, 2.08, 
    NA, 2.29, 1.87, 1.68, 1.62, 2.45, NA, 1.66, NA, 0.42, NA, 
    NA, 0.73, 1.87, 1.03, NA, NA, 1.04, NA, 2.5, 1.46, 1.87, 
    1.03, NA, NA, 1.15, 1.46, NA, 2.76, 1.95, 1.43, NA, NA, 1.62, 
    NA, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 1.63, NA, 1.99, 1.35, 
    3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, NA, 
    1.42, NA, 1.63, 2.08, 2.49, NA, NA, NA, NA, 1.04, NA, NA, 
    1.61, 0.83, 1.87, 1.83, NA, 1.71, NA, 2.91, 1.35, 1.77, NA, 
    1.9, NA, 0.94, NA, 0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 
    NA, 0.75, NA, 0.55, NA, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 
    1.25, 1.26, 1.56, NA, NA, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 
    NA, NA, 0.94, NA, 1.15, 1.46, 0.97, 1.54, 1.11, 0.73, 0.64, 
    1.24, 1.06, 1.41, 1.35, 1.46, 1.35, 5.64, 4.38, 5.68, 7.8, 
    NA, 6.75, NA, 5.05, NA, 7.22, 5.75, NA, 5.6, NA, 7.5, NA, 
    NA, NA, 8.7, 5.1, 5.65, 4.5, NA, 3.93, 3.52, 3.58, 4.8, 3.95, 
    4.5, 4.7, 5.7, NA, NA, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 
    6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, NA, 8.9, 7.2, 
    NA, NA, 6.3, 5.85, 6.25, NA, NA, 6.8, NA, 3.27, 5.75, 3.8, 
    4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, NA, 3.38, 3.74, NA, 
    3.21, 3.8, 4.6, NA, NA, NA, 2.5, 3.9, NA, 4.8, 3.05, 2.62, 
    2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, NA, 
    2.9, NA, NA, NA, NA, 2.8, NA, 2.94, 2.7, 3.4, 3.3, 2.7, NA, 
    2.9, NA, NA, 3.08, 2.9, 1.9, NA, NA, 3.4, 1.28, 3.25, NA, 
    NA, NA, 2.8, NA, 3.94, NA, 2.12, NA, NA, NA, NA, 5, 5.45, 
    7.1, NA, 5, 5.7, 4.92, NA, NA, 4.35, 4.4, 8.21, 4, 4.9, NA, 
    8.42, 9.4, NA, NA, 7.1, 10.52, 7.6, 7.9, NA, 7.5, 13, NA, 
    7.65, NA, 5.58, NA, NA, 6.62, 10.68, 10.26, NA, 8.5, NA, 
    NA, 9.7, 7.7, 7.3, NA, 9.3, 9.2, 1.04, 1.05, NA, 0.86, NA, 
    1.05, 1.02, NA, 1.08, NA, 1.25, NA, 1.15, 1.25, 1.2, 1.28, 
    NA, 1.13, 1.23, NA, 1.09, NA, 1.11, 1.09, 1.12, 1.13, NA, 
    1.02, NA, 1.04, 1.19, 1.09, NA, 1.25, 1.1, NA, NA, NA, 1.18, 
    NA, 0.95, 0.91, 0.88, 0.82, NA, 0.87, NA, NA, 1.07, 1.12, 
    1.12, NA, NA, NA, NA, NA, 0.94, 1.07, 0.89, NA, 1.25, 0.98, 
    1.23, 1.22, NA, NA, NA, 1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 
    1.31, NA, 1.23, 1.19, 0.96, 1.06, 1.19, 1.38, NA, 1.31, 0.84, 
    NA, 1.23, NA, NA, NA, 1.07, 1.08, 1.05, NA, 1.15, NA, NA, 
    NA, NA, NA, 1.42, 1.27, NA, 0.8, NA, 1.04, NA, 1, NA, 0.86, 
    NA, NA, 0.9, NA, 1.1, 0.93, 1.71, 0.95, NA, 0.7, NA, 0.8, 
    NA, 0.92, 0.73, NA, 0.86, 0.69, 0.97, 0.89, 0.79, NA, 0.74, 
    0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, NA, NA, NA, 0.89, 
    NA, 0.65, 0.6, NA, 0.54, 0.55, NA, NA, 0.48, 0.61, 0.56, 
    NA, 0.6, 0.57, 0.67, 0.57, NA, 0.56, 0.96, 0.87, 0.68, 0.7, 
    0.78, 0.85, NA, 0.74, 0.67, NA, NA, NA, 0.64, 0.7, 0.59, 
    0.6, 0.61, 3.92, 3.4, 3.17, NA, 2.93, NA, 3.58, 3.58, 2.85, 
    3.55, 3.17, 2.82, 2.9, NA, 3, NA, 2.65, NA, 2.82, 3.36, 3.71, 
    3.52, 4, 3.63, NA, 3.2, NA, 2.77, NA, NA, 2.71, 2.88, 2.87, 
    3, 2.87, NA, NA, NA, 2.69, NA, 3.38, 3, NA, 3, 3.35, 3.33, 
    NA, 3.33, 2.75, 3.1, NA, NA, 3.26, 2.93, 3.2, 3.03, NA, 2.84, 
    2.87, 1.82, NA, 1.59, 2.46, 2.87, NA, 2.3, 3.18, 3.48, NA, 
    NA, 1.82, NA, NA, 3.5, 3.13, 2.14, 2.48, 2.52, NA, NA, NA, 
    3.14, 2.72, 2.01, NA, NA, NA, 3.21, 2.75, 3.21, NA, 2.65, 
    NA, 3.3, NA, 2.63, NA, 2.74, NA, 2.83, 2.96, 2.77, 3.38, 
    NA, 3.57, 3.3, 3.17, 2.42, 3.02, 3.26, 2.81, 2.78, 2.5, NA, 
    3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, NA, 
    3.64, NA, 2.84, NA, 2.78, NA, NA, 1.42, 1.36, 1.29, 1.51, 
    1.58, 1.27, 1.69, 1.82, 2.15, NA, NA, NA, 2.05, 2, NA, 1.33, 
    NA, 1.62, 1.33, NA, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 
    1.96, 1.78, 1.58, NA, NA, 1.75, 1.68, 1.75, NA, NA, NA, NA, 
    NA, NA, NA, NA, NA, 1.56, 1.62, 1.6, 1065, 1050, NA, NA, 
    735, 1450, 1290, 1295, 1045, 1045, NA, 1280, 1320, NA, 1547, 
    1310, 1280, NA, 1680, 845, NA, 770, 1035, 1015, 845, NA, 
    1195, 1285, 915, 1035, 1285, NA, NA, 1235, NA, 920, 880, 
    NA, NA, 760, 795, 1035, 1095, 680, 885, 1080, 1065, 985, 
    1060, NA, 1150, 1265, 1190, NA, 1060, 1120, 970, 1270, 1285, 
    NA, NA, NA, 630, NA, NA, NA, 502, NA, NA, 718, 870, 410, 
    472, NA, 886, NA, 392, NA, NA, 463, 278, 714, NA, 515, NA, 
    NA, NA, 562, 680, 625, 480, 450, 495, NA, 345, NA, 625, 428, 
    660, 406, NA, 562, 438, 415, 672, NA, 510, NA, 312, 680, 
    562, 325, NA, 434, 385, NA, NA, NA, 372, 564, NA, NA, 365, 
    380, NA, 378, NA, 466, NA, 580, NA, NA, 560, 600, NA, 695, 
    720, 515, 580, 590, 600, NA, 520, 550, NA, 830, 415, NA, 
    NA, 550, 500, 480, 425, 675, 640, NA, 480, 880, 660, NA, 
    NA, 680, 570, NA, 615, NA, 695, 685, NA, NA, 510, 470, NA, 
    740, NA, 835, 840, NA), dim = c(178L, 13L), dimnames = list(
        NULL, c("alco", "malic", "ash", "alca", "mg", "phe", 
        "fla", "nfla", "pro", "col", "hue", "ratio", "prol"))), 
        method = "FCS-homo", nb.clust = 3, m = 20, maxit = 20, 
        Lstart = 100, L = 20, method.mice = NULL, predictmat = NULL, 
        data.type = "continuous", bootstrap = FALSE)), listvar = "alco", 
    nb.clust = NULL, nnodes = 2, sizeblock = 5, method.select = "knockoff", 
    B = 50, r = 0.3, graph = FALSE, printflag = TRUE, path.outfile = NULL, 
    mar = c(2.1, 4.1, 2.1, 0.6), cex.names = 0.7, modelNames = NULL))

res.varsel$predictormatrix["alco", ]

## ----eval=FALSE---------------------------------------------------------------
#  # multiple imputation with the new model
#  res.imp.select <- imputedata(data.na = wine.na,
#                               method = "FCS-homo",
#                               nb.clust = nb.clust,
#                               maxit = maxit,
#                               m = m,
#                               predictmat = res.varsel$predictormatrix,
#                               verbose = FALSE)

## ----eval=FALSE---------------------------------------------------------------
#  varselbest(res.imputedata = res.imp.FCS, B = B, nnodes = nnodes) # (time consuming)

## ----convb,fig.height = 4, fig.width = 4, fig.align = "center"----------------
res.B <- chooseB(res.varsel)

## ----fig.height = 4, fig.width = 4, fig.align = "center"----------------------
# check the variable importance
round(res.varsel$proportion["alco",], 2)
barplot(sort(res.varsel$proportion["alco",], decreasing=TRUE),
        ylab = "proportion",
        main = "alco",
        ylim = c(0, 1),
        las = 2,
        cex.names = .5)

r <- 0.2 # a new threshold value (r = 0.3 by default)
abline(h = r, col = 2, lty = 2)

## ----fig.height = 4, fig.width = 4, fig.align = "center"----------------------
predictormatrix <- res.varsel$predictormatrix
predictormatrix[res.varsel$proportion>r] <- 1
predictormatrix[res.varsel$proportion<=r] <- 0
predictormatrix["alco", ]

## ----eval=FALSE---------------------------------------------------------------
#  chooser(res.varsel = res.varsel)

## ----eval=FALSE---------------------------------------------------------------
#  # kmeans clustering
#  res.pool.kmeans <- clusterMI(res.imp.JM, nnodes = nnodes)

## ----echo=FALSE---------------------------------------------------------------
res.pool.kmeans <- list(part = c(3L, 3L, 3L, 3L, 1L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 1L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 3L, 3L, 3L, 3L, 1L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 1L, 
1L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 3L, 2L, 1L, 1L, 3L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 2L, 1L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 3L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L), instability = list(U = c(0.0153642216891807, 
0.0311576821108446, 0.0264650927913142, 0.0410289104910996, 0.0270458275470269, 
0.0310768842317889, 0.0231687918192147, 0.0233291251104659, 0.0344931195556117, 
0.0206956192399949, 0.019561923999495, 0.0200239868703447, 0.0141838151748517, 
0.0244653452846863, 0.023736901906325, 0.0171884863022346, 0.017634137103901, 
0.0187021840676682, 0.0118911753566469, 0.0202625931069309), 
    Ubar = 0.0230737911879813, B = 0.0818651084004049, Tot = 0.104938899588386), 
    call = list(output = list(res.imp = list(structure(list(alco = c(14.23, 
    13.2, 13.16, 15.6017781817434, 13.24, 14.1218699379843, 14.39, 
    14.06, 14.6154433351557, 13.86, 13.3262534999375, 13.8658322778452, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 13.9985682392331, 
    13.64, 13.5652144957771, 12.93, 13.71, 12.85, 13.5, 14.156831857401, 
    13.39, 13.3, 13.87, 12.2631127253604, 13.73, 13.58, 13.68, 
    13.76, 13.9461153779751, 13.4389375006348, 12.5858139394117, 
    13.05, 12.2994067239849, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 14.1277970339251, 13.9, 14.4008246306274, 13.8553626327926, 
    13.05, 13.2124765110196, 13.82, 13.5853685889256, 13.74, 
    15.0155830625855, 14.22, 13.29, 13.437316124706, 12.37, 12.33, 
    12.64, 11.5275845179006, 12.37, 12.17, 12.37, 13.2608639506802, 
    12.1896107752542, 13.34, 12.21, 11.7312521144202, 13.86, 
    12.9440532817319, 12.99, 11.96, 11.66, 12.793225457513, 11.84, 
    12.6912884260666, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.0448502957956, 
    12.16, 12.5763621014964, 12.0363869721096, 12.08, 12.08, 
    12, 12.69, 12.8177756889821, 11.62, 12.0785575856062, 11.81, 
    12.29, 11.4649648008883, 12.29, 12.08, 12.6, 13.0882445507017, 
    13.1253432101123, 12.51, 12.8850859124291, 11.4513063245918, 
    12.72, 12.22, 11.61, 12.2991103369056, 12.496747450495, 11.76, 
    12.9230672064307, 12.08, 11.03, 11.82, 12.5925532373004, 
    13.4460097711112, 11.9830348896708, 11.45, 14.1275072110409, 
    12.42, 13.05, 11.87, 12.07, 11.6862515697781, 11.79, 12.802595878783, 
    12.04, 12.86, 12.88, 13.5898095897997, 12.7, 12.51, 13.1880001377116, 
    12.25, 12.53, 11.2790108153082, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 13.6588081355931, 13.7790120071936, 12.87, 
    13.32, 12.6311694283065, 11.8667284088852, 12.79, 13.720376648635, 
    13.23, 13.9332706009388, 13.17, 13.84, 12.45, 14.34, 13.48, 
    13.2491366752794, 13.69, 12.85, 12.96, 13.78, 13.73, 14.1381100373571, 
    13.5470285329412, 13.58, 13.4, 14.1353894445137, 12.77, 14.16, 
    13.2407486343743, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    2.42154412035699, 0.248318884643582, 1.95, -0.375609715470145, 
    1.29607993551841, 1.87, 2.15, 1.17476821996456, 1.35, 2.16, 
    1.48, 2.15047250415326, 1.73, 1.87, 1.04300075617574, 1.92, 
    2.05556751179309, 1.06317532497557, 3.1, 1.63, 3.8, 2.74469383796024, 
    1.6, 1.81, 2.45293075647531, 1.49024832294939, 4.12968534192407, 
    1.9, 0.693726783707004, 1.5, 2.24724393531273, 1.83, 1.53, 
    2.73409318269464, 0.710468195966053, 1.70823876617972, 1.65, 
    1.5, 1.71826883575261, 1.71, 3.52790494059343, 3.06625375539715, 
    3.98, 0.517755909539689, 4.04, 1.99241853442338, 1.68, 1.74482245239705, 
    1.70838173372699, 1.511603377532, 1.65, 1.44222762121893, 
    1.9, 1.67, 1.1719062658585, 1.7, 1.97, 2.14759470964472, 
    0.94, 1.72316659271012, 1.36, 1.25, 1.98944974027033, 2.78190503661569, 
    1.21, 1.01, 1.17, 1.14259211254288, 3.41904718177594, 2.31930535322114, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 2.69937264138786, 0.99, 
    3.87, 1.55610756611846, 0.582745857009625, 1.13, 3.51298245577227, 
    2.04991228039932, 0.98, 1.61, 1.67, 0.660569183492346, 2.47076054050208, 
    1.15698236599592, 1.51, 3.05864859256983, 2.07537351051294, 
    1.57167620514933, 1.17190131109915, 3.95841914473814, 1.41, 
    3.04519976899137, 1.88668689757113, 2.08, 1.44467980131773, 
    2.69940891228486, 1.72, 2.46456756160996, 2.55, 1.73, 2.74420923217179, 
    1.29, 0.0779285474275988, 3.74, 3.59755391841639, 2.68, 0.74, 
    1.39, -0.477586640174561, 0.597461599685756, 1.61, 3.43, 
    1.81474268050269, 2.4, 2.05, 4.43, 3.43782462657812, 4.31, 
    2.16, 1.53, 2.13, 3.65317555864573, 4.3, 1.35, 2.99, 2.31, 
    3.55, 1.24, 4.99827778184474, 5.09847581642237, 5.51, 3.59, 
    1.49911473617177, 2.81, 2.56, 2.58215738561278, 2.78428838737931, 
    3.88, 4.27170619564131, 5.04, 4.61, 3.24, 5.82752156887076, 
    3.12, 2.67, 1.9, 3.30741005353268, 1.29, 5.19, 4.12, 3.03, 
    1.68, 1.67, 3.83, 3.18466928033047, 2.94251598808884, 3.45, 
    2.76, 4.7142594150108, 3.07097722769589, 3.37, 2.58, 4.6, 
    3.61012361120767, 2.39, 3.90475199759286, 5.65, 3.91, 4.28568534839395, 
    2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.29501978428351, 
    2.5861945521218, 2.45, 2.61, 2.48344094355234, 2.46361325382418, 
    2.3, 2.39297475096481, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
    2.48, 2.56, 2.54542501891736, 2.65, 2.5550085897168, 2.52, 
    2.61, 3.22, 2.88132017464925, 2.28335044676038, 2.83402369953638, 
    1.9049246037256, 2.49656440763856, 2.36, 3.05796289335719, 
    2.7, 2.65, 2.25351064333874, 2.03187344683388, 2.55, 2.1, 
    2.51, 2.14924130495618, 2.12, 2.59, 2.29, 2.1, 2.44, 2.68507909022341, 
    2.12, 2.4, 2.27, 2.04, 2.577364915047, 2.33057304418992, 
    2.64783628584703, 2.53896752488166, 2.92943169811593, 2.3, 
    2.68, 2.5, 1.36, 2.14238881088827, 2.14608654562533, 1.92, 
    2.16, 2.53, 2.56, 2.11470371269316, 1.92, 2.09884405884664, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.7528641825973, 1.71, 
    2.09753300949209, 2.25710377697334, 2.4, 2, 2.46502012415666, 
    2.51, 2.32, 2.58, 2.24, 2.33680722508396, 2.62, 2.49516771934425, 
    2.3, 2.32, 2.06841959052399, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.35837771462825, 1.96538111395539, 1.7, 2.17543021944238, 
    2.46, 1.88, 2.18245315895885, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.03714573468205, 
    2.1743342132785, 2, 2.42, 3.23, 2.71868076369531, 2.13, 2.39, 
    2.17, 2.29, 2.42561310942944, 2.00314141584919, 2.38, 2.65254069785467, 
    2.4, 2.4, 2.36, 2.25, 2.90514439240849, 2.54, 2.64, 2.19, 
    2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.51097736624406, 2.79646067055905, 
    2.48, 2.29740747211344, 2.36, 2.62, 2.48, 2.38256957559424, 
    2.28, 2.51431197149656, 2.32, 2.38, 1.94122285404199, 2.42023974544759, 
    2.64, 2.38, 2.54, 2.80055993958424, 2.35, 2.3, 3.03945345144785, 
    2.6, 2.3113094452511, 2.69, 2.34831551549428, 2.32, 2.28, 
    2.59395081793817, 2.33536505541983, 2.48, 2.26, 2.37, 2.74
    ), alca = c(15.6, 14.553817716937, 19.9233705136328, 16.8, 
    21, 15.2, 11.3156622782084, 17.6, 14, 16, 16.0125693817899, 
    16.8, 16, 11.4, 11.549401855977, 15.4528807074889, 15.9499721363575, 
    20, 15.8113210007464, 15.2, 16, 18.6, 13.5440020048092, 17.8, 
    20, 18.639648378709, 16.1, 17, 20.6406592680748, 19.8444051389991, 
    19.8686554912568, 19.1, 20.5362872749405, 18.9450592567171, 
    19, 20.5, 15.5, 22.6188098467494, 15.5, 13.2, 16.2, 18.8, 
    18.5238700913061, 17.2072838070603, 17, 12.4481203173608, 
    16, 16.7940906516483, 18.8, 17.4, 12.4, 20.3594966421723, 
    16.1175070997585, 10.9048441379479, 12.5898985524201, 20.5, 
    16.3, 16.8, 13.781973736365, 10.6, 19.2984433104451, 16.8, 
    18, 19, 19, 18.1, 20.8939677901434, 19.6, 24.6220456284571, 
    16.8, 20.4, 23.8050694615356, 24, 30, 21, 15.9978554392586, 
    16, 18, 14.8, 19.6338085617133, 19, 23.6494185577908, 24, 
    22.5, 18, 18, 22.8, 22.1916664242633, 21.6, 17.6441354901318, 
    23.3144941177839, 22, 24.7709416325944, 18, 18, 19, 21.5, 
    16, 17.0099339370524, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
    19, 22.5, 19, 20, 19.5, 18.1034212389761, 25.1524816040513, 
    21.3511961377935, 20.5258499582751, 21.5, 20.8, 18.2621546987676, 
    16, 22.6490040396435, 21.5085136332461, 18.9006638665131, 
    30.0213250155678, 21.5, 21, 21, 21.5, 28.5, 16.7151051364086, 
    22, 26.3378403272774, 20, 24, 21.5, 19.2560460285177, 21.2209285641988, 
    18.2253476933361, 25, 19.5, 22.9828695857353, 21, 21.0382004526796, 
    23.5, 20, 21.6446089084455, 21, 20, 26.7512418611832, 16.4282038596885, 
    21.5, 24, 22, 21.1797272390201, 18.5, 20, 22, 19.5, 13.2737795672193, 
    25, 23.9111104969087, 15.8571468043197, 21.7355333756858, 
    25.6004569427485, 18.5, 17.4261079740806, 22.5, 23, 19.5, 
    24.5, 25, 19.105748318559, 19.5, 19.7165371972574, 20.5, 
    23, 20, 21.4271781341878, 24.5), mg = c(127, 100, 101, 113, 
    91.0229825487791, 112, 96, 121, 97, 91.833630413955, 136.471350505954, 
    99.1484080387972, 89, 91, 102, 112, 120, 101.679898420623, 
    101.221754916276, 116, 126, 102, 105.594536916868, 95, 112.653881782854, 
    124, 93, 126.011111646424, 107, 96, 101, 106, 104, 108.242011266649, 
    110, 103.827655341911, 110, 68.6197899274282, 62.039153418463, 
    107.84456942003, 117, 116.252265935657, 101, 83.1226692774287, 
    107, 111, 102, 101, 121.32644910418, 108, 92, 106.267712741679, 
    111, 115, 110.820115481043, 132.671323499955, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 77.0684543671153, 
    151, 118.685009041049, 86, 87, 139, 101, 96.851062311415, 
    74.1262813519563, 112, 84.4332412126606, 98.8116699982816, 
    86, 75.7910634534667, 78, 85, 70.7196427805501, 99, 90, 88, 
    84, 70, 81, 113.02757894573, 96.781163192223, 88, 83.1675226399144, 
    98.254885773536, 134, 85, 88, 87.3207979179069, 97, 88, 97.1953835180261, 
    86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 
    117.260814487596, 90.2067049357306, 87, 96, 129.193891326159, 
    102, 86, 77.6542960053088, 85, 96.0926453616909, 92, 88, 
    80, 122, 87.751911503402, 98, 106, 84.96026232138, 94, 89, 
    92.6911568798051, 88, 101, 100.777740652937, 89, 97, 102.530771316384, 
    112, 102, 94.219188081828, 112.063525982365, 92, 113, 123, 
    126.902437107777, 116, 98, 103, 86.7135767967371, 89, 97, 
    98, 89, 82.3978116108568, 87.0913096748242, 106, 106, 90, 
    88, 84.2659536467007, 88, 105, 112, 96, 86, 91, 95, 102, 
    113.864660052246, 120, 126.039798809601), phe = c(2.8, 2.65, 
    2.8, 3.85, 2.8, 3.51093729967228, 2.5, 2.6, 3.26229758461829, 
    2.98, 2.95, 2.2, 2.6, 3.1, 3.24005259736258, 2.85, 2.8, 3.15014141047382, 
    3.3, 2.61842453031566, 3, 2.41, 2.61, 2.48, 2.53, 2.71207541056519, 
    2.33774456679943, 2.4, 2.95, 2.65, 3, 2.64536274611829, 2.42, 
    2.95, 2.80379376488765, 2.7, 2.6, 2.45, 3.18063491160795, 
    3.29509420403834, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 
    3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.74974790591968, 3.02959612517719, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.37050863134826, 3.5, 
    1.89, 2.7625146580331, 2.97962679982043, 2.11, 2.53, 3.20551875104229, 
    1.1, 2.95, 1.88, 2.65581665847429, 2.98871436681411, 2.10031848625747, 
    2.13369069139754, 2.04218721921785, 1.9, 2.83, 2.75582711372462, 
    2.2, 2.15225573747865, 1.65, 2.19768917424423, 2.2, 2.06134416958, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.34739160427809, 
    1.24627228580831, 2.55, 3.52, 3.21254178482119, 2.63052972588875, 
    1.6839610015031, 2.56, 2.5, 2.2, 2.43947939372298, 1.65, 
    1.98860395146043, 2.36, 2.74, 3.18, 2.55, 1.75, 2.85954603731984, 
    2.56, 2.46, 1.66708152302441, 2, 1.57490772201153, 2, 2.9, 
    3.18, 2.27438413768598, 1.93496780073968, 2.86, 2.99953199932702, 
    2.74, 2.13, 2.22, 2.1, 1.51, 1.40764128912344, 1.50226138879316, 
    1.7, 1.05739107313535, 1.62, 1.38, 1.73822054770627, 1.62, 
    1.28961626525058, 1.60063365919427, 1.4, 1.84717222701903, 
    2, 1.38, 0.94420571667253, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 1.52817561515422, 1.48, 1.16290837914416, 1.8, 1.9, 
    1.42144653748509, 1.81313368537143, 2.3, 1.83, 1.38270686342009, 
    1.39, 1.35, 1.61363042976108, 1.7, 1.48, 1.55, 1.89107689770186, 
    1.25, 1.39, 1.68, 1.19764192769733, 1.92916853924595, 1.24115403938012, 
    1.65, 2.05), fla = c(2.91521377307154, 2.76, 3.64512311734352, 
    3.49, 2.13866612851558, 3.39, 2.52, 2.51, 2.98, 3.66485336641042, 
    3.32, 2.43, 2.76, 3.35229578388323, 3.64, 2.91, 3.14, 3.4, 
    3.5862865939836, 3.16492166030643, 3.41655419658463, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.41815426635109, 2.97, 2.33, 
    3.25, 3.19, 2.69, 3.27874093356547, 2.63210404455527, 2.98, 
    1.89461683955464, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.29448319634148, 2.92, 3.54, 4.14588136621389, 
    2.99, 3.74, 2.79, 2.9, 2.78, 3.44337420587568, 3.23, 3.67, 
    0.57, 1.93570604397047, 1.41, 1.79, 3.22276892317212, 1.75, 
    2.65, 2.38110999292412, 2, 1.3, 3.93828806948348, 1.02, 2.86, 
    1.14698777320084, 2.89, 2.14, 1.57, 1.19858362797014, 1.77657105883018, 
    1.85, 2.09776834981722, 2.26, 2.53, 1.58, 1.59, 2.21, 1.65526004994645, 
    1.69, 1.61, 1.69, 2.140134991537, 1.36998034354641, 1.25, 
    1.46, 2.04204086678555, 3.23712757776066, 2.27, 0.99, 1.87614509373718, 
    3.78824912497386, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 1.96099896005822, 2.92, 2.93293088124372, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 0.330087759851425, 1.64, 
    3.64882293268214, 4.0139832466307, 2.39274089422142, 1.79932385982643, 
    3.03, 2.65, 3.15, 2.24, 1.81751348033171, 1.75, 0.856389229127289, 
    0.391104347878773, 0.797696880215831, 1.2, 0.58, 0.66, 0.47, 
    0.912692992707671, 1.06857637808961, 0.6, 0.5, 0.956723896476065, 
    0.52, 0.8, 0.99666692231153, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.933675734926611, 0.58, 
    1.31, 1.1, 0.92, 0.633304727620532, 0.6, 0.695367761168557, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.42456772564185, 
    0.7, 0.61, 0.795823271587648, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.352131302885271, 0.3, 0.24, 0.39, 0.375796504321666, 0.347979351938333, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.283670495877178, 
    0.447281815474505, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.194712105214202, 
    0.18751985437, 0.34, 0.27, 0.410342000560825, 0.26, 0.29, 
    0.283696086236744, 0.42, 0.376462820733232, 0.29, 0.26, 0.357193296375685, 
    0.221826449904422, 0.28, 0.2, 0.34, 0.390567013822018, 0.357350568940093, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.350102118019358, 0.240614712516699, 
    0.28, 0.364398663664442, 0.53, 0.32, 0.406261443405028, 0.45, 
    0.37, 0.26, 0.27, 0.456988826140004, 0.230374363899938, 0.324673270481636, 
    0.321601620680508, 0.464861399089234, 0.21, 0.13, 0.34, 0.24, 
    0.43, 0.35, 0.43, 0.3, 0.367312655533543, 0.4, 0.61, 0.22, 
    0.3, 0.418324883904891, 0.463105985955331, 0.160635657929509, 
    0.412920799408438, 0.406509187954983, 0.5, 0.455194556928011, 
    0.25, 0.17, 0.32, 0.445278931912291, 0.357631771159959, 0.24, 
    0.390611858124363, 0.296194301203296, 0.29, 0.34, 0.37, 0.331767476060162, 
    0.66, 0.306334482745553, 0.48, 0.39, 0.29, 0.24, 0.26, 0.422473236269116, 
    0.42, 0.361298770318868, 0.52, 0.3, 0.221969570521311, 0.502767368639029, 
    0.412376740556598, 0.32, 0.384617802654203, 0.43, 0.3, 0.562566930703972, 
    0.251276134688029, 0.39, 0.384292537882866, 0.4, 0.36919080660252, 
    0.353801810633917, 0.394625366230721, 0.27, 0.17, 0.6, 0.63, 
    0.53, 0.63, 0.58, 0.53, 0.53, 0.346479881776669, 0.392012316765183, 
    0.305879804459512, 0.490309869264976, 0.43, 0.4, 0.47, 0.45, 
    0.404508297546472, 0.22, 0.274607678831061, 0.26, 0.61, 0.53, 
    0.624348916308511, 0.405608837794976, 0.63, 0.53, 0.52, 0.471569023074841, 
    0.5, 0.6, 0.4, 0.41, 0.60103279072447, 0.43, 0.4, 0.39, 0.310461815488074, 
    0.477654075992197, 0.48, 0.355774309349967, 0.535374217098237, 
    0.43, 0.43, 0.53, 0.491084435479867), pro = c(2.29, 1.28, 
    2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 
    1.81, 3.02266191321139, 2.96, 1.46, 2.54719909007917, 2.03705530537954, 
    1.86, 1.66, 2.1, 0.0484216722638837, 1.69, 1.46, 1.66, 1.92, 
    1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.85141327808064, 2.39352520013806, 
    1.54, 1.86, 1.36, 1.44, 1.37, 1.84514927066655, 1.84491947831327, 
    1.77075941837106, 1.99624438681422, 1.4755731236605, 2.03, 
    0.909157742828402, 2.19, 2.14, 2.38, 2.08, 2.78223957220699, 
    2.29, 1.87, 1.68, 1.62, 2.45, 2.37145196391104, 1.66, 1.39284530443389, 
    0.42, 2.2013947679463, 1.89359285141882, 0.73, 1.87, 1.03, 
    1.91238196101845, 1.41156858505882, 1.04, 1.33637526144509, 
    2.5, 1.46, 1.87, 1.03, 1.71884368453984, 0.263596860374118, 
    1.15, 1.46, 1.38807893092307, 2.76, 1.95, 1.43, 2.79666864578529, 
    1.27004050945517, 1.62, 1.03172580622596, 1.46, 1.56, 1.34, 
    1.35, 1.38, 1.64, 1.63, 1.67636217384354, 1.99, 1.35, 3.28, 
    1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.81697197963474, 
    1.42, 1.28860141163, 1.63, 2.08, 2.49, 1.81588624369839, 
    1.5594500756897, 1.91306912150629, 1.78304871982632, 1.04, 
    2.03234719988555, 1.74362596145738, 1.61, 0.83, 1.87, 1.83, 
    2.73515914402521, 1.71, 1.25551142108904, 2.91, 1.35, 1.77, 
    1.79508063770383, 1.9, 0.855940860726685, 0.94, 0.7477577709777, 
    0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 0.509025850434436, 
    0.75, 1.48008660532274, 0.55, 0.610971757623382, 1.14, 1.3, 
    0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.2191578022545, 
    1.09468147805873, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.77770059565988, 
    1.54944252018768, 0.94, 1.24480641035462, 1.15, 1.46, 0.97, 
    1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
    ), col = c(5.64, 4.38, 5.68, 7.8, 5.45226697255022, 6.75, 
    5.16438370632865, 5.05, 7.53037626426632, 7.22, 5.75, 7.35266743544202, 
    5.6, 6.21487092569802, 7.5, 6.84802698652534, 7.35409642975307, 
    3.44449836286631, 8.7, 5.1, 5.65, 4.5, 4.10629211810956, 
    3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 6.67688250317908, 
    4.42944789563439, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
    4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 7.64776533587632, 
    8.9, 7.2, 3.86141269506811, 6.0670864341981, 6.3, 5.85, 6.25, 
    6.77250869212169, 5.02257819723373, 6.8, 6.55007738310661, 
    3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 
    4.00032064926474, 3.38, 3.74, 3.67147660319945, 3.21, 3.8, 
    4.6, 2.94753984761392, 4.2098824633867, 6.03040205722529, 
    2.5, 3.9, 1.18804447631428, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 
    1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 7.47836711082916, 
    2.9, 1.77084508318845, 3.11748261282212, 4.23116642469846, 
    4.24909082831934, 2.8, 4.35295980299839, 2.94, 2.7, 3.4, 
    3.3, 2.7, 1.39145429661308, 2.9, 2.4815000861174, 1.22043232438793, 
    3.08, 2.9, 1.9, 2.49436617962049, 5.86595872771045, 3.4, 
    1.28, 3.25, 6.05008155850801, 0.140412973192086, 6.11015157908419, 
    2.8, 2.57406661628558, 3.94, 0.962345459535009, 2.12, 5.72475790428963, 
    6.05868230251805, 8.84996129565093, 6.8243814294473, 5, 5.45, 
    7.1, 5.84616773488345, 5, 5.7, 4.92, 8.41018170618694, 4.50487120430957, 
    4.35, 4.4, 8.21, 4, 4.9, 5.88867136610654, 8.42, 9.4, 1.361870563758, 
    6.84019137048076, 7.1, 10.52, 7.6, 7.9, 9.77808839898116, 
    7.5, 13, 9.00844677082518, 7.65, 7.66905974550211, 5.58, 
    9.70260298355627, 6.95514422203028, 6.62, 10.68, 10.26, 3.81297057593024, 
    8.5, 11.8064253355167, 10.9870812323815, 9.7, 7.7, 7.3, 10.2725900985565, 
    9.3, 9.2), hue = c(1.04, 1.05, 1.39056270880011, 0.86, 0.954934636517517, 
    1.05, 1.02, 1.00481147900113, 1.08, 0.980123442808032, 1.25, 
    0.956407260578228, 1.15, 1.25, 1.2, 1.28, 0.769262789349839, 
    1.13, 1.23, 0.724687622781733, 1.09, 0.41306183484283, 1.11, 
    1.09, 1.12, 1.13, 1.10201833955666, 1.02, 0.8269302287791, 
    1.04, 1.19, 1.09, 0.878940564861654, 1.25, 1.1, 1.08636114021068, 
    0.524218417981897, 0.825625193401014, 1.18, 1.17267012155875, 
    0.95, 0.91, 0.88, 0.82, 1.52556958179514, 0.87, 1.24381859418041, 
    1.19481342033244, 1.07, 1.12, 1.12, 1.51301430204659, 1.01989315355396, 
    1.185488802149, 1.15399419570614, 0.919478711431709, 0.94, 
    1.07, 0.89, 1.14969696749453, 1.25, 0.98, 1.23, 1.22, 0.841356890352039, 
    1.05758472325953, 0.997526563914834, 1.12, 1.02, 1.28, 0.906, 
    1.36, 0.98, 1.31, 1.17087087778638, 1.23, 1.19, 0.96, 1.06, 
    1.19, 1.38, 1.12040322021225, 1.31, 0.84, 1.09418308846564, 
    1.23, 1.00650670789839, 1.03269764983834, 1.1493855807052, 
    1.07, 1.08, 1.05, 0.603626597171563, 1.15, 1.23766176330536, 
    1.48516004161864, 0.667863438952731, 1.10036829699012, 0.998657180210806, 
    1.42, 1.27, 0.804631935848298, 0.8, 0.992195916793937, 1.04, 
    1.21429991338817, 1, 0.618307196675524, 0.86, 1.09473348369904, 
    0.837843495881548, 0.9, 0.915776411068128, 1.1, 0.93, 1.71, 
    0.95, 0.69015580648524, 0.7, 1.29256026215794, 0.8, 0.778264316858999, 
    0.92, 0.73, 1.12519131900562, 0.86, 0.69, 0.97, 0.89, 0.79, 
    0.779198012935583, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
    0.81, 0.709548794631781, 0.664381335007522, 0.824095181394706, 
    0.89, 0.615035385023368, 0.65, 0.6, 0.44044874042974, 0.54, 
    0.55, 0.347004960078006, 0.730148712001213, 0.48, 0.61, 0.56, 
    0.59510662010402, 0.6, 0.57, 0.67, 0.57, 0.946624859974712, 
    0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.634392367809825, 
    0.74, 0.67, 0.416669311996427, 0.631642857141285, 0.684295365486056, 
    0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 3.70180792359379, 
    2.93, 2.91902326984382, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
    2.9, 3.71083354322603, 3, 2.7981288279675, 2.65, 3.85838520962389, 
    2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.75975866120248, 3.2, 3.67867298432911, 
    2.77, 3.23339969315419, 2.584952783435, 2.71, 2.88, 2.87, 
    3, 2.87, 3.60234177759828, 2.10398998501251, 3.08539457130378, 
    2.69, 3.6503238151549, 3.38, 3, 2.85898597675497, 3, 3.35, 
    3.33, 3.05396634888541, 3.33, 2.75, 3.1, 4.00476723117154, 
    3.69616753410611, 3.26, 2.93, 3.2, 3.03, 3.57904802297012, 
    2.84, 2.87, 1.82, 3.05695173161965, 1.59, 2.46, 2.87, 2.73291059704304, 
    2.3, 3.18, 3.48, 2.59003493285651, 4.05778177084562, 1.82, 
    3.56503922432124, 2.58123083825815, 3.5, 3.13, 2.14, 2.48, 
    2.52, 3.04588742114323, 2.82308698362478, 3.31273372192657, 
    3.14, 2.72, 2.01, 2.82172136567331, 2.98632368275498, 2.31586765284304, 
    3.21, 2.75, 3.21, 2.57405725695321, 2.65, 2.82217362079794, 
    3.3, 3.85250005434767, 2.63, 1.49055083662418, 2.74, 3.09845821795548, 
    2.83, 2.96, 2.77, 3.38, 2.48319932783799, 3.57, 3.3, 3.17, 
    2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.57872965678139, 3.19, 
    2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 1.99825573960537, 
    3.64, 2.70839758325957, 2.84, 2.9922467097727, 2.78, 2.75584321954305, 
    1.44000090183954, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
    1.82, 2.15, 1.64374360918895, 2.62579439355793, 1.58483114936738, 
    2.05, 2, 1.79673515682077, 1.33, 2.22291932409513, 1.62, 
    1.33, 3.00744338157327, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
    1.73, 1.96, 1.78, 1.58, 1.83931650765701, 2.47692953726245, 
    1.75, 1.68, 1.75, 1.85493487769825, 2.04366454506958, 1.66376403821416, 
    1.68638621748622, 2.10019618272636, 1.05234975146542, 1.92396104928136, 
    2.02794976007388, 1.76147576887807, 1.56, 1.62, 1.6), prol = c(1065, 
    1050, 1367.15711961451, 582.898880355269, 735, 1450, 1290, 
    1295, 1045, 1045, 1122.13918001733, 1280, 1320, 981.146744091597, 
    1547, 1310, 1280, 1332.14999408665, 1680, 845, 1270.431636709, 
    770, 1035, 1015, 845, 1210.74777619245, 1195, 1285, 915, 
    1035, 1285, 1447.44083730959, 1030.79829411208, 1235, 1077.89310322228, 
    920, 880, 500.250949443596, 548.896012388606, 760, 795, 1035, 
    1095, 680, 885, 1080, 1065, 985, 1060, 1114.33109598462, 
    1150, 1265, 1190, 1568.57950569124, 1060, 1120, 970, 1270, 
    1285, 628.863673338595, 585.693076503666, 658.666486106837, 
    630, 581.484184076542, 639.442365669625, 1145.55006249417, 
    502, 667.949644964069, 196.426917474909, 718, 870, 410, 472, 
    572.809720302213, 886, 673.035333555969, 392, 857.095851059905, 
    490.005591129383, 463, 278, 714, 484.209068601025, 515, 604.85087529574, 
    444.350743344369, 665.515656896463, 562, 680, 625, 480, 450, 
    495, 541.445828677582, 345, 283.805415772681, 625, 428, 660, 
    406, 368.690716060905, 562, 438, 415, 672, 299.503659593908, 
    510, 535.896028347027, 312, 680, 562, 325, 651.404236100093, 
    434, 385, 561.024168727431, 489.999644286147, 413.431483417841, 
    372, 564, 652.34690809418, 1173.52861607787, 365, 380, 440.164759739252, 
    378, 702.676738736036, 466, 369.655160126723, 580, 303.635898554294, 
    518.638492903301, 560, 600, 668.950271257219, 695, 720, 515, 
    580, 590, 600, 348.466097909479, 520, 550, 908.976058086569, 
    830, 415, 129.603565303425, 557.197648972336, 550, 500, 480, 
    425, 675, 640, 537.269383315545, 480, 880, 660, 398.744360701036, 
    498.477523182843, 680, 570, 996.667896369511, 615, 465.21361356541, 
    695, 685, 527.313175658265, 606.871776276433, 510, 470, 484.315157058041, 
    740, 602.629313996836, 835, 840, 432.266380491809)), row.names = c(NA, 
    -178L), class = "data.frame"), structure(list(alco = c(14.23, 
    13.2, 13.16, 14.869321332286, 13.24, 13.5944999938917, 14.39, 
    14.06, 14.0161398827589, 13.86, 13.856647234763, 13.5979332399701, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 14.3428836574343, 
    13.64, 14.1452475925078, 12.93, 13.71, 12.85, 13.5, 14.0858840349417, 
    13.39, 13.3, 13.87, 13.8591446880074, 13.73, 13.58, 13.68, 
    13.76, 13.3799287188922, 13.9885101868331, 13.6344916376119, 
    13.05, 11.9177973524711, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 14.0097999118789, 13.9, 13.8281495690603, 13.7337982636901, 
    13.05, 13.7040611481761, 13.82, 14.0455182524748, 13.74, 
    14.477207550075, 14.22, 13.29, 14.3378122676622, 12.37, 12.33, 
    12.64, 11.1214608107665, 12.37, 12.17, 12.37, 12.565490543558, 
    12.3340749266081, 13.34, 12.21, 12.4460452808973, 13.86, 
    12.7312840569751, 12.99, 11.96, 11.66, 12.1334417904223, 
    11.84, 12.5597140230321, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 
    11.5935689011402, 12.16, 12.7607828487, 11.695475235909, 
    12.08, 12.08, 12, 12.69, 11.8713598547478, 11.62, 11.9948098860403, 
    11.81, 12.29, 14.0466365116629, 12.29, 12.08, 12.6, 12.9992477661547, 
    12.9640866814497, 12.51, 13.6762972359182, 12.0799926618289, 
    12.72, 12.22, 11.61, 12.7003120583082, 13.0070463977138, 
    11.76, 12.027356728267, 12.08, 11.03, 11.82, 12.5801847179927, 
    12.2888183792037, 11.7983103480678, 11.45, 13.5768789730122, 
    12.42, 13.05, 11.87, 12.07, 11.7207216659793, 11.79, 11.9818314844657, 
    12.04, 12.86, 12.88, 13.0323166090468, 12.7, 12.51, 12.6976654060383, 
    12.25, 12.53, 11.6469564451715, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 12.5264065648202, 13.6927552742241, 12.87, 
    13.32, 13.2872752971736, 12.7625294624941, 12.79, 13.6580651955098, 
    13.23, 13.2379228861789, 13.17, 13.84, 12.45, 14.34, 13.48, 
    13.178841387174, 13.69, 12.85, 12.96, 13.78, 13.73, 14.0444813985234, 
    13.0922716747764, 13.58, 13.4, 12.9887094220644, 12.77, 14.16, 
    13.129243651805, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    1.84171524617059, 0.698476983758165, 1.95, 1.92171701522828, 
    1.68046611754168, 1.87, 2.15, 1.69850199205553, 1.35, 2.16, 
    1.48, 1.60351708408405, 1.73, 1.87, 2.17507631301375, 1.92, 
    -0.148498805913885, 1.0935938736459, 3.1, 1.63, 3.8, 2.32420663994338, 
    1.6, 1.81, 2.23311269184323, 1.06125318768408, 2.20951519999828, 
    1.9, 0.973064329358708, 1.5, 1.48782077670805, 1.83, 1.53, 
    2.63741878611672, 2.06648135043083, 2.51302580820795, 1.65, 
    1.5, 2.63092332820103, 1.71, 2.43790758379177, 1.79226292310224, 
    3.98, 0.231541528401514, 4.04, 1.67106763493484, 1.68, 2.19893097126174, 
    1.032048113359, 0.00732171614206445, 1.65, 1.44288489504526, 
    1.9, 1.67, 1.7938491287837, 1.7, 1.97, 2.6426456266375, 0.94, 
    0.97504000237884, 1.36, 1.25, 1.62251408228098, 3.09964351034086, 
    1.21, 1.01, 1.17, 1.79418160500682, 1.75550908311226, 2.00249688716905, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.32118701172929, 0.99, 
    3.87, 1.76331503133775, 2.01439964482295, 1.13, 3.54364957440497, 
    0.90962840774049, 0.98, 1.61, 1.67, 0.606984351397613, 0.852112768910142, 
    3.84630665257074, 1.51, 2.43539246064689, 1.4870627311265, 
    1.52477270313614, -0.50824607741691, 4.5662617004694, 1.41, 
    0.881465064306481, 0.220508998746645, 2.08, 0.547929402392431, 
    2.76900075058579, 1.72, 1.21590611601274, 2.55, 1.73, 3.13341468197294, 
    1.29, 0.189113800951239, 3.74, 1.93074719968366, 2.68, 0.74, 
    1.39, -0.640054031369675, -0.162266091962984, 1.61, 3.43, 
    1.36986212955453, 2.4, 2.05, 4.43, 1.61477897660444, 4.31, 
    2.16, 1.53, 2.13, 1.10920529289271, 4.3, 1.35, 2.99, 2.31, 
    3.55, 1.24, 3.28845241003995, 3.62534913260274, 5.51, 3.59, 
    6.15442250206162, 2.81, 2.56, 4.80661303268526, 1.48705706774244, 
    3.88, 2.09212903204618, 5.04, 4.61, 3.24, 2.59338312606722, 
    3.12, 2.67, 1.9, 4.74716288820079, 1.29, 5.19, 4.12, 3.03, 
    1.68, 1.67, 3.83, 2.39776835645715, 2.8302309541737, 3.45, 
    2.76, 4.65338648471553, 4.26234378025465, 3.37, 2.58, 4.6, 
    3.5006854381886, 2.39, 3.91115912944149, 5.65, 3.91, 3.40120002422895, 
    2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.26381007717816, 
    2.61952014645759, 2.45, 2.61, 2.03200288396898, 2.48577823710122, 
    2.3, 2.0422730617117, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
    2.48, 2.56, 2.17078902984699, 2.65, 2.38735733635809, 2.52, 
    2.61, 3.22, 2.6979998333232, 2.40542402212905, 2.41937891718746, 
    2.53738213260576, 2.67828821050478, 2.36, 2.4420151222941, 
    2.7, 2.65, 2.85143545325828, 2.54631222100013, 2.55, 2.1, 
    2.51, 2.28552996552741, 2.12, 2.59, 2.29, 2.1, 2.44, 2.26742771824758, 
    2.12, 2.4, 2.27, 2.04, 2.59915714830052, 2.53392228768446, 
    2.44336484977018, 2.49901120403294, 2.51345548440374, 2.3, 
    2.68, 2.5, 1.36, 2.24874852825115, 1.79019466403048, 1.92, 
    2.16, 2.53, 2.56, 2.11485281811379, 1.92, 2.31860380938305, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.05736691117123, 1.71, 
    1.90008960353341, 2.31010739477794, 2.4, 2, 2.73765982644088, 
    2.51, 2.32, 2.58, 2.24, 2.26154232683478, 2.62, 2.12023601695505, 
    2.3, 2.32, 1.78864334118589, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.4689750546229, 2.04656082711151, 1.7, 2.17002882184993, 
    2.46, 1.88, 2.27399859117164, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.47745382907606, 
    2.22562217684278, 2, 2.42, 3.23, 2.74803631508974, 2.13, 
    2.39, 2.17, 2.29, 2.48722556600843, 2.55947896324899, 2.38, 
    2.4603955924811, 2.4, 2.4, 2.36, 2.25, 2.76612906377286, 
    2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.42085002820009, 
    2.19184955720896, 2.48, 2.22809295323588, 2.36, 2.62, 2.48, 
    2.36394088563424, 2.28, 2.08927487410994, 2.32, 2.38, 1.5927491161397, 
    2.35961345465926, 2.64, 2.38, 2.54, 2.36372805527296, 2.35, 
    2.3, 2.42187254215488, 2.6, 2.13634888166232, 2.69, 2.51744527113656, 
    2.32, 2.28, 2.73026449437004, 2.31599240037899, 2.48, 2.26, 
    2.37, 2.74), alca = c(15.6, 19.3902799471684, 16.7875821113822, 
    16.8, 21, 15.2, 17.8155294118677, 17.6, 14, 16, 17.1937554324922, 
    16.8, 16, 11.4, 13.5182196313374, 14.525493470991, 17.2605929322777, 
    20, 12.6676074236875, 15.2, 16, 18.6, 19.2543004354512, 17.8, 
    20, 19.4367371194862, 16.1, 17, 19.0328727638514, 18.6185850566768, 
    15.0269863269927, 19.1, 20.0392121410151, 17.9611994610573, 
    19, 20.5, 15.5, 19.4855252082657, 15.5, 13.2, 16.2, 18.8, 
    15.9644448562867, 16.8383791931537, 17, 13.195783650567, 
    16, 13.6619971652941, 18.8, 17.4, 12.4, 19.2763096707114, 
    15.6710281323992, 19.8935497256714, 18.3376450443871, 20.5, 
    16.3, 16.8, 14.7302558511526, 10.6, 21.9226696208057, 16.8, 
    18, 19, 19, 18.1, 20.7764087183208, 19.6, 18.8924160678445, 
    16.8, 20.4, 23.3091485513372, 24, 30, 21, 18.7233048786212, 
    16, 18, 14.8, 23.4694545225688, 19, 19.5852128889799, 24, 
    22.5, 18, 18, 22.8, 22.5051973485997, 21.6, 16.582349130156, 
    27.2896288645203, 22, 19.8423276091641, 18, 18, 19, 21.5, 
    16, 15.9154373523595, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
    19, 22.5, 19, 20, 19.5, 17.5368151236965, 27.6375184220962, 
    20.14454733865, 23.2392592319999, 21.5, 20.8, 22.0447626554603, 
    16, 18.1891272676944, 19.1291468017567, 24.8505705951888, 
    25.7765755957047, 21.5, 21, 21, 21.5, 28.5, 22.2503988586879, 
    22, 20.3456713594515, 20, 24, 21.5, 21.8332530404961, 23.9824520806817, 
    22.5355211509227, 25, 19.5, 22.6327170187736, 21, 20.2975577113077, 
    23.5, 20, 23.2610072353248, 21, 20, 21.5582735557004, 21.2447380131107, 
    21.5, 24, 22, 19.2257657281427, 18.5, 20, 22, 19.5, 14.3467772125449, 
    25, 25.7379730665252, 22.0639736730651, 18.6136064508654, 
    23.5654918592316, 18.5, 20.2991735524164, 22.5, 23, 19.5, 
    24.5, 25, 18.7623329441428, 19.5, 21.4604131757256, 20.5, 
    23, 20, 25.4279804225348, 24.5), mg = c(127, 100, 101, 113, 
    108.635847221251, 112, 96, 121, 97, 116.517540570877, 99.3517565523078, 
    117.441421287108, 89, 91, 102, 112, 120, 85.0998651930735, 
    96.6600097808631, 116, 126, 102, 112.772454322027, 95, 108.577611822787, 
    124, 93, 118.380348783064, 107, 96, 101, 106, 104, 104.662795459539, 
    110, 119.291666472978, 110, 112.259216507348, 86.6438777853623, 
    114.00536119775, 117, 95.0863794610428, 101, 114.176070673955, 
    107, 111, 102, 101, 111.214364999134, 108, 92, 108.679369740444, 
    111, 115, 99.9371834347108, 138.606299922453, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 76.6765758928763, 
    151, 116.208696896135, 86, 87, 139, 101, 76.9348783311193, 
    106.485711252781, 112, 104.020338181653, 100.03037782488, 
    86, 88.3533737400545, 78, 85, 88.0969408363609, 99, 90, 88, 
    84, 70, 81, 95.7985450990077, 87.9022012076328, 88, 114.769866188715, 
    97.1690860749779, 134, 85, 88, 91.5288707405662, 97, 88, 
    98.8408853993322, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 
    88, 84, 85, 86, 119.445469332352, 103.660711339836, 87, 96, 
    101.400120912243, 102, 86, 94.8122434521383, 85, 81.7927144603469, 
    92, 88, 80, 122, 103.419935852552, 98, 106, 97.9746410015037, 
    94, 89, 86.761760438656, 88, 101, 102.747372635698, 89, 97, 
    92.8096582307512, 112, 102, 86.9483821860565, 96.589118565896, 
    92, 113, 123, 85.0887171379081, 116, 98, 103, 102.858505634877, 
    89, 97, 98, 89, 101.495721289767, 81.5356799443214, 106, 
    106, 90, 88, 98.0797840763086, 88, 105, 112, 96, 86, 91, 
    95, 102, 92.7511368331608, 120, 107.201057268985), phe = c(2.8, 
    2.65, 2.8, 3.85, 2.8, 3.02044988215643, 2.5, 2.6, 2.92099813618839, 
    2.98, 2.95, 2.2, 2.6, 3.1, 3.00175489204244, 2.85, 2.8, 3.24604307650877, 
    3.3, 3.6901973402691, 3, 2.41, 2.61, 2.48, 2.53, 2.67251410410879, 
    2.91719256470729, 2.4, 2.95, 2.65, 3, 2.97619634658548, 2.42, 
    2.95, 2.66626227870701, 2.7, 2.6, 2.45, 3.21056349810722, 
    3.09669640059601, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 
    3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.80966767774797, 2.65028757012281, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.14681083455038, 3.5, 
    1.89, 2.64117889780151, 1.68867108154637, 2.11, 2.53, 2.68279644570831, 
    1.1, 2.95, 1.88, 2.57422053790843, 2.19764098978022, 1.65996096280173, 
    2.4815852826909, 1.27459161398199, 1.9, 2.83, 2.70461014524587, 
    2.2, 2.034750251899, 1.65, 2.20345237007548, 2.2, 2.30659328270603, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.72643899290713, 
    1.28199430870772, 2.55, 3.52, 3.13394218398262, 2.35047667486763, 
    1.90603599388787, 2.56, 2.5, 2.2, 2.21881145625662, 1.65, 
    2.0806995312454, 2.36, 2.74, 3.18, 2.55, 1.75, 2.03958249144393, 
    2.56, 2.46, 2.24778913109421, 2, 2.21232149903468, 2, 2.9, 
    3.18, 2.93514769537466, 2.0056141164098, 2.86, 2.91462558740783, 
    2.74, 2.13, 2.22, 2.1, 1.51, 1.63137698336964, 1.68241254705859, 
    1.7, 1.13055758257652, 1.62, 1.38, 1.83446405677507, 1.62, 
    1.19881238308577, 1.60095982386067, 1.4, 1.1158117427031, 
    2, 1.38, 0.693319324130144, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 1.42955752627054, 1.48, 1.20053879332468, 1.8, 1.9, 
    1.47154566941437, 1.67104096738621, 2.3, 1.83, 1.08245029649258, 
    1.39, 1.35, 1.90575322754925, 1.7, 1.48, 1.55, 2.07282785602772, 
    1.25, 1.39, 1.68, 1.38416261563822, 1.61336285335532, 1.84216135329985, 
    1.65, 2.05), fla = c(2.99518504611284, 2.76, 3.00344011527451, 
    3.49, 2.91323045979376, 3.39, 2.52, 2.51, 2.98, 3.35022892580221, 
    3.32, 2.43, 2.76, 2.40292325811244, 3.64, 2.91, 3.14, 3.4, 
    3.44829919326691, 3.75911899659805, 2.78886327047952, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.17472306984601, 2.97, 2.33, 
    3.25, 3.19, 2.69, 3.1742337652566, 2.85166552352943, 2.98, 
    2.63408364739451, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.18166376358404, 2.92, 3.54, 2.76145800788311, 
    2.99, 3.74, 2.79, 2.9, 2.78, 2.49220506243246, 3.23, 3.67, 
    0.57, 2.03562802328415, 1.41, 1.79, 3.17478784813135, 1.75, 
    2.65, 2.00314321139646, 2, 1.3, 2.75979599177224, 1.02, 2.86, 
    1.28952107140914, 2.89, 2.14, 1.57, 1.63637360913159, 0.910628901137842, 
    1.85, 2.39278828105449, 2.26, 2.53, 1.58, 1.59, 2.21, 1.8333760290144, 
    1.69, 1.61, 1.69, 1.77932900270355, 0.765291483360076, 1.25, 
    1.46, 2.53664740476918, 2.41608578019038, 2.27, 0.99, 1.96634432866165, 
    3.27491073596043, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 2.02179299400739, 2.92, 2.4237703565388, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.62282350669105, 1.64, 
    2.42214802610535, 3.73904053468266, 2.19554279942477, 1.58913101095313, 
    3.03, 2.65, 3.15, 2.24, 2.20482889414939, 1.75, 1.04141844300105, 
    1.04222437737912, 0.959317876960491, 1.2, 0.58, 0.66, 0.47, 
    1.06379410038321, 0.845015233991935, 0.6, 0.5, 0.82227377801686, 
    0.52, 0.8, 0.976218352686069, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.642387573003209, 0.58, 
    1.31, 1.1, 0.92, 1.67755293297847, 0.6, 0.379779511521754, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.876851496545594, 
    0.7, 0.61, 0.472800293468749, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.0600187082076738, 0.3, 0.24, 0.39, 0.19562276221361, 0.324840416054614, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.289686686451684, 
    0.191047031214322, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.315811232807695, 
    0.295368967261752, 0.34, 0.27, 0.324498495629916, 0.26, 0.29, 
    0.231654454069237, 0.42, 0.335231870167572, 0.29, 0.26, 0.207640964817082, 
    0.319847992554239, 0.28, 0.2, 0.34, 0.337119589384327, 0.42790988722662, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.397143150929958, 0.16805968620837, 
    0.28, 0.371549123254939, 0.53, 0.32, 0.190364215961, 0.45, 
    0.37, 0.26, 0.27, 0.508939250905837, 0.217898879566243, 0.201945158906055, 
    0.403081453827165, 0.523865629944693, 0.21, 0.13, 0.34, 0.24, 
    0.43, 0.35, 0.43, 0.3, 0.220230456629123, 0.4, 0.61, 0.22, 
    0.3, 0.356521482040256, 0.480525019591072, 0.337719874673149, 
    0.272846241768671, 0.677142788909001, 0.5, 0.319706983432149, 
    0.25, 0.17, 0.32, 0.45160520033498, 0.272132994441054, 0.24, 
    0.131859229863138, 0.323531912187373, 0.29, 0.34, 0.37, 0.329670688515583, 
    0.66, 0.573933064502277, 0.48, 0.39, 0.29, 0.24, 0.26, 0.490356539178066, 
    0.42, 0.312295120179462, 0.52, 0.3, 0.215043459239084, 0.304409042816548, 
    0.415300907500161, 0.32, 0.447030605164853, 0.43, 0.3, 0.296234075867235, 
    0.27172728851916, 0.39, 0.463143566896953, 0.4, 0.388393794796535, 
    0.323818293653166, 0.469373528816984, 0.27, 0.17, 0.6, 0.63, 
    0.53, 0.63, 0.58, 0.53, 0.53, 0.320813932157253, 0.507666214867731, 
    0.562197977794879, 0.403744265307401, 0.43, 0.4, 0.47, 0.45, 
    0.304843086701887, 0.22, 0.436268305674674, 0.26, 0.61, 0.53, 
    0.618040036058466, 0.423817231010275, 0.63, 0.53, 0.52, 0.407453897191028, 
    0.5, 0.6, 0.4, 0.41, 0.622410914462538, 0.43, 0.4, 0.39, 
    0.463564847631544, 0.374634220860484, 0.48, 0.508419918609273, 
    0.288824999633363, 0.43, 0.43, 0.53, 0.40646606818555), pro = c(2.29, 
    1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 
    1.57, 1.81, 1.8467646757196, 2.96, 1.46, 2.74102414169393, 
    2.56919403084473, 1.86, 1.66, 2.1, 0.548815004275222, 1.69, 
    1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.11063757244405, 
    2.20783570506213, 1.54, 1.86, 1.36, 1.44, 1.37, 2.1944809866276, 
    1.74359385675469, 1.80667347310703, 2.29981099361995, 2.0067498024677, 
    2.03, 0.763944387019292, 2.19, 2.14, 2.38, 2.08, 1.43883976854417, 
    2.29, 1.87, 1.68, 1.62, 2.45, 0.854012897540544, 1.66, 2.1353198144322, 
    0.42, 2.03197759966334, 2.14092942832659, 0.73, 1.87, 1.03, 
    2.37637396742469, 2.1172070544672, 1.04, 1.47033275081009, 
    2.5, 1.46, 1.87, 1.03, 1.86109539302824, 1.66138288506597, 
    1.15, 1.46, 0.447166162647622, 2.76, 1.95, 1.43, 1.89114303361576, 
    1.59605588670555, 1.62, 2.33814131212785, 1.46, 1.56, 1.34, 
    1.35, 1.38, 1.64, 1.63, 1.84788864990898, 1.99, 1.35, 3.28, 
    1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.33576798105152, 
    1.42, 2.01340107104863, 1.63, 2.08, 2.49, 1.30007535168376, 
    1.4261319282229, 1.03033563645413, 2.34785016515876, 1.04, 
    2.27464754617954, 1.27137975438926, 1.61, 0.83, 1.87, 1.83, 
    2.43915507220354, 1.71, 1.5913920497914, 2.91, 1.35, 1.77, 
    1.27982220740766, 1.9, 1.34460928308663, 0.94, 1.48250942891949, 
    0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 0.912919415220259, 
    0.75, 0.686374897694968, 0.55, 1.38539169913529, 1.14, 1.3, 
    0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 0.734901735235562, 
    1.1421241952718, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 0.759952399363677, 
    0.661081236939638, 0.94, 0.614195659907133, 1.15, 1.46, 0.97, 
    1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
    ), col = c(5.64, 4.38, 5.68, 7.8, 4.55259875958816, 6.75, 
    4.51284905470668, 5.05, 7.15654024552037, 7.22, 5.75, 7.64187463816128, 
    5.6, 6.90788907264473, 7.5, 5.20545982744043, 6.66022535543556, 
    4.96552741312976, 8.7, 5.1, 5.65, 4.5, 4.16174085422211, 
    3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 6.25963101153402, 
    5.73152270733452, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
    4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 5.33267304620994, 
    8.9, 7.2, 6.55566437322967, 5.73987672246384, 6.3, 5.85, 
    6.25, 7.48073323464649, 4.93241311513842, 6.8, 6.46570422357521, 
    3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 
    5.75555993150267, 3.38, 3.74, 4.42069506160535, 3.21, 3.8, 
    4.6, 3.6312496281375, 5.4874348662679, 6.86965191976054, 
    2.5, 3.9, 1.59240514353783, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 
    1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 6.24147624988054, 
    2.9, 4.12673297877991, 4.82319223089096, 5.0328416305694, 
    4.87509819608541, 2.8, 5.69924786257408, 2.94, 2.7, 3.4, 
    3.3, 2.7, 1.84223064119955, 2.9, 5.921230711639, 2.66942751742339, 
    3.08, 2.9, 1.9, 1.94418264707968, 5.39427120568279, 3.4, 
    1.28, 3.25, 3.20941353880743, 3.32090517056405, 5.27141994620603, 
    2.8, 3.55202677422688, 3.94, 1.34565596645223, 2.12, 4.52380756625253, 
    8.29098788838242, 9.50865812518686, 5.99895593048065, 5, 
    5.45, 7.1, 6.51881579326662, 5, 5.7, 4.92, 9.80374200660709, 
    7.77902385642172, 4.35, 4.4, 8.21, 4, 4.9, 5.87925098486418, 
    8.42, 9.4, 5.15167399413448, 5.3982865974542, 7.1, 10.52, 
    7.6, 7.9, 7.94169781690704, 7.5, 13, 3.64017753042169, 7.65, 
    4.79624552035243, 5.58, 8.39336851706126, 7.5911821645897, 
    6.62, 10.68, 10.26, 9.92792707025594, 8.5, 4.07888952383566, 
    5.25483699691218, 9.7, 7.7, 7.3, 7.60727141069187, 9.3, 9.2
    ), hue = c(1.04, 1.05, 1.08963382170154, 0.86, 0.860156748182529, 
    1.05, 1.02, 1.13099443389752, 1.08, 1.11308478697141, 1.25, 
    1.10826605608877, 1.15, 1.25, 1.2, 1.28, 1.10509480684354, 
    1.13, 1.23, 1.23130230126448, 1.09, 0.995602120631763, 1.11, 
    1.09, 1.12, 1.13, 1.37999789649266, 1.02, 1.00950529497813, 
    1.04, 1.19, 1.09, 1.28105178543111, 1.25, 1.1, 0.995230212157301, 
    1.03175723159248, 0.916880788667174, 1.18, 1.05389410392301, 
    0.95, 0.91, 0.88, 0.82, 1.18719230813849, 0.87, 0.779486627482723, 
    1.17611471122281, 1.07, 1.12, 1.12, 1.15753446545834, 1.00940009165212, 
    0.906320914772117, 1.13183958216931, 1.07449770417984, 0.94, 
    1.07, 0.89, 1.07778255465994, 1.25, 0.98, 1.23, 1.22, 0.868209286353222, 
    1.05032702283791, 1.47678947434944, 1.12, 1.02, 1.28, 0.906, 
    1.36, 0.98, 1.31, 1.24845963761014, 1.23, 1.19, 0.96, 1.06, 
    1.19, 1.38, 0.952734453773719, 1.31, 0.84, 1.14668327107186, 
    1.23, 1.04309978564008, 0.87541317438713, 1.00938168482971, 
    1.07, 1.08, 1.05, 0.910646511002634, 1.15, 0.969263695679089, 
    1.5245180910238, 0.865560965198094, 1.08744583200686, 1.21675237595536, 
    1.42, 1.27, 1.2825043238087, 0.8, 0.855407340229776, 1.04, 
    1.14535781200186, 1, 1.18247248578978, 0.86, 1.2824339034904, 
    0.774432223579557, 0.9, 0.796334233670414, 1.1, 0.93, 1.71, 
    0.95, 0.953961105189857, 0.7, 1.24223219077823, 0.8, 1.48250900595044, 
    0.92, 0.73, 1.19547136392871, 0.86, 0.69, 0.97, 0.89, 0.79, 
    0.769541904836584, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
    0.81, 0.541298844073867, 0.587998731148396, 0.582826488608536, 
    0.89, 0.729830656652096, 0.65, 0.6, 0.37703204809815, 0.54, 
    0.55, 0.648096997008563, 0.520046782036451, 0.48, 0.61, 0.56, 
    1.2849280371729, 0.6, 0.57, 0.67, 0.57, 0.906112292046848, 
    0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.732937208785652, 
    0.74, 0.67, 0.899578233713052, 0.732510026213003, 0.532796959288398, 
    0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 2.69838245450731, 
    2.93, 2.60278631826905, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
    2.9, 2.40693130794854, 3, 3.21808227977611, 2.65, 3.50767636976062, 
    2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.64120076899659, 3.2, 3.35481566479497, 
    2.77, 3.77669866261637, 2.53538917326979, 2.71, 2.88, 2.87, 
    3, 2.87, 3.47302705625672, 3.27139673876753, 2.6697594682151, 
    2.69, 3.14651008337072, 3.38, 3, 3.18858934197664, 3, 3.35, 
    3.33, 3.23266249331285, 3.33, 2.75, 3.1, 2.64643828295881, 
    2.54939964165327, 3.26, 2.93, 3.2, 3.03, 2.6063540067656, 
    2.84, 2.87, 1.82, 3.58668187602091, 1.59, 2.46, 2.87, 2.60207365123712, 
    2.3, 3.18, 3.48, 2.19210305852156, 3.26867185628391, 1.82, 
    3.81908650879188, 2.8582988117252, 3.5, 3.13, 2.14, 2.48, 
    2.52, 1.71871496996227, 2.989705998958, 3.38185301707093, 
    3.14, 2.72, 2.01, 2.12684137582921, 2.76978809071436, 2.42101690958406, 
    3.21, 2.75, 3.21, 2.69556126609551, 2.65, 2.98665185213615, 
    3.3, 2.45401680373648, 2.63, 1.59375357063421, 2.74, 4.10314011925603, 
    2.83, 2.96, 2.77, 3.38, 1.85935121619962, 3.57, 3.3, 3.17, 
    2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.33551975122851, 3.19, 
    2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.27493449424599, 
    3.64, 2.62619774840037, 2.84, 3.47002373492891, 2.78, 2.74835095854079, 
    1.29699195396864, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
    1.82, 2.15, 0.927231727002023, 0.945603602630303, 1.78126282287256, 
    2.05, 2, 1.91179249941505, 1.33, 1.78058608455061, 1.62, 
    1.33, 2.34752572594602, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
    1.73, 1.96, 1.78, 1.58, 2.68618328668206, 1.99233271640228, 
    1.75, 1.68, 1.75, 1.43664360089226, 1.37304810328889, 1.94981374053375, 
    1.59190651475918, 1.72515248808131, 1.95400079947891, 1.44361286373842, 
    1.14605725936754, 1.72934446685968, 1.56, 1.62, 1.6), prol = c(1065, 
    1050, 727.060043228383, 791.401945536871, 735, 1450, 1290, 
    1295, 1045, 1045, 964.258808320528, 1280, 1320, 779.383840778735, 
    1547, 1310, 1280, 880.121259807643, 1680, 845, 760.500761923268, 
    770, 1035, 1015, 845, 1346.72607185946, 1195, 1285, 915, 
    1035, 1285, 972.238075563128, 878.080207949591, 1235, 1228.89377613312, 
    920, 880, 828.017826129351, 344.864718620363, 760, 795, 1035, 
    1095, 680, 885, 1080, 1065, 985, 1060, 1122.74054751551, 
    1150, 1265, 1190, 988.678269065213, 1060, 1120, 970, 1270, 
    1285, -68.8197456503109, 529.646732527554, 480.106071743898, 
    630, 390.752029189483, 475.20974511073, 763.450880519329, 
    502, 659.814133687635, 450.088887947591, 718, 870, 410, 472, 
    595.434116289151, 886, 525.152898690785, 392, 472.246301026171, 
    798.994232638948, 463, 278, 714, 486.661294003673, 515, 628.296475437978, 
    509.019385398467, 294.320658359579, 562, 680, 625, 480, 450, 
    495, 968.918768125399, 345, 247.902559852086, 625, 428, 660, 
    406, 595.782038430592, 562, 438, 415, 672, 465.399588624115, 
    510, 373.984259649002, 312, 680, 562, 325, 632.307230403567, 
    434, 385, 391.828085935834, 449.988163689548, 654.912450389201, 
    372, 564, 867.346452112471, 901.008587332477, 365, 380, 882.626838465211, 
    378, 562.178161391738, 466, 649.576167577926, 580, 681.24659878354, 
    1113.27089215929, 560, 600, 342.904393335335, 695, 720, 515, 
    580, 590, 600, 578.944522970699, 520, 550, 286.178209803425, 
    830, 415, 582.902066737758, 628.078744849197, 550, 500, 480, 
    425, 675, 640, 725.728469006353, 480, 880, 660, 735.797710767859, 
    390.61337923719, 680, 570, 585.194085766536, 615, 567.056324443675, 
    695, 685, 550.04867899558, 276.170613807844, 510, 470, 829.135888458357, 
    740, 497.382662892407, 835, 840, 151.048534432588)), row.names = c(NA, 
    -178L), class = "data.frame"), structure(list(alco = c(14.23, 
    13.2, 13.16, 14.9346660480161, 13.24, 14.0696510627186, 14.39, 
    14.06, 13.6908958378912, 13.86, 13.5546642026438, 13.2112780816, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 14.6380845418911, 
    13.64, 13.9194824260199, 12.93, 13.71, 12.85, 13.5, 13.5172581646065, 
    13.39, 13.3, 13.87, 13.1538223808876, 13.73, 13.58, 13.68, 
    13.76, 12.8121616958956, 13.4871167385428, 12.6743577849002, 
    13.05, 11.7054762563849, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 13.8464351637932, 13.9, 13.5337343959759, 13.9068639254132, 
    13.05, 14.2946129449228, 13.82, 13.2131830805516, 13.74, 
    14.4376208058756, 14.22, 13.29, 14.1442504059769, 12.37, 
    12.33, 12.64, 11.7017913650128, 12.37, 12.17, 12.37, 12.8570932507154, 
    12.2189384870883, 13.34, 12.21, 12.7102735124503, 13.86, 
    12.2832522600771, 12.99, 11.96, 11.66, 12.1674626815485, 
    11.84, 13.5964299182373, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 
    11.7649303014179, 12.16, 12.179128035445, 12.293593063437, 
    12.08, 12.08, 12, 12.69, 12.0320968164051, 11.62, 12.1164808281952, 
    11.81, 12.29, 13.6427658180593, 12.29, 12.08, 12.6, 12.5359081629665, 
    12.6330562880808, 12.51, 11.7744036008169, 12.1245297790835, 
    12.72, 12.22, 11.61, 11.5176014113202, 12.2830885561154, 
    11.76, 11.7089266890118, 12.08, 11.03, 11.82, 12.0472237764426, 
    12.6230382330544, 12.2061672643947, 11.45, 14.4062927641237, 
    12.42, 13.05, 11.87, 12.07, 12.9623719128962, 11.79, 12.7871164203771, 
    12.04, 12.86, 12.88, 12.7118994604516, 12.7, 12.51, 12.5370440373913, 
    12.25, 12.53, 12.4653033217075, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 12.7312876406219, 11.7296914314344, 12.87, 
    13.32, 12.5223324144939, 11.8111332764352, 12.79, 13.263646620987, 
    13.23, 13.9980490302689, 13.17, 13.84, 12.45, 14.34, 13.48, 
    13.0094997323742, 13.69, 12.85, 12.96, 13.78, 13.73, 13.9792605527585, 
    13.8475914093224, 13.58, 13.4, 12.6007654093716, 12.77, 14.16, 
    13.7087319247636, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    2.42420739732838, 0.934444349922624, 1.95, 1.92403076223105, 
    1.53657071685561, 1.87, 2.15, 2.05393119669511, 1.35, 2.16, 
    1.48, 1.47129943743027, 1.73, 1.87, 3.12343398237575, 1.92, 
    0.317455036662817, 0.621422252475804, 3.1, 1.63, 3.8, 0.77704940002195, 
    1.6, 1.81, 2.72929845037922, 1.6619575027743, 2.02838851100752, 
    1.9, 1.24048849553409, 1.5, 0.530932983988422, 1.83, 1.53, 
    2.80752321073658, 1.97883727820717, 1.59980107852125, 1.65, 
    1.5, 3.57552984746136, 1.71, 2.55026162200026, 2.2346390845058, 
    3.98, 0.806888510133869, 4.04, 1.05610727350555, 1.68, 1.76148870904789, 
    1.41801050725449, -0.787280138314428, 1.65, 1.60815883105086, 
    1.9, 1.67, -0.191360179069195, 1.7, 1.97, 1.82116610944346, 
    0.94, 1.79157344019366, 1.36, 1.25, 0.654526269586227, 3.32274998224297, 
    1.21, 1.01, 1.17, 2.03943372627, 1.40701163448787, 1.13009659198369, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.38347415835829, 0.99, 
    3.87, 1.8810916223629, 1.04145190175562, 1.13, 3.45941177425381, 
    1.90420642582157, 0.98, 1.61, 1.67, 0.51720340111101, 2.471496104151, 
    1.9498556964427, 1.51, 2.98411895956403, 1.40047726417028, 
    0.556523550245382, 1.05002341092146, 4.09076955969018, 1.41, 
    2.75963066056934, 0.767491587610228, 2.08, 1.09424334563268, 
    1.86248506042625, 1.72, 0.945240073147612, 2.55, 1.73, 1.58644296281902, 
    1.29, 1.15919411563019, 3.74, 1.49495784168708, 2.68, 0.74, 
    1.39, 1.32977849878375, 0.430701686608856, 1.61, 3.43, 2.66824295165754, 
    2.4, 2.05, 4.43, 0.995612804380755, 4.31, 2.16, 1.53, 2.13, 
    2.25887022528441, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 3.34082597159902, 
    4.51330454209435, 5.51, 3.59, 3.92433081108239, 2.81, 2.56, 
    2.98225389612809, 1.74155106683282, 3.88, 3.75625436309127, 
    5.04, 4.61, 3.24, 4.99367346145089, 3.12, 2.67, 1.9, 3.19869208896228, 
    1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.86944387983738, 
    4.04952633217628, 3.45, 2.76, 4.20517036639702, 3.14619994269769, 
    3.37, 2.58, 4.6, 3.97962060084999, 2.39, 2.98675286362778, 
    5.65, 3.91, 1.52878140070513, 2.59, 4.1), ash = c(2.43, 2.14, 
    2.67, 2.5, 2.3178383482741, 2.47329781653207, 2.45, 2.61, 
    2.18869567030191, 2.09676922947641, 2.3, 2.49897060125251, 
    2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.50616578425331, 
    2.65, 2.81171911623206, 2.52, 2.61, 3.22, 2.30106983687973, 
    2.90098482661378, 2.70438049556565, 2.85049364077197, 2.57804101220865, 
    2.36, 2.26850325786882, 2.7, 2.65, 2.29454002931905, 2.16050587398271, 
    2.55, 2.1, 2.51, 2.26263056896379, 2.12, 2.59, 2.29, 2.1, 
    2.44, 2.23555957374173, 2.12, 2.4, 2.27, 2.04, 2.55704692321723, 
    2.56047187979815, 2.17661275367832, 2.57534501081788, 2.54760261114933, 
    2.3, 2.68, 2.5, 1.36, 2.36484556287961, 2.46653303108286, 
    1.92, 2.16, 2.53, 2.56, 2.52104820672325, 1.92, 2.46927850133152, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.93914203478492, 1.71, 
    1.67705532494115, 2.5028579220682, 2.4, 2, 2.51547959772033, 
    2.51, 2.32, 2.58, 2.24, 2.37828154939987, 2.62, 2.91330884709876, 
    2.3, 2.32, 2.22824542624591, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.43584770632758, 2.20499877494589, 1.7, 2.15981716021748, 
    2.46, 1.88, 2.21384767037017, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.39644330144316, 
    2.44774166176788, 2, 2.42, 3.23, 2.63007681598745, 2.13, 
    2.39, 2.17, 2.29, 2.51621973200974, 2.53583336098112, 2.38, 
    2.46075416371163, 2.4, 2.4, 2.36, 2.25, 2.53892796143381, 
    2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 3.00790695218961, 
    2.69371404845968, 2.48, 2.17479323063464, 2.36, 2.62, 2.48, 
    1.91836749086359, 2.28, 2.40108054532304, 2.32, 2.38, 1.72268676720783, 
    2.41740271486179, 2.64, 2.38, 2.54, 2.51856439646415, 2.35, 
    2.3, 2.85185300215743, 2.6, 2.7273768786626, 2.69, 2.66092452763995, 
    2.32, 2.28, 2.66211815825709, 2.27513496873908, 2.48, 2.26, 
    2.37, 2.74), alca = c(15.6, 17.341847169009, 19.2350325562699, 
    16.8, 21, 15.2, 10.9766033879955, 17.6, 14, 16, 17.1784946293487, 
    16.8, 16, 11.4, 16.3532661455145, 14.3458183162358, 18.109978167083, 
    20, 15.177071167156, 15.2, 16, 18.6, 17.9790475168069, 17.8, 
    20, 21.2515943685262, 16.1, 17, 17.8924304632082, 15.9769896479224, 
    18.3980536968151, 19.1, 17.2263255636966, 15.5801863554197, 
    19, 20.5, 15.5, 17.1481654822906, 15.5, 13.2, 16.2, 18.8, 
    12.7948733084216, 20.6405836415007, 17, 11.4509171242684, 
    16, 16.3591176042318, 18.8, 17.4, 12.4, 20.183555202078, 
    17.6186319479464, 16.2779269865113, 16.4516768194973, 20.5, 
    16.3, 16.8, 14.0473987297718, 10.6, 21.9504401323231, 16.8, 
    18, 19, 19, 18.1, 15.5760486681704, 19.6, 19.0398136268751, 
    16.8, 20.4, 18.2037137174392, 24, 30, 21, 19.9257756162501, 
    16, 18, 14.8, 16.337605866941, 19, 15.164050129799, 24, 22.5, 
    18, 18, 22.8, 15.8128302382249, 21.6, 25.3380909578189, 29.2669226461133, 
    22, 22.3752237381273, 18, 18, 19, 21.5, 16, 19.3813044240103, 
    18, 17.5, 18.5, 21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 
    17.5695792002104, 18.202873106751, 20.8607716127164, 20.3307293439691, 
    21.5, 20.8, 17.7200451829831, 16, 21.8649913805395, 24.0504304815959, 
    16.1613017194419, 18.5006056447225, 21.5, 21, 21, 21.5, 28.5, 
    26.9862041001873, 22, 22.186423422835, 20, 24, 21.5, 18.9270871404247, 
    21.9301237714784, 21.2171335183124, 25, 19.5, 18.9886465618803, 
    21, 17.5901974063342, 23.5, 20, 24.3614033786281, 21, 20, 
    18.3408113454997, 18.3160711209709, 21.5, 24, 22, 18.8977601501616, 
    18.5, 20, 22, 19.5, 17.3978411004661, 25, 21.0376832922575, 
    21.2019512657537, 14.7816308057731, 22.0359756568777, 18.5, 
    17.2453304157393, 22.5, 23, 19.5, 24.5, 25, 27.1814837533162, 
    19.5, 17.5449336334506, 20.5, 23, 20, 25.1946874324979, 24.5
    ), mg = c(127, 100, 101, 113, 126.608780544816, 112, 96, 
    121, 97, 102.517108804055, 111.163342282521, 126.154242445217, 
    89, 91, 102, 112, 120, 88.0476259615363, 98.409126515624, 
    116, 126, 102, 100.091565987769, 95, 114.235965519354, 124, 
    93, 124.539519640758, 107, 96, 101, 106, 104, 94.7624307151813, 
    110, 119.837564557922, 110, 80.357093555135, 92.2312961560751, 
    112.777546143788, 117, 100.712629290987, 101, 112.016161956277, 
    107, 111, 102, 101, 111.639561272852, 108, 92, 122.641643504068, 
    111, 115, 123.509033193323, 114.577936799232, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 95.9078987184585, 
    151, 91.024143286588, 86, 87, 139, 101, 106.266674128686, 
    89.7717198425467, 112, 72.9963578312565, 123.216380685856, 
    86, 85.9343821715497, 78, 85, 106.218294584851, 99, 90, 88, 
    84, 70, 81, 108.61939317139, 97.7661350383759, 88, 107.705049115534, 
    81.1198902913673, 134, 85, 88, 62.5657911250672, 97, 88, 
    107.470777021291, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 
    88, 84, 85, 86, 87.3212374912028, 85.3426824024164, 87, 96, 
    74.2575282969849, 102, 86, 89.6545796126582, 85, 80.091619128933, 
    92, 88, 80, 122, 95.3784725762665, 98, 106, 100.454739686125, 
    94, 89, 90.0887885486618, 88, 101, 93.206947083613, 89, 97, 
    89.1053984729165, 112, 102, 98.4266559240184, 100.366288820236, 
    92, 113, 123, 80.4052931279186, 116, 98, 103, 83.5507449893154, 
    89, 97, 98, 89, 108.190320035932, 84.5121784751094, 106, 
    106, 90, 88, 119.252021983829, 88, 105, 112, 96, 86, 91, 
    95, 102, 96.5187414372604, 120, 120.592833157714), phe = c(2.8, 
    2.65, 2.8, 3.85, 2.8, 3.24587580825232, 2.5, 2.6, 2.88967494012493, 
    2.98, 2.95, 2.2, 2.6, 3.1, 2.98471069170328, 2.85, 2.8, 2.56601644510089, 
    3.3, 2.68116821138582, 3, 2.41, 2.61, 2.48, 2.53, 2.05801389443022, 
    2.69061754361136, 2.4, 2.95, 2.65, 3, 2.60896880811029, 2.42, 
    2.95, 1.82663918956401, 2.7, 2.6, 2.45, 3.09842965400656, 
    2.9406650157121, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
    2.75, 2.88, 2.72, 2.45, 3.88, 2.48529776614585, 2.86988083069066, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.65120857014757, 3.5, 
    1.89, 2.85804045500146, 2.24018537857284, 2.11, 2.53, 3.55836992912531, 
    1.1, 2.95, 1.88, 2.5738374181957, 2.39958556815196, 2.32822155354865, 
    2.43478473414477, 1.91501995310377, 1.9, 2.83, 2.72103188735397, 
    2.2, 1.6168353628582, 1.65, 2.25940622766717, 2.2, 1.73299162905665, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 1.87201441918904, 
    1.89376457653807, 2.55, 3.52, 2.81148991649949, 2.87158029660607, 
    1.82259003601198, 2.56, 2.5, 2.2, 1.71438016806141, 1.65, 
    2.4068971017545, 2.36, 2.74, 3.18, 2.55, 1.75, 1.88985698849646, 
    2.56, 2.46, 1.92209331577761, 2, 2.2581164467929, 2, 2.9, 
    3.18, 2.29890440128239, 1.73998799605247, 2.86, 2.3326905465951, 
    2.74, 2.13, 2.22, 2.1, 1.51, 2.21304270163441, 1.3401064287209, 
    1.7, 1.08674469137604, 1.62, 1.38, 1.97357835051711, 1.62, 
    1.56155889152724, 1.63385863284752, 1.4, 1.33378886760547, 
    2, 1.38, 1.55976217001403, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 1.77424381196917, 1.48, 1.01396743669473, 1.8, 1.9, 
    1.28405438408, 1.3016423272011, 2.3, 1.83, 1.91427467683146, 
    1.39, 1.35, 1.82437306789091, 1.7, 1.48, 1.55, 1.80923562676459, 
    1.25, 1.39, 1.68, 1.93418566124393, 1.63863032626769, 1.06240250522116, 
    1.65, 2.05), fla = c(2.95001434893841, 2.76, 3.73823997298893, 
    3.49, 2.8792052927691, 3.39, 2.52, 2.51, 2.98, 3.22747584490069, 
    3.32, 2.43, 2.76, 2.85394164983816, 3.64, 2.91, 3.14, 3.4, 
    3.18716533854206, 3.00144815585851, 3.14033974932444, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.36774471599862, 2.97, 2.33, 
    3.25, 3.19, 2.69, 2.89305264033255, 1.84082704677555, 2.98, 
    1.53624051189393, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.28866877842923, 2.92, 3.54, 2.40745425720286, 
    2.99, 3.74, 2.79, 2.9, 2.78, 3.28545466533264, 3.23, 3.67, 
    0.57, 1.74057864702391, 1.41, 1.79, 2.93606226249396, 1.75, 
    2.65, 2.21122072379586, 2, 1.3, 3.95534130199875, 1.02, 2.86, 
    1.346007842334, 2.89, 2.14, 1.57, 1.69423474642285, 1.71534622188622, 
    1.85, 2.90792291213706, 2.26, 2.53, 1.58, 1.59, 2.21, 2.0504294862168, 
    1.69, 1.61, 1.69, 2.29643667641375, 1.53050480074525, 1.25, 
    1.46, 2.26035958111556, 2.69018075334164, 2.27, 0.99, 1.76459298547967, 
    4.17629975203496, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 1.94285508304309, 2.92, 2.85778930231334, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.70885147871082, 1.64, 
    2.37820055679449, 3.76209822937132, 1.78312641370235, 1.42057628407583, 
    3.03, 2.65, 3.15, 2.24, 1.93253648090871, 1.75, 0.783782709476205, 
    1.31912957901353, 0.599506786988994, 1.2, 0.58, 0.66, 0.47, 
    1.14714862584516, 0.298967158943631, 0.6, 0.5, 0.701380758355029, 
    0.52, 0.8, 1.12018380192378, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.811467785118316, 0.58, 
    1.31, 1.1, 0.92, 0.617329514240456, 0.6, 0.0157386473941978, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.70851100523679, 
    0.7, 0.61, 0.539293407767133, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.332224829426326, 0.3, 0.24, 0.39, 0.151859951871039, 0.332065108476828, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.106459654638938, 
    0.328673402044132, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.409222243577849, 
    0.26617358914109, 0.34, 0.27, 0.437537699103405, 0.26, 0.29, 
    0.270007970314747, 0.42, 0.465405942597835, 0.29, 0.26, 0.349070961188307, 
    0.337058448093174, 0.28, 0.2, 0.34, 0.308617452487817, 0.0950824176178444, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.215244835949804, 0.231200320877312, 
    0.28, 0.399625433589391, 0.53, 0.32, 0.25801641746173, 0.45, 
    0.37, 0.26, 0.27, 0.607590169474296, 0.0360160589397858, 
    0.322106686442943, 0.319753330369408, 0.515030030625402, 
    0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.253350212723955, 
    0.4, 0.61, 0.22, 0.3, 0.272096955894109, 0.428840261901934, 
    0.35509693090252, 0.456214541239755, 0.46988597372303, 0.5, 
    0.332979212599061, 0.25, 0.17, 0.32, 0.244482172631415, 0.43496047042457, 
    0.24, 0.442306063195197, 0.429319097670905, 0.29, 0.34, 0.37, 
    0.29828838863787, 0.66, 0.394767649607249, 0.48, 0.39, 0.29, 
    0.24, 0.26, 0.217733071841953, 0.42, 0.429978325781865, 0.52, 
    0.3, 0.473884072959631, 0.34049535688689, 0.4761944678815, 
    0.32, 0.40500593134469, 0.43, 0.3, 0.242204767586368, 0.279443514605757, 
    0.39, 0.384923940228425, 0.4, 0.345597619529028, 0.357849756596016, 
    0.523603693403033, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 
    0.53, 0.53, 0.377467519827105, 0.490115909336088, 0.442096700561801, 
    0.395748560810639, 0.43, 0.4, 0.47, 0.45, 0.303787499135802, 
    0.22, 0.385233170336197, 0.26, 0.61, 0.53, 0.710798299099574, 
    0.626097482698658, 0.63, 0.53, 0.52, 0.457575067516427, 0.5, 
    0.6, 0.4, 0.41, 0.631500108196158, 0.43, 0.4, 0.39, 0.515362649237375, 
    0.554672113945324, 0.48, 0.534850588941401, 0.580178964532745, 
    0.43, 0.43, 0.53, 0.473998396945596), pro = c(2.29, 1.28, 
    2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 
    1.81, 1.59437723256644, 2.96, 1.46, 2.42480336183608, 2.80450524407023, 
    1.86, 1.66, 2.1, 0.472085544702012, 1.69, 1.46, 1.66, 1.92, 
    1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.36158700873521, 1.61477998874887, 
    1.54, 1.86, 1.36, 1.44, 1.37, 1.40006328819621, 1.90298045970288, 
    1.95884785107792, 1.65178374495503, 0.664853959408949, 2.03, 
    0.822715247719342, 2.19, 2.14, 2.38, 2.08, 1.73165582379722, 
    2.29, 1.87, 1.68, 1.62, 2.45, 2.68508833772259, 1.66, 2.42407097546761, 
    0.42, 1.67174889669811, 1.24158068523059, 0.73, 1.87, 1.03, 
    2.09853913517633, 1.45156983698178, 1.04, 1.32027830021255, 
    2.5, 1.46, 1.87, 1.03, 2.16481341467205, 1.33406618300906, 
    1.15, 1.46, 1.3251482910031, 2.76, 1.95, 1.43, 2.95302108739567, 
    1.43339349129381, 1.62, 1.0779700406579, 1.46, 1.56, 1.34, 
    1.35, 1.38, 1.64, 1.63, 1.1452070018744, 1.99, 1.35, 3.28, 
    1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.51131843356052, 
    1.42, 2.04972977556465, 1.63, 2.08, 2.49, 0.906253256276854, 
    1.27496488709544, 1.82975779459413, 1.42685139140958, 1.04, 
    1.46917840663671, 0.786879781248338, 1.61, 0.83, 1.87, 1.83, 
    2.06936210233163, 1.71, 1.72462223783288, 2.91, 1.35, 1.77, 
    1.19049849246739, 1.9, 1.60502985224151, 0.94, 2.24370434493231, 
    0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.69967990452418, 
    0.75, 0.411115068444657, 0.55, 1.00550879158465, 1.14, 1.3, 
    0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 0.771093755937607, 
    1.32313034035266, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 0.469452363196101, 
    1.47216273639933, 0.94, 1.40987897209874, 1.15, 1.46, 0.97, 
    1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
    ), col = c(5.64, 4.38, 5.68, 7.8, 5.41666448692333, 6.75, 
    5.27306908057555, 5.05, 5.90617680506871, 7.22, 5.75, 5.62362442291607, 
    5.6, 6.40628395431922, 7.5, 4.5115907726328, 6.57964023158132, 
    5.31841023173293, 8.7, 5.1, 5.65, 4.5, 4.23745592509906, 
    3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 8.33870094186899, 
    5.35313494346557, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
    4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 4.8672433701904, 
    8.9, 7.2, 6.92474964780619, 5.50271190065568, 6.3, 5.85, 
    6.25, 5.41812336916008, 4.67958829172768, 6.8, 5.10085117046539, 
    3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 
    6.35152776462192, 3.38, 3.74, 3.48262240513945, 3.21, 3.8, 
    4.6, 4.15271138627359, 5.90704668437116, 2.30789325059446, 
    2.5, 3.9, 2.93996435989708, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 
    1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 3.26581086979034, 
    2.9, 3.41982637098018, 4.27269880592984, 4.21752216837632, 
    4.26688417922753, 2.8, 4.44202443025492, 2.94, 2.7, 3.4, 
    3.3, 2.7, 1.80152569115315, 2.9, 4.68852729284135, 2.20885368825278, 
    3.08, 2.9, 1.9, 3.69515791726314, 3.72511200435248, 3.4, 
    1.28, 3.25, 4.94189369579737, 4.886697720239, 4.11578776161636, 
    2.8, 4.64335505446782, 3.94, 0.829556849750927, 2.12, 3.97100314603665, 
    6.08175092535877, 7.50243720024758, 4.16377613139735, 5, 
    5.45, 7.1, 5.56310407952225, 5, 5.7, 4.92, 6.71175878862377, 
    4.99871764699864, 4.35, 4.4, 8.21, 4, 4.9, 6.31698938653407, 
    8.42, 9.4, 2.61778356709723, 4.83336328235962, 7.1, 10.52, 
    7.6, 7.9, 8.80556737230422, 7.5, 13, 5.43385256925178, 7.65, 
    5.78314883854544, 5.58, 6.7304025673783, 5.69435711970093, 
    6.62, 10.68, 10.26, 10.7376292076369, 8.5, 6.24949863825741, 
    5.22526017584216, 9.7, 7.7, 7.3, 6.59729350744497, 9.3, 9.2
    ), hue = c(1.04, 1.05, 1.18653799570608, 0.86, 1.07851635532796, 
    1.05, 1.02, 1.00401494220512, 1.08, 1.30133686717838, 1.25, 
    0.901861928840646, 1.15, 1.25, 1.2, 1.28, 0.868113975778523, 
    1.13, 1.23, 1.1369784071897, 1.09, 1.07063248554495, 1.11, 
    1.09, 1.12, 1.13, 0.748178295265458, 1.02, 1.09259767262384, 
    1.04, 1.19, 1.09, 0.967760426901975, 1.25, 1.1, 1.12301694375609, 
    0.97461353188498, 1.30937491993143, 1.18, 1.02454902989808, 
    0.95, 0.91, 0.88, 0.82, 1.340359750126, 0.87, 1.01951539310515, 
    1.0804991966621, 1.07, 1.12, 1.12, 1.19755969235116, 1.19114706816, 
    1.03450264880017, 1.08182257614103, 0.731338082116362, 0.94, 
    1.07, 0.89, 0.712077070215841, 1.25, 0.98, 1.23, 1.22, 1.05191239017311, 
    0.973349888728293, 1.08990449486214, 1.12, 1.02, 1.28, 0.906, 
    1.36, 0.98, 1.31, 1.29384956585943, 1.23, 1.19, 0.96, 1.06, 
    1.19, 1.38, 1.20369455686444, 1.31, 0.84, 1.18639592783144, 
    1.23, 1.16850706144605, 1.47177855410487, 1.14667745171971, 
    1.07, 1.08, 1.05, 0.82978169519157, 1.15, 1.14493646747527, 
    0.828774279908959, 0.900116585860642, 1.22611167687865, 1.06147019910668, 
    1.42, 1.27, 0.977627236001872, 0.8, 0.938154163087067, 1.04, 
    1.35073423213983, 1, 0.989965123480476, 0.86, 1.61117109373629, 
    1.00145009967228, 0.9, 1.27918484834566, 1.1, 0.93, 1.71, 
    0.95, 1.03394701606485, 0.7, 1.18223278986569, 0.8, 1.07749441762195, 
    0.92, 0.73, 1.11216514951182, 0.86, 0.69, 0.97, 0.89, 0.79, 
    0.928693772796491, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
    0.81, 0.564119178449272, 0.697101397101362, 0.619998895718054, 
    0.89, 0.624692273281044, 0.65, 0.6, 0.706571558636227, 0.54, 
    0.55, 0.162775087632952, 0.961897754561017, 0.48, 0.61, 0.56, 
    0.652589823848642, 0.6, 0.57, 0.67, 0.57, 1.08777100413648, 
    0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.507707842807023, 
    0.74, 0.67, 0.575653923199119, 0.968310226115288, 0.597401244700275, 
    0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 2.37097226479456, 
    2.93, 3.08586823159819, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
    2.9, 3.97437484133321, 3, 3.02594469946169, 2.65, 3.62630683513265, 
    2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.77046353798102, 3.2, 3.26649095873995, 
    2.77, 3.48829084220745, 2.47847609312435, 2.71, 2.88, 2.87, 
    3, 2.87, 3.60741938185245, 1.96602683673784, 3.99581089333672, 
    2.69, 3.81297028513332, 3.38, 3, 3.56439923646343, 3, 3.35, 
    3.33, 3.62223936343867, 3.33, 2.75, 3.1, 3.47775746113413, 
    3.53191826407858, 3.26, 2.93, 3.2, 3.03, 3.56800262308593, 
    2.84, 2.87, 1.82, 2.78565411276775, 1.59, 2.46, 2.87, 2.72736011519206, 
    2.3, 3.18, 3.48, 2.50999482064433, 5.01578134928968, 1.82, 
    4.01159370152856, 2.34373513730123, 3.5, 3.13, 2.14, 2.48, 
    2.52, 2.18858558341638, 2.96752145715707, 3.44831916526297, 
    3.14, 2.72, 2.01, 3.44570858235225, 3.55715197770957, 3.39492964302194, 
    3.21, 2.75, 3.21, 2.35329267534931, 2.65, 3.58432162858104, 
    3.3, 3.52095168543838, 2.63, 2.34049319805024, 2.74, 4.69457475366463, 
    2.83, 2.96, 2.77, 3.38, 2.10816521204109, 3.57, 3.3, 3.17, 
    2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.2237577640919, 3.19, 
    2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 3.21235081686808, 
    3.64, 2.88403886291925, 2.84, 3.74215472594976, 2.78, 2.68636233261629, 
    1.83740939966112, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
    1.82, 2.15, 2.20608713587711, 2.3690382870502, 2.09766688900761, 
    2.05, 2, 2.02324213512347, 1.33, 1.69980536573483, 1.62, 
    1.33, 2.58943760861184, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
    1.73, 1.96, 1.78, 1.58, 1.49525011591806, 1.75090343787422, 
    1.75, 1.68, 1.75, 2.21365113548423, 1.39344134080216, 1.36320803512792, 
    1.57199771713636, 2.25341795781455, 1.74505864071001, 2.0195928028465, 
    1.25061141251457, 1.87161691716278, 1.56, 1.62, 1.6), prol = c(1065, 
    1050, 988.729959337361, 1434.22156620598, 735, 1450, 1290, 
    1295, 1045, 1045, 916.602884130928, 1280, 1320, 1028.00094527831, 
    1547, 1310, 1280, 1518.55432951454, 1680, 845, 990.425076337591, 
    770, 1035, 1015, 845, 1124.6605626688, 1195, 1285, 915, 1035, 
    1285, 1052.29224218706, 1120.05280838501, 1235, 973.70693339587, 
    920, 880, 621.80451228818, 483.989700052922, 760, 795, 1035, 
    1095, 680, 885, 1080, 1065, 985, 1060, 1222.23388923182, 
    1150, 1265, 1190, 1078.04712663974, 1060, 1120, 970, 1270, 
    1285, 1199.63588297914, 540.914278635177, 1014.27910153751, 
    630, 459.327441930729, 523.384353601838, 493.055164290968, 
    502, 266.815194145765, 665.705811336895, 718, 870, 410, 472, 
    707.055678725443, 886, 248.176906152305, 392, 319.94826630935, 
    834.148472718268, 463, 278, 714, 699.165499984221, 515, 571.080099360633, 
    625.529272681795, 257.070480099687, 562, 680, 625, 480, 450, 
    495, 410.55278951119, 345, 481.48073069021, 625, 428, 660, 
    406, -19.132009036589, 562, 438, 415, 672, 352.628283837181, 
    510, 816.480328996016, 312, 680, 562, 325, 539.15399871624, 
    434, 385, 453.755743120505, 236.83901855307, 522.264329925042, 
    372, 564, 351.493958933693, 860.060471584961, 365, 380, 875.117210887117, 
    378, 931.558357919869, 466, 388.566691890165, 580, 678.553776864934, 
    615.236224897037, 560, 600, 894.926734181514, 695, 720, 515, 
    580, 590, 600, 646.954346266857, 520, 550, 617.849999322941, 
    830, 415, 808.785256764838, 833.055028169695, 550, 500, 480, 
    425, 675, 640, 655.501862070014, 480, 880, 660, 889.104599303437, 
    169.425863287909, 680, 570, 756.827053065876, 615, 593.433577615607, 
    695, 685, 624.207553633326, 594.356666216436, 510, 470, 1014.24061777441, 
    740, 273.021706439357, 835, 840, 598.405786883127)), row.names = c(NA, 
    -178L), class = "data.frame"), structure(list(alco = c(14.23, 
    13.2, 13.16, 14.9568684046377, 13.24, 13.6226412158977, 14.39, 
    14.06, 13.2371982647001, 13.86, 13.4910034989928, 13.4816993675913, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 13.9601066564404, 
    13.64, 13.3610717058769, 12.93, 13.71, 12.85, 13.5, 14.2066208330374, 
    13.39, 13.3, 13.87, 14.5045695954848, 13.73, 13.58, 13.68, 
    13.76, 13.4420319940588, 14.5839918978755, 13.9162635995113, 
    13.05, 11.7756009609266, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 14.3736151416925, 13.9, 14.3627992393379, 14.3147198116103, 
    13.05, 12.7242670562875, 13.82, 14.1326343856295, 13.74, 
    14.5523050771477, 14.22, 13.29, 14.5449374276579, 12.37, 
    12.33, 12.64, 11.6430262806099, 12.37, 12.17, 12.37, 12.0071893571952, 
    12.2926876025965, 13.34, 12.21, 12.1757210257516, 13.86, 
    12.506214627051, 12.99, 11.96, 11.66, 12.5140387346638, 11.84, 
    14.3399466688679, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 11.4311834758078, 
    12.16, 12.7605081989956, 12.4965302495311, 12.08, 12.08, 
    12, 12.69, 11.9856348123037, 11.62, 11.8823477038481, 11.81, 
    12.29, 13.0385913661522, 12.29, 12.08, 12.6, 13.0729699027619, 
    12.6244251304856, 12.51, 12.1053724411988, 12.3430342395681, 
    12.72, 12.22, 11.61, 12.2654364799974, 11.5657998142527, 
    11.76, 12.3701955892979, 12.08, 11.03, 11.82, 12.4837913180862, 
    12.6848638678166, 12.1512406488516, 11.45, 14.6278674855426, 
    12.42, 13.05, 11.87, 12.07, 12.1220398952205, 11.79, 13.3317119886716, 
    12.04, 12.86, 12.88, 13.0266592745799, 12.7, 12.51, 12.7736605911959, 
    12.25, 12.53, 11.9014961268096, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 13.0473886236848, 13.6384393003146, 12.87, 
    13.32, 13.5977843736461, 12.0295116135838, 12.79, 13.6928257628119, 
    13.23, 12.3446162449254, 13.17, 13.84, 12.45, 14.34, 13.48, 
    13.3061447958921, 13.69, 12.85, 12.96, 13.78, 13.73, 13.676744012068, 
    13.1896724974516, 13.58, 13.4, 12.8421544945813, 12.77, 14.16, 
    13.6667415724582, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    2.4672742039099, 1.0764637749509, 1.95, 3.85494190297049, 
    1.80648495389647, 1.87, 2.15, 1.05977423106141, 1.35, 2.16, 
    1.48, 1.44475338288088, 1.73, 1.87, 1.99036302688928, 1.92, 
    2.21925673384778, 0.804654886372151, 3.1, 1.63, 3.8, 1.90268435855934, 
    1.6, 1.81, 2.23853320215766, 1.79219724317076, 3.42173009808273, 
    1.9, 2.59635264872073, 1.5, 2.3864508539064, 1.83, 1.53, 
    1.3265500087969, 3.11660789617157, 2.83940001957675, 1.65, 
    1.5, 2.41001687586841, 1.71, 3.62352933585641, 1.7710383153964, 
    3.98, 0.856116928068804, 4.04, 2.17886530410872, 1.68, 1.99817863733487, 
    2.1294228553145, 0.947080654924481, 1.65, 0.519601633196245, 
    1.9, 1.67, 1.66546190312026, 1.7, 1.97, 3.03366654412286, 
    0.94, 1.8942555253237, 1.36, 1.25, 1.34878640727181, 3.83480518779336, 
    1.21, 1.01, 1.17, 2.87558364537685, 1.91755432742982, 2.11209748960222, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 4.19508107655999, 0.99, 
    3.87, 2.0464133087837, 2.09949935241299, 1.13, 3.44263200313617, 
    0.740357925888753, 0.98, 1.61, 1.67, 2.50630123722297, 0.641759768081541, 
    1.09326000695961, 1.51, 3.82450217008856, 1.35241735851652, 
    1.90453840598966, -0.518874164604607, 2.79578401932674, 1.41, 
    0.785394911097515, -0.387945592862604, 2.08, 1.81827923427472, 
    2.1423547891201, 1.72, 1.10250388105456, 2.55, 1.73, 1.94961619857784, 
    1.29, 0.591881868781844, 3.74, 1.91258446120554, 2.68, 0.74, 
    1.39, -0.114037292362701, 1.50036361603838, 1.61, 3.43, 2.2843629704277, 
    2.4, 2.05, 4.43, 4.28706669111231, 4.31, 2.16, 1.53, 2.13, 
    3.53469184060579, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 4.48068428711217, 
    2.77335204558641, 5.51, 3.59, 4.0363708528226, 2.81, 2.56, 
    2.68087486378542, 1.74895181991751, 3.88, 3.28308727762457, 
    5.04, 4.61, 3.24, 2.99647865221536, 3.12, 2.67, 1.9, 2.96065684773236, 
    1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 3.33692562625014, 
    2.34542411510196, 3.45, 2.76, 1.49860233459634, 3.26174993534967, 
    3.37, 2.58, 4.6, 3.87579758166711, 2.39, 4.66956811558408, 
    5.65, 3.91, 4.10599389687047, 2.59, 4.1), ash = c(2.43, 2.14, 
    2.67, 2.5, 2.43248425958536, 2.56346818831666, 2.45, 2.61, 
    2.15788319107017, 2.20475943309043, 2.3, 2.11708500598486, 
    2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.09204225631177, 
    2.65, 2.46277861710195, 2.52, 2.61, 3.22, 2.66915676160639, 
    3.04177364521413, 2.87390223681036, 2.64908406900609, 2.47514870266218, 
    2.36, 2.72069831265569, 2.7, 2.65, 3.03531664016734, 2.43974570340901, 
    2.55, 2.1, 2.51, 1.95643077588016, 2.12, 2.59, 2.29, 2.1, 
    2.44, 2.44010579937741, 2.12, 2.4, 2.27, 2.04, 2.15097890788744, 
    2.19391357392572, 2.72402611606692, 2.18210165666153, 2.41348125437965, 
    2.3, 2.68, 2.5, 1.36, 2.06880486792544, 2.17564407193612, 
    1.92, 2.16, 2.53, 2.56, 1.66995854862809, 1.92, 2.21067858526891, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.72447127192463, 1.71, 
    2.17972690557457, 2.46744315957352, 2.4, 2, 2.21471038994475, 
    2.51, 2.32, 2.58, 2.24, 2.42292233260495, 2.62, 2.17337135921802, 
    2.3, 2.32, 1.88722619516077, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.10197012543061, 2.32731086722778, 1.7, 2.04788051425028, 
    2.46, 1.88, 2.29378020509231, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.09439424581723, 
    2.61953109731385, 2, 2.42, 3.23, 2.50572177566581, 2.13, 
    2.39, 2.17, 2.29, 2.72789001442636, 2.63724440316606, 2.38, 
    2.46989139902093, 2.4, 2.4, 2.36, 2.25, 2.38703973506705, 
    2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.38530351336311, 
    2.42165673803391, 2.48, 2.32785316022074, 2.36, 2.62, 2.48, 
    2.37197279625505, 2.28, 2.51765699671336, 2.32, 2.38, 1.84698061955124, 
    2.09906114447661, 2.64, 2.38, 2.54, 2.46362663975013, 2.35, 
    2.3, 2.35908000534391, 2.6, 1.93079546879682, 2.69, 2.52977139671279, 
    2.32, 2.28, 2.14855564672043, 2.5570281356744, 2.48, 2.26, 
    2.37, 2.74), alca = c(15.6, 15.4456312986881, 18.1935938619311, 
    16.8, 21, 15.2, 14.8079089682151, 17.6, 14, 16, 15.136490912437, 
    16.8, 16, 11.4, 13.5793871137278, 15.5772144810994, 19.0500246722209, 
    20, 12.1556830030571, 15.2, 16, 18.6, 21.4606629459754, 17.8, 
    20, 15.9260663447563, 16.1, 17, 16.9915765105738, 15.8239863520503, 
    19.0760619811241, 19.1, 17.7570779276265, 17.1932752091793, 
    19, 20.5, 15.5, 21.4886679251091, 15.5, 13.2, 16.2, 18.8, 
    18.4632850736021, 12.7508129079122, 17, 20.1044185958599, 
    16, 12.2836778336497, 18.8, 17.4, 12.4, 18.9012641589642, 
    9.54778539460271, 15.0763398675191, 11.7301665244757, 20.5, 
    16.3, 16.8, 17.6145545136834, 10.6, 17.1422463432317, 16.8, 
    18, 19, 19, 18.1, 18.7528599945385, 19.6, 19.5816568128138, 
    16.8, 20.4, 17.3428786055493, 24, 30, 21, 19.1696710483584, 
    16, 18, 14.8, 16.3879635325499, 19, 23.8439894338856, 24, 
    22.5, 18, 18, 22.8, 23.5191616336353, 21.6, 19.6306813101455, 
    21.6309897469514, 22, 19.3643566807463, 18, 18, 19, 21.5, 
    16, 16.5911318570263, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
    19, 22.5, 19, 20, 19.5, 14.7667870439852, 24.41501560276, 
    19.2867520617929, 19.1501604423105, 21.5, 20.8, 26.7935394701756, 
    16, 21.3056653635983, 22.0225738491505, 21.3645533857454, 
    22.2666344745347, 21.5, 21, 21, 21.5, 28.5, 25.5675033552111, 
    22, 16.5581284471585, 20, 24, 21.5, 18.1207113480793, 21.2927919416775, 
    16.8442464839247, 25, 19.5, 19.4061382532648, 21, 23.076849986421, 
    23.5, 20, 23.9252511275768, 21, 20, 21.6057489678558, 18.1472411054493, 
    21.5, 24, 22, 19.7705554085871, 18.5, 20, 22, 19.5, 16.0408390704652, 
    25, 21.6130077833659, 23.1703057096318, 21.1691087817376, 
    20.229428556728, 18.5, 21.8226470487109, 22.5, 23, 19.5, 
    24.5, 25, 18.8762376246083, 19.5, 20.396338068217, 20.5, 
    23, 20, 26.3813306039989, 24.5), mg = c(127, 100, 101, 113, 
    104.044875265013, 112, 96, 121, 97, 116.790797950125, 121.43368860763, 
    110.331834397978, 89, 91, 102, 112, 120, 105.408365030123, 
    103.763317558541, 116, 126, 102, 113.48359494521, 95, 99.324338571199, 
    124, 93, 111.631821222508, 107, 96, 101, 106, 104, 112.305577072328, 
    110, 122.657182794977, 110, 74.3703396881892, 93.3221994732647, 
    115.65985429979, 117, 117.434148970416, 101, 127.378016173618, 
    107, 111, 102, 101, 126.938474539045, 108, 92, 109.073895347063, 
    111, 115, 110.550212708879, 124.932080971033, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 88.1800617632134, 
    151, 93.3859013651239, 86, 87, 139, 101, 111.583527843362, 
    81.6706512865748, 112, 117.945894689874, 96.813830029383, 
    86, 84.9131228622911, 78, 85, 89.9452810828145, 99, 90, 88, 
    84, 70, 81, 93.240599038086, 108.139631649814, 88, 114.097954732832, 
    84.0789575167124, 134, 85, 88, 75.0400308985315, 97, 88, 
    81.3376176851549, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 
    88, 84, 85, 86, 69.2880017906307, 116.910201053243, 87, 96, 
    107.052731093148, 102, 86, 83.5371093563204, 85, 62.4868017547856, 
    92, 88, 80, 122, 107.28293778299, 98, 106, 93.1636476859916, 
    94, 89, 124.827764193088, 88, 101, 105.972700403143, 89, 
    97, 79.5744113367426, 112, 102, 103.89904962149, 118.24131141604, 
    92, 113, 123, 107.797686569391, 116, 98, 103, 115.450622415334, 
    89, 97, 98, 89, 101.166634020374, 120.126582023685, 106, 
    106, 90, 88, 109.817886504698, 88, 105, 112, 96, 86, 91, 
    95, 102, 113.726544829649, 120, 121.045584160717), phe = c(2.8, 
    2.65, 2.8, 3.85, 2.8, 2.70239208344512, 2.5, 2.6, 2.50773786036635, 
    2.98, 2.95, 2.2, 2.6, 3.1, 3.04248749939535, 2.85, 2.8, 2.86559959333792, 
    3.3, 3.09045373743236, 3, 2.41, 2.61, 2.48, 2.53, 3.04958572720952, 
    2.59734575541072, 2.4, 2.95, 2.65, 3, 2.88781532316068, 2.42, 
    2.95, 1.91025247775283, 2.7, 2.6, 2.45, 3.18214950728491, 
    3.4413305517154, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 
    2.75, 2.88, 2.72, 2.45, 3.88, 2.90348505424494, 3.23466854530932, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.59578016275778, 3.5, 
    1.89, 3.14420453212297, 2.78853122361593, 2.11, 2.53, 4.0684911596927, 
    1.1, 2.95, 1.88, 1.83809384105792, 2.11240649240397, 2.52171862479197, 
    2.17938848583204, 2.26527130773244, 1.9, 2.83, 2.72640100301705, 
    2.2, 1.7632713903957, 1.65, 2.25633829794321, 2.2, 2.05485300390733, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.32544706305691, 
    1.68201152874828, 2.55, 3.52, 2.91790701897531, 2.6637895198859, 
    1.86024684087685, 2.56, 2.5, 2.2, 1.65703512564318, 1.65, 
    2.30782157304231, 2.36, 2.74, 3.18, 2.55, 1.75, 2.41956783172221, 
    2.56, 2.46, 1.83153416196934, 2, 1.8961946974221, 2, 2.9, 
    3.18, 3.01227842171619, 2.19513728224075, 2.86, 2.94545916763035, 
    2.74, 2.13, 2.22, 2.1, 1.51, 2.16233105430978, 1.61014099548088, 
    1.7, 1.13859467605562, 1.62, 1.38, 0.743132183457506, 1.62, 
    1.53037410996538, 1.36213793114421, 1.4, 1.08866402657785, 
    2, 1.38, 1.1298704850567, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 1.3895504694166, 1.48, 1.412944314619, 1.8, 1.9, 1.56692107982976, 
    2.12294042779684, 2.3, 1.83, 0.981897959205869, 1.39, 1.35, 
    1.0407631928759, 1.7, 1.48, 1.55, 1.62309938939402, 1.25, 
    1.39, 1.68, 1.55612919683777, 2.00835572627832, 1.58606185829347, 
    1.65, 2.05), fla = c(2.9445633940337, 2.76, 3.90130933867654, 
    3.49, 1.75567877242181, 3.39, 2.52, 2.51, 2.98, 3.1759623385283, 
    3.32, 2.43, 2.76, 2.99640953070676, 3.64, 2.91, 3.14, 3.4, 
    3.55039623036965, 3.41367278685594, 3.30562883713118, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.88033498349657, 2.97, 2.33, 
    3.25, 3.19, 2.69, 3.12851395931135, 1.88885730795031, 2.98, 
    2.63444008823736, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.04748859930034, 2.92, 3.54, 2.90564143474993, 
    2.99, 3.74, 2.79, 2.9, 2.78, 2.73644742601475, 3.23, 3.67, 
    0.57, 1.64037330431557, 1.41, 1.79, 3.0687003689237, 1.75, 
    2.65, 2.84493259104951, 2, 1.3, 3.77832008456696, 1.02, 2.86, 
    1.35216414839206, 2.89, 2.14, 1.57, 1.35100895721754, 1.69652025967715, 
    1.85, 2.34275001619933, 2.26, 2.53, 1.58, 1.59, 2.21, 1.87192423872195, 
    1.69, 1.61, 1.69, 1.75925170887467, 1.37505331519669, 1.25, 
    1.46, 2.36408847836472, 2.62791465287782, 2.27, 0.99, 1.94825185836631, 
    3.07185677351251, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 1.97507059928384, 2.92, 2.46940309468717, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.1742376848669, 1.64, 
    2.26050604570646, 3.31109683464581, 2.47834644982484, 1.67743022280597, 
    3.03, 2.65, 3.15, 2.24, 1.9805646306265, 1.75, 1.33368803071379, 
    1.20665352921188, 1.04777509067338, 1.2, 0.58, 0.66, 0.47, 
    -0.180645910417233, 0.616094028355583, 0.6, 0.5, 1.04906141891479, 
    0.52, 0.8, 0.985242493345039, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.569577868242152, 0.58, 
    1.31, 1.1, 0.92, 0.553296610822639, 0.6, 0.918972205985303, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.570497581132647, 
    0.7, 0.61, 1.04160790105131, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.394825386747141, 0.3, 0.24, 0.39, 0.0550801598990394, 0.296639440288843, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.257844100011294, 
    0.28716615604631, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.426359959040341, 
    0.432740752365428, 0.34, 0.27, 0.397977121315197, 0.26, 0.29, 
    0.415528618859298, 0.42, 0.193076360155522, 0.29, 0.26, 0.203741467280669, 
    0.306996775508834, 0.28, 0.2, 0.34, 0.312183803490606, 0.28888431078033, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.182940909091632, 0.256589715804764, 
    0.28, 0.479589539764778, 0.53, 0.32, 0.46092002706093, 0.45, 
    0.37, 0.26, 0.27, 0.418091048633603, 0.0229443686270767, 
    0.274999760498916, 0.487164442058443, 0.440314074652892, 
    0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.371374680907366, 
    0.4, 0.61, 0.22, 0.3, 0.469273346640076, 0.445646400861306, 
    0.354979895817128, 0.277124335480905, 0.422886340856397, 
    0.5, 0.246940477633354, 0.25, 0.17, 0.32, 0.50126376748246, 
    0.219006761724794, 0.24, 0.433903028717232, 0.316215529460321, 
    0.29, 0.34, 0.37, 0.327874236970759, 0.66, 0.230200570978142, 
    0.48, 0.39, 0.29, 0.24, 0.26, 0.325012071590623, 0.42, 0.351941511857941, 
    0.52, 0.3, 0.484136824183702, 0.376929148589891, 0.402464187208146, 
    0.32, 0.266275626694634, 0.43, 0.3, 0.332875131067683, 0.240285419500056, 
    0.39, 0.212739484447106, 0.4, 0.515624417377899, 0.266370601072956, 
    0.450552612361953, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 
    0.53, 0.53, 0.360014882435373, 0.546162429665357, 0.424679940682907, 
    0.601233477642914, 0.43, 0.4, 0.47, 0.45, 0.331476451585898, 
    0.22, 0.189010217196053, 0.26, 0.61, 0.53, 0.514751769762967, 
    0.394899019995958, 0.63, 0.53, 0.52, 0.404365620018757, 0.5, 
    0.6, 0.4, 0.41, 0.348246419208772, 0.43, 0.4, 0.39, 0.465527266895732, 
    0.522769326338951, 0.48, 0.560851617057326, 0.50429338008094, 
    0.43, 0.43, 0.53, 0.550497345501697), pro = c(2.29, 1.28, 
    2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 
    1.81, 2.54204810285957, 2.96, 1.46, 1.77884208141676, 2.18596957814361, 
    1.86, 1.66, 2.1, 0.408307844817833, 1.69, 1.46, 1.66, 1.92, 
    1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 2.11571899012007, 2.15553892325657, 
    1.54, 1.86, 1.36, 1.44, 1.37, 1.36963500258399, 2.16679631580652, 
    1.71795898900498, 1.68045145109745, 1.20671024827363, 2.03, 
    0.342505387684921, 2.19, 2.14, 2.38, 2.08, 1.37531097383955, 
    2.29, 1.87, 1.68, 1.62, 2.45, 2.23651580688627, 1.66, 2.15222358513532, 
    0.42, 2.10545242560285, 1.92861067994918, 0.73, 1.87, 1.03, 
    1.84966904413923, 1.80231491016414, 1.04, 1.11883820259158, 
    2.5, 1.46, 1.87, 1.03, 2.78293238671162, 1.64340952760309, 
    1.15, 1.46, 1.81090502871308, 2.76, 1.95, 1.43, 2.23985130681176, 
    1.4857891525768, 1.62, 1.70519643915952, 1.46, 1.56, 1.34, 
    1.35, 1.38, 1.64, 1.63, 1.39401141680175, 1.99, 1.35, 3.28, 
    1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.40283449084002, 
    1.42, 2.22354199665261, 1.63, 2.08, 2.49, 1.02516635289823, 
    1.38057194833224, 1.94054737492544, 1.32936293011493, 1.04, 
    2.12284850992631, 1.17245932160338, 1.61, 0.83, 1.87, 1.83, 
    2.20958087738781, 1.71, 0.590779920473697, 2.91, 1.35, 1.77, 
    1.40091635996117, 1.9, 1.43485335241775, 0.94, 2.3782737920522, 
    0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.88646844337037, 
    0.75, 2.02337468634347, 0.55, 0.966084509150401, 1.14, 1.3, 
    0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.6199630483723, 
    2.11590139332699, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.85551862651578, 
    1.03069465814106, 0.94, 1.57378528024675, 1.15, 1.46, 0.97, 
    1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
    ), col = c(5.64, 4.38, 5.68, 7.8, 6.18793598522207, 6.75, 
    4.71417859202784, 5.05, 6.78340796089165, 7.22, 5.75, 6.09253573247964, 
    5.6, 6.53903021623318, 7.5, 5.59437673076652, 8.28767853012952, 
    4.38870231462109, 8.7, 5.1, 5.65, 4.5, 4.3671922812881, 3.93, 
    3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.65514113955794, 4.92545063721602, 
    5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 
    5.04, 5.24, 4.9, 6.1, 7.59956334733661, 8.9, 7.2, 4.69482771735648, 
    8.65346302839256, 6.3, 5.85, 6.25, 5.87169865924662, 4.58912062164465, 
    6.8, 6.10077839128428, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 
    5.3, 4.68, 3.17, 2.85, 4.02840741647166, 3.38, 3.74, 3.99573445329424, 
    3.21, 3.8, 4.6, 5.67912687096895, 7.98579063121141, 6.21479599298405, 
    2.5, 3.9, 2.40014843010042, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 
    1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 6.17127269405172, 
    2.9, 6.61234396052806, 5.13444237412176, 4.22800846751738, 
    3.54459628016309, 2.8, 2.99720379294108, 2.94, 2.7, 3.4, 
    3.3, 2.7, 1.74155339549892, 2.9, 2.29286563613589, 0.861870016326128, 
    3.08, 2.9, 1.9, 2.00893772178779, 3.73877979964638, 3.4, 
    1.28, 3.25, 3.8604533461318, 5.29866870073037, 4.03553131791572, 
    2.8, 2.08476302181165, 3.94, 0.683748827455972, 2.12, 3.51172458967993, 
    4.43004575850686, 7.38853001763099, 5.891470474294, 5, 5.45, 
    7.1, 7.95015424631651, 5, 5.7, 4.92, 9.27687068215506, 2.83232552008367, 
    4.35, 4.4, 8.21, 4, 4.9, 7.13938738300824, 8.42, 9.4, 3.66953978954028, 
    5.7319333458589, 7.1, 10.52, 7.6, 7.9, 7.58439176954998, 
    7.5, 13, 8.40445333315588, 7.65, 8.87198491946051, 5.58, 
    10.4103898589807, 5.02037401731167, 6.62, 10.68, 10.26, 2.82453194726984, 
    8.5, 7.16005744786557, 1.3082842599897, 9.7, 7.7, 7.3, 8.94246550815124, 
    9.3, 9.2), hue = c(1.04, 1.05, 1.09921919729946, 0.86, 1.05645287322363, 
    1.05, 1.02, 0.943556036027713, 1.08, 1.28225422377109, 1.25, 
    0.677334988197824, 1.15, 1.25, 1.2, 1.28, 1.06874405155959, 
    1.13, 1.23, 1.02978300359398, 1.09, 0.998350474289009, 1.11, 
    1.09, 1.12, 1.13, 0.90180458535976, 1.02, 0.92902459174925, 
    1.04, 1.19, 1.09, 1.20801630215092, 1.25, 1.1, 0.58826125190221, 
    0.823755259987624, 0.721948211358631, 1.18, 1.0594504059179, 
    0.95, 0.91, 0.88, 0.82, 1.49870804672287, 0.87, 0.930133472940036, 
    1.4810503477789, 1.07, 1.12, 1.12, 1.10291708715459, 1.51352798568133, 
    1.1459232516866, 1.21966238749831, 1.21650618092588, 0.94, 
    1.07, 0.89, 0.962426859138874, 1.25, 0.98, 1.23, 1.22, 0.911321279005044, 
    1.27754237542301, 1.33127631512934, 1.12, 1.02, 1.28, 0.906, 
    1.36, 0.98, 1.31, 1.05563772261533, 1.23, 1.19, 0.96, 1.06, 
    1.19, 1.38, 0.88019339361827, 1.31, 0.84, 1.09898441987893, 
    1.23, 0.957333719938097, 0.905997772308126, 0.836125923250533, 
    1.07, 1.08, 1.05, 0.829578136574825, 1.15, 1.1775223899064, 
    1.25847517764108, 0.733740428932106, 0.921831252425614, 0.99508464540096, 
    1.42, 1.27, 0.941630136432837, 0.8, 1.18571104797037, 1.04, 
    0.857859433071482, 1, 1.00447068473159, 0.86, 1.26099922597159, 
    0.936191464224242, 0.9, 0.936992741971988, 1.1, 0.93, 1.71, 
    0.95, 0.862956733635551, 0.7, 1.10516007682724, 0.8, 1.0137091638522, 
    0.92, 0.73, 0.713910344145762, 0.86, 0.69, 0.97, 0.89, 0.79, 
    0.989830469433018, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
    0.81, 0.839466053555596, 0.567093689745516, 0.874272473800819, 
    0.89, 0.913060538584804, 0.65, 0.6, 0.631943166511348, 0.54, 
    0.55, 0.748184370202624, 0.645288466432499, 0.48, 0.61, 0.56, 
    0.944899826531366, 0.6, 0.57, 0.67, 0.57, 0.952400972976138, 
    0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.458211216833643, 
    0.74, 0.67, 0.541451505084179, 0.866016509583565, 0.456353341957358, 
    0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 2.64476920668173, 
    2.93, 3.06407012529038, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
    2.9, 3.11472490544195, 3, 3.01792406153864, 2.65, 3.84503688478327, 
    2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.4703193494886, 3.2, 3.06904406001535, 
    2.77, 3.16829772719212, 2.457899444906, 2.71, 2.88, 2.87, 
    3, 2.87, 3.24716842074086, 3.47193962028923, 2.93433618972084, 
    2.69, 3.69260026571562, 3.38, 3, 3.61175611783536, 3, 3.35, 
    3.33, 3.37122441121468, 3.33, 2.75, 3.1, 2.48659128201015, 
    2.72766483427577, 3.26, 2.93, 3.2, 3.03, 2.53824694528229, 
    2.84, 2.87, 1.82, 2.84920166668275, 1.59, 2.46, 2.87, 2.31601448276194, 
    2.3, 3.18, 3.48, 2.50023969142623, 3.43542647884002, 1.82, 
    3.97070681748423, 2.5052013946465, 3.5, 3.13, 2.14, 2.48, 
    2.52, 1.66994478265216, 3.56352779670641, 3.27175984752155, 
    3.14, 2.72, 2.01, 2.87703410806824, 2.75798043485582, 2.53347091434204, 
    3.21, 2.75, 3.21, 2.76108918414243, 2.65, 3.1386308575743, 
    3.3, 2.91942159459802, 2.63, 2.09632903215758, 2.74, 2.66045784692, 
    2.83, 2.96, 2.77, 3.38, 3.02532114493257, 3.57, 3.3, 3.17, 
    2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.41566159359668, 3.19, 
    2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.29458250771747, 
    3.64, 3.24741986054872, 2.84, 3.17146084201527, 2.78, 3.33875146312842, 
    2.40184597167144, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
    1.82, 2.15, 1.52719644269701, 2.67417496182192, 2.67774284694935, 
    2.05, 2, 1.31664592005128, 1.33, 1.68843774182545, 1.62, 
    1.33, 2.13251740749054, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
    1.73, 1.96, 1.78, 1.58, 1.55716161095188, 1.79519010814113, 
    1.75, 1.68, 1.75, 1.43893998592415, 1.37125969910099, 2.5965296984499, 
    1.95656134601439, 2.53156725525281, 3.3669043224609, 1.7830683724508, 
    1.48570827332937, 2.42095785898871, 1.56, 1.62, 1.6), prol = c(1065, 
    1050, 1421.91002393981, 720.621547039354, 735, 1450, 1290, 
    1295, 1045, 1045, 1246.89566948847, 1280, 1320, 1096.43963205938, 
    1547, 1310, 1280, 1076.41055621589, 1680, 845, 1250.89422610346, 
    770, 1035, 1015, 845, 993.189039917919, 1195, 1285, 915, 
    1035, 1285, 1053.21346688848, 1160.99658786881, 1235, 1145.22839237877, 
    920, 880, 996.729993657793, 431.957943437386, 760, 795, 1035, 
    1095, 680, 885, 1080, 1065, 985, 1060, 1279.80761324394, 
    1150, 1265, 1190, 1082.22007010603, 1060, 1120, 970, 1270, 
    1285, 204.687819555606, 194.663746830677, 305.922542964159, 
    630, 469.469890514292, 707.69649619559, 592.223768295279, 
    502, 184.76837549361, 4.29378707820467, 718, 870, 410, 472, 
    1375.85031224156, 886, 248.831250473369, 392, 558.565140173787, 
    891.44530551743, 463, 278, 714, 641.152115025497, 515, 335.509013183328, 
    612.619841097281, 539.427091150568, 562, 680, 625, 480, 450, 
    495, 804.057850675844, 345, 772.468832996454, 625, 428, 660, 
    406, 332.893565896835, 562, 438, 415, 672, 323.35654271606, 
    510, 495.407443815905, 312, 680, 562, 325, 980.231281362339, 
    434, 385, 442.421031818022, 608.44046037453, 394.233423963727, 
    372, 564, 383.490152853063, 1215.60730682787, 365, 380, 256.299774415905, 
    378, 639.546508980418, 466, 576.689170608852, 580, 1261.45675074629, 
    669.378828926338, 560, 600, 582.280630505166, 695, 720, 515, 
    580, 590, 600, 364.537780529936, 520, 550, 587.324214763771, 
    830, 415, 792.584918885926, 554.558769549002, 550, 500, 480, 
    425, 675, 640, 357.769881970898, 480, 880, 660, 679.196138720703, 
    467.703076517658, 680, 570, 1031.38544694445, 615, 665.585663047416, 
    695, 685, 257.202422467647, 884.067090268586, 510, 470, 550.908678107949, 
    740, 217.656733967041, 835, 840, 574.128955528658)), row.names = c(NA, 
    -178L), class = "data.frame"), structure(list(alco = c(14.23, 
    13.2, 13.16, 14.967065017825, 13.24, 14.0530705410562, 14.39, 
    14.06, 14.075030553904, 13.86, 13.8658350395638, 14.0265839375164, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 14.4889349941546, 
    13.64, 14.4003759966083, 12.93, 13.71, 12.85, 13.5, 13.8390505249049, 
    13.39, 13.3, 13.87, 14.1372889852918, 13.73, 13.58, 13.68, 
    13.76, 14.4247270828903, 13.4620001373468, 13.9685893328252, 
    13.05, 12.6485381846133, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 13.1541917954047, 13.9, 14.2164482152764, 13.7803455639669, 
    13.05, 14.329002827246, 13.82, 14.0102549539707, 13.74, 14.1593642817302, 
    14.22, 13.29, 13.6415079512178, 12.37, 12.33, 12.64, 11.3536857768222, 
    12.37, 12.17, 12.37, 13.6109321745699, 12.4198632301306, 
    13.34, 12.21, 12.9992307795231, 13.86, 11.7040725749415, 
    12.99, 11.96, 11.66, 11.4646965280485, 11.84, 14.2339263996282, 
    12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.1000730029535, 12.16, 
    12.8414150296321, 12.4107180541926, 12.08, 12.08, 12, 12.69, 
    12.1657854275141, 11.62, 12.3342459322625, 11.81, 12.29, 
    12.0772552041824, 12.29, 12.08, 12.6, 12.6011191644067, 12.4338896272592, 
    12.51, 12.5947482433427, 12.4362159808393, 12.72, 12.22, 
    11.61, 10.9622578956108, 10.9817699307621, 11.76, 11.9594419675857, 
    12.08, 11.03, 11.82, 12.9821883838267, 12.5376900856667, 
    12.7326410617991, 11.45, 13.8244783763399, 12.42, 13.05, 
    11.87, 12.07, 11.9544628411773, 11.79, 12.4923154948712, 
    12.04, 12.86, 12.88, 13.3301252948682, 12.7, 12.51, 12.9281265581493, 
    12.25, 12.53, 12.8755366021086, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 14.2083590266192, 13.4734339836844, 12.87, 
    13.32, 11.9932256967724, 12.8003908695213, 12.79, 13.2964859795147, 
    13.23, 13.9419899424805, 13.17, 13.84, 12.45, 14.34, 13.48, 
    13.2377703093398, 13.69, 12.85, 12.96, 13.78, 13.73, 13.4175535771481, 
    13.4656783760255, 13.58, 13.4, 12.306151691985, 12.77, 14.16, 
    12.9356286401801, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    2.15129782900562, 0.482895375782752, 1.95, 3.23272108720891, 
    1.77201572579836, 1.87, 2.15, 1.57047528048074, 1.35, 2.16, 
    1.48, 1.67416398449412, 1.73, 1.87, 2.07863076349216, 1.92, 
    0.732040031299252, 2.41405659079162, 3.1, 1.63, 3.8, 2.3055070249786, 
    1.6, 1.81, 3.22296836616448, 1.73834113510998, 3.52160746599111, 
    1.9, 1.82271383309753, 1.5, 1.82758847591334, 1.83, 1.53, 
    1.83242541343076, 2.17133799435134, 3.73706594397505, 1.65, 
    1.5, 1.55631777474895, 1.71, 2.55603344004071, 1.56527539882343, 
    3.98, 1.10589809723709, 4.04, 0.794345772008427, 1.68, 0.248692048509811, 
    0.477706301880594, 0.776303230267047, 1.65, 2.01730445136446, 
    1.9, 1.67, 0.356109391146169, 1.7, 1.97, 1.9997918736938, 
    0.94, 2.26342258936944, 1.36, 1.25, -0.971356344803635, 3.44962153612446, 
    1.21, 1.01, 1.17, 1.27370480128413, -0.12924416427924, 2.03763218974377, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 2.60704626362802, 0.99, 
    3.87, 0.551589217961705, 2.78004859084283, 1.13, 3.09979633169139, 
    1.24595213425214, 0.98, 1.61, 1.67, 2.17416863975771, 2.24945308553776, 
    1.21708687646495, 1.51, 2.96977324464861, 1.09769312141527, 
    0.376550907361444, -0.79480982070008, 3.84573741301969, 1.41, 
    1.57120580326285, 0.204830772545911, 2.08, 0.644481831221683, 
    1.87218016676827, 1.72, 3.13268134537294, 2.55, 1.73, 3.0619841439171, 
    1.29, 0.495153377359808, 3.74, 1.34437143969458, 2.68, 0.74, 
    1.39, 0.497362473284802, 1.72711094324209, 1.61, 3.43, 3.44382426340449, 
    2.4, 2.05, 4.43, 3.83873123811496, 4.31, 2.16, 1.53, 2.13, 
    1.4264424544941, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 4.35143851820264, 
    4.29768321793988, 5.51, 3.59, 3.78541852871306, 2.81, 2.56, 
    4.54787509653522, 2.34859571548941, 3.88, 3.79342013697534, 
    5.04, 4.61, 3.24, 4.34045252824399, 3.12, 2.67, 1.9, 3.31311988208822, 
    1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.95359213651006, 
    3.63443645806031, 3.45, 2.76, 1.77320985433112, 2.63256658337369, 
    3.37, 2.58, 4.6, 4.2345095152556, 2.39, 4.87056382836909, 
    5.65, 3.91, 3.78546852874935, 2.59, 4.1), ash = c(2.43, 2.14, 
    2.67, 2.5, 2.88922926900899, 2.81499505541237, 2.45, 2.61, 
    2.56294578176571, 2.47145749116204, 2.3, 2.57563320479658, 
    2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.41562703394304, 
    2.65, 2.86693070377289, 2.52, 2.61, 3.22, 2.75129262058822, 
    2.47430498440746, 2.95021233667056, 2.21742004927264, 2.52336402727063, 
    2.36, 2.05114269073087, 2.7, 2.65, 2.48241052429753, 2.78003884074743, 
    2.55, 2.1, 2.51, 2.03817773311047, 2.12, 2.59, 2.29, 2.1, 
    2.44, 2.22519309813497, 2.12, 2.4, 2.27, 2.04, 3.13504080367337, 
    2.44305505456448, 3.02879327286139, 2.54285490416986, 2.39393870714086, 
    2.3, 2.68, 2.5, 1.36, 2.29093613001459, 2.16146013983886, 
    1.92, 2.16, 2.53, 2.56, 1.92634832369576, 1.92, 2.06549127880937, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.12026137595695, 1.71, 
    2.2971516911142, 2.55974714252338, 2.4, 2, 2.535933091314, 
    2.51, 2.32, 2.58, 2.24, 2.50863013064009, 2.62, 2.42435334908638, 
    2.3, 2.32, 2.13979727101823, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.24414896012024, 1.96122040236727, 1.7, 1.83165131380317, 
    2.46, 1.88, 2.12483719398209, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.3859005455342, 2.01949182555246, 
    2, 2.42, 3.23, 2.29128336539254, 2.13, 2.39, 2.17, 2.29, 
    2.06471094921396, 2.75753683052806, 2.38, 2.28467707085726, 
    2.4, 2.4, 2.36, 2.25, 2.70985467890998, 2.54, 2.64, 2.19, 
    2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.76863379265843, 2.4615436185058, 
    2.48, 2.35479012529694, 2.36, 2.62, 2.48, 1.97935516809913, 
    2.28, 2.22270793540466, 2.32, 2.38, 1.82325078641043, 2.00744982098613, 
    2.64, 2.38, 2.54, 2.37063857438874, 2.35, 2.3, 2.26758452866063, 
    2.6, 2.23504249149127, 2.69, 2.63286076355996, 2.32, 2.28, 
    2.09600351467713, 2.19228970474602, 2.48, 2.26, 2.37, 2.74
    ), alca = c(15.6, 16.7366971097538, 22.1870089903755, 16.8, 
    21, 15.2, 19.1212665805281, 17.6, 14, 16, 16.2358764950034, 
    16.8, 16, 11.4, 13.5014493235298, 18.3901532118641, 19.129453062118, 
    20, 13.2719582810559, 15.2, 16, 18.6, 20.368675926956, 17.8, 
    20, 17.6432532690602, 16.1, 17, 19.7264979836954, 12.337568212044, 
    16.0736473591285, 19.1, 16.0321179716015, 18.9686472821074, 
    19, 20.5, 15.5, 21.641263191031, 15.5, 13.2, 16.2, 18.8, 
    12.1967498417875, 17.7103236142841, 17, 14.6097849016928, 
    16, 15.2322671936164, 18.8, 17.4, 12.4, 18.4073223642072, 
    16.2676798699447, 18.2624340440409, 16.4693469461353, 20.5, 
    16.3, 16.8, 16.3390709109094, 10.6, 26.1155560008261, 16.8, 
    18, 19, 19, 18.1, 18.9697389757986, 19.6, 22.3754128245817, 
    16.8, 20.4, 23.2269607554943, 24, 30, 21, 16.5721776365135, 
    16, 18, 14.8, 24.6755028441709, 19, 23.1809998246816, 24, 
    22.5, 18, 18, 22.8, 20.4252512045841, 21.6, 22.4039208837456, 
    28.3764501461796, 22, 22.7030965839355, 18, 18, 19, 21.5, 
    16, 17.1479324107122, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
    19, 22.5, 19, 20, 19.5, 17.2612495921808, 24.5662736224789, 
    17.2858952403486, 19.1304178538434, 21.5, 20.8, 19.8984740252876, 
    16, 21.682882207548, 18.0884103712079, 15.6772798535883, 
    25.7613986198137, 21.5, 21, 21, 21.5, 28.5, 21.1268367049788, 
    22, 17.8172332079151, 20, 24, 21.5, 20.6977524565126, 22.0198807845379, 
    21.4501856066443, 25, 19.5, 20.9791877836126, 21, 20.173169879903, 
    23.5, 20, 24.8927731008763, 21, 20, 17.9890529745628, 16.6557926770419, 
    21.5, 24, 22, 20.7496061198136, 18.5, 20, 22, 19.5, 9.04899747534266, 
    25, 20.9924500826584, 19.6254420452733, 21.5402558477936, 
    23.7512921480044, 18.5, 21.0855275597822, 22.5, 23, 19.5, 
    24.5, 25, 19.3306971737705, 19.5, 19.7939100040713, 20.5, 
    23, 20, 24.8200717140759, 24.5), mg = c(127, 100, 101, 113, 
    83.5638156879111, 112, 96, 121, 97, 130.378687564694, 105.288671153791, 
    105.646026080177, 89, 91, 102, 112, 120, 103.89185551096, 
    114.428720482484, 116, 126, 102, 114.768095445456, 95, 94.7105328346632, 
    124, 93, 121.190426751524, 107, 96, 101, 106, 104, 110.088433052578, 
    110, 112.617378443734, 110, 84.5676660450385, 106.542380781225, 
    100.113689189938, 117, 107.065548204537, 101, 99.4152919850139, 
    107, 111, 102, 101, 110.996798883885, 108, 92, 130.41457625743, 
    111, 115, 115.598336025624, 107.394918938211, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 93.7091686517197, 
    151, 92.2496036556021, 86, 87, 139, 101, 76.548932884982, 
    79.2868253359304, 112, 92.4664494339878, 108.569129940794, 
    86, 87.2452711135797, 78, 85, 85.1162256473953, 99, 90, 88, 
    84, 70, 81, 112.059037537562, 73.1756178685057, 88, 93.3669135960581, 
    70.2690578481042, 134, 85, 88, 76.991122258334, 97, 88, 105.020517511668, 
    86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 88, 84, 85, 86, 
    102.941500011247, 101.679709345242, 87, 96, 122.362953643899, 
    102, 86, 89.1880848857656, 85, 77.4770254618373, 92, 88, 
    80, 122, 119.266770314856, 98, 106, 99.2428464788609, 94, 
    89, 104.725672415927, 88, 101, 103.482014094123, 89, 97, 
    102.252503220054, 112, 102, 101.723541281976, 94.8170475496544, 
    92, 113, 123, 109.498459951131, 116, 98, 103, 115.826082816747, 
    89, 97, 98, 89, 95.1562293910563, 104.21965887923, 106, 106, 
    90, 88, 107.091658648069, 88, 105, 112, 96, 86, 91, 95, 102, 
    106.194852119844, 120, 115.6395209738), phe = c(2.8, 2.65, 
    2.8, 3.85, 2.8, 3.00012572665156, 2.5, 2.6, 3.23682743414553, 
    2.98, 2.95, 2.2, 2.6, 3.1, 2.86795059499517, 2.85, 2.8, 2.86098098402031, 
    3.3, 3.41989069885541, 3, 2.41, 2.61, 2.48, 2.53, 2.4412182285771, 
    2.5322775118531, 2.4, 2.95, 2.65, 3, 2.55875550075003, 2.42, 
    2.95, 2.33672897567896, 2.7, 2.6, 2.45, 3.14376002456484, 
    3.19958538974569, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 
    3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.45821910239649, 2.96367053780895, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.45794966324143, 3.5, 
    1.89, 2.84023161276419, 2.26622039622995, 2.11, 2.53, 4.15365882967341, 
    1.1, 2.95, 1.88, 2.82375216620499, 2.43447023973314, 1.61711780883116, 
    3.14057690573611, 2.51000183064438, 1.9, 2.83, 3.00438284894884, 
    2.2, 2.07414888982719, 1.65, 2.34123997202894, 2.2, 1.68104311529759, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.5646273140176, 
    1.51347755459235, 2.55, 3.52, 3.29570831865182, 2.55346459539175, 
    2.10467622836495, 2.56, 2.5, 2.2, 1.98403211995723, 1.65, 
    1.95401715081646, 2.36, 2.74, 3.18, 2.55, 1.75, 2.181388263012, 
    2.56, 2.46, 1.52114931153575, 2, 1.89009386714787, 2, 2.9, 
    3.18, 1.97894770471183, 2.00503329854617, 2.86, 2.81847126124356, 
    2.74, 2.13, 2.22, 2.1, 1.51, 1.45118244452779, 1.59994215030866, 
    1.7, 1.11566264688501, 1.62, 1.38, 1.85551143770165, 1.62, 
    2.01385271852867, 0.882626099785251, 1.4, 1.36880021799467, 
    2, 1.38, 1.14966477380593, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 1.68750837497551, 1.48, 1.4561377715923, 1.8, 1.9, 1.55806055836805, 
    1.77602289350694, 2.3, 1.83, 1.5621314525242, 1.39, 1.35, 
    0.88729416905846, 1.7, 1.48, 1.55, 1.6972206515242, 1.25, 
    1.39, 1.68, 1.21726108725299, 1.74118451913936, 1.20203367854945, 
    1.65, 2.05), fla = c(2.94363336119689, 2.76, 3.02188558162895, 
    3.49, 2.67056092056753, 3.39, 2.52, 2.51, 2.98, 3.18107865378477, 
    3.32, 2.43, 2.76, 2.91914327975171, 3.64, 2.91, 3.14, 3.4, 
    3.51178608946304, 3.32317289259373, 2.71034269144666, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.36734855364442, 2.97, 2.33, 
    3.25, 3.19, 2.69, 3.43083531127901, 1.60961584247333, 2.98, 
    2.43699641889939, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.32162584122185, 2.92, 3.54, 2.67935815949354, 
    2.99, 3.74, 2.79, 2.9, 2.78, 3.04094719096, 3.23, 3.67, 0.57, 
    1.94867998296358, 1.41, 1.79, 2.50972333852127, 1.75, 2.65, 
    1.48584536338849, 2, 1.3, 4.07833087740946, 1.02, 2.86, 1.99933140286237, 
    2.89, 2.14, 1.57, 2.83644437544111, 1.98311356650589, 1.85, 
    2.82764094446354, 2.26, 2.53, 1.58, 1.59, 2.21, 2.09960087593162, 
    1.69, 1.61, 1.69, 2.29230164657096, 1.83850142956075, 1.25, 
    1.46, 2.16473612454046, 2.67795614768065, 2.27, 0.99, 2.0806892185743, 
    3.81770572298925, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 2.04591023814805, 2.92, 3.74732833054214, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.08777088182798, 1.64, 
    3.62503827854263, 3.91310537072321, 2.21208959912506, 1.68939999432634, 
    3.03, 2.65, 3.15, 2.24, 1.88974294045683, 1.75, 0.957121478311358, 
    0.968292890522554, 1.14159822537881, 1.2, 0.58, 0.66, 0.47, 
    0.85017145779219, 1.39168149071427, 0.6, 0.5, 0.441284066975772, 
    0.52, 0.8, 1.01174622234463, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.961331054931375, 0.58, 
    1.31, 1.1, 0.92, 0.960860147114287, 0.6, 0.958608355776487, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.488345575727378, 
    0.7, 0.61, 0.272555445639025, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.244494527794434, 0.3, 0.24, 0.39, 0.160180196373743, 0.314682442549956, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.30572369874087, 
    0.314444235746581, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.32077211606385, 
    0.32695259110275, 0.34, 0.27, 0.413895788468249, 0.26, 0.29, 
    0.230875557507345, 0.42, 0.490157131799419, 0.29, 0.26, 0.158229987000812, 
    0.36950162632504, 0.28, 0.2, 0.34, 0.181128013880203, 0.215272679392322, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.420389884178741, 0.18986695293494, 
    0.28, 0.415366602583002, 0.53, 0.32, 0.275107047129665, 0.45, 
    0.37, 0.26, 0.27, 0.349221143502963, 0.0749378877240786, 
    0.338179842548033, 0.239227459851186, 0.262643150881171, 
    0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.395204004826764, 
    0.4, 0.61, 0.22, 0.3, 0.382901875355128, 0.460338164444048, 
    0.263418295611016, 0.48719109196551, 0.0991130297805289, 
    0.5, 0.327484009114994, 0.25, 0.17, 0.32, 0.413253102927591, 
    0.254541868812032, 0.24, 0.516293435746875, 0.384337922136899, 
    0.29, 0.34, 0.37, 0.467865929166678, 0.66, 0.315299915630438, 
    0.48, 0.39, 0.29, 0.24, 0.26, 0.349120052157023, 0.42, 0.218547339812448, 
    0.52, 0.3, 0.34605273845465, 0.305852054217774, 0.338599276699152, 
    0.32, 0.315334176037856, 0.43, 0.3, 0.387533029521154, 0.39707830036395, 
    0.39, 0.282771362384061, 0.4, 0.480597482442275, 0.347474083402395, 
    0.504817635770842, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 
    0.53, 0.53, 0.57845679355824, 0.452008895310061, 0.452870565977328, 
    0.435754555788899, 0.43, 0.4, 0.47, 0.45, 0.199630240928313, 
    0.22, 0.230266027215466, 0.26, 0.61, 0.53, 0.576991440276633, 
    0.509668370933667, 0.63, 0.53, 0.52, 0.448693156018442, 0.5, 
    0.6, 0.4, 0.41, 0.213087149109867, 0.43, 0.4, 0.39, 0.519882424781586, 
    0.496596551365142, 0.48, 0.46098699966132, 0.374910384903149, 
    0.43, 0.43, 0.53, 0.559901597750014), pro = c(2.29, 1.28, 
    2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 
    1.81, 2.05179561857197, 2.96, 1.46, 2.23986844476176, 2.68343976804404, 
    1.86, 1.66, 2.1, 0.324510178318878, 1.69, 1.46, 1.66, 1.92, 
    1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.89699540994972, 2.91834943960607, 
    1.54, 1.86, 1.36, 1.44, 1.37, 2.38698170609207, 1.94074051065244, 
    2.00968272604096, 2.1947538713773, 1.49812423602417, 2.03, 
    0.718008088924239, 2.19, 2.14, 2.38, 2.08, 1.46675331191182, 
    2.29, 1.87, 1.68, 1.62, 2.45, 1.47917575450394, 1.66, 1.74338238083161, 
    0.42, 1.14074492915038, 1.70203996706259, 0.73, 1.87, 1.03, 
    2.08254381271185, 0.911878383900254, 1.04, 1.71408909236577, 
    2.5, 1.46, 1.87, 1.03, 1.99379687697567, 1.25724310366624, 
    1.15, 1.46, 1.3977034590626, 2.76, 1.95, 1.43, 1.56596789978443, 
    1.64881389686607, 1.62, 1.85532400894759, 1.46, 1.56, 1.34, 
    1.35, 1.38, 1.64, 1.63, 0.797243397918744, 1.99, 1.35, 3.28, 
    1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.633917904700549, 
    1.42, 2.12457143097065, 1.63, 2.08, 2.49, 1.85926023037721, 
    1.40620320675042, 1.13402255314023, 2.200181821721, 1.04, 
    1.75364683226156, 1.32314546018015, 1.61, 0.83, 1.87, 1.83, 
    2.22630928670381, 1.71, 1.3613657199472, 2.91, 1.35, 1.77, 
    1.57103747841198, 1.9, 0.911260127648634, 0.94, 0.974909873480644, 
    0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 1.23581306947186, 
    0.75, 0.863607571401216, 0.55, 1.21081194970706, 1.14, 1.3, 
    0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 1.00673009287532, 
    1.33279396386387, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 0.660971482175891, 
    0.844448344187911, 0.94, 1.34615277437322, 1.15, 1.46, 0.97, 
    1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
    ), col = c(5.64, 4.38, 5.68, 7.8, 3.52091379789774, 6.75, 
    4.47727930733637, 5.05, 5.4127977849643, 7.22, 5.75, 7.28862049341839, 
    5.6, 5.95160097843933, 7.5, 5.10424884218682, 7.41804593944892, 
    4.69313928365409, 8.7, 5.1, 5.65, 4.5, 3.44701667319773, 
    3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 7.25368944317449, 
    6.23924602206673, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 
    4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 8.19228843544761, 
    8.9, 7.2, 7.19578785769448, 5.22229342836925, 6.3, 5.85, 
    6.25, 8.40018507295849, 4.81038433219655, 6.8, 5.74748954943414, 
    3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 
    6.53577318789923, 3.38, 3.74, 3.61416612757713, 3.21, 3.8, 
    4.6, 2.80664618368064, 3.98731765239359, 2.18475010774027, 
    2.5, 3.9, 1.39454092171029, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 
    1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 7.33052279440812, 
    2.9, 2.6926468041677, 3.8249123509948, 4.38027534387994, 
    4.34085902715981, 2.8, 3.36203471674091, 2.94, 2.7, 3.4, 
    3.3, 2.7, 1.77511031419881, 2.9, 1.33775352399132, 2.89097282272132, 
    3.08, 2.9, 1.9, 2.74542623905622, 5.92054336584324, 3.4, 
    1.28, 3.25, 5.75290878265224, 2.00778915224241, 2.57807447819465, 
    2.8, 1.44772500311465, 3.94, 0.729836901735172, 2.12, 6.0876566784602, 
    6.64033575193315, 6.5282328172191, 7.32228662112922, 5, 5.45, 
    7.1, 6.31882447978366, 5, 5.7, 4.92, 4.79812556787945, 10.5664570512747, 
    4.35, 4.4, 8.21, 4, 4.9, 7.03280449120719, 8.42, 9.4, 5.73743993302512, 
    5.99461949803738, 7.1, 10.52, 7.6, 7.9, 9.13133519171792, 
    7.5, 13, 9.52978011755434, 7.65, 6.30667043382087, 5.58, 
    10.4926178296851, 5.46696688038129, 6.62, 10.68, 10.26, 4.58675139175446, 
    8.5, 6.95526901545023, 4.91903304857576, 9.7, 7.7, 7.3, 10.1828985225417, 
    9.3, 9.2), hue = c(1.04, 1.05, 1.09105598250904, 0.86, 1.24543586615847, 
    1.05, 1.02, 0.987609410876064, 1.08, 0.97801245363958, 1.25, 
    1.24888510592512, 1.15, 1.25, 1.2, 1.28, 1.08955318869088, 
    1.13, 1.23, 1.39325905027768, 1.09, 0.907015639099093, 1.11, 
    1.09, 1.12, 1.13, 0.83948204745575, 1.02, 1.15145516904687, 
    1.04, 1.19, 1.09, 1.09823442975338, 1.25, 1.1, 1.12791536891669, 
    0.785750945940748, 1.02210232048872, 1.18, 1.40079522122278, 
    0.95, 0.91, 0.88, 0.82, 1.38904715956942, 0.87, 1.17250907946828, 
    0.9854446705938, 1.07, 1.12, 1.12, 0.912762852152571, 0.985507951591501, 
    0.774940010548841, 1.07166634373609, 1.23057631306931, 0.94, 
    1.07, 0.89, 1.1409310077231, 1.25, 0.98, 1.23, 1.22, 0.95516946626894, 
    1.21580227378923, 0.724626324370307, 1.12, 1.02, 1.28, 0.906, 
    1.36, 0.98, 1.31, 1.00106366411507, 1.23, 1.19, 0.96, 1.06, 
    1.19, 1.38, 0.994074249996937, 1.31, 0.84, 1.27812573617574, 
    1.23, 1.26321877057461, 0.870388940974243, 1.1574133985387, 
    1.07, 1.08, 1.05, 1.09712540302035, 1.15, 1.00006480779499, 
    1.3031234734343, 1.04861132545831, 0.679654281409126, 1.5337427382549, 
    1.42, 1.27, 1.07827080884091, 0.8, 0.982322204028589, 1.04, 
    0.862108244824973, 1, 0.810008115801744, 0.86, 1.13982664050972, 
    0.918247219728815, 0.9, 1.04241658401718, 1.1, 0.93, 1.71, 
    0.95, 0.738943386937404, 0.7, 1.00593860653062, 0.8, 1.30155587781742, 
    0.92, 0.73, 0.823406221247525, 0.86, 0.69, 0.97, 0.89, 0.79, 
    1.13905074047959, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
    0.81, 0.742481251521182, 0.636513532711965, 0.825835946514982, 
    0.89, 0.72658937990364, 0.65, 0.6, 0.620195190769576, 0.54, 
    0.55, 0.45744942862964, 0.912684929425807, 0.48, 0.61, 0.56, 
    0.580866152336631, 0.6, 0.57, 0.67, 0.57, 0.875443579383131, 
    0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.593769298646394, 
    0.74, 0.67, 0.738447596709742, 0.889713754942719, 0.256996433816939, 
    0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 2.57998890636276, 
    2.93, 3.04872530018159, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
    2.9, 3.68668332708641, 3, 3.33868457974303, 2.65, 4.09659590400938, 
    2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.42546560695037, 3.2, 3.03410850811122, 
    2.77, 3.36149876703862, 2.86375417679276, 2.71, 2.88, 2.87, 
    3, 2.87, 3.43177822143106, 2.23758017811536, 3.49832306915496, 
    2.69, 3.39897899490833, 3.38, 3, 3.17831871871324, 3, 3.35, 
    3.33, 2.96942087357994, 3.33, 2.75, 3.1, 2.11641682751032, 
    2.86079484856321, 3.26, 2.93, 3.2, 3.03, 2.61075949609572, 
    2.84, 2.87, 1.82, 3.58739493063954, 1.59, 2.46, 2.87, 2.59047691581125, 
    2.3, 3.18, 3.48, 3.07648067931727, 5.4278352741245, 1.82, 
    3.84924159985733, 2.58957265092541, 3.5, 3.13, 2.14, 2.48, 
    2.52, 2.85973200782694, 3.79300290357224, 3.81555303952523, 
    3.14, 2.72, 2.01, 2.38163115170492, 4.08887761053196, 3.53216298933758, 
    3.21, 2.75, 3.21, 3.22000821375709, 2.65, 3.37950600596408, 
    3.3, 3.18575904465455, 2.63, 1.5565675813916, 2.74, 3.59372551580625, 
    2.83, 2.96, 2.77, 3.38, 2.50402022585955, 3.57, 3.3, 3.17, 
    2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.55946938197145, 3.19, 
    2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.87882628017713, 
    3.64, 3.78183199763751, 2.84, 4.22268198502138, 2.78, 2.56504975835622, 
    2.4577752541437, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
    1.82, 2.15, 1.92822508885369, 1.38898187704408, 1.41426650186689, 
    2.05, 2, 2.12656191990088, 1.33, 1.35415353030915, 1.62, 
    1.33, 2.55314841313768, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
    1.73, 1.96, 1.78, 1.58, 1.95118217566667, 3.39619573147228, 
    1.75, 1.68, 1.75, 1.38122996705839, 1.39233095767583, 1.77079605565437, 
    1.94640255996535, 1.80805583625267, 1.492307399647, 1.52113101468112, 
    1.30021535197028, 1.2285287206535, 1.56, 1.62, 1.6), prol = c(1065, 
    1050, 444.370918907353, 982.76469281739, 735, 1450, 1290, 
    1295, 1045, 1045, 1191.34445017938, 1280, 1320, 1096.75478956825, 
    1547, 1310, 1280, 1063.21040173624, 1680, 845, 810.931465382991, 
    770, 1035, 1015, 845, 996.408470623766, 1195, 1285, 915, 
    1035, 1285, 795.097415411733, 1305.06189306605, 1235, 927.683560452069, 
    920, 880, 609.407525781659, 1028.86465979707, 760, 795, 1035, 
    1095, 680, 885, 1080, 1065, 985, 1060, 1201.45640575239, 
    1150, 1265, 1190, 1071.13742744468, 1060, 1120, 970, 1270, 
    1285, 544.744832372233, 435.348624036508, 851.390790899855, 
    630, 253.541029519808, 565.043254503943, 724.516521878359, 
    502, 554.804215546575, 106.108345361703, 718, 870, 410, 472, 
    696.218137614446, 886, 634.861854762619, 392, 434.332989920528, 
    1140.80525643466, 463, 278, 714, 233.9341875621, 515, 692.41965524355, 
    696.855051107443, 491.606628478565, 562, 680, 625, 480, 450, 
    495, 799.175653077654, 345, 355.502856227256, 625, 428, 660, 
    406, 301.913666612009, 562, 438, 415, 672, 334.408973580365, 
    510, 367.999270931127, 312, 680, 562, 325, 791.452352205577, 
    434, 385, 425.238884573612, 479.824330031713, 733.785466227658, 
    372, 564, 570.812575509442, 1393.27882695416, 365, 380, 361.46966566804, 
    378, 564.085098895868, 466, 275.107539815611, 580, 889.801777747812, 
    708.66404302008, 560, 600, 693.726232810955, 695, 720, 515, 
    580, 590, 600, 679.741903595559, 520, 550, 772.082006340532, 
    830, 415, 1114.98499626559, 882.276464728299, 550, 500, 480, 
    425, 675, 640, 670.472934365685, 480, 880, 660, 646.248501655716, 
    680.599767084713, 680, 570, 785.908571346749, 615, 824.993283891837, 
    695, 685, 365.478898485035, 518.591030606609, 510, 470, 773.434804880992, 
    740, 440.500522032836, 835, 840, 652.341457477591)), row.names = c(NA, 
    -178L), class = "data.frame"), structure(list(alco = c(14.23, 
    13.2, 13.16, 14.9470298963647, 13.24, 14.1511979303617, 14.39, 
    14.06, 13.5446224455248, 13.86, 13.3949098624723, 13.8524253930062, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 14.5020873259602, 
    13.64, 13.6226832867076, 12.93, 13.71, 12.85, 13.5, 13.7095655473592, 
    13.39, 13.3, 13.87, 14.1455607883355, 13.73, 13.58, 13.68, 
    13.76, 13.3050922341679, 13.3899625770017, 12.3958684564065, 
    13.05, 11.5435895601664, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 13.7510799922997, 13.9, 14.2798308343568, 13.7776000693749, 
    13.05, 13.9490142526555, 13.82, 13.1501558910944, 13.74, 
    15.2191366842333, 14.22, 13.29, 13.6852300355433, 12.37, 
    12.33, 12.64, 11.7565064357975, 12.37, 12.17, 12.37, 13.1865318099216, 
    12.2056192365801, 13.34, 12.21, 12.1681452676864, 13.86, 
    12.7990685370401, 12.99, 11.96, 11.66, 12.5465469227178, 
    11.84, 13.4526432591655, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 
    11.6654011312318, 12.16, 12.5566987314818, 12.9438074742724, 
    12.08, 12.08, 12, 12.69, 12.9391363768961, 11.62, 12.5398376425499, 
    11.81, 12.29, 11.8262287619548, 12.29, 12.08, 12.6, 12.4805233683017, 
    13.7860563323782, 12.51, 12.6851908591746, 11.8692849973508, 
    12.72, 12.22, 11.61, 11.4382133692334, 11.8936958699003, 
    11.76, 12.727256028904, 12.08, 11.03, 11.82, 11.4908705386375, 
    12.0316620720929, 11.2401090372364, 11.45, 13.1302986141389, 
    12.42, 13.05, 11.87, 12.07, 11.5605455890135, 11.79, 13.0451177490006, 
    12.04, 12.86, 12.88, 13.0674446275422, 12.7, 12.51, 12.5178404495788, 
    12.25, 12.53, 12.2581589643294, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 12.0858819582023, 11.6029021111302, 12.87, 
    13.32, 12.7316258378672, 10.9886122468992, 12.79, 13.0278770405948, 
    13.23, 13.0956029029956, 13.17, 13.84, 12.45, 14.34, 13.48, 
    13.2576417869954, 13.69, 12.85, 12.96, 13.78, 13.73, 14.654669467562, 
    13.4458059904848, 13.58, 13.4, 13.3156174683819, 12.77, 14.16, 
    14.7392196605706, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    2.67724153241672, 2.35514322058255, 1.95, 2.39200854765863, 
    1.78812013938626, 1.87, 2.15, 1.36672974563777, 1.35, 2.16, 
    1.48, 1.85382207042399, 1.73, 1.87, 1.80824642805067, 1.92, 
    3.02422768947966, 0.572720429924252, 3.1, 1.63, 3.8, 2.56309503318211, 
    1.6, 1.81, 2.27354885300442, 1.8576131141817, 3.5943487673281, 
    1.9, 2.37580260422966, 1.5, 2.53445006674186, 1.83, 1.53, 
    1.52558561037432, 2.213510150309, 1.55964987919227, 1.65, 
    1.5, 0.646331424114684, 1.71, 3.6400150185158, 2.14860639713648, 
    3.98, 0.770285416305535, 4.04, 0.90801276557332, 1.68, 2.2969258040833, 
    1.1312527634262, 1.30435114576899, 1.65, 1.7225301390912, 
    1.9, 1.67, 2.22452735742239, 1.7, 1.97, 2.41381955638989, 
    0.94, 3.7910881865658, 1.36, 1.25, 1.34143018232741, 2.68172648464087, 
    1.21, 1.01, 1.17, 3.47638978090862, 2.68269714734767, 2.62125894142521, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 4.00404681022441, 0.99, 
    3.87, 1.72649735238891, 2.5942804925892, 1.13, 3.22863104414369, 
    1.53070710967346, 0.98, 1.61, 1.67, 3.23572084589419, 2.23237337610845, 
    2.69627614570983, 1.51, 2.44290013098995, 3.15673798386279, 
    1.33445478210974, 1.1114661741305, 3.6948574958596, 1.41, 
    1.1111925413398, 2.13450549576093, 2.08, 0.944355066558488, 
    1.26136961215747, 1.72, 1.97042108276951, 2.55, 1.73, 2.42668318515902, 
    1.29, 0.787638347378205, 3.74, 1.90561832063909, 2.68, 0.74, 
    1.39, -0.475423492891264, -1.02999405930389, 1.61, 3.43, 
    0.107717911943612, 2.4, 2.05, 4.43, 2.2978346450617, 4.31, 
    2.16, 1.53, 2.13, 2.14028412335218, 4.3, 1.35, 2.99, 2.31, 
    3.55, 1.24, 4.50783629489681, 3.20124037715013, 5.51, 3.59, 
    4.00131376000043, 2.81, 2.56, 2.4588040114391, 2.44403067762773, 
    3.88, 3.97908236762327, 5.04, 4.61, 3.24, 5.77570732509575, 
    3.12, 2.67, 1.9, 3.07958504022528, 1.29, 5.19, 4.12, 3.03, 
    1.68, 1.67, 3.83, 2.15960776531855, 1.86385074616352, 3.45, 
    2.76, 2.20316766314679, 4.4890183529539, 3.37, 2.58, 4.6, 
    4.06472870278673, 2.39, 4.22433835281601, 5.65, 3.91, 2.78485771817004, 
    2.59, 4.1), ash = c(2.43, 2.14, 2.67, 2.5, 2.30182734824593, 
    2.69984596069384, 2.45, 2.61, 2.14873318912876, 2.47632649873904, 
    2.3, 2.25086195485511, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
    2.48, 2.56, 2.24717802958828, 2.65, 2.61133523831522, 2.52, 
    2.61, 3.22, 2.49028201764212, 2.30005764969346, 2.50613037795093, 
    2.12233897992259, 2.71779120555805, 2.36, 2.68008328256143, 
    2.7, 2.65, 2.45562470869631, 2.29100220295469, 2.55, 2.1, 
    2.51, 2.29632933586717, 2.12, 2.59, 2.29, 2.1, 2.44, 2.47078018070777, 
    2.12, 2.4, 2.27, 2.04, 2.43350154638381, 2.87416476509145, 
    2.28525986295694, 2.63148049484508, 2.91474484891564, 2.3, 
    2.68, 2.5, 1.36, 1.68040961345916, 1.90299759328727, 1.92, 
    2.16, 2.53, 2.56, 2.14422074739839, 1.92, 2.27679550722079, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.12628782102195, 1.71, 
    2.45048923018798, 2.25571827501402, 2.4, 2, 2.47172026111485, 
    2.51, 2.32, 2.58, 2.24, 2.33393970423044, 2.62, 2.50597138289849, 
    2.3, 2.32, 2.02667265563097, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.41504418957739, 1.94829156740143, 1.7, 2.2783528680951, 
    2.46, 1.88, 2.45270138404608, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.53023063658165, 
    1.86450576789531, 2, 2.42, 3.23, 2.32347828807289, 2.13, 
    2.39, 2.17, 2.29, 2.34853415259157, 2.10547951747208, 2.38, 
    2.74083979710165, 2.4, 2.4, 2.36, 2.25, 1.97747399497406, 
    2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.86875225974219, 
    2.2460766677134, 2.48, 2.37644275875544, 2.36, 2.62, 2.48, 
    1.86796847959754, 2.28, 2.29080934960612, 2.32, 2.38, 1.64350156483689, 
    2.09687305271061, 2.64, 2.38, 2.54, 2.00981898774305, 2.35, 
    2.3, 1.85526394456194, 2.6, 2.30107267522498, 2.69, 2.81434291325814, 
    2.32, 2.28, 2.58426242628337, 2.74111504430598, 2.48, 2.26, 
    2.37, 2.74), alca = c(15.6, 15.1649875614738, 23.7149795124167, 
    16.8, 21, 15.2, 15.6735541795409, 17.6, 14, 16, 16.2120178143182, 
    16.8, 16, 11.4, 13.0000376382993, 15.3813877682537, 15.7982376300048, 
    20, 12.7445694095516, 15.2, 16, 18.6, 19.0648068495548, 17.8, 
    20, 18.1971227571785, 16.1, 17, 16.6919395132842, 11.92365625365, 
    16.270498831185, 19.1, 19.2526738302785, 19.2878330556592, 
    19, 20.5, 15.5, 16.3482018509228, 15.5, 13.2, 16.2, 18.8, 
    12.4540434851911, 16.3153635948514, 17, 17.3030755971534, 
    16, 12.7109806617041, 18.8, 17.4, 12.4, 17.3192074950252, 
    14.3512643811853, 16.8092533713763, 20.627695187034, 20.5, 
    16.3, 16.8, 10.2593072316414, 10.6, 18.8057320198898, 16.8, 
    18, 19, 19, 18.1, 20.1366570632127, 19.6, 19.3806979246743, 
    16.8, 20.4, 19.3903265448634, 24, 30, 21, 22.1231219049462, 
    16, 18, 14.8, 20.9093239414392, 19, 23.6440267249527, 24, 
    22.5, 18, 18, 22.8, 25.2958589555365, 21.6, 16.3188071472904, 
    24.6853481878673, 22, 24.4084396446676, 18, 18, 19, 21.5, 
    16, 17.5924254607479, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
    19, 22.5, 19, 20, 19.5, 20.8096346443721, 22.979185139354, 
    20.5595400703395, 20.7547259052959, 21.5, 20.8, 25.9961955463628, 
    16, 20.1035343915486, 21.6493720075892, 18.9886941127439, 
    19.7538993554029, 21.5, 21, 21, 21.5, 28.5, 21.7451122323866, 
    22, 21.2465320636203, 20, 24, 21.5, 23.3461080076454, 20.7641544565259, 
    20.3305325421204, 25, 19.5, 20.4045966387304, 21, 25.9374357837032, 
    23.5, 20, 19.0160310911802, 21, 20, 18.576933372425, 16.434729509828, 
    21.5, 24, 22, 16.9901652352702, 18.5, 20, 22, 19.5, 12.1420669463116, 
    25, 27.3492611797341, 17.4101540928925, 27.8665753582244, 
    22.7491470936721, 18.5, 18.8635451771362, 22.5, 23, 19.5, 
    24.5, 25, 20.8743344005908, 19.5, 20.9792212815356, 20.5, 
    23, 20, 21.3014600425396, 24.5), mg = c(127, 100, 101, 113, 
    93.5958186351747, 112, 96, 121, 97, 117.21823331529, 129.951808057099, 
    105.59597946342, 89, 91, 102, 112, 120, 122.278467970175, 
    100.899847143872, 116, 126, 102, 96.5942646844708, 95, 106.639035961726, 
    124, 93, 115.690210822542, 107, 96, 101, 106, 104, 130.083614286272, 
    110, 127.318407428096, 110, 132.368188328174, 98.365730684756, 
    102.135046688919, 117, 122.90693311695, 101, 103.364080613132, 
    107, 111, 102, 101, 120.044797752846, 108, 92, 100.970607703381, 
    111, 115, 124.272004568494, 115.855138064088, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 94.9256010045764, 
    151, 105.659186064747, 86, 87, 139, 101, 98.111223038398, 
    95.3035923228853, 112, 117.957056515931, 91.8630683423604, 
    86, 94.363004428879, 78, 85, 133.986953762611, 99, 90, 88, 
    84, 70, 81, 109.129799748798, 76.1490283160261, 88, 106.42346196472, 
    91.6913876844487, 134, 85, 88, 82.1624400252151, 97, 88, 
    87.8884682862867, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 
    88, 84, 85, 86, 86.6190861144421, 116.404372761501, 87, 96, 
    139.8019251238, 102, 86, 99.524022198334, 85, 81.3451490075091, 
    92, 88, 80, 122, 111.762406353975, 98, 106, 101.22108627876, 
    94, 89, 102.36663223879, 88, 101, 90.7320497185909, 89, 97, 
    89.0054815803422, 112, 102, 100.717793626691, 89.2714847977517, 
    92, 113, 123, 109.385256309514, 116, 98, 103, 113.275298079644, 
    89, 97, 98, 89, 89.3515162937237, 87.676320993286, 106, 106, 
    90, 88, 106.596849235865, 88, 105, 112, 96, 86, 91, 95, 102, 
    126.862845741121, 120, 118.160814407269), phe = c(2.8, 2.65, 
    2.8, 3.85, 2.8, 3.29014956585085, 2.5, 2.6, 3.03473357211941, 
    2.98, 2.95, 2.2, 2.6, 3.1, 3.05439260248913, 2.85, 2.8, 2.93456297893475, 
    3.3, 3.19385750296035, 3, 2.41, 2.61, 2.48, 2.53, 2.42510340801793, 
    2.72143738431192, 2.4, 2.95, 2.65, 3, 2.55064495999532, 2.42, 
    2.95, 2.17778856356088, 2.7, 2.6, 2.45, 3.08622773906943, 
    3.25217909042982, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 
    3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.30847840284886, 2.87905559391409, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.82499952978144, 3.5, 
    1.89, 2.73631171552898, 2.90764067112657, 2.11, 2.53, 3.20467833122379, 
    1.1, 2.95, 1.88, 2.01533623705859, 2.0565884275819, 1.91231893998152, 
    2.40988607929206, 1.77355472135541, 1.9, 2.83, 2.82398148417076, 
    2.2, 1.63545188971871, 1.65, 1.9751233442288, 2.2, 2.23596107692296, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.35440915409847, 
    1.69617508186985, 2.55, 3.52, 2.56909989880325, 2.33770225475856, 
    2.37088257091693, 2.56, 2.5, 2.2, 2.11534897229277, 1.65, 
    2.17290914934642, 2.36, 2.74, 3.18, 2.55, 1.75, 2.1525136747528, 
    2.56, 2.46, 1.49560098269012, 2, 2.12079359518142, 2, 2.9, 
    3.18, 2.17164396070261, 1.6765441101933, 2.86, 2.7624588590519, 
    2.74, 2.13, 2.22, 2.1, 1.51, 2.14992462081343, 2.06857266150962, 
    1.7, 0.958414058099508, 1.62, 1.38, 1.80415074933042, 1.62, 
    2.10643509062292, 1.54970254180542, 1.4, 1.70438534067765, 
    2, 1.38, 0.805454048722795, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 2.19555679771965, 1.48, 1.11071001556194, 1.8, 1.9, 
    1.45496062822675, 1.37789696291256, 2.3, 1.83, 1.57872152170671, 
    1.39, 1.35, 1.39475338721856, 1.7, 1.48, 1.55, 1.20644167771659, 
    1.25, 1.39, 1.68, 1.47976823167386, 2.18571972533011, 1.95779146894801, 
    1.65, 2.05), fla = c(2.97254044194433, 2.76, 3.43337709434217, 
    3.49, 2.05673430593296, 3.39, 2.52, 2.51, 2.98, 3.31438585693654, 
    3.32, 2.43, 2.76, 2.86233101823825, 3.64, 2.91, 3.14, 3.4, 
    3.14709247595116, 3.41715255064339, 3.20995882573223, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.43683479575575, 2.97, 2.33, 
    3.25, 3.19, 2.69, 2.68506519773852, 2.19399274626764, 2.98, 
    2.67333641889538, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.1277712973923, 2.92, 3.54, 3.27432939039718, 
    2.99, 3.74, 2.79, 2.9, 2.78, 3.1520460335241, 3.23, 3.67, 
    0.57, 2.03083291862922, 1.41, 1.79, 3.04929114285681, 1.75, 
    2.65, 2.07219365106662, 2, 1.3, 3.65412842927516, 1.02, 2.86, 
    0.939331075979496, 2.89, 2.14, 1.57, 1.31767895328647, 0.974681926997677, 
    1.85, 2.5595822474741, 2.26, 2.53, 1.58, 1.59, 2.21, 1.66560989845596, 
    1.69, 1.61, 1.69, 2.0219592615428, 1.69789165590355, 1.25, 
    1.46, 2.19372023898014, 2.383289549538, 2.27, 0.99, 1.87679429162036, 
    3.30389041912462, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 2.06250994755475, 2.92, 3.301344334986, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.42643758568951, 1.64, 
    3.55215336626284, 3.67791139692924, 1.58244664814572, 1.94992656067106, 
    3.03, 2.65, 3.15, 2.24, 1.34199927421128, 1.75, 2.10950692044378, 
    1.41828622598391, 1.75169038061528, 1.2, 0.58, 0.66, 0.47, 
    0.985976994815753, 0.466465652340326, 0.6, 0.5, 1.08447898196969, 
    0.52, 0.8, 1.10590707140663, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 0.762002009936779, 0.58, 
    1.31, 1.1, 0.92, 1.27009600459392, 0.6, 0.101095942861293, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.652725404555059, 
    0.7, 0.61, 1.09001723471991, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.416588365114721, 0.3, 0.24, 0.39, 0.222584181213115, 0.435748417008153, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.118962923459115, 
    0.210859797862584, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.406293523358149, 
    0.215945990003668, 0.34, 0.27, 0.260237748053384, 0.26, 0.29, 
    0.437727496557851, 0.42, 0.280925983200893, 0.29, 0.26, 0.109342432747018, 
    0.254293665733009, 0.28, 0.2, 0.34, 0.234487914621985, 0.32225543248571, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.381361322467839, 0.219377967296493, 
    0.28, 0.188335051357001, 0.53, 0.32, 0.37773439687909, 0.45, 
    0.37, 0.26, 0.27, 0.390874776441946, 0.0512766395232832, 
    0.360011507036122, 0.1864045701702, 0.489431727090673, 0.21, 
    0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.414919595116356, 
    0.4, 0.61, 0.22, 0.3, 0.399173699741563, 0.251187379200568, 
    0.382350347150331, 0.389290611780335, 0.651429702864512, 
    0.5, 0.377537457031759, 0.25, 0.17, 0.32, 0.471948218449657, 
    0.376539837774243, 0.24, 0.458868148023323, 0.314339745708658, 
    0.29, 0.34, 0.37, 0.443963611491115, 0.66, 0.390920272750923, 
    0.48, 0.39, 0.29, 0.24, 0.26, 0.10213101841498, 0.42, 0.401929039399551, 
    0.52, 0.3, 0.275073168888336, 0.318076907030329, 0.343462612763786, 
    0.32, 0.0935094977468406, 0.43, 0.3, 0.232499345858632, 0.224928818357109, 
    0.39, 0.374734461263219, 0.4, 0.625536515369103, 0.274375198835245, 
    0.283368628990385, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 
    0.53, 0.53, 0.459124953992748, 0.305077860735485, 0.453963027602083, 
    0.459658901324774, 0.43, 0.4, 0.47, 0.45, 0.127778604678571, 
    0.22, 0.264884444052892, 0.26, 0.61, 0.53, 0.483263004009714, 
    0.64069248375482, 0.63, 0.53, 0.52, 0.494494764781062, 0.5, 
    0.6, 0.4, 0.41, 0.28620488787624, 0.43, 0.4, 0.39, 0.479667548473565, 
    0.645820847713557, 0.48, 0.409965492625458, 0.478694047438492, 
    0.43, 0.43, 0.53, 0.536609577227438), pro = c(2.29, 1.28, 
    2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 2.38, 1.57, 
    1.81, 1.80230543924911, 2.96, 1.46, 2.27472355276048, 2.6534852709673, 
    1.86, 1.66, 2.1, 0.46675949619201, 1.69, 1.46, 1.66, 1.92, 
    1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.71530781498851, 1.68638383299175, 
    1.54, 1.86, 1.36, 1.44, 1.37, 2.42882693582796, 1.89893489313099, 
    1.75610886163354, 1.60376261480708, 1.90740003642668, 2.03, 
    0.825120792331713, 2.19, 2.14, 2.38, 2.08, 1.87445602847907, 
    2.29, 1.87, 1.68, 1.62, 2.45, 1.97054326777798, 1.66, 1.64909494772941, 
    0.42, 1.83578782079128, 2.00430897621194, 0.73, 1.87, 1.03, 
    2.32385256022196, 1.21719160354876, 1.04, 0.628371478608712, 
    2.5, 1.46, 1.87, 1.03, 2.09841984698137, 1.91397674907889, 
    1.15, 1.46, 1.08278901084267, 2.76, 1.95, 1.43, 2.00533744716894, 
    1.96576122213504, 1.62, 1.33501530096811, 1.46, 1.56, 1.34, 
    1.35, 1.38, 1.64, 1.63, 1.80052629384743, 1.99, 1.35, 3.28, 
    1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.10596455465153, 
    1.42, 1.68377369092833, 1.63, 2.08, 2.49, 1.97995709967061, 
    1.11180080133509, 1.43023710520546, 2.160879769209, 1.04, 
    2.13367110414718, 2.15104234768551, 1.61, 0.83, 1.87, 1.83, 
    2.21983432906087, 1.71, 1.76478379632997, 2.91, 1.35, 1.77, 
    1.69320631203602, 1.9, 1.88043911580396, 0.94, 0.527718261898732, 
    0.83, 0.84, 1.25, 0.94, 0.8, 1.1, 0.88, 0.890170757291548, 
    0.75, 2.09931393903647, 0.55, 0.990601927656008, 1.14, 1.3, 
    0.68, 0.86, 1.25, 1.14, 1.25, 1.26, 1.56, 0.37404195069596, 
    0.236386883084396, 1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.34057643954813, 
    0.973554826287833, 0.94, 1.41613535576636, 1.15, 1.46, 0.97, 
    1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 1.46, 1.35
    ), col = c(5.64, 4.38, 5.68, 7.8, 3.3971976949946, 6.75, 
    5.16603523510938, 5.05, 5.98857119294687, 7.22, 5.75, 7.11127864535092, 
    5.6, 8.19848165020882, 7.5, 5.4312566364632, 7.55088066040246, 
    4.5020504285724, 8.7, 5.1, 5.65, 4.5, 3.46032903895036, 3.93, 
    3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 6.47148554373682, 5.69439077636136, 
    5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 
    5.04, 5.24, 4.9, 6.1, 7.10718436226983, 8.9, 7.2, 6.25537958929371, 
    5.07646605920069, 6.3, 5.85, 6.25, 7.40989035788439, 4.90861076624044, 
    6.8, 5.90012386001283, 3.27, 5.75, 3.8, 4.45, 2.95, 4.6, 
    5.3, 4.68, 3.17, 2.85, 3.83209462600236, 3.38, 3.74, 5.46089644165591, 
    3.21, 3.8, 4.6, 3.84893069015981, 3.04793161423012, 0.517791876847236, 
    2.5, 3.9, 2.15530459170885, 4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 
    1.74, 2.4, 3.6, 3.05, 2.15, 3.25, 2.6, 5.26669634990354, 
    2.9, 3.14719286946576, 4.67588242039557, 4.84261982941112, 
    2.78260801268887, 2.8, 7.19304421911403, 2.94, 2.7, 3.4, 
    3.3, 2.7, 1.47407281588841, 2.9, 3.79050235624172, 2.32366096549093, 
    3.08, 2.9, 1.9, 4.03304571237934, 1.62112083708118, 3.4, 
    1.28, 3.25, 3.17788219264077, 4.36262047563352, 4.5753260837281, 
    2.8, 3.02005265241405, 3.94, 3.46375805629175, 2.12, 3.8824771661302, 
    6.77065395848006, 6.67701645138466, 13.7767600888763, 5, 
    5.45, 7.1, 6.91311466684771, 5, 5.7, 4.92, 7.06627350210067, 
    5.00613567657154, 4.35, 4.4, 8.21, 4, 4.9, 6.30935414114455, 
    8.42, 9.4, 0.276091968350133, 6.35456013447109, 7.1, 10.52, 
    7.6, 7.9, 8.25055897080625, 7.5, 13, 4.79513502240404, 7.65, 
    5.65562068392578, 5.58, 6.78067219116322, 5.64936724740508, 
    6.62, 10.68, 10.26, 2.17834828533175, 8.5, 8.60517734858897, 
    9.00512718293989, 9.7, 7.7, 7.3, 9.77849011236203, 9.3, 9.2
    ), hue = c(1.04, 1.05, 0.930342135888664, 0.86, 1.12267436905456, 
    1.05, 1.02, 0.910448747892175, 1.08, 1.07539071386313, 1.25, 
    0.777038572615583, 1.15, 1.25, 1.2, 1.28, 1.13779164664433, 
    1.13, 1.23, 1.08990546471207, 1.09, 0.710535428728662, 1.11, 
    1.09, 1.12, 1.13, 1.07115119078592, 1.02, 1.09618196521802, 
    1.04, 1.19, 1.09, 0.963898971628126, 1.25, 1.1, 0.980350000731466, 
    1.30401706491118, 1.17472585804352, 1.18, 1.05580143280251, 
    0.95, 0.91, 0.88, 0.82, 1.37908494954216, 0.87, 1.3327132116034, 
    1.03890140431643, 1.07, 1.12, 1.12, 1.24675826317588, 0.998287960092099, 
    1.22926664602041, 0.872325106487931, 0.732528001114205, 0.94, 
    1.07, 0.89, 1.11208433945204, 1.25, 0.98, 1.23, 1.22, 1.12703529243692, 
    1.21778599328356, 1.01816624254371, 1.12, 1.02, 1.28, 0.906, 
    1.36, 0.98, 1.31, 1.05921499908196, 1.23, 1.19, 0.96, 1.06, 
    1.19, 1.38, 1.13191698131966, 1.31, 0.84, 0.948429207337312, 
    1.23, 0.910190523692787, 0.819041738635708, 0.811154434085372, 
    1.07, 1.08, 1.05, 0.917445842691179, 1.15, 1.16525531243897, 
    1.21149934586988, 0.640870630168468, 0.625790858644992, 1.28441237301128, 
    1.42, 1.27, 0.952645707356392, 0.8, 0.903980565855941, 1.04, 
    0.938506901658579, 1, 1.02686265182814, 0.86, 1.10230207367925, 
    1.28420178195921, 0.9, 0.83683955817111, 1.1, 0.93, 1.71, 
    0.95, 1.30420774715801, 0.7, 1.04286613548493, 0.8, 1.07200075153936, 
    0.92, 0.73, 1.09483016454295, 0.86, 0.69, 0.97, 0.89, 0.79, 
    0.533975556712571, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 
    0.81, 0.746936671265891, 0.750418874418579, 1.09447688391585, 
    0.89, 0.888140365376118, 0.65, 0.6, 0.498935046175834, 0.54, 
    0.55, 0.657813786605516, 1.06481268859686, 0.48, 0.61, 0.56, 
    0.853146770969073, 0.6, 0.57, 0.67, 0.57, 0.701551862946966, 
    0.56, 0.96, 0.87, 0.68, 0.7, 0.78, 0.85, 0.672443017765451, 
    0.74, 0.67, 0.517865406688061, 0.533843089719026, 0.503507218006571, 
    0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 3.02621323457128, 
    2.93, 2.99937062545647, 3.58, 3.58, 2.85, 3.55, 3.17, 2.82, 
    2.9, 2.49879896059485, 3, 3.22825797225166, 2.65, 3.14414516992054, 
    2.82, 3.36, 3.71, 3.52, 4, 3.63, 3.78117249024447, 3.2, 3.40338042118081, 
    2.77, 3.76630290191691, 3.05678676231825, 2.71, 2.88, 2.87, 
    3, 2.87, 3.46599795731402, 2.54644393914919, 3.55066052785557, 
    2.69, 3.68118550160606, 3.38, 3, 3.78222237849821, 3, 3.35, 
    3.33, 3.57864611281792, 3.33, 2.75, 3.1, 2.7247525086108, 
    3.29149918969658, 3.26, 2.93, 3.2, 3.03, 3.08878226387263, 
    2.84, 2.87, 1.82, 2.88573768939451, 1.59, 2.46, 2.87, 2.79514090635278, 
    2.3, 3.18, 3.48, 2.73007998127198, 4.11221793589984, 1.82, 
    3.99539150897127, 2.39342730200208, 3.5, 3.13, 2.14, 2.48, 
    2.52, 2.98439848177121, 3.88209445752225, 3.22268025114118, 
    3.14, 2.72, 2.01, 2.4354287962325, 2.54722583004941, 2.65466269938878, 
    3.21, 2.75, 3.21, 2.52941258717392, 2.65, 2.64484118446274, 
    3.3, 2.64058933986743, 2.63, 1.57245032681295, 2.74, 3.07071721732041, 
    2.83, 2.96, 2.77, 3.38, 2.24321979003526, 3.57, 3.3, 3.17, 
    2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.35801326006239, 3.19, 
    2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 3.45194304757366, 
    3.64, 3.40907990997477, 2.84, 2.33109708434558, 2.78, 2.49866738296609, 
    2.3153194530124, 1.42, 1.36, 1.29, 1.51, 1.58, 1.27, 1.69, 
    1.82, 2.15, 2.02486756522743, 1.48546110879198, 1.8372251949458, 
    2.05, 2, 2.30762935376866, 1.33, 1.69719520353264, 1.62, 
    1.33, 2.65307599048676, 1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 
    1.73, 1.96, 1.78, 1.58, 2.356766356055, 1.34689342230938, 
    1.75, 1.68, 1.75, 1.97795936497418, 1.39883002725606, 2.91838830845348, 
    2.22337025759756, 1.97482837780464, 2.13158223995554, 1.32406282222445, 
    2.76340700342626, 2.22820613880148, 1.56, 1.62, 1.6), prol = c(1065, 
    1050, 892.63010989369, 935.571120074397, 735, 1450, 1290, 
    1295, 1045, 1045, 1242.6217324171, 1280, 1320, 1555.07254649847, 
    1547, 1310, 1280, 1212.49663011771, 1680, 845, 1319.81061264238, 
    770, 1035, 1015, 845, 1257.88376027063, 1195, 1285, 915, 
    1035, 1285, 1208.3506383014, 903.051461734146, 1235, 1124.56305893728, 
    920, 880, 1095.3520033317, 940.260311805869, 760, 795, 1035, 
    1095, 680, 885, 1080, 1065, 985, 1060, 1231.93797353206, 
    1150, 1265, 1190, 1081.61562114366, 1060, 1120, 970, 1270, 
    1285, 401.708793305181, 635.988137481728, 703.105751147502, 
    630, 490.73202016002, 645.186733635904, 1044.10484231364, 
    502, 545.421048021988, 532.734471423705, 718, 870, 410, 472, 
    998.897558736849, 886, 545.347193628704, 392, 777.592966406163, 
    1087.36736530579, 463, 278, 714, 662.261643556531, 515, 1223.77015614099, 
    796.325100390627, 49.4787131955443, 562, 680, 625, 480, 450, 
    495, 459.662562820159, 345, 696.421061140855, 625, 428, 660, 
    406, 602.642111815915, 562, 438, 415, 672, 582.304231841038, 
    510, 611.165265235856, 312, 680, 562, 325, 666.241650933504, 
    434, 385, 706.322228158984, 441.728646122389, 541.830330323832, 
    372, 564, 349.50865671116, 1249.02969936957, 365, 380, 656.057714220997, 
    378, 552.098434902738, 466, 315.900674671074, 580, 1113.65373348142, 
    560.418897883655, 560, 600, 793.196343307815, 695, 720, 515, 
    580, 590, 600, 596.234984462667, 520, 550, 600.318724273143, 
    830, 415, 746.144912876279, 548.2239885392, 550, 500, 480, 
    425, 675, 640, 883.923408102412, 480, 880, 660, 598.176807648505, 
    452.972137814953, 680, 570, 675.790990494458, 615, 299.445271104259, 
    695, 685, 610.268118567053, 749.989034056268, 510, 470, 818.11805645727, 
    740, 627.996478855753, 835, 840, 583.053784342456)), row.names = c(NA, 
    -178L), class = "data.frame"), structure(list(alco = c(14.23, 
    13.2, 13.16, 14.7882379795619, 13.24, 14.1861048958039, 14.39, 
    14.06, 13.9456489888515, 13.86, 13.6130917033022, 14.4017336208512, 
    13.75, 14.75, 14.38, 13.63, 14.3, 13.83, 14.0481858443269, 
    13.64, 13.234642158162, 12.93, 13.71, 12.85, 13.5, 14.012917489641, 
    13.39, 13.3, 13.87, 14.1402349151204, 13.73, 13.58, 13.68, 
    13.76, 13.3771969219002, 13.9133842364, 13.120026069491, 
    13.05, 11.5386345788937, 14.22, 13.56, 13.41, 13.88, 13.24, 
    13.05, 14.21, 13.4222779192955, 13.9, 13.2968605526261, 13.4779563184208, 
    13.05, 13.2180322733555, 13.82, 13.8628154793284, 13.74, 
    14.9873215583333, 14.22, 13.29, 14.1462622690679, 12.37, 
    12.33, 12.64, 11.2210727498613, 12.37, 12.17, 12.37, 13.4513357160663, 
    11.8037585092432, 13.34, 12.21, 12.6569589429067, 13.86, 
    12.0331811944696, 12.99, 11.96, 11.66, 11.7852480335391, 
    11.84, 13.9835951735063, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 
    12.1200654936338, 12.16, 12.7374756545636, 12.2434001938597, 
    12.08, 12.08, 12, 12.69, 12.7181343877849, 11.62, 13.7884138093905, 
    11.81, 12.29, 14.268130792769, 12.29, 12.08, 12.6, 12.7989521304119, 
    12.7275954028922, 12.51, 12.138543244942, 12.0273277355878, 
    12.72, 12.22, 11.61, 11.5242437244287, 12.1214066472598, 
    11.76, 12.2862063070144, 12.08, 11.03, 11.82, 12.7104690336372, 
    12.2149961856002, 11.9962945410751, 11.45, 12.7139652404527, 
    12.42, 13.05, 11.87, 12.07, 11.858625206265, 11.79, 12.2980727687315, 
    12.04, 12.86, 12.88, 13.0055157795447, 12.7, 12.51, 12.7328279272309, 
    12.25, 12.53, 11.8651794140102, 12.84, 12.93, 13.36, 13.52, 
    13.62, 12.25, 12.8644190885425, 12.9043387218984, 12.87, 
    13.32, 11.8214109431107, 12.1654072178516, 12.79, 12.4667996167863, 
    13.23, 13.4547629135166, 13.17, 13.84, 12.45, 14.34, 13.48, 
    12.729684076715, 13.69, 12.85, 12.96, 13.78, 13.73, 13.2544779334432, 
    13.3538063115808, 13.58, 13.4, 12.9521341885481, 12.77, 14.16, 
    13.6143552768108, 13.4, 13.27, 13.17, 14.13), malic = c(1.71, 
    2.28870009751914, 0.323158038672311, 1.95, 2.55384713776104, 
    2.72925923736678, 1.87, 2.15, 2.53699227779454, 1.35, 2.16, 
    1.48, 0.607719688045241, 1.73, 1.87, 1.02169572716106, 1.92, 
    4.75147969833458, 0.156735908129039, 3.1, 1.63, 3.8, 1.84688992504502, 
    1.6, 1.81, 1.10545531778818, 1.09623828016446, 2.37582476338249, 
    1.9, 1.67593384401425, 1.5, 1.57875117319718, 1.83, 1.53, 
    2.1919970947661, 2.23972434751912, 1.43354351035889, 1.65, 
    1.5, 2.67065676197916, 1.71, 3.94618503007692, 2.72369224222513, 
    3.98, 0.600495992177426, 4.04, 0.929776518983813, 1.68, 1.22956876295047, 
    0.508880241177494, 1.23354172996049, 1.65, 2.89304301868294, 
    1.9, 1.67, 3.26447142038422, 1.7, 1.97, 2.38885487152929, 
    0.94, 1.97645352941904, 1.36, 1.25, 0.954010454495864, 3.14587689180291, 
    1.21, 1.01, 1.17, 2.83413652808651, 3.8587235307953, 2.23706901515381, 
    1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 1.04933212032949, 0.99, 
    3.87, 2.16214243226775, 2.93397108427221, 1.13, 3.76613893520055, 
    2.88877647558588, 0.98, 1.61, 1.67, 2.81096183416511, 1.89493831103586, 
    2.43023868956203, 1.51, 1.76367267304229, 3.27773892560668, 
    2.58591493378346, 1.67297051944677, 3.90474435135626, 1.41, 
    2.14425733881583, 1.91046773498294, 2.08, 0.782915164957916, 
    2.93123733176747, 1.72, 2.25828097583032, 2.55, 1.73, 2.5168578667631, 
    1.29, 1.4383836094579, 3.74, 2.2117841182534, 2.68, 0.74, 
    1.39, 0.205066491128616, 3.50507832694762, 1.61, 3.43, 2.56264671197419, 
    2.4, 2.05, 4.43, 2.89191588938206, 4.31, 2.16, 1.53, 2.13, 
    3.12774561149621, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 4.35503417553596, 
    2.46094972236331, 5.51, 3.59, 3.95500799471758, 2.81, 2.56, 
    3.74396931741495, 3.67499411596186, 3.88, 2.26534093969181, 
    5.04, 4.61, 3.24, 2.99250938279667, 3.12, 2.67, 1.9, 3.84375055909396, 
    1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.0932166127576, 
    4.39328963464758, 3.45, 2.76, 2.52590760570174, 1.81737557047681, 
    3.37, 2.58, 4.6, 2.78644753008532, 2.39, 3.9790086896092, 
    5.65, 3.91, 4.30576095416976, 2.59, 4.1), ash = c(2.43, 2.14, 
    2.67, 2.5, 2.20978931088028, 2.50281735140737, 2.45, 2.61, 
    2.55465067105362, 2.59677552909367, 2.3, 2.53490170514589, 
    2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.67830429603487, 
    2.65, 2.60857560258163, 2.52, 2.61, 3.22, 2.76772916632617, 
    2.51913050047989, 3.07580592768427, 2.55359827242427, 2.41627991548406, 
    2.36, 2.894639575078, 2.7, 2.65, 3.04490142071839, 1.98162160206007, 
    2.55, 2.1, 2.51, 2.37357120279206, 2.12, 2.59, 2.29, 2.1, 
    2.44, 2.28736573212584, 2.12, 2.4, 2.27, 2.04, 2.422841865929, 
    2.24443376482833, 2.46105330873697, 2.57209713447841, 2.43495310901515, 
    2.3, 2.68, 2.5, 1.36, 2.59975862392728, 2.43969636467986, 
    1.92, 2.16, 2.53, 2.56, 2.23859723888784, 1.92, 2.25705141242093, 
    1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 2.20734630220881, 1.71, 
    2.02419357192624, 2.52990565680768, 2.4, 2, 2.66461443522375, 
    2.51, 2.32, 2.58, 2.24, 2.15134966951743, 2.62, 2.06747057866829, 
    2.3, 2.32, 2.41592705134185, 2.26, 2.22, 2.28, 2.2, 2.74, 
    1.98, 2.34379211422295, 1.91578808750291, 1.7, 2.19835422078783, 
    2.46, 1.88, 2.50852681233493, 2.27, 2.12, 2.28, 1.94, 2.7, 
    1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.21168987250519, 
    2.06280546773188, 2, 2.42, 3.23, 2.42982297016603, 2.13, 
    2.39, 2.17, 2.29, 2.81025590619738, 2.26522690603701, 2.38, 
    2.08897695047917, 2.4, 2.4, 2.36, 2.25, 2.38824686159415, 
    2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.39187098989337, 
    2.22583787216127, 2.48, 2.27582895222495, 2.36, 2.62, 2.48, 
    1.75302386558619, 2.28, 2.57089990400871, 2.32, 2.38, 2.08136562769115, 
    2.17794207095821, 2.64, 2.38, 2.54, 2.88924522415583, 2.35, 
    2.3, 2.02159954683099, 2.6, 2.08074851934186, 2.69, 2.29031842646061, 
    2.32, 2.28, 2.75932054008045, 2.21192125765415, 2.48, 2.26, 
    2.37, 2.74), alca = c(15.6, 13.1180045332016, 18.8093819713918, 
    16.8, 21, 15.2, 18.0781500940239, 17.6, 14, 16, 17.400208068628, 
    16.8, 16, 11.4, 14.9702676129991, 18.0106916577655, 14.1733702698287, 
    20, 11.8947426344264, 15.2, 16, 18.6, 19.5858987414273, 17.8, 
    20, 22.1013815753661, 16.1, 17, 21.4535883366227, 14.842023051839, 
    17.5768106487872, 19.1, 14.7336599276119, 17.979798809109, 
    19, 20.5, 15.5, 20.8399812983216, 15.5, 13.2, 16.2, 18.8, 
    17.9617318993523, 22.4463656302518, 17, 13.1400669564164, 
    16, 13.7205789632138, 18.8, 17.4, 12.4, 16.385974452807, 
    12.3640278428432, 16.4606746528055, 16.5011571880681, 20.5, 
    16.3, 16.8, 16.7755606818031, 10.6, 23.1641199238386, 16.8, 
    18, 19, 19, 18.1, 19.9109490711865, 19.6, 19.4235736607926, 
    16.8, 20.4, 24.4623428788759, 24, 30, 21, 21.8467968458024, 
    16, 18, 14.8, 21.2428469808327, 19, 18.9104563691634, 24, 
    22.5, 18, 18, 22.8, 22.6807769022727, 21.6, 20.1002009142648, 
    21.1377679174228, 22, 23.1209552344657, 18, 18, 19, 21.5, 
    16, 19.3907824853804, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
    19, 22.5, 19, 20, 19.5, 18.2467607322381, 24.8339794201401, 
    25.1173320185135, 18.7494131297563, 21.5, 20.8, 18.4334322182979, 
    16, 19.9641937659493, 16.0104417455725, 20.5803561561318, 
    20.4564383711994, 21.5, 21, 21, 21.5, 28.5, 20.6779085400118, 
    22, 16.4393260762662, 20, 24, 21.5, 19.3519318461741, 16.467562853975, 
    17.2165560280639, 25, 19.5, 24.5144753348718, 21, 19.2391762240876, 
    23.5, 20, 22.5474201774562, 21, 20, 21.9215056884594, 18.1639531895307, 
    21.5, 24, 22, 13.2018625137295, 18.5, 20, 22, 19.5, 14.4451964909574, 
    25, 24.6838903387796, 17.2988257096632, 23.9893898709941, 
    27.3803466160562, 18.5, 21.4170863152923, 22.5, 23, 19.5, 
    24.5, 25, 24.389275834477, 19.5, 24.4077990568853, 20.5, 
    23, 20, 21.004501384384, 24.5), mg = c(127, 100, 101, 113, 
    78.4286395605749, 112, 96, 121, 97, 109.265555643494, 116.774364658037, 
    115.270554351095, 89, 91, 102, 112, 120, 120.343802157498, 
    110.91024334298, 116, 126, 102, 106.033074053257, 95, 96.0219835892472, 
    124, 93, 111.608353898419, 107, 96, 101, 106, 104, 89.7897449050771, 
    110, 113.77059519869, 110, 135.737813877588, 70.2425666828389, 
    130.292135485042, 117, 125.360431130665, 101, 80.4738911969617, 
    107, 111, 102, 101, 126.47716392211, 108, 92, 102.174018900591, 
    111, 115, 109.793812283659, 137.650145407361, 118, 102, 108, 
    88, 101, 100, 94, 87, 104, 98, 78, 78, 85.6340562487956, 
    151, 82.8190596711693, 86, 87, 139, 101, 101.448947128825, 
    85.4999530059461, 112, 106.935177098289, 107.229378358327, 
    86, 103.947425524885, 78, 85, 82.6261689368827, 99, 90, 88, 
    84, 70, 81, 117.118969876943, 73.0598001110935, 88, 111.626120916264, 
    90.6619783923801, 134, 85, 88, 77.3779008287125, 97, 88, 
    81.8312676928067, 86, 85, 90, 80, 84, 92, 94, 107, 88, 103, 
    88, 84, 85, 86, 91.0257794004935, 92.942693107069, 87, 96, 
    112.513440994655, 102, 86, 76.5615334027849, 85, 60.3451146008685, 
    92, 88, 80, 122, 93.5497179346968, 98, 106, 78.8824010389639, 
    94, 89, 102.831605388044, 88, 101, 94.965918462203, 89, 97, 
    103.721309512837, 112, 102, 99.3593427760933, 113.310395283378, 
    92, 113, 123, 96.7772449700643, 116, 98, 103, 118.694134714775, 
    89, 97, 98, 89, 106.795811100958, 85.8215310268283, 106, 
    106, 90, 88, 118.585921659647, 88, 105, 112, 96, 86, 91, 
    95, 102, 101.230324517021, 120, 106.972154369811), phe = c(2.8, 
    2.65, 2.8, 3.85, 2.8, 3.02496061503862, 2.5, 2.6, 3.09520016380483, 
    2.98, 2.95, 2.2, 2.6, 3.1, 3.07607613026282, 2.85, 2.8, 3.2259625394288, 
    3.3, 2.76057087627381, 3, 2.41, 2.61, 2.48, 2.53, 2.89991159374916, 
    3.07136245551583, 2.4, 2.95, 2.65, 3, 3.3167946864705, 2.42, 
    2.95, 2.47223426926002, 2.7, 2.6, 2.45, 2.80166012125891, 
    3.24303918728312, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 
    3.1, 2.75, 2.88, 2.72, 2.45, 3.88, 2.91085817545721, 2.6377737696052, 
    2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.07433235593193, 3.5, 
    1.89, 3.08288107033733, 2.60849780551427, 2.11, 2.53, 3.42229597093067, 
    1.1, 2.95, 1.88, 2.29231067724225, 1.90936839123986, 1.67294956161576, 
    2.63876647583107, 1.86868006381509, 1.9, 2.83, 2.4086334509695, 
    2.2, 1.76979895536668, 1.65, 1.99568783981888, 2.2, 2.0704599102681, 
    1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.37732488521757, 
    1.52082994223317, 2.55, 3.52, 3.09044755698295, 2.21821508525119, 
    1.95285167872063, 2.56, 2.5, 2.2, 2.22886102148067, 1.65, 
    2.43852339996975, 2.36, 2.74, 3.18, 2.55, 1.75, 1.8774346645058, 
    2.56, 2.46, 2.03222127776409, 2, 2.4834744979205, 2, 2.9, 
    3.18, 2.25398348491022, 1.23803153222887, 2.86, 2.84434184150143, 
    2.74, 2.13, 2.22, 2.1, 1.51, 2.09915398692524, 1.04353790157351, 
    1.7, 1.03320057122621, 1.62, 1.38, 1.5004555392534, 1.62, 
    1.18820828133635, 1.70542029965953, 1.4, 1.21391507746837, 
    2, 1.38, 1.32045080648782, 0.98, 1.7, 1.93, 1.41, 1.4, 1.48, 
    2.2, 1.40421714644708, 1.48, 1.51648008504642, 1.8, 1.9, 
    1.89990219500084, 1.25851262232488, 2.3, 1.83, 1.17747726107638, 
    1.39, 1.35, 1.25723385502364, 1.7, 1.48, 1.55, 1.65424628320296, 
    1.25, 1.39, 1.68, 1.55755652323844, 1.23145350449344, 1.46814030850604, 
    1.65, 2.05), fla = c(2.75063510582594, 2.76, 3.45726287881867, 
    3.49, 2.41987148462466, 3.39, 2.52, 2.51, 2.98, 3.42557407526181, 
    3.32, 2.43, 2.76, 2.71115563700541, 3.64, 2.91, 3.14, 3.4, 
    3.38417275302925, 3.12263765640605, 3.5510922511755, 2.41, 
    2.88, 2.37, 2.61, 2.68, 2.94, 2.65799051938484, 2.97, 2.33, 
    3.25, 3.19, 2.69, 3.30759494071156, 2.54038105921335, 2.98, 
    2.57109937942944, 2.43, 2.64, 3.04, 3.29, 2.68, 3.56, 2.63, 
    3, 2.65, 3.17, 3.07879504270087, 2.92, 3.54, 3.09045088106712, 
    2.99, 3.74, 2.79, 2.9, 2.78, 2.90536576483442, 3.23, 3.67, 
    0.57, 2.12647342926097, 1.41, 1.79, 3.11067125886702, 1.75, 
    2.65, 1.58305104811392, 2, 1.3, 3.92665840910765, 1.02, 2.86, 
    1.36055887002182, 2.89, 2.14, 1.57, 2.00312904457201, 1.3966247237055, 
    1.85, 2.97191046613999, 2.26, 2.53, 1.58, 1.59, 2.21, 2.02944702083962, 
    1.69, 1.61, 1.69, 2.13705341905299, 1.39909089280911, 1.25, 
    1.46, 3.20537396605033, 2.64157880253067, 2.27, 0.99, 1.85887322959182, 
    3.52037721622872, 2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 
    2.03, 1.76, 1.71277876451837, 2.92, 3.14367285184518, 2.27, 
    2.03, 2.01, 2.29, 2.17, 1.6, 2.09, 1.52565518827832, 1.64, 
    3.24135038910534, 4.37761163741656, 2.01625212795121, 0.348125786979278, 
    3.03, 2.65, 3.15, 2.24, 2.0859458060643, 1.75, 0.821270098854624, 
    1.43195591873053, 0.815967164465479, 1.2, 0.58, 0.66, 0.47, 
    0.695282872225907, 0.915872611872577, 0.6, 0.5, 0.349849843621699, 
    0.52, 0.8, 1.09740470721443, 0.55, 0.34, 0.65, 0.76, 1.39, 
    1.57, 1.36, 1.28, 0.83, 0.58, 0.63, 1.08435699685239, 0.58, 
    1.31, 1.1, 0.92, 1.19392493600469, 0.6, 0.579052834448508, 
    0.68, 0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.74583723899247, 
    0.7, 0.61, 0.849766837688722, 0.69, 0.68, 0.76), nfla = c(0.28, 
    0.192003726129585, 0.3, 0.24, 0.39, 0.344881354028354, 0.215872198901418, 
    0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.245879644031235, 
    0.193208605293696, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.229469416522564, 
    0.377708953805465, 0.34, 0.27, 0.412230343171592, 0.26, 0.29, 
    0.243400837872372, 0.42, 0.444778246860985, 0.29, 0.26, 0.136364411257925, 
    0.0567685931247233, 0.28, 0.2, 0.34, 0.180941103578169, 0.256810575635374, 
    0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 
    0.39, 0.21, 0.2, 0.26, 0.281380039503556, 0.0784371754464746, 
    0.28, 0.314211407743816, 0.53, 0.32, 0.398385189770785, 0.45, 
    0.37, 0.26, 0.27, 0.4225945152996, 0.168254713527602, 0.35012160576605, 
    0.354398255269779, 0.52794012761591, 0.21, 0.13, 0.34, 0.24, 
    0.43, 0.35, 0.43, 0.3, 0.34436090288754, 0.4, 0.61, 0.22, 
    0.3, 0.366884605459837, 0.272235385156871, 0.360209804664528, 
    0.445582235697674, 0.500249217071017, 0.5, 0.243518090568608, 
    0.25, 0.17, 0.32, 0.506016729800227, 0.460446606286673, 0.24, 
    0.435565047713413, 0.311155619510858, 0.29, 0.34, 0.37, 0.403362609348924, 
    0.66, 0.259850340654783, 0.48, 0.39, 0.29, 0.24, 0.26, 0.54798795470161, 
    0.42, 0.344220308319755, 0.52, 0.3, 0.166542377320732, 0.167234838698917, 
    0.415049499993182, 0.32, 0.253344014617747, 0.43, 0.3, 0.558947738631298, 
    0.250691997688012, 0.39, 0.360647886068305, 0.4, 0.318445086229597, 
    0.178817118026036, 0.57920005061871, 0.27, 0.17, 0.6, 0.63, 
    0.53, 0.63, 0.58, 0.53, 0.53, 0.45403722273339, 0.448029159413634, 
    0.29407845174477, 0.379131038901975, 0.43, 0.4, 0.47, 0.45, 
    0.457426516313833, 0.22, 0.278835498276157, 0.26, 0.61, 0.53, 
    0.42362582413516, 0.53000844228387, 0.63, 0.53, 0.52, 0.408697649636382, 
    0.5, 0.6, 0.4, 0.41, 0.407152687361817, 0.43, 0.4, 0.39, 
    0.351480606615877, 0.314791214676589, 0.48, 0.494353065642504, 
    0.509842831408165, 0.43, 0.43, 0.53, 0.563451774365414), 
        pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 
        1.98, 1.85, 2.38, 1.57, 1.81, 2.61419510981585, 2.96, 
        1.46, 2.18180106442451, 1.58378831984752, 1.86, 1.66, 
        2.1, 0.512022774358078, 1.69, 1.46, 1.66, 1.92, 1.45, 
        1.35, 1.76, 1.98, 2.38, 1.95, 1.71053677911509, 1.75238361269284, 
        1.54, 1.86, 1.36, 1.44, 1.37, 2.64543536717851, 2.12101888144995, 
        1.68616510924658, 1.53758834124188, 1.71250905168923, 
        2.03, 0.931785838810586, 2.19, 2.14, 2.38, 2.08, 1.49944737016387, 
        2.29, 1.87, 1.68, 1.62, 2.45, 1.68649046419126, 1.66, 
        1.83115754278479, 0.42, 1.96318160362933, 1.82934714719366, 
        0.73, 1.87, 1.03, 1.96529250322904, 0.693884771882943, 
        1.04, 1.17747709624378, 2.5, 1.46, 1.87, 1.03, 1.65145492403597, 
        1.93926540643754, 1.15, 1.46, 1.9848293168083, 2.76, 
        1.95, 1.43, 1.96429755287819, 1.4803314656767, 1.62, 
        0.570355371848282, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
        1.63, 1.639834474316, 1.99, 1.35, 3.28, 1.56, 1.77, 1.95, 
        2.81, 1.4, 1.35, 1.31, 1.42, 0.821864612564567, 1.42, 
        2.09701989158088, 1.63, 2.08, 2.49, 1.82479291011283, 
        0.510597477761409, 1.63046386745844, 2.33119217237805, 
        1.04, 1.46562439383456, 0.632449549298013, 1.61, 0.83, 
        1.87, 1.83, 1.64283138440073, 1.71, 1.05222490642817, 
        2.91, 1.35, 1.77, 1.80653032549622, 1.9, 0.700503865826036, 
        0.94, 1.43539124027294, 0.83, 0.84, 1.25, 0.94, 0.8, 
        1.1, 0.88, 1.5570114813993, 0.75, 1.22008845466584, 0.55, 
        0.509314225803019, 1.14, 1.3, 0.68, 0.86, 1.25, 1.14, 
        1.25, 1.26, 1.56, 1.72778361804919, 1.00742916332152, 
        1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.51695654132911, 
        1.23002310790288, 0.94, 1.72355148478294, 1.15, 1.46, 
        0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
        1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.82645457796619, 
        6.75, 4.06227698508778, 5.05, 4.50143152309565, 7.22, 
        5.75, 8.93987033078502, 5.6, 7.35308713886883, 7.5, 5.7961450345509, 
        7.95001798418812, 2.96041918955541, 8.7, 5.1, 5.65, 4.5, 
        3.73585462165824, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 
        5.7, 5.78688632026802, 5.25121303665358, 5.4, 4.2, 5.1, 
        4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 
        4.9, 6.1, 6.30203939368666, 8.9, 7.2, 4.67740238900209, 
        6.11880310345485, 6.3, 5.85, 6.25, 7.15740269457369, 
        4.82034608999949, 6.8, 6.90488634624748, 3.27, 5.75, 
        3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 4.22732763917728, 
        3.38, 3.74, 4.10562222207537, 3.21, 3.8, 4.6, 3.8769884300926, 
        4.91596965685424, 3.58998387393584, 2.5, 3.9, 1.78430203823659, 
        4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 
        2.15, 3.25, 2.6, 4.90435438638578, 2.9, 5.09530110561168, 
        3.85539447326044, 4.98632872817546, 4.51623614458939, 
        2.8, 5.19916195830347, 2.94, 2.7, 3.4, 3.3, 2.7, 0.910238096705336, 
        2.9, 5.40597557555117, 3.15580599884229, 3.08, 2.9, 1.9, 
        1.55418850623828, 5.50265149876654, 3.4, 1.28, 3.25, 
        4.04315103554779, 0.888066254650163, 2.1103913814822, 
        2.8, 3.74653397780484, 3.94, -0.849627771542023, 2.12, 
        1.17053724705917, 9.00690323030775, 9.00849068959986, 
        6.11610328922744, 5, 5.45, 7.1, 7.05762431552137, 5, 
        5.7, 4.92, 7.14689816475096, 2.76338630101778, 4.35, 
        4.4, 8.21, 4, 4.9, 8.02556806609971, 8.42, 9.4, 4.57988253213029, 
        5.60683813335161, 7.1, 10.52, 7.6, 7.9, 6.15240630083345, 
        7.5, 13, 4.74298013214299, 7.65, 4.75260808525378, 5.58, 
        8.55190089679011, 4.66940915510994, 6.62, 10.68, 10.26, 
        3.95753096477924, 8.5, 3.61441905788521, 5.96352996840038, 
        9.7, 7.7, 7.3, 7.9074159789968, 9.3, 9.2), hue = c(1.04, 
        1.05, 1.28550933205833, 0.86, 0.972908566398182, 1.05, 
        1.02, 1.14631739494375, 1.08, 1.16661385206197, 1.25, 
        0.839794978959476, 1.15, 1.25, 1.2, 1.28, 0.879117656164233, 
        1.13, 1.23, 1.07067068656632, 1.09, 0.916008472655054, 
        1.11, 1.09, 1.12, 1.13, 0.813649213314057, 1.02, 1.12171521438111, 
        1.04, 1.19, 1.09, 1.31053691347187, 1.25, 1.1, 1.16504113618176, 
        1.11653269905697, 0.827498564648551, 1.18, 0.853754404776948, 
        0.95, 0.91, 0.88, 0.82, 1.21631667163955, 0.87, 1.2628633361508, 
        1.08310865670057, 1.07, 1.12, 1.12, 1.24858639374672, 
        0.90727492407245, 1.06656541514666, 1.13393727000253, 
        0.775736281456715, 0.94, 1.07, 0.89, 0.841200748716247, 
        1.25, 0.98, 1.23, 1.22, 0.891206276370194, 1.02539958299397, 
        1.08457469548207, 1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 
        1.31, 0.95059239836255, 1.23, 1.19, 0.96, 1.06, 1.19, 
        1.38, 0.997042837442641, 1.31, 0.84, 0.80550676175146, 
        1.23, 1.15719559860259, 0.790205097828081, 0.886861031228246, 
        1.07, 1.08, 1.05, 1.07552872339596, 1.15, 1.04237838451456, 
        1.16561129578945, 0.819558804388257, 1.23626701815869, 
        0.739348694978906, 1.42, 1.27, 0.928899737087903, 0.8, 
        0.764346633833947, 1.04, 1.30599737121356, 1, 0.911583121005702, 
        0.86, 1.17396532537685, 0.870775796134011, 0.9, 0.919268191585246, 
        1.1, 0.93, 1.71, 0.95, 0.925452787180973, 0.7, 0.988918349899895, 
        0.8, 1.0760662539104, 0.92, 0.73, 1.23451421576249, 0.86, 
        0.69, 0.97, 0.89, 0.79, 0.658307087348913, 0.74, 0.66, 
        0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.772844460780219, 
        0.679322737264503, 0.654204305175752, 0.89, 0.743450184173861, 
        0.65, 0.6, 0.277454065813563, 0.54, 0.55, 0.938434688796168, 
        0.688594079310759, 0.48, 0.61, 0.56, 0.477433762565698, 
        0.6, 0.57, 0.67, 0.57, 0.873747119439224, 0.56, 0.96, 
        0.87, 0.68, 0.7, 0.78, 0.85, 0.640808804538126, 0.74, 
        0.67, 0.546488174213264, 0.807530584323418, 0.516986828786458, 
        0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 
        2.83103149697653, 2.93, 3.2726974527116, 3.58, 3.58, 
        2.85, 3.55, 3.17, 2.82, 2.9, 2.37198146654179, 3, 2.7920969245548, 
        2.65, 3.62215351530758, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 
        2.81512641195773, 3.2, 2.83746322380935, 2.77, 2.94317882711239, 
        2.64802439922594, 2.71, 2.88, 2.87, 3, 2.87, 2.93107231657605, 
        3.64347034692005, 2.87815409737423, 2.69, 3.0133111209259, 
        3.38, 3, 3.73035006620999, 3, 3.35, 3.33, 2.90609822159416, 
        3.33, 2.75, 3.1, 2.5558500362453, 2.30362916921454, 3.26, 
        2.93, 3.2, 3.03, 3.23264350150646, 2.84, 2.87, 1.82, 
        2.67563223609208, 1.59, 2.46, 2.87, 2.98000097635954, 
        2.3, 3.18, 3.48, 2.39661455173291, 3.7698309844459, 1.82, 
        3.21106162548619, 2.54216957075176, 3.5, 3.13, 2.14, 
        2.48, 2.52, 2.32486845549337, 3.22129376615075, 3.20325179319334, 
        3.14, 2.72, 2.01, 2.75298325764739, 3.36277221008469, 
        2.70659047896407, 3.21, 2.75, 3.21, 2.32629612550937, 
        2.65, 3.14756072655829, 3.3, 2.77002720875992, 2.63, 
        2.42840406141286, 2.74, 3.68990232406762, 2.83, 2.96, 
        2.77, 3.38, 2.42603001886302, 3.57, 3.3, 3.17, 2.42, 
        3.02, 3.26, 2.81, 2.78, 2.5, 3.31233134565699, 3.19, 
        2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 2.42020941367093, 
        3.64, 3.01096183477547, 2.84, 3.49756604227474, 2.78, 
        2.52740597454741, 1.19726774983271, 1.42, 1.36, 1.29, 
        1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.29654544425996, 
        1.78095959859791, 1.65787630774655, 2.05, 2, 1.53059694733805, 
        1.33, 1.34070806335585, 1.62, 1.33, 2.31871702430656, 
        1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
        1.58, 2.20626993819098, 1.64603257090006, 1.75, 1.68, 
        1.75, 1.58618594014455, 1.09466566783576, 2.17843133667131, 
        1.94150759478818, 1.98706785861693, 1.79443126180233, 
        0.974242531798576, 1.80393754818406, 2.61267535742029, 
        1.56, 1.62, 1.6), prol = c(1065, 1050, 803.23169203573, 
        1137.30130079188, 735, 1450, 1290, 1295, 1045, 1045, 
        853.62436156275, 1280, 1320, 1398.58708190801, 1547, 
        1310, 1280, 1007.90388416967, 1680, 845, 1356.9894470174, 
        770, 1035, 1015, 845, 1356.11249717148, 1195, 1285, 915, 
        1035, 1285, 810.167711550548, 1447.64118182253, 1235, 
        1076.46957814597, 920, 880, 1152.20813401961, 789.574532203399, 
        760, 795, 1035, 1095, 680, 885, 1080, 1065, 985, 1060, 
        1219.07067975653, 1150, 1265, 1190, 1154.43980995479, 
        1060, 1120, 970, 1270, 1285, 735.102308716343, 256.574597681266, 
        888.943267991511, 630, 728.623330939013, 768.893491984608, 
        658.721800016855, 502, 392.232385688241, 491.347968540875, 
        718, 870, 410, 472, 735.08345105379, 886, 605.282294741399, 
        392, 182.925988136993, 1155.41734004822, 463, 278, 714, 
        577.321281085199, 515, 675.456053499745, 1176.87796625898, 
        669.231997099383, 562, 680, 625, 480, 450, 495, 948.567481661721, 
        345, 905.708341237795, 625, 428, 660, 406, 513.778641848376, 
        562, 438, 415, 672, 308.916028256076, 510, 55.3518469774806, 
        312, 680, 562, 325, 326.320550598889, 434, 385, 272.066841068465, 
        255.616731129711, 864.914507123092, 372, 564, 624.214499570849, 
        1340.6281787595, 365, 380, 764.09431624601, 378, 810.562355621452, 
        466, 334.25950650665, 580, 628.794210740941, 956.887848469011, 
        560, 600, 829.6183624437, 695, 720, 515, 580, 590, 600, 
        819.257617529059, 520, 550, 414.791781555709, 830, 415, 
        665.407964994081, 790.243245405906, 550, 500, 480, 425, 
        675, 640, 469.387547954288, 480, 880, 660, 830.127245616988, 
        713.501049983438, 680, 570, 760.872721706495, 615, 387.173365892672, 
        695, 685, 573.649240342428, 647.281478047993, 510, 470, 
        754.564394905673, 740, 805.518545493371, 835, 840, 463.241618875185
        )), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 15.6114646692452, 
        13.24, 14.2201818465998, 14.39, 14.06, 13.8528811660818, 
        13.86, 13.0089326084036, 13.8298628590906, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 15.6243783917044, 13.64, 13.3143204668827, 
        12.93, 13.71, 12.85, 13.5, 14.1326534453925, 13.39, 13.3, 
        13.87, 13.4244120657752, 13.73, 13.58, 13.68, 13.76, 
        12.6560226423586, 13.8904247750335, 13.14372307472, 13.05, 
        11.6384605170218, 14.22, 13.56, 13.41, 13.88, 13.24, 
        13.05, 14.21, 15.2355289509562, 13.9, 14.6708681371062, 
        13.746362417633, 13.05, 12.9912042654637, 13.82, 13.7065898228379, 
        13.74, 15.0933960769327, 14.22, 13.29, 14.1001062316664, 
        12.37, 12.33, 12.64, 11.5658468898568, 12.37, 12.17, 
        12.37, 12.3654917274318, 12.5383748447339, 13.34, 12.21, 
        12.5351941901551, 13.86, 12.6431844531134, 12.99, 11.96, 
        11.66, 12.0771797972426, 11.84, 12.4132321641902, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.4600703933361, 12.16, 
        12.7687085641907, 12.9335609772908, 12.08, 12.08, 12, 
        12.69, 12.5029541711002, 11.62, 12.4371755049936, 11.81, 
        12.29, 13.0281984912924, 12.29, 12.08, 12.6, 12.9226713607441, 
        12.5416564466252, 12.51, 12.9354504934471, 12.0746914469272, 
        12.72, 12.22, 11.61, 13.0443867709036, 12.5132053069826, 
        11.76, 11.7364670329173, 12.08, 11.03, 11.82, 11.4625006203684, 
        12.5209101590021, 11.8075407463978, 11.45, 15.4140493060421, 
        12.42, 13.05, 11.87, 12.07, 11.4480808350771, 11.79, 
        11.5316270881575, 12.04, 12.86, 12.88, 13.385130803158, 
        12.7, 12.51, 12.3867957399783, 12.25, 12.53, 13.249094959489, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 13.2758310817062, 
        11.6190649919942, 12.87, 13.32, 11.7504082154126, 12.0171362350263, 
        12.79, 11.2800254425147, 13.23, 11.5344812909663, 13.17, 
        13.84, 12.45, 14.34, 13.48, 12.6800007527548, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.2876197457501, 13.1727591198703, 
        13.58, 13.4, 12.2330591442261, 12.77, 14.16, 13.8019396994474, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.63424919030751, 
        -0.776730442252869, 1.95, 2.21645627872691, 2.20110796870466, 
        1.87, 2.15, 1.88997826236767, 1.35, 2.16, 1.48, 1.28826469287194, 
        1.73, 1.87, 3.37096870466481, 1.92, -0.278109177703096, 
        2.16792031392951, 3.1, 1.63, 3.8, 2.22844937645926, 1.6, 
        1.81, 2.71738952388285, 1.62832088226258, 1.26919927670835, 
        1.9, 2.4214499490981, 1.5, 0.170439213852588, 1.83, 1.53, 
        0.995626431158108, 3.0800838830739, 1.00398439079141, 
        1.65, 1.5, 3.16758770406126, 1.71, 2.75383833644687, 
        2.0640127666784, 3.98, -0.756393954849654, 4.04, 3.55277669343035, 
        1.68, 1.98354527985954, 1.19616537353065, 0.804661031769977, 
        1.65, 3.43454544298386, 1.9, 1.67, 2.77060258468974, 
        1.7, 1.97, 3.24497254147763, 0.94, 1.96124712276678, 
        1.36, 1.25, 1.66134306916704, 3.22461531454409, 1.21, 
        1.01, 1.17, 1.98454430677383, 1.70351219471333, -0.924378040747437, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.21480807239869, 
        0.99, 3.87, 3.08925223426537, 3.165420662841, 1.13, 4.18703074910203, 
        2.24952470780755, 0.98, 1.61, 1.67, 3.04230034090562, 
        2.60040518616535, 2.16691714610354, 1.51, 2.01029893541545, 
        1.62794686762329, 2.15510320547261, 2.55222380914944, 
        3.72190429679178, 1.41, 2.24580270481424, -0.17285733323041, 
        2.08, 2.06471634609936, 3.61719351692051, 1.72, 1.37901835768139, 
        2.55, 1.73, 2.08159035323672, 1.29, 0.876311354046731, 
        3.74, 2.94199673055269, 2.68, 0.74, 1.39, -1.0656367451708, 
        2.39152003942207, 1.61, 3.43, 1.47238663859078, 2.4, 
        2.05, 4.43, 2.40166167278459, 4.31, 2.16, 1.53, 2.13, 
        1.24296246918438, 4.3, 1.35, 2.99, 2.31, 3.55, 1.24, 
        5.11922383511286, 1.71197493477582, 5.51, 3.59, 5.32180735108766, 
        2.81, 2.56, 2.95702614506379, 3.8079649294485, 3.88, 
        2.55295022730687, 5.04, 4.61, 3.24, 3.67997377136101, 
        3.12, 2.67, 1.9, 3.20888823703477, 1.29, 5.19, 4.12, 
        3.03, 1.68, 1.67, 3.83, 3.82274394420774, 2.94047453044249, 
        3.45, 2.76, 4.11307307296485, 1.57531162926219, 3.37, 
        2.58, 4.6, 2.09316589364279, 2.39, 4.15203361178, 5.65, 
        3.91, 2.64399293536618, 2.59, 4.1), ash = c(2.43, 2.14, 
        2.67, 2.5, 2.29461984627594, 2.74479266066017, 2.45, 
        2.61, 2.45841827445169, 2.33579267035678, 2.3, 2.47092139885057, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 3.0616432906625, 
        2.65, 2.63697599259676, 2.52, 2.61, 3.22, 2.63633105447822, 
        2.86448163952573, 2.4954657274412, 2.56172633857976, 
        2.66383295876355, 2.36, 2.65404500043523, 2.7, 2.65, 
        2.57044446983272, 2.20133052359454, 2.55, 2.1, 2.51, 
        2.47787376936887, 2.12, 2.59, 2.29, 2.1, 2.44, 2.64245659347866, 
        2.12, 2.4, 2.27, 2.04, 2.14021442188941, 2.50737222441806, 
        2.77447805542215, 2.10918008012758, 2.4664716220147, 
        2.3, 2.68, 2.5, 1.36, 2.21743930318246, 2.52750958485961, 
        1.92, 2.16, 2.53, 2.56, 2.59862876601129, 1.92, 2.37872492203056, 
        1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.57257816897016, 1.71, 
        2.43836892696332, 1.93713436760471, 2.4, 2, 2.04541942858994, 
        2.51, 2.32, 2.58, 2.24, 2.42842958657328, 2.62, 2.53731345087935, 
        2.3, 2.32, 2.19901393510174, 2.26, 2.22, 2.28, 2.2, 2.74, 
        1.98, 2.43800553414642, 1.92035519597071, 1.7, 2.2209399743826, 
        2.46, 1.88, 2.27059765753528, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.54513841808949, 
        1.50939818537217, 2, 2.42, 3.23, 1.9911274146022, 2.13, 
        2.39, 2.17, 2.29, 2.97845144107404, 1.96274299163692, 
        2.38, 2.41659548725007, 2.4, 2.4, 2.36, 2.25, 2.07429679498307, 
        2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.53260649296287, 
        2.34433346877418, 2.48, 2.55641921598027, 2.36, 2.62, 
        2.48, 1.90777518849689, 2.28, 2.34186518842978, 2.32, 
        2.38, 2.2874623962767, 2.65418046237081, 2.64, 2.38, 
        2.54, 2.69640819060969, 2.35, 2.3, 2.83054315582494, 
        2.6, 2.36274689299171, 2.69, 2.80608552107681, 2.32, 
        2.28, 2.40948731255645, 2.72168800145048, 2.48, 2.26, 
        2.37, 2.74), alca = c(15.6, 10.2773412615315, 14.1828896811393, 
        16.8, 21, 15.2, 12.9052818902791, 17.6, 14, 16, 20.6777549518187, 
        16.8, 16, 11.4, 14.2176041996623, 16.5602809818062, 19.3191691750729, 
        20, 7.63522239354662, 15.2, 16, 18.6, 15.9460378668797, 
        17.8, 20, 20.3240016189139, 16.1, 17, 13.138307954912, 
        20.1652820142877, 16.7980068852438, 19.1, 17.7812832497091, 
        17.6762005130375, 19, 20.5, 15.5, 21.8891898254565, 15.5, 
        13.2, 16.2, 18.8, 15.6332610813019, 20.3784740695438, 
        17, 11.9246910555007, 16, 14.497439952878, 18.8, 17.4, 
        12.4, 13.198039137707, 14.772810082289, 19.6955052952294, 
        13.1691231582674, 20.5, 16.3, 16.8, 15.9086405768288, 
        10.6, 20.1925515586706, 16.8, 18, 19, 19, 18.1, 20.751534115819, 
        19.6, 23.2555870213291, 16.8, 20.4, 22.6547255431285, 
        24, 30, 21, 17.3413220838808, 16, 18, 14.8, 25.640833187454, 
        19, 17.1198336423834, 24, 22.5, 18, 18, 22.8, 22.3170550737138, 
        21.6, 16.2164577410243, 19.0000718296165, 22, 29.355799658112, 
        18, 18, 19, 21.5, 16, 14.9745295931811, 18, 17.5, 18.5, 
        21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 19.9914785611084, 
        22.0857156418386, 23.9266838259214, 23.4331668971688, 
        21.5, 20.8, 15.7715078663178, 16, 16.27740704533, 20.4198736915077, 
        21.2625446184754, 20.7620584197747, 21.5, 21, 21, 21.5, 
        28.5, 16.1176799252038, 22, 29.3358563579639, 20, 24, 
        21.5, 23.2879193880663, 18.3401312418921, 21.6328798309467, 
        25, 19.5, 20.6008707033981, 21, 21.6654923625991, 23.5, 
        20, 22.6994007197394, 21, 20, 22.1648818759219, 17.5300543669877, 
        21.5, 24, 22, 15.456246854959, 18.5, 20, 22, 19.5, 14.7298754098645, 
        25, 26.3468464954838, 17.2748896466804, 24.2557371586574, 
        26.9012468925271, 18.5, 20.4458238685818, 22.5, 23, 19.5, 
        24.5, 25, 26.5508265089808, 19.5, 22.4117926199371, 20.5, 
        23, 20, 23.8474258672868, 24.5), mg = c(127, 100, 101, 
        113, 94.7144312840625, 112, 96, 121, 97, 111.980265328953, 
        86.3038580703723, 120.736663963947, 89, 91, 102, 112, 
        120, 80.5502282055545, 82.7437826334593, 116, 126, 102, 
        120.240992932046, 95, 112.12843546461, 124, 93, 123.761974947286, 
        107, 96, 101, 106, 104, 87.0995884926497, 110, 120.198963161166, 
        110, 74.2575679791198, 75.0880010231079, 93.3984949812894, 
        117, 106.674134764598, 101, 90.4925668965758, 107, 111, 
        102, 101, 98.0664013792742, 108, 92, 109.239554265948, 
        111, 115, 109.787489211613, 119.662777336259, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 111.61144390631, 
        151, 113.154432040436, 86, 87, 139, 101, 103.129669191656, 
        80.4335184879369, 112, 78.0943023718415, 95.769160674428, 
        86, 123.710390893742, 78, 85, 96.0780820495759, 99, 90, 
        88, 84, 70, 81, 100.431796505144, 102.767475693991, 88, 
        87.0246320667219, 103.190221645565, 134, 85, 88, 87.5527454888241, 
        97, 88, 108.092087811864, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 98.7666714064965, 87.7271725355287, 
        87, 96, 101.124828621488, 102, 86, 87.4777160355728, 
        85, 88.0238435035393, 92, 88, 80, 122, 100.219746657362, 
        98, 106, 97.9704110744429, 94, 89, 116.75736283804, 88, 
        101, 93.3495669233275, 89, 97, 98.2589421431721, 112, 
        102, 104.701716410963, 125.633238496846, 92, 113, 123, 
        90.3622979191259, 116, 98, 103, 113.939158595015, 89, 
        97, 98, 89, 130.17056182627, 107.634657425054, 106, 106, 
        90, 88, 95.9512609223797, 88, 105, 112, 96, 86, 91, 95, 
        102, 100.866426990365, 120, 111.458473210016), phe = c(2.8, 
        2.65, 2.8, 3.85, 2.8, 2.9892342467516, 2.5, 2.6, 2.82791816547559, 
        2.98, 2.95, 2.2, 2.6, 3.1, 3.43634360769037, 2.85, 2.8, 
        2.8175673176919, 3.3, 3.07105900644891, 3, 2.41, 2.61, 
        2.48, 2.53, 2.84285175049713, 2.83743514173893, 2.4, 
        2.95, 2.65, 3, 2.92479553370943, 2.42, 2.95, 2.65640218143008, 
        2.7, 2.6, 2.45, 2.62135326986456, 2.90821649944919, 3.15, 
        2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 
        2.45, 3.88, 2.33504060376725, 2.80042298361154, 2.96, 
        3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.14424999884235, 3.5, 
        1.89, 1.94272309214772, 3.04229236499924, 2.11, 2.53, 
        2.67021963353107, 1.1, 2.95, 1.88, 2.82392599924872, 
        2.07044300065831, 2.1024647941627, 2.44515288543285, 
        1.91222027907737, 1.9, 2.83, 2.33304643772956, 2.2, 1.68442294744066, 
        1.65, 1.99653451372232, 2.2, 1.69798940578496, 1.92, 
        1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.24926223395734, 
        1.49010452122639, 2.55, 3.52, 3.18680749728026, 2.7679762077897, 
        1.62414314830529, 2.56, 2.5, 2.2, 2.43802662603336, 1.65, 
        2.11143520670059, 2.36, 2.74, 3.18, 2.55, 1.75, 1.80561585604463, 
        2.56, 2.46, 1.95589523682074, 2, 1.48851271750229, 2, 
        2.9, 3.18, 3.01590812350824, 2.41643434733889, 2.86, 
        2.02403961104933, 2.74, 2.13, 2.22, 2.1, 1.51, 1.24573380569991, 
        1.14082227792992, 1.7, 1.10086393249253, 1.62, 1.38, 
        1.73860153777682, 1.62, 1.85462235680075, 1.65293721934646, 
        1.4, 1.58670742138016, 2, 1.38, 1.52791832037773, 0.98, 
        1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.79202860658626, 1.48, 
        1.44148685910248, 1.8, 1.9, 1.6490039233655, 1.55257771348613, 
        2.3, 1.83, 1.63029186251833, 1.39, 1.35, 1.74122364303336, 
        1.7, 1.48, 1.55, 1.73518828425924, 1.25, 1.39, 1.68, 
        1.8974123724288, 1.52348364022109, 1.35988177202151, 
        1.65, 2.05), fla = c(2.86012573475873, 2.76, 3.32651127961389, 
        3.49, 1.9031293332123, 3.39, 2.52, 2.51, 2.98, 3.35465154126724, 
        3.32, 2.43, 2.76, 2.41478582213271, 3.64, 2.91, 3.14, 
        3.4, 2.63830705801627, 3.42603737204319, 3.80036541866666, 
        2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.20245701099052, 
        2.97, 2.33, 3.25, 3.19, 2.69, 3.29390739184661, 3.08725728641901, 
        2.98, 2.51979855821696, 2.43, 2.64, 3.04, 3.29, 2.68, 
        3.56, 2.63, 3, 2.65, 3.17, 3.25416060885832, 2.92, 3.54, 
        3.481223269871, 2.99, 3.74, 2.79, 2.9, 2.78, 3.06409300254211, 
        3.23, 3.67, 0.57, 1.16001002017759, 1.41, 1.79, 3.41087445697052, 
        1.75, 2.65, 3.55134046627733, 2, 1.3, 2.45685792880423, 
        1.02, 2.86, 1.68490923045063, 2.89, 2.14, 1.57, 1.98287898620036, 
        1.42420609068284, 1.85, 2.33281603993826, 2.26, 2.53, 
        1.58, 1.59, 2.21, 1.99061723954254, 1.69, 1.61, 1.69, 
        2.21028052508926, 1.80072535947506, 1.25, 1.46, 2.0897855919333, 
        3.22710902275549, 2.27, 0.99, 2.07869900448295, 3.634808832929, 
        2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 
        1.95201222880984, 2.92, 2.34042346389783, 2.27, 2.03, 
        2.01, 2.29, 2.17, 1.6, 2.09, 0.790305556886548, 1.64, 
        3.59143561715232, 3.49755722336648, 2.18214293077671, 
        2.14935425234671, 3.03, 2.65, 3.15, 2.24, 1.90917868239614, 
        1.75, 0.58915219549845, -0.0589690365404243, 0.200020834471462, 
        1.2, 0.58, 0.66, 0.47, 0.665854661402259, 1.00689737699346, 
        0.6, 0.5, 0.603290003552872, 0.52, 0.8, 0.908905616762405, 
        0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
        0.58, 0.63, 0.544722309121781, 0.58, 1.31, 1.1, 0.92, 
        1.19658099839017, 0.6, 0.374965000237021, 0.68, 0.47, 
        0.92, 0.66, 0.84, 0.96, 0.49, 0.675985422586386, 0.7, 
        0.61, 0.499502373564576, 0.69, 0.68, 0.76), nfla = c(0.28, 
        0.253251415249587, 0.3, 0.24, 0.39, 0.1854377539488, 
        0.351993263350957, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 
        0.43, 0.29, 0.3, 0.417596489972257, 0.39126564780642, 
        0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.32406400419807, 
        0.43813297562616, 0.34, 0.27, 0.156211065286167, 0.26, 
        0.29, 0.234717556688679, 0.42, 0.292952847429445, 0.29, 
        0.26, 0.213187827479919, 0.222199756501411, 0.28, 0.2, 
        0.34, 0.368774753944577, 0.339430825237132, 0.32, 0.28, 
        0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 
        0.21, 0.2, 0.26, 0.325207722763428, 0.459577193819195, 
        0.28, 0.41644259265861, 0.53, 0.32, 0.16639545426961, 
        0.45, 0.37, 0.26, 0.27, 0.33003791356433, 0.283707478265952, 
        0.159248786110357, 0.490912830779199, 0.415420387401016, 
        0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.316454450419779, 
        0.4, 0.61, 0.22, 0.3, 0.384926366483987, 0.451438927814774, 
        0.343978922268601, 0.250884981078133, 0.246180179173654, 
        0.5, 0.370497543503926, 0.25, 0.17, 0.32, 0.318262792806265, 
        0.276296740017895, 0.24, 0.213838647046333, 0.286724379649068, 
        0.29, 0.34, 0.37, 0.487955537386054, 0.66, 0.236137524739571, 
        0.48, 0.39, 0.29, 0.24, 0.26, 0.301458131120961, 0.42, 
        0.588204114393442, 0.52, 0.3, 0.188104998789876, 0.280837806714801, 
        0.314751523912532, 0.32, 0.272432567236527, 0.43, 0.3, 
        0.333722267963788, 0.026808149312435, 0.39, 0.4990071605613, 
        0.4, 0.547336095289198, 0.386864245156354, 0.551584260837644, 
        0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 
        0.485659666069468, 0.593299975416824, 0.555085848857796, 
        0.458818166497824, 0.43, 0.4, 0.47, 0.45, 0.289727757736233, 
        0.22, 0.39173343233616, 0.26, 0.61, 0.53, 0.356001460665866, 
        0.650703254314282, 0.63, 0.53, 0.52, 0.248016049820638, 
        0.5, 0.6, 0.4, 0.41, 0.485760276616462, 0.43, 0.4, 0.39, 
        0.427746761312056, 0.567280494947528, 0.48, 0.443714447932485, 
        0.535240392685694, 0.43, 0.43, 0.53, 0.479743206016821
        ), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 
        1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.94366803565211, 
        2.96, 1.46, 2.97499957868489, 2.1748282798651, 1.86, 
        1.66, 2.1, 0.866738041198274, 1.69, 1.46, 1.66, 1.92, 
        1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.59891375703482, 
        1.88041372546284, 1.54, 1.86, 1.36, 1.44, 1.37, 1.26732527147237, 
        1.79005205789094, 2.15062560873745, 1.95392057304393, 
        1.71114355351041, 2.03, 1.1423102631219, 2.19, 2.14, 
        2.38, 2.08, 2.07918000629419, 2.29, 1.87, 1.68, 1.62, 
        2.45, 2.11014588564392, 1.66, 2.3743942900023, 0.42, 
        0.767179377671363, 2.01342298168267, 0.73, 1.87, 1.03, 
        1.81455988484629, 1.86136604306669, 1.04, 1.32055363877941, 
        2.5, 1.46, 1.87, 1.03, 1.14834279768405, 1.54770317159058, 
        1.15, 1.46, 1.15674691607224, 2.76, 1.95, 1.43, 1.19010920325655, 
        1.49237398229497, 1.62, 1.29244573881389, 1.46, 1.56, 
        1.34, 1.35, 1.38, 1.64, 1.63, 1.99690266406754, 1.99, 
        1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 
        1.42, 1.6866050502624, 1.42, 1.71941699232761, 1.63, 
        2.08, 2.49, 1.8657061477664, 1.45332107427335, 0.942654716685708, 
        1.62301859145361, 1.04, 1.96999143844038, 0.459708325304826, 
        1.61, 0.83, 1.87, 1.83, 2.4293134881551, 1.71, 1.58537934362057, 
        2.91, 1.35, 1.77, 1.53563445960919, 1.9, 1.76905418413595, 
        0.94, 0.550725301948942, 0.83, 0.84, 1.25, 0.94, 0.8, 
        1.1, 0.88, 0.700315424531156, 0.75, 0.447298035570978, 
        0.55, 1.05926735352008, 1.14, 1.3, 0.68, 0.86, 1.25, 
        1.14, 1.25, 1.26, 1.56, 1.60431249560246, 0.476579391028327, 
        1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 2.18501972449819, 
        2.10854464289182, 0.94, 1.39001947969711, 1.15, 1.46, 
        0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
        1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 2.65659332557705, 
        6.75, 6.39405345775852, 5.05, 6.2697113044977, 7.22, 
        5.75, 7.117015529319, 5.6, 6.2979124182716, 7.5, 4.20998428107727, 
        7.13174923206989, 5.48271375538223, 8.7, 5.1, 5.65, 4.5, 
        4.89270798094489, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 
        5.7, 6.71727438862335, 6.0867392877146, 5.4, 4.2, 5.1, 
        4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 
        4.9, 6.1, 7.46697799315698, 8.9, 7.2, 5.92471050097493, 
        3.95748686908483, 6.3, 5.85, 6.25, 7.18216096433949, 
        4.90587035317549, 6.8, 5.57507471424354, 3.27, 5.75, 
        3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 8.70682585235686, 
        3.38, 3.74, 2.13574670956833, 3.21, 3.8, 4.6, 1.58069991814932, 
        3.55192252674536, 0.488778352877487, 2.5, 3.9, 1.86569078741545, 
        4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 
        2.15, 3.25, 2.6, 6.90030179230038, 2.9, 1.42347315508015, 
        4.69586518209698, 2.95377882582133, 3.71315354119883, 
        2.8, 1.46051393220223, 2.94, 2.7, 3.4, 3.3, 2.7, 1.38350701464991, 
        2.9, 2.44349782716661, 2.79157743245204, 3.08, 2.9, 1.9, 
        0.933615236173091, 2.39344326007954, 3.4, 1.28, 3.25, 
        6.69338084042654, -0.0777334423244076, 3.70293658184597, 
        2.8, 4.60478448475508, 3.94, -0.00949392754758716, 2.12, 
        -0.378971734143652, 3.43256539296659, 8.21470579328793, 
        9.85957400435106, 5, 5.45, 7.1, 8.60899712398753, 5, 
        5.7, 4.92, 6.78817913857956, 10.0926102109927, 4.35, 
        4.4, 8.21, 4, 4.9, 7.62954884883998, 8.42, 9.4, 9.21648779784635, 
        10.6269739646288, 7.1, 10.52, 7.6, 7.9, 9.25267802703071, 
        7.5, 13, 9.49897455650705, 7.65, 9.09617555987269, 5.58, 
        8.42629980214138, 11.4865860643931, 6.62, 10.68, 10.26, 
        9.4970994723599, 8.5, 8.80847544495237, 10.5528179688117, 
        9.7, 7.7, 7.3, 11.6030237596063, 9.3, 9.2), hue = c(1.04, 
        1.05, 1.28202602181109, 0.86, 1.07154901894294, 1.05, 
        1.02, 0.819547572904835, 1.08, 1.02052058900414, 1.25, 
        0.94439976994705, 1.15, 1.25, 1.2, 1.28, 1.19761609729792, 
        1.13, 1.23, 1.14260953043862, 1.09, 0.792828797678176, 
        1.11, 1.09, 1.12, 1.13, 1.12859386363781, 1.02, 1.08484377217421, 
        1.04, 1.19, 1.09, 0.972927712483146, 1.25, 1.1, 1.15747809765486, 
        1.2329836879557, 0.73380121004336, 1.18, 0.727676438748151, 
        0.95, 0.91, 0.88, 0.82, 1.39466340077386, 0.87, 0.87338253655898, 
        1.14064881681167, 1.07, 1.12, 1.12, 1.33944455085763, 
        1.1028497893771, 1.15873289197804, 0.937549911646595, 
        1.29730996069341, 0.94, 1.07, 0.89, 1.18471224751996, 
        1.25, 0.98, 1.23, 1.22, 0.81196357730362, 1.21896609341907, 
        1.05222085479588, 1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 
        1.31, 1.08688304845097, 1.23, 1.19, 0.96, 1.06, 1.19, 
        1.38, 1.10179750889597, 1.31, 0.84, 0.929116291240123, 
        1.23, 1.03052687435034, 0.989665575857064, 0.866347872882391, 
        1.07, 1.08, 1.05, 1.03748244954304, 1.15, 1.36674467411569, 
        1.10765368969074, 0.498627601369587, 1.12413040038418, 
        1.27739464838611, 1.42, 1.27, 0.918054103359373, 0.8, 
        0.985401202307415, 1.04, 1.00975927499933, 1, 0.884859718682752, 
        0.86, 0.992749780334514, 0.944288179964603, 0.9, 0.806696526156972, 
        1.1, 0.93, 1.71, 0.95, 0.953577924310695, 0.7, 0.897989023856557, 
        0.8, 0.999889328593087, 0.92, 0.73, 1.09407656800056, 
        0.86, 0.69, 0.97, 0.89, 0.79, 1.07605676218687, 0.74, 
        0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.832624331712086, 
        0.629341730436947, 0.435796817774937, 0.89, 0.710002667028627, 
        0.65, 0.6, 0.409028205950267, 0.54, 0.55, 0.845213688154954, 
        0.521505404998795, 0.48, 0.61, 0.56, 0.655206517295612, 
        0.6, 0.57, 0.67, 0.57, 0.949448406642951, 0.56, 0.96, 
        0.87, 0.68, 0.7, 0.78, 0.85, 0.546939770493829, 0.74, 
        0.67, 0.826310116086908, 0.711014048416124, 0.83258474773195, 
        0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 
        3.73600015669162, 2.93, 3.47373710604151, 3.58, 3.58, 
        2.85, 3.55, 3.17, 2.82, 2.9, 2.91045158035785, 3, 2.77380187229265, 
        2.65, 3.34840335227841, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 
        3.01098317055629, 3.2, 2.75497676630308, 2.77, 3.52935996708876, 
        3.04461800371007, 2.71, 2.88, 2.87, 3, 2.87, 3.04449787850259, 
        3.0890245510333, 3.08579233906651, 2.69, 3.67507522802499, 
        3.38, 3, 3.71948178633647, 3, 3.35, 3.33, 4.01254274118507, 
        3.33, 2.75, 3.1, 2.97074260710256, 2.84894418272333, 
        3.26, 2.93, 3.2, 3.03, 3.06803164277322, 2.84, 2.87, 
        1.82, 2.33520889263463, 1.59, 2.46, 2.87, 2.64126183671637, 
        2.3, 3.18, 3.48, 2.76584096062672, 2.64588991164163, 
        1.82, 3.0678813132085, 3.02878351175215, 3.5, 3.13, 2.14, 
        2.48, 2.52, 3.037373374384, 3.01853130861475, 2.84849947761091, 
        3.14, 2.72, 2.01, 2.65572193609808, 2.69865416031756, 
        2.78822814262901, 3.21, 2.75, 3.21, 2.8972201713964, 
        2.65, 3.10540456501675, 3.3, 2.80624575890614, 2.63, 
        1.44318448946013, 2.74, 3.47271806963373, 2.83, 2.96, 
        2.77, 3.38, 3.12278910646049, 3.57, 3.3, 3.17, 2.42, 
        3.02, 3.26, 2.81, 2.78, 2.5, 2.35197052698859, 3.19, 
        2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 3.0026838459932, 
        3.64, 2.95933509742579, 2.84, 2.68182135516612, 2.78, 
        2.51876390516356, 2.141708506163, 1.42, 1.36, 1.29, 1.51, 
        1.58, 1.27, 1.69, 1.82, 2.15, 1.94152795273731, 2.19927021587626, 
        2.26682821482795, 2.05, 2, 1.64756971452019, 1.33, 1.51767806243552, 
        1.62, 1.33, 1.88289705291996, 1.47, 1.33, 1.51, 1.55, 
        1.48, 1.64, 1.73, 1.96, 1.78, 1.58, 2.0491795990191, 
        1.76093682417339, 1.75, 1.68, 1.75, 1.4152891080551, 
        1.05760311898312, 1.90774758627777, 1.62121362766566, 
        1.09074676239948, 1.33477444228164, 2.0120018079442, 
        1.23784859616502, 1.80629344548648, 1.56, 1.62, 1.6), 
            prol = c(1065, 1050, 1423.86649396011, 1505.55948808698, 
            735, 1450, 1290, 1295, 1045, 1045, 679.535183228284, 
            1280, 1320, 1461.85503948402, 1547, 1310, 1280, 1073.13582779702, 
            1680, 845, 1537.94072260953, 770, 1035, 1015, 845, 
            1102.39830886567, 1195, 1285, 915, 1035, 1285, 984.992833853764, 
            994.364585354562, 1235, 810.266261835829, 920, 880, 
            440.578081818249, 1044.86976438756, 760, 795, 1035, 
            1095, 680, 885, 1080, 1065, 985, 1060, 1043.94840818377, 
            1150, 1265, 1190, 818.924767290176, 1060, 1120, 970, 
            1270, 1285, 592.833623572582, 337.990160187688, 922.278816833972, 
            630, 459.087278137198, 806.91071308423, 1105.72465990236, 
            502, 398.100677540564, 399.385357963615, 718, 870, 
            410, 472, 578.967928427712, 886, 246.382253387891, 
            392, 890.999453920507, 841.466857613351, 463, 278, 
            714, 592.119587224358, 515, 519.549481822358, 758.332634908753, 
            445.544056048605, 562, 680, 625, 480, 450, 495, 648.988522657979, 
            345, 481.351264444357, 625, 428, 660, 406, 385.220815607573, 
            562, 438, 415, 672, 459.984716496452, 510, 462.951397829476, 
            312, 680, 562, 325, 1024.0014303768, 434, 385, 364.030967941166, 
            351.191895217217, 831.2201820067, 372, 564, 529.922677496532, 
            1105.37957098796, 365, 380, 590.2732071473, 378, 
            84.153647055767, 466, 550.094235265298, 580, 371.269575469588, 
            691.148006601539, 560, 600, 718.903223558328, 695, 
            720, 515, 580, 590, 600, 361.218604308406, 520, 550, 
            414.607439347163, 830, 415, 387.051487891937, 713.922027383862, 
            550, 500, 480, 425, 675, 640, 811.29398394672, 480, 
            880, 660, 575.947898972294, 496.297296461542, 680, 
            570, 761.868906121678, 615, 980.33742326854, 695, 
            685, 829.951034088368, 565.324270389287, 510, 470, 
            717.518296605821, 740, 644.625264424497, 835, 840, 
            704.605032200273)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.8294691529913, 
        13.24, 12.8243427627165, 14.39, 14.06, 13.2631677913804, 
        13.86, 13.3704330103177, 13.0356444382066, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.579320677628, 13.64, 13.9547149083943, 
        12.93, 13.71, 12.85, 13.5, 13.9348358213695, 13.39, 13.3, 
        13.87, 13.8347900424714, 13.73, 13.58, 13.68, 13.76, 
        13.5476920722553, 14.1014182517649, 12.3707065800401, 
        13.05, 11.7957212532256, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 13.4522718375016, 13.9, 13.7799360054433, 
        14.0977671739547, 13.05, 13.8370122577226, 13.82, 13.9691016978601, 
        13.74, 14.4501168473674, 14.22, 13.29, 14.6066389483825, 
        12.37, 12.33, 12.64, 11.7722723665341, 12.37, 12.17, 
        12.37, 12.2214424659522, 12.723486093341, 13.34, 12.21, 
        12.2899939005681, 13.86, 12.9079156904112, 12.99, 11.96, 
        11.66, 12.1783954677125, 11.84, 13.1306906033303, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 10.9827142419322, 12.16, 
        12.4772837241449, 12.4471662921656, 12.08, 12.08, 12, 
        12.69, 12.3817186606938, 11.62, 12.4079652234728, 11.81, 
        12.29, 13.3619389389115, 12.29, 12.08, 12.6, 13.2467237177995, 
        12.9343229013517, 12.51, 12.6366886950466, 11.9036033426239, 
        12.72, 12.22, 11.61, 12.5427375625765, 12.456589668225, 
        11.76, 11.9718706579857, 12.08, 11.03, 11.82, 11.2068754499156, 
        12.4554924829974, 12.2263199960887, 11.45, 12.8514669724283, 
        12.42, 13.05, 11.87, 12.07, 11.6767562780863, 11.79, 
        11.6042602611765, 12.04, 12.86, 12.88, 14.0642923756846, 
        12.7, 12.51, 12.6959490170022, 12.25, 12.53, 12.8306089375381, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 11.7497978555959, 
        11.5150584946978, 12.87, 13.32, 11.5630181996224, 11.4994738268957, 
        12.79, 12.3062513880516, 13.23, 12.1231817165663, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.558312664742, 13.69, 12.85, 
        12.96, 13.78, 13.73, 13.6185952618781, 13.7585625571953, 
        13.58, 13.4, 13.5022215692802, 12.77, 14.16, 14.0785290347363, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.49707369773019, 
        -0.0802868509917687, 1.95, 3.5606349409851, 0.808845115710245, 
        1.87, 2.15, 1.07304078233467, 1.35, 2.16, 1.48, 1.69694373553803, 
        1.73, 1.87, 1.32050236110253, 1.92, 2.06792934935236, 
        -0.0658636569423856, 3.1, 1.63, 3.8, 3.55510323140341, 
        1.6, 1.81, 2.85261052797988, 2.75962642068805, 3.45843158567191, 
        1.9, 1.60299604204137, 1.5, 0.747593763415982, 1.83, 
        1.53, 2.12266560095146, 1.87777940955545, 1.07970480532754, 
        1.65, 1.5, 1.96448195433002, 1.71, 1.89763742657025, 
        1.96852890157256, 3.98, -0.499422367972177, 4.04, -0.416255006859322, 
        1.68, 1.97530939107245, 1.95473450919321, -1.36956044404002, 
        1.65, -0.881684703575916, 1.9, 1.67, 0.654617514002779, 
        1.7, 1.97, 5.10169392900337, 0.94, 2.11201008954279, 
        1.36, 1.25, -0.147095151084097, 3.16438276271221, 1.21, 
        1.01, 1.17, 3.99789416727104, 0.198540036644739, -1.09498625995732, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 4.62165160745694, 
        0.99, 3.87, -0.112039633415792, 3.39681770547229, 1.13, 
        5.39678878066029, 2.7049243879202, 0.98, 1.61, 1.67, 
        1.86978431642209, 2.24908298994351, 1.73376804553852, 
        1.51, 5.44713029399153, 1.16290472179043, 0.125479671719992, 
        3.34545170203251, 3.88597883927013, 1.41, 0.821421658080289, 
        0.513754938094749, 2.08, 1.27305039740288, 4.76325671907839, 
        1.72, 1.93889190122078, 2.55, 1.73, 2.53480510634268, 
        1.29, -0.323881785221834, 3.74, 3.60537305074486, 2.68, 
        0.74, 1.39, -2.0065521187652, -0.538944658760583, 1.61, 
        3.43, 2.88679256675838, 2.4, 2.05, 4.43, 3.38491227371604, 
        4.31, 2.16, 1.53, 2.13, 2.31881317516722, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 3.54230916700009, 2.78980413000018, 
        5.51, 3.59, 5.06747376365999, 2.81, 2.56, 4.62877742547338, 
        2.81922725670734, 3.88, 5.4856515960438, 5.04, 4.61, 
        3.24, 0.900065245174796, 3.12, 2.67, 1.9, 5.37656804233256, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 4.05120276773888, 
        2.50630996330734, 3.45, 2.76, 3.95218486509268, 0.674019621160778, 
        3.37, 2.58, 4.6, 0.4714943783084, 2.39, 4.6607129791491, 
        5.65, 3.91, 3.85921207166366, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.56415076893218, 2.26910159719317, 
        2.45, 2.61, 2.34501253207858, 2.34045585341114, 2.3, 
        2.4457556303176, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 
        2.56, 2.80227668435561, 2.65, 2.82362242924509, 2.52, 
        2.61, 3.22, 2.57977826018108, 2.67049291013774, 2.33008252461029, 
        2.38702153081304, 2.60096262321088, 2.36, 2.37873336920974, 
        2.7, 2.65, 2.72213449141273, 2.35318351572078, 2.55, 
        2.1, 2.51, 1.9391725443624, 2.12, 2.59, 2.29, 2.1, 2.44, 
        2.66906862930696, 2.12, 2.4, 2.27, 2.04, 2.78463870597674, 
        2.20425534284839, 2.79800185056162, 2.59284231975921, 
        2.57076930497814, 2.3, 2.68, 2.5, 1.36, 2.41166644817745, 
        2.59327450396125, 1.92, 2.16, 2.53, 2.56, 1.96308489992072, 
        1.92, 2.23635183626623, 1.75, 2.21, 2.67, 2.24, 2.6, 
        2.3, 2.32944117734916, 1.71, 1.94203688613014, 2.07211893626532, 
        2.4, 2, 2.68867964394153, 2.51, 2.32, 2.58, 2.24, 2.46111600955792, 
        2.62, 2.47691494972012, 2.3, 2.32, 2.08384079427332, 
        2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 1.91866189558139, 
        2.16935725402846, 1.7, 2.05972885577625, 2.46, 1.88, 
        2.07483987545771, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 
        2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.68055160468452, 2.00073499045404, 
        2, 2.42, 3.23, 2.23909838801451, 2.13, 2.39, 2.17, 2.29, 
        2.81046379445109, 2.17287164126377, 2.38, 2.54055988723537, 
        2.4, 2.4, 2.36, 2.25, 2.34278570694555, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.84162237272875, 2.23292107980356, 
        2.48, 2.43445414012302, 2.36, 2.62, 2.48, 2.47522703298087, 
        2.28, 2.46862768422772, 2.32, 2.38, 2.03975346066193, 
        2.38155689052698, 2.64, 2.38, 2.54, 2.42579407580384, 
        2.35, 2.3, 2.56172012353829, 2.6, 2.62182252322209, 2.69, 
        2.60080092132434, 2.32, 2.28, 2.65987288097771, 2.42936986648197, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 15.4009957679017, 
        20.2486162154, 16.8, 21, 15.2, 11.596975751605, 17.6, 
        14, 16, 14.3687209432871, 16.8, 16, 11.4, 10.3091782052788, 
        16.9380217102493, 14.8213982865024, 20, 12.8198472845742, 
        15.2, 16, 18.6, 22.0107349583876, 17.8, 20, 21.0473033169523, 
        16.1, 17, 18.6909303895991, 17.7247290497303, 14.8639904415822, 
        19.1, 20.1923394052577, 11.2472897838512, 19, 20.5, 15.5, 
        18.6586509679717, 15.5, 13.2, 16.2, 18.8, 19.871260455149, 
        20.1091437671819, 17, 14.8932401743033, 16, 16.6960777613671, 
        18.8, 17.4, 12.4, 14.5637172644078, 13.3809770672208, 
        19.1395317287223, 20.7464220802332, 20.5, 16.3, 16.8, 
        11.1481188692502, 10.6, 22.51296038646, 16.8, 18, 19, 
        19, 18.1, 16.5688373187449, 19.6, 16.0190071522295, 16.8, 
        20.4, 23.0777586479627, 24, 30, 21, 21.3942951640454, 
        16, 18, 14.8, 26.7343453216526, 19, 17.7524382701649, 
        24, 22.5, 18, 18, 22.8, 21.4456681474311, 21.6, 19.106490539356, 
        22.0305601689495, 22, 22.7047472557896, 18, 18, 19, 21.5, 
        16, 18.0439737624937, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 23.5559591563571, 24.4688160474793, 
        19.8979105776274, 18.1103682745606, 21.5, 20.8, 22.2023131571169, 
        16, 21.4804022455172, 18.4638660472065, 21.1983293654661, 
        22.3542980736064, 21.5, 21, 21, 21.5, 28.5, 23.3682003292297, 
        22, 18.9874143371553, 20, 24, 21.5, 22.4499311011747, 
        26.9146774867015, 23.4943106907276, 25, 19.5, 27.3674854675389, 
        21, 22.8007423184424, 23.5, 20, 21.4252382863955, 21, 
        20, 24.3914053618848, 19.3585117082664, 21.5, 24, 22, 
        24.2552213009672, 18.5, 20, 22, 19.5, 15.546364235865, 
        25, 25.333569089643, 23.2527580722848, 18.7410086767296, 
        21.9267754842911, 18.5, 23.22365971088, 22.5, 23, 19.5, 
        24.5, 25, 20.0127128815087, 19.5, 22.313271050142, 20.5, 
        23, 20, 23.5799014055677, 24.5), mg = c(127, 100, 101, 
        113, 95.874963344884, 112, 96, 121, 97, 114.096440168617, 
        124.45210471479, 119.026338328511, 89, 91, 102, 112, 
        120, 137.853267558419, 98.6696499931089, 116, 126, 102, 
        121.133305887156, 95, 95.6675814985935, 124, 93, 131.649778650061, 
        107, 96, 101, 106, 104, 77.4732099000034, 110, 114.067580080789, 
        110, 112.215635272367, 90.8306013580583, 98.8545596654088, 
        117, 103.999438989638, 101, 89.8313074052998, 107, 111, 
        102, 101, 124.048808917551, 108, 92, 117.812367000187, 
        111, 115, 102.33288327356, 125.563244471782, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 87.7453057925892, 
        151, 80.2204710784213, 86, 87, 139, 101, 93.8998030131493, 
        71.5447142151699, 112, 78.9271666764509, 96.4640765064977, 
        86, 76.2058659292709, 78, 85, 98.2919328026454, 99, 90, 
        88, 84, 70, 81, 108.940619007623, 98.2480580590545, 88, 
        90.9242397898724, 119.792834017745, 134, 85, 88, 93.9441500381584, 
        97, 88, 102.53518635047, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 94.5254962114253, 69.5822037929094, 
        87, 96, 130.200827765036, 102, 86, 144.364841853584, 
        85, 87.9216276009135, 92, 88, 80, 122, 82.1800904801251, 
        98, 106, 93.0261586407082, 94, 89, 120.576210070308, 
        88, 101, 90.8597839488748, 89, 97, 95.7192392078999, 
        112, 102, 104.984516257915, 84.8837340674025, 92, 113, 
        123, 95.2136980995291, 116, 98, 103, 96.9265188110358, 
        89, 97, 98, 89, 95.7864653019893, 110.498392101979, 106, 
        106, 90, 88, 94.3386392865568, 88, 105, 112, 96, 86, 
        91, 95, 102, 102.801615743917, 120, 101.272915664916), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 2.41934599753885, 
            2.5, 2.6, 2.60595214065912, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.47489923847807, 2.85, 2.8, 3.00370166881643, 
            3.3, 3.00684661906409, 3, 2.41, 2.61, 2.48, 2.53, 
            2.66502366958683, 2.62473892513348, 2.4, 2.95, 2.65, 
            3, 2.66782742172398, 2.42, 2.95, 3.21382897002, 2.7, 
            2.6, 2.45, 2.84847616690573, 3.31266436027547, 3.15, 
            2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 
            2.72, 2.45, 3.88, 2.71248487983872, 2.36974134048477, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.36545367600366, 
            3.5, 1.89, 2.7098146929602, 2.01159795518632, 2.11, 
            2.53, 3.20600092784287, 1.1, 2.95, 1.88, 2.94036041017303, 
            1.92271003000794, 1.37745773002939, 1.56161797278635, 
            1.81682327579644, 1.9, 2.83, 3.25230075699247, 2.2, 
            1.81489009251795, 1.65, 2.04328917832422, 2.2, 1.989760721911, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.26477967856356, 
            1.91879946570357, 2.55, 3.52, 3.50869975928516, 2.96538210827451, 
            1.59481677459778, 2.56, 2.5, 2.2, 2.54087848550577, 
            1.65, 2.19570551087657, 2.36, 2.74, 3.18, 2.55, 1.75, 
            2.44492244923542, 2.56, 2.46, 1.89793408123424, 2, 
            2.43621612378986, 2, 2.9, 3.18, 2.02087814061821, 
            1.57703454627255, 2.86, 2.95251031193748, 2.74, 2.13, 
            2.22, 2.1, 1.51, 0.910236035130099, 1.41861052162141, 
            1.7, 2.29741088532522, 1.62, 1.38, 1.20150545831028, 
            1.62, 1.30219397342883, 1.50396985425659, 1.4, 2.74128657955122, 
            2, 1.38, 1.14260788490987, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.40340215469794, 1.48, 1.54861206198361, 
            1.8, 1.9, 1.47793329622099, 1.93476321370803, 2.3, 
            1.83, 2.16282988523183, 1.39, 1.35, 2.22987600593574, 
            1.7, 1.48, 1.55, 1.94251046033302, 1.25, 1.39, 1.68, 
            1.91652942822825, 1.73377583121266, 1.55862789706144, 
            1.65, 2.05), fla = c(2.89919758555502, 2.76, 3.66046213773645, 
            3.49, 2.55012904881712, 3.39, 2.52, 2.51, 2.98, 3.41969449869702, 
            3.32, 2.43, 2.76, 2.95265349851685, 3.64, 2.91, 3.14, 
            3.4, 3.03771514052307, 3.23637746055967, 3.20290444978165, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.46223215750316, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.25068947212731, 3.08506053980559, 
            2.98, 2.90079206917432, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.37544467568705, 2.92, 
            3.54, 3.17854837329687, 2.99, 3.74, 2.79, 2.9, 2.78, 
            2.85271983322703, 3.23, 3.67, 0.57, 2.24813365470416, 
            1.41, 1.79, 2.96806132794407, 1.75, 2.65, 1.98409271791504, 
            2, 1.3, 2.87850581773946, 1.02, 2.86, 1.24893045415002, 
            2.89, 2.14, 1.57, 1.37160210523712, 1.76840173232956, 
            1.85, 2.35639181363443, 2.26, 2.53, 1.58, 1.59, 2.21, 
            2.51985852185586, 1.69, 1.61, 1.69, 2.050526530486, 
            1.21392932478834, 1.25, 1.46, 2.94719559149587, 2.91599709767889, 
            2.27, 0.99, 2.00952557042533, 3.03338415975394, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.8856278125537, 
            2.92, 2.45916201397947, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.51585935072395, 1.64, 3.34271041234454, 
            3.57362721008476, 1.59300301137404, 1.09480601018058, 
            3.03, 2.65, 3.15, 2.24, 2.36829049006763, 1.75, 0.489625506713759, 
            0.395515133383065, 0.312109913751918, 1.2, 0.58, 
            0.66, 0.47, 0.156459958820635, 0.0745304576823289, 
            0.6, 0.5, 0.427360340258156, 0.52, 0.8, 1.03114866389673, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.321932018138647, 0.58, 1.31, 1.1, 0.92, 
            0.506541069441598, 0.6, 0.946171614939173, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.404334751378228, 
            0.7, 0.61, 0.788972435835575, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.282605234774452, 0.3, 0.24, 0.39, 
            0.23284439403575, 0.412406135591469, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.372662914703307, 
            0.29298862572475, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 
            0.340913675909293, 0.360082559220136, 0.34, 0.27, 
            0.299290273077465, 0.26, 0.29, 0.278633996225401, 
            0.42, 0.27722175310325, 0.29, 0.26, 0.424308686881554, 
            0.41688702642549, 0.28, 0.2, 0.34, 0.269795893963485, 
            0.265162731825083, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.330570371179393, 
            0.206820338663875, 0.28, 0.513455723601573, 0.53, 
            0.32, 0.320225552656906, 0.45, 0.37, 0.26, 0.27, 
            0.449449369806706, 0.292635542678943, 0.388028062871504, 
            0.30139519734178, 0.544342773581475, 0.21, 0.13, 
            0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.307124154813992, 
            0.4, 0.61, 0.22, 0.3, 0.425007078480563, 0.393129209561896, 
            0.319689973386679, 0.366078264064809, 0.455696904674498, 
            0.5, 0.451411223292936, 0.25, 0.17, 0.32, 0.461970486441147, 
            0.232226720012079, 0.24, 0.291237114336654, 0.251070598147283, 
            0.29, 0.34, 0.37, 0.327324957443843, 0.66, 0.277108472567196, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.36349806153403, 0.42, 
            0.299510208430404, 0.52, 0.3, 0.441751307061282, 
            0.493642113758094, 0.414360481819062, 0.32, 0.212951983313446, 
            0.43, 0.3, 0.529044072932065, 0.238315905763485, 
            0.39, 0.333036877078013, 0.4, 0.471541700657117, 
            0.358884073910173, 0.396371528039851, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.443003483457995, 
            0.58027610361161, 0.424416355857622, 0.358341956955672, 
            0.43, 0.4, 0.47, 0.45, 0.244879235996541, 0.22, 0.262702892822405, 
            0.26, 0.61, 0.53, 0.59681445753524, 0.724508111196761, 
            0.63, 0.53, 0.52, 0.348351127401696, 0.5, 0.6, 0.4, 
            0.41, 0.612249753109143, 0.43, 0.4, 0.39, 0.531035451811003, 
            0.364123608572181, 0.48, 0.546117224535506, 0.427489739949689, 
            0.43, 0.43, 0.53, 0.526286366927375), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 2.33962514945762, 2.96, 1.46, 2.08455574499845, 
            2.78796427521242, 1.86, 1.66, 2.1, 1.04607872292641, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 1.14650224772407, 1.84549112060087, 1.54, 1.86, 
            1.36, 1.44, 1.37, 1.91256960467556, 2.32658123175395, 
            2.0397982350954, 2.14627317455722, 2.14600599510344, 
            2.03, 1.19267085361703, 2.19, 2.14, 2.38, 2.08, 1.61165559326108, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.20815416825656, 1.66, 
            2.42807004442786, 0.42, 2.1007207907358, 1.48797864339783, 
            0.73, 1.87, 1.03, 2.66334562056107, 0.755391483391605, 
            1.04, 1.12022045038424, 2.5, 1.46, 1.87, 1.03, 1.72564931838878, 
            2.18076729052363, 1.15, 1.46, 1.38123763417152, 2.76, 
            1.95, 1.43, 1.74273611821593, 1.13691159917041, 1.62, 
            1.24923307484399, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 1.2417168064411, 1.99, 1.35, 3.28, 1.56, 1.77, 
            1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.45852600173191, 
            1.42, 1.58793296671948, 1.63, 2.08, 2.49, 1.89769200268366, 
            0.772969615700549, 1.80001010008309, 1.51180506160103, 
            1.04, 2.28415141170953, 1.38464906666583, 1.61, 0.83, 
            1.87, 1.83, 1.74264486949117, 1.71, 0.0129188602976269, 
            2.91, 1.35, 1.77, 1.6186348752488, 1.9, 0.393331158626263, 
            0.94, 1.21138587283761, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 1.38340689271107, 0.75, 1.15311908208085, 
            0.55, 1.44514756332994, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 0.87467312547857, 1.19406939018015, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 2.05040191231579, 
            2.02765598894122, 0.94, 1.14645475836147, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.17947264614673, 
            6.75, 6.3649969669321, 5.05, 5.99214842492299, 7.22, 
            5.75, 5.09152239104551, 5.6, 7.92406059837265, 7.5, 
            3.89988360422669, 8.46924741334822, 6.27372595036757, 
            8.7, 5.1, 5.65, 4.5, 2.22720422584974, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 7.40944747242261, 
            6.44898188755484, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            6.14631133030015, 8.9, 7.2, 6.05251207949204, 8.24669779059917, 
            6.3, 5.85, 6.25, 4.77312671740364, 4.35375255842865, 
            6.8, 5.08168099807083, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 11.3982990444992, 3.38, 
            3.74, 1.22970178046835, 3.21, 3.8, 4.6, 3.90307286419494, 
            7.02722668540422, 0.667076644089866, 2.5, 3.9, 2.3365001361176, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 5.93316484977779, 2.9, 8.16880073665821, 
            2.69327937388619, 2.60306370977745, 5.00148997981851, 
            2.8, 4.10496723113921, 2.94, 2.7, 3.4, 3.3, 2.7, 
            1.913125015241, 2.9, 3.11119446105947, 6.67980163292531, 
            3.08, 2.9, 1.9, 2.34017965272733, 0.574329504074268, 
            3.4, 1.28, 3.25, 1.79885673674649, 0.769688151110796, 
            4.22386579053571, 2.8, 2.3310622079289, 3.94, 1.42884862487147, 
            2.12, 0.317253554458057, 10.1805864405849, 8.10478885948586, 
            10.4969191947806, 5, 5.45, 7.1, 6.42592336405494, 
            5, 5.7, 4.92, 8.91392988470218, 8.39664515206194, 
            4.35, 4.4, 8.21, 4, 4.9, 5.73693259160536, 8.42, 
            9.4, 7.31893172185709, 10.0200525958904, 7.1, 10.52, 
            7.6, 7.9, 7.89370761155724, 7.5, 13, 8.37644731442176, 
            7.65, 8.08602115574841, 5.58, 10.8630814781806, 10.5801106317678, 
            6.62, 10.68, 10.26, 8.53614968883006, 8.5, 13.1623107829589, 
            8.7584197537109, 9.7, 7.7, 7.3, 8.81126174417007, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.12901459676332, 
            0.86, 0.84358142307674, 1.05, 1.02, 1.16442863819452, 
            1.08, 1.18897817745434, 1.25, 1.33800535202116, 1.15, 
            1.25, 1.2, 1.28, 1.03824312676182, 1.13, 1.23, 0.959615535238708, 
            1.09, 0.898242974570832, 1.11, 1.09, 1.12, 1.13, 
            1.1176028965151, 1.02, 1.10730581031263, 1.04, 1.19, 
            1.09, 1.11812761570291, 1.25, 1.1, 1.1917225624215, 
            0.940838496956985, 1.16048115252028, 1.18, 1.13886796708176, 
            0.95, 0.91, 0.88, 0.82, 1.26862507273297, 0.87, 1.21308080804793, 
            0.99069469764241, 1.07, 1.12, 1.12, 0.998235300560039, 
            1.39089637731058, 1.20164468713565, 0.974711974517476, 
            1.10231404214905, 0.94, 1.07, 0.89, 1.22077176490757, 
            1.25, 0.98, 1.23, 1.22, 0.964064423274699, 1.24813413511532, 
            1.13241303739219, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.11768800579342, 1.23, 1.19, 0.96, 1.06, 
            1.19, 1.38, 0.949713291713329, 1.31, 0.84, 0.992968087305582, 
            1.23, 1.06761497151061, 1.03653402711834, 1.2820130852467, 
            1.07, 1.08, 1.05, 0.672697196073036, 1.15, 0.950180765346357, 
            1.12911134456748, 0.737321431301575, 1.03154335659966, 
            1.17411149426858, 1.42, 1.27, 0.855707017971719, 
            0.8, 1.28606135983899, 1.04, 1.0442430922757, 1, 
            1.01000937715731, 0.86, 1.456354128717, 0.826372536640448, 
            0.9, 0.604519025762046, 1.1, 0.93, 1.71, 0.95, 1.2151674620643, 
            0.7, 1.01621008218841, 0.8, 1.10633709490225, 0.92, 
            0.73, 0.926379477870265, 0.86, 0.69, 0.97, 0.89, 
            0.79, 0.732778853366684, 0.74, 0.66, 0.78, 0.75, 
            0.73, 0.75, 0.82, 0.81, 0.495503055229021, 0.822374953565654, 
            0.605645877092292, 0.89, 0.967577590632454, 0.65, 
            0.6, 0.544114790582362, 0.54, 0.55, 0.534496874064239, 
            0.573525911556545, 0.48, 0.61, 0.56, 0.88659988266256, 
            0.6, 0.57, 0.67, 0.57, 0.916683256917856, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.401566392382091, 0.74, 
            0.67, 0.557392877473031, 0.6984944939436, 0.638797860694148, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 2.65969851005569, 2.93, 3.82541871715921, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.07304986368081, 
            3, 3.98369719817432, 2.65, 3.11939697141867, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.15087014979226, 3.2, 
            2.87184532553448, 2.77, 3.73237460843542, 2.32790852564968, 
            2.71, 2.88, 2.87, 3, 2.87, 3.34674973377184, 2.76552006580089, 
            2.20246092846767, 2.69, 3.30622038704713, 3.38, 3, 
            3.18331589352251, 3, 3.35, 3.33, 2.76837707396133, 
            3.33, 2.75, 3.1, 3.45303569557169, 3.30967071824058, 
            3.26, 2.93, 3.2, 3.03, 3.47288644418282, 2.84, 2.87, 
            1.82, 3.51883609870193, 1.59, 2.46, 2.87, 3.1098468406464, 
            2.3, 3.18, 3.48, 2.61040128451891, 3.24062998932576, 
            1.82, 2.92099419478601, 2.91109611692879, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.22384679485821, 3.25862457766639, 
            2.49455153415424, 3.14, 2.72, 2.01, 2.6154418274621, 
            3.51927527800371, 2.50676186908531, 3.21, 2.75, 3.21, 
            2.07947275979414, 2.65, 3.14011410618903, 3.3, 3.2064264829351, 
            2.63, 1.0583097024763, 2.74, 2.33775111974449, 2.83, 
            2.96, 2.77, 3.38, 2.8538605607665, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 1.98490017695195, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            3.13788210940303, 3.64, 3.04329324397847, 2.84, 2.55396867578139, 
            2.78, 3.38287663405959, 1.14867882432706, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.06024969539061, 
            1.71746418776933, 1.88089803774767, 2.05, 2, 1.19145357084043, 
            1.33, 1.50502746397035, 1.62, 1.33, 1.73854710223798, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 1.37762164889772, 1.80517073274116, 1.75, 1.68, 
            1.75, 1.988361084376, 1.57854697848849, 2.05839073842154, 
            1.61896871851281, 0.775509156798809, 1.46324857152023, 
            1.54093592143631, 1.91253919905678, 1.65867054301116, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 1221.83636230922, 
            1067.12864329802, 735, 1450, 1290, 1295, 1045, 1045, 
            1050.45692192293, 1280, 1320, 1347.03481841839, 1547, 
            1310, 1280, 1104.47227388835, 1680, 845, 1005.62872698261, 
            770, 1035, 1015, 845, 1209.11406808825, 1195, 1285, 
            915, 1035, 1285, 966.32877650486, 1053.35672533903, 
            1235, 1039.1211395462, 920, 880, 1110.74117038184, 
            568.170059651162, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1073.4999662613, 1150, 1265, 
            1190, 1207.53674806311, 1060, 1120, 970, 1270, 1285, 
            517.49590682139, 427.376623537978, 774.573328934155, 
            630, 518.820513723445, 946.342368911095, 932.622353617652, 
            502, 327.597337462315, 485.198028006564, 718, 870, 
            410, 472, 320.232071685749, 886, 557.618072977692, 
            392, 918.577758073082, 781.721249006588, 463, 278, 
            714, 623.18173905145, 515, 584.031367800244, 542.037886477185, 
            359.81917400819, 562, 680, 625, 480, 450, 495, 927.776264650897, 
            345, 653.408668243451, 625, 428, 660, 406, 617.49874148867, 
            562, 438, 415, 672, 439.622325897863, 510, 480.586307809509, 
            312, 680, 562, 325, 855.794954157349, 434, 385, 763.45368589424, 
            519.884276025195, 727.858497756011, 372, 564, 758.541719961197, 
            895.748036528869, 365, 380, 663.874622089393, 378, 
            681.286465799119, 466, 355.464040360369, 580, 820.484759349371, 
            524.818588367463, 560, 600, 185.158656686314, 695, 
            720, 515, 580, 590, 600, 631.888742970881, 520, 550, 
            653.626642629918, 830, 415, 749.490943995419, 573.58828025089, 
            550, 500, 480, 425, 675, 640, 695.380285610218, 480, 
            880, 660, 428.940628497504, 700.40388464871, 680, 
            570, 1005.79965788924, 615, 471.479039635094, 695, 
            685, 418.567759475434, 580.341193814739, 510, 470, 
            1137.28340126025, 740, 363.020716458471, 835, 840, 
            847.147542268569)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.5870096609308, 
        13.24, 12.8833108944218, 14.39, 14.06, 14.7193424405472, 
        13.86, 13.8593583191395, 13.2344649598756, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 13.9376106345913, 13.64, 14.0467950191129, 
        12.93, 13.71, 12.85, 13.5, 14.1190546515346, 13.39, 13.3, 
        13.87, 13.4369194936023, 13.73, 13.58, 13.68, 13.76, 
        13.7006850269071, 13.8310898800757, 14.4142718339795, 
        13.05, 12.7960147381117, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 13.7428061309459, 13.9, 13.3026466837656, 
        13.6546217453022, 13.05, 13.0960562774082, 13.82, 14.011286307661, 
        13.74, 14.4371112331236, 14.22, 13.29, 13.6878547465993, 
        12.37, 12.33, 12.64, 11.5340550651117, 12.37, 12.17, 
        12.37, 12.8757137040747, 12.5712390363372, 13.34, 12.21, 
        12.9642634878579, 13.86, 12.2302038327671, 12.99, 11.96, 
        11.66, 11.9884557417659, 11.84, 12.5061984625527, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.9909344633312, 12.16, 
        12.2473675523369, 11.7267933494187, 12.08, 12.08, 12, 
        12.69, 11.9843650878019, 11.62, 13.6852575863336, 11.81, 
        12.29, 14.0350473624373, 12.29, 12.08, 12.6, 13.0690959594547, 
        11.4056920762322, 12.51, 12.7522775104213, 11.5699475883694, 
        12.72, 12.22, 11.61, 12.8468025558148, 12.2841445522023, 
        11.76, 12.3797527001935, 12.08, 11.03, 11.82, 12.8038226162292, 
        12.2435362286143, 11.9908819087834, 11.45, 12.8439076915301, 
        12.42, 13.05, 11.87, 12.07, 13.1225602713372, 11.79, 
        12.1432515326574, 12.04, 12.86, 12.88, 12.9498298969133, 
        12.7, 12.51, 13.1106477696058, 12.25, 12.53, 12.8677397348482, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 13.1814637034211, 
        12.3030216069314, 12.87, 13.32, 11.9179535807618, 12.485731789366, 
        12.79, 12.128393532999, 13.23, 12.1621564747262, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.5502277022245, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.0511014665077, 13.8767013070143, 
        13.58, 13.4, 13.5547560481377, 12.77, 14.16, 14.5696110384654, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.11552540880564, 
        0.826127011124252, 1.95, 1.66223839637405, 0.647084782448839, 
        1.87, 2.15, 1.75469439209041, 1.35, 2.16, 1.48, 1.13358093379798, 
        1.73, 1.87, 0.255518396152084, 1.92, 0.983902666682396, 
        1.07823970603587, 3.1, 1.63, 3.8, 1.35164665422255, 1.6, 
        1.81, 0.690797549052618, 1.64315812852356, 1.10207095919419, 
        1.9, 3.11971357747493, 1.5, 2.46936677189558, 1.83, 1.53, 
        2.63141476030503, 3.17150205621143, 3.08228979592864, 
        1.65, 1.5, 1.30078041155061, 1.71, 1.7843922322219, -0.740551616728197, 
        3.98, -0.56510848578582, 4.04, 1.23451234322778, 1.68, 
        -0.678694650436196, 1.46865047239217, -0.491400799942839, 
        1.65, 3.58165747654167, 1.9, 1.67, 0.933197102908393, 
        1.7, 1.97, 1.54477624979304, 0.94, 1.27059918997602, 
        1.36, 1.25, 1.13101646787899, 3.5491778361099, 1.21, 
        1.01, 1.17, 2.50954289119307, 0.934596785718112, 2.69994028211563, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.71977123831946, 
        0.99, 3.87, 1.65523263613685, 2.4907690458998, 1.13, 
        4.41048583130622, 1.95981280458257, 0.98, 1.61, 1.67, 
        -0.0221065121570758, -0.00735330979099791, 1.49411173021169, 
        1.51, 2.46467863183759, 0.817379634765605, 1.04588369066585, 
        1.59139609003966, 1.65339772069488, 1.41, 0.778483851000626, 
        1.93409845865145, 2.08, 3.86452160447638, 2.14442912222598, 
        1.72, 0.659405964712618, 2.55, 1.73, 3.26291332970607, 
        1.29, 0.137192632705555, 3.74, 1.40171031616643, 2.68, 
        0.74, 1.39, -1.33624520194029, 2.45644989392828, 1.61, 
        3.43, 3.68721303373156, 2.4, 2.05, 4.43, 3.10054944366922, 
        4.31, 2.16, 1.53, 2.13, 2.33408036346989, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 4.25091829986934, 2.3032999389123, 
        5.51, 3.59, 3.12741581935843, 2.81, 2.56, 5.07775657934886, 
        4.05422174378793, 3.88, 3.64454610169794, 5.04, 4.61, 
        3.24, 2.48600486949924, 3.12, 2.67, 1.9, 2.8437613360507, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.68141718843094, 
        2.91547012834366, 3.45, 2.76, 4.17188113743241, 2.70375133353879, 
        3.37, 2.58, 4.6, 4.7484306038179, 2.39, 2.32197895463874, 
        5.65, 3.91, 4.47815310149069, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.41780171505723, 2.37638738435138, 
        2.45, 2.61, 2.63529035762237, 2.33956700120809, 2.3, 
        2.38281295283196, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
        2.48, 2.56, 2.66326520917, 2.65, 2.58490943435352, 2.52, 
        2.61, 3.22, 2.40536232140752, 2.58327832211964, 2.4394287636858, 
        2.46894355166843, 2.65514780379113, 2.36, 2.59418263003521, 
        2.7, 2.65, 2.43978221365121, 2.38309536437723, 2.55, 
        2.1, 2.51, 2.5404227240252, 2.12, 2.59, 2.29, 2.1, 2.44, 
        2.40368209104573, 2.12, 2.4, 2.27, 2.04, 2.34451320410859, 
        2.4271435513516, 2.54482728329433, 2.54945523686029, 
        2.47059566781532, 2.3, 2.68, 2.5, 1.36, 2.43231881856662, 
        2.32393574387733, 1.92, 2.16, 2.53, 2.56, 1.97972264903396, 
        1.92, 1.84273435718407, 1.75, 2.21, 2.67, 2.24, 2.6, 
        2.3, 1.97612687375865, 1.71, 2.45980830718673, 1.91565559732211, 
        2.4, 2, 2.35078880567094, 2.51, 2.32, 2.58, 2.24, 1.91909819944436, 
        2.62, 2.47170769917773, 2.3, 2.32, 2.09615881413932, 
        2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 1.95671729419253, 
        1.77079854844326, 1.7, 2.35805359220927, 2.46, 1.88, 
        2.53693208238342, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 
        2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.24031258519796, 1.74466907691108, 
        2, 2.42, 3.23, 2.46378687588443, 2.13, 2.39, 2.17, 2.29, 
        2.42541240904695, 2.59195853118077, 2.38, 2.34988944982447, 
        2.4, 2.4, 2.36, 2.25, 2.86107000762743, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.34160580613957, 2.5342856743446, 
        2.48, 1.91516149080543, 2.36, 2.62, 2.48, 2.39996784082972, 
        2.28, 2.40019026955948, 2.32, 2.38, 1.91615508546741, 
        2.26350713181326, 2.64, 2.38, 2.54, 2.41263274745297, 
        2.35, 2.3, 2.47054862677259, 2.6, 2.31222862713662, 2.69, 
        2.69412285396781, 2.32, 2.28, 2.74952345510189, 2.5677312367352, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 14.5355133424393, 
        21.2044990310505, 16.8, 21, 15.2, 15.4839910461681, 17.6, 
        14, 16, 19.6179978920797, 16.8, 16, 11.4, 15.0810098214672, 
        18.6728200303164, 22.3428318284675, 20, 17.4542143219554, 
        15.2, 16, 18.6, 17.6735558529275, 17.8, 20, 24.0538268366036, 
        16.1, 17, 17.1638114206146, 18.6584335232392, 20.5130446871284, 
        19.1, 18.8702274978836, 15.2910344115165, 19, 20.5, 15.5, 
        18.7152532827721, 15.5, 13.2, 16.2, 18.8, 16.4572464470183, 
        15.5233989989977, 17, 13.580839783031, 16, 11.1885811601553, 
        18.8, 17.4, 12.4, 20.1881676503618, 14.8480396358705, 
        15.7202572451151, 19.8715155999299, 20.5, 16.3, 16.8, 
        14.3214941357691, 10.6, 19.9414702607194, 16.8, 18, 19, 
        19, 18.1, 13.6717553438071, 19.6, 11.4127459716551, 16.8, 
        20.4, 20.3213065634342, 24, 30, 21, 17.997246427303, 
        16, 18, 14.8, 13.9721975744038, 19, 16.4749989556988, 
        24, 22.5, 18, 18, 22.8, 19.2787913517276, 21.6, 20.6507625422016, 
        18.6280307903592, 22, 19.8489837608423, 18, 18, 19, 21.5, 
        16, 10.76075407142, 18, 17.5, 18.5, 21, 19.5, 20.5, 22, 
        19, 22.5, 19, 20, 19.5, 20.2215158176716, 26.6666363066012, 
        25.3366046919954, 14.996077298976, 21.5, 20.8, 18.0493415728313, 
        16, 16.4412426195218, 17.9290510055093, 27.0219283790791, 
        19.9603576209842, 21.5, 21, 21, 21.5, 28.5, 16.7775378550074, 
        22, 19.9227572027402, 20, 24, 21.5, 24.0569712148229, 
        25.5359536239084, 21.9487446953376, 25, 19.5, 20.4724384736411, 
        21, 19.6855836706504, 23.5, 20, 13.6226364277909, 21, 
        20, 22.0245972853505, 23.2765888905915, 21.5, 24, 22, 
        23.1599206848627, 18.5, 20, 22, 19.5, 12.9869405662721, 
        25, 24.9701741841812, 19.5911413593419, 20.8129022469683, 
        24.6694617670162, 18.5, 21.6282060620651, 22.5, 23, 19.5, 
        24.5, 25, 18.8076375177446, 19.5, 22.8528016724033, 20.5, 
        23, 20, 20.2336040016919, 24.5), mg = c(127, 100, 101, 
        113, 119.213522600627, 112, 96, 121, 97, 112.658982595626, 
        96.8330700177402, 107.081235396771, 89, 91, 102, 112, 
        120, 124.248232518504, 150.172372309217, 116, 126, 102, 
        91.6095972465565, 95, 92.4200647677279, 124, 93, 105.396174833996, 
        107, 96, 101, 106, 104, 113.623628349752, 110, 111.912308031358, 
        110, 100.721794971494, 89.6838070554951, 119.892896840778, 
        117, 100.821745283388, 101, 105.120734329373, 107, 111, 
        102, 101, 121.707744553472, 108, 92, 107.363565027779, 
        111, 115, 110.89210241821, 111.842285894161, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 62.9442078185791, 
        151, 90.9008509974924, 86, 87, 139, 101, 87.4351141801065, 
        76.8338636816338, 112, 83.2853046590064, 119.86236276044, 
        86, 77.7121484765716, 78, 85, 111.364328091025, 99, 90, 
        88, 84, 70, 81, 103.183058865131, 52.6501930107542, 88, 
        99.3744542784275, 115.279442232473, 134, 85, 88, 98.6103115180976, 
        97, 88, 85.9236003398431, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 92.4847315760655, 77.8624229614756, 
        87, 96, 131.315332675564, 102, 86, 138.154690267462, 
        85, 82.0987862484388, 92, 88, 80, 122, 108.161074647779, 
        98, 106, 87.452745945505, 94, 89, 85.4913439373055, 88, 
        101, 103.847639862028, 89, 97, 75.5064479229242, 112, 
        102, 86.7785451316809, 120.00292226397, 92, 113, 123, 
        91.3540458544207, 116, 98, 103, 107.594261276674, 89, 
        97, 98, 89, 81.027755574026, 72.7679285533797, 106, 106, 
        90, 88, 129.299368568585, 88, 105, 112, 96, 86, 91, 95, 
        102, 91.5065695204549, 120, 114.961070526608), phe = c(2.8, 
        2.65, 2.8, 3.85, 2.8, 2.90957148613944, 2.5, 2.6, 3.2701630256449, 
        2.98, 2.95, 2.2, 2.6, 3.1, 3.32001018245463, 2.85, 2.8, 
        3.20590481392894, 3.3, 2.79777687408855, 3, 2.41, 2.61, 
        2.48, 2.53, 2.90873643933855, 2.65777441900242, 2.4, 
        2.95, 2.65, 3, 2.97739511752863, 2.42, 2.95, 2.84226202752183, 
        2.7, 2.6, 2.45, 2.71868548388819, 3.32147121837746, 3.15, 
        2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 2.88, 2.72, 
        2.45, 3.88, 3.16249717652252, 2.57256540829936, 2.96, 
        3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.21993557728357, 3.5, 
        1.89, 2.92164480245573, 3.53134694115371, 2.11, 2.53, 
        3.48678312528678, 1.1, 2.95, 1.88, 2.38186873181803, 
        1.72736883591592, 1.88047027306666, 2.4640133063169, 
        2.45537186823304, 1.9, 2.83, 2.66853010707096, 2.2, 2.040149085981, 
        1.65, 2.24489028216998, 2.2, 2.43638239750781, 1.92, 
        1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.62756431850254, 
        1.38922195426576, 2.55, 3.52, 3.16204983072392, 2.83272134380044, 
        1.97272380951346, 2.56, 2.5, 2.2, 2.70632177759312, 1.65, 
        2.25465888540283, 2.36, 2.74, 3.18, 2.55, 1.75, 2.21285374509886, 
        2.56, 2.46, 2.16562151249547, 2, 1.55039960790163, 2, 
        2.9, 3.18, 2.05914775831742, 2.49570599228904, 2.86, 
        2.8814761335477, 2.74, 2.13, 2.22, 2.1, 1.51, 2.00961634699376, 
        1.88717272663769, 1.7, 2.18385174029123, 1.62, 1.38, 
        1.13039471952312, 1.62, 1.86982790377154, 0.970448522884164, 
        1.4, 0.904367347947505, 2, 1.38, 0.872725853686041, 0.98, 
        1.7, 1.93, 1.41, 1.4, 1.48, 2.2, 1.59677617490032, 1.48, 
        1.52817352168181, 1.8, 1.9, 1.53941326985966, 1.21487416017866, 
        2.3, 1.83, 1.64043041090696, 1.39, 1.35, 2.05376763952943, 
        1.7, 1.48, 1.55, 1.80807940706841, 1.25, 1.39, 1.68, 
        2.25800564263433, 1.58129072937607, 1.62622931768632, 
        1.65, 2.05), fla = c(2.86946787274986, 2.76, 3.73760308063321, 
        3.49, 2.78049410559267, 3.39, 2.52, 2.51, 2.98, 3.524103233785, 
        3.32, 2.43, 2.76, 2.81103319968797, 3.64, 2.91, 3.14, 
        3.4, 3.09137191404526, 3.1701247012165, 3.1251607030113, 
        2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.18226644406807, 
        2.97, 2.33, 3.25, 3.19, 2.69, 2.83667374008886, 2.75143765398459, 
        2.98, 2.1996772310616, 2.43, 2.64, 3.04, 3.29, 2.68, 
        3.56, 2.63, 3, 2.65, 3.17, 3.16160044957406, 2.92, 3.54, 
        3.137658544192, 2.99, 3.74, 2.79, 2.9, 2.78, 3.32578453909532, 
        3.23, 3.67, 0.57, 1.59593009801183, 1.41, 1.79, 3.0371496657947, 
        1.75, 2.65, 2.97481169494079, 2, 1.3, 2.70389629846579, 
        1.02, 2.86, 1.30524997728058, 2.89, 2.14, 1.57, 2.11082280627116, 
        1.5761051975811, 1.85, 2.01484839119476, 2.26, 2.53, 
        1.58, 1.59, 2.21, 1.79462900824006, 1.69, 1.61, 1.69, 
        1.82585792556869, 1.6558786585384, 1.25, 1.46, 2.31499730575975, 
        2.9664221889085, 2.27, 0.99, 1.88660261750425, 2.91611307843818, 
        2.99, 2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 
        2.08202601728284, 2.92, 2.14025140268432, 2.27, 2.03, 
        2.01, 2.29, 2.17, 1.6, 2.09, 1.08852115245535, 1.64, 
        3.30234451006823, 3.75425402305463, 1.61809564932452, 
        2.21901739781586, 3.03, 2.65, 3.15, 2.24, 2.24216784804923, 
        1.75, 0.65790998571781, 0.392347855553761, 0.322657515290861, 
        1.2, 0.58, 0.66, 0.47, 0.382597394974128, 0.0682336709388778, 
        0.6, 0.5, 0.73723545031628, 0.52, 0.8, 0.903410099544455, 
        0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
        0.58, 0.63, 0.897826444252832, 0.58, 1.31, 1.1, 0.92, 
        1.1649012237637, 0.6, 0.493232733040647, 0.68, 0.47, 
        0.92, 0.66, 0.84, 0.96, 0.49, -0.0993031435990419, 0.7, 
        0.61, 0.371718964038583, 0.69, 0.68, 0.76), nfla = c(0.28, 
        0.113814090335851, 0.3, 0.24, 0.39, 0.339740317058052, 
        0.297639178053656, 0.31, 0.29, 0.22, 0.22, 0.26, 0.29, 
        0.43, 0.29, 0.3, 0.372367624940314, 0.437302570046666, 
        0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.264993461878555, 
        0.37162131102386, 0.34, 0.27, 0.258867861927583, 0.26, 
        0.29, 0.301551707337891, 0.42, 0.26249950257889, 0.29, 
        0.26, 0.297654255462513, 0.554817632629397, 0.28, 0.2, 
        0.34, 0.284404712646611, 0.0966662875654035, 0.32, 0.28, 
        0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 0.32, 0.39, 
        0.21, 0.2, 0.26, 0.36499234478687, 0.222291105923285, 
        0.28, 0.48573678399159, 0.53, 0.32, 0.340509368504043, 
        0.45, 0.37, 0.26, 0.27, 0.395547462288552, 0.246966058668051, 
        0.320799240062303, 0.288642251882805, 0.543265492682391, 
        0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.355956542834921, 
        0.4, 0.61, 0.22, 0.3, 0.254290689882322, 0.598582297590677, 
        0.463651982353915, 0.367208333491085, 0.483522851208208, 
        0.5, 0.523667813879225, 0.25, 0.17, 0.32, 0.414591440664027, 
        0.311482323018483, 0.24, 0.369175049215547, 0.0578349738248828, 
        0.29, 0.34, 0.37, 0.281531648879636, 0.66, 0.325728203700606, 
        0.48, 0.39, 0.29, 0.24, 0.26, 0.384505146276355, 0.42, 
        0.359208701486274, 0.52, 0.3, 0.341159016488259, 0.387603050472201, 
        0.441119510129833, 0.32, 0.150425081212981, 0.43, 0.3, 
        0.380639455347626, 0.130514285248675, 0.39, 0.171545103316642, 
        0.4, 0.369274363486644, 0.493065601546996, 0.581051412630112, 
        0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 
        0.303409184586431, 0.606208900159421, 0.476763309442706, 
        0.380493374821076, 0.43, 0.4, 0.47, 0.45, 0.153742224835407, 
        0.22, 0.312215951889835, 0.26, 0.61, 0.53, 0.58291409245026, 
        0.435553113056112, 0.63, 0.53, 0.52, 0.452637668628189, 
        0.5, 0.6, 0.4, 0.41, 0.583101389044034, 0.43, 0.4, 0.39, 
        0.514446208783783, 0.407624499001059, 0.48, 0.650975381577934, 
        0.573704706952477, 0.43, 0.43, 0.53, 0.432798650291575
        ), pro = c(2.29, 1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 
        1.25, 1.98, 1.85, 2.38, 1.57, 1.81, 1.98778692492483, 
        2.96, 1.46, 2.44974299945564, 1.9347638908434, 1.86, 
        1.66, 2.1, 0.84341506303134, 1.69, 1.46, 1.66, 1.92, 
        1.45, 1.35, 1.76, 1.98, 2.38, 1.95, 1.79617498383991, 
        1.67543734890021, 1.54, 1.86, 1.36, 1.44, 1.37, 2.14790161451048, 
        1.60293649679901, 2.14481146647585, 2.15044263313909, 
        0.922519932408355, 2.03, 1.13572846277203, 2.19, 2.14, 
        2.38, 2.08, 1.92497653592335, 2.29, 1.87, 1.68, 1.62, 
        2.45, 1.90515503611185, 1.66, 1.95421138951343, 0.42, 
        1.70400255112503, 1.70235963958864, 0.73, 1.87, 1.03, 
        2.18682047318116, 1.74513787856415, 1.04, 1.53327329057746, 
        2.5, 1.46, 1.87, 1.03, 2.66557203068391, 1.34889602275922, 
        1.15, 1.46, 0.864158237571249, 2.76, 1.95, 1.43, 2.47397869005356, 
        1.40144322416115, 1.62, 1.40615102991858, 1.46, 1.56, 
        1.34, 1.35, 1.38, 1.64, 1.63, 1.61216848051751, 1.99, 
        1.35, 3.28, 1.56, 1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 
        1.42, 0.812109916362553, 1.42, 0.731687438529217, 1.63, 
        2.08, 2.49, 1.86369662533398, 1.32183501458199, 1.62103988383187, 
        2.01996395793841, 1.04, 2.31692362945388, 1.24323927970805, 
        1.61, 0.83, 1.87, 1.83, 1.58603499600796, 1.71, 1.50795332976424, 
        2.91, 1.35, 1.77, 1.23441753699992, 1.9, 1.53047394367297, 
        0.94, 0.679334437261709, 0.83, 0.84, 1.25, 0.94, 0.8, 
        1.1, 0.88, 1.51988660469347, 0.75, 1.79690337048521, 
        0.55, 2.68542694549891, 1.14, 1.3, 0.68, 0.86, 1.25, 
        1.14, 1.25, 1.26, 1.56, 1.2380243514287, 0.696661272525374, 
        1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.35043685308415, 
        1.78908209441552, 0.94, 1.51007759902125, 1.15, 1.46, 
        0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
        1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.68706746710288, 
        6.75, 4.63122759563287, 5.05, 6.881334972643, 7.22, 5.75, 
        4.62666367555528, 5.6, 6.41099391148259, 7.5, 7.06502729225774, 
        7.73646288851865, 7.33224657569601, 8.7, 5.1, 5.65, 4.5, 
        3.43633649884023, 3.93, 3.52, 3.58, 4.8, 3.95, 4.5, 4.7, 
        5.7, 5.7555487474853, 5.54935104143046, 5.4, 4.2, 5.1, 
        4.6, 4.25, 3.7, 5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 
        4.9, 6.1, 6.20320149499747, 8.9, 7.2, 4.9775701068934, 
        5.57952902028672, 6.3, 5.85, 6.25, 8.24657769728753, 
        5.15398433753875, 6.8, 9.84234327266229, 3.27, 5.75, 
        3.8, 4.45, 2.95, 4.6, 5.3, 4.68, 3.17, 2.85, 10.3293219622979, 
        3.38, 3.74, 2.97640559133931, 3.21, 3.8, 4.6, 1.44610472248638, 
        2.55769971551951, 3.28450558198132, 2.5, 3.9, 0.853858579195264, 
        4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 3.05, 
        2.15, 3.25, 2.6, 7.03268927986125, 2.9, 5.25940498656652, 
        5.6858099627908, 2.59289519853828, 1.27064569719776, 
        2.8, 0.867948185505119, 2.94, 2.7, 3.4, 3.3, 2.7, 1.77107386277591, 
        2.9, 3.50642152736872, 1.74062946228516, 3.08, 2.9, 1.9, 
        -0.690189872234685, 5.62593794874112, 3.4, 1.28, 3.25, 
        3.32661423068467, 0.268654108504337, 4.86214303151455, 
        2.8, 2.92614720819959, 3.94, 1.38104085016472, 2.12, 
        1.94507190839083, 9.78427601671583, 8.11264029571323, 
        5.98954209378846, 5, 5.45, 7.1, 6.86203960374822, 5, 
        5.7, 4.92, 7.89936207427663, 7.94763698073914, 4.35, 
        4.4, 8.21, 4, 4.9, 6.84974432606857, 8.42, 9.4, 3.97509970648944, 
        9.96160828699938, 7.1, 10.52, 7.6, 7.9, 8.94337326763621, 
        7.5, 13, 10.0136090457054, 7.65, 8.48231835317375, 5.58, 
        8.11829207117853, 9.77613642726133, 6.62, 10.68, 10.26, 
        8.33982999869198, 8.5, 8.12189426680671, 5.35398368513547, 
        9.7, 7.7, 7.3, 7.88780891636668, 9.3, 9.2), hue = c(1.04, 
        1.05, 1.30853723030063, 0.86, 0.884658550009095, 1.05, 
        1.02, 0.830350356474506, 1.08, 0.702788740207387, 1.25, 
        0.872999736147323, 1.15, 1.25, 1.2, 1.28, 1.09422897102594, 
        1.13, 1.23, 0.728810506592378, 1.09, 0.980221227876393, 
        1.11, 1.09, 1.12, 1.13, 1.3839728286307, 1.02, 1.01624126331594, 
        1.04, 1.19, 1.09, 1.08254342462746, 1.25, 1.1, 1.02851175785435, 
        0.794737050645381, 1.06654458582079, 1.18, 1.15559801547775, 
        0.95, 0.91, 0.88, 0.82, 1.32809958425419, 0.87, 1.11776517711455, 
        1.01123725981876, 1.07, 1.12, 1.12, 1.16976299462479, 
        1.16821020110965, 1.21968632061531, 0.996598895246502, 
        1.33109201813856, 0.94, 1.07, 0.89, 0.798234332078525, 
        1.25, 0.98, 1.23, 1.22, 0.557176405202615, 0.949297390657983, 
        0.972712171361778, 1.12, 1.02, 1.28, 0.906, 1.36, 0.98, 
        1.31, 0.894693725053466, 1.23, 1.19, 0.96, 1.06, 1.19, 
        1.38, 1.20854045477634, 1.31, 0.84, 1.00034031308653, 
        1.23, 0.920504999143292, 1.17994192726734, 1.24753788874141, 
        1.07, 1.08, 1.05, 1.11404415488811, 1.15, 1.23551159330984, 
        1.35018667096805, 0.825544192409448, 1.19251989920128, 
        1.06166601588414, 1.42, 1.27, 0.847114341221924, 0.8, 
        1.45457351814336, 1.04, 1.30213671856723, 1, 0.882269068788545, 
        0.86, 1.11666584923318, 1.33001358094673, 0.9, 1.04288589164071, 
        1.1, 0.93, 1.71, 0.95, 0.862842877821058, 0.7, 1.31556373554769, 
        0.8, 1.04553504209912, 0.92, 0.73, 1.38866840748971, 
        0.86, 0.69, 0.97, 0.89, 0.79, 0.670627958922346, 0.74, 
        0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.890486419245618, 
        0.260134589180047, 0.667408375628838, 0.89, 0.908883309940167, 
        0.65, 0.6, 0.468850043450725, 0.54, 0.55, 0.482920054312289, 
        0.603911296651063, 0.48, 0.61, 0.56, 0.647086923289811, 
        0.6, 0.57, 0.67, 0.57, 0.627191049405796, 0.56, 0.96, 
        0.87, 0.68, 0.7, 0.78, 0.85, 0.615455129589447, 0.74, 
        0.67, 0.323276797074065, 0.952726465762209, 0.807373517389527, 
        0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 3.17, 
        2.63401852068831, 2.93, 2.63221139092767, 3.58, 3.58, 
        2.85, 3.55, 3.17, 2.82, 2.9, 3.33742331128204, 3, 2.6909016913529, 
        2.65, 3.44308572959472, 2.82, 3.36, 3.71, 3.52, 4, 3.63, 
        3.28574563588489, 3.2, 3.52847463903799, 2.77, 3.64467070556974, 
        2.67462246719159, 2.71, 2.88, 2.87, 3, 2.87, 3.2476735920294, 
        3.15779773751266, 2.68973014515459, 2.69, 3.81193738153805, 
        3.38, 3, 3.46518844300452, 3, 3.35, 3.33, 3.08289949587549, 
        3.33, 2.75, 3.1, 2.82295755744723, 2.86786198496654, 
        3.26, 2.93, 3.2, 3.03, 3.13005689837655, 2.84, 2.87, 
        1.82, 2.93183174224492, 1.59, 2.46, 2.87, 2.78074812170631, 
        2.3, 3.18, 3.48, 2.19235872642661, 2.87250751336543, 
        1.82, 3.49392983811852, 1.85348450660967, 3.5, 3.13, 
        2.14, 2.48, 2.52, 2.96387865120022, 2.91142137246326, 
        3.45191047089752, 3.14, 2.72, 2.01, 2.82867963332549, 
        3.0811922945948, 2.37965664870291, 3.21, 2.75, 3.21, 
        3.27440102573602, 2.65, 2.61177856552105, 3.3, 3.66907511196844, 
        2.63, 1.95367508339944, 2.74, 3.36088123765603, 2.83, 
        2.96, 2.77, 3.38, 2.45716500634462, 3.57, 3.3, 3.17, 
        2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.60549670829075, 
        3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
        2.65509285382366, 3.64, 2.689265873633, 2.84, 2.93471525485119, 
        2.78, 1.70276458906283, 2.24247555677083, 1.42, 1.36, 
        1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 2.12533732394248, 
        1.84877767127435, 2.31399170953121, 2.05, 2, 2.04956618175587, 
        1.33, 2.05497659940559, 1.62, 1.33, 2.45958621979346, 
        1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
        1.58, 2.5531118167038, 2.0605195144421, 1.75, 1.68, 1.75, 
        1.32431629491434, 1.51691312115911, 1.85189974463576, 
        1.81668407291821, 2.32652613406415, 1.94843891672801, 
        2.11811728858553, 2.07314856850643, 1.17690382367889, 
        1.56, 1.62, 1.6), prol = c(1065, 1050, 1295.8700368258, 
        1103.09926778968, 735, 1450, 1290, 1295, 1045, 1045, 
        846.424117620129, 1280, 1320, 1119.84001110395, 1547, 
        1310, 1280, 1213.13575706972, 1680, 845, 1112.90563983674, 
        770, 1035, 1015, 845, 1186.07155977544, 1195, 1285, 915, 
        1035, 1285, 1001.7783671297, 1192.69657163, 1235, 1232.49747737372, 
        920, 880, 1155.22135323772, 790.675707063083, 760, 795, 
        1035, 1095, 680, 885, 1080, 1065, 985, 1060, 1157.62223977785, 
        1150, 1265, 1190, 1158.41190366311, 1060, 1120, 970, 
        1270, 1285, 718.922476595916, 734.482859072185, 786.86587376593, 
        630, 321.783681707886, 500.360389444751, 604.31343077502, 
        502, 351.507835696267, 485.054262431246, 718, 870, 410, 
        472, 888.233131511469, 886, 460.171327966172, 392, 597.121693924359, 
        489.017993217719, 463, 278, 714, 552.546737163729, 515, 
        852.179436745834, 321.747302248337, 300.618916720571, 
        562, 680, 625, 480, 450, 495, 587.60721163935, 345, 1058.02218456022, 
        625, 428, 660, 406, 432.462570056397, 562, 438, 415, 
        672, 491.2915549514, 510, 485.316668752527, 312, 680, 
        562, 325, 590.455324327774, 434, 385, 484.405519820186, 
        114.579118059967, 461.073115854322, 372, 564, 335.54996794731, 
        661.126498061794, 365, 380, 724.521406133658, 378, 818.675907408663, 
        466, 595.620630289892, 580, 882.989911416034, 567.672452864521, 
        560, 600, 447.590756789942, 695, 720, 515, 580, 590, 
        600, 344.082543756984, 520, 550, 609.412938055295, 830, 
        415, 546.650847501937, 467.075224364793, 550, 500, 480, 
        425, 675, 640, 813.54415255201, 480, 880, 660, 849.11234179253, 
        386.071083042285, 680, 570, 750.618079571551, 615, 538.330806944369, 
        695, 685, 965.710319480747, 568.006705297115, 510, 470, 
        699.741234781342, 740, 539.973324370866, 835, 840, 550.549503201551
        )), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.4200794924232, 
        13.24, 13.6884432063133, 14.39, 14.06, 13.7162613739393, 
        13.86, 13.7168311823541, 14.4305869109065, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.456782802202, 13.64, 13.1310606937368, 
        12.93, 13.71, 12.85, 13.5, 13.7462461991422, 13.39, 13.3, 
        13.87, 12.2833907593754, 13.73, 13.58, 13.68, 13.76, 
        13.4257504174857, 14.5201503915367, 14.3696910736484, 
        13.05, 11.1256563773773, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 14.4535209015849, 13.9, 14.419527737047, 
        13.8816169650364, 13.05, 14.0425761980077, 13.82, 13.8661318609772, 
        13.74, 14.4559591366051, 14.22, 13.29, 13.8205121878983, 
        12.37, 12.33, 12.64, 11.9909606259663, 12.37, 12.17, 
        12.37, 12.2713511195675, 12.6349682519885, 13.34, 12.21, 
        11.7063204103739, 13.86, 12.7642685542875, 12.99, 11.96, 
        11.66, 12.2181962030774, 11.84, 12.6156888125805, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.799690605888, 12.16, 
        12.9189976221528, 13.2102701840084, 12.08, 12.08, 12, 
        12.69, 11.6278024003963, 11.62, 11.7986965154255, 11.81, 
        12.29, 14.5989865379806, 12.29, 12.08, 12.6, 12.6865505834116, 
        12.6771175340415, 12.51, 12.1658979085355, 12.2698963973318, 
        12.72, 12.22, 11.61, 12.1145801254081, 12.0026038853637, 
        11.76, 12.2046275737123, 12.08, 11.03, 11.82, 12.1814536522893, 
        12.1796688592047, 12.0478717677037, 11.45, 14.6435615859467, 
        12.42, 13.05, 11.87, 12.07, 11.3259645110219, 11.79, 
        12.5072280598293, 12.04, 12.86, 12.88, 13.2682936495979, 
        12.7, 12.51, 12.5932079684565, 12.25, 12.53, 12.1980963195913, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.9067054285279, 
        12.1013623060957, 12.87, 13.32, 12.2100935855538, 12.6450141842037, 
        12.79, 12.4268030869381, 13.23, 12.652392747251, 13.17, 
        13.84, 12.45, 14.34, 13.48, 12.576344090472, 13.69, 12.85, 
        12.96, 13.78, 13.73, 13.7121817025587, 13.5765758398519, 
        13.58, 13.4, 12.7095089031222, 12.77, 14.16, 13.3936184664081, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 1.71470875066601, 
        -0.229828509954581, 1.95, 3.50830859519075, 2.51084414636923, 
        1.87, 2.15, 1.37854690156346, 1.35, 2.16, 1.48, 0.634616400898448, 
        1.73, 1.87, 0.750598232481153, 1.92, 1.23150088475923, 
        0.181021459750729, 3.1, 1.63, 3.8, 3.56031092615626, 
        1.6, 1.81, 2.46289975514061, 0.872582241272953, 2.53731347283231, 
        1.9, 1.48301137142405, 1.5, 1.53022468197761, 1.83, 1.53, 
        3.36472001236486, 0.438900578260588, 5.44159534794715, 
        1.65, 1.5, 1.7625075471612, 1.71, 2.79948377605399, 1.70268586842076, 
        3.98, -0.159091938477993, 4.04, 3.38940161232037, 1.68, 
        3.11705997958009, 2.1451154246248, -0.260351231914296, 
        1.65, 1.63510829058599, 1.9, 1.67, 1.5820767057114, 1.7, 
        1.97, 2.11448846687167, 0.94, 1.90693549670557, 1.36, 
        1.25, 1.73400154394704, 2.70045187415364, 1.21, 1.01, 
        1.17, 2.07834637730312, 3.1105910946261, 1.76042835980411, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 5.15099086699218, 
        0.99, 3.87, 1.01712063460395, 3.03876249993474, 1.13, 
        5.19874578431028, 0.961434901819694, 0.98, 1.61, 1.67, 
        3.4097742566782, 1.41576524323302, 2.9578263690895, 1.51, 
        3.88036037157543, 0.14472577874149, 2.32578964434421, 
        0.577751332440646, 4.40442597041812, 1.41, 1.31662247287728, 
        -0.116987414078536, 2.08, 1.98725105963832, 3.77730349287592, 
        1.72, 2.7212292058849, 2.55, 1.73, 2.40075802897074, 
        1.29, 1.38702328952621, 3.74, 1.60739224748919, 2.68, 
        0.74, 1.39, -0.800438761035321, 2.46159953958435, 1.61, 
        3.43, 2.03372781232167, 2.4, 2.05, 4.43, 1.54256262547378, 
        4.31, 2.16, 1.53, 2.13, 4.00293215844417, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 3.41814109936171, 2.0765340106441, 
        5.51, 3.59, 5.16796893701153, 2.81, 2.56, 5.82655255392132, 
        3.55819453553572, 3.88, 5.64780564989779, 5.04, 4.61, 
        3.24, 4.98176753534741, 3.12, 2.67, 1.9, 3.67277747924243, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 4.37959426090481, 
        3.23892918775278, 3.45, 2.76, 4.70068678636291, 2.81560143182778, 
        3.37, 2.58, 4.6, 4.64439792055472, 2.39, 2.61754728946012, 
        5.65, 3.91, 3.10614804901488, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.71098605980716, 2.89223694168923, 
        2.45, 2.61, 2.27285972925169, 2.43953587207553, 2.3, 
        2.6016416054292, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 
        2.56, 2.4962125083489, 2.65, 2.27998786514461, 2.52, 
        2.61, 3.22, 2.12552165507453, 2.53086399648121, 2.79789774451788, 
        2.09211369101756, 2.60276988600255, 2.36, 2.5910962048554, 
        2.7, 2.65, 3.23618172092833, 2.37491500925585, 2.55, 
        2.1, 2.51, 2.14418963686914, 2.12, 2.59, 2.29, 2.1, 2.44, 
        2.30106173248286, 2.12, 2.4, 2.27, 2.04, 2.66053790859493, 
        2.38400097126107, 2.42521696853874, 2.53248305322958, 
        2.30857659653409, 2.3, 2.68, 2.5, 1.36, 2.51726984372799, 
        2.04386655129391, 1.92, 2.16, 2.53, 2.56, 1.88649189138956, 
        1.92, 1.50493847251282, 1.75, 2.21, 2.67, 2.24, 2.6, 
        2.3, 1.98300837690464, 1.71, 1.78432439958691, 1.60783532566295, 
        2.4, 2, 2.65284412166024, 2.51, 2.32, 2.58, 2.24, 2.33920158357566, 
        2.62, 2.77788201791756, 2.3, 2.32, 1.99891246824508, 
        2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.73982757123675, 
        1.98339020947531, 1.7, 2.09960064201484, 2.46, 1.88, 
        2.42824867905613, 2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 
        2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.15280662084324, 1.9230131707642, 
        2, 2.42, 3.23, 2.03241357375344, 2.13, 2.39, 2.17, 2.29, 
        2.88516941013645, 2.17750117265843, 2.38, 3.3465009864413, 
        2.4, 2.4, 2.36, 2.25, 2.43142464824766, 2.54, 2.64, 2.19, 
        2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.801760354961, 2.57856074568702, 
        2.48, 2.01014836739415, 2.36, 2.62, 2.48, 2.88334888406037, 
        2.28, 2.8087061456797, 2.32, 2.38, 1.61081416045015, 
        2.33220535229478, 2.64, 2.38, 2.54, 2.74501827212144, 
        2.35, 2.3, 2.92848458922526, 2.6, 2.25682845965622, 2.69, 
        3.14234698926996, 2.32, 2.28, 2.37416497844838, 2.46564831097274, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 15.0304749520659, 
        16.2260039472065, 16.8, 21, 15.2, 18.5839561493918, 17.6, 
        14, 16, 18.8785380469984, 16.8, 16, 11.4, 14.9353919080774, 
        16.5976888530349, 18.3082040615177, 20, 14.6673637351826, 
        15.2, 16, 18.6, 16.7578761900078, 17.8, 20, 19.9983991552741, 
        16.1, 17, 18.6895752785608, 13.1853441859473, 19.0026628436913, 
        19.1, 17.0844077647915, 19.1686158066548, 19, 20.5, 15.5, 
        16.2641966593054, 15.5, 13.2, 16.2, 18.8, 15.3178455641832, 
        18.8345008863623, 17, 15.2656905461942, 16, 13.6172181845944, 
        18.8, 17.4, 12.4, 18.3577117810818, 14.4589620300777, 
        18.8306458835191, 14.280994243169, 20.5, 16.3, 16.8, 
        15.0406300206532, 10.6, 23.1467738521847, 16.8, 18, 19, 
        19, 18.1, 20.7279009418787, 19.6, 16.5603027640122, 16.8, 
        20.4, 16.645176600332, 24, 30, 21, 17.0280648102222, 
        16, 18, 14.8, 21.2294686985081, 19, 20.4810999659765, 
        24, 22.5, 18, 18, 22.8, 22.5438126762685, 21.6, 18.3476806187319, 
        23.6468792985904, 22, 20.9566067726157, 18, 18, 19, 21.5, 
        16, 15.9298545285942, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 19.126942376477, 27.2533190531185, 
        21.0655082182363, 20.2732022254041, 21.5, 20.8, 20.9228674943268, 
        16, 22.3383190590289, 22.0044617650485, 25.8648006975606, 
        18.7138780607704, 21.5, 21, 21, 21.5, 28.5, 18.3696352636182, 
        22, 27.9382181877716, 20, 24, 21.5, 20.8185189089512, 
        20.7443687073363, 22.0270338726843, 25, 19.5, 19.9680101820812, 
        21, 22.9280411142108, 23.5, 20, 20.3745949513871, 21, 
        20, 20.4356206695399, 19.3436887881551, 21.5, 24, 22, 
        22.8600487418278, 18.5, 20, 22, 19.5, 17.6883961605131, 
        25, 19.1842535549345, 22.2987141933413, 22.168225786675, 
        25.7033037283795, 18.5, 19.5979405600547, 22.5, 23, 19.5, 
        24.5, 25, 19.881692335449, 19.5, 19.7170339505445, 20.5, 
        23, 20, 19.6950501561898, 24.5), mg = c(127, 100, 101, 
        113, 113.465979121168, 112, 96, 121, 97, 109.893013509385, 
        108.264282422152, 95.1464504033818, 89, 91, 102, 112, 
        120, 108.883998145703, 97.0489112189665, 116, 126, 102, 
        109.009395610724, 95, 116.47489001545, 124, 93, 100.736734106656, 
        107, 96, 101, 106, 104, 103.189915909194, 110, 99.666295182507, 
        110, 126.85703026743, 95.3069057790888, 109.395080947014, 
        117, 103.97589418866, 101, 139.811097644857, 107, 111, 
        102, 101, 116.404578783833, 108, 92, 108.365081777747, 
        111, 115, 113.000471465531, 118.355348043054, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 69.3600207579685, 
        151, 75.5559999672115, 86, 87, 139, 101, 91.8889182754765, 
        69.812219623709, 112, 77.415747882362, 107.053896150289, 
        86, 73.7543499752976, 78, 85, 99.9685941089, 99, 90, 
        88, 84, 70, 81, 103.603321158911, 72.9066047973857, 88, 
        106.724127692628, 98.6584912722934, 134, 85, 88, 69.2916432780245, 
        97, 88, 108.173146876381, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 83.5628084438955, 86.6699749516557, 
        87, 96, 119.140809989437, 102, 86, 135.021856875684, 
        85, 102.238225091533, 92, 88, 80, 122, 103.508277266311, 
        98, 106, 79.1946482277403, 94, 89, 110.762701233006, 
        88, 101, 95.7667659746579, 89, 97, 88.362100514666, 112, 
        102, 98.44706223774, 106.606121561143, 92, 113, 123, 
        114.378109420804, 116, 98, 103, 112.352784191848, 89, 
        97, 98, 89, 142.456483208516, 94.6712824572013, 106, 
        106, 90, 88, 94.9560548953141, 88, 105, 112, 96, 86, 
        91, 95, 102, 101.487997646965, 120, 101.007676526873), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 2.73784870710804, 
            2.5, 2.6, 3.36932827641446, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.50156977187278, 2.85, 2.8, 3.07533798428301, 
            3.3, 3.17049533598967, 3, 2.41, 2.61, 2.48, 2.53, 
            2.41375179911308, 3.12489920953089, 2.4, 2.95, 2.65, 
            3, 2.43723786846594, 2.42, 2.95, 2.49875399932524, 
            2.7, 2.6, 2.45, 2.48609080215244, 3.51464029720587, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 2.83628517470435, 2.87224666680091, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.58150589945361, 
            3.5, 1.89, 3.14920565535422, 2.21499400275933, 2.11, 
            2.53, 3.25154522495468, 1.1, 2.95, 1.88, 2.12972844802762, 
            2.14296517451557, 1.85865438365767, 2.21751340613076, 
            1.97260585201294, 1.9, 2.83, 2.54440001053213, 2.2, 
            1.36161652775077, 1.65, 2.47736447017446, 2.2, 1.96414867689599, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.27951649902756, 
            0.791883708058387, 2.55, 3.52, 3.09389087358254, 
            2.69113633122026, 1.7781081840159, 2.56, 2.5, 2.2, 
            1.89993973600888, 1.65, 2.44701357855964, 2.36, 2.74, 
            3.18, 2.55, 1.75, 2.2934988566756, 2.56, 2.46, 1.09842435967496, 
            2, 2.34532137510713, 2, 2.9, 3.18, 2.00268794069697, 
            2.840102599887, 2.86, 2.53002005220753, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.93461364533711, 1.29366032910538, 
            1.7, 2.16306688247021, 1.62, 1.38, 1.44418577094882, 
            1.62, 2.03021869165454, 0.778893417115176, 1.4, 1.87660775856228, 
            2, 1.38, 1.37345077359285, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.67490464026983, 1.48, 1.58876816282253, 
            1.8, 1.9, 1.45363740038471, 1.49832632077792, 2.3, 
            1.83, 1.59021032482671, 1.39, 1.35, 1.65126218427678, 
            1.7, 1.48, 1.55, 1.54706268531455, 1.25, 1.39, 1.68, 
            1.58392245262645, 1.54325508723778, 1.49454270524676, 
            1.65, 2.05), fla = c(2.82268225311552, 2.76, 3.57127247059225, 
            3.49, 2.9843324535372, 3.39, 2.52, 2.51, 2.98, 3.41146377979945, 
            3.32, 2.43, 2.76, 2.29879868670526, 3.64, 2.91, 3.14, 
            3.4, 3.25676690154984, 3.33101713687479, 3.63458020660721, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.72733969993592, 
            2.97, 2.33, 3.25, 3.19, 2.69, 2.86542407574469, 2.63333810785181, 
            2.98, 2.52627896702586, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.08081031971355, 2.92, 
            3.54, 3.02199494736778, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.01107083009269, 3.23, 3.67, 0.57, 2.52487676876108, 
            1.41, 1.79, 3.06685658045277, 1.75, 2.65, 2.57657565482996, 
            2, 1.3, 3.28685739865826, 1.02, 2.86, 1.28360109576862, 
            2.89, 2.14, 1.57, 1.87273420367477, 1.35823109472698, 
            1.85, 2.12980855932302, 2.26, 2.53, 1.58, 1.59, 2.21, 
            1.76426164113712, 1.69, 1.61, 1.69, 2.00564928243096, 
            2.15197252772799, 1.25, 1.46, 2.47664890031266, 2.87261115430642, 
            2.27, 0.99, 1.95035030859962, 3.17905124104873, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.94091616690195, 
            2.92, 2.47808962223848, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.49698128567786, 1.64, 3.45080915398389, 
            3.19038711538706, 1.25430068391506, 1.68294575084627, 
            3.03, 2.65, 3.15, 2.24, 1.83488914184084, 1.75, 1.11215045582152, 
            1.27705950497068, 0.500497450249986, 1.2, 0.58, 0.66, 
            0.47, 0.413090721529072, 0.632998851020468, 0.6, 
            0.5, 0.407248443748798, 0.52, 0.8, 1.18571521043347, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.835262243060206, 0.58, 1.31, 1.1, 0.92, 
            0.418120080927583, 0.6, 0.485807183101236, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.381527762530266, 
            0.7, 0.61, 0.623664494337661, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.313265662684665, 0.3, 0.24, 0.39, 
            0.27697262673894, 0.489470800272594, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.191960858306985, 
            0.168467225493991, 0.32, 0.17, 0.24, 0.25, 0.27, 
            0.26, 0.405180530842977, 0.270735989793614, 0.34, 
            0.27, 0.387325353961528, 0.26, 0.29, 0.1522108373414, 
            0.42, 0.259167864131269, 0.29, 0.26, 0.365337778303923, 
            0.29015475798208, 0.28, 0.2, 0.34, 0.226033307281236, 
            0.258240516900515, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.227469730238609, 
            0.343270318206838, 0.28, 0.206611345281795, 0.53, 
            0.32, 0.173397007080683, 0.45, 0.37, 0.26, 0.27, 
            0.635027188225473, 0.115256776591004, 0.457609549672697, 
            0.424339562264815, 0.415845692258567, 0.21, 0.13, 
            0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.344649508757531, 
            0.4, 0.61, 0.22, 0.3, 0.47592838944127, 0.402089023690535, 
            0.301128978404813, 0.396174839286074, 0.30343688497055, 
            0.5, 0.577648429579831, 0.25, 0.17, 0.32, 0.1758329826009, 
            0.318074089269308, 0.24, 0.284258852426025, 0.234760371887084, 
            0.29, 0.34, 0.37, 0.463751239178379, 0.66, 0.31014555364125, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.197033057912235, 
            0.42, 0.348461586580682, 0.52, 0.3, 0.306834072846865, 
            0.412161794235928, 0.418999156857741, 0.32, 0.305083927221987, 
            0.43, 0.3, 0.472347989408486, 0.386407315056312, 
            0.39, 0.430492654446141, 0.4, 0.682811765248649, 
            0.294184312110896, 0.382309722065587, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.544873147196446, 
            0.458559466766343, 0.445642390993542, 0.342683681610388, 
            0.43, 0.4, 0.47, 0.45, 0.396023405858942, 0.22, 0.264924022497135, 
            0.26, 0.61, 0.53, 0.483562221693247, 0.390853014232111, 
            0.63, 0.53, 0.52, 0.342716038077295, 0.5, 0.6, 0.4, 
            0.41, 0.475578286557734, 0.43, 0.4, 0.39, 0.453313547985439, 
            0.442130727899778, 0.48, 0.339935732017964, 0.556249328965293, 
            0.43, 0.43, 0.53, 0.534447008631189), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 1.79582876300883, 2.96, 1.46, 2.31018295098309, 
            1.71638645835151, 1.86, 1.66, 2.1, 1.2661907937544, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 1.49388302467908, 1.00546031163139, 1.54, 1.86, 
            1.36, 1.44, 1.37, 2.85375580753325, 2.32301422579295, 
            1.62319144339691, 2.75080135267364, 1.94284056360998, 
            2.03, 1.03412912511763, 2.19, 2.14, 2.38, 2.08, 1.70928963248257, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.30657982201043, 1.66, 
            2.54022700419683, 0.42, 2.24576188808463, 1.99787325715987, 
            0.73, 1.87, 1.03, 2.14171106364475, 1.89747496386023, 
            1.04, 2.39984350083554, 2.5, 1.46, 1.87, 1.03, 1.366206561772, 
            1.13899101838759, 1.15, 1.46, 1.06490239162686, 2.76, 
            1.95, 1.43, 1.69940530404442, 1.27185805812048, 1.62, 
            1.04506059911461, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 0.655229952099324, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.79948882906183, 
            1.42, 2.08630535355769, 1.63, 2.08, 2.49, 1.7676487273441, 
            1.81407423074156, 1.13778896423079, 1.32385100151978, 
            1.04, 1.75238095974775, 1.40525252287476, 1.61, 0.83, 
            1.87, 1.83, 1.45107776683196, 1.71, 1.24301176107117, 
            2.91, 1.35, 1.77, 1.39316776448431, 1.9, 1.04573944398168, 
            0.94, 1.73609762972847, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 0.646140602213886, 0.75, 1.34121077534531, 
            0.55, 0.57593986193396, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 0.919074883170375, 1.29962637327384, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.46962018419937, 
            1.25725855291624, 0.94, 1.71343439663868, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 3.91880446741283, 
            6.75, 5.15448344675843, 5.05, 5.23135243699437, 7.22, 
            5.75, 7.42503849895409, 5.6, 5.99395684829717, 7.5, 
            5.0430565885609, 7.57934066216995, 6.27992055835637, 
            8.7, 5.1, 5.65, 4.5, 4.12503979273474, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 6.9677485189241, 
            6.02179466819984, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            7.77695279100176, 8.9, 7.2, 5.64184680245204, 7.41138589994469, 
            6.3, 5.85, 6.25, 6.6548974744204, 4.49873847375374, 
            6.8, 8.70961378000716, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 5.67719713189618, 3.38, 
            3.74, 4.50107602530822, 3.21, 3.8, 4.6, 2.20470983028403, 
            4.19734580071814, 2.60986093287918, 2.5, 3.9, 2.97263281196013, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 6.15799731106106, 2.9, 5.86768809283806, 
            4.56397182236862, 4.39161479874828, 3.15414037186272, 
            2.8, 3.27551986406516, 2.94, 2.7, 3.4, 3.3, 2.7, 
            0.825170737464287, 2.9, 2.53145563426818, 1.95865514317848, 
            3.08, 2.9, 1.9, 1.49036613965315, 3.52182638939116, 
            3.4, 1.28, 3.25, 5.56440430555293, 0.289916381490303, 
            8.29312419355162, 2.8, 4.00857284574579, 3.94, 1.80589345137705, 
            2.12, 0.383460665664842, 8.04021853962554, 9.40605258559117, 
            8.84275827042369, 5, 5.45, 7.1, 6.69408609919134, 
            5, 5.7, 4.92, 7.82374325659644, 8.29291474456611, 
            4.35, 4.4, 8.21, 4, 4.9, 6.41761911055753, 8.42, 
            9.4, 10.0458519957281, 10.7594798398665, 7.1, 10.52, 
            7.6, 7.9, 9.2075365879035, 7.5, 13, 9.11662596940167, 
            7.65, 6.47921048217725, 5.58, 8.09893529110251, 10.1041572215417, 
            6.62, 10.68, 10.26, 8.53942658897697, 8.5, 8.21687488702444, 
            7.13569444175706, 9.7, 7.7, 7.3, 9.76057485059999, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.13372716619926, 
            0.86, 0.924937224504965, 1.05, 1.02, 0.951545345031691, 
            1.08, 0.984366916522461, 1.25, 1.02909289171306, 
            1.15, 1.25, 1.2, 1.28, 1.02681416966267, 1.13, 1.23, 
            0.87192274554264, 1.09, 1.00360909467719, 1.11, 1.09, 
            1.12, 1.13, 1.27208591638552, 1.02, 1.25141283129141, 
            1.04, 1.19, 1.09, 0.779300098195566, 1.25, 1.1, 1.00809919814553, 
            0.774717453983449, 1.10619275201654, 1.18, 1.27620260486538, 
            0.95, 0.91, 0.88, 0.82, 1.34196797732238, 0.87, 0.949562066821787, 
            1.32443975637107, 1.07, 1.12, 1.12, 1.28058716368165, 
            0.982110887126538, 1.0632194614891, 1.21610258057514, 
            1.34779839930867, 0.94, 1.07, 0.89, 0.838950820836633, 
            1.25, 0.98, 1.23, 1.22, 0.880271571682047, 0.907980988827681, 
            1.10700240529535, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.33729265279697, 1.23, 1.19, 0.96, 1.06, 
            1.19, 1.38, 0.982056207454548, 1.31, 0.84, 1.28100658823058, 
            1.23, 1.12007712055774, 0.910601525939724, 0.760995880994109, 
            1.07, 1.08, 1.05, 0.750591193577966, 1.15, 1.07697355334131, 
            1.42477237329489, 0.703226970792231, 0.899896126142796, 
            1.10791758484713, 1.42, 1.27, 0.885393165306996, 
            0.8, 1.21403792387779, 1.04, 0.829190446852135, 1, 
            0.951915436992399, 0.86, 1.05467940303258, 1.19959255592348, 
            0.9, 1.14542534571954, 1.1, 0.93, 1.71, 0.95, 0.963500436145204, 
            0.7, 1.17439022526801, 0.8, 0.843396153500305, 0.92, 
            0.73, 0.876339662062046, 0.86, 0.69, 0.97, 0.89, 
            0.79, 1.02532189374062, 0.74, 0.66, 0.78, 0.75, 0.73, 
            0.75, 0.82, 0.81, 0.746157790128558, 0.39211321147074, 
            0.872332760854397, 0.89, 0.693174200976326, 0.65, 
            0.6, 0.553801412390214, 0.54, 0.55, 0.430682115953915, 
            0.663854111101563, 0.48, 0.61, 0.56, 0.922755330053857, 
            0.6, 0.57, 0.67, 0.57, 0.704308993689102, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.408384931320368, 0.74, 
            0.67, 0.611272940509287, 0.688537574434884, 0.881995896512271, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 2.36638834625453, 2.93, 2.95948256016523, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 2.7708373831381, 
            3, 3.57992978899419, 2.65, 3.4981432578473, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.77723674132463, 3.2, 
            3.31585744614835, 2.77, 3.70879564681977, 2.60984060273688, 
            2.71, 2.88, 2.87, 3, 2.87, 3.80894483789714, 3.65498854952737, 
            3.05093720091986, 2.69, 3.11591618904959, 3.38, 3, 
            3.167991724994, 3, 3.35, 3.33, 3.80215011196125, 
            3.33, 2.75, 3.1, 2.22047736242435, 3.57308737768901, 
            3.26, 2.93, 3.2, 3.03, 2.83761596793811, 2.84, 2.87, 
            1.82, 3.34028354985754, 1.59, 2.46, 2.87, 3.22683883472388, 
            2.3, 3.18, 3.48, 2.64032533651554, 3.2110970997403, 
            1.82, 4.27714351777251, 2.74236069031785, 3.5, 3.13, 
            2.14, 2.48, 2.52, 3.17576655920989, 2.94782321305452, 
            3.70202125641114, 3.14, 2.72, 2.01, 3.0648360400023, 
            2.92355627562174, 2.58557763783052, 3.21, 2.75, 3.21, 
            3.0716418663545, 2.65, 3.41618160071715, 3.3, 2.95604757506243, 
            2.63, 1.61051115907069, 2.74, 3.33193456345098, 2.83, 
            2.96, 2.77, 3.38, 3.36217829407802, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.02401662552052, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            2.27612255590249, 3.64, 3.33617574651943, 2.84, 2.59657610341607, 
            2.78, 3.03124537675011, 2.29255436713483, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.98364027403082, 
            1.51043428850742, 2.68017296056107, 2.05, 2, 1.91222413436439, 
            1.33, 1.81932645513737, 1.62, 1.33, 2.12266933362011, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 2.08579165641367, 2.44842999133904, 1.75, 1.68, 
            1.75, 1.2256349549053, 1.28248217654057, 1.8661269834534, 
            1.53443021757984, 1.16244761144779, 2.08345694056497, 
            2.56640384597033, 1.82493654517359, 2.1858753019723, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 911.636590827471, 
            1076.00033196792, 735, 1450, 1290, 1295, 1045, 1045, 
            1211.12050081612, 1280, 1320, 1586.13063658482, 1547, 
            1310, 1280, 1012.4671248071, 1680, 845, 913.695884460312, 
            770, 1035, 1015, 845, 1047.63330791394, 1195, 1285, 
            915, 1035, 1285, 1004.36028102469, 1151.59114498993, 
            1235, 690.679091203377, 920, 880, 1081.28086342439, 
            424.945701829353, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1008.96996690874, 1150, 1265, 
            1190, 1296.53745961719, 1060, 1120, 970, 1270, 1285, 
            728.054420749676, 560.192834438849, 1059.70151194634, 
            630, 315.484358252487, 756.641356897212, 478.791109348569, 
            502, 508.270014117405, 527.108759306631, 718, 870, 
            410, 472, 1034.90505706621, 886, 632.062059557028, 
            392, 427.20110255073, 442.367673313951, 463, 278, 
            714, 621.957989271875, 515, 712.195080513117, 374.559137960306, 
            962.23146174384, 562, 680, 625, 480, 450, 495, 498.02360705795, 
            345, 494.270523980162, 625, 428, 660, 406, 343.296826000984, 
            562, 438, 415, 672, 772.922764088534, 510, 418.076266754646, 
            312, 680, 562, 325, 476.671261960146, 434, 385, 353.672537657233, 
            829.5212534197, 382.864504741192, 372, 564, 346.77131820413, 
            950.834392521518, 365, 380, 389.121081343628, 378, 
            478.772061305648, 466, 625.909985752827, 580, 465.407745739805, 
            599.407235432411, 560, 600, 474.800343458254, 695, 
            720, 515, 580, 590, 600, 549.201563406923, 520, 550, 
            457.798617462279, 830, 415, 523.699824038104, 627.417276059918, 
            550, 500, 480, 425, 675, 640, 433.363996595689, 480, 
            880, 660, 675.077174463685, 544.153738839371, 680, 
            570, 606.353406130865, 615, 532.838266680964, 695, 
            685, 776.624191943515, 541.851373876983, 510, 470, 
            886.747509860769, 740, 389.938487603865, 835, 840, 
            419.139432750313)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.9732394925565, 
        13.24, 13.5924650605161, 14.39, 14.06, 13.627273990826, 
        13.86, 13.2598677573495, 14.2781014938665, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.3310494761221, 13.64, 12.9464189320715, 
        12.93, 13.71, 12.85, 13.5, 13.7224079827162, 13.39, 13.3, 
        13.87, 12.8824970417075, 13.73, 13.58, 13.68, 13.76, 
        13.1984862384065, 13.1399033855484, 13.6778285589283, 
        13.05, 12.9596593521991, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 13.0626693510283, 13.9, 13.4973702401742, 
        13.8179401553592, 13.05, 13.7322879466435, 13.82, 13.2491005247795, 
        13.74, 14.0223054016252, 14.22, 13.29, 14.2942557443568, 
        12.37, 12.33, 12.64, 11.4583003936826, 12.37, 12.17, 
        12.37, 12.719742448921, 12.5400450773914, 13.34, 12.21, 
        12.6260960055425, 13.86, 12.8387161859294, 12.99, 11.96, 
        11.66, 11.8358344079069, 11.84, 13.9675839510633, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 12.730239424356, 12.16, 
        12.4810408258328, 12.4360081084399, 12.08, 12.08, 12, 
        12.69, 12.5690463240753, 11.62, 12.2166135781457, 11.81, 
        12.29, 14.2285090160565, 12.29, 12.08, 12.6, 13.0913586922772, 
        12.5163112219574, 12.51, 12.0454335919254, 11.9861931772592, 
        12.72, 12.22, 11.61, 11.4508763431925, 12.4034602212718, 
        11.76, 11.8233885843382, 12.08, 11.03, 11.82, 11.4504574506974, 
        12.1593199373854, 12.3788781089987, 11.45, 13.3965385417073, 
        12.42, 13.05, 11.87, 12.07, 12.6987962526289, 11.79, 
        11.6666526314685, 12.04, 12.86, 12.88, 12.7171465201154, 
        12.7, 12.51, 12.7786144404373, 12.25, 12.53, 12.5552568255685, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.4982939513345, 
        12.0141647762966, 12.87, 13.32, 12.1545330087898, 11.4966756240889, 
        12.79, 11.6313471758013, 13.23, 12.281413525473, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.3194527235884, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.930483030916, 13.963405121001, 
        13.58, 13.4, 12.6457503989663, 12.77, 14.16, 13.6523082965082, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 0.885890237687826, 
        1.0673232588127, 1.95, 2.75204777322646, 2.09434319368527, 
        1.87, 2.15, 1.2890241133101, 1.35, 2.16, 1.48, 0.529828557655717, 
        1.73, 1.87, 1.17872207743802, 1.92, 3.43885173639159, 
        -0.782403853868717, 3.1, 1.63, 3.8, 0.624934877345113, 
        1.6, 1.81, 2.795466512921, 2.17253975008252, 0.503454737487613, 
        1.9, 2.02646562514354, 1.5, 2.57814300945716, 1.83, 1.53, 
        1.7641891933139, 1.39308806862841, 2.82833464390989, 
        1.65, 1.5, 4.50225704607774, 1.71, 1.55305134518907, 
        3.99776094115947, 3.98, -1.10951971097563, 4.04, -0.626277530701727, 
        1.68, 0.382957445809741, 1.36397550189181, 2.40487447290018, 
        1.65, 4.05854198025088, 1.9, 1.67, -0.779841824398581, 
        1.7, 1.97, 3.62123438891919, 0.94, 3.19899342575994, 
        1.36, 1.25, 2.29221183853693, 4.8788583105352, 1.21, 
        1.01, 1.17, 3.02682127744569, 1.74905147118511, 1.16027245950068, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 5.66115103722129, 
        0.99, 3.87, 1.98160387700857, 2.20576935952322, 1.13, 
        5.12527508982775, 0.905753376613783, 0.98, 1.61, 1.67, 
        2.88832410953373, 2.03170439281919, 0.660640770957968, 
        1.51, 1.50632296320751, 1.93605411628874, 2.37386271110999, 
        3.97218575072282, 5.50424081563304, 1.41, 2.61365471752178, 
        -0.681000986289279, 2.08, 2.39991453670236, 4.43108106947323, 
        1.72, 1.68793805476057, 2.55, 1.73, 2.51197943369715, 
        1.29, 1.75500065997575, 3.74, 3.41526002636393, 2.68, 
        0.74, 1.39, -1.3655286649221, 0.852583702402723, 1.61, 
        3.43, 2.34475686880773, 2.4, 2.05, 4.43, 2.59841184493921, 
        4.31, 2.16, 1.53, 2.13, 1.08424696073801, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 4.05181393864992, 1.49637695754005, 
        5.51, 3.59, 2.99051712321113, 2.81, 2.56, 4.45967637602665, 
        -0.341824655571146, 3.88, 5.78779847990231, 5.04, 4.61, 
        3.24, 5.089838401614, 3.12, 2.67, 1.9, 1.14316803640029, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.13900354735911, 
        3.4166031860814, 3.45, 2.76, 4.9690941270402, 1.25258339303236, 
        3.37, 2.58, 4.6, 3.75259166876878, 2.39, 5.61252315167243, 
        5.65, 3.91, 3.21773569977848, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.27606923669179, 2.50217376341569, 
        2.45, 2.61, 2.31813161797123, 2.35530857433641, 2.3, 
        2.51937563960735, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
        2.48, 2.56, 2.53971758080727, 2.65, 2.32455107564895, 
        2.52, 2.61, 3.22, 2.46900646687199, 2.07476604286473, 
        2.61843698373039, 2.48494781260478, 2.62748375196544, 
        2.36, 2.5093870700862, 2.7, 2.65, 2.44549795840768, 2.08855544912196, 
        2.55, 2.1, 2.51, 2.08093930196425, 2.12, 2.59, 2.29, 
        2.1, 2.44, 2.43960628977368, 2.12, 2.4, 2.27, 2.04, 2.80333282731888, 
        2.75477296653065, 2.52148454807646, 2.37608199231236, 
        2.26297953226609, 2.3, 2.68, 2.5, 1.36, 2.48875442588519, 
        2.31607222713765, 1.92, 2.16, 2.53, 2.56, 1.84840224945748, 
        1.92, 2.21500784628558, 1.75, 2.21, 2.67, 2.24, 2.6, 
        2.3, 2.08997343343702, 1.71, 1.90301381042219, 1.9961323855888, 
        2.4, 2, 2.79214515022098, 2.51, 2.32, 2.58, 2.24, 2.62336588614793, 
        2.62, 2.0729673571037, 2.3, 2.32, 1.93777832291978, 2.26, 
        2.22, 2.28, 2.2, 2.74, 1.98, 2.31733193320293, 1.95166768556885, 
        1.7, 1.7884332386607, 2.46, 1.88, 2.36290442490552, 2.27, 
        2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 
        1.99, 2.10980957636642, 1.80300824910695, 2, 2.42, 3.23, 
        2.16957654295873, 2.13, 2.39, 2.17, 2.29, 2.52852152395126, 
        2.09244293972061, 2.38, 2.37708009342346, 2.4, 2.4, 2.36, 
        2.25, 2.48955383903462, 2.54, 2.64, 2.19, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.85840776433761, 2.76923858514928, 
        2.48, 2.19555461825192, 2.36, 2.62, 2.48, 1.99898333154169, 
        2.28, 2.22622210825528, 2.32, 2.38, 1.86820367970111, 
        2.16383016820987, 2.64, 2.38, 2.54, 2.54769217293595, 
        2.35, 2.3, 2.74372283843654, 2.6, 2.29285248300457, 2.69, 
        2.15029223374407, 2.32, 2.28, 2.4881686980096, 2.4438524367063, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 19.3206877007641, 
        16.3256735174862, 16.8, 21, 15.2, 18.8200895010992, 17.6, 
        14, 16, 12.3376437805647, 16.8, 16, 11.4, 11.521076041354, 
        21.5720304012754, 19.9788631519872, 20, 14.3140074493222, 
        15.2, 16, 18.6, 16.5271546739334, 17.8, 20, 24.7861205799005, 
        16.1, 17, 22.8351113055329, 17.2301890279304, 17.1691938913591, 
        19.1, 14.4537500785034, 19.1021763929753, 19, 20.5, 15.5, 
        15.4797246865624, 15.5, 13.2, 16.2, 18.8, 21.4691748188777, 
        20.8167937604897, 17, 14.5328161801239, 16, 13.1651488185772, 
        18.8, 17.4, 12.4, 18.5755193316258, 14.9378717346593, 
        19.494888921078, 15.8755726671703, 20.5, 16.3, 16.8, 
        17.7361781110508, 10.6, 19.0531685332775, 16.8, 18, 19, 
        19, 18.1, 21.8482848451872, 19.6, 18.31077047243, 16.8, 
        20.4, 17.225366110647, 24, 30, 21, 18.741924365228, 16, 
        18, 14.8, 16.6644648099272, 19, 24.6083595111635, 24, 
        22.5, 18, 18, 22.8, 15.5683900472011, 21.6, 19.5378668672924, 
        26.4833633482236, 22, 21.225740872426, 18, 18, 19, 21.5, 
        16, 19.8447567205711, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 22.2788176221844, 15.8781718712397, 
        17.1448528063348, 18.5009126464052, 21.5, 20.8, 24.1533306693115, 
        16, 24.4474598278348, 18.6603291972675, 20.7717520621679, 
        19.23066770699, 21.5, 21, 21, 21.5, 28.5, 19.6965046891919, 
        22, 18.4861684034934, 20, 24, 21.5, 23.3872892408705, 
        20.0887808483579, 18.7318954062079, 25, 19.5, 22.3625824415328, 
        21, 20.9819348678298, 23.5, 20, 19.7568539428487, 21, 
        20, 19.0594305112269, 21.2769312367737, 21.5, 24, 22, 
        20.3429088886516, 18.5, 20, 22, 19.5, 12.0169701068029, 
        25, 23.0907455564484, 14.7285727141735, 19.3253476664587, 
        22.6979753151124, 18.5, 18.0440116943606, 22.5, 23, 19.5, 
        24.5, 25, 19.0060158550979, 19.5, 20.6257723590717, 20.5, 
        23, 20, 21.226673386754, 24.5), mg = c(127, 100, 101, 
        113, 96.0325597072045, 112, 96, 121, 97, 83.0947528391081, 
        119.821388789066, 107.124048746166, 89, 91, 102, 112, 
        120, 108.519021318526, 96.5780707571703, 116, 126, 102, 
        101.51955387551, 95, 118.93302766363, 124, 93, 113.743079393798, 
        107, 96, 101, 106, 104, 107.027523867385, 110, 107.262178972746, 
        110, 104.588439218744, 101.850494354772, 92.6808952527501, 
        117, 119.691101244206, 101, 62.2089787341194, 107, 111, 
        102, 101, 110.045967340395, 108, 92, 116.088326521589, 
        111, 115, 108.829052360531, 108.072592010491, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 81.6369397328926, 
        151, 83.7557619052554, 86, 87, 139, 101, 113.379559536549, 
        74.0922930800386, 112, 116.788486036818, 92.6015855831205, 
        86, 102.405583725205, 78, 85, 90.9596250866161, 99, 90, 
        88, 84, 70, 81, 112.144251214524, 85.5654526297903, 88, 
        109.358588306658, 108.825794128945, 134, 85, 88, 74.4508185833205, 
        97, 88, 95.9757059267779, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 93.4693943391765, 82.2096412125505, 
        87, 96, 106.312146934804, 102, 86, 129.547363682408, 
        85, 102.378791305595, 92, 88, 80, 122, 100.494581450845, 
        98, 106, 75.0709073922108, 94, 89, 96.0529937996846, 
        88, 101, 92.8976870910082, 89, 97, 70.9043051979912, 
        112, 102, 103.188680347977, 106.794265997901, 92, 113, 
        123, 106.09739060905, 116, 98, 103, 107.327017276053, 
        89, 97, 98, 89, 79.2096781313919, 110.618249726059, 106, 
        106, 90, 88, 94.3619468232965, 88, 105, 112, 96, 86, 
        91, 95, 102, 95.62281117391, 120, 106.664838218852), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.04314283253328, 
            2.5, 2.6, 2.61740202315595, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.4427138881428, 2.85, 2.8, 3.55682356603601, 
            3.3, 2.71930871548388, 3, 2.41, 2.61, 2.48, 2.53, 
            2.26475391334346, 2.92741380068056, 2.4, 2.95, 2.65, 
            3, 2.79485454396013, 2.42, 2.95, 2.56551086708378, 
            2.7, 2.6, 2.45, 3.03356844299811, 3.32203712937868, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 1.9253382807296, 2.72490555571762, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.1056217362106, 
            3.5, 1.89, 2.87984676402993, 2.6614297846408, 2.11, 
            2.53, 3.04417078378195, 1.1, 2.95, 1.88, 2.30208471665193, 
            2.03676488703429, 2.11792283025845, 2.40009596683677, 
            2.35580335548519, 1.9, 2.83, 2.66076945023624, 2.2, 
            1.82977884406384, 1.65, 2.52763641060253, 2.2, 1.99617242155119, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.06563168159028, 
            1.10274124864312, 2.55, 3.52, 2.59188680166486, 2.3803581423609, 
            1.83161241844535, 2.56, 2.5, 2.2, 1.94181774786349, 
            1.65, 2.31662832182439, 2.36, 2.74, 3.18, 2.55, 1.75, 
            1.95664058122476, 2.56, 2.46, 1.82677111972796, 2, 
            1.68095920733933, 2, 2.9, 3.18, 3.01420239856852, 
            2.43446990740386, 2.86, 2.62329339484904, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.11376301470134, 0.95334984631247, 
            1.7, 2.0635348110772, 1.62, 1.38, 0.914406099444656, 
            1.62, 2.13396909911967, 1.62843067919444, 1.4, 2.32962239090201, 
            2, 1.38, 1.22739911599607, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.6483102213613, 1.48, 1.45088537738685, 
            1.8, 1.9, 1.36427508699393, 1.9461568832796, 2.3, 
            1.83, 1.84576693207121, 1.39, 1.35, 1.90638517288779, 
            1.7, 1.48, 1.55, 1.86917886489623, 1.25, 1.39, 1.68, 
            1.69531965770764, 1.81947361787082, 1.37893859665342, 
            1.65, 2.05), fla = c(2.87330006396987, 2.76, 3.88915162328894, 
            3.49, 1.93955996079019, 3.39, 2.52, 2.51, 2.98, 3.76773582864797, 
            3.32, 2.43, 2.76, 2.75227879858377, 3.64, 2.91, 3.14, 
            3.4, 3.26794068385429, 3.05887916036214, 3.85393436771355, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 1.94935508741508, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.35667381553067, 2.75771896385982, 
            2.98, 2.56355517070767, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.22147770284586, 2.92, 
            3.54, 3.65617761975159, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.6539633226482, 3.23, 3.67, 0.57, 1.48783774995417, 
            1.41, 1.79, 3.00264984854308, 1.75, 2.65, 2.31996930259179, 
            2, 1.3, 2.90143906719469, 1.02, 2.86, 1.79314016253943, 
            2.89, 2.14, 1.57, 2.20721371374147, 2.51096023642645, 
            1.85, 2.55932900412488, 2.26, 2.53, 1.58, 1.59, 2.21, 
            1.21017102159559, 1.69, 1.61, 1.69, 2.11631290978994, 
            1.6946299184736, 1.25, 1.46, 2.05862171098149, 3.04584581416652, 
            2.27, 0.99, 1.71453588076577, 3.42640220852872, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.93592520437762, 
            2.92, 2.86204488977523, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.29260195860831, 1.64, 3.35546594820305, 
            2.41357354225245, 2.4792285422057, 2.10626822192806, 
            3.03, 2.65, 3.15, 2.24, 2.08266135582103, 1.75, 0.095568050801901, 
            0.574838579243318, 0.711010494022154, 1.2, 0.58, 
            0.66, 0.47, 0.310187138082426, 0.379523482378605, 
            0.6, 0.5, 0.673349806555163, 0.52, 0.8, 1.18356375319561, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.421902581761532, 0.58, 1.31, 1.1, 0.92, 
            0.385868021025568, 0.6, 0.583086313419116, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.40982620700814, 
            0.7, 0.61, 0.689088249043625, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.280990897627611, 0.3, 0.24, 0.39, 
            0.369486493728984, 0.345783733021131, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.253807525095901, 
            0.362951319773501, 0.32, 0.17, 0.24, 0.25, 0.27, 
            0.26, 0.364089105272575, 0.322474078615735, 0.34, 
            0.27, 0.253132862165364, 0.26, 0.29, 0.211181745121964, 
            0.42, 0.270420599269098, 0.29, 0.26, 0.246848551894058, 
            0.359317591622118, 0.28, 0.2, 0.34, 0.0847644121151589, 
            0.196747495459365, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.42835546361132, 
            0.164508694955991, 0.28, 0.487612808521765, 0.53, 
            0.32, 0.4061359880596, 0.45, 0.37, 0.26, 0.27, 0.486313871148569, 
            0.12832524157278, 0.227152522848632, 0.258786833177821, 
            0.306487855278474, 0.21, 0.13, 0.34, 0.24, 0.43, 
            0.35, 0.43, 0.3, 0.393313418353746, 0.4, 0.61, 0.22, 
            0.3, 0.377887774056745, 0.480163239821798, 0.383471115621592, 
            0.404219401107953, 0.512146105531858, 0.5, 0.411859581066583, 
            0.25, 0.17, 0.32, 0.452125118275897, 0.461254945327795, 
            0.24, 0.476992610036187, 0.320616778670904, 0.29, 
            0.34, 0.37, 0.45596646395944, 0.66, 0.380974601139512, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.382129506843521, 
            0.42, 0.398345820119544, 0.52, 0.3, 0.183764922806094, 
            0.217108918866371, 0.358858056454255, 0.32, 0.481506159647677, 
            0.43, 0.3, 0.615862767290169, 0.217548493879858, 
            0.39, 0.422567920778894, 0.4, 0.566555714709772, 
            0.415597412510034, 0.543641619743028, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.439594297466851, 
            0.426278660159439, 0.359601960613423, 0.385605847977602, 
            0.43, 0.4, 0.47, 0.45, 0.52262574678613, 0.22, 0.309979647006425, 
            0.26, 0.61, 0.53, 0.483884795090769, 0.468143409468256, 
            0.63, 0.53, 0.52, 0.393631679097806, 0.5, 0.6, 0.4, 
            0.41, 0.667149995167642, 0.43, 0.4, 0.39, 0.495159494658709, 
            0.487785834599119, 0.48, 0.589585261665006, 0.475042575260153, 
            0.43, 0.43, 0.53, 0.549092091757683), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 1.2455748423938, 2.96, 1.46, 2.57819367110859, 
            1.68099930242535, 1.86, 1.66, 2.1, 1.21328165720458, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 2.42339471986806, 2.62478685613486, 1.54, 1.86, 
            1.36, 1.44, 1.37, 0.926743148087258, 2.27333406066419, 
            2.47905651325422, 1.09435307987383, 1.54299507566125, 
            2.03, 1.20206706321592, 2.19, 2.14, 2.38, 2.08, 1.44678904702087, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.02829609208366, 1.66, 
            2.25848948586812, 0.42, 1.20967078940846, 1.61566768129885, 
            0.73, 1.87, 1.03, 2.58803102973392, 1.05451422755329, 
            1.04, 0.598150638257415, 2.5, 1.46, 1.87, 1.03, 2.27990341767456, 
            1.49245210879286, 1.15, 1.46, 2.02949755507358, 2.76, 
            1.95, 1.43, 2.52353527599931, 1.31216960583081, 1.62, 
            1.65144115053749, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 2.47024916073229, 1.99, 1.35, 3.28, 1.56, 1.77, 
            1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.970317140124549, 
            1.42, 1.56084903253124, 1.63, 2.08, 2.49, 1.37331664413767, 
            1.1318547936881, 1.29563395401754, 1.17980280459963, 
            1.04, 1.73706249177402, 1.33981135603104, 1.61, 0.83, 
            1.87, 1.83, 1.21868651358622, 1.71, 1.39711742549833, 
            2.91, 1.35, 1.77, 1.50810284594456, 1.9, 1.31247234077203, 
            0.94, 1.21348660406609, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 1.11595059536539, 0.75, 1.03428788118085, 
            0.55, 1.20069861417997, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 1.54214399529624, 1.1914708393865, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.43255061589243, 
            1.13037734604135, 0.94, 1.58602759587283, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 3.54966640593471, 
            6.75, 4.44346458503337, 5.05, 6.49395849112111, 7.22, 
            5.75, 6.95997740697371, 5.6, 7.20147288518965, 7.5, 
            4.83790767436364, 7.0672203178447, 5.1605471358946, 
            8.7, 5.1, 5.65, 4.5, 4.8336282694586, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.77764048065827, 
            5.13232366012678, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            5.59810014874204, 8.9, 7.2, 5.49628921057997, 4.76630574416465, 
            6.3, 5.85, 6.25, 8.48495664271804, 5.08217955938134, 
            6.8, 9.20180073124709, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 4.1989853387248, 3.38, 
            3.74, 3.40041944213532, 3.21, 3.8, 4.6, 4.29673052133985, 
            5.45656127727545, 3.82861657005924, 2.5, 3.9, 1.81227606772959, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 6.57028722229607, 2.9, 4.21896493801891, 
            7.1595665284004, 5.20986791960604, 3.65967487397606, 
            2.8, 3.19094492882076, 2.94, 2.7, 3.4, 3.3, 2.7, 
            1.06782614710555, 2.9, 3.51304829227853, 3.30762232894506, 
            3.08, 2.9, 1.9, 1.39727543483039, 1.82053417142853, 
            3.4, 1.28, 3.25, 0.289192378868506, 2.90803479220878, 
            3.22371910622434, 2.8, 2.30131869444993, 3.94, 2.32443816277373, 
            2.12, 0.394356281789822, 8.1061199553905, 9.22896347234031, 
            9.31788549640474, 5, 5.45, 7.1, 7.17458340700789, 
            5, 5.7, 4.92, 8.00639928862883, 9.86375846427541, 
            4.35, 4.4, 8.21, 4, 4.9, 6.72484140024752, 8.42, 
            9.4, 1.9152151187226, 10.4201620351554, 7.1, 10.52, 
            7.6, 7.9, 7.42485292313577, 7.5, 13, 8.09525550655013, 
            7.65, 8.66590029904685, 5.58, 8.39547603673263, 10.2420443370062, 
            6.62, 10.68, 10.26, 7.87809041107416, 8.5, 7.44771498655068, 
            7.84343995051412, 9.7, 7.7, 7.3, 9.06546013174856, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.05611370254135, 
            0.86, 1.17429779627297, 1.05, 1.02, 1.02943335837049, 
            1.08, 1.07904582421767, 1.25, 0.994299744059989, 
            1.15, 1.25, 1.2, 1.28, 0.937470209604577, 1.13, 1.23, 
            0.793549131840459, 1.09, 1.17921175712422, 1.11, 
            1.09, 1.12, 1.13, 1.25629568995709, 1.02, 1.1068205682627, 
            1.04, 1.19, 1.09, 1.17687632362703, 1.25, 1.1, 1.28557949761584, 
            1.04519743854587, 0.954855580058701, 1.18, 1.06301415951544, 
            0.95, 0.91, 0.88, 0.82, 1.12763833559211, 0.87, 1.4752747173821, 
            0.8251279410679, 1.07, 1.12, 1.12, 1.17464446191691, 
            0.951550228385608, 0.947774622846812, 0.959868066179983, 
            1.59649555559253, 0.94, 1.07, 0.89, 0.755779406490562, 
            1.25, 0.98, 1.23, 1.22, 1.06251758266475, 1.29500567664094, 
            1.12566953310671, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.39548177466958, 1.23, 1.19, 0.96, 1.06, 
            1.19, 1.38, 1.30868821329296, 1.31, 0.84, 1.18871797761515, 
            1.23, 1.20053301619598, 1.04055524570079, 1.02168345938606, 
            1.07, 1.08, 1.05, 0.989378314022892, 1.15, 1.03048496046447, 
            1.08162614321054, 0.649124165025672, 1.32293933212835, 
            1.16214495464336, 1.42, 1.27, 1.07181352719867, 0.8, 
            1.24957499565678, 1.04, 1.36288177967506, 1, 0.985379905188662, 
            0.86, 1.31225522045432, 0.81328643509274, 0.9, 1.22190463492455, 
            1.1, 0.93, 1.71, 0.95, 1.24944423074701, 0.7, 0.998688066582878, 
            0.8, 1.61744933749361, 0.92, 0.73, 0.973329136619988, 
            0.86, 0.69, 0.97, 0.89, 0.79, 0.790504976649348, 
            0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.879932093980516, 
            0.706640286009924, 0.651171513854181, 0.89, 0.984519025592676, 
            0.65, 0.6, 0.626092763348461, 0.54, 0.55, 0.464235852429524, 
            1.01590933709663, 0.48, 0.61, 0.56, 0.681226328701244, 
            0.6, 0.57, 0.67, 0.57, 0.990062845046571, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.957437405535537, 0.74, 
            0.67, 0.845713839668192, 0.627085206330128, 0.372340601984905, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 3.20601569621742, 2.93, 2.89615342273767, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.79768085587039, 
            3, 3.33182110351264, 2.65, 3.27693278912743, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.76328618411835, 3.2, 
            2.83136302821668, 2.77, 3.31613020055105, 2.56738706352051, 
            2.71, 2.88, 2.87, 3, 2.87, 2.74837376927256, 3.63413631984052, 
            2.31223158337995, 2.69, 3.39973952687318, 3.38, 3, 
            3.39763470691285, 3, 3.35, 3.33, 2.86564639523275, 
            3.33, 2.75, 3.1, 3.25538934219633, 3.00520798159818, 
            3.26, 2.93, 3.2, 3.03, 3.78629671051032, 2.84, 2.87, 
            1.82, 2.27102428734628, 1.59, 2.46, 2.87, 2.50026813741111, 
            2.3, 3.18, 3.48, 3.13215805406549, 3.35514951156935, 
            1.82, 3.70740601930524, 3.5195878361155, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.52588854038891, 3.15106114308794, 
            3.45217238456737, 3.14, 2.72, 2.01, 2.25619876689563, 
            2.94892431582082, 2.308151988179, 3.21, 2.75, 3.21, 
            2.9978287844551, 2.65, 2.65391102717392, 3.3, 3.68561900365835, 
            2.63, 1.44277904546295, 2.74, 4.0637552972571, 2.83, 
            2.96, 2.77, 3.38, 2.75161243973105, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.30574331741324, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            3.75576928146374, 3.64, 3.73317393503419, 2.84, 2.53659712537388, 
            2.78, 2.68005045886197, 1.50074385966156, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.02008929103013, 
            1.41130931439512, 2.72508280082676, 2.05, 2, 1.65960996201878, 
            1.33, 1.87357673478602, 1.62, 1.33, 2.39906966533965, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 1.85122434048693, 2.42593278448758, 1.75, 1.68, 
            1.75, 1.8494105881088, 2.04496932343146, 2.2859028988029, 
            2.25124158537716, 1.84565448516602, 1.4762468253431, 
            1.38765723888324, 1.70694197562892, 1.69563566193074, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 1375.1058522515, 
            1441.99280976122, 735, 1450, 1290, 1295, 1045, 1045, 
            1043.20774857222, 1280, 1320, 1071.25178225279, 1547, 
            1310, 1280, 499.31091589525, 1680, 845, 1474.41319440265, 
            770, 1035, 1015, 845, 911.313625943893, 1195, 1285, 
            915, 1035, 1285, 1030.69675904656, 1118.60195891257, 
            1235, 1087.26145843321, 920, 880, 1010.1118439417, 
            1165.51695653436, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 700.252282394984, 1150, 1265, 
            1190, 1090.38021806593, 1060, 1120, 970, 1270, 1285, 
            424.867178369301, 813.18003232251, 832.003784471125, 
            630, 687.003601060631, 769.848884686476, 793.065434642101, 
            502, 174.958526752583, 454.932348953836, 718, 870, 
            410, 472, 518.320081773266, 886, 629.945488119064, 
            392, 169.35079151512, 1265.22838814731, 463, 278, 
            714, 477.819553867483, 515, 475.319019870402, 419.904025962002, 
            463.920749634377, 562, 680, 625, 480, 450, 495, 658.101193910056, 
            345, 472.893493199861, 625, 428, 660, 406, 406.707887990212, 
            562, 438, 415, 672, 388.158445599663, 510, 523.43207144977, 
            312, 680, 562, 325, 892.353834136319, 434, 385, 684.371819631767, 
            262.121717519398, 130.712963554276, 372, 564, 231.664605742941, 
            783.643728893364, 365, 380, 76.432225862403, 378, 
            901.622778330172, 466, 516.512400873787, 580, 867.244291009242, 
            558.404645883825, 560, 600, 550.641818939214, 695, 
            720, 515, 580, 590, 600, 741.372359188292, 520, 550, 
            783.830578796596, 830, 415, 661.693227322074, 354.325101243194, 
            550, 500, 480, 425, 675, 640, 319.249707681213, 480, 
            880, 660, 788.737528759091, 383.947416900429, 680, 
            570, 656.511400995881, 615, 383.100320046822, 695, 
            685, 737.969118896183, 794.559424225346, 510, 470, 
            693.58177719554, 740, 586.640240582447, 835, 840, 
            589.463590139746)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.6487961536583, 
        13.24, 14.5268708715502, 14.39, 14.06, 14.1762716336061, 
        13.86, 13.52692257583, 14.1579068262291, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.078307362268, 13.64, 13.5796991351775, 
        12.93, 13.71, 12.85, 13.5, 14.1058346416415, 13.39, 13.3, 
        13.87, 15.0844842637658, 13.73, 13.58, 13.68, 13.76, 
        12.2865303358689, 14.2526022670372, 11.8144583025664, 
        13.05, 11.7711367078294, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 14.4082365275679, 13.9, 13.8104947376494, 
        13.8366880803347, 13.05, 14.5160812057922, 13.82, 13.8011568331561, 
        13.74, 15.1461221683539, 14.22, 13.29, 13.9679364901953, 
        12.37, 12.33, 12.64, 11.4850651290412, 12.37, 12.17, 
        12.37, 12.6698577820111, 12.6680310111445, 13.34, 12.21, 
        11.6796446610696, 13.86, 12.5349845780266, 12.99, 11.96, 
        11.66, 12.4212423193886, 11.84, 12.0961783124533, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.7692209965131, 12.16, 
        12.8019464969396, 12.6870435092236, 12.08, 12.08, 12, 
        12.69, 12.6270083618119, 11.62, 11.9433416461006, 11.81, 
        12.29, 11.5269586106776, 12.29, 12.08, 12.6, 12.9301724711844, 
        11.9328825357895, 12.51, 12.6126783326716, 12.180932253292, 
        12.72, 12.22, 11.61, 12.1564848696336, 12.6453875350535, 
        11.76, 12.4112264654098, 12.08, 11.03, 11.82, 12.1709467471621, 
        12.2367314528055, 11.743750987942, 11.45, 14.877329769597, 
        12.42, 13.05, 11.87, 12.07, 11.5781003701701, 11.79, 
        12.3042695877027, 12.04, 12.86, 12.88, 13.1297851427839, 
        12.7, 12.51, 13.1522635247443, 12.25, 12.53, 12.6862566393565, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.2657750321584, 
        11.3622791099609, 12.87, 13.32, 11.7662861687385, 13.0061309721694, 
        12.79, 12.3868872612028, 13.23, 12.2980615806431, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.0095315274616, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.1308428086219, 13.301814192581, 
        13.58, 13.4, 12.5274225956912, 12.77, 14.16, 14.5398382820254, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 1.6895693823967, 
        2.81035905719391, 1.95, 2.69377591297357, 2.58514997898188, 
        1.87, 2.15, 0.491207286942471, 1.35, 2.16, 1.48, 0.0251235601448765, 
        1.73, 1.87, 2.38389078438295, 1.92, -0.548204164776031, 
        0.231190494667918, 3.1, 1.63, 3.8, 3.06561219546188, 
        1.6, 1.81, 1.60472482321685, 0.737283336431253, 1.8573908435487, 
        1.9, 3.00116804118677, 1.5, 3.00363159871674, 1.83, 1.53, 
        2.16190331966175, 3.31277074978331, 0.173988190942782, 
        1.65, 1.5, 5.13301965105504, 1.71, 2.81682321995075, 
        1.70788734134666, 3.98, -1.06898807614897, 4.04, 2.93374187392046, 
        1.68, 0.193612865225642, 0.645227675876598, -0.737668763749733, 
        1.65, 2.2276858012606, 1.9, 1.67, 3.95733518874672, 1.7, 
        1.97, 1.84779072150971, 0.94, 2.01969646034815, 1.36, 
        1.25, 2.04537573048385, 4.00540537724207, 1.21, 1.01, 
        1.17, 2.70084421350038, 3.81981001365094, 1.94797994058024, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 4.66134903391904, 
        0.99, 3.87, 1.46360422605957, 0.568780520964551, 1.13, 
        5.43375118847561, 0.851490820405824, 0.98, 1.61, 1.67, 
        3.47799250605397, 1.74635844833846, 1.75851016083272, 
        1.51, 2.65380203417016, 1.78409775044908, 1.28683779315479, 
        2.29827361360263, 3.95788817301397, 1.41, -0.682603358630821, 
        -0.914291229003934, 2.08, 0.473204458490934, 3.1632726404724, 
        1.72, 1.54752916490138, 2.55, 1.73, 2.07992548075848, 
        1.29, 1.35278311843888, 3.74, 2.33105686829984, 2.68, 
        0.74, 1.39, -1.67917722582676, 0.0782226645345494, 1.61, 
        3.43, 2.67868735395994, 2.4, 2.05, 4.43, 3.64150038998409, 
        4.31, 2.16, 1.53, 2.13, 2.95172073969935, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 5.03868708457088, 2.92382119834137, 
        5.51, 3.59, 2.76743895434962, 2.81, 2.56, 4.93416403581681, 
        1.58063381238624, 3.88, 3.37073749307095, 5.04, 4.61, 
        3.24, 2.87082724712907, 3.12, 2.67, 1.9, 2.53759361135035, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.16679964312919, 
        4.7833830469587, 3.45, 2.76, 2.92095531369264, 2.99502029734676, 
        3.37, 2.58, 4.6, 1.70994473723608, 2.39, 2.32491666966235, 
        5.65, 3.91, 0.66195040904362, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.47585144697327, 2.76045556223485, 
        2.45, 2.61, 2.06442682682909, 2.23836116330446, 2.3, 
        2.24651689383667, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
        2.48, 2.56, 2.27674244045512, 2.65, 2.53190986339105, 
        2.52, 2.61, 3.22, 2.57721740105256, 2.43515238583159, 
        2.66604523761089, 2.33699293951853, 2.71039147008506, 
        2.36, 2.5239510820732, 2.7, 2.65, 2.70369265989468, 1.69966363437755, 
        2.55, 2.1, 2.51, 2.18059087763529, 2.12, 2.59, 2.29, 
        2.1, 2.44, 2.85023601928306, 2.12, 2.4, 2.27, 2.04, 1.85996297925378, 
        2.79895833870589, 2.31151478731174, 2.18430219828058, 
        2.71104340071825, 2.3, 2.68, 2.5, 1.36, 2.17122532974875, 
        2.29066524569786, 1.92, 2.16, 2.53, 2.56, 1.95017408140766, 
        1.92, 2.1265222286569, 1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 
        2.20739406756646, 1.71, 2.60808910651974, 1.65177428014859, 
        2.4, 2, 2.29587607432459, 2.51, 2.32, 2.58, 2.24, 2.49725762749625, 
        2.62, 2.34316374660085, 2.3, 2.32, 1.75222453513868, 
        2.26, 2.22, 2.28, 2.2, 2.74, 1.98, 2.30469534537924, 
        2.06952312294501, 1.7, 2.062961736323, 2.46, 1.88, 2.31300438973588, 
        2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 
        2.2, 1.99, 2.17531003657848, 1.93187038648557, 2, 2.42, 
        3.23, 2.76661630474822, 2.13, 2.39, 2.17, 2.29, 2.99483094730562, 
        2.12692766521806, 2.38, 2.44612145876064, 2.4, 2.4, 2.36, 
        2.25, 2.87125389627754, 2.54, 2.64, 2.19, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.91570994775365, 2.24899161831897, 
        2.48, 2.28566162719317, 2.36, 2.62, 2.48, 2.63065849479079, 
        2.28, 2.4095155603538, 2.32, 2.38, 1.94093862442368, 
        1.90044437077123, 2.64, 2.38, 2.54, 3.12662304684567, 
        2.35, 2.3, 2.88731644055229, 2.6, 2.09130780517907, 2.69, 
        2.54467777080341, 2.32, 2.28, 2.89990578779342, 2.75849896814259, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 12.9320427502005, 
        23.55076252003, 16.8, 21, 15.2, 18.2329681783189, 17.6, 
        14, 16, 16.9073658885385, 16.8, 16, 11.4, 12.5479778290552, 
        19.367718451798, 20.1693406673578, 20, 17.3482721769145, 
        15.2, 16, 18.6, 22.1086673605792, 17.8, 20, 20.8559867301104, 
        16.1, 17, 22.3527604469682, 19.857749694147, 14.0377432167918, 
        19.1, 21.4920093598159, 22.3068976619543, 19, 20.5, 15.5, 
        20.8074724515805, 15.5, 13.2, 16.2, 18.8, 17.1329983929835, 
        18.8415286035453, 17, 17.9224476259548, 16, 12.9971880960582, 
        18.8, 17.4, 12.4, 16.3924905099526, 17.3897721487995, 
        15.8704546220628, 17.2580465117022, 20.5, 16.3, 16.8, 
        13.9230122947838, 10.6, 18.446676358697, 16.8, 18, 19, 
        19, 18.1, 18.7619397092034, 19.6, 22.7640090031299, 16.8, 
        20.4, 22.685549591525, 24, 30, 21, 20.4461812334482, 
        16, 18, 14.8, 26.3409011080097, 19, 20.3376358906585, 
        24, 22.5, 18, 18, 22.8, 22.3475085754873, 21.6, 20.0873890294279, 
        18.8580796520262, 22, 19.1328991073403, 18, 18, 19, 21.5, 
        16, 13.9171314450971, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 19.8387656234938, 18.2614942193808, 
        24.9739619265753, 20.807846696337, 21.5, 20.8, 18.245850238305, 
        16, 17.0017222044952, 20.2049274730477, 21.1412221557741, 
        22.8086115804141, 21.5, 21, 21, 21.5, 28.5, 18.2332920024108, 
        22, 19.8799905382156, 20, 24, 21.5, 21.4268172794734, 
        19.6121527269854, 20.2856794866951, 25, 19.5, 23.0271929070764, 
        21, 20.0404410578138, 23.5, 20, 15.4518865319442, 21, 
        20, 21.3426700428658, 17.7512834285694, 21.5, 24, 22, 
        25.3500841808603, 18.5, 20, 22, 19.5, 10.4440070636367, 
        25, 22.9542686620217, 19.6151888445906, 18.7066929618697, 
        28.7162165977427, 18.5, 20.629026022559, 22.5, 23, 19.5, 
        24.5, 25, 23.2800372481327, 19.5, 22.4342772499782, 20.5, 
        23, 20, 19.5910064419451, 24.5), mg = c(127, 100, 101, 
        113, 84.8948061388647, 112, 96, 121, 97, 98.1452868647755, 
        111.87216114515, 98.7804554633982, 89, 91, 102, 112, 
        120, 93.593683803305, 117.631552057534, 116, 126, 102, 
        110.34753754865, 95, 110.500175440439, 124, 93, 103.670263537032, 
        107, 96, 101, 106, 104, 122.885179067286, 110, 100.210359876821, 
        110, 95.61130848637, 96.0999445053747, 115.577811225998, 
        117, 111.822664855869, 101, 122.939124779051, 107, 111, 
        102, 101, 114.448836201555, 108, 92, 106.994306608557, 
        111, 115, 111.684151323916, 117.525701922851, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 87.0276326412856, 
        151, 98.5639901970859, 86, 87, 139, 101, 102.51228290688, 
        76.3904073433173, 112, 102.122347035076, 120.07033876876, 
        86, 120.125928725609, 78, 85, 78.9232349700694, 99, 90, 
        88, 84, 70, 81, 100.671264652185, 90.8509722125622, 88, 
        72.567420810149, 115.493725020834, 134, 85, 88, 76.4491642527508, 
        97, 88, 91.0336365528057, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 103.705693452651, 69.7144122556011, 
        87, 96, 103.76865874681, 102, 86, 133.307435043597, 85, 
        82.6933780137409, 92, 88, 80, 122, 121.962849933067, 
        98, 106, 83.0194926909437, 94, 89, 98.8016075122729, 
        88, 101, 81.3828729201055, 89, 97, 80.8873305694322, 
        112, 102, 112.538413865544, 98.4513165441199, 92, 113, 
        123, 108.553715930252, 116, 98, 103, 99.3354563590904, 
        89, 97, 98, 89, 103.565368057021, 89.7128897233311, 106, 
        106, 90, 88, 112.189804486272, 88, 105, 112, 96, 86, 
        91, 95, 102, 102.441894625939, 120, 108.299598518075), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.360503397835, 
            2.5, 2.6, 2.79319533682274, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.39446866318377, 2.85, 2.8, 3.37247418727744, 
            3.3, 2.92169351330075, 3, 2.41, 2.61, 2.48, 2.53, 
            2.8996016327791, 2.67050080060832, 2.4, 2.95, 2.65, 
            3, 3.32808394236116, 2.42, 2.95, 2.79244618124797, 
            2.7, 2.6, 2.45, 3.32719865764962, 3.57161164211344, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 2.96882441786797, 2.7179084708322, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.13286988361998, 
            3.5, 1.89, 3.21021689908692, 2.86431015260982, 2.11, 
            2.53, 2.33983970435793, 1.1, 2.95, 1.88, 2.04728619740822, 
            2.07545080343368, 1.73811873653861, 2.60963626263147, 
            2.46607048552465, 1.9, 2.83, 3.00766568658537, 2.2, 
            1.91489914224931, 1.65, 1.88104374155235, 2.2, 1.63721239247812, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.27046851167617, 
            1.27586739046149, 2.55, 3.52, 3.62588799015237, 2.94896689095724, 
            2.20777519473386, 2.56, 2.5, 2.2, 2.58364227853, 
            1.65, 2.47418508949298, 2.36, 2.74, 3.18, 2.55, 1.75, 
            1.80555868123795, 2.56, 2.46, 1.9856226782648, 2, 
            2.68367545477904, 2, 2.9, 3.18, 2.50774580209799, 
            1.82254572946681, 2.86, 2.98470831151418, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.34795600258994, 0.824852318762334, 
            1.7, 2.26050766872846, 1.62, 1.38, 1.87187001586725, 
            1.62, 1.96377638272224, 1.53440604751561, 1.4, 2.45500418908195, 
            2, 1.38, 1.11158815156285, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.32749684908812, 1.48, 1.61255045747455, 
            1.8, 1.9, 1.63565618741007, 1.97814348029578, 2.3, 
            1.83, 1.41526700908981, 1.39, 1.35, 1.73691226818429, 
            1.7, 1.48, 1.55, 1.65247702875741, 1.25, 1.39, 1.68, 
            1.85162900977941, 1.37932321350927, 1.54111619513342, 
            1.65, 2.05), fla = c(2.85293872165088, 2.76, 3.18603789465077, 
            3.49, 2.52593398736419, 3.39, 2.52, 2.51, 2.98, 3.48751425644986, 
            3.32, 2.43, 2.76, 2.85853106098093, 3.64, 2.91, 3.14, 
            3.4, 3.41211781233055, 3.11081209552293, 3.12623141814225, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.44338139386786, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.01734195553021, 2.5583025491684, 
            2.98, 2.9095312471361, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.10530481206523, 2.92, 
            3.54, 3.34545330008135, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.19479406645247, 3.23, 3.67, 0.57, 1.61227160876077, 
            1.41, 1.79, 2.8304923998338, 1.75, 2.65, 2.43308384105957, 
            2, 1.3, 2.86620604531169, 1.02, 2.86, 1.78805637548257, 
            2.89, 2.14, 1.57, 1.90361515262887, 2.42816859997618, 
            1.85, 2.25306126984518, 2.26, 2.53, 1.58, 1.59, 2.21, 
            1.70936636800792, 1.69, 1.61, 1.69, 2.12491735051832, 
            1.96822551523125, 1.25, 1.46, 2.02980506359345, 3.02316978061461, 
            2.27, 0.99, 1.75344284050154, 3.27506516913572, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.94093506329121, 
            2.92, 2.29514817960015, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.89721415840706, 1.64, 3.40625676357966, 
            2.78300070919645, 2.4334513219359, 1.37297406910705, 
            3.03, 2.65, 3.15, 2.24, 1.87937544753779, 1.75, 1.05565256548054, 
            0.87295661794014, -0.167956227147265, 1.2, 0.58, 
            0.66, 0.47, 0.77520624821626, 0.26108370720046, 0.6, 
            0.5, 0.340107145458102, 0.52, 0.8, 0.990628287318268, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.516179548994336, 0.58, 1.31, 1.1, 0.92, 
            0.319113356189156, 0.6, 0.325209768245391, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.277971415190305, 
            0.7, 0.61, 0.555842945767659, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.160768216192819, 0.3, 0.24, 0.39, 
            0.17753815791444, 0.309954991533465, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.339547434652655, 
            0.32992138576763, 0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 
            0.196050283194741, 0.270028303916576, 0.34, 0.27, 
            0.254801438398814, 0.26, 0.29, 0.257350952654078, 
            0.42, 0.334291925433649, 0.29, 0.26, 0.2962359649087, 
            0.215426255868809, 0.28, 0.2, 0.34, 0.183418782397516, 
            0.206020415485284, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.366700146783932, 
            0.250030903903196, 0.28, 0.29128786179426, 0.53, 
            0.32, 0.424445849915854, 0.45, 0.37, 0.26, 0.27, 
            0.453008630412081, 0.159824845821394, 0.162965153435593, 
            0.29721578343221, 0.336860342590241, 0.21, 0.13, 
            0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.1375229843687, 
            0.4, 0.61, 0.22, 0.3, 0.418313739645013, 0.350783964324691, 
            0.373592333039357, 0.381067110817805, 0.350557415766794, 
            0.5, 0.429872635553531, 0.25, 0.17, 0.32, 0.444692418571512, 
            0.461777967519441, 0.24, 0.494805533398473, 0.227533518768509, 
            0.29, 0.34, 0.37, 0.268539640905124, 0.66, 0.320174474942866, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.471713641135552, 
            0.42, 0.490902836437823, 0.52, 0.3, 0.281676266004054, 
            0.432629089648149, 0.623890429144143, 0.32, 0.129975978097269, 
            0.43, 0.3, 0.339820620589512, 0.39602127569495, 0.39, 
            0.182845961630859, 0.4, 0.500956808826969, 0.227554651265215, 
            0.472103582978231, 0.27, 0.17, 0.6, 0.63, 0.53, 0.63, 
            0.58, 0.53, 0.53, 0.406621824327851, 0.319181625131825, 
            0.491935122952248, 0.464103023646835, 0.43, 0.4, 
            0.47, 0.45, 0.449181435076216, 0.22, 0.324458508815356, 
            0.26, 0.61, 0.53, 0.733305953526781, 0.369998558462117, 
            0.63, 0.53, 0.52, 0.493516477334012, 0.5, 0.6, 0.4, 
            0.41, 0.454247652902036, 0.43, 0.4, 0.39, 0.253649236794886, 
            0.348997624723678, 0.48, 0.442764242468432, 0.330626531090813, 
            0.43, 0.43, 0.53, 0.524406336243651), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 2.33643350559869, 2.96, 1.46, 2.5143566086706, 
            2.34287895299038, 1.86, 1.66, 2.1, 1.03670312963889, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 1.78414382349335, 2.002916228187, 1.54, 1.86, 
            1.36, 1.44, 1.37, 1.42727995458862, 2.25000156090317, 
            1.88342703833134, 2.24470659041295, 1.29816609391181, 
            2.03, 1.05793418599775, 2.19, 2.14, 2.38, 2.08, 1.92282803764232, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.19496708793372, 1.66, 
            2.62320616032345, 0.42, 1.53318136915693, 1.65921122553019, 
            0.73, 1.87, 1.03, 2.15754274237178, 0.974566969184376, 
            1.04, 1.32301740495123, 2.5, 1.46, 1.87, 1.03, 2.20126770132944, 
            1.23248601288687, 1.15, 1.46, 1.45512969135068, 2.76, 
            1.95, 1.43, 1.43839190456709, 1.50338500197269, 1.62, 
            1.25998825014941, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 2.54185068248641, 1.99, 1.35, 3.28, 1.56, 1.77, 
            1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.741701639764464, 
            1.42, 1.79922667210277, 1.63, 2.08, 2.49, 1.24462825451717, 
            1.94709657013682, 1.47574150344242, 1.76483493139539, 
            1.04, 1.96516942362915, 1.11342398220516, 1.61, 0.83, 
            1.87, 1.83, 1.58792544318574, 1.71, 1.21255371125318, 
            2.91, 1.35, 1.77, 0.653568821100054, 1.9, 0.86869350732197, 
            0.94, 1.605798268892, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 1.36602130355788, 0.75, 1.08939147963422, 
            0.55, 1.31157955366662, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 1.63262399020917, 1.15904269097544, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.31055932599309, 
            1.29095933697185, 0.94, 1.57591738376772, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.9567595598722, 
            6.75, 4.33823421456827, 5.05, 8.67674512475783, 7.22, 
            5.75, 6.92287547503583, 5.6, 7.82017695459524, 7.5, 
            5.36609829931983, 7.32026331571625, 4.26671205488359, 
            8.7, 5.1, 5.65, 4.5, 3.12498724511688, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 4.38097145718213, 
            5.11089482237662, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            6.71347048329331, 8.9, 7.2, 8.5809460401823, 5.52095391449389, 
            6.3, 5.85, 6.25, 7.33787956886534, 4.86899815953862, 
            6.8, 9.27542430514794, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 6.19075261455478, 3.38, 
            3.74, 4.01418489325595, 3.21, 3.8, 4.6, 3.2773119577061, 
            3.67564643181317, 2.63606487022298, 2.5, 3.9, 0.6762183680074, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 6.07651996743033, 2.9, 4.82709948175562, 
            4.44335584545582, 3.74080412964491, 2.87371422163617, 
            2.8, 1.55987103663457, 2.94, 2.7, 3.4, 3.3, 2.7, 
            1.21512196898295, 2.9, 3.93251181588417, 2.63320695183813, 
            3.08, 2.9, 1.9, 1.74268749514766, 4.66058013569141, 
            3.4, 1.28, 3.25, 3.57019975180075, 2.28253202410479, 
            2.56902522888832, 2.8, 4.17469410259205, 3.94, -0.198102432508469, 
            2.12, 1.65318107278792, 11.5983259701108, 9.97727017005118, 
            7.04995635251152, 5, 5.45, 7.1, 6.05066599085658, 
            5, 5.7, 4.92, 7.74970252665949, 7.55204021486261, 
            4.35, 4.4, 8.21, 4, 4.9, 7.36560652465257, 8.42, 
            9.4, 5.97567816994627, 10.6525134079311, 7.1, 10.52, 
            7.6, 7.9, 7.59674205624422, 7.5, 13, 8.19455187954533, 
            7.65, 8.8670249994009, 5.58, 7.22025669368885, 10.0526276369942, 
            6.62, 10.68, 10.26, 7.09873350831494, 8.5, 7.91614358315918, 
            8.59115470801311, 9.7, 7.7, 7.3, 10.1282474635107, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.11660227415308, 
            0.86, 1.03927018458823, 1.05, 1.02, 0.986031244038267, 
            1.08, 1.07690107084811, 1.25, 1.07677282407631, 1.15, 
            1.25, 1.2, 1.28, 1.02759449759445, 1.13, 1.23, 0.985533967077316, 
            1.09, 0.969373988592381, 1.11, 1.09, 1.12, 1.13, 
            1.00161518165732, 1.02, 0.97129988181659, 1.04, 1.19, 
            1.09, 1.07294647480367, 1.25, 1.1, 0.995110112228674, 
            1.29481807026868, 0.983266626671598, 1.18, 1.02888459448506, 
            0.95, 0.91, 0.88, 0.82, 1.221314003855, 0.87, 1.03319991420914, 
            1.19284608862816, 1.07, 1.12, 1.12, 0.892946824160915, 
            1.23288916772079, 1.08493121106084, 0.604026268080175, 
            1.27018363150828, 0.94, 1.07, 0.89, 0.790773232598887, 
            1.25, 0.98, 1.23, 1.22, 0.926198941158364, 1.31377453308911, 
            0.952373697215473, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.21148212939025, 1.23, 1.19, 0.96, 1.06, 
            1.19, 1.38, 1.05053109632551, 1.31, 0.84, 1.06385784643426, 
            1.23, 1.21727904849557, 1.2223980857645, 1.06261112524971, 
            1.07, 1.08, 1.05, 1.24515404713035, 1.15, 0.853329331668142, 
            1.27514416948971, 0.792321454910918, 1.15840085258159, 
            1.2509300025636, 1.42, 1.27, 1.05969821096902, 0.8, 
            1.39621487518496, 1.04, 1.30759956839131, 1, 1.04719207200861, 
            0.86, 1.32208687706702, 0.883467203532052, 0.9, 1.12249294288768, 
            1.1, 0.93, 1.71, 0.95, 1.24242930101536, 0.7, 1.23371743965122, 
            0.8, 1.0844318335321, 0.92, 0.73, 1.2541726578479, 
            0.86, 0.69, 0.97, 0.89, 0.79, 0.554622428844216, 
            0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.796487668942346, 
            0.45514378821368, 0.77948441510885, 0.89, 1.172548648918, 
            0.65, 0.6, 0.683696869449595, 0.54, 0.55, 0.668655955396799, 
            0.875879434667218, 0.48, 0.61, 0.56, 0.952367666140196, 
            0.6, 0.57, 0.67, 0.57, 0.992135424116199, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.777443632400019, 0.74, 
            0.67, 0.719858941789172, 0.538793982387095, 0.79705736140242, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 2.76698416141821, 2.93, 3.1587080442072, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 2.6242264346872, 
            3, 2.76727516649812, 2.65, 3.62195758198389, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.29862948402822, 3.2, 
            3.13742168700122, 2.77, 3.11270798817924, 3.69523811296502, 
            2.71, 2.88, 2.87, 3, 2.87, 3.25190071586972, 3.01643077918331, 
            3.79197302859136, 2.69, 3.22769511809936, 3.38, 3, 
            3.25620575216333, 3, 3.35, 3.33, 3.22991766602832, 
            3.33, 2.75, 3.1, 2.90827963598507, 3.42090391183344, 
            3.26, 2.93, 3.2, 3.03, 3.13358909925426, 2.84, 2.87, 
            1.82, 2.83808306464225, 1.59, 2.46, 2.87, 2.81135569580177, 
            2.3, 3.18, 3.48, 2.72544342472647, 3.87808472764442, 
            1.82, 3.44171711980053, 3.11079469064061, 3.5, 3.13, 
            2.14, 2.48, 2.52, 3.15956871596036, 3.37772736900279, 
            3.16570606879364, 3.14, 2.72, 2.01, 2.88063564198226, 
            2.479410076623, 2.82899400798112, 3.21, 2.75, 3.21, 
            3.52148493415873, 2.65, 2.8192343017213, 3.3, 2.71400695708567, 
            2.63, 1.99111864589631, 2.74, 2.48694703346125, 2.83, 
            2.96, 2.77, 3.38, 2.87525230320625, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.35959302441444, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            3.11037151298106, 3.64, 2.54391302291418, 2.84, 3.61867520544673, 
            2.78, 2.35228062449865, 1.13556551154556, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 0.781388175686553, 
            2.02223153001006, 2.33440374615958, 2.05, 2, 1.93760262445044, 
            1.33, 1.32228281011861, 1.62, 1.33, 2.51735147129228, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 1.11532244197601, 1.37721497417381, 1.75, 1.68, 
            1.75, 0.602099818773946, 1.24886747699013, 2.62685038792302, 
            1.94057035419862, 1.44456116813833, 0.768526571682887, 
            1.58787482510309, 2.36070207907815, 2.12026766444374, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 621.344833241296, 
            1065.42068660614, 735, 1450, 1290, 1295, 1045, 1045, 
            1076.33085822757, 1280, 1320, 1430.55471185605, 1547, 
            1310, 1280, 1237.53114806507, 1680, 845, 843.243070466562, 
            770, 1035, 1015, 845, 1135.59180075625, 1195, 1285, 
            915, 1035, 1285, 1038.50700165469, 1188.47359965175, 
            1235, 588.358892808303, 920, 880, 710.804945512412, 
            561.710433908574, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1241.06825575098, 1150, 1265, 
            1190, 1136.83111598024, 1060, 1120, 970, 1270, 1285, 
            573.491028864885, 448.708828035157, 695.136300819503, 
            630, 102.849260326282, 579.905461623002, 470.397966076324, 
            502, 612.184846911055, 291.167821608298, 718, 870, 
            410, 472, 760.277955477338, 886, 391.099454388219, 
            392, 649.853177036128, 375.675475434029, 463, 278, 
            714, 899.39856152276, 515, 562.048532143852, 520.735642444973, 
            715.469557777773, 562, 680, 625, 480, 450, 495, 795.68260662236, 
            345, 611.213672097538, 625, 428, 660, 406, 313.722540157741, 
            562, 438, 415, 672, 541.407103976691, 510, 438.764782541326, 
            312, 680, 562, 325, 1015.1284352766, 434, 385, 373.131813189038, 
            236.50778549569, 757.61049143874, 372, 564, 370.285084854791, 
            922.691972523592, 365, 380, 161.765481044306, 378, 
            312.903585103986, 466, 628.834370116409, 580, 885.453766822087, 
            912.421659486574, 560, 600, 477.825690563739, 695, 
            720, 515, 580, 590, 600, 390.442082800907, 520, 550, 
            879.408901866776, 830, 415, 446.623033981827, 546.9188849254, 
            550, 500, 480, 425, 675, 640, 504.197214889117, 480, 
            880, 660, 601.247583340778, 597.566691483834, 680, 
            570, 797.229077613041, 615, 933.872789501117, 695, 
            685, 908.428286862413, 452.717890029836, 510, 470, 
            816.999665600495, 740, 558.563629264899, 835, 840, 
            614.026997633087)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 15.2073587031096, 
        13.24, 13.6914591974963, 14.39, 14.06, 14.2791273480633, 
        13.86, 13.195503411003, 13.7472324883162, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.0013904562978, 13.64, 13.7301022775948, 
        12.93, 13.71, 12.85, 13.5, 13.6636409539765, 13.39, 13.3, 
        13.87, 15.1078959701591, 13.73, 13.58, 13.68, 13.76, 
        13.1067064689922, 11.822101280998, 12.903863062936, 13.05, 
        11.6602466045153, 14.22, 13.56, 13.41, 13.88, 13.24, 
        13.05, 14.21, 14.0890392950009, 13.9, 13.9195982391062, 
        13.7779509800509, 13.05, 14.1123072987306, 13.82, 13.1793509908915, 
        13.74, 15.3006962810999, 14.22, 13.29, 13.4051102458779, 
        12.37, 12.33, 12.64, 11.7795347726973, 12.37, 12.17, 
        12.37, 13.0475762678639, 12.6736445719704, 13.34, 12.21, 
        12.3621569971178, 13.86, 11.9283252240973, 12.99, 11.96, 
        11.66, 12.1536704003835, 11.84, 12.4412674535788, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.2567052781971, 12.16, 
        13.0150979142441, 12.7018061028318, 12.08, 12.08, 12, 
        12.69, 12.6570267839968, 11.62, 12.1561211563191, 11.81, 
        12.29, 13.5196487386721, 12.29, 12.08, 12.6, 12.6572803329657, 
        12.5841989485362, 12.51, 12.0266369400201, 11.7766072517641, 
        12.72, 12.22, 11.61, 11.0866903491574, 11.707189728624, 
        11.76, 12.6703343660579, 12.08, 11.03, 11.82, 11.2505638530002, 
        12.136181583277, 11.8062110228662, 11.45, 14.9679893979159, 
        12.42, 13.05, 11.87, 12.07, 12.4185613586629, 11.79, 
        12.5724613241278, 12.04, 12.86, 12.88, 12.4777236562065, 
        12.7, 12.51, 12.6634000712551, 12.25, 12.53, 12.8545546178999, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 13.4538976863236, 
        11.5839953683472, 12.87, 13.32, 11.5835611582831, 11.942561223469, 
        12.79, 11.8365767428733, 13.23, 12.3281236214949, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.0004457517043, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.298187036938, 14.1191368162613, 
        13.58, 13.4, 13.3308412616443, 12.77, 14.16, 12.9078063749841, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.33204833297019, 
        -0.11373506053932, 1.95, 0.642245918199782, 0.386866715602751, 
        1.87, 2.15, 0.938890595052958, 1.35, 2.16, 1.48, 0.0175598524479872, 
        1.73, 1.87, 2.02110236848508, 1.92, 1.48590780458562, 
        -0.0430215581250257, 3.1, 1.63, 3.8, 1.81464499850619, 
        1.6, 1.81, 2.08711489300111, -0.0197877584134591, 2.1690220327597, 
        1.9, 2.96097403536784, 1.5, 0.226504021464334, 1.83, 
        1.53, 3.3886306155137, -1.0719765429873, 2.11175388360416, 
        1.65, 1.5, 2.97343986316579, 1.71, 3.11438918242021, 
        1.71618690920283, 3.98, -1.13700033570805, 4.04, 1.30094198888394, 
        1.68, 3.9988574939618, -0.0854302630984343, -0.894438827786507, 
        1.65, 2.33458377336133, 1.9, 1.67, 1.15024779503608, 
        1.7, 1.97, 0.337990359401509, 0.94, 1.3177515016019, 
        1.36, 1.25, 1.74778847017103, 3.55396653065568, 1.21, 
        1.01, 1.17, 3.9054239686641, 3.0598890292322, 1.8837752743644, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 2.18968217894646, 
        0.99, 3.87, 1.55522926133734, 1.92005504125891, 1.13, 
        5.61251653441199, 1.82934902133542, 0.98, 1.61, 1.67, 
        2.28984285615794, 1.41360655953836, 3.10887309655828, 
        1.51, 4.7380428585264, 2.25416620807183, 0.178843120547409, 
        4.17376737487805, 3.32080138490073, 1.41, 2.2287193931887, 
        2.0407371016817, 2.08, 2.39673662101251, 2.37410313026993, 
        1.72, 0.0108408162806772, 2.55, 1.73, 3.88982150813857, 
        1.29, 1.91959159742842, 3.74, 2.27786917660089, 2.68, 
        0.74, 1.39, -0.0100862404914697, 3.70921489244266, 1.61, 
        3.43, 0.948789230471261, 2.4, 2.05, 4.43, 3.61651317572575, 
        4.31, 2.16, 1.53, 2.13, 2.08956193460527, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 3.87496433253868, 2.96674780877123, 
        5.51, 3.59, 4.3647661751681, 2.81, 2.56, 4.82600543332538, 
        3.87281914068346, 3.88, 5.51303673227839, 5.04, 4.61, 
        3.24, 2.54811800318382, 3.12, 2.67, 1.9, 3.64198055038854, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 5.53099000390849, 
        3.77123732820978, 3.45, 2.76, 4.85602495760318, 2.12697371105974, 
        3.37, 2.58, 4.6, 3.4065890044071, 2.39, 4.64297320705795, 
        5.65, 3.91, 4.25555946210169, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.54460832880879, 2.37682277195226, 
        2.45, 2.61, 2.14605062069651, 2.0770133273809, 2.3, 2.35549069248543, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.33013801384274, 
        2.65, 1.96340521570456, 2.52, 2.61, 3.22, 2.57637000213641, 
        2.74738721929027, 2.27016230420731, 2.44387061128535, 
        2.57192966036058, 2.36, 2.36661904976368, 2.7, 2.65, 
        2.46081210467207, 1.75673594326396, 2.55, 2.1, 2.51, 
        2.27791749073071, 2.12, 2.59, 2.29, 2.1, 2.44, 2.83123339300833, 
        2.12, 2.4, 2.27, 2.04, 2.41056277038957, 2.7018408150082, 
        2.00223211808212, 2.49208638831399, 2.54695773698986, 
        2.3, 2.68, 2.5, 1.36, 2.26337556907631, 2.02963757325186, 
        1.92, 2.16, 2.53, 2.56, 1.96250902871728, 1.92, 2.08328322529138, 
        1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.97123676101403, 1.71, 
        2.36885581465234, 2.22104244714945, 2.4, 2, 2.01004903112214, 
        2.51, 2.32, 2.58, 2.24, 2.31678502717642, 2.62, 2.40919821424513, 
        2.3, 2.32, 1.72990977224824, 2.26, 2.22, 2.28, 2.2, 2.74, 
        1.98, 1.90309895565102, 1.79948995454798, 1.7, 1.79596170481273, 
        2.46, 1.88, 2.16769063225142, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.18594918090523, 
        2.39584254917236, 2, 2.42, 3.23, 2.55716119069351, 2.13, 
        2.39, 2.17, 2.29, 2.39601080976947, 2.61215470022989, 
        2.38, 2.03344567400303, 2.4, 2.4, 2.36, 2.25, 2.50130478026161, 
        2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 3.07701419343643, 
        2.43905498041156, 2.48, 2.27531371335778, 2.36, 2.62, 
        2.48, 2.21945579302487, 2.28, 2.276179942554, 2.32, 2.38, 
        2.02772152864781, 2.03776020081285, 2.64, 2.38, 2.54, 
        2.89429632037865, 2.35, 2.3, 2.67241367521056, 2.6, 2.62299672230441, 
        2.69, 2.476215143111, 2.32, 2.28, 2.46251741857631, 2.53956416964357, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 14.3149776308251, 
        16.4185533394256, 16.8, 21, 15.2, 16.3238259644481, 17.6, 
        14, 16, 16.3955563856239, 16.8, 16, 11.4, 13.5615460815552, 
        21.952220246936, 17.3960361958678, 20, 8.82454288481284, 
        15.2, 16, 18.6, 11.2959818820155, 17.8, 20, 20.5106511736495, 
        16.1, 17, 20.5228545852478, 20.4382663075963, 17.1034990847316, 
        19.1, 23.0436759590243, 17.4809258839869, 19, 20.5, 15.5, 
        19.174757942425, 15.5, 13.2, 16.2, 18.8, 13.2672750551463, 
        17.1062439715449, 17, 13.317013169144, 16, 13.5189421734251, 
        18.8, 17.4, 12.4, 17.0947082870455, 14.7014268652258, 
        20.2085140813824, 18.3160969290218, 20.5, 16.3, 16.8, 
        17.9079072048737, 10.6, 18.907838934586, 16.8, 18, 19, 
        19, 18.1, 17.2216564432077, 19.6, 18.2629026784499, 16.8, 
        20.4, 22.7496018889897, 24, 30, 21, 19.9091622907858, 
        16, 18, 14.8, 17.7508604777907, 19, 18.6193186828503, 
        24, 22.5, 18, 18, 22.8, 24.3494304834316, 21.6, 17.6915913719601, 
        21.8509379277126, 22, 23.0561485424636, 18, 18, 19, 21.5, 
        16, 11.2024258963156, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 22.8783980792152, 25.4174164371701, 
        22.2027034571438, 17.2696107128188, 21.5, 20.8, 18.7828040208968, 
        16, 19.7800523197441, 16.6206255609168, 21.6886549959373, 
        23.933010838795, 21.5, 21, 21, 21.5, 28.5, 22.7691329714077, 
        22, 23.7813781784701, 20, 24, 21.5, 18.825435280098, 
        19.4410905126745, 20.9522879905454, 25, 19.5, 21.0661943244943, 
        21, 24.5660648808689, 23.5, 20, 19.0799408203454, 21, 
        20, 25.0153938687677, 20.4336693986301, 21.5, 24, 22, 
        19.9807285804454, 18.5, 20, 22, 19.5, 8.35510938660669, 
        25, 21.6732065040803, 18.3128511992805, 22.270012339206, 
        24.1535055780594, 18.5, 25.7602434488271, 22.5, 23, 19.5, 
        24.5, 25, 22.2932603573399, 19.5, 17.1075685350252, 20.5, 
        23, 20, 20.0681943939843, 24.5), mg = c(127, 100, 101, 
        113, 103.525499439417, 112, 96, 121, 97, 114.499149757131, 
        127.353453934216, 109.672447542417, 89, 91, 102, 112, 
        120, 116.531296956348, 89.5185639008087, 116, 126, 102, 
        94.3470894171661, 95, 124.031198932077, 124, 93, 107.984059608095, 
        107, 96, 101, 106, 104, 104.991093329751, 110, 98.2624363861615, 
        110, 64.4003466376068, 75.9326928324461, 104.704358367551, 
        117, 107.858250155292, 101, 124.405934393174, 107, 111, 
        102, 101, 119.179385192131, 108, 92, 105.142448550992, 
        111, 115, 123.803680426611, 87.4094489043474, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 102.594088429403, 
        151, 88.8969626838863, 86, 87, 139, 101, 95.7626505603808, 
        88.7372388112822, 112, 90.6822575796483, 101.180621256743, 
        86, 119.190614281796, 78, 85, 108.906191840755, 99, 90, 
        88, 84, 70, 81, 99.1974950959391, 88.5731051186122, 88, 
        84.8772298443393, 95.2783054854652, 134, 85, 88, 91.9698135688904, 
        97, 88, 98.5791451498075, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 93.5772080543183, 86.0898146828655, 
        87, 96, 94.516999402912, 102, 86, 152.406072965128, 85, 
        114.372895094431, 92, 88, 80, 122, 89.6967842209984, 
        98, 106, 74.3702651880391, 94, 89, 110.995372011143, 
        88, 101, 89.8044576321386, 89, 97, 106.046134051985, 
        112, 102, 106.995307347055, 104.448397336312, 92, 113, 
        123, 94.3453092593052, 116, 98, 103, 112.105723626091, 
        89, 97, 98, 89, 99.2269643610043, 114.935102371271, 106, 
        106, 90, 88, 117.854023577931, 88, 105, 112, 96, 86, 
        91, 95, 102, 114.691476876093, 120, 107.405998094464), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.02627969580108, 
            2.5, 2.6, 3.40101131011994, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.38093164408566, 2.85, 2.8, 2.93247105573062, 
            3.3, 2.78168639776696, 3, 2.41, 2.61, 2.48, 2.53, 
            2.34437094280473, 2.79098645025104, 2.4, 2.95, 2.65, 
            3, 3.30191139384631, 2.42, 2.95, 1.47668113737102, 
            2.7, 2.6, 2.45, 2.76212826322234, 3.14981753585145, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 2.38712427696302, 2.87563554956074, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.49366556761806, 
            3.5, 1.89, 1.77030362041894, 2.24149802521893, 2.11, 
            2.53, 1.96496671143393, 1.1, 2.95, 1.88, 1.96550326120439, 
            2.13012560989984, 1.74764508090478, 2.43622328727359, 
            1.77005335630555, 1.9, 2.83, 2.69147380371827, 2.2, 
            2.63873611581073, 1.65, 2.33499597850052, 2.2, 1.88449105039681, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 1.66282811026055, 
            1.22269318625198, 2.55, 3.52, 3.08882993252464, 1.69536743038122, 
            1.8510753806104, 2.56, 2.5, 2.2, 1.86442268425295, 
            1.65, 1.9377817350828, 2.36, 2.74, 3.18, 2.55, 1.75, 
            2.30147861390332, 2.56, 2.46, 1.57494730843651, 2, 
            2.37965882088547, 2, 2.9, 3.18, 2.14843194401184, 
            2.28815923919226, 2.86, 2.73797070254744, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.51408789518894, 1.69008038181592, 
            1.7, 2.06881287550351, 1.62, 1.38, 1.17096672036316, 
            1.62, 1.79855669926896, 1.57035423841645, 1.4, 2.58637785803816, 
            2, 1.38, 2.04317837070817, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.55848024052437, 1.48, 1.62596479480357, 
            1.8, 1.9, 1.49847759287034, 2.30960364528707, 2.3, 
            1.83, 1.63257862808887, 1.39, 1.35, 2.04899058651439, 
            1.7, 1.48, 1.55, 1.77949751935699, 1.25, 1.39, 1.68, 
            1.21223228775566, 1.62134828924141, 1.62709525638304, 
            1.65, 2.05), fla = c(2.85642104125455, 2.76, 3.74661626000094, 
            3.49, 2.40753546344071, 3.39, 2.52, 2.51, 2.98, 3.14781519834624, 
            3.32, 2.43, 2.76, 2.80688446865223, 3.64, 2.91, 3.14, 
            3.4, 3.55308235025593, 3.06160231788463, 3.14714919720564, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.60270335664361, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.0882085551567, 2.38821429136698, 
            2.98, 2.57142950137817, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.18274900797239, 2.92, 
            3.54, 3.17859734839161, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.67186109910766, 3.23, 3.67, 0.57, 2.03137030598526, 
            1.41, 1.79, 3.06920929042229, 1.75, 2.65, 2.01707907852025, 
            2, 1.3, 2.63283374310609, 1.02, 2.86, 1.73796771299017, 
            2.89, 2.14, 1.57, 1.82632333389415, 1.50310994462317, 
            1.85, 1.89397080137254, 2.26, 2.53, 1.58, 1.59, 2.21, 
            2.33340745026924, 1.69, 1.61, 1.69, 1.98783133340001, 
            1.91453923299783, 1.25, 1.46, 2.02252537966727, 2.72422031893174, 
            2.27, 0.99, 1.9913323989646, 3.74754956283627, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.94818335366376, 
            2.92, 2.8177159332485, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.86507755893917, 1.64, 3.29316294856099, 
            3.53790670450585, 2.04531016799254, 1.86824085068164, 
            3.03, 2.65, 3.15, 2.24, 2.03016903460078, 1.75, 0.171829757361212, 
            0.471650982361201, 0.857416167484689, 1.2, 0.58, 
            0.66, 0.47, 0.22544706606627, 0.150148152101651, 
            0.6, 0.5, 0.329350827728496, 0.52, 0.8, 0.984374029034161, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.717974977289747, 0.58, 1.31, 1.1, 0.92, 
            0.592553083585001, 0.6, 0.139713726490477, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 1.02391479766843, 
            0.7, 0.61, 0.553963045660433, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.309499961267945, 0.3, 0.24, 0.39, 
            0.123419200435916, 0.412139059807223, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.219406204078106, 
            0.186441886438915, 0.32, 0.17, 0.24, 0.25, 0.27, 
            0.26, 0.281032462254795, 0.390098084122328, 0.34, 
            0.27, 0.363926870784929, 0.26, 0.29, 0.257543136338757, 
            0.42, 0.455366064382124, 0.29, 0.26, 0.292497513507248, 
            0.332002367144822, 0.28, 0.2, 0.34, 0.311172605989023, 
            0.226568038167601, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.302321874772112, 
            0.38503647297273, 0.28, 0.285899138300747, 0.53, 
            0.32, 0.24267441676515, 0.45, 0.37, 0.26, 0.27, 0.460603579602744, 
            0.314334519573422, 0.462406827886717, 0.416688199587558, 
            0.307808643593803, 0.21, 0.13, 0.34, 0.24, 0.43, 
            0.35, 0.43, 0.3, 0.184879929985442, 0.4, 0.61, 0.22, 
            0.3, 0.384390389865063, 0.34371039367987, 0.255455576479932, 
            0.421049742708075, 0.291718585082143, 0.5, 0.315606006009229, 
            0.25, 0.17, 0.32, 0.586896909879645, 0.275538706771195, 
            0.24, 0.248732840734252, 0.294765966438292, 0.29, 
            0.34, 0.37, 0.225665726396115, 0.66, 0.320959930790853, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.262271856489184, 
            0.42, 0.362875311404664, 0.52, 0.3, 0.399329646275432, 
            0.418006420227499, 0.368703988037486, 0.32, 0.359230901027883, 
            0.43, 0.3, 0.238274945296138, 0.303611818751778, 
            0.39, 0.414005616095685, 0.4, 0.519897000535557, 
            0.672768205923446, 0.479665680094919, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.532398721767607, 
            0.561787602681741, 0.387316779173563, 0.464358735704621, 
            0.43, 0.4, 0.47, 0.45, 0.532247007385088, 0.22, 0.408443701054675, 
            0.26, 0.61, 0.53, 0.419789478465565, 0.602046439638828, 
            0.63, 0.53, 0.52, 0.439903464292715, 0.5, 0.6, 0.4, 
            0.41, 0.627015714616421, 0.43, 0.4, 0.39, 0.473746168545588, 
            0.539299820760077, 0.48, 0.401569933350509, 0.41551951684995, 
            0.43, 0.43, 0.53, 0.54188075680646), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 2.5599689947746, 2.96, 1.46, 2.08018035070065, 
            2.4018481669309, 1.86, 1.66, 2.1, 1.10249144089429, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 1.78209622560131, 2.2018401021241, 1.54, 1.86, 
            1.36, 1.44, 1.37, 1.90743644150887, 2.07519477896852, 
            1.95614287758586, 2.06105352353365, 1.45444594199092, 
            2.03, 1.39081982066474, 2.19, 2.14, 2.38, 2.08, 1.49391225609775, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.78035664431957, 1.66, 
            1.95016620771629, 0.42, 2.36445711619074, 1.8928981332598, 
            0.73, 1.87, 1.03, 1.1396707675501, 1.79504519925657, 
            1.04, 1.73654842735369, 2.5, 1.46, 1.87, 1.03, 2.11228087408462, 
            1.07523806441527, 1.15, 1.46, 1.30482265310815, 2.76, 
            1.95, 1.43, 1.86331233286219, 0.90837631404447, 1.62, 
            1.00147602295044, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 0.849481208429391, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.27644819626908, 
            1.42, 1.06682557052199, 1.63, 2.08, 2.49, 1.09724629638786, 
            1.20462361522256, 1.6789855929588, 1.86610522820413, 
            1.04, 1.30528850347978, 1.19108408611166, 1.61, 0.83, 
            1.87, 1.83, 2.75914748710283, 1.71, 1.54711286651704, 
            2.91, 1.35, 1.77, 1.50400190919635, 1.9, 1.21745524265459, 
            0.94, 1.05354592738425, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 1.02537994832312, 0.75, 0.954578775333945, 
            0.55, 0.674516487798673, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 0.946592534140921, 1.17848906969268, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.67257670294494, 
            1.61494046572447, 0.94, 1.28404731182595, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 2.1691915687998, 
            6.75, 4.79556358436122, 5.05, 6.98320106029232, 7.22, 
            5.75, 5.97638154007391, 5.6, 5.81963328664272, 7.5, 
            3.80151647089998, 7.70298818026148, 5.38444422974127, 
            8.7, 5.1, 5.65, 4.5, 5.47067720785739, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.57089738026798, 
            5.79283943154795, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            5.6229526227595, 8.9, 7.2, 6.68923012022917, 5.60558330762333, 
            6.3, 5.85, 6.25, 7.33380181139836, 4.68393402434574, 
            6.8, 9.61328017676873, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 8.6743883521418, 3.38, 
            3.74, 4.21489578079988, 3.21, 3.8, 4.6, 2.91688355332159, 
            3.34520203235545, 1.0582918107814, 2.5, 3.9, 0.0552223840341748, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 6.75070298002944, 2.9, 6.15166479296958, 
            5.33230921547134, 6.78254813571625, 3.64590548216176, 
            2.8, 2.87757981146904, 2.94, 2.7, 3.4, 3.3, 2.7, 
            0.951810672839692, 2.9, 1.95576854829168, 1.32469802427606, 
            3.08, 2.9, 1.9, 1.24517854892402, 1.49597724623425, 
            3.4, 1.28, 3.25, 5.25287992451748, 1.86560045014112, 
            4.57765710304928, 2.8, 3.64110212321093, 3.94, 2.54472624692268, 
            2.12, 1.09581068226986, 10.0832938079634, 7.16541616037602, 
            7.46358176704497, 5, 5.45, 7.1, 5.99853730358431, 
            5, 5.7, 4.92, 6.57210637964909, 7.83481411430543, 
            4.35, 4.4, 8.21, 4, 4.9, 7.32438424015595, 8.42, 
            9.4, 2.12777365371294, 10.1653651946871, 7.1, 10.52, 
            7.6, 7.9, 9.11092701649439, 7.5, 13, 7.09861538173255, 
            7.65, 7.14302973403674, 5.58, 6.65083265822746, 9.88886010022363, 
            6.62, 10.68, 10.26, 7.85484461662281, 8.5, 8.80351768384575, 
            4.52394356128607, 9.7, 7.7, 7.3, 8.95892464121712, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.16760422037495, 
            0.86, 0.981694136569562, 1.05, 1.02, 1.04879326493107, 
            1.08, 1.23050644114738, 1.25, 1.0553199941163, 1.15, 
            1.25, 1.2, 1.28, 0.827683494252365, 1.13, 1.23, 0.883860603895219, 
            1.09, 1.06625678344852, 1.11, 1.09, 1.12, 1.13, 0.818870309194124, 
            1.02, 1.41485889291673, 1.04, 1.19, 1.09, 1.18236065795561, 
            1.25, 1.1, 0.822294783934458, 1.3226169005003, 1.00608175537127, 
            1.18, 0.59262899040527, 0.95, 0.91, 0.88, 0.82, 1.26198251122192, 
            0.87, 1.1653255478576, 1.00064750056628, 1.07, 1.12, 
            1.12, 0.983930259539645, 0.892999561502215, 0.895153670064874, 
            1.10850049930029, 0.90415274306061, 0.94, 1.07, 0.89, 
            0.905727443968967, 1.25, 0.98, 1.23, 1.22, 0.92386136414076, 
            0.812323775375128, 1.17082733614248, 1.12, 1.02, 
            1.28, 0.906, 1.36, 0.98, 1.31, 1.10349832520763, 
            1.23, 1.19, 0.96, 1.06, 1.19, 1.38, 1.23345457831552, 
            1.31, 0.84, 0.865720095690351, 1.23, 1.01761840708867, 
            0.99071877857835, 1.01456522499041, 1.07, 1.08, 1.05, 
            0.755677135089277, 1.15, 1.04520796084607, 1.1501123186205, 
            0.728666680232218, 0.812627089937824, 1.06443965863281, 
            1.42, 1.27, 1.10714959710074, 0.8, 1.19288694546861, 
            1.04, 1.21320328808827, 1, 0.770088385705683, 0.86, 
            1.29647725455396, 1.24554546411894, 0.9, 1.10206792771224, 
            1.1, 0.93, 1.71, 0.95, 1.02241796921907, 0.7, 1.3127480027204, 
            0.8, 1.00975720855696, 0.92, 0.73, 1.08746984400535, 
            0.86, 0.69, 0.97, 0.89, 0.79, 0.991994592560856, 
            0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.507194673950792, 
            0.713132014386773, 0.758712589711518, 0.89, 0.518234557960747, 
            0.65, 0.6, 0.753252144650128, 0.54, 0.55, 0.715616832043682, 
            0.589586849541134, 0.48, 0.61, 0.56, 0.705801178288908, 
            0.6, 0.57, 0.67, 0.57, 0.78325143822566, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.295366621660746, 0.74, 
            0.67, 0.666493091313226, 1.05843958792168, 0.411094333884392, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 3.6116254563495, 2.93, 3.22685399545308, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.39397249132542, 
            3, 3.55663910826601, 2.65, 3.54244547536953, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.77070237734231, 3.2, 
            2.99666062785294, 2.77, 4.27861809222989, 3.5165748804129, 
            2.71, 2.88, 2.87, 3, 2.87, 2.66891109993861, 3.40595972027372, 
            3.00387536664589, 2.69, 3.28286165363042, 3.38, 3, 
            3.69692946963644, 3, 3.35, 3.33, 3.22069878307598, 
            3.33, 2.75, 3.1, 2.91698372856216, 3.02290079268516, 
            3.26, 2.93, 3.2, 3.03, 3.80854411272631, 2.84, 2.87, 
            1.82, 2.93003096739853, 1.59, 2.46, 2.87, 2.91381608717092, 
            2.3, 3.18, 3.48, 2.68266364497842, 4.28043348161938, 
            1.82, 3.63296770828141, 1.9506754208634, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.55463637280894, 3.38740303766007, 
            3.60020279325957, 3.14, 2.72, 2.01, 2.84860675914615, 
            3.08444956607996, 2.60696308751802, 3.21, 2.75, 3.21, 
            2.78318193270777, 2.65, 2.91195624299889, 3.3, 2.78479525728868, 
            2.63, 2.01417707243926, 2.74, 3.84653045016618, 2.83, 
            2.96, 2.77, 3.38, 3.08245897746658, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.19086620543443, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            2.27346200333857, 3.64, 3.04907224651669, 2.84, 2.57255213877224, 
            2.78, 2.40959384192116, 1.3572451874441, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.78303355519482, 
            1.91135838810507, 2.54588987132339, 2.05, 2, 1.86551460875284, 
            1.33, 1.30195793982893, 1.62, 1.33, 2.67131724583378, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 2.11130935977119, 1.66519813266024, 1.75, 1.68, 
            1.75, 1.24800807773669, 1.44430390349236, 2.32612565304199, 
            2.23871255767544, 2.14590959346557, 2.54928797982802, 
            1.65536518979062, 1.02891727563506, 1.72855316979343, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 1333.21921907573, 
            1175.78286372291, 735, 1450, 1290, 1295, 1045, 1045, 
            1008.71868131099, 1280, 1320, 1395.59098204339, 1547, 
            1310, 1280, 1139.44776923702, 1680, 845, 1185.14792992327, 
            770, 1035, 1015, 845, 1362.53878328872, 1195, 1285, 
            915, 1035, 1285, 1201.76952506572, 889.092693700894, 
            1235, 1138.16959295737, 920, 880, 1037.4523954573, 
            604.099772531517, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1305.35692557884, 1150, 1265, 
            1190, 773.653816742827, 1060, 1120, 970, 1270, 1285, 
            427.401336779361, 882.706280445637, 888.325352907362, 
            630, 753.459548216657, 532.084415779332, 1095.04856134872, 
            502, 647.701430710928, 619.853771203199, 718, 870, 
            410, 472, 949.422645404278, 886, 647.797406872829, 
            392, 621.715779057631, 847.145386681113, 463, 278, 
            714, 606.006595261391, 515, 765.048629397679, 666.118386423283, 
            613.328984599551, 562, 680, 625, 480, 450, 495, 688.400014275981, 
            345, 761.966897955847, 625, 428, 660, 406, 660.019534206306, 
            562, 438, 415, 672, 521.6722380135, 510, 594.49502135153, 
            312, 680, 562, 325, 633.452233068477, 434, 385, 724.070710811621, 
            -3.67598235899254, 672.257334486343, 372, 564, 325.060109883807, 
            1030.33058534341, 365, 380, 725.413510076143, 378, 
            564.968135206634, 466, 783.576623903962, 580, 583.598018593598, 
            714.586926750387, 560, 600, 786.628373172811, 695, 
            720, 515, 580, 590, 600, 326.926742364348, 520, 550, 
            480.394204875037, 830, 415, 438.220464098523, 260.686386380283, 
            550, 500, 480, 425, 675, 640, 217.810331064345, 480, 
            880, 660, 668.679019393256, 376.301441629128, 680, 
            570, 541.206232422284, 615, 313.205858227823, 695, 
            685, 731.678556712692, 481.279787096286, 510, 470, 
            675.82987492427, 740, 650.8202137466, 835, 840, 665.223411090299
            )), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.5326218321355, 
        13.24, 13.9378428009591, 14.39, 14.06, 13.7618591559937, 
        13.86, 13.5132255752285, 13.7896492979089, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.8076345759451, 13.64, 13.0008392522683, 
        12.93, 13.71, 12.85, 13.5, 13.9576378796168, 13.39, 13.3, 
        13.87, 12.943594370815, 13.73, 13.58, 13.68, 13.76, 12.6920087719575, 
        14.2920986063263, 13.3769703163572, 13.05, 12.7755648768358, 
        14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 14.21, 14.3223653389171, 
        13.9, 13.4597675621388, 13.9825535474947, 13.05, 14.4090138158745, 
        13.82, 13.7596175189377, 13.74, 14.5226530733415, 14.22, 
        13.29, 13.9194046907303, 12.37, 12.33, 12.64, 11.275515613755, 
        12.37, 12.17, 12.37, 12.9927365824799, 12.6605124084405, 
        13.34, 12.21, 13.1993027181384, 13.86, 11.8742311286812, 
        12.99, 11.96, 11.66, 12.0085035202642, 11.84, 12.6014053730905, 
        12.7, 12, 12.72, 12.08, 13.05, 11.84, 12.1105055187267, 
        12.16, 12.7039003295417, 12.792439270465, 12.08, 12.08, 
        12, 12.69, 12.0477438950387, 11.62, 11.9405022957423, 
        11.81, 12.29, 12.976590755257, 12.29, 12.08, 12.6, 13.1849841273864, 
        12.928098492298, 12.51, 12.4885808042825, 11.8421561534963, 
        12.72, 12.22, 11.61, 12.0403038809447, 12.7058513625161, 
        11.76, 12.1936851525621, 12.08, 11.03, 11.82, 11.3676447848324, 
        13.0457476560927, 12.0002820324625, 11.45, 14.9214647244162, 
        12.42, 13.05, 11.87, 12.07, 11.1553646366053, 11.79, 
        12.3999588386094, 12.04, 12.86, 12.88, 12.9706196659506, 
        12.7, 12.51, 12.8975806146608, 12.25, 12.53, 12.4287849224524, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.3063952122519, 
        11.8382608516853, 12.87, 13.32, 12.4128650503548, 11.4066201492488, 
        12.79, 12.0219174960481, 13.23, 12.965796217667, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.1005887473566, 13.69, 
        12.85, 12.96, 13.78, 13.73, 14.1559018509367, 13.6603518368221, 
        13.58, 13.4, 12.6078581341884, 12.77, 14.16, 13.4302505215991, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.42520833111963, 
        1.49015158418122, 1.95, -0.244775360016412, 1.87528085503532, 
        1.87, 2.15, 0.102073886599873, 1.35, 2.16, 1.48, -0.0495156996039952, 
        1.73, 1.87, 1.03863994368827, 1.92, 0.290186626188524, 
        0.650380273810456, 3.1, 1.63, 3.8, 4.13525018074017, 
        1.6, 1.81, 3.51345633674464, 0.920386803609414, 3.23916163699479, 
        1.9, 0.155403606473223, 1.5, 1.23756173333703, 1.83, 
        1.53, 1.16146401740678, 2.10694496541168, 1.66456114275753, 
        1.65, 1.5, 3.87624007843935, 1.71, 1.22521362912191, 
        3.18192240208806, 3.98, -0.811667426956893, 4.04, 3.00625453661821, 
        1.68, 4.24304704358984, 1.71572085013536, 0.64483811639471, 
        1.65, 2.51114448753463, 1.9, 1.67, -0.184805280918666, 
        1.7, 1.97, 1.67588293995729, 0.94, 1.46345942976057, 
        1.36, 1.25, 1.8759018418024, 3.61298952512113, 1.21, 
        1.01, 1.17, 2.36784266402056, 1.86898813596282, 0.0252276992193399, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 2.24829708919369, 
        0.99, 3.87, 0.162734418888402, 2.71367040449059, 1.13, 
        5.57818316700998, 0.643720786387457, 0.98, 1.61, 1.67, 
        2.74264894390388, 0.941058627791417, 0.772819812338736, 
        1.51, 3.14578804924988, 1.48908114090601, 2.69567667348295, 
        1.91477128618633, 3.73271841614554, 1.41, 1.08470862216554, 
        0.385243754757618, 2.08, 2.85366045928832, 5.36237616987224, 
        1.72, 1.84681153766648, 2.55, 1.73, 2.33495540910401, 
        1.29, 1.50468994937228, 3.74, 3.61686440745983, 2.68, 
        0.74, 1.39, -1.99565284166488, 2.44501892163477, 1.61, 
        3.43, 2.31166753169524, 2.4, 2.05, 4.43, 3.56700682491751, 
        4.31, 2.16, 1.53, 2.13, 2.03657643811233, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 3.76592051626105, 2.95686024032367, 
        5.51, 3.59, 4.60224204270844, 2.81, 2.56, 2.78918019532017, 
        3.14342692639796, 3.88, 4.27358680306891, 5.04, 4.61, 
        3.24, 4.65050758168107, 3.12, 2.67, 1.9, 4.39566810291108, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 2.95192556820218, 
        5.08202082851212, 3.45, 2.76, 4.87291358595182, 2.86512466482102, 
        3.37, 2.58, 4.6, 5.32557456555837, 2.39, 4.22930658076269, 
        5.65, 3.91, 3.29489977515631, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.90720161182954, 3.00118796856347, 
        2.45, 2.61, 1.8973028117966, 2.27605161563677, 2.3, 2.66555089666362, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.57183962780482, 
        2.65, 2.11267000045384, 2.52, 2.61, 3.22, 2.42568966728222, 
        2.71985636359413, 2.79209134345939, 2.75177611768342, 
        2.55719500652106, 2.36, 2.48176001560413, 2.7, 2.65, 
        2.5784017475921, 2.40151893704166, 2.55, 2.1, 2.51, 2.557012598188, 
        2.12, 2.59, 2.29, 2.1, 2.44, 2.59189884762127, 2.12, 
        2.4, 2.27, 2.04, 2.62230154384191, 2.47861110530602, 
        2.11646677679681, 2.3504398329261, 2.34040205121236, 
        2.3, 2.68, 2.5, 1.36, 2.2912991829892, 2.19759174589713, 
        1.92, 2.16, 2.53, 2.56, 1.93898868265228, 1.92, 1.76076179466498, 
        1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.75210087204244, 1.71, 
        2.26865925059208, 1.92000249398297, 2.4, 2, 2.59683683779304, 
        2.51, 2.32, 2.58, 2.24, 2.32231961021066, 2.62, 2.50917113960929, 
        2.3, 2.32, 2.01420040441313, 2.26, 2.22, 2.28, 2.2, 2.74, 
        1.98, 2.39877383741995, 1.95998563129586, 1.7, 1.78373883244523, 
        2.46, 1.88, 2.311276580421, 2.27, 2.12, 2.28, 1.94, 2.7, 
        1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.17035664231365, 
        1.76768289066895, 2, 2.42, 3.23, 2.36681245006941, 2.13, 
        2.39, 2.17, 2.29, 2.76741557284119, 2.72726126497109, 
        2.38, 3.03280638297614, 2.4, 2.4, 2.36, 2.25, 2.71395528262202, 
        2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.6943081017502, 
        2.61452615942252, 2.48, 2.2140435203558, 2.36, 2.62, 
        2.48, 2.28161157543032, 2.28, 2.54375473481904, 2.32, 
        2.38, 1.94611881404338, 2.56958257782349, 2.64, 2.38, 
        2.54, 2.42662175724466, 2.35, 2.3, 2.6365821405761, 2.6, 
        2.30211184554773, 2.69, 2.70190435168387, 2.32, 2.28, 
        2.52284229099749, 2.28885037602917, 2.48, 2.26, 2.37, 
        2.74), alca = c(15.6, 13.4552162773788, 21.4851350999236, 
        16.8, 21, 15.2, 15.9938299165562, 17.6, 14, 16, 20.8101242244742, 
        16.8, 16, 11.4, 12.454892476723, 14.3480835309794, 19.0607080102154, 
        20, 11.6905109882977, 15.2, 16, 18.6, 17.2068759739698, 
        17.8, 20, 25.1829746676314, 16.1, 17, 20.0251269456141, 
        20.8169957383464, 17.560578888348, 19.1, 16.0548408489979, 
        19.2030123601529, 19, 20.5, 15.5, 10.8139346721559, 15.5, 
        13.2, 16.2, 18.8, 14.6727415019262, 20.0598051076619, 
        17, 14.1227101300797, 16, 13.707559528512, 18.8, 17.4, 
        12.4, 16.7933433705344, 17.5103483312928, 17.1942199229296, 
        21.3873136695532, 20.5, 16.3, 16.8, 13.1326635929738, 
        10.6, 18.8818939863422, 16.8, 18, 19, 19, 18.1, 19.8061226037324, 
        19.6, 16.7638418951857, 16.8, 20.4, 23.1593771134972, 
        24, 30, 21, 17.7954312582434, 16, 18, 14.8, 21.6858603631635, 
        19, 22.0767672053748, 24, 22.5, 18, 18, 22.8, 22.2542177595384, 
        21.6, 20.2896509864536, 23.4983832147096, 22, 24.5867935651582, 
        18, 18, 19, 21.5, 16, 13.0670252591212, 18, 17.5, 18.5, 
        21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 22.1572042033367, 
        25.7058064014789, 23.9058112483767, 17.9756916573545, 
        21.5, 20.8, 16.6197514186269, 16, 17.6323397110789, 21.1596851935163, 
        18.358670510838, 23.8087062002929, 21.5, 21, 21, 21.5, 
        28.5, 23.3340137347675, 22, 22.5487427930681, 20, 24, 
        21.5, 14.0433369187336, 21.1410764621996, 20.4717158345987, 
        25, 19.5, 17.5923545446281, 21, 15.7988164168135, 23.5, 
        20, 22.5723976736493, 21, 20, 22.4410356953704, 19.7254253061366, 
        21.5, 24, 22, 23.4669470812227, 18.5, 20, 22, 19.5, 9.99037816870617, 
        25, 26.5311376512432, 20.4577569330863, 19.4993559282251, 
        23.252641953076, 18.5, 23.3712474458354, 22.5, 23, 19.5, 
        24.5, 25, 18.6762888673366, 19.5, 24.1924512658064, 20.5, 
        23, 20, 20.0667681479015, 24.5), mg = c(127, 100, 101, 
        113, 84.7720728318828, 112, 96, 121, 97, 99.1152569660776, 
        119.176550987306, 120.557098683143, 89, 91, 102, 112, 
        120, 113.033459746982, 112.660780394511, 116, 126, 102, 
        102.478501367584, 95, 108.861326513224, 124, 93, 116.151872854425, 
        107, 96, 101, 106, 104, 116.721735038291, 110, 102.719947991787, 
        110, 113.980872475162, 95.5886613220932, 97.165329131005, 
        117, 110.573111025251, 101, 116.865422605426, 107, 111, 
        102, 101, 123.447429549893, 108, 92, 72.5050380680376, 
        111, 115, 116.451822392505, 105.628309107311, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 71.0003057415109, 
        151, 85.3634803342267, 86, 87, 139, 101, 96.6989868673805, 
        86.9411090701551, 112, 94.9053432495253, 106.017958065889, 
        86, 78.6502338267228, 78, 85, 95.9747126467248, 99, 90, 
        88, 84, 70, 81, 112.72544654821, 84.7088634237967, 88, 
        101.713729408661, 107.449527465131, 134, 85, 88, 77.1297072210594, 
        97, 88, 93.8600899277268, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 93.3549790244067, 64.7525159576087, 
        87, 96, 110.233857864661, 102, 86, 131.721604495059, 
        85, 100.649607236698, 92, 88, 80, 122, 81.1969024227247, 
        98, 106, 73.0551875456654, 94, 89, 105.148748320412, 
        88, 101, 101.922369009798, 89, 97, 84.8192507914495, 
        112, 102, 115.470162147203, 94.2878353477864, 92, 113, 
        123, 89.5955665735812, 116, 98, 103, 104.197358779021, 
        89, 97, 98, 89, 92.5458898611943, 106.836825953876, 106, 
        106, 90, 88, 89.4959775581343, 88, 105, 112, 96, 86, 
        91, 95, 102, 66.9407608511463, 120, 106.973355974049), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 2.72795004313234, 
            2.5, 2.6, 2.98153130583379, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.42248157359784, 2.85, 2.8, 2.82816224134057, 
            3.3, 3.42052120642947, 3, 2.41, 2.61, 2.48, 2.53, 
            2.45243920884716, 3.0579251672599, 2.4, 2.95, 2.65, 
            3, 3.01099963135765, 2.42, 2.95, 2.78852613276495, 
            2.7, 2.6, 2.45, 2.81211942885458, 2.89601219490941, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 3.14779842169983, 2.82452102057672, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 1.90938716120666, 
            3.5, 1.89, 2.82223489681449, 2.23719904096955, 2.11, 
            2.53, 2.98604055205735, 1.1, 2.95, 1.88, 2.12767035404704, 
            1.53771802984138, 2.23853098570333, 2.95503828639963, 
            2.14994071225096, 1.9, 2.83, 3.07438379852508, 2.2, 
            1.88235046827701, 1.65, 1.84335733239335, 2.2, 1.48091259154255, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 1.89606092416106, 
            1.26724618180465, 2.55, 3.52, 3.55004560707121, 3.35106213049936, 
            1.89582048353764, 2.56, 2.5, 2.2, 2.52282501481605, 
            1.65, 2.32212060576555, 2.36, 2.74, 3.18, 2.55, 1.75, 
            1.89804327435613, 2.56, 2.46, 1.78754339730929, 2, 
            2.13851504311656, 2, 2.9, 3.18, 1.67815202060009, 
            2.18698292852732, 2.86, 2.49859888952034, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.314046184822, 1.55938400061403, 
            1.7, 2.20058617510869, 1.62, 1.38, 1.18811276295154, 
            1.62, 1.7054677584808, 1.51056511886242, 1.4, 1.34437229354222, 
            2, 1.38, 1.59205361749849, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.35030899036835, 1.48, 1.55181755504198, 
            1.8, 1.9, 0.981791837674806, 1.48967718969699, 2.3, 
            1.83, 1.99049647098076, 1.39, 1.35, 2.05383836448414, 
            1.7, 1.48, 1.55, 2.01346249937247, 1.25, 1.39, 1.68, 
            1.57056418433262, 1.34661686941249, 0.948520791269623, 
            1.65, 2.05), fla = c(2.85325739114704, 2.76, 2.99466791127912, 
            3.49, 2.21201455270393, 3.39, 2.52, 2.51, 2.98, 3.50896216858535, 
            3.32, 2.43, 2.76, 2.59788842200546, 3.64, 2.91, 3.14, 
            3.4, 2.88529360273085, 3.43836001735645, 3.76484193014524, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.60045634547213, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.13113032386474, 3.14294065451978, 
            2.98, 2.36016873809942, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.16511253611618, 2.92, 
            3.54, 3.58773699176637, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.41542695800443, 3.23, 3.67, 0.57, 1.72768008701951, 
            1.41, 1.79, 2.98151197036685, 1.75, 2.65, 1.81956339367498, 
            2, 1.3, 2.78332588280881, 1.02, 2.86, 1.98828759295329, 
            2.89, 2.14, 1.57, 2.24229656192118, 1.84106989748747, 
            1.85, 2.33001103496778, 2.26, 2.53, 1.58, 1.59, 2.21, 
            1.70212035555315, 1.69, 1.61, 1.69, 2.03826888317602, 
            1.80904278112497, 1.25, 1.46, 2.32366995343547, 2.91461903851636, 
            2.27, 0.99, 1.83745776295829, 3.31285086640505, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.94370797450654, 
            2.92, 2.56345864819382, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.07737412059005, 1.64, 3.41058276792925, 
            3.79530842003991, 1.71229118939702, 1.71935210369595, 
            3.03, 2.65, 3.15, 2.24, 2.16200272636091, 1.75, 0.733554901063605, 
            0.464261034260893, 0.770450501075528, 1.2, 0.58, 
            0.66, 0.47, 0.305520655272725, 0.430403075455442, 
            0.6, 0.5, -0.150130178121449, 0.52, 0.8, 1.07859176896075, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.767162715193118, 0.58, 1.31, 1.1, 0.92, 
            1.96831327334617, 0.6, 1.0044987743086, 0.68, 0.47, 
            0.92, 0.66, 0.84, 0.96, 0.49, 0.71224349086453, 0.7, 
            0.61, 0.503904676979124, 0.69, 0.68, 0.76), nfla = c(0.28, 
            0.294926593297046, 0.3, 0.24, 0.39, 0.206621786164345, 
            0.201333434724668, 0.31, 0.29, 0.22, 0.22, 0.26, 
            0.29, 0.43, 0.29, 0.3, 0.268219689254811, 0.233085567501817, 
            0.32, 0.17, 0.24, 0.25, 0.27, 0.26, 0.262296333783146, 
            0.43379189868968, 0.34, 0.27, 0.304083581159431, 
            0.26, 0.29, 0.163485360796649, 0.42, 0.342117246808919, 
            0.29, 0.26, 0.322835687532365, 0.430856680446737, 
            0.28, 0.2, 0.34, 0.202755039840086, 0.28988204859725, 
            0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 0.32, 0.17, 0.22, 
            0.32, 0.39, 0.21, 0.2, 0.26, 0.334101829805765, 0.117433682739223, 
            0.28, 0.386142518084727, 0.53, 0.32, 0.280992988745173, 
            0.45, 0.37, 0.26, 0.27, 0.426557473463103, 0.295798724978752, 
            0.339381445287591, 0.444729482018257, 0.349264651185917, 
            0.21, 0.13, 0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.436732072356184, 
            0.4, 0.61, 0.22, 0.3, 0.328657391258726, 0.447126193304667, 
            0.298170857015465, 0.383026557093163, 0.334362558239145, 
            0.5, 0.398934126503971, 0.25, 0.17, 0.32, 0.397601081660911, 
            0.388558739672773, 0.24, 0.397140409871437, 0.322256013263346, 
            0.29, 0.34, 0.37, 0.283638972642465, 0.66, 0.337560616297643, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.314739584559402, 
            0.42, 0.397826538038938, 0.52, 0.3, 0.491428580551884, 
            0.335301426510966, 0.440262716918903, 0.32, 0.354444959411411, 
            0.43, 0.3, 0.431991948088354, 0.286566807106321, 
            0.39, 0.287880947527308, 0.4, 0.492605867093343, 
            0.479292043705074, 0.572460243135341, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.531728185808853, 
            0.440710105579081, 0.560393086679681, 0.426723546247198, 
            0.43, 0.4, 0.47, 0.45, 0.308401252870299, 0.22, 0.430086713825725, 
            0.26, 0.61, 0.53, 0.568254713622583, 0.375570380349327, 
            0.63, 0.53, 0.52, 0.5291735461298, 0.5, 0.6, 0.4, 
            0.41, 0.696118672252175, 0.43, 0.4, 0.39, 0.432369688625097, 
            0.522792540063083, 0.48, 0.433775946860936, 0.437244105469425, 
            0.43, 0.43, 0.53, 0.534559568873324), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 2.4310612664946, 2.96, 1.46, 2.37197787510278, 
            2.16174015480457, 1.86, 1.66, 2.1, 1.02888817059936, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 1.64044300670032, 2.40033870978008, 1.54, 1.86, 
            1.36, 1.44, 1.37, 1.84654055931423, 1.63471691897409, 
            2.3583700818717, 1.76328538440032, 1.00005995205734, 
            2.03, 1.31941170562202, 2.19, 2.14, 2.38, 2.08, 1.82885123792623, 
            2.29, 1.87, 1.68, 1.62, 2.45, 2.92569664823037, 1.66, 
            2.37135613227133, 0.42, 2.06745746983384, 1.63955617940433, 
            0.73, 1.87, 1.03, 2.26728404026007, 1.01622191218799, 
            1.04, 1.31109010861879, 2.5, 1.46, 1.87, 1.03, 2.52923396338884, 
            1.98339147382309, 1.15, 1.46, 2.0211662268568, 2.76, 
            1.95, 1.43, 1.84785901818123, 1.34558324182838, 1.62, 
            1.8961475638048, 1.46, 1.56, 1.34, 1.35, 1.38, 1.64, 
            1.63, 1.32534654410921, 1.99, 1.35, 3.28, 1.56, 1.77, 
            1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.653618759251335, 
            1.42, 1.20732155679646, 1.63, 2.08, 2.49, 1.86152091543116, 
            1.46248346218271, 2.13294883797605, 1.51790441952005, 
            1.04, 2.06058823659305, 1.19592707980458, 1.61, 0.83, 
            1.87, 1.83, 2.60353354631539, 1.71, 2.12584691635924, 
            2.91, 1.35, 1.77, 1.0422912704105, 1.9, 2.40028300203484, 
            0.94, 0.745370654659278, 0.83, 0.84, 1.25, 0.94, 
            0.8, 1.1, 0.88, 1.45286914740307, 0.75, 0.112300998474886, 
            0.55, 1.02925526076818, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 1.05774585781644, 2.13790552048291, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 2.24417577956077, 
            1.54648970145992, 0.94, 1.44040598907379, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.52662915203912, 
            6.75, 4.19756616832697, 5.05, 7.52140557468842, 7.22, 
            5.75, 5.90230012784034, 5.6, 7.1468059221079, 7.5, 
            6.52884762432988, 7.36033722139903, 6.41435254812431, 
            8.7, 5.1, 5.65, 4.5, 3.87112465741136, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.82507456419738, 
            6.06927424769229, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            4.42824895952419, 8.9, 7.2, 5.43389854891941, 6.17808930202428, 
            6.3, 5.85, 6.25, 7.00350928589472, 4.7833638267535, 
            6.8, 8.99167619014259, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 11.3705888012818, 3.38, 
            3.74, 3.58115670330288, 3.21, 3.8, 4.6, 2.7794810768226, 
            5.46399207516399, 3.19071819736169, 2.5, 3.9, 1.46601154883649, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 7.18899405246447, 2.9, 6.48512491432057, 
            4.14094778045427, 2.84382508722822, 3.03536820555432, 
            2.8, 4.57088574419673, 2.94, 2.7, 3.4, 3.3, 2.7, 
            0.883785884074689, 2.9, 4.04661835365122, 0.956764463331528, 
            3.08, 2.9, 1.9, 0.690954989912899, 2.38739844123313, 
            3.4, 1.28, 3.25, 7.79530968712658, 2.09241088337187, 
            8.08168039440182, 2.8, 4.65167450644815, 3.94, 1.0545715918862, 
            2.12, 5.08455499412787, 3.20164717872927, 8.12260985874047, 
            8.82856321929577, 5, 5.45, 7.1, 6.0257403039227, 
            5, 5.7, 4.92, 7.20163481151933, 7.65518411674212, 
            4.35, 4.4, 8.21, 4, 4.9, 6.76885099560831, 8.42, 
            9.4, 1.33304982176137, 9.93110115315849, 7.1, 10.52, 
            7.6, 7.9, 8.68450056216526, 7.5, 13, 8.94694874575837, 
            7.65, 6.32369265884461, 5.58, 10.0356697113203, 9.92318661422149, 
            6.62, 10.68, 10.26, 8.97041014839015, 8.5, 6.83874516844159, 
            9.6144388385897, 9.7, 7.7, 7.3, 8.53576524299095, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.20900615045548, 
            0.86, 1.53601682194865, 1.05, 1.02, 1.03982081133138, 
            1.08, 1.03037488173342, 1.25, 1.06287129621758, 1.15, 
            1.25, 1.2, 1.28, 0.943013975588705, 1.13, 1.23, 0.849253253043451, 
            1.09, 0.989436444249055, 1.11, 1.09, 1.12, 1.13, 
            1.23092015182518, 1.02, 0.990068715743513, 1.04, 
            1.19, 1.09, 1.14066549783007, 1.25, 1.1, 1.14845328173117, 
            1.11460204418148, 1.07493316792615, 1.18, 0.807901046235785, 
            0.95, 0.91, 0.88, 0.82, 1.32860833944588, 0.87, 1.18691402707504, 
            1.0397563478477, 1.07, 1.12, 1.12, 1.00583570008171, 
            1.18694641256297, 1.1960160865148, 1.26154905576333, 
            1.0435459217376, 0.94, 1.07, 0.89, 0.731631126136514, 
            1.25, 0.98, 1.23, 1.22, 0.739440909465569, 0.978837903125772, 
            1.18722192271976, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.05567641837405, 1.23, 1.19, 0.96, 1.06, 
            1.19, 1.38, 0.965587343034752, 1.31, 0.84, 1.04074057056781, 
            1.23, 1.24412223322932, 1.24985092409461, 1.05089265146552, 
            1.07, 1.08, 1.05, 1.05121988283934, 1.15, 1.05581707151305, 
            0.813764847701138, 0.589209803363039, 1.03538259944141, 
            0.976310389980807, 1.42, 1.27, 0.869902963607825, 
            0.8, 0.999832166303536, 1.04, 1.2460063974971, 1, 
            0.967281390232942, 0.86, 1.11838489456543, 0.863398615940552, 
            0.9, 1.13116077990759, 1.1, 0.93, 1.71, 0.95, 1.18247235479612, 
            0.7, 1.00313906090868, 0.8, 1.25259261862527, 0.92, 
            0.73, 1.18535131842004, 0.86, 0.69, 0.97, 0.89, 0.79, 
            1.31623289981054, 0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 
            0.82, 0.81, 0.609078422596731, 0.777888718984402, 
            0.684706951358167, 0.89, 1.10155898418501, 0.65, 
            0.6, 0.255256442452409, 0.54, 0.55, 0.124484168971498, 
            0.71130488640777, 0.48, 0.61, 0.56, 0.901705551370519, 
            0.6, 0.57, 0.67, 0.57, 0.827724861822777, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.499451209613893, 0.74, 
            0.67, 0.518210223801058, 0.72319943322606, 0.549362305181081, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 2.65492838294356, 2.93, 2.95322044463619, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 2.1885875951742, 
            3, 2.75180033264667, 2.65, 3.67898347174265, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.30971083301752, 3.2, 
            2.94122264055941, 2.77, 3.18086780248616, 2.41546924350301, 
            2.71, 2.88, 2.87, 3, 2.87, 3.96194675728583, 2.77906970193211, 
            2.89997982924494, 2.69, 3.40656121748171, 3.38, 3, 
            3.61807190719402, 3, 3.35, 3.33, 3.52649949674904, 
            3.33, 2.75, 3.1, 3.25149796914233, 2.99605590748507, 
            3.26, 2.93, 3.2, 3.03, 3.26051515191316, 2.84, 2.87, 
            1.82, 2.6428390054698, 1.59, 2.46, 2.87, 2.60516325126472, 
            2.3, 3.18, 3.48, 3.03008276789809, 3.50677954666822, 
            1.82, 3.66349157602344, 2.54887495496477, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.29938622252888, 3.18096996200254, 
            3.38953363776765, 3.14, 2.72, 2.01, 2.68701637616381, 
            2.88236090880736, 3.19233785070098, 3.21, 2.75, 3.21, 
            3.04807601324231, 2.65, 3.05483487826695, 3.3, 3.13077624858774, 
            2.63, 1.20758636866035, 2.74, 3.31632947263945, 2.83, 
            2.96, 2.77, 3.38, 2.58694734218499, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.15838568733609, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            2.29236458849604, 3.64, 2.80218683052829, 2.84, 3.06871593182863, 
            2.78, 1.80292875939288, 2.3415334749832, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.91257899258104, 
            1.84819500469604, 1.9165939585574, 2.05, 2, 1.44582537083527, 
            1.33, 1.4432302791699, 1.62, 1.33, 2.32904016214475, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 3.32096433180929, 1.61302590584962, 1.75, 1.68, 
            1.75, 1.63871625008569, 1.29522845240715, 1.68390722849925, 
            1.32963321523513, 1.68017574213081, 1.36811836386969, 
            1.50420075493181, 1.76753175817811, 1.95229420901993, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 777.397465341688, 
            929.240186783248, 735, 1450, 1290, 1295, 1045, 1045, 
            1030.06576265456, 1280, 1320, 1545.67965765315, 1547, 
            1310, 1280, 980.801448135437, 1680, 845, 1181.6526535337, 
            770, 1035, 1015, 845, 921.874187119348, 1195, 1285, 
            915, 1035, 1285, 966.209665245813, 1291.78383286846, 
            1235, 878.562895258833, 920, 880, 1230.4942000472, 
            863.654628877011, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1181.29216131927, 1150, 1265, 
            1190, 1051.28980209371, 1060, 1120, 970, 1270, 1285, 
            725.199267221999, 682.255974902859, 891.064570736982, 
            630, 473.634713474562, 835.569957710437, 835.404820746004, 
            502, 601.641897650984, 511.10133800445, 718, 870, 
            410, 472, 562.633872543898, 886, 271.405194285115, 
            392, 405.574608309094, 662.357732225077, 463, 278, 
            714, 437.336063765151, 515, 967.237115995484, 757.257623059659, 
            726.387883511503, 562, 680, 625, 480, 450, 495, 558.096027529737, 
            345, 687.185226373052, 625, 428, 660, 406, 515.145543665369, 
            562, 438, 415, 672, 286.041395756819, 510, 580.353539416331, 
            312, 680, 562, 325, 441.624143957675, 434, 385, 357.93620171081, 
            597.891396307254, 884.327584537799, 372, 564, 349.511968648801, 
            1292.6563657673, 365, 380, 381.986877942579, 378, 
            88.3772845366051, 466, 561.10954975996, 580, 873.101094356593, 
            518.833847122894, 560, 600, 887.546946676888, 695, 
            720, 515, 580, 590, 600, 644.729959342138, 520, 550, 
            394.070844533334, 830, 415, 420.108289672562, 493.318545359572, 
            550, 500, 480, 425, 675, 640, 454.527660423421, 480, 
            880, 660, 428.961473882287, 305.933324348278, 680, 
            570, 984.761824831084, 615, 567.493590810377, 695, 
            685, 788.511256665077, 851.731070491728, 510, 470, 
            469.73470218885, 740, 672.663641517143, 835, 840, 
            607.160512752604)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 14.7719525137479, 
        13.24, 13.963817155989, 14.39, 14.06, 14.3182434300859, 
        13.86, 13.2793134459978, 12.9895462447097, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.231911665296, 13.64, 13.2612445592928, 
        12.93, 13.71, 12.85, 13.5, 13.4400896364744, 13.39, 13.3, 
        13.87, 14.3515336093023, 13.73, 13.58, 13.68, 13.76, 
        13.1267052007204, 13.762344596521, 13.1625374339564, 
        13.05, 12.7019732955046, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 13.5440156949781, 13.9, 14.04108058171, 
        13.8399001625205, 13.05, 14.6468539258357, 13.82, 13.4682652514091, 
        13.74, 15.0918453869188, 14.22, 13.29, 13.5715331982808, 
        12.37, 12.33, 12.64, 11.7832569758484, 12.37, 12.17, 
        12.37, 13.0275509415921, 12.6816145826774, 13.34, 12.21, 
        11.3013379617456, 13.86, 12.5409702023783, 12.99, 11.96, 
        11.66, 11.8777529387377, 11.84, 12.4358918921312, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 12.2090111777485, 12.16, 
        12.9691789482122, 12.587141894723, 12.08, 12.08, 12, 
        12.69, 11.6189254830754, 11.62, 12.4534336481658, 11.81, 
        12.29, 13.0562664164038, 12.29, 12.08, 12.6, 14.3369354528039, 
        12.5212685327666, 12.51, 12.7681976880843, 11.8064594515153, 
        12.72, 12.22, 11.61, 12.2288379600738, 12.0096889619938, 
        11.76, 12.3474661590231, 12.08, 11.03, 11.82, 13.332005331433, 
        11.7959922279656, 11.8074376466475, 11.45, 14.5609996066482, 
        12.42, 13.05, 11.87, 12.07, 12.7407096008893, 11.79, 
        12.3164733118555, 12.04, 12.86, 12.88, 12.2719505914568, 
        12.7, 12.51, 12.6876202148105, 12.25, 12.53, 11.9976310864511, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 11.6096026089359, 
        11.6020991783004, 12.87, 13.32, 12.1429690674063, 11.7036709359065, 
        12.79, 12.3372280180126, 13.23, 12.5143273205996, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.0547253616141, 13.69, 
        12.85, 12.96, 13.78, 13.73, 14.1604020488627, 14.3397643890369, 
        13.58, 13.4, 12.2254681665543, 12.77, 14.16, 12.9312339247451, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 2.35642852798696, 
        0.0357876828811992, 1.95, 4.46522374694716, 2.26408986826437, 
        1.87, 2.15, 1.02201698378372, 1.35, 2.16, 1.48, -0.0994603517015133, 
        1.73, 1.87, 1.68050438137647, 1.92, 2.06142647977236, 
        0.645410874549166, 3.1, 1.63, 3.8, 1.81665283315527, 
        1.6, 1.81, 3.1193406476322, 0.449961768186864, 3.50003633849858, 
        1.9, 4.56517297604202, 1.5, 2.51102155071467, 1.83, 1.53, 
        2.02065502775616, 1.86621483262389, 2.54797581375345, 
        1.65, 1.5, 3.2641402537411, 1.71, 1.67903570264744, 2.67293636999156, 
        3.98, -0.827803292649601, 4.04, 0.18611729585102, 1.68, 
        0.236105737759747, 0.764435760823864, -0.109074557573849, 
        1.65, 1.50648689748468, 1.9, 1.67, 1.3725060781378, 1.7, 
        1.97, 3.77489118505862, 0.94, 2.96912356443937, 1.36, 
        1.25, 0.959735697304576, 3.80757387778606, 1.21, 1.01, 
        1.17, 0.55831126442417, 4.53916175731655, 2.35653308536129, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.52601187163129, 
        0.99, 3.87, 1.79231792129377, 1.72442952553897, 1.13, 
        5.58555105747459, 3.30574161060571, 0.98, 1.61, 1.67, 
        1.36737338239643, 2.79161787070367, 2.62906071906637, 
        1.51, 3.04846544509256, 1.78520949671271, 1.78846658758568, 
        2.79132102065594, 2.63752109471534, 1.41, 0.230868459277043, 
        -0.197584706813677, 2.08, 1.53076804443067, 5.96176402801948, 
        1.72, 2.61630097743861, 2.55, 1.73, 3.05877926324716, 
        1.29, 1.0694079856455, 3.74, 2.94449135866344, 2.68, 
        0.74, 1.39, -0.171104038478568, 1.77508756777651, 1.61, 
        3.43, 0.84591340420676, 2.4, 2.05, 4.43, 1.34500534283119, 
        4.31, 2.16, 1.53, 2.13, 1.63293238742961, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 4.44131436721276, 4.11468784045714, 
        5.51, 3.59, 5.48325132075924, 2.81, 2.56, 5.00600998856532, 
        2.53581090130941, 3.88, 4.59367313789983, 5.04, 4.61, 
        3.24, 3.62335021642043, 3.12, 2.67, 1.9, 1.67222632005602, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 3.17328279916526, 
        4.47389298605931, 3.45, 2.76, 3.47947339476511, 2.59227258406531, 
        3.37, 2.58, 4.6, 3.04236456140979, 2.39, 2.34201972277787, 
        5.65, 3.91, 3.30217629637233, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.76698790145262, 2.41368108240706, 
        2.45, 2.61, 2.88249466797338, 2.29303397955385, 2.3, 
        2.27730384103803, 2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 
        2.48, 2.56, 2.4160186017753, 2.65, 2.65522825832992, 
        2.52, 2.61, 3.22, 2.32316810630747, 2.52536802091619, 
        2.36331340207652, 3.383466215958, 2.46829891892739, 2.36, 
        2.40456748312694, 2.7, 2.65, 2.35125929951913, 2.31176582367563, 
        2.55, 2.1, 2.51, 2.26359003701323, 2.12, 2.59, 2.29, 
        2.1, 2.44, 2.39677527545502, 2.12, 2.4, 2.27, 2.04, 2.60753208416494, 
        2.32213245442783, 2.11192871263248, 2.70041568290833, 
        2.8273310301857, 2.3, 2.68, 2.5, 1.36, 2.67019890657201, 
        2.24768371076317, 1.92, 2.16, 2.53, 2.56, 2.39133516647846, 
        1.92, 2.06676661330091, 1.75, 2.21, 2.67, 2.24, 2.6, 
        2.3, 2.00396038428371, 1.71, 1.98825601515193, 2.49555523032116, 
        2.4, 2, 2.07192385240001, 2.51, 2.32, 2.58, 2.24, 2.38572118218203, 
        2.62, 2.74561882639693, 2.3, 2.32, 1.6934714474786, 2.26, 
        2.22, 2.28, 2.2, 2.74, 1.98, 2.21251737171405, 2.27281026819019, 
        1.7, 2.27877361291965, 2.46, 1.88, 2.08512562930779, 
        2.27, 2.12, 2.28, 1.94, 2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 
        2.2, 1.99, 2.58071365713754, 2.1711816195252, 2, 2.42, 
        3.23, 2.76846503464812, 2.13, 2.39, 2.17, 2.29, 2.53997582888171, 
        2.36015295254804, 2.38, 2.25390180499998, 2.4, 2.4, 2.36, 
        2.25, 2.50003084854224, 2.54, 2.64, 2.19, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.76658007562069, 2.18060940658103, 
        2.48, 2.09586364783894, 2.36, 2.62, 2.48, 2.71393022979002, 
        2.28, 2.45219195760061, 2.32, 2.38, 1.85774760976407, 
        2.21727686601736, 2.64, 2.38, 2.54, 3.10259599848585, 
        2.35, 2.3, 2.82930803592538, 2.6, 2.64607356252179, 2.69, 
        3.26891366990336, 2.32, 2.28, 2.52515698293455, 2.58208972208958, 
        2.48, 2.26, 2.37, 2.74), alca = c(15.6, 13.6195194510284, 
        21.6031971590071, 16.8, 21, 15.2, 15.543979379005, 17.6, 
        14, 16, 15.0080135043379, 16.8, 16, 11.4, 9.77664752947461, 
        11.0961287682811, 16.5881362593358, 20, 11.9646345676421, 
        15.2, 16, 18.6, 16.8750394341267, 17.8, 20, 24.6465858818677, 
        16.1, 17, 18.9083821274686, 19.0705331506318, 19.5420805531859, 
        19.1, 17.3952147582222, 18.8345935139095, 19, 20.5, 15.5, 
        13.3414968885808, 15.5, 13.2, 16.2, 18.8, 19.3436047437646, 
        17.4760974664184, 17, 14.9543459413352, 16, 12.6770446615996, 
        18.8, 17.4, 12.4, 15.8031496352764, 15.3564918449499, 
        21.8252218521253, 18.5402328564248, 20.5, 16.3, 16.8, 
        9.47173782012726, 10.6, 24.3963147529248, 16.8, 18, 19, 
        19, 18.1, 18.5408746793963, 19.6, 22.1714296388233, 16.8, 
        20.4, 24.1264293073115, 24, 30, 21, 11.3525308027687, 
        16, 18, 14.8, 21.1005509122401, 19, 22.0307999681434, 
        24, 22.5, 18, 18, 22.8, 24.797071902377, 21.6, 12.818501624972, 
        19.5904291327735, 22, 25.4291770387146, 18, 18, 19, 21.5, 
        16, 11.4392141266955, 18, 17.5, 18.5, 21, 19.5, 20.5, 
        22, 19, 22.5, 19, 20, 19.5, 13.5957003552333, 27.0349558580957, 
        26.6694305765058, 15.4360062817235, 21.5, 20.8, 19.1099051652369, 
        16, 22.3270625933805, 14.2373349947931, 19.3345879095534, 
        20.7100658734123, 21.5, 21, 21, 21.5, 28.5, 23.3481378858755, 
        22, 17.9799138438012, 20, 24, 21.5, 24.0798766042383, 
        17.4119314710194, 15.5038885339676, 25, 19.5, 20.8263639639963, 
        21, 22.2089321505912, 23.5, 20, 16.8221941828138, 21, 
        20, 18.3107637195309, 22.1273280005307, 21.5, 24, 22, 
        22.2709333404635, 18.5, 20, 22, 19.5, 11.5794565066056, 
        25, 18.3833426688168, 19.328033933928, 23.1569600198256, 
        23.7137409610886, 18.5, 22.0397058483984, 22.5, 23, 19.5, 
        24.5, 25, 21.3326494023661, 19.5, 20.0965183548748, 20.5, 
        23, 20, 19.7619158724914, 24.5), mg = c(127, 100, 101, 
        113, 133.15031031577, 112, 96, 121, 97, 99.5588623758706, 
        120.510008617977, 118.620184324922, 89, 91, 102, 112, 
        120, 104.803030665344, 113.854682719903, 116, 126, 102, 
        114.555596980024, 95, 95.9544149646489, 124, 93, 105.593808731356, 
        107, 96, 101, 106, 104, 96.6021392630524, 110, 106.472190655868, 
        110, 114.02050541722, 105.154786412441, 88.7745190834362, 
        117, 84.7665221936879, 101, 117.676734349614, 107, 111, 
        102, 101, 118.379351843545, 108, 92, 97.9581854075512, 
        111, 115, 118.647912750165, 106.612052382948, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 55.2754151394206, 
        151, 114.574592272346, 86, 87, 139, 101, 106.056589078241, 
        87.9110123111151, 112, 108.713129941678, 111.147745514196, 
        86, 110.059123066522, 78, 85, 106.983236754315, 99, 90, 
        88, 84, 70, 81, 97.8841996812246, 87.0077393161356, 88, 
        106.380625103132, 95.9972727343807, 134, 85, 88, 65.3198355589066, 
        97, 88, 108.447967206714, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 87.3426857105908, 100.296492169586, 
        87, 96, 106.950911269793, 102, 86, 117.010546598614, 
        85, 101.835763841774, 92, 88, 80, 122, 96.1490217949701, 
        98, 106, 91.1955813673366, 94, 89, 94.8443772158679, 
        88, 101, 92.8756148456866, 89, 97, 101.961942019722, 
        112, 102, 98.7147615792433, 68.8247967898631, 92, 113, 
        123, 86.136603523396, 116, 98, 103, 92.9753424804986, 
        89, 97, 98, 89, 108.312449762901, 97.1310861540787, 106, 
        106, 90, 88, 83.6569280719749, 88, 105, 112, 96, 86, 
        91, 95, 102, 120.26129115897, 120, 99.9468358238871), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 3.19384567339994, 
            2.5, 2.6, 2.82561129505843, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.53593248745833, 2.85, 2.8, 2.17536330101079, 
            3.3, 3.18132023260759, 3, 2.41, 2.61, 2.48, 2.53, 
            2.02144318252405, 3.07565969739954, 2.4, 2.95, 2.65, 
            3, 3.01066160657688, 2.42, 2.95, 2.98530137517229, 
            2.7, 2.6, 2.45, 2.83847534484279, 3.08784427297642, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 2.50628893070069, 2.54846824130575, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 2.48610418063939, 
            3.5, 1.89, 2.8753059022445, 2.36605600315428, 2.11, 
            2.53, 2.69314901975024, 1.1, 2.95, 1.88, 2.02789461000187, 
            2.05032788020752, 2.22500509730439, 2.39032238291271, 
            1.84188229136291, 1.9, 2.83, 2.5219104667161, 2.2, 
            1.50779320116125, 1.65, 1.90653285917784, 2.2, 1.71728246668638, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.56528342598511, 
            1.60935991926054, 2.55, 3.52, 2.34502010502313, 2.01041695538884, 
            1.69946588681392, 2.56, 2.5, 2.2, 2.55640888637679, 
            1.65, 2.223888976492, 2.36, 2.74, 3.18, 2.55, 1.75, 
            1.90640144769072, 2.56, 2.46, 1.96888124378938, 2, 
            2.2550274627818, 2, 2.9, 3.18, 2.31117709630791, 
            1.9531767580463, 2.86, 2.41083431783943, 2.74, 2.13, 
            2.22, 2.1, 1.51, 1.74080308461826, 1.96008336731137, 
            1.7, 2.29249907571513, 1.62, 1.38, 0.617825120295405, 
            1.62, 1.6698734115781, 1.22467319119835, 1.4, 0.884976588363479, 
            2, 1.38, 1.21569292262164, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 1.85557607700125, 1.48, 1.23340807382963, 
            1.8, 1.9, 1.33192207319439, 1.61838584963827, 2.3, 
            1.83, 1.3852682227958, 1.39, 1.35, 1.74468799518956, 
            1.7, 1.48, 1.55, 1.44175966488499, 1.25, 1.39, 1.68, 
            1.49029293274668, 1.74004257139423, 1.7949197413027, 
            1.65, 2.05), fla = c(2.85552676028877, 2.76, 3.78845670397225, 
            3.49, 3.18603508490765, 3.39, 2.52, 2.51, 2.98, 3.51008055354133, 
            3.32, 2.43, 2.76, 2.61337305208702, 3.64, 2.91, 3.14, 
            3.4, 3.27569150042839, 3.23478234990858, 3.48307029463915, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.64365223935986, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.38260662559891, 3.04918695986019, 
            2.98, 2.87299434744493, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.1237148752862, 2.92, 
            3.54, 3.28357842683659, 2.99, 3.74, 2.79, 2.9, 2.78, 
            3.23199103319969, 3.23, 3.67, 0.57, 1.4302110127231, 
            1.41, 1.79, 2.733520308018, 1.75, 2.65, 2.52805591076522, 
            2, 1.3, 2.58304319416268, 1.02, 2.86, 1.46537047405383, 
            2.89, 2.14, 1.57, 1.9937862530455, 2.0426901826425, 
            1.85, 2.14677119490131, 2.26, 2.53, 1.58, 1.59, 2.21, 
            1.60473137391231, 1.69, 1.61, 1.69, 2.15384961157892, 
            1.78746701435784, 1.25, 1.46, 2.65064586170148, 2.46756352680288, 
            2.27, 0.99, 1.77711437355712, 3.13852483188015, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.94000283018682, 
            2.92, 2.14732764213515, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.87996037392409, 1.64, 3.33817689571924, 
            3.87355968138206, 2.28380735826232, 1.63332049228006, 
            3.03, 2.65, 3.15, 2.24, 1.98332372323319, 1.75, 0.158347225349933, 
            0.963822783765869, 1.18036372517573, 1.2, 0.58, 0.66, 
            0.47, 0.106632367264755, 0.726265512305957, 0.6, 
            0.5, 0.724848092945423, 0.52, 0.8, 0.876652915866927, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 0.572870524518049, 0.58, 1.31, 1.1, 0.92, 
            0.792034991045944, 0.6, 0.711089792250686, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.488697023031863, 
            0.7, 0.61, 0.308262705486986, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.299544054581474, 0.3, 0.24, 0.39, 
            0.151383973513425, 0.477659920684787, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.199103796089447, 
            0.402662444188226, 0.32, 0.17, 0.24, 0.25, 0.27, 
            0.26, 0.316994722132966, 0.518264250999469, 0.34, 
            0.27, 0.3445578637741, 0.26, 0.29, 0.148797913656585, 
            0.42, 0.300898776422134, 0.29, 0.26, 0.267887989220062, 
            0.219443624203073, 0.28, 0.2, 0.34, 0.365443669252355, 
            0.162298152873337, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.273649133321561, 
            0.260935019374842, 0.28, 0.464620722436504, 0.53, 
            0.32, 0.432835170987061, 0.45, 0.37, 0.26, 0.27, 
            0.466415914521633, 0.495788451933884, 0.313399985805546, 
            0.493182879590171, 0.417055964903472, 0.21, 0.13, 
            0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.326414453824025, 
            0.4, 0.61, 0.22, 0.3, 0.40230353182653, 0.377400052243214, 
            0.444041973582296, 0.194063940531977, 0.415715196213488, 
            0.5, 0.29720671062414, 0.25, 0.17, 0.32, 0.625181001077368, 
            0.335968202125542, 0.24, 0.306827866752553, 0.282047452620518, 
            0.29, 0.34, 0.37, 0.211954844269691, 0.66, 0.478699982654968, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.313174316493534, 
            0.42, 0.312285485835235, 0.52, 0.3, 0.392665613111844, 
            0.183700187516253, 0.493620548853759, 0.32, 0.111501229584255, 
            0.43, 0.3, 0.609298735496029, 0.240434247985957, 
            0.39, 0.419585316729299, 0.4, 0.361809270080513, 
            0.430592970189288, 0.392637603105275, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.400859613675599, 
            0.403260810670084, 0.26799910182386, 0.550226979610487, 
            0.43, 0.4, 0.47, 0.45, 0.31486507996276, 0.22, 0.444429080848343, 
            0.26, 0.61, 0.53, 0.538742044175634, 0.570211395019458, 
            0.63, 0.53, 0.52, 0.435649683740319, 0.5, 0.6, 0.4, 
            0.41, 0.468406182041558, 0.43, 0.4, 0.39, 0.559249073946839, 
            0.385128432182506, 0.48, 0.440180157416538, 0.578130754410887, 
            0.43, 0.43, 0.53, 0.614136598019105), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 1.69708002415893, 2.96, 1.46, 2.16420134734588, 
            2.50737359198175, 1.86, 1.66, 2.1, 0.806509774748303, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 2.31695209299174, 2.36475365572183, 1.54, 1.86, 
            1.36, 1.44, 1.37, 1.54534910668159, 2.09815997342996, 
            1.66584076224246, 1.43296367948143, 1.20049953892333, 
            2.03, 1.25101244084989, 2.19, 2.14, 2.38, 2.08, 1.89595527194421, 
            2.29, 1.87, 1.68, 1.62, 2.45, 1.9667867327209, 1.66, 
            1.65197765391927, 0.42, 1.30577706288066, 1.57919756651797, 
            0.73, 1.87, 1.03, 2.51944854057803, 2.32533182370004, 
            1.04, 0.938505027861539, 2.5, 1.46, 1.87, 1.03, 1.79172414220516, 
            1.18291395365625, 1.15, 1.46, 0.908759064326379, 
            2.76, 1.95, 1.43, 1.37577932068402, 1.23549024582267, 
            1.62, 1.79121571835944, 1.46, 1.56, 1.34, 1.35, 1.38, 
            1.64, 1.63, 1.01912924443348, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 0.679686972990425, 
            1.42, 1.39157842994187, 1.63, 2.08, 2.49, 1.11395721405249, 
            1.90547019075144, 1.51055001667522, 2.14965377579705, 
            1.04, 1.56865395718917, 0.85482222007811, 1.61, 0.83, 
            1.87, 1.83, 1.81172283398319, 1.71, 1.16059189674955, 
            2.91, 1.35, 1.77, 1.38753928086487, 1.9, 0.458631971983241, 
            0.94, 1.18435036603611, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 1.09932454297604, 0.75, 1.08655864143421, 
            0.55, 1.02203583610746, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 0.947980235447176, 1.32671271531149, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 2.07314967738422, 
            1.33788984502833, 0.94, 1.86633827974426, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 4.87866841379458, 
            6.75, 5.03897020615047, 5.05, 6.65398818788038, 7.22, 
            5.75, 4.22396492956453, 5.6, 6.7285888892605, 7.5, 
            5.00595090220935, 7.88856788720706, 6.17665878034652, 
            8.7, 5.1, 5.65, 4.5, 4.14921781263196, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 5.21522542815183, 
            5.36887689366941, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            7.55608987371929, 8.9, 7.2, 7.33417017367419, 7.04923422518705, 
            6.3, 5.85, 6.25, 8.48918766934743, 4.59179239690005, 
            6.8, 8.31820913546438, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 6.2321014057296, 3.38, 
            3.74, 4.55063780656175, 3.21, 3.8, 4.6, 1.76357578511272, 
            9.79569305353666, 4.22016298244945, 2.5, 3.9, 2.73883172149061, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 7.20449854604807, 2.9, 6.31632289366875, 
            5.75991901237758, 1.73064709937277, 3.41851717141375, 
            2.8, 3.08950770362445, 2.94, 2.7, 3.4, 3.3, 2.7, 
            1.47793642990553, 2.9, 2.52548785092953, 1.49816267230176, 
            3.08, 2.9, 1.9, 1.56138803736104, 5.85732399888618, 
            3.4, 1.28, 3.25, 7.18056087096927, 2.37484148451639, 
            6.29844989283111, 2.8, 3.77275740149516, 3.94, 2.10987423294555, 
            2.12, -0.457586886403456, 9.28902798158959, 8.69854360122531, 
            7.57619077459296, 5, 5.45, 7.1, 6.05332077005719, 
            5, 5.7, 4.92, 8.98773297226609, 7.97885086799679, 
            4.35, 4.4, 8.21, 4, 4.9, 5.85739265035255, 8.42, 
            9.4, 2.25410697151555, 9.79029612170214, 7.1, 10.52, 
            7.6, 7.9, 8.39510990805924, 7.5, 13, 8.9086872766279, 
            7.65, 7.87234974965057, 5.58, 9.23450804409082, 9.44038443610426, 
            6.62, 10.68, 10.26, 9.75986950854319, 8.5, 6.41235846891144, 
            8.06908325486562, 9.7, 7.7, 7.3, 9.23707794311501, 
            9.3, 9.2), hue = c(1.04, 1.05, 1.61091582380983, 
            0.86, 0.907867381595449, 1.05, 1.02, 1.06800400862944, 
            1.08, 1.05893455719139, 1.25, 0.949544036208342, 
            1.15, 1.25, 1.2, 1.28, 1.04693873400972, 1.13, 1.23, 
            1.06375288964723, 1.09, 0.767096685549909, 1.11, 
            1.09, 1.12, 1.13, 1.16470721670102, 1.02, 1.15162010291645, 
            1.04, 1.19, 1.09, 1.14610869173101, 1.25, 1.1, 1.27790812754185, 
            1.01188287410545, 1.4850200736591, 1.18, 0.897695372603421, 
            0.95, 0.91, 0.88, 0.82, 1.33842055268216, 0.87, 1.37902261278353, 
            1.08361758702283, 1.07, 1.12, 1.12, 1.28019759393948, 
            1.06376500240573, 0.975333723739914, 1.11252359527053, 
            1.13810548683532, 0.94, 1.07, 0.89, 0.344361067759627, 
            1.25, 0.98, 1.23, 1.22, 1.15978924435709, 1.28839078623367, 
            1.26801819977037, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 1.08615183320677, 1.23, 1.19, 0.96, 1.06, 
            1.19, 1.38, 1.05927352210559, 1.31, 0.84, 1.07035542279689, 
            1.23, 1.13896155550021, 1.03992899323599, 1.19493183010947, 
            1.07, 1.08, 1.05, 1.13918373532327, 1.15, 0.966501758883057, 
            1.40180382078692, 0.933791803572478, 1.32965957390532, 
            1.21057407003422, 1.42, 1.27, 1.26506931518317, 0.8, 
            0.896035641088689, 1.04, 0.803825321695394, 1, 0.922137844471716, 
            0.86, 1.3972524688982, 0.937444463076212, 0.9, 1.12748750145793, 
            1.1, 0.93, 1.71, 0.95, 0.911770042772165, 0.7, 1.21275737863402, 
            0.8, 1.12817222638436, 0.92, 0.73, 1.38427535642186, 
            0.86, 0.69, 0.97, 0.89, 0.79, 0.729914901937623, 
            0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.673552095117909, 
            0.71470146854462, 0.409423810682815, 0.89, 0.884504811961145, 
            0.65, 0.6, 0.584611817349566, 0.54, 0.55, 0.365156963519201, 
            0.959872565198344, 0.48, 0.61, 0.56, 0.814375186674332, 
            0.6, 0.57, 0.67, 0.57, 0.670693967624161, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.411897264591676, 0.74, 
            0.67, 0.835614140646646, 0.609561508663751, 0.621273564290688, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 3.03324512250373, 2.93, 2.9912177824828, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.24139411523961, 
            3, 3.60052799336156, 2.65, 3.92643848831024, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 3.02452778876115, 3.2, 
            3.10026278401079, 2.77, 4.02799299359842, 2.90048932992122, 
            2.71, 2.88, 2.87, 3, 2.87, 3.68520116283392, 3.40203092002019, 
            2.59923516360594, 2.69, 3.37784658080454, 3.38, 3, 
            3.54361102512189, 3, 3.35, 3.33, 3.34713048466822, 
            3.33, 2.75, 3.1, 2.61697872163896, 3.37982142644892, 
            3.26, 2.93, 3.2, 3.03, 2.79998268770053, 2.84, 2.87, 
            1.82, 1.63937988249369, 1.59, 2.46, 2.87, 3.05564024524284, 
            2.3, 3.18, 3.48, 2.65641794673189, 3.11630224218055, 
            1.82, 3.69042419766868, 2.46413429160816, 3.5, 3.13, 
            2.14, 2.48, 2.52, 1.67149380797435, 2.361258965229, 
            3.77282265035355, 3.14, 2.72, 2.01, 2.54017200600836, 
            2.84876537038804, 2.93660146303423, 3.21, 2.75, 3.21, 
            2.92897332352468, 2.65, 3.26265660926231, 3.3, 2.34085704175308, 
            2.63, 1.58391648404501, 2.74, 3.84425322176554, 2.83, 
            2.96, 2.77, 3.38, 2.65133486517019, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 2.70861662003821, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            2.09987185458083, 3.64, 3.33821316080998, 2.84, 2.72843588032381, 
            2.78, 3.63823996017147, 1.05953852026123, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 0.996040104207481, 
            2.15048193790534, 1.96111767627384, 2.05, 2, 0.661743466163204, 
            1.33, 1.40533686680475, 1.62, 1.33, 2.49639564944784, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 1.85091418344443, 1.78994329776969, 1.75, 1.68, 
            1.75, 1.54016181046911, 1.84809329867449, 1.33737170789191, 
            1.50014829654879, 1.6944639134684, 1.56179409871922, 
            2.00694912660131, 0.891877151810012, 0.923057053036951, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 1512.29252984039, 
            974.952014482717, 735, 1450, 1290, 1295, 1045, 1045, 
            1054.67417190804, 1280, 1320, 1205.17365848419, 1547, 
            1310, 1280, 1175.55212255801, 1680, 845, 1070.72387471217, 
            770, 1035, 1015, 845, 925.692969313637, 1195, 1285, 
            915, 1035, 1285, 1016.4116834178, 963.712300041707, 
            1235, 865.37834588301, 920, 880, 1422.66331479324, 
            1083.90226108766, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1132.7430771714, 1150, 1265, 
            1190, 993.584841343513, 1060, 1120, 970, 1270, 1285, 
            846.283832012302, 403.568237032715, 797.116412913765, 
            630, 191.922169796867, 459.36193451136, 766.487117394648, 
            502, 663.015521725004, 465.399567457084, 718, 870, 
            410, 472, 981.559651547666, 886, 815.618185391047, 
            392, 982.949916943297, 1065.75746887701, 463, 278, 
            714, 247.225925968924, 515, 731.775023236565, 659.137569956897, 
            482.475806769297, 562, 680, 625, 480, 450, 495, 608.173813801449, 
            345, 425.253499981608, 625, 428, 660, 406, 932.263073500604, 
            562, 438, 415, 672, 377.049839328635, 510, 452.56287412503, 
            312, 680, 562, 325, 712.928288835048, 434, 385, 300.331637627079, 
            91.165892161285, 890.550707849633, 372, 564, 748.930134143892, 
            1227.69490440987, 365, 380, 384.418067845742, 378, 
            867.350399526745, 466, 616.135223868176, 580, 808.21695208824, 
            667.144417491624, 560, 600, 517.128432397409, 695, 
            720, 515, 580, 590, 600, 678.662063067353, 520, 550, 
            449.583669658294, 830, 415, 521.234301630818, 537.270959111826, 
            550, 500, 480, 425, 675, 640, 568.654607815538, 480, 
            880, 660, 802.270754336598, 859.712363966028, 680, 
            570, 463.435235845817, 615, 821.392850100374, 695, 
            685, 635.008947873551, 904.624712582694, 510, 470, 
            935.460427946504, 740, 745.095242757026, 835, 840, 
            450.894664825477)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 15.2024556166842, 
        13.24, 13.9095501349331, 14.39, 14.06, 13.6513378645255, 
        13.86, 13.4511671122563, 12.8081835539496, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 14.2349831644538, 13.64, 13.1678685778828, 
        12.93, 13.71, 12.85, 13.5, 13.8710180399858, 13.39, 13.3, 
        13.87, 12.9209055266084, 13.73, 13.58, 13.68, 13.76, 
        12.6572209866971, 13.8745996187946, 12.4918709092429, 
        13.05, 12.4188355932936, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 13.3372513799399, 13.9, 12.8735506195761, 
        14.0845660583679, 13.05, 15.2998022650519, 13.82, 14.05981036266, 
        13.74, 14.3805409812188, 14.22, 13.29, 13.7439514398516, 
        12.37, 12.33, 12.64, 11.1391174552116, 12.37, 12.17, 
        12.37, 13.191902217347, 12.6312815114059, 13.34, 12.21, 
        12.8795509827671, 13.86, 12.4787439370201, 12.99, 11.96, 
        11.66, 12.0530965883104, 11.84, 13.0931431020383, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.6631586555244, 12.16, 
        12.7834296412533, 12.3173242854147, 12.08, 12.08, 12, 
        12.69, 11.8840978019519, 11.62, 12.2938537732808, 11.81, 
        12.29, 12.2384817133936, 12.29, 12.08, 12.6, 13.0284856009531, 
        12.7864967755987, 12.51, 12.4263766005093, 11.9779329261137, 
        12.72, 12.22, 11.61, 11.9334506866862, 12.4754625892715, 
        11.76, 12.388397506091, 12.08, 11.03, 11.82, 12.1546869407381, 
        11.7517351949314, 12.1825154621897, 11.45, 15.0609197825219, 
        12.42, 13.05, 11.87, 12.07, 13.095512798113, 11.79, 12.3686024490402, 
        12.04, 12.86, 12.88, 13.0380378464761, 12.7, 12.51, 12.6526626345641, 
        12.25, 12.53, 12.7927789823896, 12.84, 12.93, 13.36, 
        13.52, 13.62, 12.25, 12.1687680903004, 11.555680226306, 
        12.87, 13.32, 11.7741324842546, 11.093165025404, 12.79, 
        12.4897131300439, 13.23, 12.2101862780541, 13.17, 13.84, 
        12.45, 14.34, 13.48, 12.9490643043892, 13.69, 12.85, 
        12.96, 13.78, 13.73, 13.4586829286697, 13.6511912893875, 
        13.58, 13.4, 12.8458855726218, 12.77, 14.16, 13.4951549197596, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 3.18266650399164, 
        2.17498499181326, 1.95, 2.1088381632206, 1.54868387491887, 
        1.87, 2.15, 2.88833369445705, 1.35, 2.16, 1.48, 0.395937216062862, 
        1.73, 1.87, 1.67152983268064, 1.92, 2.1000514532439, 
        1.60587821566327, 3.1, 1.63, 3.8, 1.92872628144174, 1.6, 
        1.81, 3.31383748335325, 3.33645010343986, 1.46257000757697, 
        1.9, 2.23600058639179, 1.5, 2.03565384427411, 1.83, 1.53, 
        1.07353578119541, 3.49838589551448, 0.693385708762767, 
        1.65, 1.5, 2.44934550213912, 1.71, 1.66294718696525, 
        2.65150215999097, 3.98, -0.159374847613499, 4.04, 0.620195969418887, 
        1.68, 2.33351910655934, 2.3415827795377, -0.314368283701232, 
        1.65, 4.70909447031438, 1.9, 1.67, 2.60854165196832, 
        1.7, 1.97, 3.00632375439224, 0.94, 2.32688260125546, 
        1.36, 1.25, 1.0789450257523, 3.26332235414524, 1.21, 
        1.01, 1.17, 2.73856492699608, 3.36118280654744, 1.41792779718702, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 1.13449758253883, 
        0.99, 3.87, 1.37088276264119, 0.128506314003666, 1.13, 
        5.14818028051295, 1.50533557670705, 0.98, 1.61, 1.67, 
        1.28510023953271, 0.547232775944501, 2.30483195958637, 
        1.51, 3.2803800979148, 0.234224756699591, 0.711377214749254, 
        1.08550080393363, 5.38410966509929, 1.41, 0.611344189686935, 
        2.68232308401061, 2.08, 1.34499604899275, 4.74053857339665, 
        1.72, 0.756125085304434, 2.55, 1.73, 2.28173078094637, 
        1.29, -0.277914713331568, 3.74, 3.31232163904113, 2.68, 
        0.74, 1.39, 0.00892354106835658, 2.5563642742405, 1.61, 
        3.43, 3.41921359511758, 2.4, 2.05, 4.43, 3.02576714984505, 
        4.31, 2.16, 1.53, 2.13, 2.98516309316059, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 3.95447566837704, 2.1052521726583, 
        5.51, 3.59, 1.93503343315601, 2.81, 2.56, 5.09022476498734, 
        3.09447242327048, 3.88, 3.30084917143672, 5.04, 4.61, 
        3.24, 3.15708171620368, 3.12, 2.67, 1.9, 0.892000648360898, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 3.15242986371003, 
        3.32702797323114, 3.45, 2.76, 2.53410510116337, 2.43630020277773, 
        3.37, 2.58, 4.6, 2.10111276736174, 2.39, 2.5428595977034, 
        5.65, 3.91, 0.64883338385031, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.24315276954555, 2.6993815704618, 2.45, 
        2.61, 2.25779693765891, 2.56009312744434, 2.3, 2.33079432209221, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.4527087370774, 
        2.65, 2.42550826039467, 2.52, 2.61, 3.22, 2.33230710041103, 
        2.33164102217233, 2.34332823889077, 2.0204320220446, 
        2.58782472508994, 2.36, 2.64479441346587, 2.7, 2.65, 
        2.64589504387606, 2.16834745364255, 2.55, 2.1, 2.51, 
        2.30508612084848, 2.12, 2.59, 2.29, 2.1, 2.44, 2.24547915995477, 
        2.12, 2.4, 2.27, 2.04, 2.68898471025902, 2.53931396572657, 
        2.76070176194045, 2.2848206831582, 2.50461966724156, 
        2.3, 2.68, 2.5, 1.36, 2.0164973861159, 2.12804623322398, 
        1.92, 2.16, 2.53, 2.56, 2.29883776444623, 1.92, 2.20037892390573, 
        1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.72815367901828, 1.71, 
        2.17053196813381, 2.14800448062577, 2.4, 2, 2.33084885154111, 
        2.51, 2.32, 2.58, 2.24, 2.10916317146149, 2.62, 2.28287192162247, 
        2.3, 2.32, 2.03494482816691, 2.26, 2.22, 2.28, 2.2, 2.74, 
        1.98, 2.45168238263021, 2.03839033699202, 1.7, 1.90526693275092, 
        2.46, 1.88, 2.11152541215195, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.43271288490423, 
        1.5114944708078, 2, 2.42, 3.23, 2.61137461830783, 2.13, 
        2.39, 2.17, 2.29, 2.26359648165535, 2.36878834719063, 
        2.38, 2.59740948980427, 2.4, 2.4, 2.36, 2.25, 2.43580928396216, 
        2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.25071817898716, 
        2.33095697070069, 2.48, 2.25225409835425, 2.36, 2.62, 
        2.48, 2.79653010937263, 2.28, 2.35870730277473, 2.32, 
        2.38, 1.82457380442904, 2.34869106304587, 2.64, 2.38, 
        2.54, 2.50714856973371, 2.35, 2.3, 2.79595905498353, 
        2.6, 2.20964788145036, 2.69, 2.56092476746147, 2.32, 
        2.28, 2.41143579650275, 2.74973020201168, 2.48, 2.26, 
        2.37, 2.74), alca = c(15.6, 16.2591282340833, 17.9489875759059, 
        16.8, 21, 15.2, 18.3905722259154, 17.6, 14, 16, 16.1528639565896, 
        16.8, 16, 11.4, 15.2615400448126, 15.2744768567953, 21.9070827792478, 
        20, 11.8808381529829, 15.2, 16, 18.6, 20.5528473115268, 
        17.8, 20, 26.8746706535859, 16.1, 17, 25.1442897900269, 
        17.634241961175, 16.6132964106823, 19.1, 17.3396407425232, 
        14.6433735845832, 19, 20.5, 15.5, 21.5844870806554, 15.5, 
        13.2, 16.2, 18.8, 16.4178532020718, 17.9921575172363, 
        17, 12.8807398552838, 16, 13.3138153874988, 18.8, 17.4, 
        12.4, 17.6319313071316, 16.8227738062776, 19.6830566798224, 
        19.8091458330779, 20.5, 16.3, 16.8, 16.465029689088, 
        10.6, 20.2837363183178, 16.8, 18, 19, 19, 18.1, 17.481192183713, 
        19.6, 16.0313613827912, 16.8, 20.4, 19.3112472907221, 
        24, 30, 21, 16.112756840928, 16, 18, 14.8, 25.4668837926804, 
        19, 18.7522499048262, 24, 22.5, 18, 18, 22.8, 19.6954354734021, 
        21.6, 18.2944167588819, 24.6347681170591, 22, 27.8930644099005, 
        18, 18, 19, 21.5, 16, 19.2372066538993, 18, 17.5, 18.5, 
        21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 20.4073370445664, 
        23.6356206285692, 21.0965186840466, 15.8166036012545, 
        21.5, 20.8, 17.9075603181772, 16, 18.0664367103265, 21.4796792530573, 
        20.0359865382332, 20.8903800879288, 21.5, 21, 21, 21.5, 
        28.5, 17.404854244521, 22, 22.8971088426096, 20, 24, 
        21.5, 22.4153109025266, 17.672112822722, 19.6930198419316, 
        25, 19.5, 21.8833042544633, 21, 18.1940047655507, 23.5, 
        20, 22.5853187091625, 21, 20, 22.179471681434, 17.0086311423192, 
        21.5, 24, 22, 22.2564225997819, 18.5, 20, 22, 19.5, 16.9694001211959, 
        25, 17.9698020852842, 18.9529303941434, 20.2229547736242, 
        25.2644611480892, 18.5, 25.0221152362537, 22.5, 23, 19.5, 
        24.5, 25, 18.4258266208636, 19.5, 17.6501245921963, 20.5, 
        23, 20, 16.8036046056386, 24.5), mg = c(127, 100, 101, 
        113, 106.262916084365, 112, 96, 121, 97, 114.401952518885, 
        102.438442129768, 112.289832966764, 89, 91, 102, 112, 
        120, 96.4060836708055, 90.9582449208259, 116, 126, 102, 
        89.5643131900805, 95, 96.36400493899, 124, 93, 93.2263866149705, 
        107, 96, 101, 106, 104, 98.2686585335069, 110, 118.870745151472, 
        110, 82.6233498828117, 122.519971495558, 101.805158126487, 
        117, 108.760369613751, 101, 117.78700186079, 107, 111, 
        102, 101, 111.051938143686, 108, 92, 100.16156365913, 
        111, 115, 103.153597358055, 141.226232666688, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 110.319225718077, 
        151, 89.2129049603294, 86, 87, 139, 101, 74.5370708032666, 
        91.4648844075276, 112, 111.927036808029, 84.8252883575054, 
        86, 100.924605094373, 78, 85, 106.767467127016, 99, 90, 
        88, 84, 70, 81, 106.891486479483, 129.155519449252, 88, 
        93.6346690109297, 107.553880995177, 134, 85, 88, 120.085895166965, 
        97, 88, 97.1785783357187, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 89.5891500610269, 93.3366823629253, 
        87, 96, 98.814648984417, 102, 86, 116.996353703217, 85, 
        81.4880254857975, 92, 88, 80, 122, 107.207864591084, 
        98, 106, 97.9366289145464, 94, 89, 104.872450545253, 
        88, 101, 104.583633378009, 89, 97, 104.801042036288, 
        112, 102, 108.849932953728, 107.173863965082, 92, 113, 
        123, 83.0848136844493, 116, 98, 103, 108.12294166679, 
        89, 97, 98, 89, 109.09335106895, 115.927375888601, 106, 
        106, 90, 88, 119.238985190097, 88, 105, 112, 96, 86, 
        91, 95, 102, 98.1954095918978, 120, 116.913648914031), 
            phe = c(2.8, 2.65, 2.8, 3.85, 2.8, 2.7764362679311, 
            2.5, 2.6, 2.97877947339252, 2.98, 2.95, 2.2, 2.6, 
            3.1, 3.60927830191152, 2.85, 2.8, 3.43494744676237, 
            3.3, 2.91909646894994, 3, 2.41, 2.61, 2.48, 2.53, 
            2.28315254282791, 2.90746783898646, 2.4, 2.95, 2.65, 
            3, 2.90130585430668, 2.42, 2.95, 2.07266532788472, 
            2.7, 2.6, 2.45, 3.0641055167741, 3.70932708323242, 
            3.15, 2.45, 3.25, 2.64, 3, 2.85, 3.25, 3.1, 2.75, 
            2.88, 2.72, 2.45, 3.88, 2.50857792201859, 2.57266232715872, 
            2.96, 3.2, 3, 3.4, 1.98, 2.05, 2.02, 1.66836477526786, 
            3.5, 1.89, 2.2580872263959, 2.41673180820233, 2.11, 
            2.53, 3.65482130295887, 1.1, 2.95, 1.88, 2.71161459297092, 
            1.92371051527929, 1.71816995169826, 2.76301470296027, 
            1.79652570648859, 1.9, 2.83, 2.79475662076635, 2.2, 
            2.12812374083279, 1.65, 2.30784200710854, 2.2, 1.97487530018717, 
            1.92, 1.95, 2.2, 1.6, 1.45, 1.38, 2.45, 3.02, 2.94599584288507, 
            2.46137061397278, 2.55, 3.52, 3.41585187981362, 3.15052906182803, 
            1.82057223420479, 2.56, 2.5, 2.2, 2.11867803360958, 
            1.65, 2.33969686951796, 2.36, 2.74, 3.18, 2.55, 1.75, 
            1.78411080422554, 2.56, 2.46, 1.76877079684462, 2, 
            1.87315124004564, 2, 2.9, 3.18, 2.25354075697299, 
            2.74252522190316, 2.86, 2.39267158963006, 2.74, 2.13, 
            2.22, 2.1, 1.51, 2.03992309295598, 1.25037721413672, 
            1.7, 2.36770413137406, 1.62, 1.38, 1.52724003034416, 
            1.62, 1.94108259209595, 1.79634113516097, 1.4, 1.77770854326316, 
            2, 1.38, 1.47093705840827, 0.98, 1.7, 1.93, 1.41, 
            1.4, 1.48, 2.2, 2.0378563899424, 1.48, 1.70822803903804, 
            1.8, 1.9, 1.29134568329748, 1.80115344181362, 2.3, 
            1.83, 1.2987256847715, 1.39, 1.35, 1.77618071144755, 
            1.7, 1.48, 1.55, 2.05046825125269, 1.25, 1.39, 1.68, 
            1.7849684377709, 1.30223241730108, 1.26245514594206, 
            1.65, 2.05), fla = c(2.754042611998, 2.76, 2.955882441079, 
            3.49, 2.97694918877615, 3.39, 2.52, 2.51, 2.98, 3.46093566393778, 
            3.32, 2.43, 2.76, 2.62661999339348, 3.64, 2.91, 3.14, 
            3.4, 3.54013286330932, 3.08083953205744, 3.42837932631388, 
            2.41, 2.88, 2.37, 2.61, 2.68, 2.94, 2.39021868160617, 
            2.97, 2.33, 3.25, 3.19, 2.69, 3.23767070017704, 1.8154642529939, 
            2.98, 2.7651864638907, 2.43, 2.64, 3.04, 3.29, 2.68, 
            3.56, 2.63, 3, 2.65, 3.17, 3.05638336729194, 2.92, 
            3.54, 3.37679310550515, 2.99, 3.74, 2.79, 2.9, 2.78, 
            2.88421337532157, 3.23, 3.67, 0.57, 1.77766602134265, 
            1.41, 1.79, 2.93053656153422, 1.75, 2.65, 2.35168912099067, 
            2, 1.3, 2.78008815130336, 1.02, 2.86, 1.75919883082407, 
            2.89, 2.14, 1.57, 2.05520944045928, 1.82086834827935, 
            1.85, 2.2096922768631, 2.26, 2.53, 1.58, 1.59, 2.21, 
            2.20120789717769, 1.69, 1.61, 1.69, 2.06477774728582, 
            1.60620285430722, 1.25, 1.46, 2.27745732642477, 2.80326057585114, 
            2.27, 0.99, 1.94420067084428, 3.10458067597475, 2.99, 
            2.17, 1.36, 2.11, 1.64, 1.92, 1.84, 2.03, 1.76, 1.91654290808702, 
            2.92, 2.4632673334509, 2.27, 2.03, 2.01, 2.29, 2.17, 
            1.6, 2.09, 1.3220509727929, 1.64, 3.35503408078223, 
            3.83377423463778, 2.16569174982692, 2.26873802250709, 
            3.03, 2.65, 3.15, 2.24, 1.88727243757419, 1.75, 0.165809448475304, 
            0.882026784085989, 0.306871539420407, 1.2, 0.58, 
            0.66, 0.47, 0.510797851773959, 0.221002791659999, 
            0.6, 0.5, -0.00261565299940325, 0.52, 0.8, 1.09163742487333, 
            0.55, 0.34, 0.65, 0.76, 1.39, 1.57, 1.36, 1.28, 0.83, 
            0.58, 0.63, 1.14127242879906, 0.58, 1.31, 1.1, 0.92, 
            0.640556522022377, 0.6, 0.586489218837478, 0.68, 
            0.47, 0.92, 0.66, 0.84, 0.96, 0.49, 0.566333307762354, 
            0.7, 0.61, 0.238294280706027, 0.69, 0.68, 0.76), 
            nfla = c(0.28, 0.326366008182143, 0.3, 0.24, 0.39, 
            0.210730663641691, 0.273866419389725, 0.31, 0.29, 
            0.22, 0.22, 0.26, 0.29, 0.43, 0.29, 0.3, 0.268506442561178, 
            0.279640275610766, 0.32, 0.17, 0.24, 0.25, 0.27, 
            0.26, 0.380646929926011, 0.562038976597756, 0.34, 
            0.27, 0.339740824620725, 0.26, 0.29, 0.23830626856726, 
            0.42, 0.361294078146718, 0.29, 0.26, 0.34007112425626, 
            0.302394994340331, 0.28, 0.2, 0.34, 0.285916803782075, 
            0.259219819210084, 0.32, 0.28, 0.3, 0.27, 0.21, 0.32, 
            0.32, 0.17, 0.22, 0.32, 0.39, 0.21, 0.2, 0.26, 0.259055635685493, 
            0.168729669997768, 0.28, 0.358193176715259, 0.53, 
            0.32, 0.149698307422512, 0.45, 0.37, 0.26, 0.27, 
            0.253419375307108, 0.339908055435293, 0.381452085068936, 
            0.371208990786196, 0.304536174977858, 0.21, 0.13, 
            0.34, 0.24, 0.43, 0.35, 0.43, 0.3, 0.449823271576348, 
            0.4, 0.61, 0.22, 0.3, 0.281824267227589, 0.393082059467285, 
            0.404868201788752, 0.230481010286479, 0.340656488588915, 
            0.5, 0.24709755093342, 0.25, 0.17, 0.32, 0.781337853514113, 
            0.348362467186983, 0.24, 0.496868077992198, 0.248981347314907, 
            0.29, 0.34, 0.37, 0.241383721121072, 0.66, 0.226102135424113, 
            0.48, 0.39, 0.29, 0.24, 0.26, 0.524317003477307, 
            0.42, 0.385679518704013, 0.52, 0.3, 0.312113921918777, 
            0.340834876714805, 0.359149053292621, 0.32, 0.16801111763257, 
            0.43, 0.3, 0.412903449095923, 0.243310122150834, 
            0.39, 0.477391989881734, 0.4, 0.612827488104321, 
            0.451848193610439, 0.448329593711045, 0.27, 0.17, 
            0.6, 0.63, 0.53, 0.63, 0.58, 0.53, 0.53, 0.471918120892838, 
            0.399346090089654, 0.453122203726728, 0.413255031330077, 
            0.43, 0.4, 0.47, 0.45, 0.367335143132598, 0.22, 0.454728385487118, 
            0.26, 0.61, 0.53, 0.623993504221109, 0.411990961191757, 
            0.63, 0.53, 0.52, 0.473587670030225, 0.5, 0.6, 0.4, 
            0.41, 0.574810539303277, 0.43, 0.4, 0.39, 0.519237053128769, 
            0.510873399604155, 0.48, 0.499701547854446, 0.512409898137189, 
            0.43, 0.43, 0.53, 0.468274619804016), pro = c(2.29, 
            1.28, 2.81, 2.18, 1.82, 1.97, 1.98, 1.25, 1.98, 1.85, 
            2.38, 1.57, 1.81, 2.09135673173752, 2.96, 1.46, 2.49041059915335, 
            2.13130840617874, 1.86, 1.66, 2.1, 1.28005281870975, 
            1.69, 1.46, 1.66, 1.92, 1.45, 1.35, 1.76, 1.98, 2.38, 
            1.95, 2.01218238021871, 2.51324672667403, 1.54, 1.86, 
            1.36, 1.44, 1.37, 1.33950823125772, 2.18424977678219, 
            2.60848557080431, 2.13949375424125, 1.38921374989328, 
            2.03, 1.3261008117594, 2.19, 2.14, 2.38, 2.08, 1.80343814103312, 
            2.29, 1.87, 1.68, 1.62, 2.45, 1.72599574751846, 1.66, 
            1.82253895113957, 0.42, 1.61733035635333, 1.77940573721481, 
            0.73, 1.87, 1.03, 0.52883398609767, 2.08803616433743, 
            1.04, 1.04022780110903, 2.5, 1.46, 1.87, 1.03, 3.30878393355802, 
            1.49076962249684, 1.15, 1.46, 1.96706944676082, 2.76, 
            1.95, 1.43, 1.40387647459901, 0.970706108721164, 
            1.62, 1.11831884344399, 1.46, 1.56, 1.34, 1.35, 1.38, 
            1.64, 1.63, 1.95103213967753, 1.99, 1.35, 3.28, 1.56, 
            1.77, 1.95, 2.81, 1.4, 1.35, 1.31, 1.42, 1.22774977531793, 
            1.42, 1.28770066812897, 1.63, 2.08, 2.49, 1.65764115406828, 
            1.29721712771851, 0.609531386144873, 1.42731024399141, 
            1.04, 1.44879480156929, 0.930567640427243, 1.61, 
            0.83, 1.87, 1.83, 2.57012490317235, 1.71, 1.39424095456086, 
            2.91, 1.35, 1.77, 1.98446949999496, 1.9, 1.62180953663698, 
            0.94, 1.73607391155764, 0.83, 0.84, 1.25, 0.94, 0.8, 
            1.1, 0.88, 1.55877654361509, 0.75, 0.426085342892536, 
            0.55, 1.4761831782504, 1.14, 1.3, 0.68, 0.86, 1.25, 
            1.14, 1.25, 1.26, 1.56, 0.355302062436978, 0.993021922417075, 
            1.55, 1.56, 1.14, 2.7, 2.29, 1.04, 1.25347060030942, 
            1.60491657676208, 0.94, 1.01933016818372, 1.15, 1.46, 
            0.97, 1.54, 1.11, 0.73, 0.64, 1.24, 1.06, 1.41, 1.35, 
            1.46, 1.35), col = c(5.64, 4.38, 5.68, 7.8, 5.29513016136556, 
            6.75, 4.38302154428785, 5.05, 5.74943906367376, 7.22, 
            5.75, 3.76235111176515, 5.6, 6.20610321066349, 7.5, 
            5.44238639965208, 7.66200455281397, 4.87585883573943, 
            8.7, 5.1, 5.65, 4.5, 2.73311631091517, 3.93, 3.52, 
            3.58, 4.8, 3.95, 4.5, 4.7, 5.7, 6.11453867816125, 
            5.49197813213303, 5.4, 4.2, 5.1, 4.6, 4.25, 3.7, 
            5.1, 6.13, 4.28, 5.43, 4.36, 5.04, 5.24, 4.9, 6.1, 
            3.48977700322169, 8.9, 7.2, 7.87108460216918, 5.69446535114455, 
            6.3, 5.85, 6.25, 5.30099458500423, 4.74826068107638, 
            6.8, 4.33164068725733, 3.27, 5.75, 3.8, 4.45, 2.95, 
            4.6, 5.3, 4.68, 3.17, 2.85, 10.3483302077849, 3.38, 
            3.74, 5.46489804343251, 3.21, 3.8, 4.6, 4.37145079839948, 
            6.65338950679006, 0.884644673681495, 2.5, 3.9, 0.921036653513473, 
            4.8, 3.05, 2.62, 2.45, 2.6, 2.8, 1.74, 2.4, 3.6, 
            3.05, 2.15, 3.25, 2.6, 5.01714379469649, 2.9, 3.34904794466245, 
            2.1822990797623, 3.26878679332223, 3.66396248011783, 
            2.8, 3.93116140561533, 2.94, 2.7, 3.4, 3.3, 2.7, 
            2.08646289237072, 2.9, 3.91192864636398, 3.3177370902347, 
            3.08, 2.9, 1.9, 1.45116030425533, 3.12336712502313, 
            3.4, 1.28, 3.25, 7.14289281060455, 2.64499399066081, 
            5.12626596412829, 2.8, 3.75174120943015, 3.94, 2.30497608364642, 
            2.12, 1.24851280140987, 6.51539280591084, 7.56770708250472, 
            7.448469071824, 5, 5.45, 7.1, 6.06521945720992, 5, 
            5.7, 4.92, 6.67207794371956, 6.56266238423166, 4.35, 
            4.4, 8.21, 4, 4.9, 6.48658157343903, 8.42, 9.4, 0.867052021168611, 
            9.15932322719345, 7.1, 10.52, 7.6, 7.9, 10.1328237861966, 
            7.5, 13, 8.26654795986522, 7.65, 9.20048558278853, 
            5.58, 8.17473517299385, 10.3610110354049, 6.62, 10.68, 
            10.26, 7.6787321052442, 8.5, 7.9392381032431, 8.25405326386627, 
            9.7, 7.7, 7.3, 10.0593442987959, 9.3, 9.2), hue = c(1.04, 
            1.05, 1.14813529824281, 0.86, 0.915962159965684, 
            1.05, 1.02, 1.11229500772457, 1.08, 1.32267987434203, 
            1.25, 1.19540339512324, 1.15, 1.25, 1.2, 1.28, 1.00275060200244, 
            1.13, 1.23, 1.00260175251669, 1.09, 0.979421034917946, 
            1.11, 1.09, 1.12, 1.13, 1.1728722628701, 1.02, 1.32205581506189, 
            1.04, 1.19, 1.09, 1.25676705299071, 1.25, 1.1, 1.1926655709037, 
            1.14222793610826, 1.11420287629923, 1.18, 0.881966593939961, 
            0.95, 0.91, 0.88, 0.82, 1.53583023232934, 0.87, 1.24482538066046, 
            1.11559777682776, 1.07, 1.12, 1.12, 0.944033939525624, 
            1.12522101468962, 1.0253834827041, 1.00579854115452, 
            0.970729450224899, 0.94, 1.07, 0.89, 0.988261734699048, 
            1.25, 0.98, 1.23, 1.22, 0.93353442505973, 1.1582466880826, 
            1.03623199181794, 1.12, 1.02, 1.28, 0.906, 1.36, 
            0.98, 1.31, 0.980595119005944, 1.23, 1.19, 0.96, 
            1.06, 1.19, 1.38, 1.16993279816553, 1.31, 0.84, 1.1934406354553, 
            1.23, 1.01491706665237, 1.15927393359845, 1.17542353326267, 
            1.07, 1.08, 1.05, 0.755288891363011, 1.15, 1.09789984514863, 
            1.21042096678112, 0.728030824529886, 1.08151228053025, 
            1.16770868614659, 1.42, 1.27, 1.01263897284, 0.8, 
            0.944258826066885, 1.04, 0.997433662843542, 1, 0.947890965856259, 
            0.86, 1.32666168767975, 0.876171513074265, 0.9, 0.995620225374877, 
            1.1, 0.93, 1.71, 0.95, 0.962447907501517, 0.7, 1.08995098001645, 
            0.8, 1.25108714268948, 0.92, 0.73, 1.31505257682823, 
            0.86, 0.69, 0.97, 0.89, 0.79, 0.706675022825434, 
            0.74, 0.66, 0.78, 0.75, 0.73, 0.75, 0.82, 0.81, 0.924501841235281, 
            0.878693273941907, 0.567602013179991, 0.89, 1.03447860889976, 
            0.65, 0.6, 0.533681179712274, 0.54, 0.55, 0.698731374865048, 
            1.25571467370868, 0.48, 0.61, 0.56, 0.853944568668634, 
            0.6, 0.57, 0.67, 0.57, 0.792685969634878, 0.56, 0.96, 
            0.87, 0.68, 0.7, 0.78, 0.85, 0.422390055780102, 0.74, 
            0.67, 0.602040822327014, 0.590096371536645, 0.693270443993927, 
            0.64, 0.7, 0.59, 0.6, 0.61), ratio = c(3.92, 3.4, 
            3.17, 3.57801928137772, 2.93, 3.42798944629768, 3.58, 
            3.58, 2.85, 3.55, 3.17, 2.82, 2.9, 3.19565932779182, 
            3, 3.05833344837634, 2.65, 2.83187401238883, 2.82, 
            3.36, 3.71, 3.52, 4, 3.63, 2.97068009039322, 3.2, 
            2.85694203115169, 2.77, 3.78532539696513, 3.30324227534004, 
            2.71, 2.88, 2.87, 3, 2.87, 3.38608324594805, 2.88184505446612, 
            3.48013385480418, 2.69, 3.03480014145067, 3.38, 3, 
            3.24371991160888, 3, 3.35, 3.33, 3.22659160988914, 
            3.33, 2.75, 3.1, 3.11448471302832, 3.64817452330805, 
            3.26, 2.93, 3.2, 3.03, 3.86775999328059, 2.84, 2.87, 
            1.82, 3.39047948205068, 1.59, 2.46, 2.87, 2.83764059970657, 
            2.3, 3.18, 3.48, 3.23338291966355, 2.71454258207013, 
            1.82, 3.27265237300883, 2.96509507671815, 3.5, 3.13, 
            2.14, 2.48, 2.52, 2.64830818051698, 3.30561764993837, 
            3.51450568446089, 3.14, 2.72, 2.01, 3.27891210700163, 
            3.30615616640172, 2.61200085943422, 3.21, 2.75, 3.21, 
            2.19581504815257, 2.65, 3.17563688718947, 3.3, 2.86365581465094, 
            2.63, 1.24263729937694, 2.74, 3.01633871231629, 2.83, 
            2.96, 2.77, 3.38, 2.54384404046752, 3.57, 3.3, 3.17, 
            2.42, 3.02, 3.26, 2.81, 2.78, 2.5, 3.65160946337101, 
            3.19, 2.87, 3.33, 2.96, 2.12, 3.05, 3.39, 3.69, 3.12, 
            2.64863108821254, 3.64, 3.35246880966881, 2.84, 2.54584270742429, 
            2.78, 2.35082602741055, 1.7093369317992, 1.42, 1.36, 
            1.29, 1.51, 1.58, 1.27, 1.69, 1.82, 2.15, 1.74620731715331, 
            1.8313801488266, 3.1178003325872, 2.05, 2, 2.39465798890554, 
            1.33, 1.92901682277227, 1.62, 1.33, 2.76646471442622, 
            1.47, 1.33, 1.51, 1.55, 1.48, 1.64, 1.73, 1.96, 1.78, 
            1.58, 2.10894943229417, 2.53465582655, 1.75, 1.68, 
            1.75, 1.60167246873664, 1.25717487071263, 2.36241190674578, 
            1.71968916979206, 2.11366763074655, 1.38476507520418, 
            1.92458896802243, 1.10841493195362, 1.2676164474932, 
            1.56, 1.62, 1.6), prol = c(1065, 1050, 993.993243311041, 
            1358.05077215623, 735, 1450, 1290, 1295, 1045, 1045, 
            810.447804100404, 1280, 1320, 1336.09526702893, 1547, 
            1310, 1280, 1384.62241498804, 1680, 845, 1241.81146879834, 
            770, 1035, 1015, 845, 690.81334077829, 1195, 1285, 
            915, 1035, 1285, 1421.0887367999, 1118.56254212099, 
            1235, 447.925198574251, 920, 880, 738.98638223758, 
            811.40136467268, 760, 795, 1035, 1095, 680, 885, 
            1080, 1065, 985, 1060, 1454.8811755703, 1150, 1265, 
            1190, 1210.07729890234, 1060, 1120, 970, 1270, 1285, 
            581.894080132224, 631.574083586144, 771.727906632191, 
            630, 382.333014423881, 459.981518398508, 991.13673309516, 
            502, 417.767650616532, 529.41729967538, 718, 870, 
            410, 472, 625.331192550568, 886, 571.739959911361, 
            392, 434.457885842489, 858.829509573298, 463, 278, 
            714, -108.212398064739, 515, 364.847952036899, 352.944598622921, 
            588.108299395751, 562, 680, 625, 480, 450, 495, 591.451567938987, 
            345, 479.141540267618, 625, 428, 660, 406, 378.13146020109, 
            562, 438, 415, 672, 496.410206557734, 510, 480.589040135172, 
            312, 680, 562, 325, 762.7675530051, 434, 385, 363.311585162872, 
            526.104767435531, 813.110584582651, 372, 564, 458.182627181166, 
            1477.24144564778, 365, 380, 560.163945693582, 378, 
            1021.95284141404, 466, 524.717778551702, 580, 412.347078206073, 
            640.607525831818, 560, 600, 381.402088623608, 695, 
            720, 515, 580, 590, 600, 300.090732625459, 520, 550, 
            676.961722996909, 830, 415, 527.64794993247, 867.477616335355, 
            550, 500, 480, 425, 675, 640, 688.978337765769, 480, 
            880, 660, 1341.76153681998, 634.43267546538, 680, 
            570, 771.473123369225, 615, 770.970365526084, 695, 
            685, 626.764407714562, 874.042177369883, 510, 470, 
            1198.86263360183, 740, 1020.19779765373, 835, 840, 
            619.766484850856)), row.names = c(NA, -178L), class = "data.frame"), 
        structure(list(alco = c(14.23, 13.2, 13.16, 15.1147048328883, 
        13.24, 13.7085732256644, 14.39, 14.06, 13.7796118164295, 
        13.86, 13.6265846893057, 14.3743906070549, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 13.994522919713, 13.64, 13.5314706146006, 
        12.93, 13.71, 12.85, 13.5, 12.0951288012884, 13.39, 13.3, 
        13.87, 14.1455465418483, 13.73, 13.58, 13.68, 13.76, 
        13.1230475292659, 12.9787985870047, 12.7418727036121, 
        13.05, 13.1451304905681, 14.22, 13.56, 13.41, 13.88, 
        13.24, 13.05, 14.21, 13.2469894354119, 13.9, 14.0699201182589, 
        13.401893374171, 13.05, 12.9192485478977, 13.82, 13.5866532553472, 
        13.74, 13.9993969722987, 14.22, 13.29, 14.1065843533234, 
        12.37, 12.33, 12.64, 11.3841360506332, 12.37, 12.17, 
        12.37, 12.5728162035451, 12.6194512984398, 13.34, 12.21, 
        12.0376073475849, 13.86, 13.0673750056141, 12.99, 11.96, 
        11.66, 12.2780191168264, 11.84, 12.4416539373329, 12.7, 
        12, 12.72, 12.08, 13.05, 11.84, 11.941064784344, 12.16, 
        12.8278890382871, 12.7013948290782, 12.08, 12.08, 12, 
        12.69, 11.7170607148526, 11.62, 12.0519689766633, 11.81, 
        12.29, 13.1562545815757, 12.29, 12.08, 12.6, 12.767128735903, 
        12.2668393306717, 12.51, 12.7442658815158, 11.9847348877381, 
        12.72, 12.22, 11.61, 11.6350648285122, 12.6474454273512, 
        11.76, 12.4608289726448, 12.08, 11.03, 11.82, 12.1068345601074, 
        11.3440128419383, 12.243879263751, 11.45, 14.830870058301, 
        12.42, 13.05, 11.87, 12.07, 10.5745889937412, 11.79, 
        12.7589918643445, 12.04, 12.86, 12.88, 12.3560846458888, 
        12.7, 12.51, 13.1487940269449, 12.25, 12.53, 12.0502244182813, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 13.1009909199262, 
        11.7364858000799, 12.87, 13.32, 11.8305082841918, 11.6979130685148, 
        12.79, 11.9372352075193, 13.23, 13.0855076360521, 13.17, 
        13.84, 12.45, 14.34, 13.48, 13.3378179565105, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.3453508705329, 13.375305030557, 
        13.58, 13.4, 12.0381612035326, 12.77, 14.16, 13.0982904295089, 
        13.4, 13.27, 13.17, 14.13), malic = c(1.71, 3.76086652385337, 
        1.40811372281553, 1.95, 3.00470433015368, 1.84157013291854, 
        1.87, 2.15, 2.09710905799731, 1.35, 2.16, 1.48, 0.626466896040117, 
        1.73, 1.87, 1.96961134681209, 1.92, 0.0194067099550455, 
        2.31697442382695, 3.1, 1.63, 3.8, 3.00913445191498, 1.6, 
        1.81, 2.15831216564864, 1.64173369719895, 1.93728220175703, 
        1.9, 2.90008372822578, 1.5, 1.62128831844254, 1.83, 1.53, 
        2.29981801289354, 1.64594723164956, 0.304959881485175, 
        1.65, 1.5, 1.92524678144303, 1.71, 2.60023909451812, 
        4.66202849821493, 3.98, 0.502511842842392, 4.04, 1.35579764676497, 
        1.68, -0.0762913607539146, -1.32383303564242, -1.91715968184042, 
        1.65, -0.102484603522324, 1.9, 1.67, 1.03228376981091, 
        1.7, 1.97, 1.98891131374691, 0.94, 1.50986501280853, 
        1.36, 1.25, 0.756877158314622, 3.85336230178445, 1.21, 
        1.01, 1.17, 2.83589035376893, 3.88126352556205, 2.65248663292864, 
        1.51, 1.66, 1.67, 1.09, 1.88, 0.9, 3.99686075605434, 
        0.99, 3.87, 2.13882894692161, 0.232123493907362, 1.13, 
        4.11384605078297, 1.62597821531731, 0.98, 1.61, 1.67, 
        2.28775290539562, 0.747640586573447, 0.737176064955549, 
        1.51, 1.04689714086895, 0.391235206296873, 1.55937743938651, 
        2.14086561358778, 1.85994144057377, 1.41, 2.06105015912227, 
        0.609570251769997, 2.08, 1.52177267243026, 2.68738267446035, 
        1.72, 1.01129914403678, 2.55, 1.73, 1.2199829643131, 
        1.29, -0.213834558345201, 3.74, 2.06396043303515, 2.68, 
        0.74, 1.39, 0.394860113672418, 2.18425037784523, 1.61, 
        3.43, 2.7133704427228, 2.4, 2.05, 4.43, 1.64601559293896, 
        4.31, 2.16, 1.53, 2.13, 4.02233099940429, 4.3, 1.35, 
        2.99, 2.31, 3.55, 1.24, 3.93964336308711, 3.25402271369447, 
        5.51, 3.59, 4.36874344048364, 2.81, 2.56, 5.22509852895497, 
        1.91480148809885, 3.88, 5.79499111490362, 5.04, 4.61, 
        3.24, 3.99968661056402, 3.12, 2.67, 1.9, 2.580865763735, 
        1.29, 5.19, 4.12, 3.03, 1.68, 1.67, 3.83, 3.05457315041611, 
        3.75174421650033, 3.45, 2.76, 4.12229222453641, 3.24251710417399, 
        3.37, 2.58, 4.6, 4.6069309296191, 2.39, 4.53324198423666, 
        5.65, 3.91, 5.34070668064587, 2.59, 4.1), ash = c(2.43, 
        2.14, 2.67, 2.5, 2.91392166765321, 2.2571955915846, 2.45, 
        2.61, 2.47339638019576, 2.41408632554452, 2.3, 2.44641440795465, 
        2.41, 2.39, 2.38, 2.7, 2.72, 2.62, 2.48, 2.56, 2.14932941675561, 
        2.65, 3.05757710127753, 2.52, 2.61, 3.22, 2.58026489030039, 
        2.54188517821418, 2.54895626155787, 2.09793460192557, 
        2.62953421019109, 2.36, 2.63423127192324, 2.7, 2.65, 
        2.42219021643821, 2.07148007935726, 2.55, 2.1, 2.51, 
        2.16869561100011, 2.12, 2.59, 2.29, 2.1, 2.44, 2.04077589487559, 
        2.12, 2.4, 2.27, 2.04, 2.35519290251315, 2.79810255546447, 
        2.08917762005446, 2.34319702947898, 2.28912760912185, 
        2.3, 2.68, 2.5, 1.36, 2.33989534927995, 2.12400397324989, 
        1.92, 2.16, 2.53, 2.56, 2.06844848717624, 1.92, 2.13432262134214, 
        1.75, 2.21, 2.67, 2.24, 2.6, 2.3, 1.56125406536588, 1.71, 
        2.53002470990854, 2.03650431777351, 2.4, 2, 2.54514593712819, 
        2.51, 2.32, 2.58, 2.24, 2.2655767068586, 2.62, 2.46690513862146, 
        2.3, 2.32, 2.00050575551064, 2.26, 2.22, 2.28, 2.2, 2.74, 
        1.98, 2.11448681008585, 1.97418635327647, 1.7, 2.32106536317773, 
        2.46, 1.88, 2.39521317797345, 2.27, 2.12, 2.28, 1.94, 
        2.7, 1.82, 2.17, 2.92, 2.5, 2.5, 2.2, 1.99, 2.41922492893651, 
        2.08163175260919, 2, 2.42, 3.23, 2.35696043088541, 2.13, 
        2.39, 2.17, 2.29, 2.12080401182165, 2.15616188464534, 
        2.38, 2.65350059332508, 2.4, 2.4, 2.36, 2.25, 2.80467628893931, 
        2.54, 2.64, 2.19, 2.61, 2.7, 2.35, 2.72, 2.35, 2.2, 2.61010422753823, 
        2.24027354833935, 2.48, 2.07642947608683, 2.36, 2.62, 
        2.48, 2.43461506673815, 2.28, 2.55183870468957, 2.32, 
        2.38, 1.94217427750789, 2.67682428259351, 2.64, 2.38, 
        2.54, 2.37492819184064, 2.35, 2.3, 2.54790567838124, 
        2.6, 2.2775673978962, 2.69, 2.57330916644564, 2.32, 2.28, 
        2.66445042273397, 2.58371367936028, 2.48, 2.26, 2.37, 
        2.74), alca = c(15.6, 11.190656480152, 19.5733026158349, 
        16.8, 21, 15.2, 12.727002312024, 17.6, 14, 16, 20.7416638957188, 
        16.8, 16, 11.4, 20.0353432186347, 15.5714420077126, 16.2093111566287, 
        20, 15.5540882923009, 15.2, 16, 18.6, 19.831246478084, 
        17.8, 20, 21.4837040750183, 16.1, 17, 22.1353294532737, 
        17.3692989410386, 16.5356228710535, 19.1, 20.7046530767742, 
        15.7703573446178, 19, 20.5, 15.5, 23.3491731859625, 15.5, 
        13.2, 16.2, 18.8, 18.1837439540974, 28.7657745415895, 
        17, 14.7528118780971, 16, 12.945656012851, 18.8, 17.4, 
        12.4, 17.3739782675075, 13.4762075109341, 9.6045898129079, 
        14.0797270254182, 20.5, 16.3, 16.8, 16.2198077290869, 
        10.6, 19.8054359968054, 16.8, 18, 19, 19, 18.1, 20.7320329337011, 
        19.6, 18.9011347996155, 16.8, 20.4, 18.7512860222019, 
        24, 30, 21, 19.7272123902334, 16, 18, 14.8, 24.5017252589362, 
        19, 13.8635145620702, 24, 22.5, 18, 18, 22.8, 21.8140843566768, 
        21.6, 19.1882082893764, 21.0787686608754, 22, 23.9826254921768, 
        18, 18, 19, 21.5, 16, 20.480891329042, 18, 17.5, 18.5, 
        21, 19.5, 20.5, 22, 19, 22.5, 19, 20, 19.5, 21.0583515282048, 
        21.7406388278769, 20.8688963925013, 13.9651134974947, 
        21.5, 20.8, 17.2925800978114, 16, 22.5246522283552, 15.521844608163, 
        23.9567252518099, 18.5982358357287, 21.5, 21, 21, 21.5, 
        28.5, 22.2137226718986, 22, 22.6017100760389, 20, 24, 
        21.5, 19.8556913597636, 22.1308458842382, 18.9764322686154, 
        25, 19.5, 24.1444817967366, 21, 17.9548887535371, 23.5, 
        20, 24.8909948465615, 21, 20, 19.9516351230918, 23.1416516173682, 
        21.5, 24, 22, 20.8697257303762, 18.5, 20, 22, 19.5, 13.0514167546902, 
        25, 24.2922114856242, 18.9845351763917, 25.0022979962847, 
        23.7995576435898, 18.5, 19.4557639361563, 22.5, 23, 19.5, 
        24.5, 25, 21.7820306455442, 19.5, 20.780076587443, 20.5, 
        23, 20, 20.1528996660935, 24.5), mg = c(127, 100, 101, 
        113, 96.6042035623153, 112, 96, 121, 97, 100.936207314341, 
        108.238591866966, 121.453201785423, 89, 91, 102, 112, 
        120, 116.321606878545, 126.964198670062, 116, 126, 102, 
        118.665087032184, 95, 101.231691904827, 124, 93, 111.46413931, 
        107, 96, 101, 106, 104, 98.0581802917312, 110, 138.872695036446, 
        110, 85.3155565055864, 102.779315929593, 98.5273960975379, 
        117, 109.416581779098, 101, 104.754457079263, 107, 111, 
        102, 101, 97.3893740271144, 108, 92, 112.75144950252, 
        111, 115, 93.2358609818869, 132.47556916959, 118, 102, 
        108, 88, 101, 100, 94, 87, 104, 98, 78, 78, 80.326516326836, 
        151, 105.768455634361, 86, 87, 139, 101, 101.989176561938, 
        75.2991103129205, 112, 93.8876355473293, 105.537338107256, 
        86, 102.860847007087, 78, 85, 99.6839346045503, 99, 90, 
        88, 84, 70, 81, 108.527276707785, 79.4973279420582, 88, 
        95.0293686049133, 102.576884202766, 134, 85, 88, 93.8164009980775, 
        97, 88, 102.656208849875, 86, 85, 90, 80, 84, 92, 94, 
        107, 88, 103, 88, 84, 85, 86, 86.9333283424657, 121.949931761454, 
        87, 96, 133.67147711332, 102, 86, 132.98215624875, 85, 
        128.952450364187, 92, 88, 80, 122, 79.4574993449277, 
        98, 106, 77.408427077