% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compare_documents_dtm.r
\name{compare_documents_dtm}
\alias{compare_documents_dtm}
\title{Compare the documents in a dtm with a sliding window over time}
\usage{
compare_documents_dtm(dtm, meta = NULL, doc_col = "doc_id",
  date_col = NULL, meta_cols = NULL, hour_window = 24,
  measure = c("cosine", "overlap_pct"), min_similarity = 0,
  n_topsim = NULL, only_from = NULL, only_to = NULL,
  return_zeros = FALSE, only_complete_window = FALSE, verbose = T)
}
\arguments{
\item{dtm}{A document-term matrix in the tm \link[tm]{DocumentTermMatrix} class. It is recommended to weight the DTM beforehand, for instance using \link[tm]{weightTfIdf}.}

\item{meta}{A data.frame where rows are documents and columns are document meta information.
Should at least contain 2 columns: the document name/id and date.
The name/id column should match the document names/ids of the edgelist, and its label is specified in the `doc_col` argument.
The date column should be intepretable with \link[base]{as.POSIXct}, and its label is specified in the `date_col` argument.}

\item{doc_col}{The label for the document name/id column in the `meta` data.frame. Default is "document_id"}

\item{date_col}{The name of the document date column in the `meta` data.frame. If given, only documents within the given date window (specified in hour_window) will be compared}

\item{meta_cols}{The names of other columsn in the `meta` data.frame. If given, only documents with identical values in these columns will be compared.}

\item{hour_window}{A vector of eighter length 1 or 2. If length is 1, the same value is used for the left and right side of the window. If length is 2, the first and second value determine the left and right side. For example, the value 12 will compare each document to all documents between the previous and next 12 hours, and c(-10, 36) will compare each document to all documents between the previous 10 and the next 36 hours.}

\item{measure}{the measure that should be used to calculate similarity/distance/adjacency. Currently supports the symmetrical measure "cosine" (cosine similarity), and the assymetrical measures "overlap_pct" (percentage of term scores in the document that also occur in the other document).}

\item{min_similarity}{a threshold for similarity. lower values are deleted. Set to 0.1 by default.}

\item{n_topsim}{An alternative or additional sort of threshold for similarity. Only keep the [n_topsim] highest similarity scores for x. Can return more than [n_topsim] similarity scores in the case of duplicate similarities.}

\item{only_from}{A vector with names/ids of documents (dtm rownames), or a logical vector that matches the rows of the dtm. Use to compare only these documents to other documents.}

\item{only_to}{A vector with names/ids of documents (dtm rownames), or a logical vector that matches the rows of the dtm. Use to compare other documents to only these documents.}

\item{return_zeros}{If true, all comparison results are returned, including those with zero similarity (rarely usefull and problematic with large data)}

\item{only_complete_window}{if True, only compare articles (x) of which a full window of reference articles (y) is available. Thus, for the first and last [window.size] days, there will be no results for x.}

\item{verbose}{If TRUE, report progress}
}
\value{
A network/graph in the \link[igraph]{igraph} class
}
\description{
Given a document-term matrix (DTM) and corresponding document meta data, calculates the document similarities over time using with a sliding window.
}
\details{
The meta data.frame should have a column containing document id's that match the rownames of the DTM (i.e. document names) and should have a column indicating the publication time.
By default these columns should be labeled "document_id" and "date", but the column labels can also be set using the `doc_col` and `date_col` parameters.
Any other columns will automatically be included as document meta information in the output.

The calculation of document similarity is performed using a vector space model approach.
Inner-product based similarity measures are used, such as cosine similarity.
It is recommended to weight the DTM beforehand, for instance using Term frequency-inverse document frequency (tf.idf)
}
\examples{
\dontrun{
data(dtm)
data(meta)

dtm = tm::weightTfIdf(dtm)
g = compare_documents_dtm(dtm, meta, hour_window = c(0.1, 36))

vcount(g) # number of documents, or vertices
ecount(g) # number of document pairs, or edges

head(igraph::get.data.frame(g, 'vertices'))
head(igraph::get.data.frame(g, 'edges'))
}
}
