#' Visualize Correlation Matrix
#'
#' @author Taiyun Wei
#' @email weitaiyun@gmail.com
#  last modified by Taiyun Wei 2011-10-15
corrplot <- function(corr,
		method = c("circle", "square", "ellipse", "number", "shade", "color", "pie"),
		type = c("full", "lower", "upper"), add=FALSE, 
		col = NULL, assign.col = c("-1to1","min2max","0to1"),
		bg = "white", title = "",
		diag = TRUE, outline = FALSE, mar = c(0,0,0,0),
		addgrid.col = "gray", addnum.col= NULL,

		order = c("original", "PCA", "hclust", "alphabet"),
		hclust.method = c("complete", "ward", "single", "average",
							"mcquitty", "median", "centroid"),
		addrect = NULL, rect.col="black", rect.lwd = 2,

		addtextlabel = c("lt","ld","td","d", "no"), tl.cex = 1,
		tl.col = "red", tl.offset = 0.4,

		addcolorlabel = c("right", "bottom","no"), cl.range=c( "-1to1","min2max","0to1"),
		cl.length=11, cl.cex =0.8, cl.ratio = 0.15, cl.align.text="c",cl.offset=0.5,

		addshade = c("negtive", "positive", "all"),
		shade.lwd = 1, shade.col = "white",

		p.mat = NULL, sig.level = 0.05,
		insig = c("pch","blank", "no"),
		pch = 4, pch.col = "black", pch.cex = 3,
		
		plotCI = c("no","square", "circle", "rect"),
		lowCI.mat = NULL, uppCI.mat = NULL)
{

	if(!is.matrix(corr) )
		stop("Need a matrix!")
		
	if(min(corr) < -1 - .Machine$double.eps|| max(corr) > 1 + .Machine$double.eps)
		stop("The number in matrix should be in [-1, 1]!")
		
	if(is.null(col))
		col <- colorRampPalette(c("#67001F", "#B2182B", "#D6604D", "#F4A582", "#FDDBC7",
				"#FFFFFF", "#D1E5F0", "#92C5DE", "#4393C3", "#2166AC", "#053061"))(200)
	n <- nrow(corr)
	m <- ncol(corr)
	min.nm <- min(n,m)

	order <- match.arg(order)
	hclust.method <- match.arg(hclust.method)
	NewOrder <- function(corr, order, ...){
		if((!n==m)&(!order=="original")){
			stop("The matrix must be squre if reorder variables!")
		}

		## reorder the variables using principal component analysis
		if (order == "PCA") {
			x.eigen <- eigen(corr)$vectors[, 1:2]
			e1 <- x.eigen[, 1]
			e2 <- x.eigen[, 2]
			alpha <- ifelse(e1 > 0, atan(e2/e1), atan(e2/e1) + pi)
			ord <- order(alpha)
		}

		## reorder the variables in alphabet ordering
		if(order =="alphabet"){
			ord <- sort(rownames(corr))
		}

		## reorder the variables using hclhust
		if(order == "hclust"){
			ord <- order.dendrogram(as.dendrogram(hclust(as.dist(1-corr),
			method = hclust.method, ...)))
		}

		return(ord)
	}

	ord <- NA
	if(!order=="original"){
		ord <- NewOrder(corr, order=order)
		corr <- corr[ord,ord]
	}
	## set up variable names
	if (is.null(rownames(corr))) rownames(corr) <- 1:n
	if (is.null(colnames(corr))) colnames(corr) <- 1:m

	method <- match.arg(method)
	type <- match.arg(type)
	plotCI <- match.arg(plotCI)

	getPos.Dat <- function(mat){
		x <- matrix(1:n*m, n, m)
		tmp <- mat
		if(type=="upper")  tmp[row(x)>col(x)] <- Inf
		if(type=="lower")  tmp[row(x)<col(x)] <- Inf
		if(type=="full")   tmp <- tmp
		if(!diag)          diag(tmp) <- Inf

		Dat <- tmp[is.finite(tmp)]
		ind  <- which(is.finite(tmp),arr.ind = TRUE)
		Pos <- ind
		Pos[,1] <-  ind[,2]
		Pos[,2] <- -ind[,1] + 1 + n
		return(list(Pos, Dat))
	}

	Pos  <- getPos.Dat(corr)[[1]]
	n2 <- max(Pos[,2]); n1 <- min(Pos[,2])
	nn <- n2 -n1
	newrownames <- as.character(rownames(corr)[(n+1-n2):(n+1-n1)])
	m2 <- max(Pos[,1]); m1 <- min(Pos[,1])
	mm <- m2 -m1
	newcolnames <- as.character(colnames(corr)[m1:m2])
	DAT <- getPos.Dat(corr)[[2]]
	len.DAT <- length(DAT)
	assign.col <- match.arg(assign.col)
	if(assign.col=="-1to1")
		col.fill <- col[ceiling((DAT+1)*(length(col)-1)/2) + 1]
	if(assign.col=="min2max"){
		newcorr <- (DAT-min(DAT))/(diff(range(DAT)))*2-1
		col.fill <- col[ceiling((newcorr+1)*(length(col)-1)/2) + 1]
	}
	if(assign.col=="0to1"){
		if(any(DAT<0)) 
			stop("There are negative numbers!")
		newcorr <- DAT*2-1
		col.fill <- col[ceiling((newcorr+1)*(length(col)-1)/2) + 1]
	}	
	if(outline)
		col.border <- "black"
	if(!outline)
		col.border <- col.fill

	## calculate label-text width approximately
	par(mar = mar, bg = "white")
	if(!add) plot.new()
	plot.window(c(m1-0.5, m2+0.5), c(n1-0.5, n2+0.5), asp = 1)
	xlabwidth <- max(strwidth(newrownames, cex = tl.cex))
	ylabwidth <- max(strwidth(newcolnames, cex = tl.cex))
	S1 <- max(nn,mm)**2

	addtextlabel <- match.arg(addtextlabel)
	if(addtextlabel=="no"|addtextlabel=="d") xlabwidth <- xlabwidth <- 0
	## set up an empty plot with the appropriate dimensions

	addcolorlabel <- match.arg(addcolorlabel)
	cl.range <- match.arg(cl.range) 
	## color label
	if(addcolorlabel=="no"){
		plot.window(c(-xlabwidth + m1 - 0.5, m2 + 0.5),
					c(n1 - 0.5, n2 + 0.5 + ylabwidth),
					asp = 1, xlab="", ylab="")
		S2 <- (mm + xlabwidth)*(mm + ylabwidth)##area of figure
	}
	if(addcolorlabel=="right"){
		plot.window(c(-xlabwidth + m1 - 0.5, m2 + 0.5 + mm*cl.ratio),
					c(n1 - 0.5, n2 + 0.5 + ylabwidth),
					asp = 1, xlab="", ylab="")
		S2 <- (mm + xlabwidth+ mm*cl.ratio)*(nn + ylabwidth)
	}
	if(addcolorlabel=="bottom"){
		plot.window(c(-xlabwidth + m1 - 0.5, m2 + 0.5),
					c(n1 - 0.5 - nn*cl.ratio, n2 + 0.5 + ylabwidth),
					asp = 1, xlab="", ylab="")
		S2 <- (mm + xlabwidth+ mm*cl.ratio)*(nn + ylabwidth)
	}
	## background color
	symbols(Pos, add = TRUE, inches = FALSE,
			squares = rep(1, len.DAT), bg = bg, fg = bg)
	

	## circle
	if(method=="circle"&plotCI=="no"){
    	symbols(Pos, add = TRUE,  inches = FALSE, bg = col.fill,
			circles = 0.9*abs(DAT)^0.5/2, fg = col.border)
	}

	## ellipse
	if(method=="ellipse"&plotCI=="no"){
		ell.dat <- function(rho, length = 99){
    		k <- seq(0, 2*pi, length=length)
    		x <- cos(k + acos(rho)/2)/2
    		y <- cos(k - acos(rho)/2)/2
    		return(cbind(rbind(x,y),c(NA, NA)))
    	}
    	ELL.dat <- lapply(DAT, ell.dat)
		ELL.dat2 <- 0.85*matrix(unlist(ELL.dat),ncol=2,byrow=TRUE)
		ELL.dat2 <- ELL.dat2  + Pos[rep(1:length(DAT),each=100),]
		polygon(ELL.dat2, border=col.border, col=col.fill)
	}

	## number
	if(method=="number"&plotCI=="no"){
		text(Pos[,1], Pos[,2], font = 2, round(100 * DAT), col = col.fill)
	}

	## pie
	if(method=="pie"&plotCI=="no"){
    	symbols(Pos, add = TRUE, inches = FALSE, circles = rep(0.5, len.DAT)*0.85)
		pie.dat <- function(theta, length = 100){
			k <- seq(pi/2, pi/2 - theta, length = 0.5*length*abs(theta)/pi)
			x <- c(0, cos(k)/2, 0)
			y <- c(0, sin(k)/2, 0)
			return(cbind(rbind(x,y),c(NA, NA)))
    	}
    	PIE.dat <- lapply(DAT*2*pi, pie.dat)
    	len.pie <- unlist(lapply(PIE.dat, length))/2
		PIE.dat2 <- 0.85*matrix(unlist(PIE.dat),ncol=2,byrow=TRUE)
		PIE.dat2 <- PIE.dat2  + Pos[rep(1:length(DAT),len.pie),]
    	polygon(PIE.dat2, border = "black", col = col.fill)
	}

	## shade
	if(method=="shade"&plotCI=="no"){
    	addshade <- match.arg(addshade)
    	symbols(Pos, add = TRUE, inches = FALSE, squares = rep(1, len.DAT),
				bg = col.fill, fg = addgrid.col)
        shade.dat <- function(w){
			x <- w[1];  y <- w[2];  rho <- w[3]
			x1 <- x - 0.5
			x2 <- x + 0.5
			y1 <- y - 0.5
			y2 <- y + 0.5
			dat <- NA

			if((addshade=="positive"||addshade=="all")&rho>0){
				dat <- cbind(c(x1, x1, x), c(y, y1, y1),
						c(x, x2, x2), c(y2, y2 ,y))
			}
			if((addshade=="negtive"||addshade=="all")&rho<0){
				dat <- cbind(c(x1, x1, x), c(y, y2, y2),
						c(x, x2, x2), c(y1, y1 ,y))
			}
      	return(t(dat))
	}

	pos_corr <- rbind(cbind(Pos, DAT))
	pos_corr2 <- split(pos_corr, 1:nrow(pos_corr))
	SHADE.dat <- matrix(na.omit(unlist(lapply(pos_corr2, shade.dat))),byrow=TRUE, ncol=4)
	segments(SHADE.dat[,1], SHADE.dat[,2], SHADE.dat[,3], 
		SHADE.dat[,4], col = shade.col, lwd = shade.lwd)
	}

	##square
	if(method=="square"&plotCI=="no"){
		symbols(Pos, add = TRUE, inches = FALSE,
			squares = abs(DAT)^0.5, bg = col.fill, fg = col.border)
	}

	##  color
	if(method=="color"&plotCI=="no"){
    	symbols(Pos, add = TRUE, inches = FALSE,
			squares = rep(1, len.DAT), bg = col.fill, fg = col.border)
	}

	## add grid
	if(!is.null(addgrid.col)){
    	symbols(Pos, add=TRUE, inches = FALSE,  bg = NA,
			squares = rep(1, len.DAT), fg = addgrid.col)
	} 
	
	plotCI <- match.arg(plotCI)
	if(plotCI!="no"){
		if(is.null(lowCI.mat)||is.null(uppCI.mat))
			stop("Need lowCI.mat and uppCI.mat!")
		if(!order=="original"){
			lowCI.mat <- lowCI.mat[ord,ord]
			uppCI.mat <- uppCI.mat[ord,ord]
		}
		pos.lowNew  <- getPos.Dat(lowCI.mat)[[1]]
		lowNew      <- getPos.Dat(lowCI.mat)[[2]]
		pos.uppNew  <- getPos.Dat(uppCI.mat)[[1]]
		uppNew      <- getPos.Dat(uppCI.mat)[[2]]
		if(!(method=="circle"||method=="square"))
			stop("method shoud be circle or square if draw confidence interval!")
		k1 <- (abs(uppNew) > abs(lowNew))
		bigabs <- uppNew 
		bigabs[which(!k1)] <- lowNew[!k1]
		smallabs <- lowNew
		smallabs[which(!k1)] <- uppNew[!k1]
		sig <- sign(uppNew * lowNew)
		
		if(plotCI=="circle"){	
			symbols(pos.uppNew[,1], pos.uppNew[,2],
				add = TRUE,  inches = FALSE,
				circles = 0.95*abs(bigabs)**0.5/2,  
				bg = ifelse(sig>0, col.fill, col[ceiling((bigabs+1)*length(col)/2)]),
				fg = ifelse(sig>0, col.fill, col[ceiling((bigabs+1)*length(col)/2)]))
			symbols(pos.lowNew[,1], pos.lowNew[,2],
				add = TRUE, inches = FALSE, 
				circles = 0.95*abs(smallabs)**0.5/2, 
				bg = ifelse(sig>0, bg, col[ceiling((smallabs+1)*length(col)/2)]),
				fg = ifelse(sig>0, col.fill, col[ceiling((smallabs+1)*length(col)/2)]))
		}
		
		if(plotCI=="square"){
			symbols(pos.uppNew[,1], pos.uppNew[,2],
				add = TRUE,  inches = FALSE,
				squares = abs(bigabs)**0.5,  
				bg = ifelse(sig>0, col.fill, col[ceiling((bigabs+1)*length(col)/2)]),
				fg = ifelse(sig>0, col.fill, col[ceiling((bigabs+1)*length(col)/2)]))
			symbols(pos.lowNew[,1], pos.lowNew[,2],
				add = TRUE, inches = FALSE, 
				squares = abs(smallabs)**0.5, 
				bg = ifelse(sig>0, bg, col[ceiling((smallabs+1)*length(col)/2)]),
				fg = ifelse(sig>0, col.fill, col[ceiling((smallabs+1)*length(col)/2)]))
		}

		if(plotCI=="rect"){
			rect.width <- 0.25
			rect(pos.uppNew[,1]-rect.width, pos.uppNew[,2]+smallabs/2,
				pos.uppNew[,1] +rect.width, pos.uppNew[,2]+bigabs/2,
				col=col.fill, border=col.fill)
			segments(pos.lowNew[,1]-rect.width, pos.lowNew[,2]+DAT/2,
				pos.lowNew[,1]+rect.width, pos.lowNew[,2]+DAT/2,
				col="black",lwd=1)	
			segments(pos.uppNew[,1]-rect.width, pos.uppNew[,2]+uppNew/2,
				pos.uppNew[,1]+rect.width, pos.uppNew[,2]+uppNew/2,
				col="black",lwd=1)
			segments(pos.lowNew[,1]-rect.width, pos.lowNew[,2]+lowNew/2,
				pos.lowNew[,1]+rect.width, pos.lowNew[,2]+lowNew/2,
				col="black",lwd=1)
			segments(pos.lowNew[,1]-0.5,pos.lowNew[,2], 
				pos.lowNew[,1]+0.5, pos.lowNew[,2],col = "grey70", lty=3)
		}

	}
	
	insig <- match.arg(insig)
	if(!is.null(p.mat)&!insig=="no"){
    	if(!order=="original")
    	p.mat <- p.mat[ord, ord]
		pos.pNew  <- getPos.Dat(p.mat)[[1]]
    	pNew      <- getPos.Dat(p.mat)[[2]]
    	
		ind.p <- which(pNew > (sig.level))
    	if(insig=="pch"){
			points(pos.pNew[,1][ind.p], pos.pNew[,2][ind.p],
				pch = pch, col = pch.col, cex = pch.cex, lwd=2)
		}
		if(insig=="blank"){
			symbols(pos.pNew[,1][ind.p], pos.pNew[,2][ind.p], inches = FALSE,
				squares = rep(1, length(pos.pNew[,1][ind.p])),
				fg = addgrid.col, bg = bg, add = TRUE)
		}
	}

	if(addcolorlabel!="no"){
		if(cl.range=="min2max"){
			Range <- c(min(DAT), max(DAT))
			colRange <- col.fill[c(which.min(DAT), which.max(DAT))]
			ind1 <- which(col==colRange[1])
			ind2 <- which(col==colRange[2])
			colbar <- col[ind1:ind2]
			if(length(colbar)>20){
				labels <- round(seq(Range[1],Range[2], length=cl.length), 2)
				at <- seq(0,1,length=length(labels))
			} else {
				#warning("color is too few, cl.length is omited!")
				if(assign.col=="-1to1")
					labels <- 2*((ind1-1):ind2)/length(col) - 1
				if(assign.col=="min2max")
					labels <- seq(min(DAT),max(DAT),length=length(col)+1)
				if(assign.col=="0to1")
					labels <- seq(0,1,length=length(col)+1)	
				labels[1] <- min(DAT)
				labels[length(labels)] <- max(DAT)
				labels <- round(labels, 2)
				at <- seq(0,1,length=length(labels))
			}
		}
		if(cl.range=="-1to1"){
			if(assign.col!="-1to1") stop("assign.col shoulde be \"-1to1\"")
			colbar <- col
			labels <- round(seq(-1,1, length=min(cl.length,length(col)+1)), 2)
			at <- seq(0,1,length=length(labels))
		}
		if(cl.range=="0to1"){
			if(assign.col!="0to1") stop("assign.col shoulde be \"0to1\"")
			colbar <- col
			labels <- round(seq(0,1, length=min(cl.length,length(col)+1)), 2)
			at <- seq(0,1,length=length(labels))
		}		
		if(addcolorlabel=="right"){
			vertical <- TRUE
			xlim <- c(m2 + 0.5 + mm*0.02, m2 + 0.5 + mm*cl.ratio)
			ylim <- c(n1-0.5, n2+0.5)
		}

		if(addcolorlabel=="bottom"){
			vertical <- FALSE
			xlim <- c(m1-0.5, m2+0.5)
			ylim <- c(n1 - 0.5 - nn*cl.ratio, n1 - 0.5- nn*0.02)
		}
		colorlegend(colbar=colbar, labels=labels, offset=cl.offset,
			ratio.colbar = 0.3, cex=cl.cex,
			xlim=xlim, ylim=ylim, vertical=vertical, align=cl.align.text)
	}

	
	## add variable names and title
	if(addtextlabel!="no"){
        cex2 <- tl.cex * S1/S2
        ylabwidth2 <- strwidth(newrownames, cex = cex2)
        xlabwidth2 <- strwidth(newcolnames, cex = cex2)
        pos.xlabel <- cbind(m1:m2, n2 + 0.5 + xlabwidth2/2)
        pos.ylabel <- cbind(m1 - 0.5, n2:n1)

        if(addtextlabel=="td"){
			if(type!="upper") stop("type should be \"upper\" if addtextlabel is \"dt\".")
			pos.ylabel <- cbind(m1:(m1+nn)-0.5, n2:n1)
		}
        if(addtextlabel=="ld"){
			if(type!="lower") stop("type should be \"lower\" if addtextlabel is \"ld\".")
            pos.xlabel <- cbind(m1:m2, n2:(n2-mm) + 0.5 + xlabwidth2/2)
        }
		if(addtextlabel=="d"){
			##if(type!="full") stop("type should be \"full\" if addtextlabel is \"d\".")
			pos.ylabel <- cbind(m1:(m1+nn)-0.5, n2:n1)
			pos.ylabel <- pos.ylabel[1:min(n,m),]
			symbols(pos.ylabel[,1]+0.5, pos.ylabel[,2],add = TRUE,
				bg = bg, fg = addgrid.col,
				inches = FALSE, squares = rep(1, length(pos.ylabel[,1])))
			text(pos.ylabel[,1]+0.5, pos.ylabel[,2], newcolnames[1:min(n,m)],
                col = tl.col, cex = tl.cex * S1/S2)
		} 
		if(addtextlabel=="lt"){
			text(pos.xlabel[,1], pos.xlabel[,2], newcolnames, srt = 90,
                col = tl.col, cex = tl.cex * S1/S2, pos=3, offset=tl.offset)
			text(pos.ylabel[,1], pos.ylabel[,2], newrownames,
                col = tl.col, cex = tl.cex * S1/S2, pos=2, offset=tl.offset)
		}
	}

	title(title)
   
	## add numbers
	if(!is.null(addnum.col)&(!method == "number")){
		text(Pos[,1], Pos[,2], round(100 * DAT), col = addnum.col)
	}

	if(type=="full"&plotCI=="no"&!is.null(addgrid.col))
		rect(m1-0.5, n1-0.5, m2+0.5, n2+0.5, border=addgrid.col)
	##  draws rectangles
	if(!is.null(addrect)&order=="hclust"&type=="full"){
		tree <- hclust(as.dist(1-corr), method = hclust.method)
		hc <- cutree(tree, k = addrect)
		clustab <- table(hc)[unique(hc[tree$order])]
		cu <- c(0, cumsum(clustab))
		mat <- cbind(cu[-(addrect + 1)] + 0.5, n - cu[-(addrect + 1)] + 0.5,
					cu[-1] + 0.5, n - cu[-1] + 0.5)
		rect(mat[,1], mat[,2], mat[,3], mat[,4], border = rect.col, 
			lwd = rect.lwd)
	}

	invisible(ord)
} 

## end