% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core_dm.R, R/formatting_rest.R
\name{simulate_traces}
\alias{simulate_traces}
\alias{simulate_traces.drift_dm}
\alias{simulate_traces.fits_ids_dm}
\alias{print.traces_dm_list}
\alias{print.traces_dm}
\title{Simulate Trajectories/Traces of a Model}
\usage{
simulate_traces(object, k, ...)

\method{simulate_traces}{drift_dm}(
  object,
  k,
  ...,
  conds = NULL,
  add_x = FALSE,
  sigma = NULL,
  seed = NULL,
  unpack = FALSE
)

\method{simulate_traces}{fits_ids_dm}(object, k, ...)

\method{print}{traces_dm_list}(x, ..., round_digits = drift_dm_default_rounding(), print_steps = 5)

\method{print}{traces_dm}(
  x,
  ...,
  round_digits = drift_dm_default_rounding(),
  print_steps = 5,
  print_k = 4
)
}
\arguments{
\item{object}{an object of type \link{drift_dm} or \code{fits_ids_dm} (see
\link{load_fits_ids}).}

\item{k}{numeric, the number of traces to simulate per condition. Can be a
named numeric vector, to specify different number of traces per condition.}

\item{...}{additional arguments passed forward to the respective method.}

\item{conds}{optional character vector, conditions for which traces shall be
simulated. If \code{NULL}, then traces for all conditions are simulated.}

\item{add_x}{logical, indicating whether traces should contain a
variable starting point. If \code{TRUE}, samples from \code{x_fun} (see
\link{comp_vals}) are added to each trace. Default is \code{FALSE}.}

\item{sigma}{optional numeric, providing a value >= 0 for the diffusion
constant "sigma" to temporally override \link{prms_solve}. Useful for
exploring the model without noise.}

\item{seed}{optional numerical, a seed for reproducible sampling}

\item{unpack}{logical, indicating if the traces shall be "unpacked" (see
also \link{unpack_traces} and the return value below).}

\item{x}{an object of type \code{traces_dm_list} or \code{traces_dm}, resulting from a
call to \code{simulate_traces}.}

\item{round_digits}{integer, indicating the number of decimal places (round)
to be used when printing out the traces (default is 3).}

\item{print_steps}{integer, indicating the number of steps to show when
printing out traces (default is 5).}

\item{print_k}{integer, indicating how many traces shall be shown when
printing out traces (default is 4).}
}
\value{
\code{simulate_traces()} returns either a list of type \code{traces_dm_list}, or
directly the plain traces as matrices across conditions (if \code{unpack = TRUE}).
If the model has only one condition (and \code{unpack = TRUE}), then the matrix of
traces for this one condition is directly returned.

The returned list has as many entries as conditions requested. For example,
if only one condition is requested via the \code{conds} argument, then the list is
of length 1 (if \code{unpack = FALSE}). If \code{conds} is set to \code{NULL} (default),
then the list will have as many entries as conditions specified in the
supplied \code{object} (see also \link{conds}). If \code{unpack = FALSE}, the list
contains an additional attribute with the time space.

Each matrix of traces has \code{k} rows and \code{nt + 1} columns, stored as an
array of size (\code{k}, \code{nt + 1}). Note that \code{nt} is the number of steps in the
discretization of time; see \link{drift_dm}. If \code{unpack = FALSE}, the
array is of type \code{traces_dm}. It contains some additional attributes about
the time space, the drift rate, the boundary, and the added starting values.

The print methods \code{print.traces_dm_list()} and \code{print.traces_dm()} each
invisibly return the supplied object \code{x}.
}
\description{
Simulates single trajectories/traces of a model
(i.e., evidence accumulation processes) using forward Euler.

Might come in handy when exploring the model's behavior or when
creating figures (see also \link{plot.traces_dm_list})
}
\details{
\code{simulate_traces()} is a generic function, applicable to objects of type
\link{drift_dm} or \code{fits_ids_dm} (see \link{load_fits_ids}).

For \link{drift_dm} objects, \code{simulate_traces()} performs the simulation
on the parameter values currently set (see
\code{\link[=coef.drift_dm]{coef.drift_dm()}}).

For \code{fits_ids_dm} objects, \code{simulate_traces()} first extracts the model and
all parameter values for all IDs (see \code{\link[=coef.fits_ids_dm]{coef.fits_ids_dm()}}).
Subsequently, simulations are based on the averaged parameter values.

The algorithm for simulating traces is forward euler. See
\insertCite{Richteretal.2023;textual}{dRiftDM} and
\insertCite{Ulrichetal.2015;textual}{dRiftDM} (Appendix A) for more
information.
}
\note{
Evidence values with traces beyond the boundary of the model are set to NA
before passing them back.

The reason why \code{simulate_traces} passes back an object of type
\code{traces_dm_list} (instead of simply a list of arrays) is to provide a
\link{plot.traces_dm_list} and \link{print.traces_dm_list} function.

Users can unpack the traces even after calling \code{simulate_traces()} using
\code{\link[=unpack_traces]{unpack_traces()}}.
}
\examples{
# get a pre-built model to demonstrate the function
my_model <- dmc_dm()
some_traces <- simulate_traces(my_model, k = 1, seed = 1)
print(some_traces)

# a method is also available for fits_ids_dm objects
# (see estimate_model_ids)
# get an exemplary fits_ids_dm object
fits <- get_example_fits_ids()
some_traces <- simulate_traces(fits, k = 1, seed = 1)
print(some_traces)

# we can also print only the traces of one condition
print(some_traces$comp)

}
\seealso{
\code{\link[=unpack_traces]{unpack_traces()}}, \code{\link[=plot.traces_dm_list]{plot.traces_dm_list()}}
}
