#' Download data on the candidates' backgrounds in federal elections
#'
#' \code{candidate_fed()} downloads and aggregates data on the candidates' background who ran in
#' federal elections in Brazil. The function returns a \code{data.frame} where each observation
#' corresponds to a candidate.
#'
#' @note For the elections prior to 2002, some information can be incomplete. For the 2014 and 2018 elections, more variables are available.
#'
#' @param year Election year (\code{integer}). For this function, only the years 1994, 1998, 2002, 2006, 2010, 2014, and 2018
#' are available.
#' 
#' @param uf Federation Unit acronym (\code{character vector}).
#' 
#' @param br_archive In the TSE's data repository, some results can be obtained for the whole country by loading a single
#' within a single file by setting this argument to \code{TRUE} (may not work in for some elections and, in 
#' other, it recoverns only electoral data for presidential elections, absent in other files).
#' 
#' @param ascii (\code{logical}). Should the text be transformed from latin1 to ASCII format?
#'
#' @param encoding Data original encoding (defaults to 'latin1'). This can be changed to avoid errors
#' when \code{ascii = TRUE}.
#' 
#' @param export (\code{logical}). Should the downloaded data be saved in .dta and .sav in the current directory?
#'
#' @details If export is set to \code{TRUE}, the downloaded data is saved as .dta and .sav
#'  files in the current directory.
#'  
#'
#' @return \code{candidate_fed()} returns a \code{tbl, data.frame} with the following variables:
#'
#' \itemize{
#'   \item DATA_GERACAO: Generation date of the file (when the data was collected).
#'   \item HORA_GERACAO: Generation time of the file (when the data was collected), Brasilia Time.
#'   \item ANO_ELEICAO: Election year.
#'   \item NUM_TURNO: Round number.
#'   \item DESCRICAO_ELEICAO: Description of the election.
#'   \item SIGLA_UF: Units of the Federation's acronym in which occurred the election.
#'   \item SIGLA_UE: Units of the Federation's acronym (In case of major election is the FU's 
#'   acronym in which the candidate runs for (text) and in case of municipal election is the
#'   municipal's Supreme Electoral Court code (number)). Assume the special values BR, ZZ and
#'   VT to designate, respectively, Brazil, Overseas and Absentee Ballot.
#'   \item DESCRICAO_UE: Description of the Electoral Unit.
#'   \item CODIGO_CARGO: Code of the position that the candidate runs for.
#'   \item DESCRICAO_CARGO: Description of the position that the candidate runs for.
#'   \item NOME_CANDIDATO: Candidate's complete name.
#'   \item SEQUENCIAL_CANDIDATO: Candidate's sequence number generated internally by the electoral
#'   systems. It is not the candidate's campaign number.
#'   \item NUMERO_CANDIDATO: Candidate's number in the ballot box.
#'   \item CPF_CANDIDATO: Candidate's CPF.
#'   \item NOME_URNA_CANDIDATO: Candidate's ballot box name.
#'   \item COD_SITUACAO_CANDIDATURA: Code of the candidature situation.
#'   \item DES_SITUACAO_CANDIDATURA: Description of the candidature situation.
#'   \item NUMERO_PARTIDO: Party number.
#'   \item SIGLA_PARTIDO: Party acronym.
#'   \item NOME_PARTIDO: Party name.
#'   \item CODIGO_LEGENDA: Sequential code of the party shortname generated by the Electoral Justice.
#'   \item SIGLA_LEGENDA: Party's shortname acronym.
#'   \item COMPOSICAO_LEGENDA: Party's shortname composition.
#'   \item NOME_COLIGACAO: Coalition shortname.
#'   \item CODIGO_OCUPACAO: Candidate's occupation code.
#'   \item DESCRICAO_OCUPACAO: Candidate's occupation description.
#'   \item DATA_NASCIMENTO: Candidate's date of birth.
#'   \item NUM_TITULO_ELEITORAL_CANDIDATO: Candidate's ballot number.
#'   \item IDADE_DATA_ELEICAO: Candidate's age on the day of election.
#'   \item CODIGO_SEXO: Candidate's sex code.
#'   \item DESCRICAO_SEXO: Candidate's sex description.
#'   \item COD_GRAU_INSTRUCAO: Candidate's level of education code. Generated internally by the electoral systems.
#'   \item DESCRICAO_GRAU_INSTRUCAO: Candidate's level of education description.
#'   \item CODIGO_ESTADO_CIVIL: Candidate's marital status code.
#'   \item DESCRICAO_ESTADO_CIVIL: Candidate's marital status description.
#'   \item CODIGO_NACIONALIDADE: Candidate's nationality code.
#'   \item DESCRICAO_NACIONALIDADE: Candidate's nationality description.
#'   \item SIGLA_UF_NASCIMENTO: Candidate's Units of the Federation birth's acronym.
#'   \item COD_MUNICIPIO_NASCIMENTO: Candidate's birth city's Supreme Electoral Court code.
#'   \item COD_MUNICIPIO_NASCIMENTO: Candidate's birth city.
#'   \item DESPESA_MAX_CAMPANHA: Maximum expenditure campaign declared by the party to that position. Values in Reais.
#'   \item COD_SIT_TOT_TURNO: Candidate's totalization status code in that election round.
#'   \item DESC_SIT_TOT_TURNO: Candidate's totalization status description in that round.
#'   \item CODIGO_COR_RACA: Candidate's color/race code (self-declaration, only from 2014 election).
#'   \item DESCRICAO_COR_RACA: Candidate's color/race description (self-declaration, only from 2014 election).
#'   \item EMAIL_CANDIDATO: Candidate's e-mail adress (only from 2014 election).
#' }
#' 
#' From 2018 on, some new variables are also available:
#' \itemize{
#'   \item COD_TIPO_ELEICAO: Election type code.
#'   \item NOME_TIPO_ELEICAO: Election type.
#'   \item COD_ELEICAO: Election code.
#'   \item DATA_ELEICAO: Election date.
#'   \item ABRANGENCIA: Election scope.
#'   \item NOME_SOCIAL_CANDIDATO: Candidate's social name.
#'   \item EMAIL_CANDIDATO: Candidate's e-mail.
#'   \item COD_DETALHE_SITUACAO_CAND: Details on the status of a candidate's elegibility.
#'   \item DES_DETALHE_SITUACAO_CAND: Description of a candidate's elegibility.
#'   \item TIPO_AGREMIACAO: Type of partisan ticket (electoral coalition or single party).
#'   \item IDADE_DATA_POSSE: Candidate's age at the first day in the office.
#'   \item CODIGO_COR_RACA: Candidates' skin color code.
#'   \item DESCRICAO_COR_RACA: Candidates' skin color.
#'   \item SITUACAO_REELEICAO: Candidate's reelection status (running for reelection or not).
#'   \item SITUACAO_DECLARAR_BENS: Candidate's financial disclosure.
#'   \item NUMERO_PROTOCOLO_CANDIDATURA: Candidate's electoral protocol number.
#'   \item NUMERO_PROCESSO: Candidate's electoral process.
#'}
#' 
#' @seealso \code{\link{candidate_local}} for local elections in Brazil.
#'
#' @import utils
#' @importFrom magrittr "%>%"
#' @export
#' @examples
#' \dontrun{
#' df <- candidate_fed(2002)
#' }

candidate_fed <- function(year, uf = "all", br_archive = FALSE, ascii = FALSE, encoding = "latin1", export = FALSE){


  # Input tests
  test_encoding(encoding)
  test_fed_year(year)
  uf <- test_uf(uf)
  test_br(br_archive)

  # Download the data
  dados <- tempfile()
  sprintf("http://agencia.tse.jus.br/estatistica/sead/odsele/consulta_cand/consulta_cand_%s.zip", year) %>%
    download.file(dados)
  unzip(dados, exdir = paste0("./", year))
  unlink(dados)

  message("Processing the data...")

  # Cleans the data
  setwd(as.character(year))
  banco <- juntaDados(uf, encoding, br_archive)
  setwd("..")
  unlink(as.character(year), recursive = T)

  # Change variable names
  if(year < 2014){
    names(banco) <- c("DATA_GERACAO", "HORA_GERACAO", "ANO_ELEICAO", "NUM_TURNO", "DESCRICAO_ELEICAO",
                      "SIGLA_UF", "SIGLA_UE", "DESCRICAO_UE", "CODIGO_CARGO", "DESCRICAO_CARGO",
                      "NOME_CANDIDATO", "SEQUENCIAL_CANDIDATO", "NUMERO_CANDIDATO", "CPF_CANDIDATO",
                      "NOME_URNA_CANDIDATO", "COD_SITUACAO_CANDIDATURA", "DES_SITUACAO_CANDIDATURA",
                      "NUMERO_PARTIDO", "SIGLA_PARTIDO", "NOME_PARTIDO", "CODIGO_LEGENDA", "SIGLA_LEGENDA",
                      "COMPOSICAO_LEGENDA", "NOME_COLIGACAO", "CODIGO_OCUPACAO", "DESCRICAO_OCUPACAO",
                      "DATA_NASCIMENTO", "NUM_TITULO_ELEITORAL_CANDIDATO", "IDADE_DATA_ELEICAO",
                      "CODIGO_SEXO", "DESCRICAO_SEXO", "COD_GRAU_INSTRUCAO", "DESCRICAO_GRAU_INSTRUCAO",
                      "CODIGO_ESTADO_CIVIL", "DESCRICAO_ESTADO_CIVIL", "CODIGO_NACIONALIDADE",
                      "DESCRICAO_NACIONALIDADE", "SIGLA_UF_NASCIMENTO", "CODIGO_MUNICIPIO_NASCIMENTO",
                      "NOME_MUNICIPIO_NASCIMENTO", "DESPESA_MAX_CAMPANHA", "COD_SIT_TOT_TURNO",
                      "DESC_SIT_TOT_TURNO")

  }else{
    names(banco) <- c("DATA_GERACAO", "HORA_GERACAO", "ANO_ELEICAO", "COD_TIPO_ELEICAO", "NOME_TIPO_ELEICAO",
                      "NUM_TURNO", "COD_ELEICAO", "DESCRICAO_ELEICAO", "DATA_ELEICAO", "ABRANGENCIA", 
                      "SIGLA_UF", "SIGLA_UE", "DESCRICAO_UE", "CODIGO_CARGO", "DESCRICAO_CARGO", 
                      "SEQUENCIAL_CANDIDATO", "NUMERO_CANDIDATO", "NOME_CANDIDATO", "NOME_URNA_CANDIDATO", 
                      "NOME_SOCIAL_CANDIDATO", "CPF_CANDIDATO", "EMAIL_CANDIDATO", "COD_SITUACAO_CANDIDATURA",
                      "DES_SITUACAO_CANDIDATURA", "COD_DETALHE_SITUACAO_CAND", "DES_DETALHE_SITUACAO_CAND",
                      "TIPO_AGREMIACAO", "NUMERO_PARTIDO", "SIGLA_PARTIDO", "NOME_PARTIDO", "CODIGO_LEGENDA",
                      "NOME_COLIGACAO", "COMPOSICAO_LEGENDA", "CODIGO_NACIONALIDADE", "DESCRICAO_NACIONALIDADE",
                      "SIGLA_UF_NASCIMENTO", "CODIGO_MUNICIPIO_NASCIMENTO", "NOME_MUNICIPIO_NASCIMENTO",
                      "DATA_NASCIMENTO", "IDADE_DATA_POSSE", "NUM_TITULO_ELEITORAL_CANDIDATO", "CODIGO_SEXO",
                      "DESCRICAO_SEXO", "COD_GRAU_INSTRUCAO", "DESCRICAO_GRAU_INSTRUCAO", "CODIGO_ESTADO_CIVIL",
                      "DESCRICAO_ESTADO_CIVIL", "CODIGO_COR_RACA", "DESCRICAO_COR_RACA", "CODIGO_OCUPACAO", 
                      "DESCRICAO_OCUPACAO", "DESPESA_MAX_CAMPANHA", "COD_SIT_TOT_TURNO", "DESC_SIT_TOT_TURNO",
                      "SITUACAO_REELEICAO", "SITUACAO_DECLARAR_BENS", "NUMERO_PROTOCOLO_CANDIDATURA", 
                      "NUMERO_PROCESSO")
  }
  
  # Change to ascii
  if(ascii) banco <- to_ascii(banco, encoding)
  
  # Export
  if(export) export_data(banco)
  
  message("Done.\n")
  return(banco)
}
