% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/epifit.R
\name{epifit}
\alias{epifit}
\title{Model fitting function for epifit package}
\usage{
epifit(modeleq, equations = "", initeq = "", itereq = "", endeq = "",
  data, subset, weight, na.action, opt = c("newrap", "BFGS", "CG",
  "Nelder-Mead", "SANN"), tol1 = 1e-08, tol2 = 1e-08, maxiter = 200, init,
  verbatim = 0, ...)
}
\arguments{
\item{modeleq}{a character string specifying the model. See \sQuote{Details}.}

\item{equations}{a character string specifying R expressions executed before \sQuote{modeleq}. Multiple expressions separated by \sQuote{;} is allowed. See \sQuote{Details}.}

\item{initeq}{a character string specifying R expressions executed once at the first time. Typical use is specifying initial values for parameters using R expressions. Multiple expressions separated by \sQuote{;} is allowed.}

\item{itereq}{a character string specifying R expressions executed during Cox regression at each time of event occur. Typical use is incorporating time-dependent variables. See \sQuote{Details}.}

\item{endeq}{a character string specifying R expressions executed at the end of calculations.}

\item{data}{a data.frame in which to interpret the variables named in the formula, or in the subset and the weights argument.}

\item{subset}{expression indicating which subset of the rows of data should be used in the fit. All observations are included by default.}

\item{weight}{vector of case weights. If weights is a vector of integers, then the estimated coefficients are equivalent to estimating the model from data with the individual cases replicated as many times as indicated by weights. Only supported \sQuote{breslow} and \sQuote{efron} ties specification in Cox regression models.}

\item{na.action}{a missing-data filter function. This is applied when data.frame is supplied as \sQuote{data} parameter. Default is \code{options()$na.action}.}

\item{opt}{a character string specifying the method for optimization. When sQuote{newrap} is specified, \code{nlm} function that uses Newton-type algorithm used for obtaining maximum likelihood estimate. For the rest of specifications (\sQuote{BFGS} for quasi-Newton method, \sQuote{CQ} for conjugate gradients method, \sQuote{SANN} for simulated annealing, \sQuote{Nelder-Mead} for Nelder and Mead simplex method), \code{optim} is used. The default is \sQuote{newrap}.}

\item{tol1}{a numeric value specifying \code{gtol} in nlm, \code{abstol} in optim.}

\item{tol2}{a numeric value specifying \code{stol} in nlm, \code{reltol} in optim.}

\item{maxiter}{a integer value specifying the maximum number of iterations. Defaults is 200.}

\item{init}{Initial values for the parameters specified in the form of vector.}

\item{verbatim}{a integer value from 0 (minimum) to 2 (maximum) controlling the amount of information printed during calculation.}

\item{...}{For the arguments used in the inner functions (currently not used).}
}
\value{
A list containing the result of model fitting including parameter estimates, variance of parameter estimates, log likelihood and so on.
}
\description{
This function maximizes an arbitrary likelihood including generalized linear models and Cox partial likelihood.
}
\details{
This function provides flexible model fitting. The main model specification is written in \code{modeleq}. \code{modeleq} consisits of two parts separated by \sQuote{~}. The distribution is specified in the first part, and the second part includes variable name which follows the specified distribution in the first part. Available distribution specifications are \sQuote{cox}(Cox partial likelihood), \sQuote{poisson}(Poisson distribution), \sQuote{normal}(normal distribution), \sQuote{binomial}(binomial distribution), \sQuote{nbinom}(negative binomial distribution), \sQuote{gamma}(gamma distribution) and \sQuote{weibull}(Weibull distribution). Options can be specified for some distribution specification after \sQuote{/} in the distribution specification part. Multiple options separated by \sQuote{,} can also be specified.

For Cox regressions, time and status variable must be specified in parentheses like \code{cox(time, status)}. Some options are also available for Cox regressions, and \sQuote{efron} and \sQuote{breslow} is available for tie handling method. Strata option which specifies a variable indicating strata is also available in Cox regressions. Subset option which has same functinality as subset argument below is also available for Cox regressions and other distribution specifications. For other distribution specifications, parameters must be specified in parentheses. For poisson distribution, mean parameter must be specified as \code{poisson(mean)}. Note that each parameter specificaiton can be a variable or R equation. For other distributions, \code{normal(mean, variance)}, \code{binomial(size, probability)}, \code{nbinom(size, probability)}, \code{gamma(shape, scale)}, \code{weibull(shape, scale)}.

When distributions are specified (not Cox regressions), additional R expressions can be specified in \code{equations} argument. If you define variable \sQuote{mu} in \code{equations}, you can use \sQuote{mu} in \code{modeleq} argument. Refer examples below.
}
\examples{
library(survival)

# Make sample data
set.seed(123)
nsub <- 20
follow <- 5
x <- rnorm(nsub)
rate <- exp(-0.5 + x)
etime <- rweibull(nsub, 1, 1/rate)
status <- as.integer(etime < follow)
time <- pmin(follow, etime)
dat <- data.frame(status, x, time, ltime=log(time))

coxph(Surv(time, status)~x, data=dat)
modeleq <- "cox(time,status)~exp(beta*x)"
epifit(modeleq=modeleq, data=dat)

glm(status ~ x + offset(ltime), family=poisson(), data=dat)
equations <- "mu <- time*exp(beta0 + beta1*x)"
modeleq <- "poisson(mu) ~ status"
epifit(modeleq=modeleq, equations=equations, data=dat)

# The simplest test data set from coxph function
test1 <- list(time=c(4,3,1,1,2,2,3),
              status=c(1,1,1,0,1,1,0),
              x=c(0,2,1,1,1,0,0),
              sex=c(0,0,0,0,1,1,1))

coxph(Surv(time, status) ~ x + strata(sex), data=test1)
modeleq <- "cox(time,status)/strata(sex)~exp(beta*x)"
epifit(modeleq=modeleq, data=test1)

# Tie specification example in Cox regressions
coxph(Surv(time, status) ~ x + strata(sex), data=test1, ties="breslow")
modeleq <- "cox(time,status)/strata(sex),ties(breslow)~exp(beta*x)"
epifit(modeleq=modeleq, data=test1)
}

