




#' A Markov Chain Monte Carlo (MCMC) sampler for a linear panel model
#'
#' The sampler uses independent Normal-inverse-Gamma priors to estimate a linear panel data model. The function is
#' used for an illustration on using the \code{\link{beta_sampler}} and \code{\link{sigma_sampler}} classes.
#'
#' The considered model takes the form:
#'
#' \deqn{
#'  Y_t = X_t \beta + \varepsilon_t,
#'  }
#'
#' with \eqn{\varepsilon_t \sim N(0,I_n \sigma^2)}.
#'
#' \eqn{Y_t} (\eqn{n \times 1}) collects the \eqn{n} cross-sectional observations for time
#' \eqn{t=1,...,T}. \eqn{X_t} (\eqn{n \times k_1}) is a matrix of explanatory variables.
#' \eqn{\beta} (\eqn{k_1 \times 1}) is an unknown slope parameter matrix.
#'
#' After vertically staking the \eqn{T} cross-sections  \eqn{Y=[Y_1',...,Y_T']'} (\eqn{N \times 1}),
#' \eqn{X=[X_1',...,X_T']'} (\eqn{N \times k}), with \eqn{N=nT}, the final model can be expressed as:
#'
#' \deqn{
#'  Y = X \beta + \varepsilon,
#' }
#'
#' where \eqn{\varepsilon \sim N(0,I_N \sigma^2)}. Note that the input
#' data matrices have to be ordered first by the cross-sectional (spatial) units and then stacked by time.
#'
#' @param Y numeric \eqn{N \times 1} matrix containing the dependent variables, where \eqn{N = nT} is the number of
#' spatial (\eqn{n}) times the number of time observations (\eqn{T}, with \code{tt=T}). Note that the observations
#' have organized such that \eqn{Y = [Y_1',...,Y_T']'}.
#' @param tt single number greater or equal to 1. Denotes the number of time observations. \eqn{tt = T}.
#' @param X numeric \eqn{N \times k_1} design matrix of independent variables.
#' @param niter single number greater or equal to 1, indicating the total number of draws.
#' Will be automatically coerced to integer. The default value is 200.
#' @param nretain single number greater or equal to 0, indicating the number of draws
#' kept after the burn-in. Will be automatically coerced to integer. The default value is 100.
#' @param beta_prior list containing priors for the slope coefficients \eqn{\beta},
#' generated by the smart constructor \code{\link{beta_priors}}.
#' @param sigma_prior list containing priors for the error variance \eqn{\sigma^2},
#' generated by the smart constructor \code{\link{sigma_priors}}
#'
#' @export
#'
#' @examples
#' n = 20; tt = 10; k = 3
#' X = matrix(stats::rnorm(n*tt*k),n*tt,k)
#' Y = X %*% c(1,0,-1) + stats::rnorm(n*tt,0,.5)
#' res = normalgamma(Y,tt,X)
normalgamma <- function(Y, tt, X = matrix(1,nrow(Y),1), niter = 200, nretain = 100,
                 beta_prior = beta_priors(k = ncol(X)),sigma_prior = sigma_priors()) {
  k = ncol(X); n <- nrow(X) / tt
  ndiscard <- niter - nretain

  # save the posterior draws here
  if (is.null(colnames(X))) {colnames(X) = paste0("X",1:k)}
  varnames = colnames(X)
  postb <- matrix(0, k, nretain); rownames(postb) <- colnames(X)
  posts <- matrix(0, 1, nretain); rownames(posts) = "sigma"

  sampler_beta = beta_sampler$new(beta_prior)
  sampler_sigma = sigma_sampler$new(sigma_prior)

  ### Gibbs sampling
  pb <- utils::txtProgressBar(min = 0, max = niter, style = 3)
  for (iter in 1:niter) {
    # draw beta
    sampler_beta$sample(Y,X,sampler_sigma$curr_sigma)
    curr.xb <- X %*% sampler_beta$curr_beta

    # draw sigma
    sampler_sigma$sample(Y,curr.xb)

    # we are past the burn-in, save the draws
    if (iter > ndiscard) {
      s <- iter - ndiscard
      postb[, s] <- sampler_beta$curr_beta
      posts[s] <- sampler_sigma$curr_sigma
    }
    utils::setTxtProgressBar(pb,iter)
  }
  close(pb)

  ret = list(Y = Y, X = X,
             postb = postb, posts = posts,
             beta_prior = beta_prior,sigma_prior = sigma_prior,
             param = list(niter = niter, nretain = nretain)
  )
  class(ret) = "normalgamma"
  return(ret)
}

#' A Markov Chain Monte Carlo (MCMC) sampler for the panel spatial Durbin model (SDM) with exogenous spatial weight matrix.
#'
#' The sampler uses independent Normal-inverse-Gamma priors for the slope and variance parameters,
#' as well as a four-parameter prior for the spatial autoregressive parameter \eqn{\rho}. The function is
#' used as an illustration on using the \code{\link{beta_sampler}}, \code{\link{sigma_sampler}},
#' and \code{\link{rho_sampler}} classes.
#'
#' The considered panel spatial Durbin model (SDM) takes the form:
#'
#' \deqn{
#'  Y_t = \rho W Y_t + X_t \beta_1 + W X_t \beta_2 + Z \beta_3 + \varepsilon_t,
#'  }
#'
#' with \eqn{\varepsilon_t \sim N(0,I_n \sigma^2)}. The row-stochastic \eqn{n} by \eqn{n} spatial weight
#' matrix \eqn{W} is non-negative and has zeros on the main diagonal. \eqn{\rho} is a scalar spatial autoregressive parameter.
#'
#' \eqn{Y_t} (\eqn{n \times 1}) collects the \eqn{n} cross-sectional (spatial) observations for time
#' \eqn{t=1,...,T}. \eqn{X_t} (\eqn{n \times k_1}) and \eqn{Z_t} (\eqn{n \times k_2}) are
#' matrices of explanatory variables, where the former will also be spatially lagged. \eqn{\beta_1}
#' (\eqn{k_1 \times 1}), \eqn{\beta_2} (\eqn{k_1 \times 1}) and \eqn{\beta_3} (\eqn{k_2 \times 1})
#' are unknown slope parameter vectors.
#'
#' After vertically staking the \eqn{T} cross-sections  \eqn{Y=[Y_1',...,Y_T']'} (\eqn{N \times 1}),
#' \eqn{X=[X_1',...,X_T']'} (\eqn{N \times k_1}) and \eqn{Z=[Z_1', ..., Z_T']'} (\eqn{N \times k_2}),
#' with \eqn{N=nT}, the final model can be expressed as:
#'
#' \deqn{
#'  Y = \rho \tilde{W}Y + X \beta_1 + \tilde{W} X \beta_2 + Z \beta_3 + \varepsilon,
#' }
#'
#' where \eqn{\tilde{W}=I_T \otimes W} and \eqn{\varepsilon \sim N(0,I_N \sigma^2)}. Note that the input
#' data matrices have to be ordered first by the cross-sectional spatial units and then stacked by time.
#'
#' @inheritParams normalgamma
#' @param W numeric, non-negative and row-stochastic \eqn{n} by \eqn{n} exogenous spatial weight matrix.
#' Must have zeros on the main diagonal.
#' @param X numeric \eqn{N \times k_1} design matrix of independent variables. These will be
#' automatically spatially lagged. If no spatially lagged variable is included in the model
#' a matrix with \eqn{N} rows and zero columns should be supplied (the default value). Note:
#' either \eqn{X} or \eqn{Z} has to be a matrix with at least one column.
#' @param Z numeric \eqn{N \times k_3} design matrix of independent variables which are not
#' spatially lagged. The default value is a \eqn{N \times 1} vector of ones (i.e. an intercept for the model).
#' Note: either \eqn{X} or \eqn{Z} has to be a matrix with at least one column.
#' @param rho_prior list of prior settings for estimating \eqn{\rho},
#'  generated by the smart constructor \code{\link{rho_priors}}
#' @param beta_prior list containing priors for the slope coefficients \eqn{\beta},
#' generated by the smart constructor \code{\link{beta_priors}}. The ordering of the priors is:
#' (1) priors of \eqn{X}, (2) priors of spatially lagged \eqn{X}, (3) priors of \eqn{Z}.
#'
#' @export
#'
#' @examples
#' n = 20; tt = 10
#' dgp_dat = sim_dgp(n = n, tt = tt, rho = .5, beta1 = c(.5,1), beta2 = c(-1,.5),
#'                   beta3 = c(1.5), sigma2 = .5)
#' res = sdm(Y = dgp_dat$Y, tt = tt,  W = dgp_dat$W, X = dgp_dat$X,
#'           Z = dgp_dat$Z, niter = 100, nretain = 50)
sdm <- function(Y, tt, W, X = matrix(0,nrow(Y),0),Z = matrix(1,nrow(Y),1), niter = 200, nretain = 100,
                rho_prior = rho_priors(),beta_prior = beta_priors(k = ncol(X)*2 + ncol(Z)),
                sigma_prior = sigma_priors()) {
  if (ncol(X) == 0 && ncol(Z) == 0) {stop("Error: At least either X or Z matrix have to be provided.")}
  origX = X

  smallk = ncol(X)
  if (smallk>0 && is.null(colnames(X))) {
    colnames(X) = paste0("X",1:smallk)
  }
  k_dum = ncol(Z)
  if (k_dum>0 && is.null(colnames(Z))) {
    colnames(Z) = paste0("Z",1:k_dum)
  }
  if (smallk>0) {varnames = c(colnames(X), paste0("W_",colnames(X)))} else {varnames = c()}
  if (k_dum > 0) varnames = c(varnames,colnames(Z))

  ndiscard <- niter - nretain
  k <- smallk*2 + k_dum
  n <- nrow(X) / tt

  # map variable positions for spatial effects
  ind_baseX = ind_WX = ind_lagFX = c()
  if (smallk > 0) {
    ind_baseX = c(1:smallk)
    # the columns of XX that are spatially lagged
    ind_WX = c(1:smallk) + smallk
    # the spatial FX corresponding to these
    ind_lagFX = 1:smallk
  }
  if (k_dum > 0) {ind_baseX = c(ind_baseX,(2*smallk + 1):k)}

  # save the posterior draws here
  postb <- matrix(0, k, nretain)
  rownames(postb) <- varnames
  posts <- matrix(0, 1, nretain); rownames(posts) = "sigma"
  postr <- matrix(0, 1, nretain); rownames(postr) = "rho"

  post.direct <- matrix(0, smallk + k_dum, nretain)
  post.indirect <- matrix(0, smallk + k_dum, nretain)
  post.total <- matrix(0, smallk + k_dum, nretain)
  rownames(post.direct) <- rownames(post.indirect) <- rownames(post.total) <- varnames[ind_baseX]

  # initilize wdraws
  curr.WX = as.matrix(kronecker(Matrix::.sparseDiagonal(tt),W) %*% X)
  tX = cbind(X,curr.WX,Z)
  tY <- matrix(Y, n, tt)
  tAy <- matrix(0, n, tt)

  sampler_beta = beta_sampler$new(beta_prior)
  sampler_sigma = sigma_sampler$new(sigma_prior)
  sampler_rho <- rho_sampler$new(rho_prior)

  sampler_rho$setW(W)

  ### Gibbs sampling
  pb <- utils::txtProgressBar(min = 0, max = niter, style = 3)
  for (iter in 1:niter) {
    Ay <- matrix(sampler_rho$curr_A %*% tY, n * tt, 1)

    # draw beta
    sampler_beta$sample(Ay,tX,sampler_sigma$curr_sigma)
    curr.xb <- tX %*% sampler_beta$curr_beta
    curr.txb <- matrix(curr.xb, n, tt)

    # draw sigma
    sampler_sigma$sample(Ay,curr.xb)

    sampler_rho$sample(tY,curr.txb,sampler_sigma$curr_sigma)
    if (iter > (ndiscard / 2)) {
      sampler_rho$stopMHtune()
    }

    # we are past the burn-in, save the draws
    if (iter > ndiscard) {
      s <- iter - ndiscard
      postb[, s] <- sampler_beta$curr_beta
      posts[s] <- sampler_sigma$curr_sigma
      postr[s] <- sampler_rho$curr_rho

      post.direct[, s] <- sum(diag(sampler_rho$curr_AI)) / n * sampler_beta$curr_beta[ind_baseX]
      post.total[, s] <- sum(sampler_rho$curr_AI) / n * sampler_beta$curr_beta[ind_baseX]
      # if we have WX
      if (smallk > 0) {
        post.direct[ind_lagFX,s] = post.direct[ind_lagFX,s] +
          sum(diag(sampler_rho$curr_AI))/n * sampler_beta$curr_beta[ind_WX]
        post.total[ind_lagFX,s] = post.total[ind_lagFX,s] +
          sum(sampler_rho$curr_AI)/n * sampler_beta$curr_beta[ind_WX]
      }
      post.indirect[, s] <- post.total[, s] - post.direct[, s]
    }
    utils::setTxtProgressBar(pb,iter)
  }
  close(pb)

  ret = list(Y = Y, X = X, Z = Z, W = W,
             post.direct = post.direct, post.indirect = post.indirect, post.total = post.total,
             postb = postb, posts = posts,postr = postr,
             beta_prior = beta_prior,sigma_prior = sigma_prior,
             param = list(niter = niter, nretain = nretain)
  )
  class(ret) = "exoW"
  return(ret)
}

#' A Markov Chain Monte Carlo (MCMC) sampler for the panel spatial autoregressive model (SAR) with exogenous spatial weight matrix.
#'
#' The sampler uses independent Normal-inverse-Gamma priors for the slope and variance parameters,
#' as well as a four-parameter beta prior for the spatial autoregressive parameter \eqn{\rho}. The function is
#' used as an illustration on using the \code{\link{beta_sampler}}, \code{\link{sigma_sampler}},
#' and \code{\link{rho_sampler}} classes.
#'
#' The considered panel spatial autoregressive model (SAR) takes the form:
#'
#' \deqn{
#'  Y_t = \rho W Y_t + Z_t \beta + \varepsilon_t,
#'  }
#'
#' with \eqn{\varepsilon_t \sim N(0,I_n \sigma^2)}. The row-stochastic \eqn{n} by \eqn{n} spatial weight
#' matrix \eqn{W} is non-negative and has zeros on the main diagonal. \eqn{\rho} is a scalar spatial autoregressive parameter.
#'
#' \eqn{Y_t} (\eqn{n \times 1}) collects the \eqn{n} cross-sectional (spatial) observations for time
#' \eqn{t=1,...,T}. \eqn{Z_t} (\eqn{n \times k_3}) is a matrix of explanatory variables.
#' \eqn{\beta} (\eqn{k_3 \times 1}) is an unknown slope parameter matrix.
#'
#' After vertically staking the \eqn{T} cross-sections  \eqn{Y=[Y_1',...,Y_T']'} (\eqn{N \times 1}),
#' \eqn{Z=[Z_1',...,Z_T']'} (\eqn{N \times k_3}), with \eqn{N=nT}, the final model can be expressed as:
#'
#' \deqn{
#'  Y = \rho \tilde{W}Y + Z \beta + \varepsilon,
#' }
#'
#' where \eqn{\tilde{W}=I_T \otimes W} and \eqn{\varepsilon \sim N(0,I_N \sigma^2)}. Note that the input
#' data matrices have to be ordered first by the cross-sectional spatial units and then stacked by time.
#' This is a wrapper function calling \code{\link{sdm}} with no spatially lagged dependent variables.
#'
#' @inheritParams sdm
#' @param Z numeric \eqn{N \times k_3} design matrix of independent variables.
#' The default value is a \eqn{N \times 1} vector of ones (i.e. an intercept for the model).
#' @param beta_prior list containing priors for the slope coefficients,
#' generated by the smart constructor \code{\link{beta_priors}}.
#'
#' @export
#'
#' @examples
#' n = 20; tt = 10
#' dgp_dat = sim_dgp(n =n, tt = tt, rho = .5, beta3 = c(1,.5), sigma2 = .5)
#' res = sar(Y = dgp_dat$Y,tt = tt, W = dgp_dat$W,
#'           Z = dgp_dat$Z,niter = 100,nretain = 50)
sar <- function(Y, tt, W, Z = matrix(1,nrow(Y),1), niter = 200, nretain = 100,
                rho_prior = rho_priors(),beta_prior = beta_priors(k = ncol(Z)),
                sigma_prior = sigma_priors()) {
  ret = sdm(Y = Y, tt =tt,W = W, X = matrix(0,nrow(Y),0),Z = Z, niter = niter, nretain = nretain,
             rho_prior = rho_prior,
             beta_prior = beta_prior,sigma_prior = sigma_prior)
  return(ret)
}

