#' Get metadata on images, sounds and videos
#'
#' In addition to text data describing individual occurrences and their 
#' attributes, ALA stores images, sounds and videos associated with a given 
#' record. `atlas_media` displays metadata for any and all of the media types. 
#'
#' @param request optional `data_request` object: generated by a call to
#' [galah_call()].
#' @param identify `data.frame`: generated by a call to
#' [galah_identify()].
#' @param filter `data.frame`: generated by a call to
#' [galah_filter()]
#' @param geolocate `string`: generated by a call to
#' [galah_geolocate()]
#' @param data_profile `string`: generated by a call to
#' [galah_apply_profile()]
#' @param download_dir `string`: path to directory where downloaded media will
#' be stored. DEPRECATED; use `collect_media()` instead.
#' @param refresh_cache `logical`: if set to `TRUE` and 
#' `galah_config(caching = TRUE)` then files cached from a previous query will 
#' be replaced by the current query. DEPRECATED; use `collect_media()` instead.
#' @details [atlas_media()] works by first finding all occurrence records
#' matching the filter which contain media, then downloading the metadata for the
#' media. To actually download the files themselves, use [collect_media()].
#' It may be beneficial when requesting a large number of records to show a progress
#' bar by setting `verbose = TRUE` in [galah_config()].
#' @return An object of class `tbl_df` and `data.frame` (aka a tibble) 
#' of metadata of the requested media.
#' @seealso [atlas_counts()] to find the number of records with media; but note this
#' is not necessarily the same as the number of media files, as each record can have
#' more than one media file associated with it (see examples section for how to do this).
#' 
#' @examples 
#' \dontrun{
#' # Download Regent Honeyeater records with multimedia attached
#' galah_call() |>
#'   galah_identify("Regent Honeyeater") |>
#'   galah_filter(year == 2011) |>
#'   atlas_media()
#' 
#' # Download multimedia
#' galah_call() |>
#'   galah_identify("Regent Honeyeater") |>
#'   galah_filter(year == 2011) |>
#'   atlas_media() |>
#'   collect_media(path = "folder/your-directory")
#' 
#' # Specify a single media type to download
#' galah_call() |>
#'   galah_identify("Eolophus Roseicapilla") |>
#'   galah_filter(multimedia == "Sound") |>
#'   atlas_media()
#' 
#' # It's good to check how many records have media files before downloading
#' galah_call() |>
#'   galah_filter(multimedia == c("Image", "Sound", "Video")) |>
#'   galah_group_by(multimedia) |>
#'   atlas_counts()
#'}
#' 
#' @export
atlas_media <- function(request = NULL, 
                        identify = NULL, 
                        filter = NULL, 
                        geolocate = NULL,
                        data_profile = NULL,
                        download_dir = NULL,
                        refresh_cache = FALSE
                        ) {
  
  if(getOption("galah_config")$atlas$region != "Australia"){
    abort("`atlas_media` is currently only supported for the Atlas of Living Australia")
  }
  
  if(!is.null(request)){
    check_data_request(request)
    current_call <- update_galah_call(request, 
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      data_profile = data_profile,
      download_dir = download_dir,
      refresh_cache = refresh_cache
    ) 
     
  }else{
    current_call <- galah_call(
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      data_profile = data_profile,
      download_dir = download_dir,
      refresh_cache = refresh_cache    
    )
  }
    
  # subset to available arguments
  custom_call <- current_call[
     names(current_call) %in% names(formals(atlas_media_internal))]
  class(custom_call) <- "data_request"
     
  # check for caching
  caching <- getOption("galah_config")$package$caching
  cache_file <- cache_filename("media", unlist(custom_call))
  if (caching && file.exists(cache_file) && !refresh_cache) {
    return(read_cache_file(cache_file))
  }
  
  # run function using do.call
  result <- do.call(atlas_media_internal, custom_call)
  if(is.null(result)){
    tibble()
  }else{
    attr(result, "data_type") <- "media"
    attr(result, "data_request") <- custom_call
    
    # if caching requested, save
    if (caching) {
      write_cache_file(object = result, 
                       data_type = "media",
                       cache_file = cache_file)
    }
    
    result  
  }
}


atlas_media_internal <- function(request, 
                                 identify, 
                                 filter,
                                 data_profile, 
                                 geolocate,
                                 download_dir,
                                 refresh_cache
                                 ) {
  # set basic information
  verbose <- getOption("galah_config")$package$verbose
  if(!is.null(download_dir)){
    inform("Argument `download_dir` is deprecated; use `collect_media()` instead")
  }
  if(refresh_cache){
    inform("Argument `refresh_cache` is deprecated; use `collect_media()` instead")
  }
  if(is_tibble(identify)){if(nrow(identify) < 1){identify <- NULL}}
  
  if (is.null(identify) & is.null(filter) & is.null(geolocate)) {
    abort("No filters have been provided")
  }
  
  # Check whether any filters are media-specific filters and
  # filter to records with image/sound/video
  
  ## NOTE: This approach doesn't work very well now that galah supports
  ## dplyr-style filtering. Ergo new version doesn't try to post-filter.
  ## The approach is valid, however, and could be resurrected with some
  ## clever use of quosures.
  
  # media_fields <- show_all_fields()[show_all_fields()$type == "media", "id", drop = TRUE]
  # filter_available <- filter$variable %in% media_fields
  # media_filter <- filter[filter_available, ]
  occ_filter <- rbind(
    filter,
    # filter[!(filter_available), ],
    galah_filter('multimedia' == c("Image", "Sound", "Video"))) # quoting here to pass R CMD check
  # Q: why `multimedia == image` and not `image != ""`  or `multimedia != ""`?
  
  # Make sure media ids are included in results
  occ_columns <-  galah_select(group = c("basic", "media"))

  if (verbose) { inform("Downloading records that contain media...") }
  
  occ <- occurrences_LA(
    identify = identify, 
    filter = occ_filter, 
    geolocate = geolocate, 
    data_profile = data_profile, 
    select = occ_columns)
    
  if(nrow(occ) < 1){
    return(tibble())
  }

  all_data <- search_media(occ)
 
  return(all_data)
}