% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gisco_get_units.R
\name{gisco_get_units}
\alias{gisco_get_units}
\title{Get geospatial units data from GISCO API}
\source{
\url{https://gisco-services.ec.europa.eu/distribution/v2/}
}
\usage{
gisco_get_units(
  id_giscoR = "nuts",
  unit = "ES4",
  mode = "sf",
  year = "2016",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  resolution = "20",
  spatialtype = "RG"
)
}
\arguments{
\item{id_giscoR}{Select the \code{unit} type to be downloaded.
Accepted values are "nuts", "countries" or "urban_audit".}

\item{unit}{Unit ID to be downloaded. See Details.}

\item{mode}{Controls the output of the function. Possible values are "sf"
or "df". See Value and Details.}

\item{year}{Release year of the file. One of "2001", "2006",
"2010", "2013", "2016" or "2020".}

\item{epsg}{projection of the map: 4-digit \href{https://epsg.io/}{EPSG code}.
One of:
\itemize{
\item "4258": ETRS89
\item "4326": WGS84
\item "3035": ETRS89 / ETRS-LAEA
\item "3857": Pseudo-Mercator
}}

\item{cache}{A logical whether to do caching. Default is \code{TRUE}. See
\strong{About caching}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source
.geojson file.}

\item{cache_dir}{A path to a cache directory. See \strong{About caching}.}

\item{verbose}{Logical, displays information. Useful for debugging,
default is \code{FALSE}.}

\item{resolution}{Resolution of the geospatial data. One of
\itemize{
\item "60": 1:60million
\item "20": 1:20million
\item "10": 1:10million
\item "03": 1:3million
\item "01": 1:1million
}}

\item{spatialtype}{Type of geometry to be returned: "RG", for \code{POLYGON} and
"LB" for \code{POINT}.}
}
\value{
A \code{sf} object on \code{mode = "sf"} or a dataframe on \code{mode = "df"}.
}
\description{
Download individual shapefiles of units. Unlike \code{\link[=gisco_get_countries]{gisco_get_countries()}},
\code{\link[=gisco_get_nuts]{gisco_get_nuts()}} or \code{\link[=gisco_get_urban_audit]{gisco_get_urban_audit()}}, that downloads a full
dataset and applies filters, \code{\link[=gisco_get_units]{gisco_get_units()}} downloads a single
shapefile for each unit.
}
\details{
The function can return a dataframe on \code{mode = "df"} or a \code{sf} object
on \code{mode = "sf"}.

In order to see the available \code{unit} ids with the required
combination of \verb{spatialtype, year}, first run the function on "df"
mode. Once that you get the data frame you can select the required ids
on the \code{unit} parameter.

On \code{mode = "df"} the only relevant parameters are \verb{spatialtype, year}.
}
\note{
Country-level files would be renamed on your \code{cache_dir}
to avoid naming conflicts with NUTS-0 datasets.

Please check the download and usage provisions on \code{\link[=gisco_attributions]{gisco_attributions()}}.
}
\section{About caching}{
You can set your \code{cache_dir} with \code{\link[=gisco_set_cache_dir]{gisco_set_cache_dir()}}.

Sometimes cached files may be corrupt. On that case, try re-downloading
the data setting \code{update_cache = TRUE}.

If you experience any problem on download, try to download the
corresponding .geojson file by any other method and save it on your
\code{cache_dir}. Use the option \code{verbose = TRUE} for debugging the API query.

For a complete list of files available check \link{gisco_db}.
}

\examples{
\dontshow{if (gisco_check_access()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
cities <- gisco_get_units(
  id_giscoR = "urban_audit",
  mode = "df",
  year = "2020"
)
VAL <- cities[grep("Valencia", cities$URAU_NAME), ]
#   Order from big to small
VAL <- VAL[order(as.double(VAL$AREA_SQM), decreasing = TRUE), ]

VAL.sf <- gisco_get_units(
  id_giscoR = "urban_audit",
  year = "2020",
  unit = VAL$URAU_CODE
)
# Provincia
Provincia <-
  gisco_get_units(
    id_giscoR = "nuts",
    unit = c("ES523"),
    resolution = "01"
  )

# Reorder
VAL.sf$URAU_CATG <- factor(VAL.sf$URAU_CATG, levels = c("F", "K", "C"))

# Plot
library(ggplot2)

ggplot(Provincia) +
  geom_sf(fill = "gray1") +
  geom_sf(data = VAL.sf, aes(fill = URAU_CATG)) +
  scale_fill_viridis_d() +
  labs(
    title = "Valencia",
    subtitle = "Urban Audit",
    fill = "Urban Audit\ncategory"
  )
}
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=gisco_get_countries]{gisco_get_countries()}}

Other political: 
\code{\link{gisco_bulk_download}()},
\code{\link{gisco_get_coastallines}()},
\code{\link{gisco_get_countries}()},
\code{\link{gisco_get_lau}()},
\code{\link{gisco_get_nuts}()},
\code{\link{gisco_get_postalcodes}()},
\code{\link{gisco_get_urban_audit}()}
}
\author{
dieghernan, \url{https://github.com/dieghernan/}
}
\concept{political}
