% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg.R
\name{logLik.gkwreg}
\alias{logLik.gkwreg}
\title{Extract Log-Likelihood from a Generalized Kumaraswamy Regression Model}
\usage{
\method{logLik}{gkwreg}(object, ...)
}
\arguments{
\item{object}{An object of class \code{"gkwreg"}, typically the result of a call
to \code{\link{gkwreg}}.}

\item{...}{Additional arguments, currently ignored by this method.}
}
\value{
An object of class \code{"logLik"} representing the maximized
log-likelihood value. It has the following attributes:
\itemize{
\item \code{df}: (numeric) The number of estimated parameters in the model
(coefficients).
\item \code{nobs}: (numeric) The number of observations used for fitting the model.
}
}
\description{
This function extracts the maximized log-likelihood value from a fitted Generalized
Kumaraswamy (GKw) regression model object (class \code{"gkwreg"}). The result is
returned as an object of class \code{"logLik"}, which includes attributes for
degrees of freedom and number of observations, suitable for use with model
selection criteria like AIC and BIC.
}
\details{
The log-likelihood value is typically computed during the model fitting process
(e.g., by \code{\link{gkwreg}}) and stored within the resulting object. This
method retrieves this stored value. If the value is not directly available, it
attempts to calculate it from the stored deviance (\eqn{logLik = -deviance / 2}).

The log-likelihood for a GKw family model with parameters \eqn{\theta} is
generally defined as the sum of the log-density contributions for each observation:
\deqn{l(\theta | y) = \sum_{i=1}^n \log f(y_i; \alpha_i, \beta_i, \gamma_i, \delta_i, \lambda_i)}
where \eqn{f(y; \dots)} is the probability density function (PDF) of the specific
distribution from the GKw family used in the model (determined by the \code{family}
argument in \code{gkwreg}), and parameters (\eqn{\alpha_i, \dots, \lambda_i}) may
depend on covariates.

The function also extracts the number of estimated parameters (\code{df}) and the
number of observations (\code{nobs}) used in the fit, storing them as attributes
of the returned \code{"logLik"} object, which is essential for functions like
\code{\link[stats]{AIC}} and \code{\link[stats]{BIC}}. It attempts to find \code{df}
and \code{nobs} from various components within the \code{object} if they are not
directly stored as \code{npar} and \code{nobs}.
}
\examples{
\donttest{
# Assume 'df' exists with response 'y' and predictors 'x1', 'x2', 'x3'
# and that rkw() is available and data is appropriate (0 < y < 1).
set.seed(123)
n <- 100
x1 <- runif(n)
x2 <- rnorm(n)
x3 <- factor(rbinom(n, 1, 0.4))
alpha <- exp(0.5 + 0.2 * x1)
beta <- exp(1.0 - 0.1 * x2 + 0.3 * (x3 == "1"))
y <- rkw(n, alpha = alpha, beta = beta) # Placeholder if rkw not available
y <- pmax(pmin(y, 1 - 1e-7), 1e-7)
df <- data.frame(y = y, x1 = x1, x2 = x2, x3 = x3)

# Fit a Kumaraswamy regression model
kw_reg <- gkwreg(y ~ x1 | x2 + x3, data = df, family = "kw")

# Extract log-likelihood object
ll <- logLik(kw_reg)

# Print the log-likelihood value (with attributes)
print(ll)

# Access the value directly
ll_value <- as.numeric(ll)
print(ll_value)

# Get the number of parameters (degrees of freedom)
df_model <- attr(ll, "df")
print(paste("Number of parameters:", df_model))

# Get the number of observations
nobs_model <- attr(ll, "nobs")
print(paste("Number of observations:", nobs_model))

# Use with AIC/BIC
AIC(kw_reg)
BIC(kw_reg)
}

}
\seealso{
\code{\link{gkwreg}}, \code{\link{AIC.gkwreg}}, \code{\link{BIC.gkwreg}},
\code{\link[stats]{logLik}}, \code{\link[stats]{AIC}}, \code{\link[stats]{BIC}}
}
\author{
Lopes, J. E.
}
\keyword{likelihood}
\keyword{log-likelihood}
\keyword{models}
