\name{EvolutionStrategy.int}
\alias{EvolutionStrategy.int}
\alias{print.EvolutionStrategy.int}

\title{ 
  Evolution Strategy with Integer Chromosomes
}
\description{
Uses evolution strategy to find the minima of a given cost function.
It evolves chromosomes with limited-range integers as codons.
}
\usage{
EvolutionStrategy.int(genomeLen, codonMin, codonMax, 
    genomeMin = rep.int(codonMin, genomeLen), 
    genomeMax = rep.int(codonMax, genomeLen), 
    suggestion = NULL, popSize=4, newPerGen = 4,
    iterations = 500, terminationCost = NA, 
    mutationChance = 1/(genomeLen+1), 
    monitorFunc = NULL, evalFunc, allowrepeat = TRUE, 
    showSettings = FALSE, verbose = FALSE, plapply = lapply)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{genomeLen}{
  Number of integers (i.e, codons) in chromosome.
}
  \item{codonMin}{
  Minimum integer value range for all codons.
}
  \item{codonMax}{
  Maximum integer value range for all codons.
}
  \item{genomeMin}{
  A vector of length \code{genomeLen} containing fine-grained control
  over each codon's minimum. Overrides \code{codonMin}.
}
  \item{genomeMax}{
  A vector of length \code{genomeLen} containing fine-grained control
  over each codon's maximum. Overrides \code{codonMax}.
}
  \item{suggestion}{
  A list of suggested chromosomes to be used in the initial population.
}
  \item{popSize}{
  Size of the population generated by mutating the parent.
}
  \item{newPerGen}{
  Number of the new randomly generated chromosome in each generation.
}
  \item{iterations}{
  Number of generations to evolve the population.
}
  \item{terminationCost}{
  Target cost. If the best chromosome's cost reaches this value,
   the algorithm terminates.
}
  \item{mutationChance}{
  The chance of a codon being mutated. It must be between 0 and 1.
}
  \item{monitorFunc}{
  A function that is called at each generation. Can be used to monitor evolution of population.
}
  \item{evalFunc}{
  The cost function.
}
  \item{allowrepeat}{
  Allows or forbids repeated integers in the chromosome.
}
  \item{showSettings}{
  Enables printing GA settings.
}
  \item{verbose}{
  Enables verbose debugging info.
}
  \item{plapply}{
  \code{lapply} function used for mapping chromosomes to the cost function.
  See details for parallelization tips.
}
}
\details{
%%  ~~ If necessary, more details than the description above ~~
\code{EvolutionStrategy.int} implements evolutionary strategy search algorithm with
chromosomes created from integer values in the range of \code{codonMin} to
\code{codonMax}. \code{genomeMin} and \code{genomeMax} allow fine-grained
control of range for individual codons.
It first creates an initial population, using suggested input
\code{suggestion} or a randomly generated chromosome.
Score of each chromosome is evaluated using the cost function
\code{costFunc}. If the best chromosome reaches
\code{terminationCost}, the algorithm terminates; 
otherwise only the best candidate is selected and mutated to create a new generation,
and the cycle is repeated.
This iteration continues until the required cost is reached
or the number of generations exceeds \code{iterations}.

At each generation, the supplied \code{monitorFunc} is called with a
list similar to \code{EvolutionStrategy.int} returning value as its argument. 

The \code{evalFunc} receives integer sequences and must return a numeric value.
The goal of optimization would be to find a chromosome which minimizes this value.

To parallelize cost function evaluation, set \code{plapply} to a parallelized
\code{lapply}, such as \code{mclapply} from package \code{parallel}. 
In functions that do not handle data dependencies such as \code{parLapply}, 
variables and functions required for correct execution of \code{evalFunc} 
must be exported to worker nodes before invoking \code{EvolutionStrategy.int}.
}
\value{


  A list containing information about 
  \code{settings}, \code{population}, and the \code{best} chromosome.

  \item{settings$genomeMin}{
    Minimum of each codon. 
  }

  \item{Settings$genomeMax}{
    Maximum of each codon. 
  }

  \item{settings$popSize}{
    Size of the population created using mutation.
  }

  \item{settings$newPerGen}{
    Number of the new randomly generated chromosome in each generation.
  }

  \item{settings$totalPopulation}{
    Size of the total population.
  }

  \item{settings$iterations}{
    Number of maximum generations.}

  \item{settings$suggestion}{
    Suggested chromosomes.}

  \item{settings$mutationChance}{
    Mutation chance.}

  \item{population$population}{
    The genomic data of the current population.}

  \item{population$evaluations}{
    Cost of the latest generation.}

  \item{population$best}{
    Historical cost of the best chromosomes.}

  \item{population$mean}{
    Historical mean cost of population.}

  \item{population$currentIteration}{
   Number of generations evolved until now.}
     
  \item{best$genome}{
    The best chromosome in integer sequence format.}

  \item{best$cost}{
    The cost of the best chromosome.}
}

% \author{
% %%  ~~who you are~~
% }
% \note{
% %%  ~~further notes~~
% }
%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{GrammaticalEvolution}}, 
\code{\link{GeneticAlg.int}}
}

\examples{

# define the evaluate function
evalfunc <- function(l) {
    # maximize the odd indices and minimize the even indices
    # no repeated values are allowed
    odd <- seq(1, 20, 2)
    even <- seq(2, 20, 2)
    err <- sum(l[even]) - sum(l[odd]);

    stopifnot(!any(duplicated(l))) # no duplication allowed

    return (err)
}

monitorFunc <- function(result) {
    cat("Best of gen: ", min(result$best$cost), "\n")
}

x <- EvolutionStrategy.int(genomeLen = 20, codonMin = 0, codonMax = 20,
                allowrepeat = FALSE, terminationCost = -110,
                monitorFunc = monitorFunc, evalFunc = evalfunc)

print(x)

best.result <- x$best$genome
print("Odds:")
print(sort(best.result[seq(1, 20, 2)]))
print("Evens:")
print(sort(best.result[seq(2, 20, 2)]))
}

% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%\keyword{ ~kwd1 }
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
