% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarize-growth.R
\name{SummarizeGrowth}
\alias{SummarizeGrowth}
\title{Summarize Growth Curves}
\usage{
SummarizeGrowth(data_t, data_n, t_trim = 0, bg_correct = "min",
  blank = NA)
}
\arguments{
\item{data_t}{A vector of timepoints (data_n must also
be provided and be the same length).}

\item{data_n}{A vector of cell counts or absorbance readings.}

\item{t_trim}{Measurements taken after this time should not be included
in fitting the curve. If stationary phase is variable,
this may give you a better fit. A value of 0 means no
trimming. Defaults to no trimming (0).}

\item{bg_correct}{The background correction method to use. No background
correction is performed for the default "none". Specifying
"min" subtracts the smallest value in a column from all the
rows in that column, and specifying "blank" subtracts
the values from the blank vector from the data_n vector.}

\item{blank}{A vector of absorbance readings from a blank well
(typically contains only media) used for background
correction. The corresponding blank value is subtracted
from the data_n vector for each timepoint. Defaults to NA.}
}
\value{
An object of type gcfit containing the "fitness" proxies,
                  as well as the input data and the fitted model.
}
\description{
This function finds the parameters that describe the input data's growth.
It does so by fitting the logistic curve to your growth curve measurements.
}
\details{
The logistic curve equation is
\deqn{N_t = \frac{N_0 K} {N_0 + (K-N_0)e^{-rt}}}{N_t = N_0 K / (N_0 + (K - N_0) exp(-rt))}
where \eqn{N_t} is the number
of cells (or the absorbance reading) at time t, \eqn{N_0} is the initial
cell count (or absorbance reading), K is the carrying capacity, and r is the
growth rate.

The fitness proxies returned are the parameters of the logistic equation
and the area under the curve (a measure that integrates the effects
of \eqn{N_0}, K, and r). See \code{\link{gcfit}} for more documentation on these.
}
\examples{
# We can check that the parameters that are found are the same
# as we use to generate fake experimental data. To do so, let's first
# generate the "experimental" data using the logistic equation,
# e.g., absorbance readings from a single well in a plate reader over time.

k_in <- 0.5   # the initial carrying capacity
n0_in <- 1e-5 # the initial absorbance reading
r_in <- 1.2   # the initial growth rate
N <- 50       # the number of "measurements" collected during the growth
              # curve experiment

data_t <- 0:N * 24 / N   # the times the measurements were made (in hours)
data_n <- NAtT(k = k_in, n0 = n0_in, r = r_in, t = data_t) # the measurements

# Now summarize the "experimental" growth data that we just generated
gc <- SummarizeGrowth(data_t, data_n)

# Get the possible metrics for fitness proxies
gc$vals$r           # growth rate is a common choice for fitness
gc$vals$t_gen       # doubling time, or generation time, is also common
gc$vals$k
gc$vals$n0
gc$vals$auc_l
gc$vals$auc_e
gc$vals$t_mid

# Compare the data with the fit visually by plotting it
plot(gc)

}
\seealso{
See the accompanying Vignette for an example of how to use and interpret
SummarizeGrowth. \url{bit.ly/1p7w6dJ}.

See also \code{\link{gcfit}}.
}

