% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_design_npe.R, R/gs_power_npe.R
\name{gs_design_npe}
\alias{gs_design_npe}
\alias{gs_power_design_npe}
\alias{gs_power_npe}
\title{Group sequential design computation with non-constant effect and information.}
\usage{
gs_design_npe(
  theta = 0.1,
  theta0 = 0,
  theta1 = theta,
  info = 1,
  info0 = NULL,
  info1 = NULL,
  info_scale = c("h0_h1_info", "h0_info", "h1_info"),
  alpha = 0.025,
  beta = 0.1,
  upper = gs_b,
  upar = qnorm(0.975),
  lower = gs_b,
  lpar = -Inf,
  test_upper = TRUE,
  test_lower = TRUE,
  binding = FALSE,
  r = 18,
  tol = 1e-06
)

gs_power_npe(
  theta = 0.1,
  theta0 = 0,
  theta1 = theta,
  info = 1,
  info0 = NULL,
  info1 = NULL,
  info_scale = c("h0_h1_info", "h0_info", "h1_info"),
  upper = gs_b,
  upar = qnorm(0.975),
  lower = gs_b,
  lpar = -Inf,
  test_upper = TRUE,
  test_lower = TRUE,
  binding = FALSE,
  r = 18,
  tol = 1e-06
)
}
\arguments{
\item{theta}{Natural parameter for group sequential design representing
expected cumulative drift at all analyses; used for power calculation.
It can be a scalar (constant treatment effect) or a vector (non-constant treatment effect).
The user must provide a value for \code{theta}.}

\item{theta0}{Natural parameter for null hypothesis.
It can be a scalar (constant treatment effect) or a vector (non-constant treatment effect).
The default is 0. If a value other than 0 is provided, it affects upper bound computation.}

\item{theta1}{Natural parameter for alternate hypothesis,
if needed for lower bound computation.
It can be a scalar (constant treatment effect) or a vector (non-constant treatment effect).
The default is the same as \code{theta}, which yields the usual beta-spending.
If set to 0, spending is 2-sided under the null hypothesis.}

\item{info}{Statistical information at all analyses for input \code{theta}.
It is a vector of positive numbers in increasing order.
The user must provide values corresponding to \code{theta}.}

\item{info0}{Statistical information under null hypothesis.
It is a vector of all positive numbers with increasing order.
Default is set to be the same as \code{info}.
If \code{info0} is different than \code{info}, it impacts null hypothesis bound calculation.}

\item{info1}{Statistical information under hypothesis used for futility bound calculation.
It is a vector of all positive numbers with increasing order.
Default is set to be the same as \code{info}.
If \code{info1} is different from \code{info}, it impacts futility bound calculation.}

\item{info_scale}{Information scale for calculation. Options are:
\itemize{
\item \code{"h0_h1_info"} (default): variance under both null and alternative hypotheses is used.
\item \code{"h0_info"}: variance under null hypothesis is used.
This is often used for testing methods that use local alternatives, such as the Schoenfeld method.
\item \code{"h1_info"}: variance under alternative hypothesis is used.
}}

\item{alpha}{One-sided Type I error.}

\item{beta}{Type II error.}

\item{upper}{Function to compute upper bound.}

\item{upar}{Parameters passed to \code{upper}.}

\item{lower}{Function to compare lower bound.}

\item{lpar}{parameters passed to \code{lower}.}

\item{test_upper}{Indicator of which analyses should include
an upper (efficacy) bound;
single value of \code{TRUE} (default) indicates all analyses; otherwise,
a logical vector of the same length as \code{info} should
indicate which analyses will have an efficacy bound.}

\item{test_lower}{Indicator of which analyses should include a lower bound;
single value of \code{TRUE} (default) indicates all analyses;
single value of \code{FALSE} indicated no lower bound; otherwise,
a logical vector of the same length as \code{info} should
indicate which analyses will have a lower bound.}

\item{binding}{Indicator of whether futility bound is binding;
default of \code{FALSE} is recommended.}

\item{r}{Integer value controlling grid for numerical integration as in
Jennison and Turnbull (2000); default is 18, range is 1 to 80.
Larger values provide larger number of grid points and greater accuracy.
Normally, \code{r} will not be changed by the user.}

\item{tol}{Tolerance parameter for boundary convergence (on Z-scale).}
}
\value{
A tibble with columns of
\itemize{
\item \code{analysis}: analysis index.
\item \code{bound}: either of value \code{"upper"} or \code{"lower"}, indicating the upper and lower bound.
\item \code{z}: the Z-score bounds.
\item \code{probability}: cumulative probability of crossing the bound at or before the analysis.
\item \code{theta}: same as the input.
\item \code{theta1}: same as the input.
\item \code{info}: statistical information at each analysis.
\itemize{
\item If it is returned by \code{gs_power_npe}, the \code{info}, \code{info0}, \code{info1} are same as the input.
\item If it is returned by \code{gs_design_npe}, the \code{info}, \code{info0}, \code{info1} are changed by a constant scale factor.
factor to ensure the design has power \code{1 - beta}.
}
\item \code{info0}: statistical information under the null at each analysis.
\item \code{info1}: statistical information under the alternative at each analysis.
\item \code{info_frac}: information fraction at each analysis, i.e., \code{info / max(info)}.
}
}
\description{
The following two functions allow a non-constant treatment effect over time,
but also can be applied for the usual homogeneous effect size designs.
They require treatment effect and statistical information at each analysis
as well as a method of deriving bounds, such as spending.
Initial bound types supported are spending bounds and fixed bounds.
These routines enables two things not available in the gsDesign package: 1)
non-constant effect, 2) more flexibility in boundary selection.

\code{gs_power_npe()} derives group sequential bounds and boundary crossing probabilities
for a design, given treatment effect and information at each analysis and the
method of deriving bounds, such as spending.

\code{gs_design_npe()} derives group sequential design size,
bounds and boundary crossing probabilities based on proportionate
information and effect size at analyses, as well as the
method of deriving bounds, such as spending.

The only differences in arguments between the two functions are the \code{alpha} and \code{beta}
parameters used in the \code{gs_design_npe()}.
}
\details{
The bound specifications (\code{upper}, \code{lower}, \code{upar}, \code{lpar}) of \code{gs_design_npe()}
will be used to ensure Type I error and other boundary properties are as specified.
See the help file of \code{gs_spending_bound()} for details on spending function.
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Validate if input info is a numeric vector  or NULL, if non-NULL validate if it
   is strictly increasing and positive.
   \item Validate if input info0 is a numeric vector or NULL, if non-NULL validate if it
    is strictly increasing and positive.
   \item Validate if input info1 is a numeric vector or NULL, if non-NULL validate if it
   is strictly increasing and positive.
   \item Validate if input theta is a real vector and has the same length as info.
   \item Validate if input theta1 is a real vector and has the same length as info.
   \item Validate if input test_upper and test_lower are logical and have the same length as info.
   \item Validate if input test_upper value is TRUE.
   \item Validate if input alpha and beta are positive and of length one.
   \item Validate if input alpha and beta are from the unit interval and alpha is smaller than beta.
   \item Initialize bounds, numerical integration grids, boundary crossing probabilities.
   \item Compute fixed sample size for desired power and Type I error.
   \item Find an interval for information inflation to give correct power using \code{gs_power_npe()}.
   \item
   \item If there is no interim analysis, return a tibble including Analysis time, upper bound, Z-value,
   Probability of crossing bound, theta, info0 and info1.
   \item If the design is a group sequential design, return a tibble of Analysis,
    Bound, Z, Probability,  theta, info, info0.
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}


\if{latex}{
 \itemize{
   \item Extract the length of input info as the number of interim analysis.
   \item Validate if input info0 is NULL, so set it equal to info.
   \item Validate if the length of inputs info and info0 are the same.
   \item Validate if input theta is a scalar, so replicate
   the value for all k interim analysis.
   \item Validate if input theta1 is NULL and if it is a scalar.
   If it is NULL, set it equal to input theta. If it is a scalar,
   replicate the value for all k interim analysis.
   \item Validate if input test_upper is a scalar,
   so replicate the value for all k interim analysis.
   \item Validate if input test_lower is a scalar,
   so replicate the value for all k interim analysis.
   \item Define vector a to be -Inf with
   length equal to the number of interim analysis.
   \item Define vector b to be Inf with
   length equal to the number of interim analysis.
   \item Define hgm1_0 and hgm1 to be NULL.
   \item Define upper_prob and lower_prob to be
   vectors of NA with length of the number of interim analysis.
   \item Update lower and upper bounds using \code{gs_b()}.
   \item If there are no interim analysis, compute probabilities
   of crossing upper and lower bounds
   using \code{h1()}.
   \item Compute cross upper and lower bound probabilities
   using \code{hupdate()} and \code{h1()}.
   \item Return a tibble of analysis number, bound, z-values,
   probability of crossing bounds,
   theta, theta1, info, and info0.
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}
}

\examples{
library(gsDesign)

# Example 1 ----
# gs_design_npe with single analysis
# Lachin book p 71 difference of proportions example
pc <- .28 # Control response rate
pe <- .40 # Experimental response rate
p0 <- (pc + pe) / 2 # Ave response rate under H0

# Information per increment of 1 in sample size
info0 <- 1 / (p0 * (1 - p0) * 4)
info <- 1 / (pc * (1 - pc) * 2 + pe * (1 - pe) * 2)

# Result should round up to next even number = 652
# Divide information needed under H1 by information per patient added
gs_design_npe(theta = pe - pc, info = info, info0 = info0)


# Example 2 ----
# gs_design_npe with with fixed bound
x <- gs_design_npe(
  alpha = 0.0125,
  theta = c(.1, .2, .3),
  info = (1:3) * 80,
  info0 = (1:3) * 80,
  upper = gs_b,
  upar = gsDesign::gsDesign(k = 3, sfu = gsDesign::sfLDOF, alpha = 0.0125)$upper$bound,
  lower = gs_b,
  lpar = c(-1, 0, 0)
)
x

# Same upper bound; this represents non-binding Type I error and will total 0.025
gs_power_npe(
  theta = rep(0, 3),
  info = (x |> dplyr::filter(bound == "upper"))$info,
  upper = gs_b,
  upar = (x |> dplyr::filter(bound == "upper"))$z,
  lower = gs_b,
  lpar = rep(-Inf, 3)
)

# Example 3 ----
# gs_design_npe with spending bound
# Design with futility only at analysis 1; efficacy only at analyses 2, 3
# Spending bound for efficacy; fixed bound for futility
# NOTE: test_upper and test_lower DO NOT WORK with gs_b; must explicitly make bounds infinite
# test_upper and test_lower DO WORK with gs_spending_bound
gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  info0 = (1:3) * 40,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_b,
  lpar = c(-1, -Inf, -Inf),
  test_upper = c(FALSE, TRUE, TRUE)
)

# one can try `info_scale = "h1_info"` or `info_scale = "h0_info"` here
gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  info0 = (1:3) * 30,
  info_scale = "h1_info",
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_b,
  lpar = c(-1, -Inf, -Inf),
  test_upper = c(FALSE, TRUE, TRUE)
)

# Example 4 ----
# gs_design_npe with spending function bounds
# 2-sided asymmetric bounds
# Lower spending based on non-zero effect
gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  info0 = (1:3) * 30,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfHSD, total_spend = 0.1, param = -1, timing = NULL)
)

# Example 5 ----
# gs_design_npe with two-sided symmetric spend, O'Brien-Fleming spending
# Typically, 2-sided bounds are binding
xx <- gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  binding = TRUE,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL)
)
xx

# Re-use these bounds under alternate hypothesis
# Always use binding = TRUE for power calculations
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  binding = TRUE,
  upper = gs_b,
  lower = gs_b,
  upar = (xx |> dplyr::filter(bound == "upper"))$z,
  lpar = -(xx |> dplyr::filter(bound == "upper"))$z
)

# Example 6 ----
# Default of gs_power_npe (single analysis; Type I error controlled)
gs_power_npe(theta = 0) |> dplyr::filter(bound == "upper")

# Example 7 ----
# gs_power_npe with fixed bound
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  upper = gs_b,
  upar = gsDesign::gsDesign(k = 3, sfu = gsDesign::sfLDOF)$upper$bound,
  lower = gs_b,
  lpar = c(-1, 0, 0)
)

# Same fixed efficacy bounds, no futility bound (i.e., non-binding bound), null hypothesis
gs_power_npe(
  theta = rep(0, 3),
  info = (1:3) * 40,
  upar = gsDesign::gsDesign(k = 3, sfu = gsDesign::sfLDOF)$upper$bound,
  lpar = rep(-Inf, 3)
) |>
  dplyr::filter(bound == "upper")

# Example 8 ----
# gs_power_npe with fixed bound testing futility only at analysis 1; efficacy only at analyses 2, 3
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  upper = gs_b,
  upar = c(Inf, 3, 2),
  lower = gs_b,
  lpar = c(qnorm(.1), -Inf, -Inf)
)

# Example 9 ----
# gs_power_npe with spending function bounds
# Lower spending based on non-zero effect
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfHSD, total_spend = 0.1, param = -1, timing = NULL)
)

# Same bounds, but power under different theta
gs_power_npe(
  theta = c(.15, .25, .35),
  info = (1:3) * 40,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfHSD, total_spend = 0.1, param = -1, timing = NULL)
)

# Example 10 ----
# gs_power_npe with two-sided symmetric spend, O'Brien-Fleming spending
# Typically, 2-sided bounds are binding
x <- gs_power_npe(
  theta = rep(0, 3),
  info = (1:3) * 40,
  binding = TRUE,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL)
)

# Re-use these bounds under alternate hypothesis
# Always use binding = TRUE for power calculations
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  binding = TRUE,
  upar = (x |> dplyr::filter(bound == "upper"))$z,
  lpar = -(x |> dplyr::filter(bound == "upper"))$z
)

# Example 11 ----
# Different values of `r` and `tol` lead to different numerical accuracy
# Larger `r` and smaller `tol` give better accuracy, but leads to slow computation
n_analysis <- 5
gs_power_npe(
  theta = 0.1,
  info = 1:n_analysis,
  info0 = 1:n_analysis,
  info1 = NULL,
  info_scale = "h0_info",
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_b,
  lpar = -rep(Inf, n_analysis),
  test_upper = TRUE,
  test_lower = FALSE,
  binding = FALSE,
  # Try different combinations of (r, tol) with
  # r in 6, 18, 24, 30, 35, 40, 50, 60, 70, 80, 90, 100
  # tol in 1e-6, 1e-12
  r = 6,
  tol = 1e-6
)
}
\author{
Keaven Anderson \email{keaven_anderson@merck.com}
}
