% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect_event.R
\name{detect_event}
\alias{detect_event}
\title{Detect heatwaves and cold-spells.}
\usage{
detect_event(
  data,
  x = t,
  y = temp,
  seasClim = seas,
  threshClim = thresh,
  threshClim2 = NA,
  minDuration = 5,
  minDuration2 = minDuration,
  joinAcrossGaps = TRUE,
  maxGap = 2,
  maxGap2 = maxGap,
  coldSpells = FALSE,
  protoEvents = FALSE,
  roundRes = 4
)
}
\arguments{
\item{data}{A data frame with at least four columns. In the default setting
(i.e. ommitting the arguments \code{x}, \code{y}, \code{seas}, and \code{thresh};
see immediately below), the data set is expected to have the headers \code{t},
\code{temp}, \code{seas}, and \code{thresh}. The \code{t}
column is a vector of dates of class \code{Date}, \code{temp} is the measured
variable (by default it is assumed to be temperature), \code{seas} is the seasonal
cycle daily climatology (366 days), and \code{thresh} is the seasonal cycle
daily threshold above which events may be detected. Data of the appropriate
format are created by the function \code{\link{ts2clm}}, but your own data
can be supplied if they meet the criteria specified by \code{\link{ts2clm}}
If the column names of \code{data} match those outlined here, the following
four arguments may be ignored.}

\item{x}{This column is expected to contain a vector of dates as per the
specification of \code{\link{ts2clm}}. If a column headed \code{t} is present in
the dataframe, this argument may be ommitted; otherwise, specify the name of
the column with dates here.}

\item{y}{This is a column containing the measurement variable. If the column
name differs from the default (i.e. \code{temp}), specify the name here.}

\item{seasClim}{This function will assume that the seasonal climatology column
is called \code{seas} as this matches the output of \code{\link{ts2clm}}.
If the column name for the seasonal climatology is different, provide that here.}

\item{threshClim}{The threshold climatology column should be called
\code{thresh}. If it is not, provide the name of the threshold column here.}

\item{threshClim2}{If one wishes to provide a second climatology threshold
filter for the more rigorous detection of events, a vector or column conataining
logical values (i.e. TRUE FALSE) should be provided here. By default this
argument is ignored. It's primary purpose is to allow for the inclusion of
tMin and tMax thresholds.}

\item{minDuration}{The minimum duration for acceptance of detected events.
The default is \code{5} days.}

\item{minDuration2}{The minimum duration for acceptance of events after
filtering by \code{threshClim} and \code{threshClim}. By default
\code{minDuration2 = minDuration} and is ignored if \code{threshClim2} has not
been specified.}

\item{joinAcrossGaps}{Boolean switch indicating whether to join events which
occur before/after a short gap as specified by \code{maxGap}. The default
is \code{TRUE}.}

\item{maxGap}{The maximum length of gap allowed for the joining of MHWs. The
default is \code{2} days.}

\item{maxGap2}{The maximum gap length after applying both thresholds.
By default \code{maxGap2 = maxGap} and is ignored if \code{threshClim2} has not
been specified.}

\item{coldSpells}{Boolean specifying if the code should detect cold events
instead of warm events. The default is \code{FALSE}. Please note that the
climatological thresholds for cold-spells are considered to be the inverse
of those for MHWs. For example, the default setting for the detection of
MHWs is \code{pctile = 90}, as seen in \code{\link{ts2clm}}. Should one want
to use \code{detect_event} for MCSs, this threshold would best be generated
in \code{\link{ts2clm}} by setting \code{pctile = 10} (see example below).
Any value may be used, but this is the setting used for the calculation of
MCSs in Schlegel et al. (2017a).}

\item{protoEvents}{Boolean specifying whether the full time series must be
returned as a long table, together with columns indicating whether or not the
threshold criterion (\code{threshCriterion}) and duration criterion (\code{durationCriterion})
have been exceeded, a column showing if a heatwave is present (i.e. both
\code{threshCriterion} and \code{durationCriterion} \code{TRUE}), and a
sequential number uniquely identifying the detected event. In this case,
the heatwave metrics will not be reported. The default is \code{FALSE}.}

\item{roundRes}{This argument allows the user to choose how many decimal places
the MHW metric outputs will be rounded to. Default is 4. To
prevent rounding set \code{roundRes = FALSE}. This argument may only be given
numeric values or FALSE.}
}
\value{
The function will return a list of two data.tables (see \code{data.table}),
\code{climatology} and \code{event}, which are, surprisingly, the climatology
and event results, respectively. The climatology contains the full time series of
daily temperatures, as well as the the seasonal climatology, the threshold
and various aspects of the events that were detected. The software was
designed for detecting extreme thermal events, and the units specified below
reflect that intended purpose. However, various other kinds of extreme
events may be detected according to the specifications, and if that is the
case, the appropriate units need to be determined by the user.

The \code{climatology} results will contain the same column produced by
\code{\link{ts2clm}} as well as the following:
  \item{threshCriterion}{Boolean indicating if \code{temp} exceeds
  \code{thresh}.}
  \item{durationCriterion}{Boolean indicating whether periods of consecutive
  \code{threshCriterion} are >= \code{min_duration}.}
  \item{event}{Boolean indicating if all criteria that define an extreme event
  are met.}
  \item{event_no}{A sequential number indicating the ID and order of
  occurence of the events.}

The \code{event} results are summarised using a range of event metrics:
  \item{event_no}{A sequential number indicating the ID and order of
  the events.}
  \item{index_start}{Start index of event.}
  \item{index_end}{End index of event.}
  \item{duration}{Duration of event [days].}
  \item{date_start}{Start date of event [date].}
  \item{date_end}{End date of event [date].}
  \item{date_peak}{Date of event peak [date].}
  \item{intensity_mean}{Mean intensity [deg. C].}
  \item{intensity_max}{Maximum (peak) intensity [deg. C].}
  \item{intensity_var}{Intensity variability (standard deviation) [deg. C].}
  \item{intensity_cumulative}{Cumulative intensity [deg. C x days].}
  \item{rate_onset}{Onset rate of event [deg. C / day].}
  \item{rate_decline}{Decline rate of event [deg. C / day].}

\code{intensity_max_relThresh}, \code{intensity_mean_relThresh},
\code{intensity_var_relThresh}, and \code{intensity_cumulative_relThresh}
are as above except relative to the threshold (e.g., 90th percentile) rather
than the seasonal climatology.

\code{intensity_max_abs}, \code{intensity_mean_abs}, \code{intensity_var_abs}, and
\code{intensity_cumulative_abs} are as above except as absolute magnitudes
rather than relative to the seasonal climatology or threshold.

Note that \code{rate_onset} and \code{rate_decline} will return \code{NA}
when the event begins/ends on the first/last day of the time series. This
may be particularly evident when the function is applied to large gridded
data sets. Although the other metrics do not contain any errors and
provide sensible values, please take this into account in its
interpretation.
}
\description{
Applies the Hobday et al. (2016) marine heat wave definition to an input time
series of a given value (usually, but not necessarily limited to, temperature)
along with a daily date vector and pre-calculated seasonal and threshold
climatologies, which may either be created with \code{\link{ts2clm}} or some
other means.
}
\details{
\enumerate{
\item This function assumes that the input time series consists of continuous
daily values with few missing values. Time ranges which start and end
part-way through the calendar year are supported. The accompanying function
\code{\link{ts2clm}} aids in the preparation of a time series that is
suitable for use with \code{detect_event}, although this may also be accomplished
'by hand' as long as the criteria are met as discussed in the documentation
to \code{\link{ts2clm}}.
\item The calculation of onset and decline rates assumes that the events
started a half-day before the start day and ended a half-day after the
end-day. This is consistent with the duration definition as implemented,
which assumes duration = end day - start day + 1. An event that is already
present at the beginning of a time series, or an event that is still present
at the end of a time series, will report the rate of onset or the rate of
decline as \code{NA}, as it is impossible to know what the temperature half a
day before or after the start or end of the event is.
\item For the purposes of event detection, any missing temperature values not
interpolated over (through optional \code{maxPadLength} in \code{\link{ts2clm}})
will be set equal to the seasonal climatology. This means they will trigger
the end/start of any adjacent temperature values which satisfy the event
definition criteria.
\item If the code is used to detect cold events (\code{coldSpells = TRUE}),
then it works just as for heat waves except that events are detected as
deviations below the (100 - pctile)th percentile (e.g., the 10th instead of
90th) for at least 5 days. Intensities are reported as negative values and
represent the temperature anomaly below climatology.
}
The original Python algorithm was written by Eric Oliver, Institute for
Marine and Antarctic Studies, University of Tasmania, Feb 2015, and is
documented by Hobday et al. (2016). The marine cold spell option was
implemented in version 0.13 (21 Nov 2015) of the Python module as a result
of our preparation of Schlegel et al. (2017), wherein the cold events
receive a brief overview.
}
\examples{
res_clim <- ts2clm(sst_WA, climatologyPeriod = c("1983-01-01", "2012-12-31"))
out <- detect_event(res_clim)
# show a portion of the climatology:
out$climatology[1:10, ]
# show some of the heat waves:
out$event[1:5, 1:10]

# Or if one wants to calculate MCSs
res_clim <- ts2clm(sst_WA, climatologyPeriod = c("1983-01-01", "2012-12-31"),
                   pctile = 10)
out <- detect_event(res_clim, coldSpells = TRUE)
# show a portion of the climatology:
out$climatology[1:10, ]
# show some of the cold-spells:
out$event[1:5, 1:10]

# It is also possible to give two separate sets of threshold criteria

# To use a second static threshold we first use the exceedance function
thresh_19 <- exceedance(sst_Med, threshold = 19, minDuration = 10, maxGap = 0)$threshold
# Then we use that output when detecting our events
events_19 <- detect_event(ts2clm(sst_Med, climatologyPeriod = c("1982-01-01", "2011-12-31")),
                         threshClim2 = thresh_19$exceedance, minDuration2 = 10, maxGap2 = 0)

# If we want to use two different percentile thresholds we use detect_event
thresh_95 <- detect_event(ts2clm(sst_Med, pctile = 95,
                                 climatologyPeriod = c("1982-01-01", "2011-12-31")),
                          minDuration = 2, maxGap = 0)$climatology
# Then we use that output when detecting our events
events_95 <- detect_event(ts2clm(sst_Med, climatologyPeriod = c("1982-01-01", "2011-12-31")),
                          threshClim2 = thresh_95$event, minDuration2 = 2, maxGap2 = 0)

}
\references{
Hobday, A.J. et al. (2016). A hierarchical approach to defining
marine heatwaves, Progress in Oceanography, 141, pp. 227-238,
doi:10.1016/j.pocean.2015.12.014

Schlegel, R. W., Oliver, C. J., Wernberg, T. W., Smit, A. J. (2017).
Nearshore and offshore co-occurrences of marine heatwaves and cold-spells.
Progress in Oceanography, 151, pp. 189-205, doi:10.1016/j.pocean.2017.01.004
}
\author{
Albertus J. Smit, Robert W. Schlegel, Eric C. J. Oliver
}
