% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Estimate.R
\name{id_estimate}
\alias{id_estimate}
\title{Estimate an \code{idealstan} model}
\usage{
id_estimate(idealdata = NULL, model_type = 2, use_subset = FALSE,
  sample_it = FALSE, subset_group = NULL, subset_person = NULL,
  sample_size = 20, nchains = 4, niters = 2000, use_vb = FALSE,
  nfix = 3, restrict_params = "person", pin_vals = NULL,
  restrict_ind_high = NULL, restrict_ind_low = NULL,
  restrict_type = "constrain_twoway", fixtype = "vb",
  warmup = floor(niters/2), ncores = 4, auto_id = FALSE,
  discrim_reg_sd = 3, discrim_miss_sd = 3, person_sd = 1,
  diff_reg_sd = 4, diff_miss_sd = 4, restrict_sd = 4,
  restrict_low_bar = 0, restrict_high_bar = 0, restrict_alpha = 4,
  restrict_beta = 1, ...)
}
\arguments{
\item{idealdata}{An object produced by the \code{\link{id_make}} containing a score/vote matrix for use for estimation & plotting}

\item{model_type}{An integer reflecting the kind of model to be estimated. See below.}

\item{use_subset}{Whether a subset of the legislators/persons should be used instead of the full response matrix}

\item{sample_it}{Whether or not to use a random subsample of the response matrix. Useful for testing.}

\item{subset_group}{If person/legislative data was included in the \code{\link{id_make}} function, then you can subset by
any value in the \code{$group} column of that data if \code{use_subset} is \code{TRUE}.}

\item{subset_person}{A list of character values of names of persons/legislators to use to subset if \code{use_subset} is 
\code{TRUE} and person/legislative data was included in the \code{\link{id_make}} function with the required \code{$person.names}
column}

\item{sample_size}{If \code{sample_it} is \code{TRUE}, this value reflects how many legislators/persons will be sampled from
the response matrix}

\item{nchains}{The number of chains to use in Stan's sampler. Minimum is one. See \code{\link[rstan]{stan}} for more info.}

\item{niters}{The number of iterations to run Stan's sampler. Shouldn't be set much lower than 500. See \code{\link[rstan]{stan}} for more info.}

\item{use_vb}{Whether or not to use Stan's variational Bayesian inference engine instead of full Bayesian inference. Pros: it's much faster.
Cons: it's not quite as accurate. See \code{\link[rstan]{vb}} for more info.}

\item{nfix}{An integer specifying the number of parameters to constrain (for both high and low) if \code{fixtype} is set to \code{'vb'}}

\item{restrict_params}{Determines which parameters in the model are used for identification. If \code{fixtype} is 'vb', can be
either 'person' (person/legislator ideal points) or 'items' (discrimination parameters for items/bills). Otherwise, the value must be one
of the available classes of parameters: 'person' for persons/legislators, 'discrim_miss' for the missing-inflated item/bill discrimination
parameters, or 'discrim_reg' for the non-inflated item/bill discrimination parameters.}

\item{pin_vals}{If \code{fixtype} is set to 'pinned', then a vector of numeric values of which to pin the legislators/persons or 
bills/items should be given.}

\item{restrict_ind_high}{If \code{fixtype} is not "vb", the particular indices of legislators/persons or bills/items to constrain high}

\item{restrict_ind_low}{If \code{fixtype} is not "vb", the particular indices of legislators/persons or bills/items to constrain low. 
(Note: not used if values are pinned).}

\item{restrict_type}{Determines whether constraints will only be set on one direction 'constrain_oneway' or both high/low
constrains 'constrain_twoway'. Ignored if \code{fixtype} is equal to 'pinned'.}

\item{fixtype}{Sets the particular kind of identification used on the model, could be one of 'vb','constrained' or 'pinned'.
See details for more information.}

\item{warmup}{The number of iterations to use to calibrate Stan's sampler on a given model. Shouldn't be less than 100. 
See \code{\link[rstan]{stan}} for more info.}

\item{ncores}{The number of cores in your computer to use for parallel processing in the Stan engine. 
See \code{\link[rstan]{stan}} for more info.}

\item{auto_id}{If \code{TRUE}, a variational Bayesian version of your model will be run using \code{\link[rstan]{vb}} in a 
non-identified form. Depending on what is set via \code{restrict_type} and \code{restrict_params}, bills/items or 
persons/legislators will be used to constrain and identify the model.}

\item{discrim_reg_sd}{Set the prior standard deviation of the bimodal prior for the discrimination parameters for the non-inflated model.}

\item{discrim_miss_sd}{Set the prior standard deviation of the bimodal prior for the discrimination parameters for the inflated model.}

\item{person_sd}{Set the prior standard deviation for the legislators (persons) parameters}

\item{diff_reg_sd}{Set the prior standard deviation for the bill (item) intercepts for the non-inflated model.}

\item{diff_miss_sd}{Set the prior standard deviation for the bill (item) intercepts for the inflated model.}

\item{restrict_sd}{Set the prior standard deviation for constrained parameters}

\item{restrict_low_bar}{Set the constraint threshold for constrained parameters (parameter must be lower than this bar and no greater than zero)}

\item{restrict_high_bar}{Set the constraint threshold for constrained parameters (parameter must be higher than this bar and no less than zero)}

\item{restrict_alpha}{This is the scale (alpha) parameter passed to the gamma prior if exactly two item/person parameters are constrained, each high or low. The gamma prior pushes these two
polar parameters apart. A higher value will push these two poles farther apart, which will help identification.}

\item{restrict_beta}{The beta (shape) parameter passed to the gamma prior if exactly two item/person parameters are constrained, each high or low. The gamma prior pushes these two
polar parameters apart.}

\item{...}{Additional parameters passed on to Stan's sampling engine. See \code{\link[rstan]{stan}} for more information.}
}
\value{
A fitted \code{\link{idealstan}} object that contains posterior samples of all parameters either via full Bayesian inference
or a variational approximation if \code{use_vb} is set to \code{TRUE}. This object can then be passed to the plotting functions for further analysis.
}
\description{
This function will take a pre-processed \code{idealdata} vote/score matrix and run one of the available IRT models on the data using
Stan's MCMC engine.
}
\details{
To run an IRT ideal point model, you must first pre-process your data using the \code{\link{id_make}} function. Be sure to specify the correct options for the
kind of model you are going to run: if you want to run an ordinal and/or an inflated model, the data needs to be processed differently.
As of this version of \code{idealstan}, the following model types are available:
\enumerate{
  \item IRT 2-PL (binary response) ideal point model, no absence inflation
  \item IRT 2-PL model (binary response) with absence inflation
  \item Ordinal IRT (rating scale) model incorporating abstentions as middle category, no absence inflation
  \item Ordinal IRT (rating scale) model incorporating abstentions as middle category with absence inflation
}
Additional models are available but have not yet been tested. You can find them by searching the included stan files for more info.
}
\section{Identification}{

Identifying IRT models is challenging, and ideal point models are still more challenging because the discrimination parameters are not constrained.
As a result, more care must be taken to obtain estimates that are the same regardless of starting values. 
The parameter \code{fixtype} enables you to change the type of identification used. The default, 'vb', does not require any further
information from you in order for the model to be fit. In this version of identification, an unidentified model is run using
variational Bayesian inference (see \code{\link[rstan]{vb}}). If \code{restrict_type} is set to 'constrain_oneway', then the \code{nfix}
highest legislators/persons (if \code{restrict_params} is 'person') or bills/items (if \code{restrict_params} equals 'items') 
are used to constrain and identify the model. If \code{restrict_type} is set to 'constrain_twoway', then \code{nfix} highest and lowest legislators/persons
or bills/items are used to constrain and identify the model.
In addition, if \code{fixtype='vb'} is used, \code{auto_id} can be set to \code{TRUE}. This will run additional variational Bayesian models
using the identification achieved and will see if the signs of the estimated parameters are at least 90 percent in the same direction. If so, 
the model is considered identified. If not, the function will re-run and will increase \code{nfix} by one to see if that will identify the model, 
ad infinitum.
If \code{fixtype} is set to 'constrained', then identification is achieved by constraining a specified list of legislators/persons or
bills/items. If \code{restrict_type} is 'constrain_oneway', then the indices of all constrained legislators/persons or bills/items
should be included as the row or column indices of these parameters in the response matrix as a vector in \code{restrict_ind_high}.
If \code{restrict_type} is 'constrain_twoway,' then the indices of high constrained parameters should go in 
\code{restrict_ind_high} and the indices of low constrained parameters in \code{restrict_ind_low}. The numbers of high and low
constrained parameters should be equal. To pick the parameter to constrain, set \code{restrict_params} to 'person' for legislators/persons,
'discrim_miss' for absence-inflated bill discrimination parameters, and 'discrim_reg' for non-inflated bill discrimination parameters.
If \code{fixtype} is set to 'pinned', then identification is achieved via pinning with very tight priors a set of legislators/persons or bills/items.
The indices of legislators/persons (i.e. row indices in the response matrix) or bills/items (column indices in the response matrix) should be passed as a vector
to \code{restrict_ind_high}, while \code{restrict_ind_low} should be left blank. The particular values to pin these parameters is passed as a 
numeric vector to \code{pin_vals}.
}

\examples{
# First we can simulate data for an IRT 2-PL model that is inflated for missing data
library(ggplot2)
library(dplyr)

# This code will take at least a few minutes to run 
\dontrun{
bin_irt_2pl_abs_sim <- id_sim_gen(ordinal=FALSE,
                                  absence=TRUE,
                                  absence_diff_mean=0)

# Now we can put that directly into the id_estimate function 
# to get full Bayesian posterior estimates
# We will constrain discrimination parameters 
# for identification purposes based on the true simulated values

bin_irt_2pl_abs_est <- id_estimate(bin_irt_2pl_abs_sim,
                       model_type=2,
                       restrict_ind_high = 
                       sort(bin_irt_2pl_abs_sim@simul_data$true_reg_discrim,
                       decreasing=TRUE,
                       index=TRUE)$ix[1:3],
                       restrict_ind_low = 
                       sort(bin_irt_2pl_abs_sim@simul_data$true_reg_discrim,
                       decreasing=FALSE,
                       index=TRUE)$ix[1:3],
                       restrict_params = 'discrim_reg', 
                       restrict_type = 'constrain_twoway',
                       fixtype='constrained',
                       ncores=2,
                       nchains=2)
                                   
# We can now see how well the model recovered the true parameters

id_sim_coverage(bin_irt_2pl_abs_est) \%>\% 
         bind_rows(.id='Parameter') \%>\% 
         ggplot(aes(y=avg,x=Parameter)) +
           stat_summary(fun.args=list(mult=1.96)) + 
           theme_minimal()
 }

# In most cases, we will use pre-existing data 
# and we will need to use the id_make function first
# We will use the full rollcall voting data 
# from the 114th Senate as a rollcall object

data('senate114')

# Running this model will take at least a few minutes, even with 
# variational inference (use_vb=T) turned on
\dontrun{

to_idealstan <-   id_make(score_data = senate114,
                          ordinal = FALSE,
                          include_pres=FALSE)

sen_est <- id_estimate(senate_data,
model_type = 2,
use_vb = TRUE,
restrict_type='constrain_oneway',
restrict_params='person',
restrict_ind_high = which(row.names(senate114$votes[-1,])=='SASSE (R NE)'),
auto_id=FALSE,
fixtype='constrained')

# After running the model, we can plot 
# the results of the person/legislator ideal points

id_plot_legis(sen_est)
}

}
\references{
\enumerate{
   \item Clinton, J., Jackman, S., & Rivers, D. (2004). The Statistical Analysis of Roll Call Data. \emph{The American Political Science Review}, 98(2), 355-370. doi:10.1017/S0003055404001194
   \item Bafumi, J., Gelman, A., Park, D., & Kaplan, N. (2005). Practical Issues in Implementing and Understanding Bayesian Ideal Point Estimation. \emph{Political Analysis}, 13(2), 171-187. doi:10.1093/pan/mpi010
}
}
\seealso{
\code{\link{id_make}} for pre-processing data,
\code{\link{id_plot_legis}} for plotting results,
\code{\link{summary}} for obtaining posterior quantiles,
\code{\link{posterior_predict}} for producing predictive replications.
}
