% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/irlba.R
\name{irlba}
\alias{irlba}
\title{Find a few approximate largest singular values and corresponding
singular vectors of a matrix.}
\usage{
irlba(A, nv = 5, nu, maxit = 1000, work = nv + 5, reorth = TRUE,
  tol = 1e-05, v = NULL, right_only = FALSE, verbose = FALSE, scale,
  center, du, ds, dv, shift, mult)
}
\arguments{
\item{A}{numeric real- or complex-valued matrix or real-valued sparse matrix.}

\item{nv}{number of right singular vectors to estimate.}

\item{nu}{number of left singular vectors to estimate (defaults to \code{nv}).}

\item{maxit}{maximum number of iterations.}

\item{work}{working subspace dimension, larger values can speed convergence at the cost of more memory use.}

\item{reorth}{logical value indicating full \code{TRUE} or cheaper one-sided \code{FALSE} reorthogonalization.}

\item{tol}{convergence is determined when \eqn{\|AV - US\| < tol\|A\|}{||AV - US|| < tol*||A||}, where the spectral norm ||A|| is approximated by the largest estimated singular value, and U, V, S are the matrices corresponding to the estimated left and right singular vectors, and diagonal matrix of estimated singular values, respectively.}

\item{v}{optional starting vector or output from a previous run of \code{irlba} used to restart the algorithm from where it left off (see the notes).}

\item{right_only}{logical value indicating return only the right singular vectors (\code{TRUE}) or both sets of vectors (\code{FALSE}).}

\item{verbose}{logical value that when \code{TRUE} prints status messages during the computation.}

\item{scale}{optional column scaling vector whose values divide each column of \code{A}; must be as long as the number of columns of \code{A} (see notes).}

\item{center}{optional column centering vector whose values are subtracted from each column of \code{A}; must be as long as the number of columns of \code{A} and may not be used together with the deflation options below (see notes).}

\item{du}{optional subspace deflation vector (see notes).}

\item{ds}{optional subspace deflation scalar (see notes).}

\item{dv}{optional subspace deflation vector (see notes).}

\item{shift}{optional shift value (square matrices only, see notes).}

\item{mult}{optional custom matrix multiplication function (default is `\%*\%`, see notes).}
}
\value{
Returns a list with entries:
\itemize{
  \item{d}{ max(nu, nv) approximate singular values}
  \item{u}{ nu approximate left singular vectors (only when right_only=FALSE)}
  \item{v}{ nv approximate right singular vectors}
  \item{iter}{ The number of Lanczos iterations carried out}
  \item{mprod}{ The total number of matrix vector products carried out}
}
}
\description{
The augmented implicitly restarted Lanczos bi-diagonalization (IRLBA)
algorithm finds a few approximate largest singular values and corresponding
singular vectors of a sparse or dense matrix using a method of Baglama and
Reichel.  It is a fast and memory-efficient way to compute a partial SVD.
}
\note{
The syntax of \code{irlba} partially follows \code{svd}, with an important
exception. The usual R \code{svd} function always returns a complete set of
singular values, even if the number of singular vectors \code{nu} or \code{nv}
is set less than the maximum. The \code{irlba} function returns a number of
estimated singular values equal to the maximum of the number of specified
singular vectors \code{nu} and \code{nv}.

Use the optional \code{scale} parameter to implicitly scale each column of
the matrix \code{A} by the values in the \code{scale} vector, computing the
truncated SVD of the column-scaled \code{sweep(A,2,scale,FUN=`/`)}, or
equivalently, \code{A \%*\% diag(1/scale)}, without explicitly forming the
scaled matrix. \code{scale} must be a non-zero vector of length equal
to the number of columns of \code{A}.

Use the optional \code{center} parameter to implicitly subtract the values
in the \code{center} vector from each column of \code{A}, computing the
truncated SVD of \code{sweep(A,2,center,FUN=`-`)},
without explicitly forming the centered matrix. This option may not be
used together with the general rank 1 deflation options. \code{center}
must be a vector of length equal to the number of columns of \code{A}.
This option may be used to efficiently compute principal components without
explicitly forming the centered matrix (which can, importantly, preserve
sparsity in the matrix). See the examples.

Use the optional deflation parameters to compute the rank-one deflated
SVD of \eqn{A - ds \cdot du dv^T}{A - ds*du \%*\% t(dv)}, where
\eqn{du^T A - ds\cdot dv^T = 0}{t(du) \%*\% A - ds * t(dv) == 0}. For
example, the triple \code{ds,du,dv} may be a known singular value
and corresponding singular vectors. Or \code{ds=1} and \code{dv}
and \code{du} represent a vector of column means of A and of ones,
respectively. This is a rarely used option, but it is used internally
by the \code{center} option and the two sets of parameters are
prevented from being used in combination.

Specify an optional alternative matrix multiplication operator in the
\code{mult} parameter. \code{mult} must be a function of two arguments,
and must handle both cases where one argument is a vector and the other
a matrix. See the examples. Special care must be taken when deflation
is also desired; see the package vignette for details.

Use the \code{v} option to supply a starting vector for the iterative
method. A random vector is used by default. Optionally set \code{v} to
the ouput of a previous run of \code{irlba} to restart the method, adding
additional singular values/vectors without recomputing the already computed
subspace.
}
\examples{
set.seed(1)

A <- matrix(runif(200),nrow=20)
S <- irlba(A)
S$d

# Compare with svd
svd(A)$d[1:5]

# Principal components
P <- irlba(A, nv=1, center=colMeans(A))

# Compare with prcomp (might vary up to sign)
cbind(P$v, prcomp(A)$rotation[,1])

# A custom matrix multiplication function that scales the columns of A
# (cf the scale option). This function scales the columns of A to unit norm.
col_scale <- sqrt(apply(A,2,crossprod))
mult <- function(x,y)
        {
          # check if x is a plain, row or column vector
          if (is.vector(x) || ncol(x)==1 || nrow(x)==1)
          {
            return((x \%*\% y)/col_scale)
          }
          # else x is the matrix
          x \%*\% (y/col_scale)
        }
irlba(A, 3, mult=mult)$d

# Compare with:
irlba(A, 3, scale=col_scale)$d

# Compare with:
svd(sweep(A,2,col_scale,FUN=`/`))$d[1:3]
}
\references{
Augmented Implicitly Restarted Lanczos Bidiagonalization Methods, J. Baglama and L. Reichel, SIAM J. Sci. Comput. 2005.
}
\seealso{
\code{\link{svd}}, \code{\link{prcomp}}, \code{\link{partial_eigen}}
}

