% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/losses.R
\name{loss_binary_focal_crossentropy}
\alias{loss_binary_focal_crossentropy}
\title{Computes focal cross-entropy loss between true labels and predictions.}
\usage{
loss_binary_focal_crossentropy(
  y_true,
  y_pred,
  apply_class_balancing = FALSE,
  alpha = 0.25,
  gamma = 2,
  from_logits = FALSE,
  label_smoothing = 0,
  axis = -1L,
  ...,
  reduction = "sum_over_batch_size",
  name = "binary_focal_crossentropy",
  dtype = NULL
)
}
\arguments{
\item{y_true}{Ground truth values, of shape \verb{(batch_size, d0, .. dN)}.}

\item{y_pred}{The predicted values, of shape \verb{(batch_size, d0, .. dN)}.}

\item{apply_class_balancing}{A bool, whether to apply weight balancing on the
binary classes 0 and 1.}

\item{alpha}{A weight balancing factor for class 1, default is \code{0.25} as
mentioned in reference \href{https://arxiv.org/pdf/1708.02002}{Lin et al., 2018}.  The weight for class 0 is
\code{1.0 - alpha}.}

\item{gamma}{A focusing parameter used to compute the focal factor, default is
\code{2.0} as mentioned in the reference
\href{https://arxiv.org/pdf/1708.02002}{Lin et al., 2018}.}

\item{from_logits}{Whether to interpret \code{y_pred} as a tensor of
\href{https://en.wikipedia.org/wiki/Logit}{logit} values. By default, we
assume that \code{y_pred} are probabilities (i.e., values in \verb{[0, 1]}).}

\item{label_smoothing}{Float in \verb{[0, 1]}. When \code{0}, no smoothing occurs.
When > \code{0}, we compute the loss between the predicted labels
and a smoothed version of the true labels, where the smoothing
squeezes the labels towards \code{0.5}.
Larger values of \code{label_smoothing} correspond to heavier smoothing.}

\item{axis}{The axis along which to compute crossentropy (the features axis).
Defaults to \code{-1}.}

\item{...}{For forward/backward compatability.}

\item{reduction}{Type of reduction to apply to the loss. In almost all cases
this should be \code{"sum_over_batch_size"}. Supported options are
\code{"sum"}, \code{"sum_over_batch_size"}, \code{"mean"},
\code{"mean_with_sample_weight"} or \code{NULL}. \code{"sum"} sums the loss,
\code{"sum_over_batch_size"} and \code{"mean"} sum the loss and divide by the
sample size, and \code{"mean_with_sample_weight"} sums the loss and
divides by the sum of the sample weights. \code{"none"} and \code{NULL}
perform no aggregation. Defaults to \code{"sum_over_batch_size"}.}

\item{name}{Optional name for the loss instance.}

\item{dtype}{The dtype of the loss's computations. Defaults to \code{NULL}, which
means using \code{config_floatx()}. \code{config_floatx()} is a
\code{"float32"} unless set to different value
(via \code{config_set_floatx()}). If a \code{keras$DTypePolicy} is
provided, then the \code{compute_dtype} will be utilized.}
}
\value{
Binary focal crossentropy loss value
with shape = \verb{[batch_size, d0, .. dN-1]}.
}
\description{
According to \href{https://arxiv.org/pdf/1708.02002}{Lin et al., 2018}, it
helps to apply a focal factor to down-weight easy examples and focus more on
hard examples. By default, the focal tensor is computed as follows:

\code{focal_factor = (1 - output)^gamma} for class 1
\code{focal_factor = output^gamma} for class 0
where \code{gamma} is a focusing parameter. When \code{gamma} = 0, there is no focal
effect on the binary crossentropy loss.

If \code{apply_class_balancing == TRUE}, this function also takes into account a
weight balancing factor for the binary classes 0 and 1 as follows:

\code{weight = alpha} for class 1 (\code{target == 1})
\code{weight = 1 - alpha} for class 0
where \code{alpha} is a float in the range of \verb{[0, 1]}.

Binary cross-entropy loss is often used for binary (0 or 1) classification
tasks. The loss function requires the following inputs:
\itemize{
\item \code{y_true} (true label): This is either 0 or 1.
\item \code{y_pred} (predicted value): This is the model's prediction, i.e, a single
floating-point value which either represents a
\href{https://en.wikipedia.org/wiki/Logit}{logit}, (i.e, value in \verb{[-inf, inf]}
when \code{from_logits=TRUE}) or a probability (i.e, value in \verb{[0., 1.]} when
\code{from_logits=FALSE}).
}

According to \href{https://arxiv.org/pdf/1708.02002}{Lin et al., 2018}, it
helps to apply a "focal factor" to down-weight easy examples and focus more
on hard examples. By default, the focal tensor is computed as follows:

\code{focal_factor = (1 - output) ** gamma} for class 1
\code{focal_factor = output ** gamma} for class 0
where \code{gamma} is a focusing parameter. When \code{gamma=0}, this function is
equivalent to the binary crossentropy loss.
}
\section{Examples}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{y_true <- rbind(c(0, 1), c(0, 0))
y_pred <- rbind(c(0.6, 0.4), c(0.4, 0.6))
loss <- loss_binary_focal_crossentropy(y_true, y_pred, gamma = 2)
loss
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([0.32986466 0.20579838], shape=(2), dtype=float64)

}\if{html}{\out{</div>}}

With the \code{compile()} API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{model \%>\% compile(
    loss = loss_binary_focal_crossentropy(
        gamma = 2.0, from_logits = TRUE),
    ...
)
}\if{html}{\out{</div>}}

As a standalone function:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Example 1: (batch_size = 1, number of samples = 4)
y_true <- op_array(c(0, 1, 0, 0))
y_pred <- op_array(c(-18.6, 0.51, 2.94, -12.8))
loss <- loss_binary_focal_crossentropy(gamma = 2, from_logits = TRUE)
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.6912122, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Apply class weight
loss <- loss_binary_focal_crossentropy(
  apply_class_balancing = TRUE, gamma = 2, from_logits = TRUE)
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.5101333, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Example 2: (batch_size = 2, number of samples = 4)
y_true <- rbind(c(0, 1), c(0, 0))
y_pred <- rbind(c(-18.6, 0.51), c(2.94, -12.8))
# Using default 'auto'/'sum_over_batch_size' reduction type.
loss <- loss_binary_focal_crossentropy(
    gamma = 3, from_logits = TRUE)
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.6469951, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Apply class weight
loss <- loss_binary_focal_crossentropy(
     apply_class_balancing = TRUE, gamma = 3, from_logits = TRUE)
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.48214132, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'sample_weight' attribute with focal effect
loss <- loss_binary_focal_crossentropy(
    gamma = 3, from_logits = TRUE)
loss(y_true, y_pred, sample_weight = c(0.8, 0.2))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.13312504, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Apply class weight
loss <- loss_binary_focal_crossentropy(
     apply_class_balancing = TRUE, gamma = 3, from_logits = TRUE)
loss(y_true, y_pred, sample_weight = c(0.8, 0.2))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.09735977, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'sum' reduction` type.
loss <- loss_binary_focal_crossentropy(
    gamma = 4, from_logits = TRUE,
    reduction = "sum")
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(1.2218808, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Apply class weight
loss <- loss_binary_focal_crossentropy(
    apply_class_balancing = TRUE, gamma = 4, from_logits = TRUE,
    reduction = "sum")
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.9140807, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'none' reduction type.
loss <- loss_binary_focal_crossentropy(
    gamma = 5, from_logits = TRUE,
    reduction = NULL)
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([0.00174837 1.1561027 ], shape=(2), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Apply class weight
loss <- loss_binary_focal_crossentropy(
    apply_class_balancing = TRUE, gamma = 5, from_logits = TRUE,
    reduction = NULL)
loss(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([4.3709317e-04 8.6707699e-01], shape=(2), dtype=float32)

}\if{html}{\out{</div>}}
}

\seealso{
Other losses: \cr
\code{\link{Loss}()} \cr
\code{\link{loss_binary_crossentropy}()} \cr
\code{\link{loss_categorical_crossentropy}()} \cr
\code{\link{loss_categorical_focal_crossentropy}()} \cr
\code{\link{loss_categorical_hinge}()} \cr
\code{\link{loss_circle}()} \cr
\code{\link{loss_cosine_similarity}()} \cr
\code{\link{loss_ctc}()} \cr
\code{\link{loss_dice}()} \cr
\code{\link{loss_hinge}()} \cr
\code{\link{loss_huber}()} \cr
\code{\link{loss_kl_divergence}()} \cr
\code{\link{loss_log_cosh}()} \cr
\code{\link{loss_mean_absolute_error}()} \cr
\code{\link{loss_mean_absolute_percentage_error}()} \cr
\code{\link{loss_mean_squared_error}()} \cr
\code{\link{loss_mean_squared_logarithmic_error}()} \cr
\code{\link{loss_poisson}()} \cr
\code{\link{loss_sparse_categorical_crossentropy}()} \cr
\code{\link{loss_squared_hinge}()} \cr
\code{\link{loss_tversky}()} \cr
\code{\link{metric_binary_crossentropy}()} \cr
\code{\link{metric_binary_focal_crossentropy}()} \cr
\code{\link{metric_categorical_crossentropy}()} \cr
\code{\link{metric_categorical_focal_crossentropy}()} \cr
\code{\link{metric_categorical_hinge}()} \cr
\code{\link{metric_hinge}()} \cr
\code{\link{metric_huber}()} \cr
\code{\link{metric_kl_divergence}()} \cr
\code{\link{metric_log_cosh}()} \cr
\code{\link{metric_mean_absolute_error}()} \cr
\code{\link{metric_mean_absolute_percentage_error}()} \cr
\code{\link{metric_mean_squared_error}()} \cr
\code{\link{metric_mean_squared_logarithmic_error}()} \cr
\code{\link{metric_poisson}()} \cr
\code{\link{metric_sparse_categorical_crossentropy}()} \cr
\code{\link{metric_squared_hinge}()} \cr
}
\concept{losses}
