% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cliForestSteppe.R
\name{cliForestSteppePoints}
\alias{cliForestSteppePoints}
\title{Forest-Steppe Models}
\usage{
cliForestSteppePoints(
  temp,
  prec,
  bsdf = NULL,
  lat = NULL,
  elv = NULL,
  year = 2000,
  aprchTEMP = c("hip", "tsi", "const"),
  aprchBSDF = c("hip", "const"),
  dvTEMP = rep(0.7, 12),
  verbose = FALSE
)
}
\arguments{
\item{temp}{'numeric' R object with one-year time series of monthly mean air temperature (in °C)}

\item{prec}{'numeric' R object with one-year time series of monthly precipitation sum (in mm)}

\item{bsdf}{'numeric' R object with one-year time series of monthly mean relative sunshine duration (dimensionless)}

\item{lat}{'numeric' vector with the latitude coordinates (in decimal degrees)}

\item{elv}{'numeric' vector with the elevation values (in meters above sea level)}

\item{year}{'numeric' vector with values of the year (using astronomical year numbering)}

\item{aprchTEMP}{'character' vector of length 1 that indicates the scheme used to generate daily values of the
daily mean air temperature for a specific year. Valid values are as follows: \cr
(a) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean air temperature in order to generate daily values; \cr
(b) \code{'tsi'} -
this scheme uses an iterative interpolation technique (Lüdeke et al. 1994) to time series of the monthly mean
air temperature, in order to generate a synthetic time series of the selected meteorological variable at a
temporal resolution that is higher than the daily scale; finally, this synthetic time series is upscaled to a
daily resolution; \cr
(c) \code{'const'} -
this scheme is assumed that values of the daily mean air temperature are constant within each month.}

\item{aprchBSDF}{'character' vector of length 1 that indicates the scheme used to generate daily values of the
daily fractional sunshine duration for a specific year. Valid values are as follows: \cr
(a) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean relative sunshine duration in order to generate daily values; \cr
(b) \code{'const'} -
this scheme is assumed that values of the daily relative sunshine duration are constant within each month.}

\item{dvTEMP}{'numeric' vector of length 12 with monthly values of the damping variable for the air temperature
data.}

\item{verbose}{'logical' scalar that indicates whether or not values of the bioclimatic indices used should be
added to the output.}
}
\value{
Depending on the setting, a data frame with three or more columns where the presence/absence data are
    stored in the last three columns labelled \code{'fsp_hlz'}, \code{'fsp_fai'} and \code{'fsp_svm'}, while the
    additional columns contain the values of bioclimatic indices used. If \code{verbose = FALSE}, the return
    object is a two- or three-column data frame with the presence/absence data, depending on the available data.
}
\description{
Calculates the values of bioclimatic indices used in forest-steppe models with different theoretical
    backgrounds, and estimates the presence/absence of 'forest-steppe' ecotone, for a given geographical location
    (latitude and elevation) and year/epoch, by using the monthly time series of climate variables.
}
\details{
Here, three forest-steppe models with different theoretical backgrounds are implemented:

    \itemize{
      \item{\code{fsp_hlz}: A modified variant of the widely used Holdridge life zone (HLZ) system (see for the
      basic concept Holdridge 1947, 1967; for a proposed variant Szelepcsényi et al. 2014).}
      \item{\code{fsp_fai}: A clarified version of the forestry climate classification (see for the basic concept
      Führer et al. 2011; for a proposed variant Mátyás et al. 2018)}
      \item{\code{fsp_svm}: An initial version of the Siberian Vegetation Model (see Monserud et al. 1993)}
    }

    The HLZ system classifies the vegetation type based on the distance from the ideal (theoretical) point in the
    3-dimensional space of the following bioclimatic indices:

    \itemize{
      \item{\code{abt}: Mean Annual Biotemperature (Eq 1 in Szelepcsényi et al. (2014); in °C)}
      \item{\code{tap}: Total Annual Precipitation (in mm)}
      \item{\code{per}: Potential Evapotranspiration Ratio (Eq 4 in Szelepcsényi et al. (2014); dimensionless)}
    }

    The plotting of thresholds of the above-mentioned bioclimatic indices in the HLZ chart leads to emerge a set
    of hexagons and triangles. The hexagons indicate the so-called core HLZ types, while the so-called
    transitional HLZ types are circumscribed by equilateral triangles in the HLZ chart (see Szelepcsényi et al.
    2014). However, in contrast to this study, here, the transitional types are defined as separate zones
    designated by the centres of the triangles. As a result, hexagons appear around the triangles in the HLZ
    chart, and in parallel, the size of the hexagons denoting the core types also decreases. Thus, the size of the
    core and transitional types are the same in this approach. During the classification, all forest-steppe types
    designated by Szelepcsényi et al. (2014) (and redefined by us) are aggregated into one class. \cr
    The forestry climate classification developed by Führer et al. (2011) was reworked by Mátyás et al. (2018). In
    the context of assessing the effects of future climate change, the 'forest-steppe' climate class was
    introduced in the model. In the work of Mátyás et al. (2018), this type is characterized by the Forestry
    Aridity Index (\code{fai}, dimensionless) values between 7.25 and 8. This definition is used here. \cr
    The Siberian Vegetation Model (Monserud et al. 1993) defines numerous types of forest-steppe on the basis of
    values of the Growing Degree-Days above 5°C (\code{gdd5}, in °C day), the Budyko's Dryness Index
    (\code{bdi}, dimensionless), and the Condrad's Continentality Index (\code{cci}, in per cent). Here, all
    such ecotone types are aggregated into one class, in order to estimate the presence/absence of the
    ‘forest-steppe’ ecotone.
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
    follows: \code{'temp'} (one-year time series of monthly mean air temperature), \code{'prec'} (one-year time
    series of monthly precipitation sum), and \code{'bsdf'} (one-year time series of monthly mean relative
    sunshine duration). The objects \code{'temp'}, \code{'prec'} and \code{'bsdf'} must be either vectors of
    length 12 or 12-column matrices. The first dimensions of these matrices have to be the same length. The
    function automatically converts vectors into single-row matrices during the error handling, and then uses these
    matrices. The first dimensions of these matrices determines the number of rows in the result matrix. In the
    case of arguments that do not affect the course of the calculation procedure or the structure of the return
    object, scalar values (i.e., 'numeric' vector of length 1) may also be allowed. In this case, they are as
    follows: \code{'lat'} (latitude coordinates in decimal degrees), \code{'elv'} (elevation in meters above sea
    level), and \code{'year'} (year using astronomical year numbering). These scalars are converted to vectors by
    the function during the error handling, and these vectors are applied in the further calculations. If these
    data are stored in vectors of length at least 2, their length must be the same size of first dimension of the
    matrices containing the basic data.
}
\examples{
# Loading mandatory data for the Example 'Points'
data(inp_exPoints)

# Predict the 'forest-steppe' ecotone (using the related bioclimatic indices),
# with default settings, at a grid cell near Szeged, Hungary (46.3N, 20.2E)
# (for the normal period 1981-2010)
with(inp_exPoints, {
year <- trunc(mean(seq(1981, 2010)))
fsp <- cliForestSteppePoints(colMeans(temp), colMeans(prec), colMeans(bsdf), lat, elv,
    year = year, verbose = TRUE)
fsp
})

}
\references{
\cite{Epstein ES (1991) On Obtaining Daily Climatological Values from Monthly Means. J Clim 4(3):365–368.
    \doi{10.1175/1520-0442(1991)004<0365:OODCVF>2.0.CO;2}}

\cite{Führer E, Horváth L, Jagodics A, Machon A, Szabados I (2011) Application of a new aridity index in Hungarian
    forestry practice. Időjárás 115(3):205–216}

\cite{Holdridge LR (1947) Determination of World Plant Formations From Simple Climatic Data. Science
    105(2727):367–368. \doi{10.1126/science.105.2727.367}}

\cite{Holdridge LR (1967) Life zone ecology. Tropical Science Center, San Jose, Costa Rica}

\cite{Lüdeke MKB, Badeck FW, Otto RD, Häger C, Dönges S, Kindermann J, Würth G, Lang T, Jäkel U, Klaudius A,
    Ramge P, Habermehl S, Kohlmaier GH (1994) The Frankfurt Biosphere Model: A global process-oriented model of
    seasonal and long-term CO2 exchange between terrestrial ecosystems and the atmosphere. I. Model description
    and illustrative results for cold deciduous and boreal forests. Clim Res 4(2):143-166. \doi{10.3354/cr004143}}

\cite{Mátyás Cs, Berki I, Bidló A, Csóka Gy, Czimber K, Führer E, Gálos B, Gribovszki Z, Illés G, Hirka A, Somogyi
    Z (2018) Sustainability of Forest Cover under Climate Change on the Temperate-Continental Xeric Limits.
    Forests 9(8):489. \doi{10.3390/f9080489}}

\cite{Monserud RA, Denissenko OV, Tchebakova NM (1993) Comparison of Siberian paleovegetation to current and
    future vegetation under climate change. Clim Res 3(3):143–159. \doi{10.3354/cr003143}}

\cite{Szelepcsényi Z, Breuer H, Sümegi P (2014) The climate of Carpathian Region in the 20th century based on the
    original and modified Holdridge life zone system. Cent Eur J Geosci 6(3):293–307.
    \doi{10.2478/s13533-012-0189-5}}
}
