% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/biomassRatio.R
\name{biomassRatio}
\alias{biomassRatio}
\title{Calculates the biomass ratio between two species groups}
\usage{
biomassRatio(X, ratio.groups, species.table, metric = "BIOMASS", years)
}
\arguments{
\item{X}{A dataframe of fishery independent data derived from research vessel
survey data or model output, with columns \code{YEAR}, \code{ID},
\code{SPECIES}, and \code{BIOMASS}. \code{YEAR} indicates the year the
observation was recorded, \code{ID} is an area code indicating where the
observation was recorded, \code{SPECIES} is a numeric code indicating the
species sampled, and \code{BIOMASS} is the corresponding biomass (stratified
and corrected for catchability as required).}

\item{ratio.groups}{A dataframe with two columns, which must be named
\code{group1} and \code{group2}. Each row holds the group names for one
biomass ratio, with the numerator in column \code{group1} and the
denominator in column \code{group2}. Each entry must be a character string
matching the name of a column in \code{species.table} or \code{"ALL"}.}

\item{species.table}{A table with column names that match the entries of
\code{ratio.groups}. The entries in each column are the species codes from
\code{X} indicating which species are included that group.
\code{species.table} may also include columns for other species groups;
these will be ignored. Note that an entry in \code{ratio.groups} could be
\code{"ALL"}. In this case, a column in \code{species.table} named
\code{"ALL"} is not required; the function will automatically include all
species in \code{X}.}

\item{metric}{A character string indicating which column in \code{X} to use to
calculate indicator. Default is \code{metric = "BIOMASS"}.}

\item{years}{A vector of years for which to calculate indicator.}
}
\value{
Returns a dataframe with columns \code{ID} and \code{YEAR}, and a
 column for each biomass ratio, named after the entries in ratio.groups, e.g.
 \code{group1_group2}.

 If there is no data for spatial scale \eqn{j} in year \eqn{i}, indicator
 values is assigned \code{NA}.
}
\description{
This function calculates the biomass ratio between two
 pre-defined species groups for \eqn{j} areas and \eqn{i} years.
}
\details{
One useful biomass ratio is the proportion of predatory fish in the
 community, which is estimated by:

 PropPred = Biomass of Predatory Fish Surveyed/Total Biomass Surveyed

 Predatory fish species are defined as all surveyed fish species that are not
 largely planktivorous, ie., fish that are piscivorous, or that feed on
 invertebrates larger than the macrozooplankton category (0.2 cm; Shin et al.
 2010). Phytoplankton, zooplankton and detritus feeders should be excluded.
 This indicator captures changes in the trophic structure and changes in the
 functional diversity of fish in the ecosystem.

 Other useful biomass \eqn{(B)} ratios indicators include:
 \deqn{B_{invertebrates}/B_{demersal}} and \deqn{B_{pelagic}/B_{demersal}}
}
\examples{
# Compile data
data(X)
data(species.table)

# Species groups of interest
ratio.groups <- data.frame(rbind(c("PELAGIC", "GROUNDFISH"), c("PREDATORS", "ALL")))
names(ratio.groups) <- c("group1", "group2")

# Calculate indicators
biomassRatio(X, ratio.groups = ratio.groups, species.table = species.table,
   metric = "BIOMASS", years = c(2014:2019))
}
\references{
Bundy A, Gomez C, Cook AM. 2017. Guidance framework for the
 selection and evaluation of ecological indicators. Can. Tech. Rep. Fish.
 Aquat. Sci. 3232: xii + 212 p.

 Bundy A, Heymans JJ, Morissette L, Savenkoff C (2009) Seals, cod and forage
 fish: A comparative exploration of variations in the theme of stock collapse
 and ecosystem change in four Northwest Atlantic ecosystems. Prog Oceanogr
 81:188 206

 Shin YJ, Shannon LJ, Bundy A, Coll M, Aydin K, Bez N, Blanchard JL, Borges,
 MF, Diallo I, Diaz E, Heymans JJ, Hill L, Johannesen E, Jouffre D, Kifani S,
 Labrosse P, Link JS, Mackinson S, Masski H, Möllmann C, Neira S, Ojaveer H,
 Abdallahi KM, Perry I, Thiao D, Yemane D, and Cury PM. 2010. Using
 indicators for evaluating, comparing and communicating the ecological status
 of exploited marine ecosystems. Part 2: Setting the scene. ICES Journal of
 Marine Science, 67: 692-716
}
\seealso{
Other ecosystem structure and function indicators: \code{\link{allStructure}},
  \code{\link{communityCondition}},
  \code{\link{largeFishIndicator}},
  \code{\link{largeSpeciesIndicator}},
  \code{\link{meanTLCommunity}}
}
\author{
Danielle Dempsey \email{Danielle.Dempsey@dfo-mpo.gc.ca}, Adam Cook,
 Catalina Gomez, Alida Bundy
}
\concept{ecosystem structure and function indicators}
