% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powerMethod.R
\name{powerMethod}
\alias{powerMethod}
\title{Power Method for Eigenvectors}
\usage{
powerMethod(A, v = NULL, eps = 1e-06, maxiter = 100, plot = FALSE)
}
\arguments{
\item{A}{a square numeric matrix}

\item{v}{optional starting vector; if not supplied, it uses a unit vector of length equal to the number of rows / columns of \code{x}.}

\item{eps}{convergence threshold for terminating iterations}

\item{maxiter}{maximum number of iterations}

\item{plot}{logical; if \code{TRUE}, plot the series of iterated eigenvectors?}
}
\value{
a list containing the eigenvector (\code{vector}), eigenvalue (\code{value}), iterations (\code{iter}),
  and iteration history (\code{vector_iterations})
}
\description{
Finds a dominant eigenvalue, \eqn{\lambda_1}, and its corresponding
eigenvector, \eqn{v_1}, of a square matrix by applying Hotelling's (1933) Power Method with scaling.
}
\details{
The method is based upon the fact that repeated multiplication of a matrix \eqn{A} by a trial
vector \eqn{v_1^{(k)}} converges to the value of the eigenvector,
\deqn{v_1^{(k+1)} = A v_1^{(k)} / \vert\vert A v_1^{(k)} \vert\vert }
The corresponding eigenvalue is then found as
\deqn{\lambda_1 = \frac{v_1^T A v_1}{v_1^T  v_1}}

In pre-computer days, this method could be extended to find subsequent eigenvalue - eigenvector
pairs by "deflation", i.e., by applying the method again to the new matrix.
\eqn{A - \lambda_1 v_1 v_1^{T} }.

This method is still used in some computer-intensive applications with huge matrices where only the
dominant eigenvector is required, e.g., the Google Page Rank algorithm.
}
\examples{
A <- cbind(c(7, 3), c(3, 6))
powerMethod(A)
eigen(A)$values[1] # check
eigen(A)$vectors[,1]

# demonstrate how the power method converges to a solution
powerMethod(A, v = c(-.5, 1), plot = TRUE)

B <- cbind(c(1, 2, 0), c(2, 1, 3), c(0, 3, 1))
(rv <- powerMethod(B))

# deflate to find 2nd latent vector
l <- rv$value
v <- c(rv$vector)
B1 <- B - l * outer(v, v)
powerMethod(B1)
eigen(B)$vectors     # check

# a positive, semi-definite matrix, with eigenvalues 12, 6, 0
C <- matrix(c(7, 4, 1,  4, 4, 4,  1, 4, 7), 3, 3)
eigen(C)$vectors
powerMethod(C)
}
\references{
Hotelling, H. (1933). Analysis of a complex of statistical variables into principal components. \emph{Journal of Educational Psychology}, 24, 417-441, and 498-520.
}
\author{
Gaston Sanchez (from matrixkit)
}
