% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate_imputation_parameters.R
\name{evaluate_imputation_parameters}
\alias{evaluate_imputation_parameters}
\title{Evaluate estimated parameters after imputation}
\usage{
evaluate_imputation_parameters(
  imp_ds,
  orig_ds = NULL,
  true_pars = NULL,
  parameter = "mean",
  criterion = "RMSE",
  which_cols = seq_len(ncol(imp_ds)),
  tolerance = sqrt(.Machine$double.eps),
  ...
)
}
\arguments{
\item{imp_ds}{a data frame or matrix with imputed values}

\item{orig_ds}{a data frame or matrix with original (true) values}

\item{true_pars}{true parameters, normally a vector or a matrix}

\item{parameter}{a string specifying the estimated parameters for comparison}

\item{criterion}{a string specifying the used criterion for comparing the
imputed and original values}

\item{which_cols}{indices or names of columns used for evaluation}

\item{tolerance}{numeric, only used for \code{criterion = "precision"}:
numeric differences smaller than tolerance are treated as zero/equal}

\item{...}{further arguments passed to function for parameter estimation}
}
\value{
a numeric vector of length one
}
\description{
Compares estimated parameters after imputation to true parameters or
estimates based on the original dataset
}
\details{
Either \code{orig_ds} or \code{true_pars} must be supplied and the other one
must be \code{NULL} (default: both are \code{NULL}, just supply one, see
examples). The following \code{parameter}s are implemented:
"mean", "median", "var", "sd", "quantile", "cov", "cor".
Some details follows:
\itemize{
\item{"var": only the variances of the columns (the diagonal elements of the
covariance matrix) are compared. The whole covariance matrix can be compared
with "cov".}
\item{"quantile": the quantiles can be set via the additional
argument \code{probs} (see examples). Otherwise, the default quantiles from
\code{\link[stats]{quantile}} will be used.}
}

The argument \code{which_cols} allows the selection of columns for comparison
(see examples). If \code{true_pars} is used, it is assumed that only relevant
parameters are supplied (see examples).

Possible choices for the argument \code{criterion} are documented in
\code{\link{evaluate_imputed_values}}
}
\examples{
# only orig_ds known
orig_ds <- data.frame(X = 1:10, Y = 101:101)
imp_ds <- impute_mean(delete_MCAR(orig_ds, 0.4))
evaluate_imputation_parameters(imp_ds, orig_ds = orig_ds)

# true parameters known
orig_ds <- data.frame(X = rnorm(100), Y = rnorm(100, mean = 10))
imp_ds <- impute_mean(delete_MCAR(orig_ds, 0.3))
evaluate_imputation_parameters(imp_ds, true_pars = c(0, 10), parameter = "mean")
evaluate_imputation_parameters(imp_ds, true_pars = c(1, 1), parameter = "var")

# set quantiles
evaluate_imputation_parameters(imp_ds,
  true_pars = c(qnorm(0.3), qnorm(0.3, mean = 10)),
  parameter = "quantile", probs = 0.3
)

# compare only column Y
evaluate_imputation_parameters(imp_ds,
  true_pars = c(Y = 10), parameter = "mean",
  which_cols = "Y"
)
}
\references{
Cetin-Berber, D. D., Sari, H. I., & Huggins-Manley, A. C. (2019).
  Imputation Methods to Deal With Missing Responses in Computerized Adaptive
  Multistage Testing. \emph{Educational and psychological measurement},
  79(3), 495-511.
}
\seealso{
Other evaluation functions: 
\code{\link{evaluate_imputed_values}()},
\code{\link{evaluate_parameters}()}
}
\concept{evaluation functions}
