% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TunerHyperband.R
\name{mlr_tuners_hyperband}
\alias{mlr_tuners_hyperband}
\alias{TunerHyperband}
\title{Tuner using the Hyperband algorithm}
\source{
Li L, Jamieson K, DeSalvo G, Rostamizadeh A, Talwalkar A (2018).
\dQuote{Hyperband: A Novel Bandit-Based Approach to Hyperparameter Optimization.}
\emph{Journal of Machine Learning Research}, \bold{18}(185), 1-52.
\url{https://jmlr.org/papers/v18/16-558.html}.
}
\description{
\code{TunerHyperband} class that implements hyperband tuning. Hyperband is a
budget oriented-procedure, weeding out suboptimal performing configurations
early in a sequential training process, increasing tuning efficiency as a
consequence.

For this, several brackets are constructed with an associated set of
configurations for each bracket. Each bracket has several stages. Different
brackets are initialized with different amounts of configurations and
different budget sizes.

Within the context of hyperband each evaluation of a learner consumes a
certain budget. This budget is not fixed but controlled by a certain
hyperparameter, e.g. the number of boosting iterations or the number of trees
in a random forest. The user has to specify explicitly which hyperparameter
of the learner controls the consumption of the budget by
tagging a single hyperparameter in the \link[paradox:ParamSet]{paradox::ParamSet} with \code{"budget"}.
An alternative approach using subsampling and pipelines is described below.

Naturally, hyperband terminates once all of its brackets are evaluated, so a
\link[bbotk:Terminator]{bbotk::Terminator} in the tuning instance acts as an upper bound and should
be only set to a low value if one is unsure of how long hyperband will take
to finish under the given settings.
}
\section{Dictionary}{

This \link{Optimizer} can be instantiated via the \link[mlr3misc:Dictionary]{dictionary}
\link{mlr_optimizers} or with the associated sugar function \code{\link[=opt]{opt()}}:\preformatted{mlr_optimizers$get("hyperband")
opt("hyperband")
}
}

\section{Parameters}{

\describe{
\item{\code{eta}}{\code{numeric(1)}\cr
Fraction parameter of the successive halving algorithm: With every step the
configuration budget is increased by a factor of \code{eta} and only the best
\code{1/eta} configurations are used for the next stage. Non-integer values are
supported, but \code{eta} is not allowed to be less or equal 1.}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn during
the initialization of each bracket. The default is uniform sampling.}
}
}

\section{Archive}{

The \link[mlr3tuning:ArchiveTuning]{mlr3tuning::ArchiveTuning} holds the following additional columns that
are specific to the hyperband tuner:
\itemize{
\item \code{bracket} (\code{integer(1)})\cr
The console logs about the bracket index are actually not matching
with the original hyperband algorithm, which counts down the brackets
and stops after evaluating bracket 0. The true bracket indices are
given in this column.
\item \code{bracket_stage} (\verb{integer(1))}\cr
The bracket stage of each bracket. Hyperband starts counting at 0.
\item \code{budget_scaled} (\code{numeric(1)})\cr
The intermediate budget in each bracket stage calculated by hyperband.
Because hyperband is originally only considered for budgets starting at 1, some
rescaling is done to allow budgets starting at different values.
For this, budgets are internally divided by the lower budget bound to
get a lower budget of 1. Before the learner
receives its budgets for evaluation, the budget is transformed back to
match the original scale again.
\item \code{budget_real} (\code{numeric(1)})\cr
The real budget values the learner uses for evaluation after hyperband
calculated its scaled budget.
\item \code{n_configs} (\code{integer(1)})\cr
The amount of evaluated configurations in each stage. These correspond
to the \code{r_i} in the original paper.
}
}

\section{Hyperband without learner budget}{

Thanks to \CRANpkg{mlr3pipelines}, it is possible to use hyperband in
combination with learners lacking a natural budget parameter. For example,
any \link[mlr3:Learner]{mlr3::Learner} can be augmented with a \link[mlr3pipelines:PipeOp]{mlr3pipelines::PipeOp}
operator such as \link[mlr3pipelines:mlr_pipeops_subsample]{mlr3pipelines::PipeOpSubsample}. With the
subsampling rate as budget parameter, the resulting
\link[mlr3pipelines:mlr_learners_graph]{mlr3pipelines::GraphLearner} is fitted on small proportions of
the \link[mlr3:Task]{mlr3::Task} in the first brackets, and on the complete Task in
last brackets. See examples for some code.
}

\section{Custom sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}
}

\section{Runtime}{

The calculation of each bracket currently assumes a linear runtime in the
chosen budget parameter is always given. Hyperband is designed so each
bracket requires approximately the same runtime as the sum of the budget
over all configurations in each bracket is roughly the same. This will not
hold true once the scaling in the budget parameter is not linear
anymore, even though the sum of the budgets in each bracket remains the
same. A possible adaption would be to introduce a trafo, like it is shown in
the \emph{examples} section.
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link{Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Parallelization}{

In order to support general termination criteria and parallelization, we
evaluate points in a batch-fashion of size \code{batch_size}. The points of one
stage in a bracket are evaluated in one batch. Parallelization is supported
via package \CRANpkg{future} (see \code{\link[mlr3:benchmark]{mlr3::benchmark()}}'s section on
parallelization for more details).
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\examples{
if(requireNamespace("xgboost")) {
library(mlr3learners)

# define hyperparameter and budget parameter
search_space = ps(
  nrounds = p_int(lower = 1, upper = 16, tags = "budget"),
  eta = p_dbl(lower = 0, upper = 1),
  booster = p_fct(levels = c("gbtree", "gblinear", "dart"))
)

\donttest{
# hyperparameter tuning on the pima indians diabetes data set
instance = tune(
  method = "hyperband",
  task = tsk("pima"),
  learner = lrn("classif.xgboost", eval_metric = "logloss"),
  resampling = rsmp("cv", folds = 3),
  measure = msr("classif.ce"),
  search_space = search_space
)

# best performing hyperparameter configuration
instance$result
}
}
}
\section{Super classes}{
\code{\link[mlr3tuning:Tuner]{mlr3tuning::Tuner}} -> \code{\link[mlr3tuning:TunerFromOptimizer]{mlr3tuning::TunerFromOptimizer}} -> \code{TunerHyperband}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{TunerHyperband$new()}}
\item \href{#method-clone}{\code{TunerHyperband$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="format">}\href{../../mlr3tuning/html/Tuner.html#method-format}{\code{mlr3tuning::Tuner$format()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="print">}\href{../../mlr3tuning/html/Tuner.html#method-print}{\code{mlr3tuning::Tuner$print()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="mlr3tuning" data-topic="TunerFromOptimizer" data-id="optimize">}\href{../../mlr3tuning/html/TunerFromOptimizer.html#method-optimize}{\code{mlr3tuning::TunerFromOptimizer$optimize()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerHyperband$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerHyperband$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
