% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TuningInstanceMulticrit.R
\name{TuningInstanceMultiCrit}
\alias{TuningInstanceMultiCrit}
\title{Multi Criteria Tuning Instance}
\description{
Specifies a general multi-criteria tuning scenario, including objective
function and archive for Tuners to act upon. This class stores an
\code{ObjectiveTuning} object that encodes the black box objective function which
a \link{Tuner} has to optimize. It allows the basic operations of querying the
objective at design points (\verb{$eval_batch()}), storing the evaluations in the
internal \code{Archive} and accessing the final result (\verb{$result}).

Evaluations of hyperparameter configurations are performed in batches by
calling \code{\link[mlr3:benchmark]{mlr3::benchmark()}} internally. Before a batch is evaluated, the
\link[bbotk:Terminator]{bbotk::Terminator} is queried for the remaining budget. If the available
budget is exhausted, an exception is raised, and no further evaluations can
be performed from this point on.

The tuner is also supposed to store its final result, consisting of a
selected hyperparameter configuration and associated estimated performance
values, by calling the method \code{instance$assign_result}.
}
\examples{
library(data.table)

# define search space
search_space = ps(
  cp = p_dbl(lower = 0.001, upper = 0.1),
  minsplit = p_int(lower = 1, upper = 10)
)

# initialize instance
instance = TuningInstanceMultiCrit$new(
  task = tsk("iris"),
  learner = lrn("classif.rpart"),
  resampling = rsmp("holdout"),
  measure = msrs(c("classif.ce", "classif.acc")),
  search_space = search_space,
  terminator = trm("evals", n_evals = 5)
)

# generate design
design = data.table(cp = c(0.05, 0.01), minsplit = c(5, 3))

# eval design
instance$eval_batch(design)

# show archive
instance$archive
}
\section{Super classes}{
\code{\link[bbotk:OptimInstance]{bbotk::OptimInstance}} -> \code{\link[bbotk:OptimInstanceMultiCrit]{bbotk::OptimInstanceMultiCrit}} -> \code{TuningInstanceMultiCrit}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{result_learner_param_vals}}{(\code{list()})\cr
List of param values for the optimal learner call.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{TuningInstanceMultiCrit$new()}}
\item \href{#method-assign_result}{\code{TuningInstanceMultiCrit$assign_result()}}
\item \href{#method-clone}{\code{TuningInstanceMultiCrit$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="clear">}\href{../../bbotk/html/OptimInstance.html#method-clear}{\code{bbotk::OptimInstance$clear()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="eval_batch">}\href{../../bbotk/html/OptimInstance.html#method-eval_batch}{\code{bbotk::OptimInstance$eval_batch()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="format">}\href{../../bbotk/html/OptimInstance.html#method-format}{\code{bbotk::OptimInstance$format()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="objective_function">}\href{../../bbotk/html/OptimInstance.html#method-objective_function}{\code{bbotk::OptimInstance$objective_function()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="print">}\href{../../bbotk/html/OptimInstance.html#method-print}{\code{bbotk::OptimInstance$print()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.

This defines the resampled performance of a learner on a task, a
feasibility region for the parameters the tuner is supposed to optimize,
and a termination criterion.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceMultiCrit$new(
  task,
  learner,
  resampling,
  measures,
  terminator,
  search_space = NULL,
  store_benchmark_result = TRUE,
  store_models = FALSE,
  check_values = FALSE,
  allow_hotstart = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{task}}{(\link[mlr3:Task]{mlr3::Task})\cr
Task to operate on.}

\item{\code{learner}}{(\link[mlr3:Learner]{mlr3::Learner})\cr
Learner to tune.}

\item{\code{resampling}}{(\link[mlr3:Resampling]{mlr3::Resampling})\cr
Resampling that is used to evaluated the performance of the hyperparameter
configurations. Uninstantiated resamplings are instantiated during
construction so that all configurations are evaluated on the same data
splits. Already instantiated resamplings are kept unchanged. Specialized
\link{Tuner} change the resampling e.g. to evaluate a hyperparameter
configuration on different data splits. This field, however, always returns
the resampling passed in construction.}

\item{\code{measures}}{(list of \link[mlr3:Measure]{mlr3::Measure})\cr
Measures to optimize.}

\item{\code{terminator}}{(\link{Terminator})\cr
Stop criterion of the tuning process.}

\item{\code{search_space}}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Hyperparameter search space. If \code{NULL} (default), the search space is
constructed from the \link{TuneToken} of the learner's parameter set
(learner$param_set).}

\item{\code{store_benchmark_result}}{(\code{logical(1)})\cr
If \code{TRUE} (default), store resample result of evaluated hyperparameter
configurations in archive as \link[mlr3:BenchmarkResult]{mlr3::BenchmarkResult}.}

\item{\code{store_models}}{(\code{logical(1)})\cr
If \code{TRUE}, fitted models are stored in the benchmark result
(\code{archive$benchmark_result}). If \code{store_benchmark_result = FALSE}, models
are only stored temporarily and not accessible after the tuning. This
combination is needed for measures that require a model.}

\item{\code{check_values}}{(\code{logical(1)})\cr
If \code{TRUE}, hyperparameter values are checked before evaluation and
performance scores after. If \code{FALSE} (default), values are unchecked but
computational overhead is reduced.}

\item{\code{allow_hotstart}}{(\code{logical(1)})\cr
Allow to hotstart learners with previously fitted models. See also
\link[mlr3:HotstartStack]{mlr3::HotstartStack}. The learner must support hotstarting. Sets
\code{store_models = TRUE}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-assign_result"></a>}}
\if{latex}{\out{\hypertarget{method-assign_result}{}}}
\subsection{Method \code{assign_result()}}{
The \link{Tuner} object writes the best found points
and estimated performance values here. For internal use.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceMultiCrit$assign_result(xdt, ydt, learner_param_vals = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{xdt}}{(\code{data.table::data.table()})\cr
Hyperparameter values as \code{data.table::data.table()}. Each row is one
configuration. Contains values in the search space. Can contain additional
columns for extra information.}

\item{\code{ydt}}{(\code{data.table::data.table()})\cr
Optimal outcomes, e.g. the Pareto front.}

\item{\code{learner_param_vals}}{(List of named \verb{list()s})\cr
Fixed parameter values of the learner that are neither part of the}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceMultiCrit$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
